package com.alibaba.fastsql.sql.optimizer.visitor;

import com.alibaba.fastsql.DbType;
import com.alibaba.fastsql.sql.SQLUtils;
import com.alibaba.fastsql.sql.ast.SQLDataType;
import com.alibaba.fastsql.sql.ast.SQLExpr;
import com.alibaba.fastsql.sql.ast.SQLName;
import com.alibaba.fastsql.sql.ast.SQLObject;
import com.alibaba.fastsql.sql.ast.expr.*;
import com.alibaba.fastsql.sql.ast.statement.SQLColumnDefinition;
import com.alibaba.fastsql.sql.ast.statement.SQLJoinTableSource;
import com.alibaba.fastsql.sql.ast.statement.SQLSelectQueryBlock;
import com.alibaba.fastsql.sql.visitor.SQLASTVisitorAdapter;
import com.alibaba.fastsql.sql.visitor.functions.*;
import com.alibaba.fastsql.util.FnvHash;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

public class ConstFolding extends SQLASTVisitorAdapter {
    private int optimizedCount = 0;

    public ConstFolding() {

    }

    public ConstFolding(DbType dbType) {
        this.dbType = dbType;
    }

    public boolean visit(SQLMethodInvokeExpr x) {
        SQLExpr owner = x.getOwner();
        if (owner != null) {
            owner.accept(this);
        }

        for (SQLExpr param : x.getArguments()) {
            param.accept(this);
        }

        SQLExpr for_ = x.getFor();
        if (for_ != null) {
            for_.accept(this);
        }

        boolean allConst = true;
        for (SQLExpr param : x.getArguments()) {
            if (param instanceof SQLValuableExpr) {

            } else {
                allConst = false;
            }
        }

        if (allConst) {
            long hash = x.methodNameHashCode64();
            if (hash == FnvHash.Constants.SUBSTR) {
                // fname = substr('123', 1, 2) -> fname = '12'
                Object val = Substring.instance.eval(x);
                if (val instanceof String) {
                    if (SQLUtils.replaceInParent(x, new SQLCharExpr((String) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.CONCAT) {
                // concat('1', '2') -> '12'
                Object val = Concat.instance.eval(x);
                if (val instanceof String) {
                    if (SQLUtils.replaceInParent(x, new SQLCharExpr((String) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.LCASE || hash == FnvHash.Constants.LOWER) {
                // lcase('aBc') -> 'abc'
                Object val = Lcase.instance.eval(x);
                if (val instanceof String) {
                    if (SQLUtils.replaceInParent(x, new SQLCharExpr((String) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.UCASE || hash == FnvHash.Constants.UPPER) {
                // ucase('aBc') -> 'ABC'
                Object val = Ucase.instance.eval(x);
                if (val instanceof String) {
                    if (SQLUtils.replaceInParent(x, new SQLCharExpr((String) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.LENGTH || hash == FnvHash.Constants.LEN) {
                // len('aBc') -> 3
                Object val = Length.instance.eval(x);
                if (val instanceof Integer) {
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr((Integer) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.GREAST) {
                // greast（1，2，3) -> 3
                Object val = Greatest.instance.eval(x);
                if (val instanceof Integer) {
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr((Integer) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.LEAST) {
                // least（1，2，3) -> 1
                Object val = Least.instance.eval(x);
                if (val instanceof Integer) {
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr((Integer) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.IFNULL) {
                //
                Object val = IfNull.instance.eval(x);
                if (val instanceof Integer) {
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr((Integer) val))) {
                        optimizedCount++;
                    }
                    return false;
                } else if (val instanceof String) {
                    if (SQLUtils.replaceInParent(x, new SQLCharExpr((String) val))) {
                        optimizedCount++;
                    }
                    return false;
                }
            } else if (hash == FnvHash.Constants.IF) {
                if (x.getArguments().size() == 3) {
                    SQLExpr condExpr = x.getArguments().get(0);
                    if (condExpr instanceof SQLValuableExpr) {
                        Object condVal = ((SQLValuableExpr) condExpr).getValue();
                        if (condVal instanceof Boolean) {
                            if (((Boolean) condVal).booleanValue()) {
                                SQLUtils.replaceInParent(x, x.getArguments().get(1));
                            } else {
                                SQLUtils.replaceInParent(x, x.getArguments().get(2));
                            }
                            return false;
                        }
                    }
                }
            } else if (hash == FnvHash.Constants.TO_DATE) {
                List<SQLExpr> arguments = x.getArguments();
                boolean supportDateExpr = dbType == DbType.mysql
                        || dbType == DbType.ads
                        || dbType == DbType.hive;

                if (arguments.size() == 2
                        && arguments.get(0) instanceof SQLCharExpr
                        && arguments.get(1) instanceof SQLCharExpr
                        && supportDateExpr) {
                    String chars = ((SQLCharExpr) arguments.get(0)).getText();
                    String format = ((SQLCharExpr) arguments.get(1)).getText();
                    if (format.equals("yyyymmdd")) {
                        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
                        try {
                            Date date = dateFormat.parse(chars);
                            String dateLiteral = new SimpleDateFormat("yyyy-MM-dd").format(date);
                            if (SQLUtils.replaceInParent(x, new SQLDateExpr(dateLiteral))) {
                                optimizedCount++;
                            }
                        } catch (ParseException e) {
                            // skip
                            return false;
                        }
                    }
                }
            } else if (hash == FnvHash.Constants.DATEADD) {
                List<SQLExpr> arguments = x.getArguments();
                boolean supportDateAdd = dbType == DbType.mysql
                        || dbType == DbType.ads;
                if (arguments.size() == 3
                        && arguments.get(0) instanceof SQLDateExpr
                        && arguments.get(1) instanceof SQLIntegerExpr
                        && arguments.get(2) instanceof SQLCharExpr
                        && supportDateAdd) {
                    SQLDateExpr dateExpr = (SQLDateExpr) arguments.get(0);
                    int delta = ((SQLIntegerExpr) arguments.get(1)).getNumber().intValue();
                    String type = ((SQLCharExpr) arguments.get(2)).getText();

                    if ("day".equalsIgnoreCase(type)) {
                        SQLDateExpr dateCloned = dateExpr.clone();
                        if (dateCloned.addDay(delta)) {
                            if (SQLUtils.replaceInParent(x, dateCloned)) {
                                optimizedCount++;
                            }
                        }
                    }

                }
            } else if (hash == FnvHash.Constants.TO_CHAR) {
                List<SQLExpr> arguments = x.getArguments();
                if (arguments.size() == 2
                        && arguments.get(0) instanceof SQLDateExpr
                        && arguments.get(1) instanceof SQLCharExpr) {
                    SQLDateExpr dateExpr = (SQLDateExpr) arguments.get(0);
                    Date date = dateExpr.getDate();

                    String format = ((SQLCharExpr) arguments.get(1)).getText();
                    if (format.equals("yyyymmdd") && date != null) {
                        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
                        String chars = dateFormat.format(date);
                        if (SQLUtils.replaceInParent(x, new SQLCharExpr(chars))) {
                            optimizedCount++;
                        }
                    }
                }
            }
        }

        return false;
    }

    public boolean visit(SQLBinaryOpExpr x) {

        SQLBinaryOperator op = x.getOperator();

        x.getLeft().accept(this);
        x.getRight().accept(this);

        if ((op == SQLBinaryOperator.BooleanAnd || op == SQLBinaryOperator.BooleanOr)
                && x.getLeft() instanceof SQLBinaryOpExpr
                && ((SQLBinaryOpExpr) x.getLeft()).getOperator() == x.getOperator()) {
            SQLBinaryOpExprGroup group = new SQLBinaryOpExprGroup(op, x.getDbType());
            group.add(x.getLeft());
            group.add(x.getRight());
            if (SQLUtils.replaceInParent(x, group)) {
                optimizedCount++;
                group.accept(this);
                return false;
            }
        }

        if (op == SQLBinaryOperator.Like && x.getRight() instanceof SQLCharExpr) {
            // fname like 'aBc' -> fname = 'aBc'
            SQLCharExpr right = (SQLCharExpr) x.getRight();
            String pattern = right.getText();
            if (pattern.indexOf('%') == -1 && pattern.indexOf('_') == -1) {
                x.setOperator(SQLBinaryOperator.Equality);
            }
        } else if (op == SQLBinaryOperator.NotLike && x.getRight() instanceof SQLCharExpr) {
            // fname note like 'aBc' -> fname <> 'aBc'
            SQLCharExpr right = (SQLCharExpr) x.getRight();
            String pattern = right.getText();
            if (pattern.indexOf('%') == -1 && pattern.indexOf('_') == -1) {
                x.setOperator(SQLBinaryOperator.LessThanOrGreater);
            }
        }

        // col + 3 + 4 => col + 7
        if (isNameAndLiteral(x.getLeft()) && x.getRight() instanceof SQLLiteralExpr) {
            SQLBinaryOpExpr left = (SQLBinaryOpExpr) x.getLeft();
            SQLLiteralExpr right = (SQLLiteralExpr) x.getRight();
            SQLLiteralExpr leftRight = (SQLLiteralExpr) left.getRight();
            if (left.getOperator() == x.getOperator()) {
                switch (left.getOperator()) {
                    case Add:
                    case Subtract:
                    case Mod:
                    case Modulus:
                    case Multiply:
                    case DIV:
                    case Divide:
                    case Concat:
                        x.setLeft(left.getLeft());
                        x.setRight(new SQLBinaryOpExpr(leftRight, left.getOperator(), right));
                        x.getRight().accept(this);
                        break;
                    default:
                        break;
                }
            }
        }

        if (x.getLeft() instanceof SQLValuableExpr && x.getRight() instanceof SQLValuableExpr) {
            handValueValue(x, op
                    , (SQLValuableExpr) x.getLeft()
                    , (SQLValuableExpr) x.getRight());
            return false;
        }

        if (x.getLeft() instanceof SQLDateExpr
                && x.getRight() instanceof SQLIntervalExpr
                && (op == SQLBinaryOperator.Add || op == SQLBinaryOperator.Subtract)) {
            SQLDateExpr left = (SQLDateExpr) x.getLeft();
            SQLIntervalExpr right = (SQLIntervalExpr) x.getRight();

            String intervalText;
            int intervalInt;
            if (right.getValue() instanceof SQLCharExpr) {
                intervalText = ((SQLCharExpr) right.getValue()).getText();
                try {
                    intervalInt = Integer.parseInt(intervalText);
                } catch (NumberFormatException ex) {
                    // skip
                    return false;
                }
            } else {
                return false;
            }

            if (op == SQLBinaryOperator.Subtract) {
                intervalInt = -intervalInt;
            }

            String text = left.getValue();
            if (text.length() == 10) {
                SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                try {
                    Date date = format.parse(text);
                    Calendar calendar = Calendar.getInstance();
                    calendar.setTime(date);

                    if (right.getUnit() == SQLIntervalUnit.DAY) {
                        calendar.add(Calendar.DAY_OF_MONTH, intervalInt);
                        String dateChanged = format.format(calendar.getTime());
                        if (SQLUtils.replaceInParent(x, new SQLDateExpr(dateChanged))) {
                            optimizedCount++;
                            return false;
                        }
                    }

                } catch (ParseException ex) {
                    // skip
                }
            }
        }

        if (isLeftLiteralAndRightName(x)) {
            SQLName right = (SQLName) x.getRight();
            SQLLiteralExpr left = (SQLLiteralExpr) x.getLeft();

            if (x.getOperator() == SQLBinaryOperator.LessThan) {
                // 3 < col -> col > 3
                x.setRight(left);
                x.setLeft(right);
                x.setOperator(SQLBinaryOperator.GreaterThan);
                optimizedCount++;
            } else if (x.getOperator() == SQLBinaryOperator.Equality) {
                // 3 = id -> id = 3
                x.setRight(left);
                x.setLeft(right);
                optimizedCount++;
            } else if (x.getOperator() == SQLBinaryOperator.GreaterThanOrEqual) {
                // 3 = id -> id = 3
                x.setRight(left);
                x.setLeft(right);
                x.setOperator(SQLBinaryOperator.LessThanOrEqual);
                optimizedCount++;
            }
        }

        if (x.isLeftNameAndRightLiteral()) {
            SQLName name = (SQLName) x.getLeft();
            SQLLiteralExpr literal = (SQLLiteralExpr) x.getRight();

            handleNameLiteral(x, name, literal);
        }

        if (isLeftFunctionAndRightLiteral(x)) {
            SQLMethodInvokeExpr functionCall = (SQLMethodInvokeExpr) x.getLeft();
            SQLLiteralExpr literal = (SQLLiteralExpr) x.getRight();

            long nameHash = functionCall.methodNameHashCode64();
            SQLDataType dataType = functionCall.getResolvedReturnDataType();

            if (dataType != null) {
                final long dataTypeNameHash = dataType.nameHashCode64();
                if (dataTypeNameHash == FnvHash.Constants.BIGINT) {
                    if (literal instanceof SQLCharExpr) {
                        String text = ((SQLCharExpr) literal).getText();
                        try {
                            // id = '123' -> id = 123
                            long val = Long.parseLong(text);
                            x.setRight(new SQLIntegerExpr(val));
                            optimizedCount++;
                        } catch (NumberFormatException ex) {
                            // id = '123' -> false
                            if (SQLUtils.replaceInParent(x, new SQLBooleanExpr(false))) {
                                optimizedCount++;
                                return false;
                            }
                        }
                    }
                }
            }
        }

        if (op == SQLBinaryOperator.BooleanAnd) {
            if (isLeftNameAndRightLiteral(x.getLeft()) && isLeftNameAndRightLiteral(x.getRight())) {
                SQLBinaryOpExpr left = (SQLBinaryOpExpr) x.getLeft();
                SQLBinaryOpExpr right = (SQLBinaryOpExpr) x.getRight();

                mergeTwiceOp(x, left, right);
            } else if (isBothName(x.getLeft()) && isLeftNameAndRightLiteral(x.getRight())) {
                SQLBinaryOpExpr left = (SQLBinaryOpExpr) x.getLeft();
                SQLBinaryOpExpr right = (SQLBinaryOpExpr) x.getRight();

                if (left.getOperator() == right.getOperator()) {
                    if (left.getLeft().equals(right.getLeft())) {
                        left.setRight(right.getRight().clone());
                        optimizedCount++;
                    } else if (left.getRight().equals(right.getLeft())) {
                        left.setLeft(left.getRight());
                        left.setRight(right.getRight().clone());
                        optimizedCount++;
                    }
                }
            }
        }

        if (op == SQLBinaryOperator.Equality) {
            if(isLeftNameAndRightLiteral(x.getLeft()) && x.getRight() instanceof SQLLiteralExpr) {
                SQLBinaryOpExpr left = (SQLBinaryOpExpr) x.getLeft();
                if (left.getOperator() == SQLBinaryOperator.Add
                        && left.getRight() instanceof SQLIntegerExpr
                        && x.getRight() instanceof SQLIntegerExpr) {
                    int val = ((SQLIntegerExpr) x.getRight()).getNumber().intValue() - ((SQLIntegerExpr) left.getRight()).getNumber().intValue();
                    x.setRight(new SQLIntegerExpr(val));
                    x.setLeft(left.getLeft());
                    optimizedCount++;
                    return false;
                }
            } else if (x.isLeftNameAndRightLiteral() && x.getParent() instanceof SQLSelectQueryBlock) {
                SQLSelectQueryBlock queryBlock = (SQLSelectQueryBlock) x.getParent();
                if (queryBlock.getLimit() == null
                        && queryBlock.getFrom() instanceof SQLJoinTableSource
                        && isBothName(((SQLJoinTableSource) queryBlock.getFrom()).getCondition()))
                {
                    SQLJoinTableSource joinTableSource = (SQLJoinTableSource) queryBlock.getFrom();
                    SQLBinaryOpExpr joinCondition = (SQLBinaryOpExpr) joinTableSource.getCondition();

                    if (joinTableSource.getJoinType() == SQLJoinTableSource.JoinType.INNER_JOIN
                            && joinCondition.getOperator() == SQLBinaryOperator.Equality
                            && joinCondition.getLeft().equals(x.getLeft()))
                    {
                        SQLExpr to = SQLBinaryOpExpr.and(joinCondition.clone(),
                                x.clone(),
                                new SQLBinaryOpExpr(joinCondition.getRight().clone()
                                        , SQLBinaryOperator.Equality
                                        , x.getRight().clone()));
                        if (SQLUtils.replaceInParent(x, null)) {
                            joinTableSource.setCondition(to);
                            optimizedCount++;
                            return false;
                        }
                    }
                }
            }
        }

        return false;
    }

    private void mergeTwiceOp(SQLExpr x, SQLBinaryOpExpr left, SQLBinaryOpExpr right) {
        SQLBinaryOperator leftOp = left.getOperator();
        SQLBinaryOperator rightOp = right.getOperator();

        SQLExpr replaceTo = null;
        if (left.getLeft().equals(right.getLeft()) && left.getRight().equals(right.getRight())) {
            if (leftOp == SQLBinaryOperator.Equality
                    && (rightOp == SQLBinaryOperator.LessThanOrEqual || rightOp == SQLBinaryOperator.GreaterThanOrEqual)) {
                // B = 5 AND B >= 5 -> B = 5
                // B = 5 AND B <= 5 -> B = 5
                replaceTo = left;
            } else if (rightOp == SQLBinaryOperator.Equality
                    && (leftOp == SQLBinaryOperator.LessThanOrEqual || leftOp == SQLBinaryOperator.GreaterThanOrEqual)) {
                // B >= 5 AND B = 5 -> B = 5
                // B <= 5 AND B = 5 -> B = 5
                replaceTo = right;
            } else if ((leftOp == SQLBinaryOperator.LessThanOrEqual && rightOp == SQLBinaryOperator.GreaterThanOrEqual)
                    || (leftOp == SQLBinaryOperator.GreaterThanOrEqual && rightOp == SQLBinaryOperator.LessThanOrEqual)) {
                SQLBinaryOpExpr expr = new SQLBinaryOpExpr(left.getLeft().clone(), SQLBinaryOperator.Equality, left.getRight().clone());
                replaceTo = expr;
            }

        }

        if (replaceTo != null) {
            if (x instanceof SQLBinaryOpExprGroup) {
                SQLBinaryOpExprGroup group = (SQLBinaryOpExprGroup) x;
                if (group.getOperator() == SQLBinaryOperator.BooleanAnd) {
                    group.getItems().remove(left);
                    group.getItems().remove(right);
                    group.add(replaceTo);
                }

            } else {
                if (SQLUtils.replaceInParent(x, replaceTo)) {
                    optimizedCount++;
                }
            }
        }
    }

    private void handValueValue(SQLBinaryOpExpr x, SQLBinaryOperator op, SQLValuableExpr leftExpr, SQLValuableExpr rightExpr) {
        if (rightExpr instanceof SQLNullExpr) {
            if (op == SQLBinaryOperator.Is) {
                if (leftExpr instanceof SQLNullExpr) {
                    if (SQLUtils.replaceInParent(x, new SQLBooleanExpr(true))) {
                        optimizedCount++;
                    }
                } else {
                    if (SQLUtils.replaceInParent(x, new SQLBooleanExpr(false))) {
                        optimizedCount++;
                    }
                }
            } else if (op == SQLBinaryOperator.IsNot) {
                if (leftExpr instanceof SQLNullExpr) {
                    if (SQLUtils.replaceInParent(x, new SQLBooleanExpr(false))) {
                        optimizedCount++;
                    }
                } else {
                    if (SQLUtils.replaceInParent(x, new SQLBooleanExpr(true))) {
                        optimizedCount++;
                    }
                }
            }
        }

        Object leftVal = leftExpr.getValue();
        Object rightVal = rightExpr.getValue();

        if (leftVal instanceof Integer && rightVal instanceof Integer) {
            int leftIntVal = ((Integer) leftVal).intValue();
            int rightIntVal = ((Integer) rightVal).intValue();

            switch (op) {
                case Add:
                    // 3 + 4 -> 7
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(leftIntVal + rightIntVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Subtract:
                    // 4 - 3 -> 1
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(leftIntVal - rightIntVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Multiply:
                    // 4 * 3 -> 21
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(leftIntVal * rightIntVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Divide: {
                    // 4 * 3 -> 21
                    int result = leftIntVal / rightIntVal;
                    int mod = leftIntVal % rightIntVal;
                    if (mod == 0 && SQLUtils.replaceInParent(x, new SQLIntegerExpr(result))) {
                        optimizedCount++;
                        return;
                    }

                    break;
                }
                case Mod:
                case Modulus: {
                    int result = leftIntVal % rightIntVal;
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(result))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                }
                default:
                    break;
            }
        }

        if (leftVal instanceof Integer && rightVal instanceof Long) {
            leftVal = new Long(((Integer) leftVal).intValue());
        } else if (leftVal instanceof Long && rightVal instanceof Integer) {
            rightVal = new Long(((Integer) rightVal).intValue());
        }

        if (leftVal instanceof Long && rightVal instanceof Long) {
            long leftIntVal = ((Long) leftVal).longValue();
            long rightIntVal = ((Long) rightVal).longValue();

            switch (op) {
                case Add:
                    // 3 + 4 -> 7
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(leftIntVal + rightIntVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Subtract:
                    // 4 - 3 -> 1
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(leftIntVal - rightIntVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Multiply:
                    // 4 * 3 -> 21
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(leftIntVal * rightIntVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Divide: {
                    // 4 * 3 -> 21
                    long result = leftIntVal / rightIntVal;
                    long mod = leftIntVal % rightIntVal;
                    if (mod == 0 && SQLUtils.replaceInParent(x, new SQLIntegerExpr(result))) {
                        optimizedCount++;
                        return;
                    }

                    break;
                }
                case Mod:
                case Modulus: {
                    long result = leftIntVal % rightIntVal;
                    if (SQLUtils.replaceInParent(x, new SQLIntegerExpr(result))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                }
                default:
                    break;
            }
        }

        if (leftVal instanceof BigDecimal && (rightVal instanceof Long || rightVal instanceof Integer)) {
            rightVal = new BigDecimal(((Number) rightVal).longValue());
        } else if (rightExpr instanceof BigDecimal && (leftVal instanceof Long || leftVal instanceof Integer)) {
            leftVal = new BigDecimal(((Number) leftVal).longValue());
        }

        if (leftVal instanceof Double && rightVal instanceof Number) {
            rightVal = ((Number) rightVal).doubleValue();
        } else if (leftVal instanceof Float && rightVal instanceof Number) {
            leftVal = ((Float) leftVal).doubleValue();
            rightVal = ((Number) rightVal).doubleValue();
        }

        if (leftVal instanceof BigDecimal && rightVal instanceof BigDecimal) {
            BigDecimal leftRealVal = (BigDecimal) leftVal;
            BigDecimal rightRealVal = (BigDecimal) rightVal;

            switch (op) {
                case Add:
                    if (SQLUtils.replaceInParent(x, new SQLNumberExpr(leftRealVal.add(rightRealVal)))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Subtract:
                    if (SQLUtils.replaceInParent(x,
                                                 new SQLNumberExpr(leftRealVal.subtract(rightRealVal)))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Multiply:
                    if (SQLUtils.replaceInParent(x, new SQLNumberExpr(leftRealVal.multiply(rightRealVal)))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Divide: {
                    if (SQLUtils.replaceInParent(x, new SQLNumberExpr(leftRealVal.divide(rightRealVal)))) {
                        optimizedCount++;
                        return;
                    }

                    break;
                }
                default:
                    break;
            }
        } else if (leftVal instanceof Double && rightVal instanceof Double) {
            double leftDoubleVal = (Double) leftVal;
            double rightDoubleVal = (Double) rightVal;

            switch (op) {
                case Add:
                    if (SQLUtils.replaceInParent(x, new SQLNumberExpr(leftDoubleVal + rightDoubleVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Subtract:
                    if (SQLUtils.replaceInParent(x,
                            new SQLNumberExpr(leftDoubleVal - rightDoubleVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Multiply:
                    if (SQLUtils.replaceInParent(x, new SQLNumberExpr(leftDoubleVal * rightDoubleVal))) {
                        optimizedCount++;
                        return;
                    }
                    break;
                case Divide: {
                    if (SQLUtils.replaceInParent(x, new SQLNumberExpr(leftDoubleVal / rightDoubleVal))) {
                        optimizedCount++;
                        return;
                    }

                    break;
                }
                default:
                    break;
            }
        }

    }

    private void handleNameLiteral(SQLExpr x, SQLName name, SQLLiteralExpr literal) {
        SQLColumnDefinition column = null;
        if (name instanceof SQLIdentifierExpr) {
            column = ((SQLIdentifierExpr) name).getResolvedColumn();
        } else  if (name instanceof SQLPropertyExpr) {
            column = ((SQLPropertyExpr) name).getResolvedColumn();
        } else {
            return;
        }

        if (column != null && column.getDataType() != null) {
            SQLDataType dataType = column.getDataType();

            final long dataTypeNameHash = dataType.nameHashCode64();
            if (dataTypeNameHash == FnvHash.Constants.BIGINT
                    || dataTypeNameHash == FnvHash.Constants.INT
                    || dataTypeNameHash == FnvHash.Constants.SMALLINT
                    || dataTypeNameHash == FnvHash.Constants.TINYINT)
            {
                if (literal instanceof SQLCharExpr) {
                    String text = ((SQLCharExpr) literal).getText();
                    try {
                        // id = '123' -> id = 123
                        SQLIntegerExpr intExpr;
                        if (dataTypeNameHash == FnvHash.Constants.BIGINT) {
                            long val = Long.parseLong(text);
                            intExpr = new SQLIntegerExpr(val);
                        } else {
                            int val = Integer.parseInt(text);
                            intExpr = new SQLIntegerExpr(val);
                        }
                        if (SQLUtils.replaceInParent(literal, intExpr)) {
                            optimizedCount++;
                        }
                    } catch (NumberFormatException ex) {
                        // id = '123' -> false
                        if (SQLUtils.replaceInParent(x, new SQLBooleanExpr(false))) {
                            optimizedCount++;
                            return;
                        }
                    }
                } else if (literal instanceof SQLNumberExpr) {
                    SQLNumberExpr numberExpr = (SQLNumberExpr) literal;
                    Number number = numberExpr.getNumber();
                    if (number instanceof BigDecimal) {
                        BigDecimal decimal = (BigDecimal) number;

                        int index;
                        String strVal = decimal.toString();
                        if (decimal.doubleValue() == decimal.longValue()) {
                            BigInteger integer = decimal.toBigInteger();
                            numberExpr.setNumber(integer);
                        }
                    }
                }
            } else if (dataTypeNameHash == FnvHash.Constants.CHAR
                    || dataTypeNameHash == FnvHash.Constants.VARCHAR
                    || dataTypeNameHash == FnvHash.Constants.VARCHAR2
                    || dataTypeNameHash == FnvHash.Constants.CLOB) {
                if (literal instanceof SQLIntegerExpr) {
                    String text = literal.toString();
                    SQLCharExpr charExpr = new SQLCharExpr(text);
                    if (SQLUtils.replaceInParent(literal, charExpr)) {
                        optimizedCount++;
                    }
                }
            } else if (dataTypeNameHash == FnvHash.Constants.DATE) {
                if (literal instanceof SQLCharExpr) {
                    String text = ((SQLCharExpr) literal).getText();
                    SQLDateExpr dateExpr = new SQLDateExpr(text);
                    if (SQLUtils.replaceInParent(literal, dateExpr)) {
                        optimizedCount++;
                    }
                }
            } else if (dataTypeNameHash == FnvHash.Constants.TIMESTAMP) {
                if (literal instanceof SQLCharExpr) {
                    String text = ((SQLCharExpr) literal).getText();
                    SQLTimestampExpr timestampExpr = new SQLTimestampExpr(text);
                    if (SQLUtils.replaceInParent(literal, timestampExpr)) {
                        optimizedCount++;
                    }
                }
            } else if (dataTypeNameHash == FnvHash.Constants.TIME) {
                if (literal instanceof SQLCharExpr) {
                    String text = ((SQLCharExpr) literal).getText();
                    SQLTimeExpr timeExpr = new SQLTimeExpr(text);
                    if (SQLUtils.replaceInParent(literal, timeExpr)) {
                        optimizedCount++;
                    }
                }
            } else if ( dataTypeNameHash == FnvHash.Constants.DOUBLE) {
                if (literal instanceof SQLNumberExpr) {
                    double value = ((SQLNumberExpr) literal).getNumber().doubleValue();
                    SQLNumberExpr numberExpr = new SQLNumberExpr(value);
                    if (SQLUtils.replaceInParent(literal, numberExpr)) {
                        optimizedCount++;
                    }
                } else if (literal instanceof SQLCharExpr) {
                    String chars = ((SQLCharExpr) literal).getText();
                    double value = Double.parseDouble(chars);
                    SQLNumberExpr numberExpr = new SQLNumberExpr(value);
                    if (SQLUtils.replaceInParent(literal, numberExpr)) {
                        optimizedCount++;
                    }
                }
            } else if (dataTypeNameHash == FnvHash.Constants.FLOAT) {
                if (literal instanceof SQLNumberExpr) {
                    float value = ((SQLNumberExpr) literal).getNumber().floatValue();
                    SQLNumberExpr numberExpr = new SQLNumberExpr(value);
                    if (SQLUtils.replaceInParent(literal, numberExpr)) {
                        optimizedCount++;
                    }
                } else if (literal instanceof SQLCharExpr) {
                    String chars = ((SQLCharExpr) literal).getText();
                    float value = Float.parseFloat(chars);
                    SQLNumberExpr numberExpr = new SQLNumberExpr(value);
                    if (SQLUtils.replaceInParent(literal, numberExpr)) {
                        optimizedCount++;
                    }
                }
            } else if (dataTypeNameHash == FnvHash.Constants.DECIMAL) {
                if (literal instanceof SQLCharExpr) {
                    String chars = ((SQLCharExpr) literal).getText();
                    BigDecimal value = new BigDecimal(chars);
                    SQLNumberExpr numberExpr = new SQLNumberExpr(value);
                    if (SQLUtils.replaceInParent(literal, numberExpr)) {
                        optimizedCount++;
                    }
                }
            }
        }
    }

    public boolean visit(SQLBinaryOpExprGroup x) {
        if (x.getOperator() == SQLBinaryOperator.BooleanAnd) {
            Map<SQLExpr, SQLExpr> constFields = new HashMap<SQLExpr, SQLExpr>();

            Map<SQLName, List<SQLBinaryOpExpr>> nameConditions = new HashMap<SQLName, List<SQLBinaryOpExpr>>();
            for (int i = 0; i < x.getItems().size(); i++) {
                SQLExpr item = x.getItems().get(i);

                if (item instanceof SQLBinaryOpExpr) {
                    SQLBinaryOpExpr binaryOpExpr = (SQLBinaryOpExpr) item;
                    SQLExpr left = binaryOpExpr.getLeft();
                    SQLExpr right = binaryOpExpr.getRight();
                    if (binaryOpExpr.getOperator() == SQLBinaryOperator.Equality
                            && left instanceof SQLName
                            && right instanceof SQLValuableExpr)
                    {
                        constFields.put(left, right);
                    }

                    if (isLeftNameAndRightLiteral(item)) {
                        List<SQLBinaryOpExpr> exprList = nameConditions.get(left);
                        if (exprList == null) {
                            exprList = new ArrayList<SQLBinaryOpExpr>();
                            nameConditions.put((SQLName) left, exprList);
                        }
                        exprList.add((SQLBinaryOpExpr) item);
                    }
                }
            }

            for (SQLExpr item : x.getItems()) {
                if (item instanceof SQLBinaryOpExpr) {
                    SQLBinaryOpExpr binaryOpExpr = (SQLBinaryOpExpr) item;
                    SQLExpr left = binaryOpExpr.getLeft();
                    SQLExpr right = binaryOpExpr.getRight();
                    if (left instanceof SQLName && right instanceof SQLName) {
                        SQLExpr leftLiteral = constFields.get(left);
                        if (leftLiteral != null) {
                            binaryOpExpr.setLeft(leftLiteral.clone());
                            optimizedCount++;
                        }

                        SQLExpr rightLiteral = constFields.get(right);
                        if (rightLiteral != null) {
                            binaryOpExpr.setRight(rightLiteral.clone());
                            optimizedCount++;
                        }
                    }
                }
            }

            for (Map.Entry<SQLName, List<SQLBinaryOpExpr>> entry : nameConditions.entrySet()) {
                List<SQLBinaryOpExpr> exprList = entry.getValue();
                for (int i = 1; i < exprList.size(); i++) {
                    SQLBinaryOpExpr left = exprList.get(i - 1);
                    SQLBinaryOpExpr right = exprList.get(i);
                    mergeTwiceOp(x, left, right);
                }
            }
        }

        Collections.sort(x.getItems(), new Comparator<SQLExpr>() {

            @Override
            public int compare(SQLExpr a, SQLExpr b) {
                if (a instanceof SQLBinaryOpExpr && b instanceof SQLBinaryOpExpr) {
                    return compare((SQLBinaryOpExpr) a, (SQLBinaryOpExpr) b);
                }
                return 0;
            }

            public int compare(SQLBinaryOpExpr a, SQLBinaryOpExpr b) {
                if (isNameAndLiteral(a) && isName2(b)) {
                    return -1;
                }

                if (isNameAndLiteral(b) && isName2(a)) {
                    return 1;
                }

                if (isNameAndLiteral(a) && isNameAndLiteral(b)) {
                    SQLName aName = getName(a);
                    SQLName bName = getName(b);

                    SQLLiteralExpr aLiteral = getLiterlal(a);
                    SQLLiteralExpr bLiteral = getLiterlal(b);

                    if (aName.equals(bName) && aLiteral.getClass() == bLiteral.getClass()) {

                    }
                }

                return 0;
            }

            private SQLName getName(SQLBinaryOpExpr x) {
                SQLExpr left = x.getLeft();
                SQLExpr right = x.getRight();

                if (left instanceof SQLName) {
                    return (SQLName) left;
                }

                if (right instanceof SQLName) {
                    return (SQLName) right;
                }

                return null;
            }

            private SQLLiteralExpr getLiterlal(SQLBinaryOpExpr x) {
                SQLExpr left = x.getLeft();
                SQLExpr right = x.getRight();

                if (left instanceof SQLLiteralExpr) {
                    return (SQLLiteralExpr) left;
                }

                if (right instanceof SQLLiteralExpr) {
                    return (SQLLiteralExpr) right;
                }

                return null;
            }


        });

        for (SQLExpr item : x.getItems()) {
            item.accept(this);
        }
        return false;
    }

    public boolean visit(SQLInListExpr x) {
        SQLExpr expr = x.getExpr();
        List<SQLExpr> targetList = x.getTargetList();

        expr.accept(this);
        for (SQLExpr item : targetList) {
            item.accept(this);
        }

        if (expr instanceof SQLName) {
            SQLName name = (SQLName) expr;

            for (int i = 0; i < targetList.size(); i++) {
                SQLExpr item = targetList.get(i);
                if (item instanceof SQLLiteralExpr) {
                    SQLLiteralExpr literal = (SQLLiteralExpr) item;
                    handleNameLiteral(x, name, literal);
                }
            }
        }

        return false;
    }


    public boolean visit(SQLBetweenExpr x) {
        SQLExpr expr = x.getTestExpr();

        x.getTestExpr().accept(this);
        x.getBeginExpr().accept(this);
        x.getEndExpr().accept(this);

        if (expr instanceof SQLName) {
            SQLName name = (SQLName) expr;

            if (x.getBeginExpr() instanceof SQLLiteralExpr) {
                handleNameLiteral(x, name, (SQLLiteralExpr) x.getBeginExpr());
            }

            if (x.getEndExpr() instanceof SQLLiteralExpr) {
                handleNameLiteral(x, name, (SQLLiteralExpr) x.getEndExpr());
            }
        }
        return false;
    }

    private boolean isBothName(SQLExpr x) {
        if (x instanceof SQLBinaryOpExpr) {
            SQLBinaryOpExpr binaryOpExpr = (SQLBinaryOpExpr) x;
            return binaryOpExpr.getLeft() instanceof SQLName
                    && binaryOpExpr.getRight() instanceof SQLName;
        }
        return false;
    }

    private boolean isLeftNameAndRightLiteral(SQLExpr x) {
        if (x instanceof SQLBinaryOpExpr) {
            return ((SQLBinaryOpExpr) x).isLeftNameAndRightLiteral();
        }
        return false;
    }

    private boolean isLeftNameAndRightLiteral(SQLBinaryOpExpr x) {
        if (x.getLeft() instanceof SQLName && x.getRight() instanceof SQLLiteralExpr) {
            return true;
        }

        return false;
    }

    private boolean isLeftFunctionAndRightLiteral(SQLBinaryOpExpr x) {
        if (x.getLeft() instanceof SQLMethodInvokeExpr && x.getRight() instanceof SQLLiteralExpr) {
            return true;
        }

        return false;
    }

    private boolean isLeftLiteralAndRightName(SQLBinaryOpExpr x) {
        if (x.getRight() instanceof SQLName && x.getLeft() instanceof SQLLiteralExpr) {
            return true;
        }

        return false;
    }

    private boolean isNameAndLiteral(SQLExpr x) {
        if (x instanceof SQLBinaryOpExpr) {
            return isNameAndLiteral((SQLBinaryOpExpr) x);
        }

        return false;
    }

    private boolean isNameAndLiteral(SQLBinaryOpExpr x) {
        SQLExpr left = x.getLeft();
        SQLExpr right = x.getRight();

        if (left instanceof SQLLiteralExpr && right instanceof SQLName) {
            return true;
        }

        if (left instanceof SQLName && right instanceof SQLLiteralExpr) {
            return true;
        }

        return false;
    }

    private boolean isName2(SQLBinaryOpExpr x) {
        return x.getLeft() instanceof SQLName && x.getRight() instanceof SQLName;
    }

    public int getOptimizedCount() {
        return optimizedCount;
    }
}
