package com.alibaba.fastsql.sql.optimizer.visitor;

import com.alibaba.fastsql.DbType;
import com.alibaba.fastsql.sql.SQLUtils;
import com.alibaba.fastsql.sql.ast.*;
import com.alibaba.fastsql.sql.ast.expr.*;
import com.alibaba.fastsql.sql.ast.statement.*;
import com.alibaba.fastsql.sql.repository.SchemaRepository;
import com.alibaba.fastsql.sql.visitor.SQLASTVisitorAdapter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

public class PushUp extends SQLASTVisitorAdapter {
    private int optimizedCount = 0;

    public boolean visit(SQLSubqueryTableSource x) {
        SQLSelect select = x.getSelect();

        SQLSelectQueryBlock parentQuery = null;
        {
            SQLObject parent = x.getParent();
            while (parent instanceof SQLJoinTableSource) {
                parent = parent.getParent();
            }
            if (parent instanceof SQLSelectQueryBlock) {
                parentQuery = (SQLSelectQueryBlock) parent;
                for (SQLSelectItem selectItem : parentQuery.getSelectList()) {
                    SQLExpr expr = selectItem.getExpr();
                    if (expr instanceof SQLAllColumnExpr) {
                        return super.visit(x);
                    } else if (expr instanceof SQLPropertyExpr && ((SQLPropertyExpr) expr).getSimpleName().equalsIgnoreCase("*")) {
                        return super.visit(x);
                    } else if (expr instanceof SQLAggregateExpr) {
                        SQLAggregateExpr agg = (SQLAggregateExpr) expr;
                        for (SQLExpr arg : agg.getArguments()) {
                            if (arg instanceof SQLAllColumnExpr) {
                                return super.visit(x);
                            } else if (arg instanceof SQLPropertyExpr && ((SQLPropertyExpr) arg).getSimpleName().equalsIgnoreCase("*")) {
                                return super.visit(x);
                            }
                        }
                    }
                }
            }
        }

        SQLObject parent = x.getParent();
        SQLSelectQuery query = select.getQuery();

        query.accept(this);
        final String alias = x.getAlias();

        if (query instanceof SQLSelectQueryBlock) {
            SQLSelectQueryBlock queryBlock = (SQLSelectQueryBlock) query;
            SQLTableSource clonedFrom = queryBlock.getFrom().clone();

            SQLExpr where = queryBlock.getWhere();
            if (alias != null && where != null) {
                where = where.clone();

                final SQLExpr fromTable;
                if (clonedFrom instanceof SQLExprTableSource) {
                    fromTable = ((SQLExprTableSource) clonedFrom).getExpr();
                } else {
                    fromTable = null;
                }

                SQLASTVisitorAdapter visitor = new SQLASTVisitorAdapter() {
                    public boolean visit(SQLIdentifierExpr x) {
                        if (x.equals(fromTable)) {
                            SQLUtils.replaceInParent(x, new SQLIdentifierExpr(alias));
                            return false;
                        }

                        if (x.getParent() instanceof SQLPropertyExpr) {
                            return false;
                        }
                        SQLUtils.replaceInParent(x, new SQLPropertyExpr(alias, x.getName()));
                        return false;
                    }

                    public boolean visit(SQLPropertyExpr x) {
                        if (x.equals(fromTable)) {
                            SQLUtils.replaceInParent(x, new SQLIdentifierExpr(alias));
                            return false;
                        }

                        return true;
                    }
                };
                where.accept(visitor);
            }

            if (isSimpleQueryBlock(queryBlock)) {

                final AtomicBoolean aliasUsed = new AtomicBoolean();

                if (clonedFrom instanceof SQLJoinTableSource && x.getAlias() != null) {
                    SQLASTVisitorAdapter visitor = new SQLASTVisitorAdapter() {
                        public boolean visit(SQLIdentifierExpr x) {
                            if (x.getName().equals(alias) && x.getParent() instanceof SQLPropertyExpr) {
                                aliasUsed.set(true);
                            }
                            return false;
                        }
                    };
                    parentQuery.accept(visitor);
                    if (aliasUsed.get()) {
                        return super.visit(x);
                    }
                }

                String xAlias = x.getAlias();
                if (!(clonedFrom instanceof SQLJoinTableSource)) {
                    clonedFrom.setAlias(xAlias);
                }
                if (parent instanceof SQLJoinTableSource) {
                    SQLJoinTableSource join = (SQLJoinTableSource) parent;
                    if (join.replace(x, clonedFrom)) {
                        optimizedCount ++;
                        join.addCondition(where);

                        if (join.getJoinType() == SQLJoinTableSource.JoinType.INNER_JOIN
                                && join.getLeft() instanceof SQLExprTableSource
                                && join.getRight() instanceof SQLJoinTableSource) {
                            SQLTableSource tmp = join.getLeft();
                            join.setLeft(join.getRight());
                            join.setRight(tmp);
                        }
                    }
                } else if (parent instanceof SQLSelectQueryBlock) {
                    SQLSelectQueryBlock parentQueryBlock = (SQLSelectQueryBlock) parent;

                    boolean allMatch = true;
                    if (parentQueryBlock.getSelectList().size() == queryBlock.getSelectList().size()) {
                        for (int i = 0; i < parentQueryBlock.getSelectList().size(); i++) {
                            SQLSelectItem pItem = parentQueryBlock.getSelectList().get(i);
                            SQLExpr pExpr = pItem.getExpr();
                            SQLSelectItem item = queryBlock.getSelectList().get(i);

                            if (pExpr instanceof SQLCastExpr) {
                                pExpr = ((SQLCastExpr) pExpr).getExpr();
                            }

                            if (pItem.equals(item)) {

                            } else if (pExpr instanceof SQLIdentifierExpr
                                    && ((SQLIdentifierExpr) pExpr).getName().equals(item.computeAlias())) {

                            } else {
                                allMatch = false;
                                break;
                            }
                        }
                    } else {
                        allMatch = false;
                    }

                    if (allMatch && parentQuery.getFrom() == x) {
                        parentQuery.setFrom(clonedFrom);
                        optimizedCount ++;
                        replaceAlas(queryBlock, parentQuery, alias);
                        parentQuery.addCondition(where);
                    }
                } else {
                    return super.visit(x);
                }
            }
        }

        return super.visit(x);
    }

    private void replaceAlas(SQLSelectQueryBlock queryBlock, SQLSelectQueryBlock parent, final String tabSrcAlias) {
        final Map<Long, SQLExpr> map = new HashMap<Long, SQLExpr>();
        for (SQLSelectItem selectItem : queryBlock.getSelectList()) {
            String alias = selectItem.getAlias();
            SQLExpr expr = selectItem.getExpr().clone();

            boolean changed = false;
            if (alias != null) {
                if (expr instanceof SQLName) {
                    if (!alias.equals(expr.toString())) {
                        changed = true;
                    }
                } else {
                    changed = true;
                }
            }
            if (changed) {
                SQLASTVisitorAdapter visitor = new SQLASTVisitorAdapter() {
                    public boolean visit(SQLIdentifierExpr x) {
                        SQLUtils.replaceInParent(x, new SQLPropertyExpr(tabSrcAlias, x.getName()));
                        return false;
                    }
                };
                expr.accept(visitor);
                map.put(selectItem.alias_hash(), expr);
            }
        }
        if (map.size() == 0) {
            return;
        }

        SQLASTVisitorAdapter visitor = new SQLASTVisitorAdapter() {
            public boolean visit(SQLIdentifierExpr x) {
                Long hash = x.nameHashCode64();
                SQLExpr expr = map.get(hash);
                if (expr != null) {
                    SQLUtils.replaceInParent(x, expr.clone());
                }
                return false;
            }
        };
        parent.accept(visitor);
    }

    private static boolean isSimpleQueryBlock(SQLSelectQueryBlock x) {
        if (x.getDistionOption() != 0) {
            return false;
        }

        if (x.getLimit() != null || x.getOrderBy() != null || x.getGroupBy() != null) {
            return false;
        }

        for (SQLSelectItem selectItem : x.getSelectList()) {
            SQLExpr expr = selectItem.getExpr();

            if (expr instanceof SQLCastExpr) {
                expr = ((SQLCastExpr) expr).getExpr();
            }

            if (!(expr instanceof SQLName)) {
                return false;
            }
        }

        return isSimple(x.getFrom());
    }

    private static boolean isSimple(SQLTableSource tableSource) {
        if ((tableSource instanceof SQLExprTableSource && ((SQLExprTableSource) tableSource).getExpr() instanceof SQLName)) {
            return true;
        }

        if (tableSource instanceof SQLJoinTableSource) {
            SQLJoinTableSource join = (SQLJoinTableSource) tableSource;
            return isSimple(join.getLeft())
                    && isSimple(join.getRight());
        }

        return false;
    }

    public static String optimize(String sql, DbType dbType) {
        List<SQLStatement> stmtList = SQLUtils.parseStatements(sql, dbType);

        PushUp visitor = new PushUp();
        for (int i = 0; i < stmtList.size(); i++) {
            SQLStatement stmt = stmtList.get(i);
            stmt.accept(visitor);
        }
        return SQLUtils.toSQLString(stmtList, dbType);
    }
}
