/*
 * Copyright 1999-2017 Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.fastsql.sql.parser;

import com.alibaba.fastsql.DbType;
import com.alibaba.fastsql.sql.ast.statement.SQLSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.ads.parser.AdsStatementParser;
import com.alibaba.fastsql.sql.dialect.db2.ast.stmt.DB2SelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.db2.parser.DB2ExprParser;
import com.alibaba.fastsql.sql.dialect.db2.parser.DB2Lexer;
import com.alibaba.fastsql.sql.dialect.db2.parser.DB2StatementParser;
import com.alibaba.fastsql.sql.dialect.h2.parser.H2ExprParser;
import com.alibaba.fastsql.sql.dialect.h2.parser.H2Lexer;
import com.alibaba.fastsql.sql.dialect.h2.parser.H2StatementParser;
import com.alibaba.fastsql.sql.dialect.hive.parser.HiveExprParser;
import com.alibaba.fastsql.sql.dialect.hive.parser.HiveStatementParser;
import com.alibaba.fastsql.sql.dialect.mysql.ast.statement.MySqlSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.mysql.parser.MySqlExprParser;
import com.alibaba.fastsql.sql.dialect.mysql.parser.MySqlLexer;
import com.alibaba.fastsql.sql.dialect.mysql.parser.MySqlStatementParser;
import com.alibaba.fastsql.sql.dialect.odps.ast.OdpsSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.odps.parser.OdpsExprParser;
import com.alibaba.fastsql.sql.dialect.odps.parser.OdpsLexer;
import com.alibaba.fastsql.sql.dialect.odps.parser.OdpsStatementParser;
import com.alibaba.fastsql.sql.dialect.oracle.ast.stmt.OracleSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.oracle.parser.OracleExprParser;
import com.alibaba.fastsql.sql.dialect.oracle.parser.OracleLexer;
import com.alibaba.fastsql.sql.dialect.oracle.parser.OracleStatementParser;
import com.alibaba.fastsql.sql.dialect.phoenix.parser.PhoenixExprParser;
import com.alibaba.fastsql.sql.dialect.phoenix.parser.PhoenixLexer;
import com.alibaba.fastsql.sql.dialect.phoenix.parser.PhoenixStatementParser;
import com.alibaba.fastsql.sql.dialect.postgresql.ast.stmt.PGSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.postgresql.parser.PGExprParser;
import com.alibaba.fastsql.sql.dialect.postgresql.parser.PGLexer;
import com.alibaba.fastsql.sql.dialect.postgresql.parser.PGSQLStatementParser;
import com.alibaba.fastsql.sql.dialect.presto.parser.PrestoExprParser;
import com.alibaba.fastsql.sql.dialect.presto.parser.PrestoLexer;
import com.alibaba.fastsql.sql.dialect.presto.parser.PrestoStatementParser;
import com.alibaba.fastsql.sql.dialect.sqlserver.ast.SQLServerSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.sqlserver.parser.SQLServerExprParser;
import com.alibaba.fastsql.sql.dialect.sqlserver.parser.SQLServerStatementParser;

public class SQLParserUtils {

    public static SQLStatementParser createSQLStatementParser(String sql, DbType dbType) {
        SQLParserFeature[] features;
        if (DbType.odps == dbType || DbType.mysql == dbType) {
            features = new SQLParserFeature[] {SQLParserFeature.KeepComments};
        } else {
            features = new SQLParserFeature[] {};
        }
        return createSQLStatementParser(sql, dbType, features);
    }

    public static SQLStatementParser createSQLStatementParser(String sql, DbType dbType, boolean keepComments) {
        SQLParserFeature[] features;
        if (keepComments) {
            features = new SQLParserFeature[] {SQLParserFeature.KeepComments};
        } else {
            features = new SQLParserFeature[] {};
        }

        return createSQLStatementParser(sql, dbType, features);
    }

    public static SQLStatementParser createSQLStatementParser(String sql, String dbType, SQLParserFeature... features) {
        return createSQLStatementParser(sql, dbType == null ? null : DbType.valueOf(dbType), features);
    }

    public static SQLStatementParser createSQLStatementParser(String sql, DbType dbType, SQLParserFeature... features) {
        if (dbType == null) {
            dbType = DbType.other;
        }

        switch (dbType) {
            case oracle:
                return new OracleStatementParser(sql, features);
            case mysql:
            case mariadb:
            case elastic_search:
                return new MySqlStatementParser(sql, features);
            case postgresql:
            case edb:
                return new PGSQLStatementParser(sql, features);
            case sqlserver:
            case jtds:
                return new SQLServerStatementParser(sql);
            case h2:
                return new H2StatementParser(sql, features);
            case db2:
                return new DB2StatementParser(sql, features);
            case odps:
                return new OdpsStatementParser(sql, features);
            case phoenix:
                return new PhoenixStatementParser(sql);
            case hive:
                return new HiveStatementParser(sql, features);
            case presto:
                return new PrestoStatementParser(sql);
            case ads:
                return new AdsStatementParser(sql);
            default:
                return new SQLStatementParser(sql, dbType);
        }
    }

    public static SQLExprParser createExprParser(String sql, DbType dbType, SQLParserFeature... features) {
        if (dbType == null) {
            dbType = DbType.other;
        }

        switch (dbType) {
            case oracle:
                return new OracleExprParser(sql, features);
            case mysql:
            case mariadb:
                return new MySqlExprParser(sql, features);
            case h2:
                return new H2ExprParser(sql, features);
            case postgresql:
            case edb:
                return new PGExprParser(sql, features);
            case sqlserver:
            case jtds:
                return new SQLServerExprParser(sql, features);
            case db2:
                return new DB2ExprParser(sql, features);
            case odps:
                return new OdpsExprParser(sql, features);
            case phoenix:
                return new PhoenixExprParser(sql, features);
            case presto:
                return new PrestoExprParser(sql, features);
            case hive:
                return new HiveExprParser(sql, features);
            default:
                return new SQLExprParser(sql, dbType, features);
        }
    }

    public static Lexer createLexer(String sql, DbType dbType) {
        return createLexer(sql, dbType, new SQLParserFeature[0]);
    }

    public static Lexer createLexer(String sql, DbType dbType, SQLParserFeature... features) {
        if (dbType == null) {
            dbType = DbType.other;
        }

        switch (dbType) {
            case oracle:
                return new OracleLexer(sql);
            case mysql:
            case mariadb:
                return new MySqlLexer(sql);
            case h2:
                return new H2Lexer(sql);
            case postgresql:
            case edb:
                return new PGLexer(sql);
            case db2:
                return new DB2Lexer(sql);
            case odps:
                return new OdpsLexer(sql);
            case phoenix:
                return new PhoenixLexer(sql);
            case presto:
                return new PrestoLexer(sql);
            default:
                return new Lexer(sql, null, dbType);
        }
    }

    public static SQLSelectQueryBlock createSelectQueryBlock(DbType dbType) {
        if (dbType == null) {
            dbType = DbType.other;
        }

        switch (dbType) {
            case mysql:
                return new MySqlSelectQueryBlock();
            case oracle:
                return new OracleSelectQueryBlock();
            case db2:
                return new DB2SelectQueryBlock();
            case postgresql:
                return new PGSelectQueryBlock();
            case odps:
                return new OdpsSelectQueryBlock();
            case sqlserver:
                return new SQLServerSelectQueryBlock();
            default:
                return new SQLSelectQueryBlock();
        }
     }

     public static SQLType getSQLType(String sql, DbType dbType) {
        Lexer lexer = createLexer(sql, dbType);
        return lexer.scanSQLType();
     }
}
