package com.alibaba.fastsql.support.calcite;

import com.alibaba.fastsql.FastsqlException;
import com.alibaba.fastsql.sql.ast.*;
import com.alibaba.fastsql.sql.ast.expr.*;
import com.alibaba.fastsql.sql.ast.statement.*;
import com.alibaba.fastsql.sql.dialect.mysql.ast.statement.MySqlDeleteStatement;
import com.alibaba.fastsql.sql.dialect.mysql.ast.statement.MySqlInsertStatement;
import com.alibaba.fastsql.sql.dialect.mysql.ast.statement.MySqlSelectQueryBlock;
import com.alibaba.fastsql.sql.dialect.mysql.ast.statement.MySqlUpdateStatement;
import com.alibaba.fastsql.sql.dialect.mysql.visitor.MySqlASTVisitorAdapter;
import com.alibaba.fastsql.util.FnvHash;
import com.alibaba.fastsql.util.StringUtils;
import org.apache.calcite.sql.*;
import org.apache.calcite.sql.fun.SqlQuantifyOperator;
import org.apache.calcite.sql.fun.SqlRowOperator;
import org.apache.calcite.sql.fun.SqlStdOperatorTable;
import org.apache.calcite.sql.parser.SqlParserPos;
import org.apache.calcite.sql.parser.SqlParserUtil;

import java.util.*;

public class CalciteMySqlNodeVisitor extends MySqlASTVisitorAdapter {

    private SqlNode sqlNode;

    public SqlNode getSqlNode() {
        return sqlNode;
    }

    public boolean visit(MySqlInsertStatement x) {
        SqlNodeList keywords = new SqlNodeList(new ArrayList<SqlNode>(), SqlParserPos.ZERO);

        SQLExprTableSource tableSource = (SQLExprTableSource) x.getTableSource();
        SqlNode targetTable = convertToSqlNode(tableSource.getExpr());

        List<SQLInsertStatement.ValuesClause> valuesList = x.getValuesList();

        SqlNode[] rows = new SqlNode[valuesList.size()];
        for (int j = 0; j < valuesList.size(); j++) {

            List<SQLExpr> values = valuesList.get(j).getValues();

            SqlNode[] valueNodes = new SqlNode[values.size()];
            for (int i = 0; i < values.size(); i++) {
                SqlNode valueNode = convertToSqlNode(values.get(i));
                valueNodes[i] = valueNode;
            }
            SqlBasicCall row = new SqlBasicCall(SqlStdOperatorTable.ROW, valueNodes, SqlParserPos.ZERO);
            rows[j] = row;
        }

        SqlNode source = new SqlBasicCall(SqlStdOperatorTable.VALUES, rows, SqlParserPos.ZERO);

        SqlNodeList columnList = convertToSqlNodeList(x.getColumns());

        this.sqlNode = new SqlInsert(SqlParserPos.ZERO, keywords, targetTable, source, columnList);
        return false;
    }

    public boolean visit(MySqlUpdateStatement x) {

        if(x.getTableSource().getClass() != SQLExprTableSource.class) {
            throw new UnsupportedOperationException("Support single table only for SqlUpdate statement of calcite.");
        }
        SQLExprTableSource tableSource = (SQLExprTableSource) x.getTableSource();
        SqlNode targetTable = convertToSqlNode(tableSource.getExpr());

        List<SqlNode> columns = new ArrayList<SqlNode>();
        List<SqlNode> values = new ArrayList<SqlNode>();

        for (SQLUpdateSetItem item : x.getItems()) {
            columns.add(convertToSqlNode(item.getColumn()));
            values.add(convertToSqlNode(item.getValue()));
        }
        SqlNodeList targetColumnList = new SqlNodeList(columns, SqlParserPos.ZERO);
        SqlNodeList sourceExpressList = new SqlNodeList(values, SqlParserPos.ZERO);

        SqlNode condition = convertToSqlNode(x.getWhere());


        SqlIdentifier alias = null;
        if(x.getTableSource().getAlias() != null) {
            alias = new SqlIdentifier(tableSource.getAlias(), SqlParserPos.ZERO);
        }

        sqlNode = new SqlUpdate(SqlParserPos.ZERO, targetTable, targetColumnList, sourceExpressList, condition, null, alias);

        return false;
    }

    public boolean visit(MySqlDeleteStatement x) {

        SQLExprTableSource tableSource = (SQLExprTableSource) x.getTableSource();
        SqlNode targetTable = convertToSqlNode(tableSource.getExpr());

        SqlNode condition = convertToSqlNode(x.getWhere());


        SqlIdentifier alias = null;
        if(x.getTableSource().getAlias() != null) {
            alias = new SqlIdentifier(tableSource.getAlias(), SqlParserPos.ZERO);
        }

        sqlNode = new SqlDelete(SqlParserPos.ZERO, targetTable, condition, null, alias);

        return false;
    }

    @Override
    public boolean visit(SQLUnionQuery x) {
        SqlNode left = convertToSqlNode(x.getLeft());
        SqlNode right = convertToSqlNode(x.getRight());

        //order by
        SqlNodeList orderBySqlNode = null;
        SQLOrderBy orderBy = x.getOrderBy();
        if (orderBy != null) {
            orderBySqlNode = convertOrderby(orderBy);
        }

        //limit
        SqlNode offset = null;
        SqlNode fetch = null;
        SQLLimit limit = x.getLimit();
        if (limit != null) {
            offset = convertToSqlNode(limit.getOffset());
            fetch = convertToSqlNode(limit.getRowCount());
        }

        SQLUnionOperator operator = x.getOperator();


        SqlNode union= null;
        switch (operator) {
            case UNION_ALL:
                union = new SqlBasicCall(SqlStdOperatorTable.UNION_ALL,
                                         new SqlNode[] { left, right },
                                         SqlParserPos.ZERO);
                break;
            case UNION:
            case DISTINCT:
                union = new SqlBasicCall(SqlStdOperatorTable.UNION,
                                         new SqlNode[] { left, right },
                                         SqlParserPos.ZERO);
                break;
            default:
                throw new FastsqlException("unsupported join type: "+ operator);
        }

        if (null == orderBy && null == offset && null == fetch) {
            sqlNode = union;
        } else {
            sqlNode = new SqlOrderBy(SqlParserPos.ZERO, union, orderBySqlNode, offset, fetch);
        }

        return false;

    }

    public boolean visit(MySqlSelectQueryBlock x) {
        SqlNodeList keywordList = null;
        List<SqlNode> keywordNodes = new ArrayList<SqlNode>(5);
        int option = x.getDistionOption();
        if (option != 0) {
            if (option == SQLSetQuantifier.DISTINCT
                    || option == SQLSetQuantifier.DISTINCTROW) {
                keywordNodes.add(SqlSelectKeyword.DISTINCT.symbol(SqlParserPos.ZERO));
            }
            keywordList = new SqlNodeList(keywordNodes, SqlParserPos.ZERO);
        }

        // select list
        List<SqlNode> columnNodes = new ArrayList<SqlNode>(x.getSelectList().size());
        for (SQLSelectItem selectItem : x.getSelectList()) {
            SqlNode column = convertToSqlNode(selectItem);
            if (selectItem.getAlias() != null) {
                column = new SqlBasicCall(SqlStdOperatorTable.AS,
                                          new SqlNode[] { column, new SqlIdentifier(selectItem.getAlias(),
                                                                                    SqlParserPos.ZERO) },
                                          SqlParserPos.ZERO);
            } else if (selectItem.getExpr() instanceof SQLAggregateExpr) {
                // skip
            } else if (selectItem.getExpr() instanceof SQLMethodInvokeExpr) {
                column = new SqlBasicCall(SqlStdOperatorTable.AS,
                                          new SqlNode[] { column, new SqlIdentifier(((SQLMethodInvokeExpr) selectItem.getExpr()).getMethodName(), SqlParserPos.ZERO) },
                                          SqlParserPos.ZERO);
            }
            columnNodes.add(column);
        }

        //select item
        SqlNodeList selectList = new SqlNodeList(columnNodes, SqlParserPos.ZERO);

        //from
        SqlNode from = null;

        SQLTableSource tableSource = x.getFrom();
        if (tableSource != null) {
            from = convertToSqlNode(tableSource);
        }

        //where
        SqlNode where = convertToSqlNode(x.getWhere());

        //order by
        SqlNodeList orderBySqlNode = null;
        SQLOrderBy orderBy = x.getOrderBy();
        if (orderBy != null) {
            orderBySqlNode = convertOrderby(orderBy);
        }

        //group by
        SqlNodeList groupBySqlNode = null;
        SqlNode having = null;
        SQLSelectGroupByClause groupBys = x.getGroupBy();

        if (groupBys != null) {
            List<SqlNode> groupByNodes = new ArrayList<SqlNode>(groupBys.getItems().size());

            if(groupBys.getHaving() != null) {
                having = convertToSqlNode(groupBys.getHaving());
            }

            for (SQLExpr groupBy : groupBys.getItems()) {
                SqlNode groupByNode = convertToSqlNode(groupBy);
                groupByNodes.add(groupByNode);
            }
            groupBySqlNode = new SqlNodeList(groupByNodes, SqlParserPos.ZERO);
        }

        //limit
        SqlNode offset = null;
        SqlNode fetch = null;
        SQLLimit limit = x.getLimit();
        if (limit != null) {
            offset = convertToSqlNode(limit.getOffset());
            fetch = convertToSqlNode(limit.getRowCount());
        }

        //hints
        SqlNodeList hints = convertHints(x.getHints());

        this.sqlNode = new TDDLSqlSelect(SqlParserPos.ZERO, keywordList, selectList, from, where,
                                                        groupBySqlNode, having, null, orderBySqlNode, offset, fetch,
                                                        hints, null);


        return false;
    }

    public boolean visit(SQLTableSource x) {
        Class<?> clazz = x.getClass();
        if (clazz == SQLJoinTableSource.class) {
            visit((SQLJoinTableSource) x);
        } else  if (clazz == SQLExprTableSource.class) {
            visit((SQLExprTableSource) x);
        } else  if (clazz == SQLSubqueryTableSource.class) {
            visit((SQLSubqueryTableSource) x);
        } else {
            x.accept(this);
        }

        return false;
    }

    @Override
    public boolean visit(SQLExprTableSource x) {
        String name = x.getName().getSimpleName();
        List<String> names = Collections.singletonList(name);

        SqlIdentifier table = new SqlIdentifier(names, SqlParserPos.ZERO);
        if (x.getAlias() != null) {
            SqlIdentifier alias = new SqlIdentifier(x.computeAlias(), SqlParserPos.ZERO);
            SqlBasicCall as = new SqlBasicCall(SqlStdOperatorTable.AS, new SqlNode[] { table, alias },
                                               SqlParserPos.ZERO);
            sqlNode = as;
        } else {
            sqlNode = table;
        }

        return false;
    }

    @Override
    public boolean visit(SQLJoinTableSource x) {
        SQLJoinTableSource.JoinType joinType = x.getJoinType();

        SqlNode left = convertToSqlNode(x.getLeft());
        SqlNode right = convertToSqlNode(x.getRight());
        SqlNode condition = convertToSqlNode(x.getCondition());

        switch (joinType) {
            case COMMA:
                this.sqlNode = new SqlJoin(SqlParserPos.ZERO, left,
                                           SqlLiteral.createBoolean(false, SqlParserPos.ZERO),
                                           JoinType.COMMA.symbol(SqlParserPos.ZERO), right,
                                           JoinConditionType.NONE.symbol(SqlParserPos.ZERO),
                                           null);
                break;
            case JOIN:
            case INNER_JOIN:
                if (x.getCondition() == null) {
                    this.sqlNode = new SqlJoin(SqlParserPos.ZERO, left,
                                               SqlLiteral.createBoolean(false, SqlParserPos.ZERO),
                                               JoinType.COMMA.symbol(SqlParserPos.ZERO), right,
                                               JoinConditionType.NONE.symbol(SqlParserPos.ZERO),
                                               null);
                } else {
                    this.sqlNode = new SqlJoin(SqlParserPos.ZERO, left,
                                               SqlLiteral.createBoolean(false, SqlParserPos.ZERO),
                                               JoinType.INNER.symbol(SqlParserPos.ZERO), right,
                                               JoinConditionType.ON.symbol(SqlParserPos.ZERO),
                                               condition);
                }
                break;
            case LEFT_OUTER_JOIN:
                this.sqlNode = new SqlJoin(SqlParserPos.ZERO,
                                           left,
                                           SqlLiteral.createBoolean(false, SqlParserPos.ZERO),
                                           JoinType.LEFT.symbol(SqlParserPos.ZERO),
                                           right,
                                           JoinConditionType.ON.symbol(SqlParserPos.ZERO),
                                           condition);
                break;
            case RIGHT_OUTER_JOIN:
                this.sqlNode = new SqlJoin(SqlParserPos.ZERO,
                                           left,
                                           SqlLiteral.createBoolean(false, SqlParserPos.ZERO),
                                           JoinType.RIGHT.symbol(SqlParserPos.ZERO),
                                           right,
                                           JoinConditionType.ON.symbol(SqlParserPos.ZERO),
                                           condition);
                break;
            case NATURAL_JOIN:
                this.sqlNode = new SqlJoin(SqlParserPos.ZERO,
                                           left,
                                           SqlLiteral.createBoolean(true, SqlParserPos.ZERO),
                                           JoinType.COMMA.symbol(SqlParserPos.ZERO),
                                           right,
                                           JoinConditionType.NONE.symbol(SqlParserPos.ZERO),
                                           null);
                break;
            default:
                throw new UnsupportedOperationException("unsupported : " + joinType);
        }

        return false;
    }

    @Override
    public boolean visit(SQLSubqueryTableSource x) {
        sqlNode = convertToSqlNode(x.getSelect());

        if (x.getAlias() != null) {
            SqlIdentifier aliasIdentifier = new SqlIdentifier(x.getAlias(), SqlParserPos.ZERO);
            sqlNode = new SqlBasicCall(SqlStdOperatorTable.AS,
                                       new SqlNode[] { sqlNode, aliasIdentifier },
                                       SqlParserPos.ZERO);
        }

        return false;
    }

    @Override public boolean visit(SQLInSubQueryExpr x) {
        SqlNode left = convertToSqlNode(x.getExpr());
        SqlNode right = convertToSqlNode(x.subQuery);

        sqlNode = new SqlBasicCall(SqlStdOperatorTable.IN, new SqlNode[]{left, right}, SqlParserPos.ZERO);
        return false;
    }

    @Override
    public boolean visit(SQLSelect x) {
        SQLWithSubqueryClause withSubQuery = x.getWithSubQuery();
        if (withSubQuery != null) {
            withSubQuery.accept(this);
        }

        sqlNode = convertToSqlNode(x.getQuery());

        return false;
    }

    @Override
    public boolean visit(SQLSelectStatement x) {

        SqlNode sqlNode = convertToSqlNode(x.getSelect());

        if(sqlNode instanceof TDDLSqlSelect) {
            TDDLSqlSelect select = (TDDLSqlSelect) sqlNode;

            SqlNodeList headHints = convertHints(x.getHeadHintsDirect());
            select.setHeadHints(headHints);
            this.sqlNode = select;
        } else {
            this.sqlNode = sqlNode;
        }

        return false;
    }

    protected void visit(SQLSelectQuery x) {
        Class<?> clazz = x.getClass();
        if (clazz == MySqlSelectQueryBlock.class) {
            visit((MySqlSelectQueryBlock) x);
        } else if (clazz == SQLUnionQuery.class) {
            visit((SQLUnionQuery) x);
        } else {
            x.accept(this);
        }
    }

    public boolean visit(SQLAllExpr x) {
        sqlNode = convertToSqlNode(x.getSubQuery());
        return false;
    }

    public boolean visit(SQLAnyExpr x) {
        sqlNode = convertToSqlNode(x.getSubQuery());
        return false;
    }

    private boolean isSqlAllExpr(SQLExpr x) {
        return x.getClass() == SQLAllExpr.class;
    }

    private boolean isAnyOrSomeExpr(SQLExpr x) {
        return x.getClass() == SQLAnyExpr.class || x.getClass() == SQLSomeExpr.class;
    }

    public boolean visit(SQLSelectItem x) {
        sqlNode = convertToSqlNode(x.getExpr());

        SqlNode as = null;
        if (x.getAlias() != null) {
            as = new SqlIdentifier(x.getAlias(), SqlParserPos.ZERO);
        }

        if(as != null) {
            sqlNode = new SqlBasicCall(SqlStdOperatorTable.AS, new SqlNode[] { sqlNode, as }, SqlParserPos.ZERO);
        }

        return false;
    }

    @Override public boolean visit(SQLIdentifierExpr x) {
        sqlNode = new SqlIdentifier(x.getName(), SqlParserPos.ZERO);
        return false;
    }

    public boolean visit(SQLPropertyExpr x) {
        sqlNode = new SqlIdentifier(Arrays.asList(x.getOwnernName(), x.getName()), SqlParserPos.ZERO);
        return false;
    }


    public boolean visit(SQLBinaryOpExpr x) {
        SqlOperator operator = null;

        SqlQuantifyOperator someOrAllOperator = null;

        SQLExpr rightExpr = x.getRight();
        SqlNode right = convertToSqlNode(rightExpr);

        switch (x.getOperator()) {
            case Equality:
                if (isSqlAllExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.ALL_EQ;
                } else if (isAnyOrSomeExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.SOME_EQ;
                } else {
                    operator = SqlStdOperatorTable.EQUALS;
                }
                break;
            case GreaterThan:
                if (isSqlAllExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.ALL_GT;
                } else if (isAnyOrSomeExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.SOME_GT;
                } else {
                    operator = SqlStdOperatorTable.GREATER_THAN;
                }
                break;
            case GreaterThanOrEqual:
                if (isSqlAllExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.ALL_GE;
                } else if (isAnyOrSomeExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.SOME_GE;
                } else {
                    operator = SqlStdOperatorTable.GREATER_THAN_OR_EQUAL;
                }
                break;
            case LessThan:
                if (isSqlAllExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.ALL_LT;
                } else if (isAnyOrSomeExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.SOME_LT;
                } else {
                    operator = SqlStdOperatorTable.LESS_THAN;
                }
                break;
            case LessThanOrEqual:
                if (isSqlAllExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.ALL_LE;
                } else if (isAnyOrSomeExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.SOME_LE;
                } else {
                    operator = SqlStdOperatorTable.LESS_THAN_OR_EQUAL;
                }
                break;
            case NotEqual:
            case LessThanOrGreater:
                if (isSqlAllExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.ALL_NE;
                } else if (isAnyOrSomeExpr(rightExpr)) {
                    someOrAllOperator = SqlStdOperatorTable.SOME_NE;
                } else {
                    operator = SqlStdOperatorTable.NOT_EQUALS;
                }
                break;
            case Add:
                operator = SqlStdOperatorTable.PLUS;
                break;
            case Subtract:
                operator = SqlStdOperatorTable.MINUS;
                break;
            case Multiply:
                operator = SqlStdOperatorTable.MULTIPLY;
                break;
            case Divide:
                operator = SqlStdOperatorTable.DIVIDE;
                break;
            case Modulus:
                operator = SqlStdOperatorTable.MOD;
                break;
            case Like:
                operator = SqlStdOperatorTable.LIKE;
                break;
            case NotLike:
                operator = SqlStdOperatorTable.NOT_LIKE;
                break;
            case BooleanAnd:
                operator = SqlStdOperatorTable.AND;
                break;
            case BooleanOr:
                operator = SqlStdOperatorTable.OR;
                break;
            case Is:
                if(x.getRight() instanceof SQLNullExpr) {
                    operator = SqlStdOperatorTable.IS_NULL;
                } else if(x.getRight() instanceof SQLBooleanExpr){
                    if(((SQLBooleanExpr) rightExpr).getValue()){
                        operator = SqlStdOperatorTable.IS_TRUE;
                    } else {
                        operator = SqlStdOperatorTable.IS_NOT_TRUE;
                    }
                }
                break;
            case IsNot:
                if(rightExpr instanceof SQLNullExpr) {
                    operator = SqlStdOperatorTable.IS_NOT_NULL;
                } else if(rightExpr instanceof SQLBooleanExpr){
                    if(((SQLBooleanExpr) rightExpr).getValue()){
                        operator = SqlStdOperatorTable.IS_NOT_TRUE;
                    } else {
                        operator = SqlStdOperatorTable.IS_TRUE;
                    }
                }
                break;
            default:
                throw new FastsqlException("not support " + x.getOperator());

        }

        SqlNode left = convertToSqlNode(x.getLeft());
        if (someOrAllOperator != null) {
            this.sqlNode = new SqlBasicCall(someOrAllOperator, new SqlNode[] { left, right },
                    SqlParserPos.ZERO);
        } else {
            if(operator == SqlStdOperatorTable.IS_NULL || operator == SqlStdOperatorTable.IS_NOT_NULL || operator == SqlStdOperatorTable.IS_TRUE || operator == SqlStdOperatorTable.IS_NOT_TRUE) {
                this.sqlNode = new SqlBasicCall(operator,
                        new SqlNode[]{left},
                        SqlParserPos.ZERO);
            } else {
                this.sqlNode = new SqlBasicCall(operator,
                        new SqlNode[]{left,right},
                        SqlParserPos.ZERO);
            }
        }
        return false;
    }

    public boolean visit(SQLExistsExpr x) {
        SqlOperator sqlOperator = SqlStdOperatorTable.EXISTS;
        SqlNode sqlNode = sqlOperator.createCall(SqlParserPos.ZERO,
                convertToSqlNode(x.getSubQuery()));
        if(x.isNot()){
            sqlNode = SqlStdOperatorTable.NOT.createCall(SqlParserPos.ZERO,sqlNode);
        }
        this.sqlNode = sqlNode;
        return false;
    }

    public boolean visit(SQLAllColumnExpr x) {
        sqlNode = new SqlIdentifier(Arrays.asList(""), SqlParserPos.ZERO);

        return false;
    }

    public boolean visit(SQLCharExpr x) {
        sqlNode = SqlLiteral.createCharString(x.getText(), SqlParserPos.ZERO);
        return false;
    }

    public boolean visit(SQLNullExpr x) {
        sqlNode = SqlLiteral.createNull(SqlParserPos.ZERO);
        return false;
    }

    public boolean visit(SQLIntegerExpr x) {
        sqlNode = SqlLiteral.createExactNumeric(String.valueOf(x.getNumber().longValue()), SqlParserPos.ZERO);
        return false;
    }

    public boolean visit(SQLBooleanExpr x) {
        sqlNode = SqlLiteral.createBoolean(x.getBooleanValue(), SqlParserPos.ZERO);
        return false;
    }

    public boolean visit(SQLNumberExpr x) {
        sqlNode = SqlLiteral.createExactNumeric(x.getNumber().toString(), SqlParserPos.ZERO);
        return false;
    }

    public boolean visit(SQLAggregateExpr x) {
        SqlOperator functionOperator = null;

        String methodName = x.getMethodName();

        long hashCode64 = x.methodNameHashCode64();

        if (hashCode64 == FnvHash.Constants.COUNT) {
            functionOperator = SqlStdOperatorTable.COUNT;

        } else if (hashCode64 == FnvHash.Constants.SUM) {
            functionOperator = SqlStdOperatorTable.SUM;

        } else if (hashCode64 == FnvHash.Constants.MAX) {
            functionOperator = SqlStdOperatorTable.MAX;

        } else if (hashCode64 == FnvHash.Constants.MIN) {
            functionOperator = SqlStdOperatorTable.MIN;

        } else if (hashCode64 == FnvHash.Constants.AVG) {
            functionOperator = SqlStdOperatorTable.AVG;

        } else {
            functionOperator = new SqlUnresolvedFunction(
                    new SqlIdentifier(methodName, SqlParserPos.ZERO),
                    null,
                    null,
                    null,
                    null,
                    SqlFunctionCategory.USER_DEFINED_FUNCTION);
        }

        SqlLiteral functionQualifier = null;

        if (x.getOption() == SQLAggregateOption.DISTINCT) {
            functionQualifier = SqlSelectKeyword.DISTINCT.symbol(SqlParserPos.ZERO);
        }
        List<SQLExpr> arguments = x.getArguments();
        List<SqlNode> argNodes = new ArrayList<SqlNode>(arguments.size());

        if (arguments != null) {
            for (SQLExpr exp : arguments) {
                argNodes.add(convertToSqlNode(exp));
            }
        }

        this.sqlNode = new CalciteSqlBasicCall(functionOperator,
                                               SqlParserUtil.toNodeArray(argNodes),
                                               SqlParserPos.ZERO,
                                               false,
                                               functionQualifier);

        return false;
    }

    public boolean visit(SQLMethodInvokeExpr x) {
        SqlOperator functionOperator = null;

        String methodName = x.getMethodName();
        if (StringUtils.equals(methodName, "EXISTS")) {
            functionOperator = SqlStdOperatorTable.EXISTS;
        } else {
            functionOperator = new SqlUnresolvedFunction(
                    new SqlIdentifier(methodName, SqlParserPos.ZERO),
                    null,
                    null,
                    null,
                    null,
                    SqlFunctionCategory.USER_DEFINED_FUNCTION);
        }

        SqlLiteral functionQualifier = null;

        List<SQLExpr> arguments = x.getArguments();
        List<SqlNode> argNodes = new ArrayList<SqlNode>(arguments.size());

        if (arguments != null) {
            for (SQLExpr exp : arguments) {
                argNodes.add(convertToSqlNode(exp));
            }
        }

        this.sqlNode = new CalciteSqlBasicCall(functionOperator,
                                               SqlParserUtil.toNodeArray(argNodes),
                                               SqlParserPos.ZERO,
                                               false,
                                               functionQualifier);
        return false;
    }

    public boolean visit(SQLInListExpr x) {
        SqlNodeList sqlNodes = convertToSqlNodeList(x.getTargetList());
        SqlOperator sqlOperator = x.isNot() ? SqlStdOperatorTable.NOT_IN : SqlStdOperatorTable.IN;
        sqlNode = new SqlBasicCall(sqlOperator, new SqlNode[] { convertToSqlNode(x.getExpr()), sqlNodes },
                                   SqlParserPos.ZERO);

        return false;
    }

    public boolean visit(SQLVariantRefExpr x){
        if ("?".equals(x.getName())) {
            this.sqlNode = new SqlDynamicParam(x.getIndex(),
                    SqlParserPos.ZERO);
            return false;
        } else {
            System.out.println("end");
        }
        return false;
    }

    @Override
    public boolean visit(SQLUnaryExpr x) {
        SQLUnaryOperator operator = x.getOperator();
        switch (operator) {
            case NOT:
                this.sqlNode = SqlStdOperatorTable.NOT.createCall(SqlParserPos.ZERO,
                                                                  convertToSqlNode(x.getExpr()));
                break;
            case Negative:
                this.sqlNode = SqlStdOperatorTable.UNARY_MINUS.createCall(SqlParserPos.ZERO,
                                                                          convertToSqlNode(x.getExpr()));
                break;
            case Not:
            case Compl:
            case BINARY:
            default:
                super.visit(x);
        }
        return false;
    }


    protected SqlNodeList convertToSqlNodeList(List<? extends SQLExpr> exprList) {
        final int size = exprList.size();

        List<SqlNode> nodes = new ArrayList<SqlNode>(size);
        for (int i = 0; i < size; ++i) {
            SQLExpr expr = exprList.get(i);
            SqlNode node = convertToSqlNode(expr);
            nodes.add(node);
        }

        return new SqlNodeList(nodes, SqlParserPos.ZERO);
    }


    protected SqlNode convertToSqlNode(SQLObject ast) {
        if (ast == null) {
            return null;
        }
        CalciteMySqlNodeVisitor visitor = new CalciteMySqlNodeVisitor();
        ast.accept(visitor);
        return visitor.getSqlNode();
    }

    private SqlNodeList convertOrderby(SQLOrderBy orderBy) {
        List<SQLSelectOrderByItem> items = orderBy.getItems();
        List<SqlNode> orderByNodes = new ArrayList<SqlNode>(items.size());

        for (SQLSelectOrderByItem item : items) {
            SqlNode node = convertToSqlNode(item.getExpr());
            if (item.getType() == SQLOrderingSpecification.DESC) {
                node = new SqlBasicCall(SqlStdOperatorTable.DESC, new SqlNode[] { node }, SqlParserPos.ZERO);
            }
            orderByNodes.add(node);
        }

        return new SqlNodeList(orderByNodes, SqlParserPos.ZERO);
    }

    private SqlNodeList convertHints(List<SQLCommentHint> hints) {
        if (hints == null) {
            return null;
        }

        List<SqlNode> nodes = new ArrayList<SqlNode>(hints.size());

        for (SQLCommentHint hint : hints) {
            if (hint instanceof TDDLHint) {
                nodes.add(convertTDDLHint((TDDLHint) hint));
            }
        }

        return new SqlNodeList(nodes, SqlParserPos.ZERO);

    }

    private SqlNodeList convertTDDLHint(TDDLHint hint) {

        List<TDDLHint.Function> functions = hint.getFunctions();
        List<SqlNode> funNodes = new ArrayList<SqlNode>(functions.size());

        for (TDDLHint.Function function : functions) {
            String functionName = function.getName();

            List<TDDLHint.Argument> arguments = function.getArguments();

            SqlNode[] argNodes = new SqlNode[arguments.size()];
            for (int i = 0; i < arguments.size(); i++) {
                TDDLHint.Argument argument = arguments.get(i);
                SqlNode argName = convertToSqlNode(argument.getName());
                SqlNode argValue = convertToSqlNode(argument.getValue());

                List<SqlNode> arg = new ArrayList<SqlNode>();
                if(argName != null) {
                    arg.add(argName);
                }
                if(argValue != null) {
                    arg.add(argValue);
                }

                SqlNode argNode = null;
                if (arg.size() == 2) {
                    argNode = SqlStdOperatorTable.EQUALS.createCall(SqlParserPos.ZERO, arg);
                } else if (arg.size() == 1) {
                    argNode = argName;
                }

                argNodes[i] = argNode;
            }

            SqlNode funNode = new SqlBasicCall(
                    new SqlUnresolvedFunction(new SqlIdentifier(functionName, SqlParserPos.ZERO), null, null,
                                              null, null, SqlFunctionCategory.USER_DEFINED_FUNCTION), argNodes,
                    SqlParserPos.ZERO);

            funNodes.add(funNode);
        }

        return new SqlNodeList(funNodes, SqlParserPos.ZERO);
    }


}
