/*
 * This file is part of NavalPlan
 *
 * Copyright (C) 2009-2010 Fundación para o Fomento da Calidade Industrial e
 *                         Desenvolvemento Tecnolóxico de Galicia
 * Copyright (C) 2010-2011 Igalia, S.L.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.navalplanner.web.costcategories;

import static org.navalplanner.web.I18nHelper._;

import java.util.ConcurrentModificationException;
import java.util.List;

import org.apache.commons.logging.LogFactory;
import org.navalplanner.business.common.exceptions.ValidationException;
import org.navalplanner.business.costcategories.entities.TypeOfWorkHours;
import org.navalplanner.web.common.ConstraintChecker;
import org.navalplanner.web.common.IMessagesForUser;
import org.navalplanner.web.common.Level;
import org.navalplanner.web.common.MessagesForUser;
import org.navalplanner.web.common.OnlyOneVisible;
import org.navalplanner.web.common.Util;
import org.navalplanner.web.common.components.NewDataSortableGrid;
import org.zkoss.zk.ui.Component;
import org.zkoss.zk.ui.event.CheckEvent;
import org.zkoss.zk.ui.event.Event;
import org.zkoss.zk.ui.util.GenericForwardComposer;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.api.Window;

/**
 * Controller for CRUD actions over a {@link TypeOfWorkHours}
 *
 * @author Jacobo Aragunde Perez <jaragunde@igalia.com>
 */
@SuppressWarnings("serial")
public class TypeOfWorkHoursCRUDController extends GenericForwardComposer implements
        ITypeOfWorkHoursCRUDController {

    private static final org.apache.commons.logging.Log LOG = LogFactory
            .getLog(TypeOfWorkHoursCRUDController.class);

    private Window createWindow;

    private Window listWindow;

    private ITypeOfWorkHoursModel typeOfWorkHoursModel;

    private OnlyOneVisible visibility;

    private IMessagesForUser messagesForUser;

    private Component messagesContainer;

    private NewDataSortableGrid listing;

    @Override
    public void doAfterCompose(Component comp) throws Exception {
        super.doAfterCompose(comp);
        comp.setVariable("controller", this, true);
        messagesForUser = new MessagesForUser(messagesContainer);
        listing = (NewDataSortableGrid) listWindow.getFellowIfAny("listing");
        getVisibility().showOnly(listWindow);
    }

    @Override
    public void goToCreateForm() {
        typeOfWorkHoursModel.initCreate();
        createWindow.setTitle(_("Create Work Hours Type"));
        getVisibility().showOnly(createWindow);
        Util.reloadBindings(createWindow);
    }

    @Override
    public void goToEditForm(TypeOfWorkHours typeOfWorkHours) {
        typeOfWorkHoursModel.initEdit(typeOfWorkHours);
        createWindow.setTitle(_("Edit Work Hours Type"));
        getVisibility().showOnly(createWindow);
        Util.reloadBindings(createWindow);
    }

    @Override
    public void goToList() {
        getVisibility().showOnly(listWindow);
        Util.reloadBindings(listWindow);
    }

    public void cancel() {
        goToList();
    }

    public void saveAndExit() {
        if (save()) {
            goToList();
        }
    }

    public void saveAndContinue() {
        if (save()) {
            goToEditForm(getTypeOfWorkHours());
        }
    }

    public boolean save() {
        if(!ConstraintChecker.isValid(createWindow)) {
            return false;
        }
        try {
            typeOfWorkHoursModel.confirmSave();
            messagesForUser.showMessage(Level.INFO,
                    _("Type of work hours saved"));
            return true;
        } catch (ValidationException e) {
            messagesForUser.showInvalidValues(e);
        }
        return false;
    }

    public List<TypeOfWorkHours> getTypesOfWorkHours() {
        return typeOfWorkHoursModel.getTypesOfWorkHours();
    }

    public TypeOfWorkHours getTypeOfWorkHours() {
        return typeOfWorkHoursModel.getTypeOfWorkHours();
    }

    private OnlyOneVisible getVisibility() {
        return (visibility == null) ? new OnlyOneVisible(createWindow,
                listWindow)
                : visibility;
    }

    public void onCheckGenerateCode(Event e) {
        CheckEvent ce = (CheckEvent) e;
        if (ce.isChecked()) {
            try {
                // we have to auto-generate the code
                typeOfWorkHoursModel.setCodeAutogenerated(ce.isChecked());
            } catch (ConcurrentModificationException err) {
                messagesForUser.showMessage(Level.ERROR, err.getMessage());
            }
        }
        Util.reloadBindings(createWindow);
    }

    public void confirmDelete(TypeOfWorkHours typeOfWorkHours) {
        if (!isReferencedByOtherEntities(typeOfWorkHours)) {
            int result = showConfirmDeleteWorkHoursType(typeOfWorkHours);
            if (result == Messagebox.OK) {
                typeOfWorkHoursModel.confirmRemove(typeOfWorkHours);
                Util.reloadBindings(listing);
            }
        }
    }

    private boolean isReferencedByOtherEntities(TypeOfWorkHours typeOfWorkHours) {
        try {
            typeOfWorkHoursModel.checkIsReferencedByOtherEntities(typeOfWorkHours);
            return false;
        } catch (ValidationException e) {
            showCannotDeleteWorkHoursTypeDialog(e.getInvalidValue()
                    .getMessage(), typeOfWorkHours);
        }
        return true;
    }

    private void showCannotDeleteWorkHoursTypeDialog(String message,
            TypeOfWorkHours typeOfWorkHours) {
        try {
            Messagebox.show(_(message), _("Warning"), Messagebox.OK,
                    Messagebox.EXCLAMATION);
        } catch (InterruptedException e) {
            LOG.error(_("Error on showing warning message removing typeOfWorkHours: ", typeOfWorkHours.getId()), e);
        }
    }

    private int showConfirmDeleteWorkHoursType(TypeOfWorkHours typeOfWorkHours) {
        try {
            return Messagebox.show(_("Delete item. Are you sure?"), _("Confirm"),
                    Messagebox.OK | Messagebox.CANCEL, Messagebox.QUESTION);
        } catch (InterruptedException e) {
            LOG.error(_("Error on removing typeOfWorkHours: ", typeOfWorkHours.getId()), e);
        }
        return Messagebox.CANCEL;
    }

}
