/*
 * This file is part of NavalPlan
 *
 * Copyright (C) 2010-2011 Wireless Galicia, S.L.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.navalplanner.web.orders;

import static org.navalplanner.web.I18nHelper._;

import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.navalplanner.business.calendars.entities.BaseCalendar;
import org.navalplanner.business.externalcompanies.entities.ExternalCompany;
import org.navalplanner.business.orders.entities.Order;
import org.navalplanner.web.common.ConstraintChecker;
import org.navalplanner.web.common.Util;
import org.navalplanner.web.common.components.bandboxsearch.BandboxSearch;
import org.navalplanner.web.planner.consolidations.AdvanceConsolidationController;
import org.navalplanner.web.planner.tabs.MultipleTabsPlannerController;
import org.zkoss.zk.ui.Component;
import org.zkoss.zk.ui.Executions;
import org.zkoss.zk.ui.SuspendNotAllowedException;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.event.Event;
import org.zkoss.zk.ui.event.EventListener;
import org.zkoss.zk.ui.event.Events;
import org.zkoss.zk.ui.util.GenericForwardComposer;
import org.zkoss.zul.ComboitemRenderer;
import org.zkoss.zul.Constraint;
import org.zkoss.zul.Datebox;
import org.zkoss.zul.Grid;
import org.zkoss.zul.Window;

/**
 * Controller for the creation of an {@link order} with its principal
 * properties.
 * @author Susana Montes Pedreira <smontes@wirelessgailicia.com>
 */

public class ProjectDetailsController extends GenericForwardComposer {

    private static final Log LOG = LogFactory
            .getLog(AdvanceConsolidationController.class);

    private OrderCRUDController orderController;

    private Grid gridProjectDetails;

    private BaseCalendar defaultCalendar;

    private String oldCode;

    private boolean isCodeAutogeneratedInit;

    private MultipleTabsPlannerController tabs;

    private Window window;

    private Datebox initDate;

    private BandboxSearch bdExternalCompanies;

    private Datebox deadline;

    public ProjectDetailsController() {
        Window window = (Window) Executions.createComponents(
                "/orders/_projectDetails.zul", null,
                new HashMap<String, String>());
        try {
            doAfterCompose(window);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void doAfterCompose(Component comp) throws Exception {
        super.doAfterCompose(comp);
        window = (Window) comp;
        window.setVariable("projectController", this, true);
    }

    public void showWindow(OrderCRUDController orderController,
            MultipleTabsPlannerController tabs) {
        this.tabs = tabs;
        this.orderController = orderController;
        this.defaultCalendar = orderController.getOrder().getCalendar();
        this.isCodeAutogeneratedInit = orderController.getOrder()
                .isCodeAutogenerated();
        loadComponents();
        try {
            Util.reloadBindings(window);
            Util.createBindingsFor(gridProjectDetails);
            Util.reloadBindings(gridProjectDetails);
            window.doModal();
        } catch (SuspendNotAllowedException e) {
            throw new RuntimeException(e);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public void cancel() {
        clearProperties();
        close();
    }

    public void accept() {
        if (validate()) {
            if (tabs != null) {
                tabs.goToOrdersList();
            }
            orderController.editNewCreatedOrder(window);
        }
    }

    private boolean validate() {
        if (!ConstraintChecker.isValid(window)) {
            return false;
        }
        if (initDate.getValue() == null) {
            showWrongValue();
            return false;
        }
        return true;
    }

    private void showWrongValue() {
        throw new WrongValueException(initDate, _("cannot be null or empty"));
    }

    private void close() {
        window.setVisible(false);
    }

    public Order getOrder() {
        return orderController.getOrder();
    }

    public boolean isCodeAutogenerated() {
        return orderController.isCodeAutogenerated();
    }

    public void setCodeAutogenerated(boolean codeAutogenerated) {
        orderController.setCodeAutogeneratedInModel(codeAutogenerated);
        Util.reloadBindings(gridProjectDetails);
    }

    public List<ExternalCompany> getExternalCompaniesAreClient() {
        return orderController.getExternalCompaniesAreClient();
    }

    public List<BaseCalendar> getBaseCalendars() {
        return orderController.getBaseCalendars();
    }

    public ComboitemRenderer getBaseCalendarsComboitemRenderer() {
        return orderController.getBaseCalendarsComboitemRenderer();
    }

    public void setBaseCalendar(BaseCalendar calendar) {
        orderController.setBaseCalendar(calendar);
    }

    private void clearProperties() {
        Order order = orderController.getOrder();
        order.setName(null);
        // reset the code autogenerated property
        if (isCodeAutogeneratedInit) {
            order.setCodeAutogenerated(true);

        } else {
            order.setCodeAutogenerated(false);
            order.setCode("");
        }
        order.setCustomer(null);
        order.setDeadline(null);
        order.setInitDate(new Date());
        order.setCalendar(defaultCalendar);
    }

    public Constraint checkConstraintFinishDate() {
        return new Constraint() {
            @Override
            public void validate(Component comp, Object value)
                    throws WrongValueException {
                Date finishDate = (Date) value;
                if ((finishDate != null) && (initDate.getValue() != null)
                        && (finishDate.compareTo(initDate.getValue()) < 0)) {
                    deadline.setValue(null);
                    getOrder().setDeadline(null);
                    throw new WrongValueException(comp,
                            _("must be greater than start date"));
                }
            }
        };
    }

    public Constraint checkConstraintStartDate() {
        return new Constraint() {
            @Override
            public void validate(Component comp, Object value)
                    throws WrongValueException {
                Date startDate = (Date) value;
                if ((startDate != null) && (deadline.getValue() != null)
                        && (startDate.compareTo(deadline.getValue()) > 0)) {
                    initDate.setValue(null);
                    getOrder().setInitDate(null);
                    throw new WrongValueException(comp,
                            _("must be lower than finish date"));
                }
            }
        };
    }

    private void loadComponents() {
        bdExternalCompanies.setListboxEventListener(Events.ON_SELECT,
                new EventListener() {
                    @Override
                    public void onEvent(Event event) throws Exception {
                        final Object object = bdExternalCompanies
                                .getSelectedElement();
                        getOrder().setCustomer((ExternalCompany) object);
                    }
                });
        bdExternalCompanies.setListboxEventListener(Events.ON_OK,
                new EventListener() {
                    @Override
                    public void onEvent(Event event) throws Exception {
                        final Object object = bdExternalCompanies
                                .getSelectedElement();
                        getOrder().setCustomer((ExternalCompany) object);
                        bdExternalCompanies.close();
                    }
                });
    }

}