<?php

/*
 * This file is part of ibrand/address.
 *
 * (c) GuoJiangClub <https://www.ibrand.cc>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

use Illuminate\Foundation\Testing\DatabaseMigrations;

/**
 * Created by PhpStorm.
 * User: Administrator
 * Date: 2018/2/3
 * Time: 17:19.
 */
class AddressTest extends \Orchestra\Testbench\TestCase
{
    use DatabaseMigrations;

    public $repository;

    protected function getEnvironmentSetUp($app)
    {
        // Setup default database to use sqlite :memory:
        $app['config']->set('database.default', 'testing');
        $app['config']->set('database.connections.testing', [
            'driver' => 'sqlite',
            'database' => ':memory:',
        ]);

        $app['config']->set('repository.cache.enabled', true);
    }

    protected function getPackageProviders($app)
    {
        return [
            Prettus\Repository\Providers\RepositoryServiceProvider::class
            , Orchestra\Database\ConsoleServiceProvider::class
            , GuoJiangClub\Component\Address\ServiceProvider::class];
    }

    protected function setUp()
    {
        parent::setUp(); // TODO: Change the autogenerated stub

        $this->loadMigrationsFrom(__DIR__ . '/database');

        $this->repository = $this->app->make(\GuoJiangClub\Component\Address\RepositoryContract::class);
    }

    public function testCreate()
    {
        //1. test create default address.
        $attributes = ['user_id' => 1, 'accept_name' => '小李子', 'mobile' => '18933336666', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区', 'address' => 'xxx大道xx小区502', 'is_default' => 1,];

        $result = $this->repository->create($attributes);

        $this->assertEquals(\GuoJiangClub\Component\Address\Address::class, get_class($result));

        $this->assertEquals(1, $result->is_default);

        //1. test new default address, old address is not default.
        $attributes = ['user_id' => 1, 'accept_name' => '小陈子', 'mobile' => '18933336677', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区1', 'address' => 'xxx大道xx小区5023', 'is_default' => 1,];

        $result1 = $this->repository->create($attributes);

        $result = $this->repository->find($result->id);

        $this->assertEquals(0, $result->is_default);
        $this->assertEquals(1, $result1->is_default);

        //2. test user addresses

        $addresses = $this->repository->getByUser(1);
        $this->assertCount(2, $addresses);
    }

    public function testGetDefault()
    {
        //test get user empty address.
        $result = $this->repository->getDefaultByUser(1);
        $this->assertNull($result);

        //test user create two default address.
        $attributes = ['user_id' => 1, 'accept_name' => '小李子', 'mobile' => '18933336666', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区', 'address' => 'xxx大道xx小区502', 'is_default' => 1,];

        $this->repository->create($attributes);

        $attributes = ['user_id' => 1, 'accept_name' => '小陈子', 'mobile' => '18933336677', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区1', 'address' => 'xxx大道xx小区5023', 'is_default' => 1,];

        $result1 = $this->repository->create($attributes);

        $default = $this->repository->getDefaultByUser(1);

        $this->assertEquals($result1->id, $default->id);
        $this->assertEquals(1, $default->is_default);

        //test get user default from all is not default addresses.
        $attributes = ['user_id' => 2, 'accept_name' => '小李子', 'mobile' => '18933336666', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区', 'address' => 'xxx大道xx小区502', 'is_default' => 0,];

        $result = $this->repository->create($attributes);

        $attributes = ['user_id' => 2, 'accept_name' => '小陈子', 'mobile' => '18933336677', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区1', 'address' => 'xxx大道xx小区5023', 'is_default' => 0,];

        $result1 = $this->repository->create($attributes);

        $default = $this->repository->getDefaultByUser(2);

        $this->assertEquals($result->id, $default->id);
        $this->assertEquals(0, $default->is_default);
    }

    public function testUpdateByUser()
    {
        $attributes = ['user_id' => 1, 'accept_name' => '小李子', 'mobile' => '18933336666', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区', 'address' => 'xxx大道xx小区502', 'is_default' => 1,];

        $result = $this->repository->create($attributes);

        //1. test update own address.

        $attributes = ['accept_name' => '小李子1', 'mobile' => '18933336661'];

        $result = $this->repository->updateByUser($attributes, $result->id, 1);

        $this->assertEquals('18933336661', $result->mobile);
        $this->assertEquals('小李子1', $result->accept_name);

        //1. test update other user address.

        $attributes = ['user_id' => 2, 'accept_name' => '小李子', 'mobile' => '18933336666', 'province' => 320000,
            'city' => 320100, 'area' => '320102', 'address_name' => '江苏省 南京市 玄武区', 'address' => 'xxx大道xx小区502', 'is_default' => 1,];

        $result = $this->repository->create($attributes);

        $attributes = ['accept_name' => '小李子1', 'mobile' => '18933336661'];

        $result = $this->repository->updateByUser($attributes, $result->id, 1);

        $this->assertFalse($result);
    }
}
