<?php
/**
 * Controller.class.php
 * 
 * doitPHP控制器基类 
 * @author tommy <streen003@gmail.com>
 * @copyright  Copyright (c) 2010 Tommy Software Studio
 * @link http://www.doitphp.com
 * @license New BSD License.{@link http://www.opensource.org/licenses/bsd-license.php}
 * @version $Id: Controller.class.php 1.1 2011-8-12 22:48:01Z tommy $
 * @package core
 * @since 1.0
 */

if (!defined('IN_DOIT')) {
	exit();
}

abstract class Controller extends Base {
	
	/**
	 * 视图布局名称(layout)
	 * 
	 * @var string
	 */
	protected $layout = 'main';
	
	/**
	 * 视图主题风格名称(theme)
	 * 
	 * @var string
	 */
	protected $theme;
	
	/**
	 * 视图变量数组
	 * 
	 * @var array
	 */
	protected $_options = array();
	
	/**
	 * 视图的实例化对象(仅限于视图文件为html文件时)
	 * 
	 * @var object
	 */
	protected $view;
	
	/**
	 * 构造函数
	 * 
	 * 用于初始化本类的运行环境,或对基本变量进行赋值
	 * @access public
	 * @return boolean
	 */
	public function __construct() {
		
	    //时区设置,默认为中国(北京时区)
		date_default_timezone_set(DOIT_TIMEZONE);
		
		//设置项目系统session的存放目录
		$session_dir = APP_ROOT . 'cache/temp';
		if (is_dir($session_dir) && is_writable($session_dir)) {			
			session_save_path($session_dir);
		}
		
		//关闭魔术变量，提高PHP运行效率.
		if (get_magic_quotes_runtime()) {	
			set_magic_quotes_runtime(0);
		}
		
	    //将全局变量进行魔术变量处理
		if (get_magic_quotes_gpc()) {				
			$_COOKIE = $this->strip_slashes($_COOKIE);
		} else {			
			$_POST    = $this->add_slashes($_POST);
			$_GET     = $this->add_slashes($_GET);
			$_SESSION = $this->add_slashes($_SESSION);
		}
	}
	
	/**
	 * 第一部分：获取参数，处理$_GET, $_POST等全局超级变量数组的参数
	 * @author tommy
	 * @version 1.0 2010-10-19
	 */

	/**
	 * 获取并分析$_GET数组某参数值
	 * 
	 * 获取$_GET的全局超级变量数组的某参数值,并进行转义化处理，提升代码安全.注:参数支持数组
	 * @access public
	 * @param string $string 所要获取$_GET的参数
	 * @return string	$_GET数组某参数值
	 */
	public static function get($string) {
				
		if (!is_array($string)) {
			return isset($_GET[$string]) ? htmlspecialchars(trim($_GET[$string])) : '';			
		}
		
		foreach ($string as $key=>$value) {					
			$string[$key] = self::get($value);
		}
					
		return $string;
	}
	
	/**
	 * 获取并分析$_POST数组某参数值
	 * 
	 * 获取$_POST全局变量数组的某参数值,并进行转义等处理，提升代码安全.注:参数支持数组
	 * @access public
	 * @param string $string	所要获取$_POST的参数
	 * @return string	$_POST数组某参数值
	 */
	public static function post($string) {
				
		if (!is_array($string)) {
			return isset($_POST[$string]) ? htmlspecialchars(trim($_POST[$string])) : '';				
			
		}
		
		foreach ($string as $key=>$value) {				
			$string[$key] = self::post($value);
		}
					
		return $string;
	}
	
	/**
	 * 获取并分析 $_GET或$_POST全局超级变量数组某参数的值
	 * 
	 * 获取并分析$_POST['参数']的值 ，当$_POST['参数']不存在或为空时，再获取$_GET['参数']的值。
	 * @access public
	 * @param string $string 所要获取的参数名称
	 * @return string	$_GET或$_POST数组某参数值
	 */
	public static function get_params($string) {
		
		$param_value = self::post($string);
		
		//当$_POST[$string]值没空时
		return empty($param_value) ? self::get($string) : $param_value;
	}
	
	
	/**
	 * 第二部分：程序调试操作，用于程序调试或运行时出现错误时的信息提示
	 * @author tommy
	 * @version 1.0 2010-10-19
	 */

	/**
	 * trigger_error()的简化函数
	 * 
	 * 用于显示错误信息. 若调试模式关闭时(即:DOIT_DEBUG为false时)，则将错误信息并写入日志
	 * @access public
	 * @param string $message 所要显示的错误信息
	 * @param string $level     日志类型. 默认为Error. 参数：Warning, Error, Notice
	 * @return void
	 */
	public function halt($message, $level = 'Error') {
		
		//参数分析
		if (empty($message)) {
			return false;
		}								
		
		//调试模式下优雅输出错误信息
		$trace 			= debug_backtrace();
		$source_file 	= $trace[0]['file'] . '(' . $trace[0]['line'] . ')';
			
		$trace_string 	= '';
		foreach ($trace as $key=>$t) {
			$trace_string .= '#'. $key . ' ' . $t['file'] . '('. $t['line'] . ')' . $t['class'] . $t['type'] . $t['function'] . '(' . implode('.',  $t['args']) . ')<br/>';			
		}
			
		//加载,分析,并输出excepiton文件内容
		include_once DOIT_ROOT . 'views/html/exception.php';		
		
		if (DOIT_DEBUG === false) {			
			//写入程序运行日志
			Log::write($message, $level);				
		}
		
		//终止程序
		exit();
	}
	
	/**
	 * 显示提示信息操作
	 * 
	 * 所显示的提示信息并非完全是错误信息。如：用户登陆时用户名或密码错误，可用本方法输出提示信息
	 * 
	 * 注：显示提示信息的页面模板内容可以自定义. 方法：在项目视图目录中的error子目录中新建message.html文件,自定义该文件内容
	 * 显示错误信息处模板标签为<!--{$message}-->
	 * 
	 * 本方法支持URL的自动跳转，当显示时间有效期失效时则跳转到自定义网址，若跳转网址为空则函数不执行跳转功能，当自定义网址参数为-1时默认为:返回上一页。
	 * @access public
	 * @param string $message 		所要显示的提示信息
	 * @param string $goto_url 		所要跳转的自定义网址
	 * @param int    $limit_time 	显示信息的有效期,注:(单位:秒) 默认为5秒
	 * @return void
	 */
	public function show_message($message, $goto_url = null, $limit_time = 5) {
		
		//参数分析
		if (!$message) {
			return false;
		}
		
		//当自定义跳转网址存在时
		if (!is_null($goto_url)) {									
			$limit_time = 1000 * $limit_time;			
			//分析自定义网址是否为返回页
			if ($goto_url == -1) {				
				$goto_url = 'javascript:history.go(-1);';
				$message .= '<br/><a href="javascript:history.go(-1);" target="_self">如果你的浏览器没反应,请点击这里...</a>';
			} else{
				//防止网址过长，有换行引起跳转变不正确
				$goto_url = str_replace(array("\n","\r"), '', $goto_url);
				$message .= '<br/><a href="' . $goto_url . '" target="_self">如果你的浏览器没反应,请点击这里...</a>';
			}			
			$message .= '<script type="text/javascript">function doit_redirect_url(url){location.href=url;}setTimeout("doit_redirect_url(\'' . $goto_url . '\')", ' . $limit_time . ');</script>';
		}
		
		$message_template_file = VIEW_DIR . 'error/message.php';
				
		is_file($message_template_file) ? include_once $message_template_file : include_once DOIT_ROOT . 'views/html/message.php';
		
		exit();
	}
	
	/**
	 * 优雅输出print_r()函数所要输出的内容
	 * 
	 * 用于程序调试时,完美输出调试数据,功能相当于print_r().当第二参数为true时(默认为:false),功能相当于var_dump()。
	 * 注:本方法一般用于程序调试
	 * @access public
	 * @param array $data 		所要输出的数据
	 * @param boolean $option 	选项:true或 false
	 * @return array			所要输出的数组内容
	 */
	public function dump($data, $option = false) {
		
		//当输出print_r()内容时
		if(!$option){
			echo '<pre>';
			print_r($data);
			echo '</pre>';
		} else {
			ob_start();
			var_dump($data);
			$output = ob_get_clean();
			
			$output = str_replace('"', '', $output);
			$output = preg_replace('/\]\=\>\n(\s+)/m', '] => ', $output);
			
			echo '<pre>', $output, '</pre>';
		}

		exit;
	}
	
		
	/**
	 * 第三部分：获取当前程序运行的环境信息.如:获取域名，当前网页的根网址，当前网页的网址等信息
	 * @author tommy
	 * @version 1.0 2010-10-21
	 */
	
	/**
	 * 获取当前运行程序的网址域名
	 * 
	 * 如：http://www.doitphp.com
	 * @access public
	 * @return string	网址(域名)
	 */
	public static function get_server_name() {
		
		//获取网址域名部分.
		$server_name = !empty($_SERVER['HTTP_HOST']) ? strtolower($_SERVER['HTTP_HOST']) : $_SERVER['SERVER_NAME'];
		$server_port = ($_SERVER['SERVER_PORT'] == '80') ? '' : ':' . (int)$_SERVER['SERVER_PORT'];
		
		//获取网络协议.
		$secure = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') ? 1 : 0;		
		
		return ($secure ? 'https://' : 'http://') . $server_name . $server_port;
	}
	
	/**
	 * 获取当前项目的根目录的URL
	 * 
	 * 用于网页的CSS, JavaScript，图片等文件的调用.
	 * @access public
	 * @return string 	根目录的URL. 注:URL以反斜杠("/")结尾
	 */
	public static function get_base_url() {
		
		//处理URL中的//或\\情况,即:出现/或\重复的现象
		$url = str_replace(array('\\', '//'), URL_SEGEMENTATION, dirname($_SERVER['SCRIPT_NAME']));
		
		return (substr($url, -1) == URL_SEGEMENTATION) ? $url : $url . URL_SEGEMENTATION;
	}
	
	/**
	 * 获取当前运行的Action的URL
	 * 
	 * 获取当前Action的URL. 注:该网址由当前的控制器(Controller)及动作(Action)组成,不含有其它参数信息
	 * 如:/index.php/index/list，而非/index.php/index/list/page/5 或 /index.php/index/list/?page=5
	 * @access public
	 * @return string	URL
	 */
	public static function get_self_url() {
		
		return self::create_url(doit::get_controller_id() . URL_SEGEMENTATION . doit::get_action_id());
	}
	
	/**
	 * 获取当前Controller内的某Action的URL
	 * 
	 * 获取当前控制器(Controller)内的动作(Action)的URL. 注:该网址仅由项目入口文件和控制器(Controller)组成。
	 * @access public
	 * @param string $action_name 所要获取URL的action的名称
	 * @return string	URL
	 */
	public static function get_action_url($action_name) {
		
		//参数判断
		if (empty($action_name)) {
			return false;
		}
		
		return self::create_url(doit::get_controller_id() . URL_SEGEMENTATION . $action_name);
	}
	
	/**
	 * 获取当前项目asset目录的URL
	 * 
	 * @access public
	 * @param string $dir_name 子目录名
	 * @return string	URL
	 */
	public static function get_asset_url($dir_name = null) {
		
		//获取assets根目录的url
		$asset_url = self::get_base_url() . 'assets/';
		
		//分析assets目录下的子目录
		if (!is_null($dir_name)) {
			$asset_url .= $dir_name . '/';
		}
		
		return $asset_url;
	}
	
	/**
	 * 获取当前项目themes目录的URL
	 * 
	 * @access public
	 * @param string $theme_name 所要获取URL的主题名称
	 * @return string	URL
	 */
	public static function get_theme_url($theme_name = null){
		
		//分析主题名
		$theme_dir_name = is_null($theme_name) ? 'default' : $theme_name;
		
		return self::get_base_url() . 'themes/' . $theme_dir_name . '/';
	}
	
	
	/**
	 * 第四部分：URL处理操作. 如:URL跳转，URL组装等
	 * @author tommy
	 * @version 1.0 2010-10-21
	 */
	
	/**
	 * 网址(URL)跳转操作
	 * 
	 * 页面跳转方法，例:运行页面跳转到自定义的网址(即:URL重定向)
	 * @access public
	 * @param string $url 所要跳转的URL
	 * @return void
	 */
	public function redirect($url){
		
		//参数分析.
		if (!$url) {
			return false;
		}
				
		if (!headers_sent()) {
			header("Location:" . $url);			
		}else {
			echo '<script type="text/javascript">location.href="' . $url . '";</script>';
		}
		
		exit();
	}
	
	/**
	 * 网址(URL)组装操作
	 * 
	 * 组装绝对路径的URL
	 * @access public
	 * @param string 	$route 			controller与action
	 * @param array 	$params 		URL路由其它字段
	 * @param boolean 	$routing_mode	网址是否启用路由模式
	 * @return string	URL
	 */
	public static function create_url($route, $params = null, $routing_mode = true) {
	    
		//参数分析.
		if (!$route) {
			return false;
		}
		
		$url  = self::get_base_url() . ((DOIT_REWRITE === false) ? 'index.php' . URL_SEGEMENTATION : '') . $route;
		
		//参数$params变量的键(key),值(value)的URL组装
		if (!is_null($params) && is_array($params)) {
						
			$params_url = array();			
			if ($routing_mode == true) {				
				foreach ($params as $key=>$value) {
					$params_url[] = trim($key) . URL_SEGEMENTATION . trim($value);
				}
				$url .= URL_SEGEMENTATION . implode(URL_SEGEMENTATION, $params_url) . URL_SUFFIX;					
			} else {				
				foreach ($params as $key=>$value) {
					$params_url[] = trim($key) . '=' . trim($value);
				}
				$url .= URL_SEGEMENTATION . '?' . implode('&', $params_url);
			}
		}
				
		return str_replace('//', URL_SEGEMENTATION, $url);
	}
	
	
	/**
	 * 第五部分：类的实例化(单例模式)操作, 用于扩展类,model,扩展模块类的实例化
	 * @author tommy
	 * @version 1.0 2010-10-21
	 */
	
	/**
	 * 类的单例实例化操作
	 * 
	 * 用于类的单例模式的实例化,当某类已经实例化，第二次实例化时则直接反回初次实例化的object,避免再次实例化造成的系统资源浪费。
	 * @access public
	 * @param string $class_name 所要实例化的类名
	 * @return object	实例化后的对象
	 */
	public static function instance($class_name) {
		
		//参数判断
		if (!$class_name) {
			return false;
		}
		
		return doit::singleton($class_name);
	}
	
	/**
	 * 单例模式实例化一个Model对象
	 * 
	 * 单例模式实现化一个model对象. 初次实例化某Model后, 当第二次实例化时则直接调用初次实现化的结果(object)
	 * @access public
	 * @param string $table_name 	所要实例化的Modle名称
	 * @return object				实例化后的对象
	 */
	public static function model($table_name) {
		
		//参数判断
		if (!$table_name) {
			return false;
		}
		
		//分析model名		
		$model_name = ucfirst(strtolower($table_name)).'Model';
						
		return doit::singleton($model_name);
	}
	
	/**
	 * 加载并单例模式实例化扩展模块.
	 * 
	 * 注：这里所调用的扩展模板要放在项目extension目录里的modules子目录中
	 * @access public
	 * @param string $module_name 	模块名称
	 * @return object				扩展模块的实例化对象
	 */
	public static function module($module_name) {
		
		//参数判断.
		if (!$module_name) {
			return false;
		}
		
		static $_module_name_array = array();				
		if ($_module_name_array[$module_name] == null) {
			//加载扩展模块的引导文件
			$module_file = APP_ROOT . 'modules/' . $module_name . '/';
			$_module_name =  ucfirst(strtolower($module_name)); 
			$module_file .= $_module_name . 'Module.class.php';

			self::import($module_file);
			$_module_name_array[$module_name] = self::instance($_module_name . 'Module');
		} 
				
		return $_module_name_array[$module_name];
	}
	
	
	/**
	 * 第六部分:文件的加载, 等同于PHP函数include_once()
	 * @author tommy
	 * @version 1.0 2010-10-21
	 */
	
	/**
	 * 静态加载文件 
	 * 
	 * 相当于inclue_once()
	 * @access public
	 * @params string $file_name 所要加载的文件
	 * @return void
	 */
	public static function import($file_name) {
		
		//参数判断
		if (!$file_name) {
			return false;
		}
		
		//判断文件是不是项目扩展目录里的
		$file_url = (strpos($file_name, '/') !== false) ? realpath($file_name) : realpath(EXTENSION_DIR . $file_name . '.class.php');
		
		return doit::load_file($file_url);
	}
	
	/**
	 * 静态加载项目设置目录(config目录)中的配置文件
	 * 
	 * 加载项目设置目录(config)中的配置文件,当第一次加载后,第二次加载时则不再重新加载文件
	 * @access public
	 * @param string $file_name 所要加载的配置文件名 注：不含后缀名
	 * @return mixed	配置文件内容
	 */
	public static function load_config($file_name) {
		
		//参数分析.
		if (!$file_name) {
			return false;
		}
		
		static $_config = array();			
		if ($_config[$file_name] == null) {					
			$file_url = CONFIG_DIR . $file_name . '.ini.php';		
			//判断文件是否存在
			if (!is_file($file_url)) {			
				self::halt('The config file:' . $file_name . '.ini.class is not exists!');						
			}			
			$_config[$file_name] = include_once $file_url;
		}
				
		return $_config[$file_name];
	}
	
	
	/**
	 * 第七部分:视图操作, 注:视图操作仅限于后缀为.php的视图文件. 如果视图文件为html时,则建议用class View进行视图操作。
	 * @author tommy
	 * @version 1.0 2010-10-24
	 */
	
	/**
	 * 设置视图的主题
	 * 
	 * 用于变化网页风格的视图选择. 注:该主题的视图文件存放于项目themes目录内,而非项目controller目录内
	 * @access public
	 * @param string $theme_name 所要设置的网页模板主题名称 
	 * @return string	视图的主题名
	 */
	public function set_theme($theme_name = 'default') {
		
		return $this->theme = $theme_name;
	}
	
	/**
	 * 设置视图文件布局结构的文件名(layout)
	 * 
	 * layout默认为:main, 注:本方法无法被$this->template()所调用
	 * @access public
	 * @param string $layout_name 	所要设置的layout名称
	 * @return string				layout名称
	 */
	public function set_layout($layout_name = null) {
		
		return $this->layout = $layout_name;
	}
	
	/**
	 * 分析视图文件
	 * 
	 * 获取视图的路径,便于程序进行include操作. 注:本方法不支持视图布局结构(layout)
	 * @access publice
	 * @param string $file_name 视图文件名，注:名称中不带.php后缀。
	 * @return void
	 * 
	 * @example
	 * 法一：
	 * include $this->template('list'); 
	 * 
	 * 法二：
	 * include $this->template();	//即:当前controller视图目录下,当前action所对应的视图名称
	 * 如当前action为demo，则相当于$this->template('demo');
	 * 
	 * 法三：
	 * include $this->template('xx_controller/xx_action');
	 */
	public function template($file_name = null) {
		
		//参数分析
		if (is_null($file_name)) {						
			$file_name = doit::get_controller_id() . '/' . doit::get_action_id();			
		} else {			
			$file_name = (strpos($file_name, '/') !== false) ? $file_name : doit::get_controller_id() . '/' . $file_name;
		}
		
		//分析视图文件所在的目录,是否视图使用了主题
		$view_file  = (!empty($this->theme)) ? APP_ROOT . 'themes/' . $this->theme . '/' . $file_name : VIEW_DIR . $file_name;
		$view_file .= '.php';
		
		//分析视图文件是否存在
		if (!is_file($view_file)) {			
			$this->halt('The view file:' . $view_file . ' is not exists!');
		}
		
		return $view_file;
	}
	
	/**
	 * 视图变量赋值操作
	 * 
	 * @access public
	 * @param mixted $keys 视图变量名
	 * @param string $value 视图变量值
	 * @return mixted
	 */
	public function assign($keys, $value = null) {
		
		//参数分析
		if (!$keys) {
			return false;
		}
		
		//当$keys为数组时
		if (!is_array($keys)) {
			$this->_options[$keys] = $value;			
		} else {
			foreach ($keys as $handle=>$lines) {
				$this->_options[$handle] = $lines;
			}
		}
		
		return true;
	}
	
	/**
	 * 显示当前页面的视图内容
	 * 
	 * 包括视图页面中所含有的挂件(widget), 视图布局结构(layout), 及render()所加载的视图片段等
	 * @access public
	 * @param string $file_name 视图名称
	 * @param array  $_data     视图的变量
	 * @return void
	 */
	public function display($file_name = null, $_data = array()) {
		
		//分析视图文件路径
		$view_file = $this->template($file_name);
		
		//模板变量赋值
		if (!empty($this->_options)) {
			extract($this->_options, EXTR_PREFIX_SAME, 'data');
			//清空不必要的内存占用
			$this->_options = array();
		}
		
		if (!empty($_data) && is_array($_data)) {			
			extract($_data, EXTR_PREFIX_SAME, 'data');
			//清空不必要的内存占用
			unset($_data);
		}
		
		//获取当前视图($file_name)的内容
		ob_start();		
		include $view_file;		
		$content = ob_get_clean();
		
		//分析,加载,显示layout视图内容
		$layout_file = (!empty($this->theme)) ? APP_ROOT . 'themes/' . $this->theme . '/layout/' . $this->layout . '.php' : VIEW_DIR . 'layout/' . $this->layout . '.php';

		//分析layout文件是否存在.如果layout视图文件不存在,则直接显示$view_file视图内容
		if (!is_file($layout_file)) {
			echo $content;
		} else {			
			ob_start();		
			include $layout_file;		
			echo ob_get_clean();
		}
	}
	
	/**
	 * 加载视图文件的挂件(widget)
	 * 
	 * 加载挂件内容，一般用在视图内容中(view)
	 * @access public
	 * @param string  $widget_name 所要加载的widget名称,注没有后缀名	
	 * @return boolean
	 */
	public static function widget($widget_name) {
		
		//参数判断
		if (!$widget_name) {
			return false;
		}
		
		//分析widget名称
		$widget_name = ucfirst(strtolower($widget_name)) . 'Widget';				
	    doit::singleton($widget_name)->render();
	    
	    return true;
	}
		
	/**
	 * 加载并显示视图片段文件内容
	 * 
	 * 相当于include 代码片段，当$return为:true时返回代码代码片段内容,反之则显示代码片段内容
	 * @access public
	 * @param string  $file_name 视图片段文件名称
	 * @param array   $_data     视图模板变量，注：数组型
	 * @param boolean $return    视图内容是否为返回，当为true时为返回，为false时则为显示. 默认为:false
	 * @return mixed
	 */
	public function render($file_name, $_data = array(), $return = false) {
		
		//参数分析
		if (!$file_name) {
			return false;
		}
		
		//分析视图文件的路径
		$view_file = $this->template($file_name);
		
		//模板变量赋值
		if (!empty($_data) && is_array($_data)) {			
			extract($_data, EXTR_PREFIX_SAME, 'data');
			unset($_data);
		}
		
		//获取$file_name所对应的视图片段内容
		ob_start();				
		include $view_file;				
		$content = ob_get_clean();
				
		if (!$return) {									
			echo $content;
		} else {						
			return $content;
		}
	}
	
	/**
	 * 实例化视图
	 * 
	 * 当视图文件格式为html时,需要使用class View 对视图文件行进行编译
	 * 本函数相当于class View 单例模式的实例化
	 * @access public
	 * @return boolean
	 */
	public function load_view() {
		
		$this->view = View::getInstance();
		
		if (!empty($this->theme)) {
			$this->view->view_dir = APP_ROOT . 'themes/' . $this->theme . DIRECTORY_SEPARATOR;
		}
		
		$this->view->layout = $this->layout;
		
		return true;
	}
	
	/**
	 * Ajax调用返回
	 * 
	 * 返回json数据,供前台ajax调用
	 * @param array $data	返回数组,支持数组
	 * @param string $info	返回信息, 默认为空
	 * @param boolean $status	执行状态, 1为true, 0为false
	 * @return string
	 */
	public function ajaxReturn($info = null, $status = 1, $data = array()) {
		
		$result 			= array();
		$result['status'] 	= $status;
		$result['info']		= (!is_null($info)) ? $info : '';
		$result['data']		= $data;
		
		header("Content-Type:text/html; charset=utf-8");
        exit(json_encode($result));
	}
	
	
	/**
	 * 第八部分：其它辅助性操作,初始化运行环境辅助方法
	 * @author tommy
	 * @version 1.0 2010-10-19
	 */
	
	/**
	 * stripslashes()的同功能操作
	 * 
	 * @access protected
	 * @param string $string 	所要处理的变量
	 * @return mixed			变量
	 */
	protected static function strip_slashes($string) {
		
		//参数分析.
		if (!$string) {
			return false;
		}
				
		if (!is_array($string)) {
			return stripslashes($string);			
		}
		
		foreach ($string as $key=>$value) {					
			$string[$key] = self::strip_slashes($value);
		}
					
		return $string;
	}
	
	/**
	 * addslashes()的同功能操作
	 * 
	 * @access protected
	 * @params string $string 	所要处理的变量
	 * @return mixed			变量
	 */
	protected static function add_slashes($string) {

		//参数分析.
		if (!$string) {
			return false;
		}
		
		if (!is_array($string)) {
			return addslashes($string);			
		}
		
		foreach ($string as $key=>$value) {				
			$string[$key] = self::add_slashes($value);
		}
					
		return $string;
	}
}