<?php
/**
 * file_list class file
 * 
 * 用于文件夹内容的读取,复制,剪切等操作
 * @author tommy <streen003@gmail.com>
 * @copyright  Copyright (c) 2010 Tommy Software Studio
 * @link http://www.doitphp.com
 * @license New BSD License.{@link http://www.opensource.org/licenses/bsd-license.php}
 * @version $Id: file_list.class.php 1.0 2010-12-1 20:45:01Z tommy $
 * @package lib
 * @since 1.0
 */

if (!defined('IN_DOIT')) {
	exit();
}

class file_list extends Base {
	
	/**
	 * 构造函数
	 * 
	 * @return void
	 */
	public function __construct() {
		
	}
	
	/**
	 * 分析文件夹是否存在
	 * 
	 * @param string $dir_name 所要操作的文件目录名
	 * @return string
	 */
	protected static function parse_dir($dir_name) {		
		
		if (!is_dir($dir_name)) {
			Controller::halt('The dir:' . $dir_name . ' is not exists!');
		}
				
		return $dir_name;
	}
	
	/**
	 * 分析目标目录的读写权限
	 * 
	 * @param string $dir_name	目标目录
	 * @return string
	 */
	protected static function dest_dir($dir_name) {		
		
		if (!is_dir($dir_name)) {			
			mkdir($dir_name, 0777);
		} else {
			if (!is_writable($dir_name)) {
				chmod($dir_name, 0777);
			}
		}
				
		return $dir_name;
	}
	
	/**
	 * 获取目录内文件
	 * 
	 * @param string $dir_name	所要读取内容的目录名
	 * @return string
	 */
	public static function get_file_list($dir_name) {		
		
		$dir = self::parse_dir($dir_name);
		
		$handle = opendir($dir);
		
		$files = array();
		while (false !== ($file = readdir($handle))) {			
			if ($file == '.' || $file == '..' || $file == '.cvs' || $file == '.svn') {				
				continue;
			}			
			$files[] = $file;
		}
		
		closedir($handle);		
		
		return $files;
	}

	/**
	 * 将一个文件夹内容复制到另一个文件夹
	 * 
	 * @param string $source	被复制的文件夹名
	 * @param string $dest		所要复制文件的目标文件夹
	 * @return boolean
	 */
	public static function copy_dir($source, $dest) {
		
		//参数分析
		if (!$source || !$dest) {
			return false;
		}

		$parse_dir = self::parse_dir($source);
		$dest_dir = self::dest_dir($dest);
				
		$file_list = self::get_file_list($parse_dir);
		
		foreach ($file_list as $file) {			
			if (is_dir($parse_dir . '/' . $file)) {				
				self::copy_dir($parse_dir . '/' . $file, $dest_dir . '/' . $file);
			} else {				
				copy($parse_dir . '/' . $file, $dest_dir . '/' . $file);
			}
		}
				
		return true;
	}

	/**
	 * 移动文件夹, 相当于WIN下的ctr+x(剪切操作)
	 * 
	 * @param string $source	原目录名
	 * @param string $dest		目标目录
	 * @return boolean
	 */
	public static function move_dir($source, $dest) {

		//参数分析
		if (!$source || !$dest) {
			return false;
		}
				
		$parse_dir = self::parse_dir($source);
		$dest_dir = self::dest_dir($dest);
				
		$file_list = self::get_file_list($parse_dir);
		
		foreach ($file_list as $file) {			
			if (is_dir($parse_dir . '/' . $file)) {				
				self::move_dir($parse_dir . '/' . $file, $dest_dir . '/' . $file);
			} else {				
				if (copy($parse_dir . '/' . $file, $dest_dir . '/' . $file)) {					
					unlink($parse_dir . '/' . $file);
				}	
			}
		}
		
		rmdir($parse_dir);
				
		return true;
	}
	
	/**
	 * 删除文件夹
	 * 
	 * @param string $file_dir	所要删除文件的路径
	 * @return boolean
	 */
	public static function delete_dir($file_dir) {
		
		//参数分析
		if (!$file_dir){
			return false;
		}

		$parse_dir = self::parse_dir($file_dir);
				
		$file_list = self::get_file_list($parse_dir);
		
		foreach ($file_list as $file) {			
			if (is_dir($parse_dir . '/' . $file)) {						
				self::delete_dir($parse_dir . '/' . $file);
				rmdir($parse_dir . '/' . $file);
			} else {			
				unlink($parse_dir . '/' . $file);
			}
		}
				
		return true;
	}
}