<?php
/**
 * router class file
 * 
 * 网址路由分析
 * @author tommy <streen003@gmail.com>
 * @copyright  Copyright (c) 2010 Tommy Software Studio
 * @link http://www.doitphp.com
 * @license New BSD License.{@link http://www.opensource.org/licenses/bsd-license.php}
 * @version $Id: router.class.php 1.0 2010-12-4 22:10:01Z tommy $
 * @package lib
 * @since 1.0
 */

if (!defined('IN_DOIT')) {
	exit();
}

class router extends Base {
	
	/**
	 * 分析路由网址
	 * 
	 * @return string
	 */
	public static function get_url() {
		
		//当项目开启Rewrite设置时
		if (DOIT_REWRITE === false) {
			$path_url_string = strlen($_SERVER['SCRIPT_NAME']) > strlen($_SERVER['REQUEST_URI']) ? $_SERVER['SCRIPT_NAME'] : $_SERVER['REQUEST_URI'];
			$path_url_string = str_replace($_SERVER['SCRIPT_NAME'], '', $path_url_string);			
		} else {			
			$path_url_string = str_replace(str_replace('/index.php', '', $_SERVER['SCRIPT_NAME']), '', $_SERVER['REQUEST_URI']);
			//去掉伪静态网址后缀
			$path_url_string = str_replace(URL_SUFFIX, '', $path_url_string);
		}
		
		//如果网址(URL)含有'?'(问号),则过滤掉问号(?)及其后面的所有字符串
		$pos = strpos($path_url_string, '?');
		if ($pos !== false) {			
			$path_url_string = substr($path_url_string, 0, $pos);
		}
		
		return $path_url_string;
	}
	
	/**
	 * 路由参数信息分析
	 * 
	 * @param string $router	路由规则, 注:不支持数组
	 * @return array
	 * 
	 * @example
	 * 
	 * 例一:
	 * 当url为:http://127.0.0.1/doit/index.php/index/index/page-23
	 * 
	 * $router = $this->instance('router');
	 * $params = $router->get_params('<page:\d+>');
	 * $this->dump($params);
	 * 
	 * 运行结果:
	 * array(
	 * 	['page'] 	=> 23
	 * )
	 * 
	 * 
	 * 例二:
	 * 当url为:http://127.0.0.1/doit/index.php/index/index/id-12-page-23
	 * 
	 * $router = $this->instance('router');
	 * $params = $router->get_params('<id:\d><page:\d>');
	 * $this->dump($params);
	 * 
	 * 运行结果:
	 * array(
	 *	['id'] 		=> 12
	 * 	['page'] 	=> 23
	 * )
	 */
	public static function get_params($rules) {
		
		//参数分析
		if (!$rules || is_array($rules)) {
			return false;
		}
		
		$router_url = self::get_url();
		preg_match_all('/' . preg_replace('/<\w+:?(.*?)?>/', '(\1)', $rules) . '/', $router_url, $matches);
		preg_match_all('/<(\w+):?.*?>/', $rules, $matches2);
		
		$_params = array();
		foreach ((array)$matches2[1] as $key=>$value) {
			$_params[$value] = htmlspecialchars(trim($matches[0][$key]));
		}
		
		//清空不必要的内存占用
		unset($matches);
		unset($matches2);
		
		return $_params;
	}
	
	/**
	 * 生成含有路由信息的网址
	 * 
	 * @param string $route 	controller/action
	 * @param string $rules 	路由规则
	 * @param array  $params	参数与参数值
	 * @param string $pageEx	网址后缀, 默认为.html
	 * @param string $connector url中的分割符
	 * @return string
	 * 
	 * @example
	 * 
	 * 例一:
	 * $router = $this->instance('router');
	 * echo $router->create_url('index/index', '<id:\d+><page:\d+>', array('id'=>3, 'page'=>5));
	 * 
	 * 返回结果: /index.php/index/index/id-3-page-5.html
	 * 
	 */
	public static function create_url($route, $rules, $params, $connector = '-', $pageEx = '.html') {
		
		//参数分析.
		if (!$route || !$rules || !$params) {
			return false;
		}
		
		$url = Controller::get_base_url() . ((DOIT_REWRITE === false) ? 'index.php/' : '') . $route;
		
		preg_match_all('/<(\w+):?.*?>/', $rules, $matches);
		
		$url_array = array();
		foreach ($matches[1] as $name) {			
			$url_array[] = $name . $connector . (($params[$name]) ? trim($params[$name]) : 0);
		}	
		
		return $url . '/' . implode($connector, $url_array) . $pageEx;
	}
}