<?php
/**
 * @file			weibo.class.php
 * @CopyRight		(C)1996-2099 SINA Inc.
 * @Project			Xweibo
 * @Author			heli <heli1@staff.sina.com.cn>
 * @Author			yaoying <yaoying@staff.sina.com.cn>
 * @Create Date:	2012-07-02
 * @Modified By:	yaoying
 * @Brief			微博api操作类(OAuth 2.0)
 * @version $Id: weibo.class.php 18396 2012-08-02 03:26:46Z yaoying $
 */
require_once P_CLASS.'/conv_2/conv_2_base.class.php';

/**
 * 微博api操作类(Oauth 2.0操作兼容OAuth 1.0方法)
 * 为兼容xweibo的所有操作，特别约定：
 * OAuth 2.0的access token为xweibo session存储的oauth_token
 * OAuth 2.0的refresh token为xweibo session存储的oauth_token_secret
 */
 class weibo
 {
 	
	var $storage;
	var $format = 'json';

	/**
	 * Contains the last HTTP status code returned. 
	 *
	 * @ignore
	 */
	public $http_code;
	
	/**
	 * Contains the last API call.
	 *
	 * @ignore
	 */
	public $url;
	
	/**
	 * 最后一次访问类型
	 *
	 * @ignore
	 */
	public $last_req_type;
	
	/**
	 * Set the useragnet.
	 *
	 * @ignore
	 */
	public $useragent = 'Sae T OAuth2 v0.1/Xweibo 2.5';
	
	public $http;
	
	var $logType = 'api';	
	
	/**
	 * @ignore
	 */
	public $client_id;
	/**
	 * @ignore
	 */
	public $client_secret;
	/**
	 * @ignore
	 */
	public $access_token;
	/**
	 * @ignore
	 */
	public $refresh_token;
	
	public $_refresh_token_ed = array();
	
	/**
	 * Set API URLS
	 */
	/**
	 * @ignore
	 */
	function accessTokenURL()  { return 'https://api.weibo.com/oauth2/access_token'; }
	/**
	 * @ignore
	 */
	function authorizeURL()    { return 'https://api.weibo.com/oauth2/authorize'; }
	
	
	/**
	 * 构造函数
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * @return
	 */
	function weibo()
	{
		$this->client_id = WB_AKEY;
		$this->client_secret = WB_SKEY;
		$this->setConfig();
		$this->http = APP::ADP('http');
	}


	/**
	 * 设置
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @return
	 */
	function setConfig()
	{
		$this->setToken(1);
	}

	/**
	 * 手动设置Token
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $type 身份类型，1代表当前用户，2代表随机代理用户（如果不存在就使用内置账户），3代表指定用户，4代表内置账户，默认是1
	 * @param $oauth_token
	 * @param $oauth_token_secret
	 * @return unknown
	 */
	function setToken($type = 1, $oauth_token = null, $oauth_token_secret = null)
	{
		if ($type == 1) {
			$token = USER::getOAuthKey(true);
			$this->access_token = isset($token['oauth_token']) ? $token['oauth_token'] : '';
			$this->refresh_token = isset($token['oauth_token_secret']) ? $token['oauth_token_secret'] : '';
		} elseif ($type == 2) {
			$token = DS('accountProxy.getRandomAccount');
			if ($token) {
				$this->access_token = isset($token['token']) ? $token['token'] : '';
				$this->refresh_token = isset($token['secret']) ? $token['secret'] : '';			
			} else {
			$this->access_token = WB_USER_OAUTH_TOKEN;
			$this->refresh_token = WB_USER_OAUTH_TOKEN_SECRET;	
			}
		} elseif ($type == 3) {
			$this->access_token = $oauth_token;
			$this->refresh_token = $oauth_token_secret;	
		} elseif($type == 4) {
			$this->access_token = WB_USER_OAUTH_TOKEN;
			$this->refresh_token = WB_USER_OAUTH_TOKEN_SECRET;	
		}
		
	}
	

	/**
	 * (xweibo增加)获取access token和refresh token
	 * 请使用此方法，不要直接访问类属性！
	 * @param bool $oauth1_compact 是否采取同时返回兼容数组模式？
	 * 此时将设置两个oauth_token和oauth_token_secret，对应access_token和refresh_token
	 */
	public function getToken($oauth1_compact = true){
		$return = array(
			'access_token' => $this->access_token,
			'refresh_token' => $this->refresh_token,
		);
		if($oauth1_compact){
			$return['oauth_token'] = $this->access_token;
			$return['oauth_token_secret'] = $this->refresh_token;
		}
		return $return;
	}
	
	/**
	 * 手动设置Client信息
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $type 类型，1代表内置key，2代表指定key
	 * @param $appkey
	 * @param $appsecret
	 * @return unknown
	 */
	function setClientId($type = 1, $appkey = '', $appsecret = '', $clearToken = false){
		if(1 == $type){
			$this->client_id = WB_AKEY;
			$this->client_secret = WB_SKEY;
		}else{
			$this->client_id = $appkey;
			$this->client_secret = $appsecret;			
		}
		if($clearToken){
			$this->setToken(3, null, null);
		}
	}
	

	/**
	 * 获取client id属性
	 * @param string $name 名称，client_id、client_secret或者其它任意值
	 * @param string|array client_id或者client_secret时会返回字符串，否则将返回此两者信息组合的数组
	 */	
	public function getClientId($name = null){
		if('client_id' == $name){
			return $this->client_id;
		}elseif('client_secret' == $name){
			return $this->client_secret;
		}else{
			return array(
				'client_id' => $this->client_id,
				'client_secret' => $this->client_secret,
			);
		}
	}
		
	
	//数据集(timeline)接口

	/**
	 * 获取最新更新的公共微博消息
	 * @see http://open.weibo.com/wiki/Statuses/public_timeline
	 * @see http://open.weibo.com/wiki/2/statuses/public_timeline
	 * 
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			未审核无法使用：'errno' => 21321, 'err' => '(applications over the unaudited use restrictions!)'
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		param_change 参数增加page
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param bool $base_app 是否只获取当前应用的数据
	 * @param bool $oauth 是否使用oauth方式请求api
	 * @param int $count 条数，默认20
	 * @return array
	 */
	 function getPublicTimeline($base_app = '0', $oauth = true, $count = 20, $page = 1)
	 {
		$url = $this->oauth2_url('statuses/public_timeline');
		$params = array();
		$params['base_app'] = $base_app;
		$params['count'] = $count;
		$params['page'] = $page;

		if ($oauth == false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		$this->oauth2_compact_convert($response, 'common_only_statuses');
		return $response;
	 }


	/**
	 * @see weibo::getFriendsTimeline()
	 */
	 function getHomeTimeline($count = null, $page = null, $since_id = null, $max_id = null, $base_app = '0')
	 {
		return $this->getFriendsTimeline($count, $page, $since_id, $max_id, $base_app);
	 }


	/**
	 * 获取当前用户所关注用户的最新微博信息
	 * @see http://open.weibo.com/wiki/Statuses/friends_timeline
	 * @see http://open.weibo.com/wiki/2/statuses/friends_timeline
	 *
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param int|string $since_id 返回比since_id大的微博数据
	 * @param int|string $max_id 返回不大于max_id的微博数据
	 * @param int|string $pub_type 返回某一发布类型结果的微博  类型有全部-0, 原创-1, 转发-2默认返回全部
	 * @param int|stirng $feature 返回某一内容类型结果的微博 类型有全部-0, 图片-1, 音乐-2, 视频-3, 纯文本-4,
	 * @param int $trim_user 返回值中user字段开关，0：返回完整user字段、1：user字段仅返回user_id，默认为0。
	 * 默认返回全部
	 * @return RST_Array
	 */
	 function getFriendsTimeline($count = null, $page = null, $since_id = null, $max_id = null, $base_app = '0', $feature = 0, $trim_user = 0)
	 {
		$url = $this->oauth2_url('statuses/friends_timeline');
		$params = array();
		$params['base_app'] = $base_app;
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}
		if ($feature) {
			$params['feature'] = $feature;
		}
	 	if ($trim_user) {
			$params['trim_user'] = $trim_user;
		}		

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_statuses');
		return $response;
	 }


	/**
	 * 获取用户发布的微博信息列表
	 * @see http://open.weibo.com/wiki/Statuses/user_timeline
	 * @see http://open.weibo.com/wiki/2/statuses/user_timeline
	 *
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			未审核无法使用：'errno' => 21321, 'err' => '(applications over the unaudited use restrictions!)'
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * oauth 2.0存在问题：
	 * 			无法用此接口判定该人是否存在——“参数uid与screen_name都没有或错误，则默认返回当前登录用户的数据”
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param int|string $since_id 返回比since_id的的微博数据
	 * @parma int|string $max_id 返回不大于max_id的微博数据
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param bool $oauth 是否使用oauth方式访问api
	 * @return array
	 */
	 function getUserTimeline($id = null, $user_id = null, $name = null, $since_id = null, $max_id = null, $count = null, $page = null, $feature = 0, $oauth = true, $base_app = '0')
	 {

		$url = $this->oauth2_url('statuses/user_timeline');
	 	
		$params = array();		
	 	
		$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}
		
		$params['base_app'] = $base_app;
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}
		if ($feature) {
			$params['feature'] = $feature;
		}
		
		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		$this->oauth2_compact_convert($response, 'common_only_statuses');
		return $response;
	 }


	 /**
	  * 获取@当前用户的微博列表
	  *
	  * @see http://open.weibo.com/wiki/Statuses/mentions
	  * @see http://open.weibo.com/wiki/2/statuses/mentions
	  * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口存在参数无效问题，$filter_by_source等于1或者2均无效
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	  * 
	  * @param int $count 获取条数
	  * @param int $page 页码数
	  * @param int|string $since_id 返回比since_id大的微博数据
	  * @param int|string $max_id 返回不大于max_id的微博数据
	  * @param int $filter_by_source （参数增加）Xweibo中默认为0。来源筛选类型，0：全部、1：来自微博、2：来自微群。
	  * @return array
	  */
	 function getMentions($count = null, $page = null, $since_id = null, $max_id = null, $filter_by_source = 0)
	 {
		$url = $this->oauth2_url('statuses/mentions');

		$params = array();
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}
		$params['filter_by_source'] = $filter_by_source;

		$response = $this->oAuthRequest($url, 'get', $params);
		
		$this->oauth2_compact_convert($response, 'common_only_statuses');
		return $response;
	 }


	/**
	 * 获取当前用户发送及收到的评论列表
	 * @see http://open.weibo.com/wiki/Statuses/comments_timeline
	 * @see http://open.weibo.com/wiki/2/comments/timeline
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param int|string $since_id 返回比since_id大的微博数据
	 * @param int|string $max_id 返回不大于max_id的微博数据
	 * @return array
	 */
	 function getCommentsTimeline($count = null, $page = null, $since_id = null, $max_id = null)
	 {
		$url = $this->oauth2_url('comments/timeline');

		$params = array();
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'comments'));
		return $response;
	 }


	/**
	 * 获取当前用户发出的评论
	 * @see http://open.weibo.com/wiki/Statuses/comments_by_me
	 * @see http://open.weibo.com/wiki/2/comments/by_me
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param int|string $since_id 返回比since_id大的微博数据
	 * @param int|string $max_id 返回不大于max_id的微博数据
	  * @param int $filter_by_source （参数增加）Xweibo中默认为0。来源筛选类型，0：全部、1：来自微博、2：来自微群。
	 * @return array
	 */
	 function getCommentsByMe($count = null, $page = null, $since_id = null, $max_id = null, $filter_by_source = 0)
	 {
		$url = $this->oauth2_url('comments/by_me');

		$params = array();

		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}
		$params['filter_by_source'] = $filter_by_source;

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'comments'));
		return $response;
	 }


	/**
	 * 获取当前用户收到的评论列表
	 * @see http://open.weibo.com/wiki/2/comments/to_me
	 * @see http://open.weibo.com/wiki/Statuses/comments_to_me
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param int|string $since_id 返回比since_id大的微博数据
	 * @param int|string $max_id 返回不大于max_id的微博数据
	  * @param int $filter_by_source （参数增加）Xweibo中默认为0。来源筛选类型，0：全部、1：来自微博、2：来自微群。
	 * @return array
	 */
	function getCommentsToMe($count = null, $page = null, $since_id = null, $max_id = null, $filter_by_source = 0)
	{
		$url = $this->oauth2_url('comments/to_me');

		$params = array();
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}
		$params['filter_by_source'] = $filter_by_source;
		
		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'comments'));
		return $response;
	}
	
	/**
	 * 获取指定微博的评论列表
	 * @see http://open.weibo.com/wiki/2/comments/show
	 * @see http://open.weibo.com/wiki/Statuses/comments
	 *
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $id 微博id
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @return array
	 */
	 function getComments($id, $count = null, $page = null)
	 {
		$url = $this->oauth2_url('comments/show');

		$params = array();
		$params['id'] = $id;

		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'comments'));
		return $response;
	 }


	/**
	 * 批量获取一组微博的评论数及转发数
	 * @see http://open.weibo.com/wiki/Statuses/counts
	 * @see http://open.weibo.com/wiki/2/statuses/count
	 * 
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			未审核无法使用：'errno' => 21321, 'err' => '(applications over the unaudited use restrictions!)'
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口的api文档有问题：wiki上说返回reposts，但实质和oauth1.0返回一样的结果
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变（？）
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param string|array $ids 微博id,多个id以数组形式传入
	 * @param bool $oauth
	 * @return array
	 */
	 function getCounts($ids, $oauth = true)
	 {
		$url = $this->oauth2_url('statuses/counts');
		
		$params = array();
		if (is_array($ids)) {
			$params['ids'] = implode(',', $ids);
		} else {
			$params['ids'] = $ids;
		}
		
		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		$this->oauth2_compact_convert($response, 'getCounts');
		return $response;
	 }


	/**
	 * 获取当前用户未读消息数
	 * 
	 * @see http://open.weibo.com/wiki/Statuses/unread
	 * @see http://open.weibo.com/wiki/2/remind/unread_count
	 *
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口和其它接口不同，部署了其它地方的域名
	 * 
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *		param_change 参数改变
	 *		param_change 参数失效$with_new_status和$since_id
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $with_new_status 默认为0。1表示结果包含是否有新微博，0表示结果不包含是否有新微博
	 * @param int|string $since_id 微博id，返回此条id之后，是否有新微博产生，有返回1，没有返回0
	 * @return array
	 */
	 function getUnread($with_new_status = null, $since_id = null, $uid = 0)
	 {
		$url = 'https://rm.api.weibo.com/2/remind/unread_count.json';
		
		if($uid == 0){
			$uid = USER::uid();
		}
		$params['uid'] = $uid;
		
		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'getUnread');
		return $response;
	 }
	 
	/**
	 * 设置某个用户某个新消息的未读数为0
	 * @see http://open.weibo.com/wiki/Statuses/reset_count
	 * @see http://open.weibo.com/wiki/2/remind/set_count
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			高权限也无效，需要单独申请：array ( 'rst' => false, 'errno' => 10014, 'err' => '(Insufficient app permissions!)', 'level' => 0, 'log' => '', ) 
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口和其它接口不同，部署了其它地方的域名
	 * 			警告：此接口为高权限。
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *
	 * - oauth2_convert 
	 * 		true（无高权限，无法验证）
	 *
	 * @param string $type 1--评论数，2--@数，3--私信数，4--关注我的数。还有OAuth2有其它新增值
	 * @return array
	 */
	function resetCount($type = 1)
	{
		$url = 'https://rm.api.weibo.com/2/remind/set_count.json';
		$params = array();
		//转换
		switch($type){
			case 1:
				$type = 'cmt';
				break;
			case 2:
				$type = 'mention_status';
				break;
			case 3:
				$type = 'dm';
				break;
			case 4:
				$type = 'follower';
				break;
			default:
				break;
		}
		$params['type'] = $type;
		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	}
	
	 //访问接口

	/**
	 * 根据ID获取单条微博信息内容
	 * @see http://open.weibo.com/wiki/Statuses/show
	 * @see http://open.weibo.com/wiki/2/statuses/show
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string 微博id
	 * @return array
	 */
	 function getStatuseShow($id)
	 {
	 	$url = $this->oauth2_url('statuses/show');
		
		$params = array('id'=>$id);
		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	 }

	/**
	 * 批量获取微博信息
	 * 根据提供的MID批量获取一组微博的信息
	 * @see http://open.weibo.com/wiki/Statuses/batch_show
	 * @see http://open.weibo.com/wiki/2/statuses/show_batch
	 * 
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			未审核无法使用：'errno' => 21321, 'err' => '(applications over the unaudited use restrictions!)'
	 * 			高权限也无效，需要单独申请：array ( 'rst' => false, 'errno' => 10014, 'err' => '(Insufficient app permissions!)', 'level' => 0, 'log' => '', ) 
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口为高权限。
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param string|int $ids 根据指定id返回微博的相应内容
	 * @param int $del_ctrl (参数已无效?)是否打开estate字段的开关。1：打开开关，则在微博被删除时返回带estate字段，值为deleted。不带此开关默认为关闭，返回不带estate字段
	 * @return array
	 */
	function getStatusesBatchShow($ids, $del_ctrl = true, $oauth = true)
	{
		$url = $this->oauth2_url('statuses/show_batch');
		$params = array();
		
		$params['ids'] = $ids;
	
		if($del_ctrl) {
			$params['del_ctrl'] = $del_ctrl;
		}

		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		$this->oauth2_compact_convert($response, 'common_only_statuses');
		return $response;
	}
	
	/**
	 * 发布一条微博信息
	 * @see http://open.weibo.com/wiki/Statuses/update
	 * @see http://open.weibo.com/wiki/2/statuses/update
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变(此接口已不支持转发微博)
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param string $status 微博内容
	 * @return array
	 */
	 function update($status)
	 {
		$url = $this->oauth2_url('statuses/update');
		$params = array();
		//$params['status'] = urlencode($status);
		$params['status'] = $status;
		
		$response = $this->oAuthRequest($url, 'post', $params);
		
		return $response;
	 }
	 
	 /**
	  * 上传图片并发布一条微博信息
	  * @see http://open.weibo.com/wiki/Statuses/upload
	  * @see http://open.weibo.com/wiki/2/statuses/upload
	  * 
	 * - oauth2_convert 
	 *		true（改造完毕）
	  *
	  * @param string $status 微博内容
	  * @param string $pic 图片路径
	  * @param string $lat 纬度
	  * @param string $long 经度
	  * @return array
	  */
	 function upload($status, $pic, $lat = null, $long = null)
	 {
		$url = $this->oauth2_url('statuses/upload');

		$params = array();
		//$params['status'] = urlencode($status);
		$params['status'] = $status;
		$params['pic'] = '@'.$pic;
		
		if ($lat) {
			$params['lat'] = $lat;
		}
		if ($long) {
			$params['long'] = $long;
		}
		$response = $this->oAuthRequest($url, 'post', $params, true);
		return $response;
	 }

	/**
	 * 上传图片，返回pcid，缩略图，原图
	 * 
	 * @see http://open.weibo.com/wiki/Statuses/upload_pic
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回599999
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param string $pic 上传的图片路径
	 * @return array
	 */
	 function uploadPic($pic)
	 {
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	 }

	 /**
	  * 发布图片微博
	  * @see http://open.weibo.com/wiki/Statuses/upload_url_text
	  * @see http://open.weibo.com/wiki/2/statuses/upload_url_text
	  * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口参数已变更，不再支持$picid，会令xweibo上传图片微博流程产生很大影响
	 * 
	 * - oauth2_convert 
	 *		param_change 参数失效 不再支持$picid
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	  * 
	  * @param string $status 微博内容
	  * @param string $picid （警告：参数已不支持！）图片id
	  * @param string $picurl 图片url 必须是以http:://开头
	  * @return array
	  */
	 function uploadUrlText($status, $picid = null, $picurl = null)
	 {
		$url = $this->oauth2_url('statuses/upload_url_text');

		$params = array();
		//$params['status'] = urlencode($status);
		$params['status'] = $status;
		/*
		if ($picid) {
			$params['pic_id'] = $picid;
		}
		*/
		if ($picurl) {
			$params['url'] = $picurl;
		}
		
		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }


	/**
	 * 删除微博
	 * @see http://open.weibo.com/wiki/Statuses/destroy
	 * @see http://open.weibo.com/wiki/2/statuses/destroy
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $id 微博id
	 * @return array
	 */
	 function destroy($id)
	 {
		$url = $this->oauth2_url('statuses/destroy');

		$params = array('id'=>$id);

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }
	 
	/**
	 * 转发一条微博信息（可加评论）
	 * 
	 * @see http://open.weibo.com/wiki/Statuses/repost
	 * @see http://open.weibo.com/wiki/2/statuses/repost
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int|string 微博id
	 * @param string $status 微博内容
	 * @param int $is_comment 是否在转发的同时发表评论
	 * @return array
	 */
	 function repost($id, $status = null, $is_comment = 0)
	 {
		$url = $this->oauth2_url('statuses/repost');

		$params = array();
		$params['id'] = $id;
		if ($status) {
			//$params['status'] = urlencode($status);
			$params['status'] = $status;
		}
		$params['is_comment'] = $is_comment;

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }


	/**
	 * 对一条微博信息进行评论
	 * @see http://open.weibo.com/wiki/Statuses/comment
	 * @see http://open.weibo.com/wiki/2/comments/create
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口的api文档有问题：$comment_ori无效
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int|string $id 微博id
	 * @param string $comment 评论内容
	 * @param int|string $cid 要评论的评论id
	 * @return array
	 */
	 function comment($id, $comment, $cid = null, $comment_ori = 0)
	 {
	 	if($cid > 0){
	 		return $this->reply($id, $cid, $comment, 0, $comment_ori);
	 	}
	 	
		$url = $this->oauth2_url('comments/create');
		
		$params = array();
		$params['id'] = $id;
		$params['comment'] = $comment;
		$params['comment_ori'] = $comment_ori;
		
		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }


	/**
	 * 删除当前用户的微博评论信息
	 * @see http://open.weibo.com/wiki/Statuses/comment_destroy
	 * @see http://open.weibo.com/wiki/2/comments/destroy
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|sting 评论id
	 * @return array
	 */
	 function commentDestroy($id)
	 {
		$url = $this->oauth2_url('comments/destroy');

		$params = array('cid'=>$id);
		
		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }

	 /**
	  * 批量删除评论
	  * @see http://open.weibo.com/wiki/Statuses/comment/destroy_batch
	  * @see http://open.weibo.com/wiki/2/comments/destroy_batch
	  * 
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口的api文档有问题：文档说参数为ids，但实质是cids
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	  *
	  * @param string $ids 评论id, 多个用逗号隔开(最多20个)
	  * @return array
	  */
	 function commentDestroyBatch($ids)
	 {
		$url = $this->oauth2_url('comments/destroy_batch');

		$params = array();
		if(is_array($ids)){
			$params['cids'] = join(',', $ids);
		}else{
			$params['cids'] = $ids;
		}
		$response = $this->oAuthRequest($url, 'post', $params);
		
		return $response;
	 }


	 /**
	  * 回复微博评论信息
	  * @see http://open.weibo.com/wiki/Statuses/reply
	  * @see http://open.weibo.com/wiki/2/comments/reply
	  * 
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口的api文档有问题：$comment_ori无效
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	  * 
	  * @param int|string 微博id
	  * @param int|string 评论id
	  * @param string $comment 回复评论内容
	  * @return array
	  */
	 function reply($id, $cid, $comment, $without_mention = 0, $comment_ori = 0)
	 {
		$url = $this->oauth2_url('comments/reply');

		$params = array();
		$params['id'] = $id;
		$params['comment'] = $comment;
		$params['cid'] = $cid;
		if($without_mention > 0){
			$params['without_mention'] = 1;
		}
	 	if($comment_ori > 0){
			$params['comment_ori'] = 1;
		}
		
		$response = $this->oAuthRequest($url, 'post', $params);
		
		return $response;
	 }



	 //用户接口

	/**
	 * 根据用户ID获取用户资料（授权用户）
	 * @see http://open.weibo.com/wiki/Users/show
	 * @see http://open.weibo.com/wiki/2/users/show
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			无
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param bool $oauth 是否用户oauth方式请求api
	 * @return array
	 */
	function getUserShow($id = null, $user_id = null, $name = null, $oauth = true)
	{
		$url = $this->oauth2_url('users/show');

		$params = array();
		
		$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}
		
		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		return $response;
	}
	
	
 	/**
	 * 批量获取用户资料（授权用户） 最多获取20个
	 * @see http://open.weibo.com/wiki/Users/batch_show
	 * 
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $user_id 用户user id, 例如1234,123
	 * @param string $name 用户昵称 例如1234,123
	 * @param $is_show 是否使用user/show作为替换使用
	 * @return array
	 * 
	 */
 	function getUsersBatchShow($user_id=null, $screen_name=null, $oauth=true,$is_show = false)
	{
		if(!$is_show){
			$url = $this->oauth2_url('users/show_batch');
	
			$params = array();
			if($user_id) {
				$params['uids'] = $user_id;
			}
			if($screen_name) {
				$params['screen_name'] = $screen_name;
			}
			
			if ($oauth === false) {
				$response = $this->sourceRequest($url, 'get', $params);
			}else{
				$response = $this->oAuthRequest($url, 'get', $params);
			}
			$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'users'));
			return $response;
		}else{
			if($user_id){
				$arr_user_id = explode(",", $user_id);
				$response_all = array();
				$response_all['users'] = array();
				foreach ($arr_user_id as $id_key=>$id_vale){
					$response  = $this->getUserShow($id_vale, null, null, true);
					$value = $response['rst'];
					array_push($response_all['users'],$value);
				}
				$response_all_user = RST($response_all);
				$this->oauth2_compact_convert($response_all_user, 'common_only_body', array('body'=>'users'));
				return $response_all_user;
			}
			if($screen_name){
				$arr_screen_name = explode(",", $string);
				$response_all = array();
				$response_all['users'] = array();
				foreach ($arr_screen_name as $name_key=>$name_vale){
					$response  = $this->getUserShow(null, null, $name_vale, true);
					$value = $response['rst'];
					array_push($response_all['users'],$value);
				}
				$response_all_user = RST($response_all);
				$this->oauth2_compact_convert($response_all_user, 'common_only_body', array('body'=>'users'));
				return $response;
			}
		}
	}


	/**
	 * 获取当前用户关注对象列表及最新一条微博信息
	 * @see http://open.weibo.com/wiki/Statuses/friends
	 * @see http://open.weibo.com/wiki/2/friendships/friends
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param string $cursor 分页位置
	 * @param int $count 获取条数
	 * @return array
	 */
	 function getFriends($id = null, $user_id = null, $name = null, $cursor = null, $count = null)
	 {
		$url = $this->oauth2_url('friendships/friends');

		$params = array();		

	 	$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}
		
		if ($cursor) {
			$params['cursor'] = $cursor;
		}
		if ($count) {
			$params['count'] = $count;
		}
		
		$params['trim_status'] = 0;  //xweibo适应性改动

		$response = $this->oAuthRequest($url, 'get', $params);
		return $response;
	 }
	 

	/**
	 * 获取当前用户粉丝列表及最新一条微博信息
	 * @see http://open.weibo.com/wiki/Statuses/followers
	 * @see http://open.weibo.com/wiki/2/friendships/followers
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param string $cursos 分页位置
	 * @param int $count 获取条数
	 * @return array
	 */
	 function getFollowers($id = null, $user_id = null, $name = null, $cursor = null, $count = null)
	 {
		$url = $this->oauth2_url('friendships/followers');
		
		$params = array();		

	 	$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}
		
		if ($cursor) {
			$params['cursor'] = $cursor;
		}
		if ($count) {
			$params['count'] = $count;
		}
		$params['trim_status'] = 0;  //xweibo适应性改动

		$response = $this->oAuthRequest($url, 'get', $params);
		
		return $response;
	 }

	/**
	 * 获取当前用户感兴趣的用户列表
	 * @see http://open.weibo.com/wiki/Users/suggestions
	 * @see http://open.weibo.com/wiki/2/suggestions/users/may_interested
	 *
	 * oauth 2.0存在问题：
	 * 			警告：此接口的返回内容已大幅度改变，兼容模式会导致1次额外的api消耗
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $cursor 分页位置
	 * @param int $count 获取条数
	 * @return array
	 */
	function getUserSuggestions($cursor = null, $count = null, $with_reason = 1)
	{
		$url = $this->oauth2_url('suggestions/users/may_interested');

		$params = array();
		if ($cursor) {
			$params['cursor'] = $cursor;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($with_reason) {
			$params['with_reason'] = $with_reason;
		}

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'getUserSuggestions', array('with_reason'=>$with_reason, 'weibo'=>$this));
		return $response;
	}

	/**
	 * 添加好友备注
	 * @see http://open.weibo.com/wiki/User/friends/update_remark
	 * @see http://open.weibo.com/wiki/2/friendships/remark/update
	 * 
	 * oauth 2.0存在问题：
	 * 			无法用此接口判定该人是否存在——“参数uid与screen_name都没有或错误，则默认返回当前登录用户的数据”
	 * 			警告：此接口为高权限。
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $id 用户id
	 * @param string $remark 备注
	 * @return array
	 */
	 function updateFriendRemark($id, $remark)
	 {
		$url = $this->oauth2_url('friendships/remark/update');
		
		$params = array();
		if ($id) {
			$params['uid'] = $id;
		}
		if ($remark) {
			$params['remark'] = $remark;
		}

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }


	/**
	 *  返回系统推荐的用户列表
	 * @see http://open.weibo.com/wiki/Users/hot
	 * @see http://open.weibo.com/wiki/2/suggestions/users/hot
	 * 
	 * @param string $category 分类，返回某一类别的推荐用户，默认为default。如果不在以下分类中，返回空列表
     * default：人气关注
     * ent：影视名星
     * hk_famous：港台名人
     * model：模特
     * cooking：美食&健康
     * sport：体育名人
     * finance：商界名人
     * tech：IT互联网
     * singer：歌手
     * writer：作家
     * moderator：主持人
     * medium：媒体总编
     * stockplayer：炒股高手
	 *
	 * @return array
	 */
	 function getUsersHot($category = 'default')
	 {
		$url = $this->oauth2_url('suggestions/users/hot');

		$params = array();
		$params['category'] = $category;

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	 }

	 //私信接口

	/**
	 * 获取当前用户最新私信列表
	 * 
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param int|string $since_id 返回比since_id大的微博数据
	 * @param int|string $max_id 返回不大于max_id的微博数据
	 * @return array
	 */
	 function getDirectMessages($count = null, $page = null, $since_id = null, $max_id = null)
	 {
		$url = $this->oauth2_url('direct_messages');
		
		$params = array();
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'direct_messages'));
		
		return $response;
	 }


	/**
	 * 获取当前用户发送的最新私信列表
	 *
	 * @param int $count 获取条数
	 * @param int $page 页码数
	 * @param int|string $since_id 返回比since_id大的微博数据
	 * @param int|string $max_id 返回不大于max_id的微博数据
	 * @return array
	 */
	 function getSentDirectMessages($count = null, $page = null, $since_id = null, $max_id = null)
	 {
		$url = $this->oauth2_url('direct_messages/sent');

		$params = array();
		if ($since_id) {
			$params['since_id'] = $since_id;
		}
		if ($max_id) {
			$params['max_id'] = $max_id;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'direct_messages'));

		return $response;
	 }


	/**
	 * 发送一条私信
	 * 
	 * @param int|string $d 用户id。查询优先级最高。如果此参数大于0，则覆盖$user_id和$name
	 * @param string $text 私信内容
	 * @param string $name 用户昵称 查询最低。
	 * @param int $user_id 用户user id 查询优先级低于$user_id。如果此参数大于0，则覆盖$name
	 * @return array
	 */
	 function sendDirectMessage($id, $text, $name = null, $user_id = null)
	 {
		$url = $this->oauth2_url('direct_messages/new');

		$params = array();
		$params['text'] = $text;
		
	 	$final_uid = 0;
	 	if($id >= 1){
	 		$final_uid = $id;
	 	}elseif($user_id >= 1){
	 		$final_uid = $user_id;
	 	}
	 	if ($final_uid >= 1) {
			$params['uid'] = $final_uid;
		}elseif ($name) {
			$params['screen_name'] = $name;
		}

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }


	/**
	 * 删除一条私信
	 *
	 * @param int|string 私信id
	 * @return array
	 */
	 function deleteDirectMessage($id)
	 {
		$url = $this->oauth2_url('direct_messages/destroy');
		
		$params = array('id'=>$id);
		
		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }



	 //关注接口

	/**
	 * 关注某用户
	 * @see http://open.weibo.com/wiki/Friendships/create
	 * @see http://open.weibo.com/wiki/2/friendships/create
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param string $follow
	 * @return array
	 */
	 function createFriendship($id = null, $user_id = null, $name = null, $follow = null)
	 {
		$url = $this->oauth2_url('friendships/create');
		
		$params = array();
		
	 	$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}
		
		if ($follow) {
			$params['follow'] = $follow;
		}

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }

	 /**
	  * 批量添加关注
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口为高权限。
	  * 
	  * @param string $ids 用户id, 多个用逗号隔开(最多20个)
	  * @return array
	  */
	 function createFriendshipBatch($ids)
	 {
		$url = $this->oauth2_url('friendships/create_batch');
		$params = array();
		if(is_array($ids)){
			$params['uids'] = join(',', $ids);
		}else{
			$params['uids'] = $ids;
		}
		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }


	/**
	 * 取消关注或移除粉丝
	 * @see http://open.weibo.com/wiki/Friendships/destroy
	 * @see http://open.weibo.com/wiki/2/friendships/destroy
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $user_id 用户user id 查询优先级最高。如果此参数存在，则覆盖$name
	 * @param string $name 用户昵称 查询最低。
	 * @param int $is_follower （注意此参数已失效！）默认为0。1表示为移除粉丝，0表示为取消关注
	 * @return array
	 */
	 function deleteFriendship($user_id = null, $name = null, $is_follower = 0)
	 {
		$url = $this->oauth2_url('friendships/destroy');

		$params = array();
		$final_user = '';
		if (!empty($user_id)) {
			$final_user = $user_id;
		}elseif (!empty($name)) {
			$final_user = $name;
		}
		if(is_numeric($final_user)){
			$params['uid'] = $final_user;
		}elseif(!empty($final_user)){
			$params['screen_name'] = $final_user;
		}

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	 }

	/**
	 * 判断两个用户是否有关注关系
	 * @see http://open.weibo.com/wiki/Friendships/exists
	 * @see [OAuth_2_NOT_FOUND]
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。使用getFriendship进行替代转换
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，使用其它接口替代
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param int|string $user_a 要判断的用户UID
	 * @param int|string $user_b 要判断的被关注人用户UID
	 * @return array
	 */
	 function existsFriendship($user_a, $user_b)
	 {
		$response = $this->getFriendship($user_b, null, $user_a, null);
		$this->oauth2_compact_convert($response, 'existsFriendship');
		return $response;
	 }

	/**
	 * 获取两个用户关系的详细情况
	 * @see http://open.weibo.com/wiki/Friendships/show
	 * @see http://open.weibo.com/wiki/2/friendships/show
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $target_id 要判断的目的用户UID
	 * @param string $target_screen_name 要判断的目的微博昵称
	 * @param int $source_id 源用户UID
	 * @param string $source_screen_name 源微博昵称
	 * @return array
	 */
	 function getFriendship($target_id = null, $target_screen_name = null, $source_id = null, $source_screen_name = null)
	 {
		$url = $this->oauth2_url('friendships/show');

		$params = array();
		if ($target_id) {
			$params['target_id'] = $target_id;
		}
		if ($target_screen_name) {
			$params['target_screen_name'] = $target_screen_name;
		}
		if ($source_id) {
			$params['source_id'] = $source_id;
		}
		if ($source_screen_name) {
			$params['source_screen_name'] = $source_screen_name;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	 }

	/**
	 * 批量判断用户关注信息
	 * 判断当前登录用户是否关注批量提供的用户。如果当前用户关注其中某一用户，则返回其ID
	 * 
	 * oauth 2.0存在问题：
	 * 			注意：此接口在xweibo标准版中曾经使用，但后来废弃
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param string|int $uids 指定等待判断是否已经关注的用户id列表 默认20
	 * @return array
	 */
	function getFriendshipsBatchExists($uids)
	{
		$url = $this->oauth2_url('friendships/batch_exists');
		$params = array();

		$params['uids'] = $uids;

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	}


	 //Social Graph接口

	/**
	 * 获取用户关注对象uid列表
	 * @see http://open.weibo.com/wiki/Friends/ids
	 * @see http://open.weibo.com/wiki/2/friendships/friends/ids
	 *
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param string $cursor 分页的位置
	 * @param int 获取条数
	 * @return array
	 */
	 function getFriendIds($id = null, $user_id = null, $name = null, $cursor = null, $count = null)
	 {
		$url = $this->oauth2_url('friendships/friends/ids');

		$params = array();

	 	$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}		
		
		if ($cursor) {
			$params['cursor'] = $cursor;
		}
		if ($count) {
			$params['count'] = $count;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	 }


	/**
	 * 获取用户粉丝对象uid列表
	 * @see http://open.weibo.com/wiki/Followers/ids
	 * @see http://open.weibo.com/wiki/2/friendships/followers/ids
	 * 
	 * @param int|string $id 用户id。（用于uid中）查询优先级最高。如果此参数存在，则覆盖$user_id和$name
	 * @param int|string $user_id 用户user id （用于uid中）查询优先级低于$user_id。如果此参数存在，则覆盖$name
	 * @param string $name （用于screen_name中）用户昵称 查询最低。
	 * @param string $cursor 分页的位置
	 * @param int $count 获取条数
	 * @return array
	 */
	 function getFollowerIds($id = null, $user_id = null, $name = null, $cursor = null, $count = null)
	 {
		$url = $this->oauth2_url('friendships/followers/ids');

		$params = array();
		
	 	$param_uid = !empty($id) ? $id : (!empty($user_id) ? $user_id : null);
		$param_screen_name = !empty($name) ? $name : null;
		if(!empty($param_uid)){
			$params['uid'] = $param_uid;
		}elseif(!empty($param_screen_name)){
			$params['screen_name'] = $param_screen_name;
		}
		
		if ($cursor) {
			$params['cursor'] = $cursor;
		}
		if ($count) {
			$params['count'] = $count;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	 }

   /**
	* 获取用户优质粉丝列表，每次最多返回20条，包括用户的最新的微博
	* @see http://open.weibo.com/wiki/Statuses/magic_followers
	* @see http://open.weibo.com/wiki/2/friendships/followers/active
	* 
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			警告：疑似已经不允许（文档指明不允许登录运行） 
	 * 
	 * oauth 2.0存在问题：
	 * 			此接口的api文档有问题：文档中的示例不以users节点返回，但实质上和OAuth 1.0保持一致
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	* 
	* @param int|string $user_id 用户user id
	* @param int $count 获取条数
	* @param bool $oauth 是否要用身份认证 默认为true需要, false为不需要
	* @return array
	*/
	function getMagicFollowers($user_id, $count = null, $oauth = true)
	{
		$url = $this->oauth2_url('friendships/followers/active');
		
		$params = array();
		
		$params['uid'] = $user_id;
		if ($count) {
			$params['count'] = $count;
		}

		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		
		return $response;
	}


	 //帐号接口

	/**
	 * 验证当前用户身份是否合法
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 			请注意使用OAuth 2.0后，此处频次受到了极大限制！
	 * 
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，使用其它接口替代
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @see http://open.weibo.com/wiki/Account/verify_credentials
	 * @see http://open.weibo.com/wiki/2/account/get_uid
	 *
	 * @param bigint $uid 当前uid用户
	 * @return array
	 */
	 function verifyCredentials($uid = 0)
	 {
	 	//尝试使用session的uid
	 	if($uid <= 0){
	 		$uid = USER::uid();
	 	}
	 	
		return $this->getUserShow(null, $uid, null, true);
	 }
	 
	 /**
	  * OAuth授权之后，获取授权用户的UID
	  * @see http://open.weibo.com/wiki/2/account/get_uid
	  * @return array
	  */
	 function get_uid(){
		$url = $this->oauth2_url('account/get_uid');
		$params = array();
		$response = $this->oAuthRequest($url, 'get', $params);
		return $response;
	 }
	 
	/**
	 * 获取当前用户API访问频率限制
	 * @see http://open.weibo.com/wiki/Account/rate_limit_status
	 * @see http://open.weibo.com/wiki/2/account/rate_limit_status
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @return array
	 */
	 function getRateLimitStatus()
	 {
		$url = $this->oauth2_url('account/rate_limit_status');
		
		$params = array();
		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	 }


	/**
	 * 当前用户退出登录
	 * @see http://open.weibo.com/wiki/Account/end_session
	 * @see http://open.weibo.com/wiki/2/account/end_session
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @return array
	 */
	 function endSession()
	 {
		$url = $this->oauth2_url('account/end_session');
		$response = $this->oAuthRequest($url, 'get');
		return $response;
	 }


	/**
	 * 更改头像
	 *
	 * @param string $image 头像路径
	 * @return array
	 */
	 function updateProfileImage($image)
	 {
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	 }


	/**
	 * 更改资料
	 *
	 * @param array $params array('name' => string,
	 *								'gender' => string,
	 *								'province' => int|string,
	 *								'city' => int|string,
	 *								'description' => string)
	 * @return array
	 */
	 function updateProfile($params)
	 {
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	 }


	/**
	 * 注册新浪微博帐号
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 *
	 * @param array $params array('nick' => string,
	 *								'gender' => string,
	 *								'password' => string,
	 *								'email' => string,
	 *								'province' => int|string,
	 *								'city' => int|string,
	 *								'ip' => string)
	 * @return array|string
	 */
	 function register($params)
	 {
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	 }

	/**
	 * 二次注册新浪微博帐号
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 *
	 * @param array $params array('uid' => string,
	 *								'nickname' => string,
	 *								'gender' => string,
	 *								'email' => string,
	 *								'province' => int|string,
	 *								'city' => int|string,
	 *								'ip' => string)
	 * @return array|string
	 */
	 function activate($params)
	 {
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	 }

	/**
	 * 设置隐私信息
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param array $params array('comment' => int|string 谁可以评论当前用户的微博 0所有 1我关注的人 默认0,
	 *								'message' => int|string 谁可以给当前用户发私信 0所有 1我关注的人 默认0,
	 *								'realname' => int|string 是否允许别人通过真实名称搜索到我 0允许 1不允许 默认1,
	 *								'geo' => int|string 发布微博 是否允许微博保存并显示所处的地理位置 0允许 1不允许
	 *								默认0,
	 *								'badge' => int|string 勋章展现状态，值—1私密状态，0公开状态，默认值0)
	 * @return array|string
	 */
	 function updatePrivacy($params)
	 {
		$url = $this->oauth2_url('account/update_privacy');
		$response = $this->oAuthRequest($url, 'post', $params);
		return $response;
	 }

	/**
	 * 获取隐私信息
	 * @see http://open.weibo.com/wiki/Account/get_privacy
	 * @see http://open.weibo.com/wiki/2/account/get_privacy
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @return array|string
	 */
	 function getPrivacy()
	 {
		$url = $this->oauth2_url('account/get_privacy');
		$response = $this->oAuthRequest($url, 'get');
		$this->oauth2_compact_convert($response, 'getPrivacy');
		return $response;
	 }

   /**
	* 更新用户提醒设置
	* 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回虚值
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	* 
	* @param array $params array('comment' => int|string 新评论提醒，0—不提醒，1—提醒，默认值1,
	*							'follower' => int|string 新粉丝提醒，0—不提醒，1—提醒，默认值1,
	*							'dm' => int|string 新私信提醒，0—不提醒，1—提醒，默认值1,
	*							'mention' => int|string @提到我提醒，0—不提醒，1—提醒，默认值1,
	*							'from_user' => int|string 设置哪些提到我的微博计入提醒数，微博作者身份，0--所有人，1—关注的人,
	*							'status_type' => int|string 设置哪些提到我的微博计入提醒数，微博类型，0—所有微博，1—原创的微博)
	* @return array|string
	*/
	function updateNotice($params)
	{
		$response = RST(array());
		$this->oauth2_compact_convert($response, 'defNoticeVal', array('_force_pass'=>true));
		return $response;
	}


   /**
	* 获取用户提醒设置
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回虚值
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	* 
	* @return array|string
	*/
	function getNotice()
	{
		$response = RST(array());
		$this->oauth2_compact_convert($response, 'defNoticeVal', array('_force_pass'=>true));
		return $response;
	}
	
	 //收藏接口

	/**
	 * 获取当前用户的收藏列表
	 * @see http://open.weibo.com/wiki/2/favorites
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $page 页码数
	 * @return array
	 */
	 function getFavorites($page = null)
	 {
		$url = $this->oauth2_url('favorites');
		$params = array();
		if ($page) {
			$params['page'] = $page;
		}
		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'getFavorites');
		return $response;
	 }


	/**
	 * 添加收藏
	 * @see http://open.weibo.com/wiki/Favorites/create
	 * @see http://open.weibo.com/wiki/2/favorites/create
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string 微博id
	 * @return array
	 */
	 function createFavorite($id)
	 {
		$url = $this->oauth2_url('favorites/create');

		$params = array();
		$params['id'] = $id;
		$response = $this->oAuthRequest($url, 'post', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'status'));
		return $response;
	 }


	/**
	 * 删除当前用户收藏的微博信息
	 * @see http://open.weibo.com/wiki/Favorites/destroy
	 * @see http://open.weibo.com/wiki/2/favorites/destroy
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int $id 微博id
	 * @return array
	 */
	 function deleteFavorite($id)
	 {
		$url = $this->oauth2_url('favorites/destroy');
		$params = array('id' => $id);
		$response = $this->oAuthRequest($url, 'post', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'status'));
		return $response;
	 }


	 //oauth

    /**
     * 获取request token
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 			OAuth 2.0流程中不需要使用此步骤，请移除相关代码
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回虚值
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $oauth_callback
     * @return array a key/value array containing oauth_token and oauth_token_secret
     */
    function getRequestToken($oauth_callback = NULL)
	{
        $token = array('oauth_token' => 'REQUEST_TOKEN', 'oauth_token_secret' => 'REQUEST_TOKEN_SECRET');
        return RST($token);
    }

    /**
     * Get the authorize URL
     * @see http://open.weibo.com/wiki/Oauth2/authorize
     * 
	 * - oauth2_status: 
	 *		__code_internal__ 内部code，与外部无关
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 * - oauth2_convert:
	 * 		true（改造完毕）
	 * true 
	 * @param array|string $param 参数集合。已有参数：
	 * response_type 支持的值包括 code 和token 默认值为code
	 * state 用于保持请求和回调的状态。在回调时,会在Query Parameter中回传该参数。xweibo会将其打包为一个query值
	 * display 授权页面类型 可选范围: 
	 *  - default		默认授权页面		
	 *  - mobile		支持html5的手机		
	 *  - popup			弹窗授权页		
	 *  - wap1.2		wap1.2页面		
	 *  - wap2.0		wap2.0页面		
	 *  - js			js-sdk 专用 授权页面是弹窗，返回结果为js-sdk回掉函数		
	 *  - apponweibo	站内应用专用,站内应用不传display参数,并且response_type为token时,默认使用改display.授权后不会返回access_token，只是输出js刷新站内应用父框架
	 * forcelogin 强制登录？
	 * @param bool $sign_in_with_Weibo 参数已失效
	 * @param string $url 回调地址
	 * @param string $lang 语言
     * @return string
     */
    function getAuthorizeURL($param, $sign_in_with_Weibo = TRUE , $url, $lang = 'zh-Hans')
	{
		if(!is_array($param)){
			$param_new = array();
			$param_new = (array)parse_str($param, $param_new);
			$param = $param_new;
		}
		
		if(!isset($param['response_type'])){
			$param['response_type'] = 'code';
		}
	
		if(!empty($param['state']) && is_array($param['state'])){
			$param['state'] = xwbOAuthV2_tool_http::build_http_query($param['state']);		
		}
		
		if(!isset($param['display'])){
			$param['display'] = null;
		}
		
		$param['client_id'] = $this->client_id;
		$param['redirect_uri'] = $url;
		
		if(isset($param['forcelogin']) && true == $param['forcelogin']){
			$param['forcelogin'] = 'true';
		}
		
		$url = xwbOAuthV2_tool_http::get_url($this->authorizeURL(), $param);

		$url .= '&lang='.$lang ;
		
		return RST($url);
		
    }

	/**
	 * Get the authorize Token
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回599999
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $token
	 * @param string $user
	 * @param string $password
	 *
	 * @return array
	 */
	function getAuthorizeToken($token, $user, $password)
	{
        return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

    /**
     * 获取Access token
     * @see http://open.weibo.com/wiki/OAuth2/access_token
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $code （参数改变）此参数当且仅当$param['grant_type'] = 'authorization_code'时，等同于“调用authorize获得的code值”
	 * @param array $param （参数改变）参数集合。必填key有：
	 *  - grant_type 请求的类型,可以为:code / authorization_code, password, token / refresh_token
	 * 可选key有：
	 *  - 当grant_type为code / authorization_code时，须有： array('code'=>..., 'redirect_uri'=>...)
	 *  - 当grant_type为password时，须有： array('username'=>..., 'password'=>...)
	 *  - 当grant_type为token / refresh_token时，须有： array('refresh_token'=>...)
     * @return array array("oauth_token" => the access token,
     *                "oauth_token_secret" => the access secret)
     */
    function getAccessToken($code = FALSE, $param = array())
	{
		if(!isset($param['grant_type'])){
			$param['grant_type'] = 'authorization_code';
		}
		
		if(!empty($code) && ($param['grant_type'] == 'authorization_code' || $param['grant_type'] == 'code')){
			$param['code'] = $code;
		}
		
		switch($param['grant_type']){
			case 'token':			
			case 'refresh_token':
				$param['grant_type'] = 'refresh_token';
				break;
			case 'code':				
			case 'authorization_code':
				$param['grant_type'] = 'authorization_code';					
				break;
			case 'password':
				$param['grant_type'] = 'password';		
				break;
			default:
				return RST(false, 100000, "wrong auth type");			
		}
		
		$param['client_id'] = $this->client_id;
		$param['client_secret'] = $this->client_secret;
		$token = $this->oAuthRequest($this->accessTokenURL(), 'POST', $param);
		if(isset($token['errno']) && $token['errno'] != 0){
			return $token;
		}else{
			$this->setToken(3, $token['rst']['access_token'], isset($token['rst']['refresh_token']) ? $token['rst']['refresh_token'] : '');
			if(isset($token['rst']['uid'])){
				$token['rst']['user_id']  = $token['rst']['uid'];
			}
			return RST($this->getToken() + $token['rst']);
		}
    }
    

    /**
     * xauth登录
     * Exchange the request token and secret for an access token and
     * secret, to sign API calls.
     * 
     * @see http://open.weibo.com/wiki/OAuth2/access_token
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
     * 
     * @param string $username 用户名
     * @param string $pwd 密码
     * @return 
     */
    function getXauthAccessToken($username, $pwd)
    {
    	$param['grant_type'] = 'password';
    	$param['username'] = $username;
    	$param['password'] = $pwd;
    	
    	return $this->getAccessToken(null, $param);
    	
    }
    

// search user

	/**
	 * 搜索微博用户
	 * @see http://open.weibo.com/wiki/Users/search
	 * @see http://open.weibo.com/wiki/2/search/users
	 *
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			高权限也无效，需要单独申请：array ( 'rst' => false, 'errno' => 10014, 'err' => '(Insufficient app permissions!)', 'level' => 0, 'log' => '', )
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口为高权限。
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 		true（无高权限，无法验证）
	 * 		true（其它原因，无法验证）
	 *		false（改造未完成） 
	 *
	 *	- 张暘
	 *		此接口无权限使用使用users/show接口替换
	 *
	 * @param array $params array('base_app' => string 是否只搜索该应用数据 true false 默认为false
	 *								'q' => string 关键字,
	 *								'snick' => int|string 是否包含昵称 0不包含 1包含,
	 *								'sdomain' => int|sting 是否包含个性域名 0不包含 1包含,
	 *								'sintro' => int|string 是否包含简介 同上,
	 *								'province' => int|string 省份id,
	 *								'city' => int|string 城市id,
	 *								'gender' => string 性别 m为男 f为女,
	 *								'comorsch' => string 公司学校名称,
	 *								'sort' => int|sting 排序方式 1按更新时间 2按粉丝数,
	 *								'page' => int 页码数,
	 *								'count' => int 获取条数,
	 *								'callback' => string 此参数已失效！ ) 
	 * @param bool $oauth
	 * @return array
	 */
	function searchUser($params, $oauth = true)
	{
		$response  = $this->getUserShow(null, null, $params['q'], true);
		$value = $response['rst'];
		$response['rst'] = array('users'=>array($value));
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'users'));
		return $response;
		
		/*$url = $this->oauth2_url('search/users');
		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'users'));
		return $response;*/
		
	}


	/**
	 * 搜索微博文章
	 * @see weibo::searchStatuse()
	 * - oauth2_convert 
	 *		not_exist 已不存在，使用其它接口替代
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param array $param array('base_app' => string,
	 *								'q' => string 关键字,
	 *								'page' => int 页码数,
	 *								'count' => int 获取条数,)。
	 * @param bool $oauth
	 * @return array
	 */
	function search($params, $oauth = true)
	{
		if(!isset($params['needcount'])){
			$params['needcount'] = 'true';
		}
		return $this->searchStatuse($params, $oauth);
	}


	/**
	 * 搜索微博文章
	 * @see http://open.weibo.com/wiki/2/search/statuses
	 * @see http://open.weibo.com/wiki/Search
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口为高权限。
	 * 
	 * - oauth2_convert 
	 *		param_change 参数增加
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 * 		true（无高权限，无法验证）
	 *
	 * @param array $params array('q' => string 关键字,
	 *								'base_app' => string,
	 *								'filter_ori' => int|string 过滤器 是否原创 0为全部 5为原创 4为转发 默认为0,
	 *								'filter_pic' => int|string 过滤器 是否包含图片 0为全部 1为包含图 2为不含图,
	 *								'province' => int|string 省份id,
	 *								'city' => int|string 城市id,
	 *								'starttime' => string 开始时间(时间戳),
	 *								'endtime' => string 截止时间(时间戳),
	 *								'page' => int 页码数,
	 *								'count' => int 获取条数,
	 *								'needcount' => string 是否需要搜索总数, 'true' 是需要)
	 * @param bool $oauth
	 * @return array
	 */
	function searchStatuse($params, $oauth = true)
	{
		$url = $this->oauth2_url('search/statuses');
		if ($oauth === false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		
		$this->oauth2_compact_convert($response, 'searchStatuse', $params);
		return $response;
	}

	/**
	 * 搜索用户时的联想搜索建议
	 * @see http://open.weibo.com/wiki/Search/suggestions/at_users
	 * @see http://open.weibo.com/wiki/2/search/suggestions/users
	 *
	 * - oauth2_convert 
	 *		__code_internal__ 内部code，与外部无关
	 *		param_change 参数改变
	 *		param_change 参数失效
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $q 搜索的关键字
	 * @param int $count 每页返回结果数。默认10 
	 * @param int $type （参数已失效！）1代表粉丝，0代表关注人。另外， 粉丝最多返回1000个，关注人最多2000个 
	 * @param int $range （参数已失效！）0代表只查关注人，1代表只搜索当前用户对关注人的备注，2表示都查. 默认为2.
	 *
	 * @return array
	 */
	function getSuggestionsAtUsers($q, $type = null, $count = false, $range = false)
	{
		$url = $this->oauth2_url('search/suggestions/users');
		$params['q'] = $q;
		if ($count) {
			$params['count'] = $count;
		}
		$params['type'] = $type;
		if ($range) {
			$params['range'] = $range;
		}

		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'getSuggestionsAtUsers');
		return $response;
	}	
	
	/**
	 * 获取省份及城市编码ID与文字对应
	 * @see http://open.weibo.com/wiki/%E7%9C%81%E4%BB%BD%E5%9F%8E%E5%B8%82%E7%BC%96%E7%A0%81%E8%A1%A8
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 			进行了旧数据兼容：使用OAuth 1.0数据
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回虚值
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @return array
	 */
	function getProvinces()
	{
		$response = RST(array());
		$this->oauth2_compact_convert($response, 'getProvinces', array('_force_pass'=>true));
		return $response;
	}
	
	/**
	 * weibo表情
	 * @see http://open.weibo.com/wiki/Emotions
	 * @see http://open.weibo.com/wiki/2/emotions
	 * 
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			未审核无法使用：'errno' => 21321, 'err' => '(applications over the unaudited use restrictions!)'
	 * 
	 * oauth 2.0存在问题：
	 * 			进行了旧数据兼容：未审核无法使用、并且$type为face时，使用OAuth 1.0数据
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @return array
	 */
	function emotions($language = false, $type = false)
	{
		$url = $this->oauth2_url('emotions');
		$params = array();
		switch ($language) {
			case 'zh_cn':
			case 'cnname':				
				$lang = 'cnname';
				break;
			case 'zh_tw':
			case 'twname':					
				$lang = 'twname';
				break;
			default:
				$lang = 'cnname';
		}
		$params['language'] = $lang;
		
		if(empty($type)){
			$type = 'face';
		}
		$params['type'] = $type;
		
		$response = $this->sourceRequest($url, 'get', $params);
		if('face' == $type && (empty($response['rst']) || $response['errno'] != 0)){
			$response = RST(array());
			$this->oauth2_compact_convert($response, 'emotions_def', array('_force_pass' => true, 'language'=>$params['language']));
		}else{
			$this->oauth2_compact_convert($response, 'emotions');
		}
		
		return $response;
	}


	//热门话题

	/**
	 * 获取某人的话题
	 * @see http://open.weibo.com/wiki/Trends
	 * @see http://open.weibo.com/wiki/2/trends
	 * 
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $user_id 用户id
 	 * @param int $page 页码数
	 * @param int $count 获取条数
	 * @return array
	 */
	function getTrendsByUser($user_id, $page = null, $count = null)
	{
		$url = $this->oauth2_url('trends');
		$params = array();

		$params['uid'] = $user_id;
		if ($page) {
			$params['page'] = $page;
		}
		if ($count) {
			$params['count'] = $count;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;

	}

	/**
	 * 获取某一话题的微博(目前最多返回200条)
	 * @see http://open.weibo.com/wiki/Trends/statuses
	 * @see http://open.weibo.com/wiki/2/search/topics
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口为高权限。
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 * 		true（无高权限，无法验证）
	 *
	 * @param string $trend_name 话题
	 * @return array
	 */
	function getTrendStatuses($trend_name, $page = null, $count = null)
	{
		$url = $this->oauth2_url('search/topics');
		$params = array();

		$params['q'] = $trend_name;
		if ($page) {
			$params['page'] = $page;
		}
		if ($count) {
			$params['count'] = $count;
		}
		
		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'statuses'));

		return $response;
	}

	/**
	 * 关注某话题
	 * @see http://open.weibo.com/wiki/Trends/follow
	 * @see http://open.weibo.com/wiki/2/trends/follow
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $trend_name 话题
	 * @return array
	 */
	function createTrendsFollow($trend_name)
	{
		$url = $this->oauth2_url('trends/follow');
		$params = array();
		$params['trend_name'] = $trend_name;
		$response = $this->oAuthRequest($url, 'post', $params);
		return $response;
	}
	
	/**
	 * 获取按小时返回的热门话题
	 * @see http://open.weibo.com/wiki/Trends/hourly
	 * @see http://open.weibo.com/wiki/2/trends/hourly
	 * 
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $hour （参数未确认是否已无效）返回几个小时前的话题
	 * @param bool $base_app 是否基于当前应用来获取数据
	 * @param string $exclude （参数未确认是否已无效）从话题列表中排除的话题
	 * @return array
	 */
	function getTrendsHourly($hour = false, $base_app = '0', $exclude = null)
	{
		$url = $this->oauth2_url('trends/hourly');
		$params = array();

		if ($hour) {
			$params['hour'] = $hour;
		}
		$params['base_app'] = $base_app;

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	}

   /**
	* 获取按日期返回的热门话题
	* @see http://open.weibo.com/wiki/Trends/daily
	* @see http://open.weibo.com/wiki/2/trends/daily
	* 
	 * - oauth2_convert 
	 *		true（改造完毕）
	*
	* @param int|string $date （参数未确认是否已无效）返回此日期前的话题
	* @param bool $base_app 是否基于当前应用来获取数据
	* @param string $exclude （参数未确认是否已无效）从话题列表中排除的话题
	* @return array
	*/
	function getTrendsDaily($date = false, $base_app = '0', $exclude = null)
	{
		$url = $this->oauth2_url('trends/daily');
		$params = array();

		if ($date) {
			$params['date'] = $date;
		}
	   $params['base_app'] = $base_app;

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	}

   /**
	* 获取按周的热门话题
	* @see http://open.weibo.com/wiki/Trends/weekly
	* @see http://open.weibo.com/wiki/2/trends/weekly
	* 
	 * - oauth2_convert 
	 *		true（改造完毕）
	*
	* @param int|string $date （参数未确认是否已无效）返回此日期前的话题
	* @param bool $base_app 是否基于当前应用来获取数据
	* @param string $exclude （参数未确认是否已无效）从话题列表中排除的话题
	* @return array
	*/
	function getTrendsWeekly($date = false, $base_app = '0', $exclude = null)
	{
		$url = $this->oauth2_url('trends/weekly');
		$params = array();
		if ($date) {
			$params['date'] = $date;
		}
		$params['base_app'] = $base_app;
		$response = $this->oAuthRequest($url, 'get', $params);
		return $response;
	}

   /**
	* 获取按日的热门转发
	* @see http://open.weibo.com/wiki/Statuses/hot/repost_daily
	* @see http://open.weibo.com/wiki/2/statuses/hot/repost_daily
	* 
	 * - oauth2_convert 
	 *		true（改造完毕）
	*
	* @param int $count 获取返回的条数
	* @return array
	*/
	function getHotRepostDaily($count = null, $base_app = '0')
	{
	   $url = $this->oauth2_url('statuses/hot/repost_daily');
	   $params = array();

	   if ($count) {
		   $params['count'] = $count;
	   }
	   $params['base_app'] = $base_app;
	   $response = $this->oAuthRequest($url, 'get', $params);

	   return $response;
	}

   /**
	* 获取按周的热门转发
	* @see http://open.weibo.com/wiki/Statuses/hot/repost_weekly
	* @see http://open.weibo.com/wiki/2/statuses/hot/repost_weekly
	*
	 * - oauth2_convert 
	 *		true（改造完毕）
	*
	* @param int $count 返回条数
	* @return array
	*/
	function getHotRepostWeekly($count = null, $base_app = '0')
	{
	   $url = $this->oauth2_url('statuses/hot/repost_weekly');
	   $params = array();

	   if ($count) {
		   $params['count'] = $count;
	   }
	   $params['base_app'] = $base_app;	   
	   $response = $this->oAuthRequest($url, 'get', $params);

	   return $response;
	}

   /**
	* 获取按日的热门评论
	* @see http://open.weibo.com/wiki/Statuses/hot/comments_daily
	* @see http://open.weibo.com/wiki/2/statuses/hot/comments_daily
	* 
	 * - oauth2_convert 
	 *		true（改造完毕）
	*
	* @param int $count 返回条数
	* @return array
	*/
	function getHotCommentsDaily($count = null, $base_app = '0')
	{
	   $url = $this->oauth2_url('statuses/hot/comments_daily');
	   $params = array();

	   if ($count) {
		   $params['count'] = $count;
	   }
	   $params['base_app'] = $base_app;
	   $response = $this->oAuthRequest($url, 'get', $params);

	   return $response;
	}

   /**
	* 获取按周的热门评论
	* @see http://open.weibo.com/wiki/Statuses/hot/comments_weekly
	* @see http://open.weibo.com/wiki/2/statuses/hot/comments_weekly
	* 
	 * - oauth2_convert 
	 *		true（改造完毕）
	*
	* @param int $count 返回条数
	* @return array
	*/
	function getHotCommentsWeekly($count = null, $base_app = '0')
	{
	   $url = $this->oauth2_url('statuses/hot/comments_weekly');
	   $params = array();

	   if ($count) {
		   $params['count'] = $count;
	   }
	   $params['base_app'] = $base_app;
	   $response = $this->oAuthRequest($url, 'get', $params);

	   return $response;
	}

	//标签

	/**
	 * 获取标签列表
	 * @see http://open.weibo.com/wiki/Tags
	 * @see http://open.weibo.com/wiki/2/tags
	 * 
	 * - oauth2_convert 
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $user_id 用户user id
	 * @param int $count 获取条数 默认20
	 * @param int $page 页码数
	 * @return array
	 */
	function getTagsList($user_id, $count = null, $page = null)
	{
		$url = $this->oauth2_url('tags');
		$params = array();

		$params['uid'] = $user_id;
		if ($count) {
			$params['count'] = $count;
		}
		if ($page) {
			$params['page'] = $page;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	}

	/**
	 * 创建标签
	 * @see http://open.weibo.com/wiki/Tags/create
	 * @see http://open.weibo.com/wiki/2/tags/create
	 * 
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param string $tags 标签 多个用逗号隔开
	 * @return array
	 */
	function createTags($tags)
	{
		$url = $this->oauth2_url('tags/create');
		$params = array();
		$params['tags'] = $tags;
		$response = $this->oAuthRequest($url, 'post', $params);
		return $response;
	}


	/**
	 * 删除标签
	 * @see http://open.weibo.com/wiki/Tags/destroy
	 * @see http://open.weibo.com/wiki/2/tags/destroy
	 * 
	 * - oauth2_convert 
	 *		true（改造完毕）
	 *
	 * @param int|string $tag_id 标签id
	 * @return array
	 */
	function deleteTags($tag_id)
	{
		$url = $this->oauth2_url('tags/destroy');
		$params = array();
		$params['tag_id'] = $tag_id;
		$response = $this->oAuthRequest($url, 'post', $params);
		return $response;
	}
	
	
	/**
	 * 获取用户感兴趣的标签
	 * @see http://open.weibo.com/wiki/Tags/suggestions
	 * @see http://open.weibo.com/wiki/2/tags/suggestions
	 *
	 * @param int $page 获取页码数
	 * @param int $count 获取条数 默认为10
	 * @return array
	 */
	function getTagsSuggestions($page = null, $count = null)
	{
		$url = $this->oauth2_url('tags/suggestions');
		$params = array();
		if ($page) {
			$params['page'] = $page;
		}
		if ($count) {
			$params['count'] = $count;
		}
		$response = $this->oAuthRequest($url, 'get', $params);
		return $response;
	}

	//lists 订阅
	/**
	 * lists 订阅存在以下相同的问题和处理方案：
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回599999
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 */

	/**
	 * 创建新的订阅分类
	 *
	 * @param int|string $id （已废弃）用户id
	 * @param string $name 分类名称
	 * @param string $mode 其值（values）可以是public或private，默认为public
     * @param	$description
	 * @return array
	 */
	function createUserLists($id = null, $name, $mode = 'public', $description = null)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 列出用户的所有订阅分类 每页返回20个list
	 * 
	 * @param int|string $id 用户id
	 * @param int|string $cursor 分页位置 默认从-1开始
	 * @param int $listType 返回的列表属性，0返回公开列表，1返回私有列表，2返回系统列表，默认值0
	 * @return array
	 */
	function getUserLists($id, $cursor = null, $listType=NULL)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 更新指定的订阅分类
	 *
	 * @param int|string $id 用户id
	 * @param int|string $list_id 订阅分类id
	 * @param string $name 用户昵称
	 * @param string $mode 其值（values）可以是public或private，默认为public
     * @param	$description
	 * @return array
	 */
	function updateUserLists($id, $list_id, $name, $mode = null, $description = null)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 展示特定订阅分类的信息
	 *
	 * @param int|string $id 用户id
	 * @param int|string $list_id 订阅分类id
	 * @return array
	 */
	function getUserListId($id, $list_id)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 删除指定的订阅分类
	 *
	 * @param int|string $id 用户id
	 * @param int|string $list_id 订阅分类id
	 * @return array
	 */
	function deleteUserListId($id, $list_id)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 展示list成员的最新微博信息
	 *
	 * @param int|string $id 用户id
	 * @param int|string $list_id list id
	 * @param int $per_page 获取条数
	 * @param int $page 获取页码数
	 * @param int $since_id 返回带有比指定list id大的id
	 * @param int $max_id 返回带一个小于或等于指定list id的id结果
	 * @param int|string $base_app: 选填参数，是否基于当前应用来获取数据。1为限制本应用微博，0为不做限制
	 * @param int|stirng $content_type 返回某一内容类型结果的微博 类型有全部-0, 图片-1, 音乐-2, 视频-3, 纯文本-4,
	 * @return array
	 */
	function getUserListIdStatuses($id, $list_id, $per_page = null, $page = null, $since_id = null, $max_id = null, $base_app='0', $feature=0)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 列出用户作为成员的所有list列表
	 *
	 * @param int|string $id 用户id
	 * @param int|string $cursor 分页位置 默认从-1开始 每页包含20个list
	 * @return array
	 */
	function getUserListsMemberships($id, $cursor = null)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 获取用户的lists、subscriptions、listed数量
	 *
	 * @param int|string $id 用户id
	 * @return array
	 */
	function getUserListsCounts($id)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 列出用户订阅的所有list列表
	 *
	 * @param int|string $id 用户id
	 * @param int|string $cursor 分页位置 默认从-1开始 每页包含20个list
	 * @return array
	 */
	function getUserListsSubscriptions($id, $cursor = null)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 返回list中所有的成员
	 *
	 * @param int|string $id 用户id
	 * @param int|string $cursor 分页位置 默认从-1开始 每页包含20个list
	 * @return array
	 */
	function getUserListsMember($id, $list_id, $cursor = null)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}

	/**
	 * 将用户添加到list中。用户只能将其他用户添加到自己创建的list中。每个list最多拥有500个用户。
	 *
	 * @param int|string $id 用户id
	 * @param int|string $list_id 订阅list ID
	 * @param int $user_id 要添加的用户id
	 * @return array
	 */
	function createUserListsMember($id, $list_id, $user_id)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}
	
	
 	/**
	 * 将用户批量添加到list中。用户只能将其他用户添加到自己创建的list中。每个list最多拥有500个用户。
	 *
	 * @param int|string $uid 用户ID
	 * @param int|string $list_id 订阅list ID
	 * @param int|string $ids  要添加的用户ids
	 * @param int|string $names  要添加的用户names
	 * @return array
	 */
	function createUserListsMemberBatch($uid, $listId, $ids=NULL, $names=NULL)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}
	

	/**
	 * 将用户从list中删除
	 *
	 * @param int|string $id 用户id
	 * @param int|string $list_id 订阅list ID
	 * @param int $user_id 要删除的用户id
	 * @return array
	 */
	function deleteUserListsMember($id, $list_id, $user_id)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}


	//黑名单
	/**
	 * 将用户加入黑名单
	 * @see http://open.weibo.com/wiki/Blocks/create
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 * 
	 * @param int|string $user_id 用户id
	 * @param string $screen_name （此参数已无效）用户昵称
	 * @return array
	 */
	function createBlocks($user_id = null, $screen_name = null)
	{
		$url = $this->oauth2_url('blocks/create');
		$params = array();

		if ($user_id) {
			$params['uid'] = $user_id;
		}
		if ($screen_name) {
			return RST(false, 599998, 'PARAMTER screen_name HAS BEEN REMOVED IN OAUTH 2.0');
		}

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	}

	/**
	 * 删除黑名单
	 * @see http://open.weibo.com/wiki/Blocks/destroy
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 *
	 * @param int|string $user_id 用户id
	 * @param string $screen_name 用户昵称
	 * @return array
	 */
	function deleteBlocks($user_id = null, $screen_name = null)
	{
		$url = $this->oauth2_url('blocks/destroy');
		$params = array();

		if ($user_id) {
			$params['uid'] = $user_id;
		}
		if ($screen_name) {
			return RST(false, 599998, 'PARAMTER screen_name HAS BEEN REMOVED IN OAUTH 2.0');
		}

		$response = $this->oAuthRequest($url, 'post', $params);

		return $response;
	}
	
	/**
	 * 检测是否是黑名单用户
	 * @see http://open.weibo.com/wiki/Blocks/exists
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 * 
	 * @param int|string $user_id 用户id
	 * @param string $screen_name （警告：参数失效！）用户昵称
	 * @return array
	 */
	function existsBlocks($user_id = null, $screen_name = null)
	{
		$url = $this->oauth2_url('blocks/exists');
		$params = array();

		if ($user_id) {
			$params['uid'] = $user_id;
		}
		if ($screen_name) {
			$params['screen_name'] = $screen_name;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	}

	/**
	 * 分页黑名单用户(输出用户详细信息)
	 * @see http://open.weibo.com/wiki/Blocks/blocking
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 * 
	 * @param int $page 页码数
	 * @param int $count 获取条数
	 * @param int $with_addtime （参数未确认是否已无效）
	 * @return array
	 */
	function getBlocks($page = null, $count = null, $with_addtime=1)
	{
		$url = $this->oauth2_url('blocks/blocking');
		$params = array();

		if ($page) {
			$params['page'] = $page;
		}
		if ($count) {
			$params['count'] = $count;
		}
		if ($with_addtime) {
			$params['with_addtime'] = $with_addtime;
		}
		
		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'users'));
		return $response;
	}
	
	/**
	 * 分页黑名单用户（只输出id）
	 * @see http://open.weibo.com/wiki/Blocks/blocking/ids
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 * 		true（其它原因，无法验证）
	 * 
	 * @param int $page 页码数
	 * @param int $count 获取条数
	 * @return array
	 */
	function getBlocksIds($page = null, $count = null)
	{
		$url = $this->oauth2_url('blocks/blocking/ids');
		$params = array();
		if ($page) {
			$params['page'] = $page;
		}
		if ($count) {
			$params['count'] = $count;
		}

		$response = $this->oAuthRequest($url, 'get', $params);

		return $response;
	}

	/**
	 * 举报某条信息
	 * @see http://open.weibo.com/wiki/Report_spam
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口已不存在（被移除）。
	 * 
	 * - oauth2_convert 
	 *		not_exist 已不存在，返回599999
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param string $content 举报的内容
	 * @param string $url 举报的url
	 * @param int|string $status_id 举报的微博id 为可选参数。当status_id不为空时，允许url参数为空
	 * @return array
	 */
	function report_spam($content, $url = null, $status_id = null)
	{
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}
    
	 /**
	 * 获取一批指定用户的微博timeline
	 * @see http://open.weibo.com/wiki/Statuses/batch_timeline
	 * @see http://open.weibo.com/wiki/2/statuses/timeline_batch
	 * 
	 * oauth 2.0限制：
	 * 		oAuthRequest：
	 * 			高权限接口，但无法验证是否存在需要单独申请的问题 
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	 * 
	 * @param mixed $user_id 用户ID,一次最多20个
	 * @param mixed $screen_name 用户昵称，一次最多20个
	 * @param int $count 指定要返回的记录条数。默认20，最大200
	 * @param int $page 指定返回结果的页码
	 * @param int $feature 微博类型，0全部，1原创，2图片，3视频，4音乐. 返回指定类型的微博信息内容
	 * @param int $base_app 是否基于当前应用来获取数据。1为限制本应用微博，0为不做限制
	 * @param bool $oauth
	 */
	 function getBatchTimeline($params, $oauth = true)
	 {
		$url = $this->oauth2_url('statuses/timeline_batch');
		if(isset($params['user_id'])){
			$params['uids'] = $params['user_id'];
			unset($params['user_id']);
		}
		
		if ($oauth == false) {
			$response = $this->sourceRequest($url, 'get', $params);
		} else {
			$response = $this->oAuthRequest($url, 'get', $params);
		}
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'statuses'));
		return $response;
	 }
	 

	 /**
	  * 是否需要本地数据备份
	  */
	 function _needLocalCopy()
	 {
	 	///查询是否开启数据备份
		$plugins = DR('Plugins.get', '', 6);
		$plugins = isset($plugins['rst']) ? $plugins['rst'] : array();
		if (isset($plugins['in_use']) && $plugins['in_use'] == 1) {
			return TRUE;
		}
		
		return FALSE;
	 }
	 
	 
	 /**
	  * 获取返回结果的id数组
	  * @param array $rspList
	  */
 	 function _getRspIdList($rspList)
	 {
	 	$idList = array();
	 	
		if ( is_array($rspList) )
		{
			foreach ($rspList as $aRsp) {
				array_push($idList, $aRsp['id']);
			}
		}
		return $idList;
	 }

	/**
	 * 短链接转换
	 * @see http://open.weibo.com/wiki/ShortUrl
	 * @see http://open.weibo.com/wiki/2/short_url/shorten
	 * @see http://open.weibo.com/wiki/2/short_url/expand
	 * 
	 * oauth 2.0存在问题：
	 * 			注意：旧接口已不存在，采取了其它接口替换
	 * 
	 * - oauth2_convert 
	 *		param_change 参数改变
	 *		return_change 返回改变 
	 *		not_exist 已不存在，但保留相关代码
	 *
	 * - oauth2_convert 
	 *		true（改造完毕）
	  * 
	 * @param string|array $url 需要转换的url，无需urlencode，程序将自动转换
	 * @param boolean $is_short 默认为true，表示从长url转换成短url；false表示从短url转换成长url,true:1,false:0
	 * @param boolean $is_batch 默认为false。true则使用批量方式，false使用单条方式,true:1,false:0。批量url用英文","分割
	 * @return array
	 */
	function shortUrl($urls, $is_short = true, $is_batch = false)
	{
		if(true == $is_short){
			$url_prefix = 'url_long';
			$url = $this->oauth2_url('short_url/shorten');
		}else{
			$url_prefix = 'url_short';
			$url = $this->oauth2_url('short_url/expand');
		}
		
		$params = array();
		
		$new_urls = array();
		if(!is_array($urls)){
			if($is_batch){
				$new_urls = explode(',', $urls);
			}else{
				$new_urls[] = $urls;
			}
		}else{
			$new_urls = $urls;
		}
		$params[$url_prefix] = $new_urls;
		
		$response = $this->oAuthRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'urls'));
		return $response;
	}
 
	 /**
	  * 取得一个短链接所对应的页面信息，包含页面的title，原始的长链接，富内容（元数据）
	  * @see http://open.weibo.com/wiki/2/short_url/info
	  * 
	 * oauth 2.0限制：
	 * 		sourceRequest：
	 * 			未审核无法使用：'errno' => 21321, 'err' => '(applications over the unaudited use restrictions!)'  
	 * 
	 * oauth 2.0存在问题：
	 * 			警告：此接口为高权限。
	  * 
	  * @param string $urls 短链接id，多个用逗号隔开
	  *
	  * @return array
	  */
	 function shortUrlBatchInfo($urls)
	 {
		$url = $this->oauth2_url('short_url/info');
		$params = array();
		$new_urls = array();
		if(!is_array($urls)){
			$new_urls = explode(',', $urls);
		}else{
			$new_urls = $urls;
		}
		foreach($new_urls as $k => $row){
			if(strpos($row, 'http://t.cn') === false){
				$new_urls[$k] = 'http://t.cn/'. $row;
			}
		}
		$params['url_short'] = $new_urls;
		
		$response = $this->sourceRequest($url, 'get', $params);
		$this->oauth2_compact_convert($response, 'common_only_body', array('body'=>'urls'));
		return $response;
	 }
	 
	 /**
	  * 获取一个oauth2 url
	  * @param unknown_type $url 两边请不要有/，否则会报：Request Api not found!
	  * @param unknown_type $format
	  * @return string
	  */
	 function oauth2_url($url, $format = 'json'){
	 	return OAUTH2_WEIBO_API_URL. $url. '.'. $format;
	 }
	 
	 /**
	  * 对OAuth 2.0数据格式进行兼容到OAuth 1.0数据格式的转换
	  * @param RST_Array &$data
	  * @param string $name
	  * @param array $param 参数。其中本方法存在的特殊参数有：
	  * bool _force_pass 只要存在此参数并且不为false，就会强制将data体内的rst传入到转换类中（不管rst内是否存在值）
	  * bool _force_pass_data 只要存在此参数并且不为false，就会强制将整个$data传入到转换类中（不管是否存在错误）
	  * @return bool
	  */
	 function oauth2_compact_convert(&$data, $name, $param = array()){
	 	//_force_pass_data为true时，将整个data体传入
	 	if(isset($param['_force_pass_data']) && $param['_force_pass_data'] != false){
	 		$classname = 'conv_2/conv_2_'. $name;
	 		$o = APP::O($classname);
	 		$data = $o->convert($data, $param);
	 		return true;
	 	}
	 	
	 	if($data['errno'] != 0){
	 		return false;
	 	}elseif(empty($data['rst']) && (!isset($param['_force_pass']) || $param['_force_pass'] != true) ){
	 		return false;
	 	}
	 	
	 	$classname = 'conv_2/conv_2_'. $name;
	 	$o = APP::O($classname);
	 	$data['rst'] = $o->convert($data['rst'], $param);
	 	return true;
	 }
	 

	/**
	 * Format and sign an OAuth / API request
	 * 
	 * @param string $url
	 * @param string $method
	 * @param array $parameters
	 * @param bool $multi
	 * @param bool $origin_return
	 * @return RST_array
	 */
	function oAuthRequest($url, $method = 'get', $parameters = array(), $multi = false, $origin_return = false) {
		$this->last_req_type = 'oAuthRequest';
		$this->http->setHeader('Authorization', 'OAuth2 '. $this->access_token);
		$result = $this->http($url, $method, $parameters, $multi);
		if($this->http_code != 200){
			if(0 == $this->http_code){
				return RST(false, '1040002', 'Access Timeout or Access denied');
			}
			
			//refresh token重试
			if(strpos($result, 'expired_token') !== false){
				$_refresh = $this->_auto_refreshToken();
    			//trigger_error('_auto_refreshToken:'. var_export($_refresh, true));				
				if($_refresh){
					return $this->oAuthRequest($url, $method, $parameters, $multi, $origin_return);
				}
			}
			
			return $this->throwException($result);
		}
		
		if ($origin_return != true) {
			$result = xwbOAuthV2_tool_http::json_decode($result);
		}
		
		return RST($result);
	}

	/**
	 * 使用source方式访问api
	 *
	 * @param string $url
	 * @param string $method
	 * @param array $parameters
	 * @param bool $multi
	 * @param bool $origin_return
	 * @return RST_array
	 */
	function sourceRequest($url, $method = 'get', $parameters = array(), $multi = false, $origin_return = false)
	{
		$this->last_req_type = 'sourceRequest';
		$parameters['source'] = $this->client_id;
		$result = $this->http($url, $method, $parameters, $multi);
		
		
		if($this->http_code != 200){
			if(0 == $this->http_code){
				return RST(false, '1040002', 'Access Timeout or Access denied');
			}
			return $this->throwException($result);
		}
		
		if ($origin_return != true) {
			$result = xwbOAuthV2_tool_http::json_decode($result);
		}
		
		return RST($result);	
    }
    
    protected function _auto_refreshToken(){
    	if(empty($this->refresh_token) || isset($this->_refresh_token_ed[$this->refresh_token])){
    		return false;
    	}
    	$this->_refresh_token_ed[$this->refresh_token] = true;
    	$_refresh_param = array('grant_type' => 'refresh_token', 'refresh_token'=>$this->refresh_token);
		$acc_token = $this->getAccessToken(false, $_refresh_param, true);
		$acc_token = $acc_token['rst'];
		if(isset($acc_token['oauth_token']) && !empty($acc_token['oauth_token'])){
			$oauth_uid = $this->get_uid();
			$oauth_uid = isset($oauth_uid['rst']['uid']) && $oauth_uid['rst']['uid'] > 0 ? $oauth_uid['rst']['uid'] : 0;
			if($oauth_uid > 0){
				if(USER::uid() == $oauth_uid){
					USER::setOAuthKey($acc_token, true);
				}
				$result = DR('mgr/userCom.getByUid', '', $oauth_uid);
				$result = $result['rst'];
				if(!empty($result) && $result['uid'] > 0){
					$new_data['access_token'] = $acc_token['oauth_token'];
					$new_data['token_secret'] = $acc_token['oauth_token_secret'];
					DR('mgr/userCom.insertUser', '', $new_data, $oauth_uid);
				}
				
				//代理账号
				$proxy_file = P_COMS . "/accountProxy" . EXT_COM;
				if(is_file($proxy_file)){
					$new_data['token'] = $acc_token['oauth_token'];
					$new_data['secret'] = $acc_token['oauth_token_secret'];
					DR('accountProxy.edit', '', $new_data, $oauth_uid);
				}
				
				//user_config.php
				if ($oauth_uid === SYSTEM_SINA_UID) {
					$token = $acc_token;
					// 如果站长token发生变化，则修改user_config.php的token
					if ($token['oauth_token'] != WB_USER_OAUTH_TOKEN || $token['oauth_token_secret'] != WB_USER_OAUTH_TOKEN_SECRET) {
						if (XWB_SERVER_ENV_TYPE === 'sae') {
							$storage = new SaeStorage();
							$content = $storage->read(CONFIG_DOMAIN, md5(CONFIG_DOMAIN));
							parse_str($content, $config);
							$config['user_oauth_token'] = $token['oauth_token'];
							$config['oauth_token_secret'] = $token['user_oauth_token_secret'];
							$content = http_build_query($config);
							$storage->write(CONFIG_DOMAIN, md5(CONFIG_DOMAIN), $content);
						} else {
							// 修改user_config.php和绑定信息
							$config_file = IO::read(P_ROOT . '/../user_config.php');
							$config_arr = array(
											'WB_USER_OAUTH_TOKEN' => $token['oauth_token'],
											'WB_USER_OAUTH_TOKEN_SECRET' => $token['oauth_token_secret']
										);
							//更新user_config数据
							$config_file = F('set_define_value', $config_file, $config_arr);
							IO::write(P_ROOT . '/../user_config.php', $config_file);	
						}
					}
				}
				
				return true;
			}else{
				return false;
			}
			
		}else{
			return false;
		}
    }
	
    /**
     * Make an HTTP request
     * @param string $url 完整的URL
     * @param string $method 方法，大写 
     * @param array $parameters 参数。$method为GET时，附加到URL后；其余时候，作为body部分提交。
     * @param bool $multi 
     * @return 原始数据
     */
	function http($url, $method, $parameters = array(), $multi = false) {
		$method = strtoupper($method);
		
		$log_func_start_time = microtime(TRUE);
		
		$this->http->setHeader('API-RemoteIP', F('get_client_ip'));
		$this->http->setHeader('User-Agent', $this->useragent);
		
        switch ($method) {
        	case 'GET':
				$this->http->setUrl(xwbOAuthV2_tool_http::get_url($url, $parameters));
				$result = $this->http->request('get', true);
				break;

			case 'DELETE':
			default:
				$this->http->setUrl($url);
				$this->http->setData(xwbOAuthV2_tool_http::to_postdata($parameters, $multi));
				if($multi){
					$this->http->setHeader('Content-Type', "multipart/form-data; boundary=" . xwbOAuthV2_tool_http::$boundary);
				}
				$result = $this->http->request($method == 'DELETE' ? 'delete' : 'post', true);
				break;
      	 }
      	 	
		$this->http_code = $code = $this->http->getState();
		$this->url = $http_url = $this->http->getUrl();
			
		$logParam = array('url'=>$http_url, 'params'=>$parameters, 'code'=>$code, 'result'=>$result, 'error'=>$this->http->getError());
		LogMgr::warningLog($log_func_start_time, $this->logType, "[oAuthRequest]API Request method=$method", LOG_LEVEL_WARNING, $logParam);
		LOGSTR($this->logType, "[{$this->last_req_type}_2]API Request method=$method", LOG_LEVEL_INFO, $logParam, $log_func_start_time);
		
		if (200 != $code){
			//log
			if(!empty($this->access_token)){
				$acc_log = substr($this->access_token, 0, 5). '......';
			}else{
				$acc_log = '[X]';
			}

			$logMsg = 'url: '.$http_url." \r\ncode: ".$code." \r\nret: ".$result . "\r\nerror: " . $this->http->getError()."\r\naccess token:: ". $acc_log;
			LOGSTR($this->logType, "[{$this->last_req_type}_2]API Request method=$method&".$logMsg);
		}
		
		return $result;
	}
	
	/**
	 * 格式化错误信息
	 * @see http://open.weibo.com/wiki/Oauth2
	 * @see http://open.weibo.com/wiki/Error_code
	 * @oauth2_status __code_internal__
	 * @oauth2_convert false(待重写)
	 * @param array $error api返回的错误信息
	 * @return unknown
	 */
	function throwException($error)
	{
		$error_ori = $error;
		if(!is_array($error)){
			$error = (array)json_decode($error, true);
		}
		//目前未知OAuth 2.0和1.0的错误码关系，故进行简要处理
		/*
		 error: 错误码
		 error_code: 错误的内部编号
		 error_description: 错误的描述信息(不一定有)
		 error_url: 可读的网页URI，带有关于错误的信息，用于为终端用户提供与错误有关的额外信息。(不一定有)
		 */
		if(empty($error)){
			$error = array(
				'error' => 'unknown_error_not_json',
				'error_code' => 1050000,
				'error_description' => $error_ori,
			);
		}elseif(!isset($error['error'])){
			$error = array(
				'error' => 'unknown_error_blank',
				'error_code' => 1050000,
				'error_description' => $error_ori,			
			);
		}
		
		$final_err = 'WEIBO_OAUTH2_ERROR_CODE_'. (isset($error['error_code']) ? $error['error_code'] : 'UNKNOWN') .':'. (isset($error['error']) ? $error['error'] : 'unknown_error_blank');
		if(isset($error['error_description']) && !empty($error['error_description'])){
			$final_err .= '('. $error['error_description']. ')';
		}
		$rst = RST(false, isset($error['error_code']) ? $error['error_code'] : '1050000', $final_err, 0);
		
		return $rst;
		
	}
	
	/**
	 * 魔术方法，全部返回不存在此方法
	 * @param string $m
	 * @param array $a
	 * @return RST_Array
	 */
	function __call($m, $a){
		return RST(false, 599999, 'NO SUCH METHOD IN OAUTH 2.0');
	}
	
}


class xwbOAuthV2_tool_http{

	/**
	 * boundary of multipart
	 * @ignore
	 */
	public static $boundary = '';
	
    /**
     * builds the data one would send in a POST request
     */
    public static function to_postdata( $parameters, $multi = false ) {
    	if( $multi ){
    		return self::build_http_query_multi($parameters);
   		}else{
        	return self::build_http_query($parameters);
    	}
    }
	
	
	/**
	 * @ignore
	 */
	public static function build_http_query_multi($params) {
		if (!$params) return '';

		uksort($params, 'strcmp');

		$pairs = array();

		self::$boundary = $boundary = uniqid('------------------');
		$MPboundary = '--'.$boundary;
		$endMPboundary = $MPboundary. '--';
		$multipartbody = '';

		foreach ($params as $parameter => $value) {

			if( in_array($parameter, array('pic', 'image')) && $value{0} == '@' ) {
				$url = ltrim( $value, '@' );
				$content = file_get_contents( $url );
				$array = explode( '?', basename( $url ) );
				$filename = $array[0];

				$multipartbody .= $MPboundary . "\r\n";
				$multipartbody .= 'Content-Disposition: form-data; name="' . $parameter . '"; filename="' . $filename . '"'. "\r\n";
				$multipartbody .= "Content-Type: image/unknown\r\n\r\n";
				$multipartbody .= $content. "\r\n";
			} else {
				$multipartbody .= $MPboundary . "\r\n";
				$multipartbody .= 'content-disposition: form-data; name="' . $parameter . "\"\r\n\r\n";
				$multipartbody .= $value."\r\n";
			}

		}
		
		$multipartbody .= $endMPboundary;
		return $multipartbody;
	}
	
    public static function build_http_query($params) {
        if (!$params){
        	return '';
        }elseif(!is_array($params)){
        	return $params;
        }
        
        //value无数组时，采取内置函数进行编码
        if(!self::_check_value_has_arr($params)){
        	return http_build_query($params);
        }

        // Urlencode both keys and values
        $keys = self::urlencode_rfc3986(array_keys($params));
        $values = self::urlencode_rfc3986(array_values($params));
        $params = array_combine($keys, $values);

        // Parameters are sorted by name, using lexicographical byte value ordering.
        // Ref: Spec: 9.1.1 (1)
        //OAuth 2.0无排序要求，忽略
        //uksort($params, 'strcmp');

        $pairs = array();
        foreach ($params as $parameter => $value) {
            if (is_array($value)) {
                // If two or more parameters share the same name, they are sorted by their value
                // Ref: Spec: 9.1.1 (1)
                //OAuth 2.0无排序要求，忽略。
                //natsort($value);
                foreach ($value as $duplicate_value) {
                    $pairs[] = $parameter . '=' . $duplicate_value;
                }
            } else {
                $pairs[] = $parameter . '=' . $value;
            }
        }
        // For each parameter, the name is separated from the corresponding value by an '=' character (ASCII code 61)
        // Each name-value pair is separated by an '&' character (ASCII code 38)
        return implode('&', $pairs);
    }
    
    public static function _check_value_has_arr($params){
    	if(!is_array($params) || empty($params)){
    		return false;
    	}
    	$has_arr = false;
    	foreach($params as $val){
    		if(is_array($val)){
    			$has_arr = true;
    			break;
    		}
    	}
    	return $has_arr;
    }
    
    public static function urlencode_rfc3986($input) {
        if (is_array($input)) {
            return array_map(array('xwbOAuthV2_tool_http', 'urlencode_rfc3986'), $input);
        } else if (is_scalar($input)) {
            return str_replace(
                '+',
                ' ',
                str_replace('%7E', '~', rawurlencode($input))
            );
        } else {
            return '';
        }
    }
	
	
	/**
	 * 获取一条URL指令
	 * @param string $url
	 * @param array|string $paramter
	 */
	public static function get_url($url, $paramter = ''){
		if(empty($paramter)){
			return $url;
		}
		if(is_array($paramter)){
			$paramter = xwbOAuthV2_tool_http::build_http_query($paramter);
		}
		
		if(strpos($url, '?') === false){
			return $url. '?'. $paramter;
		}else{
			return $url. '&'. $paramter;
		}
	}
	
	public static function json_decode($result){
		return json_decode(preg_replace('#(?<=[,\{\[])\s*("\w+"):(\d{6,})(?=\s*[,\]\}])#si', '${1}:"${2}"', $result), true);
	}
	
}