(function(window, undefined){


/**
 * @module Anim
 * @namespace Anim
 */
var Anim = {
    /**
     * merge
     * @memberOf Anim
     * @param  {Object} to
     * @param  {Object} from
     * @return {Object}
     */
    merge:function(to, from){
        for(var i in from){
            to[i] = from[i];
        }
        return to;
    },
    /**
     * clone
     * @memberOf Anim
     * @param  {Object} obj
     * @return {Object}
     */
    clone:function(obj){
        if(typeof obj === "object"){
            var copy = Object.prototype.toString.call(obj) === "[object Array]"?[]:{};
            for(var i in obj){
                if(obj.hasOwnProperty(i)){
                    copy[i] = this.clone(obj[i]);
                }
            }
            return copy;
        }
        else{
            return obj;
        }
    }
};

var ua = navigator.userAgent.toLowerCase();
Anim.cssVendor = /webkit/.test(ua)?"webkit":/firefox/.test(ua)?"Moz":/msie|like gecko/.test(ua)?"ms":/opera/.test(ua)?"O":"webkit";
Anim.supportTransform3d = true;
if(Anim.cssVendor === "ms"){
    var version = ua.match(/msie ([\d.]+)/);
    if(version && version[1] && parseInt(version[1], 10) <= 9){
        Anim.supportTransform3d = false;
    }
}
window.Anim = Anim;

})(window);

(function(window, undefined){

var Anim = window.Anim;

/**
 * @module Anim/EventMix
 * @mixin EventMix
 * @requires Anim
*/
var EventMix = {
    /**
     * 监听事件
     * @param {String} type 事件类型
     * @param {Function} listener
    */
    on:function(type, listener){
        this._listener = this._listener||{};
        this._listener[type] = this._listener[type]||[];
        this._listener[type].push(listener);
    },
    /**
     * 触发事件
     * @param {String} type 事件类型
     * @param {Object} data
    */
    fire:function(type, data){
        this._listener = this._listener||{};
        var listeners = this._listener[type];
        if(listeners && listeners.length){
            listeners = listeners.slice();
            for(var i = 0, l = listeners.length;i < l;i ++){
                var e = {
                    target:this,
                    type:type
                };
                Anim.merge(e, data||{});
                listeners[i] && listeners[i].call(this, e);
            }
        }
    },
    /**
     * 取消监听
     * @param {String} type 要删除的事件类型, 若不填删除所有事件监听
     * @param {Function} listener 要删除的事件回调函数，若不填删除该类型所有事件监听
    */
    off:function(type, listener){
        if(!type){
            this._listener = {};
        }

        this._listener = this._listener||{};
        var listeners = this._listener[type];
        if(listeners){
            if(listener){
                var index = listeners.indexOf(listener);
                if(index > -1){
                    listeners.splice(index, 1);
                }
            }
            else{
                delete this._listener[type];
            }
        }
    }
};
Anim.EventMix = EventMix;

})(window);

(function(window, undefined){

var Anim = window.Anim;
var EventMix = Anim.EventMix;

var animID = 0;
var baseStyleID = "baseAnimStyle";
/**
 * @module Anim/AnimCSS
 * @requires Anim
 * @requires Anim/EventMix
 * @mixes EventMix
 * @class  AnimCSS
*/

var AnimCSS = function(elems, display){
    var that = this;
    this.display = display;
    this.elems = elems;
    Anim.merge(this, EventMix);
    elems[0].addEventListener(Anim.cssVendor + "AnimationEnd", function(){
        that.fire("animationEnd");
    });
};

AnimCSS.prototype = {
    constructor:AnimCSS,
    /**
     * 播放
     */
    play:function(){
        var elems = this.elems;
        elems.forEach(function(elem){
            var animName = elem.getAttribute("data-anim");
            elem.className = "flashAnim";
            setTimeout(function(){
                elem.className = animName + " flashAnim"
            }, 100);
            elem.style[Anim.cssVendor + "AnimationPlayState"] = "running";
        });
    },
    /**
     * 停止
     */
    stop:function(){
        var elems = this.elems;
        elems.forEach(function(elem){
            var animName = elem.getAttribute("data-anim");
            elem.className = "flashAnim";
            setTimeout(function(){
                elem.className = animName + " flashAnim"
            }, 100);
            elem.style[Anim.cssVendor + "AnimationPlayState"] = "paused";
        });
    },
    /**
     * 暂停
     */
    pause:function(){
        var elems = this.elems;
        elems.forEach(function(elem){
            elem.style[Anim.cssVendor + "AnimationPlayState"] = "paused";
        });
    },
    /**
     * 恢复
     */
    resume:function(){
        var elems = this.elems;
        elems.forEach(function(elem){
            elem.style[Anim.cssVendor + "AnimationPlayState"] = "running";
        });
    }
};

/**
 * build
 * @static
 * @param {Object} cfg
 * @param {Object} cfg.data 导出的动画数据
 * @param {String} cfg.image 图片地址
 * @param {HTMLElement} cfg.container 动画的容器，默认为document.body
 * @param {Number} cfg.iterationCount 循环次数，0为无限循环，默认无限循环
 * @param {Number} cfg.fps 帧频，默认为导出数据中的帧频
*/
AnimCSS.build = function(cfg){
    var that = this;
    var data = cfg.data;
    var image = cfg.image;
    var elems = [];

    if(!data || !image){
        console.warn("no data or no image!");
        return;
    }

    var container = cfg.container||document.body;
    var fps = cfg.fps||data.stage.fps;
    var iterationCount = cfg.iterationCount||"infinite";

    var animContainer = document.createElement("div");
    animContainer.className = "flashAnimContainer";
    animContainer.style.position = "relative";
    animContainer.style.width = cfg.data.stage.width + "px";
    animContainer.style.height = cfg.data.stage.height + "px";
    animContainer.style.overflow = "hidden";
    container.appendChild(animContainer);

    data.layers.forEach(function(layer, index){
        var elem = that._parseLayer(animContainer, layer, image, index, data.texture, fps, iterationCount);
        if(elem){
            elems.push(elem);
        }
    });

    var baseStyleElem = document.getElementById(baseStyleID);
    if(!baseStyleElem){
        this._createStyle(" \
            .flashAnim{ \
                position:absolute; \
                left:0px; \
                top:0px; \
                -webkit-animation-fill-mode: both !important; \
                -Moz-animation-fill-mode: both !important; \
                animation-fill-mode: both !important; \
            } \
        ", baseStyleID);
    }

    var anim = new AnimCSS(elems, animContainer);
    return anim;
};


/**
 * 解析每个元件
 * @private
 * @param {HTMLElement} container
 * @param {Object} layerData 层数据
 * @param {String} image 图片地址
 * @param {Number} index 层级
 * @param {Object} texture 图片数据
 * @param {Number} fps 帧频
 * @param {Number} iterationCount 播放次数
 * @return {Object} 这层的元件
*/
AnimCSS._parseLayer = function(container, layerData, image, index, texture, fps, iterationCount){
    var that = this;
    var cssVendor = Anim.cssVendor;
    var frames = layerData.frames;
    if(!frames || !frames.length){
        console.warn("no frames:", layerData.name);
        return;
    }

    var duration = frames.reduce(function(item, next){
        return {
            duration:item.duration + next.duration
        }
    }).duration;

    var styles = [];
    var t = 0;
    var r = 0;
    var lastStyle;
    var layerImgData = {
        x:0,
        y:0,
        w:0,
        h:0
    };

    for(var i = 0, l = frames.length;i<l;i ++){
        var frame = frames[i];
        if(frame.image && texture[frame.image]){
            layerImgData = texture[frame.image];
            break;
        }
    }

    frames.forEach(function(frame, i){
        var timingCss = "";
        var useStep = false;
        if(useStep){
            timingCss = "-" + cssVendor + "-animation-timing-function:step-end;\n";
        }
        else if(lastStyle && !lastStyle.tween){
            lastStyle.num = t/duration*100 - 0.0001;
            styles.push(lastStyle);
        }

        var elem = frame.elem;
        var imgData = texture[frame.image];
        if(elem && imgData){
            styles.push({
                x:elem.x - elem.originX,
                y:elem.y - elem.originY,
                scaleX:elem.scaleX,
                scaleY:elem.scaleY,
                rotation:elem.rotation,
                originX:elem.originX,
                originY:elem.originY,
                tween:frame.tween,
                num:t/duration*100,
                alpha:elem.alpha * 0.01,
                timing:frame.tween?"":timingCss,
                imgX:imgData.x,
                imgY:imgData.y,
                width:imgData.w,
                height:imgData.h
            });
        }
        else{
            styles.push({
                num:t/duration*100,
                scaleX:1,
                scaleY:1,
                alpha:0,
                rotation:0,
                x:0,
                y:0,
                originX:0,
                originY:0,
                tween:false,
                timing:timingCss,
                imgX:layerImgData.x,
                imgY:layerImgData.y,
                width:layerImgData.w,
                height:layerImgData.h
            });
        }

        lastStyle = Anim.merge({}, styles[styles.length-1]);
        t+=frame.duration;
    });

    var endStyle = Anim.merge({}, styles[styles.length-1]);
    endStyle.num = 100;
    endStyle.timing = "";
    styles.push(endStyle);

    var elemData = {
        index:100-index,
        duration:duration/fps,
        image:image
    };
    var animName = layerData.name + (animID++);
    that._addStyle(styles, elemData, animName, iterationCount);
    var elem = document.createElement("div");
    elem.className = animName + " flashAnim";
    elem.setAttribute("data-anim", animName);
    elem.setAttribute("data-layer", layerData.name);
    container.appendChild(elem);

    return elem;
};

/**
 * 给元件加动画样式
 * @private
 * @param {Array} styles
 * @param {Object} elemData
 * @param {String} animName
*/
AnimCSS._addStyle = function(styles, elemData, animName, iterationCount){
    var that = this;
    var cssVendor = Anim.cssVendor;
    var keyTpl = "\n\
        @-{cssVendor}-keyframes {anim}{\n\
        {content}\n\
        }\n\
        .{anim}{\n\
            background:url({image}) no-repeat;\n\
            z-index:{index};\n\
            -{cssVendor}-animation:{anim} {duration}s linear 0s {iterationCount};\n\
        }\n\
        ";

    var percentTpl = "\
        {num}% {\n\
            -{cssVendor}-transform:translate3d({x}px, {y}px, 0px) rotateZ({rotation}deg) scale3d({scaleX}, {scaleY}, 1);\n\
            -{cssVendor}-transform-origin:{originX}px {originY}px;\n\
            width:{width}px;\n\
            height:{height}px;\n\
            background-position:-{imgX}px -{imgY}px;\n\
            opacity:{alpha};\n\
            {timing}\
        }\n";

    var content = "";
    styles.forEach(function(s){
        s.cssVendor = cssVendor;
        content += that._renderTpl(percentTpl, s);
    });
    var style = that._renderTpl(keyTpl, Anim.merge({
        iterationCount:iterationCount,
        anim:animName,
        content:content,
        cssVendor:cssVendor
    },elemData));

    this._createStyle(style, "flashAnimStyle_" + animName);
};

/**
 * 插入css
 * @private
 * @param  {String} content css内容
 * @param  {String} id      css id
 */
AnimCSS._createStyle = function(content, id){
    var styleElem = document.createElement("style");
    styleElem.innerHTML = content;
    styleElem.id = id;
    document.getElementsByTagName("head")[0].appendChild(styleElem);
};

/**
 * 替换模板
 * @private
 * @param {String} tpl
 * @param {Object} data
 * @return {String} 替换后内容
*/
AnimCSS._renderTpl = function(tpl, data){
    data = data||{};
    for(var i in data){
        var reg = new RegExp("{" + i + "}", "g");
        tpl = tpl.replace(reg, data[i]);
    }
    return tpl;
};

/**
 * 动画结束事件
 * @event AnimCSS#animationEnd
 * @type {object}
 */
Anim.AnimCSS = AnimCSS;

})(window);

(function(window, undefined){

var Anim = window.Anim;
var EventMix = Anim.EventMix;

var zeroData = {
    alpha:0,
    scaleX:0,
    scaleY:0,
    x:0,
    y:0,
    rotation:0,
    originX:0,
    originY:0,
    image:null
};

/**
 * @module Anim/AnimElem
 * @requires Anim
 * @requires Anim/EventMix
 * @class  AnimElem
 * @property {bridge} bridge
 * @property {Display} display
 * @property {Array} frames
 * @property {Number} frameTime
 * @property {Object} layer
 * @property {String} name
 * @param {Object} data
*/
function AnimElem(data){
    this._diaplayData = null;
    this._currentTime = 0;
    this._currentIndex = 0;
    this.bridge = data.bridge;
    this.display = data.display;
    this.frames = data.frames;
    this.layer = data.layer;
    this.frameTime = data.frameTime;
    this.name = data.name;
    this.texture = data.texture;

    Anim.merge(this, EventMix);
}

AnimElem.prototype = {
    constructor:AnimElem,
    /**
     * @param {Object} data
    */
    _renderDisplay:function(data, image){
        if(this._diaplayData !== data){
            this._diaplayData = data;
            if(data){
                this.bridge.renderDisplay(this.display, data);
                if(this._image !== image){
                    this._image = image;
                    var rect = this.texture[image];
                    this.bridge.setRect(this.display, [rect.x, rect.y, rect.w, rect.h]);
                }
            }
            else{
                this.bridge.renderDisplay(this.display, zeroData);
            }
        }
    },
    /**
     * 跳转到某时间
     * @param  {Number} time 时间，单位毫秒
     */
    gotoTime:function(time){
        var frames = this.frames;
        var index, deltaTime, currentFrame, nextFrame;

        if(time >= this._currentTime){
            index = this._currentIndex;
        }
        else{
            index = 0;
        }
        this._currentTime = time;

        for(var i = index, l = frames.length;i < l;i ++){
            var frame = frames[i];
            if(time <= frame.endTime){
                deltaTime = time - frame.startTime;
                currentFrame = frame;
                nextFrame = frames[i + 1];
                this._currentIndex = i;
                break;
            }
        }

        if(!currentFrame){
            var lastIndex = frames.length - 1;
            deltaTime = 0;
            this._currentIndex = lastIndex;
            currentFrame = frames[lastIndex];
        }

        if(currentFrame.tween && nextFrame && nextFrame.elem){
            var t = deltaTime/currentFrame.duration;
            var deltaData = {};
            for(var prop in currentFrame.elem){
                deltaData[prop] = currentFrame.elem[prop] + t * (nextFrame.elem[prop] - currentFrame.elem[prop]);
            }
            deltaData.originX = currentFrame.elem.originX;
            deltaData.originY = currentFrame.elem.originY;
            this._renderDisplay(deltaData, currentFrame.image);
        }
        else{
            this._renderDisplay(currentFrame.elem, currentFrame.image);
        }
    },
    /**
     * 跳转到某帧
     * @param {Number} frameNum
    */
    gotoFrame:function(frameNum){
        this.gotoTime(frameNum * this.frameTime);
    },
    /**
     * 获取总时间
    */
    getTotalTime:function(){
        return this.frames[this.frames.length - 1].endTime;
    }
};
Anim.AnimElem = AnimElem;

})(window);

(function(window, undefined){

var Anim = window.Anim;
var EventMix = Anim.EventMix;
var AnimElem = Anim.AnimElem;

/**
 * @module Anim/AnimJS
 * @requires Anim
 * @requires Anim/EventMix
 * @requires Anim/AnimElem
 * @class AnimJS
 * @mixes EventMix
 * @property {Array} elems 元素
 * @property {Number} timeScale 时间缩放 默认为1
 * @property {display} display 显示容器
 * @property {Number} iterationCount 播放次数
 * @property {Boolean} _isPlay 是否播放
*/
function AnimJS(data){
    var that = this;

    this.elems = data.elems||[];
    this.timeScale = data.timeScale||1;
    this.frameTime = data.frameTime;
    this.display = data.display;
    this.iterationCount = data.iterationCount;
    this.bridge = data.bridge;
    this.container = data.container;
    this.width = data.width;
    this.height = data.height;
    this._data = data.data;
    this._isPlay = false;

    this._elemDict = {};
    for(var i = 0, l = this.elems.length;i < l;i ++){
        this._elemDict[this.elems[i].name] = this.elems[i];
    }

    Anim.merge(this, EventMix);

    this._currentTime = 0;
    this._startTime = 0;
    this._endTime = this._totalTime = this.elems[0].getTotalTime();

    this._setActions(data.data.actions);
}

AnimJS.prototype = {
    constructor:AnimJS,
    /**
     * 销毁
     */
    destroy:function(){
        this._isPlay = false;
        this.elems = [];
        this.bridge.destroy(this.container);
        var index = AnimJS._anims.indexOf(this);
        if(index > -1){
            AnimJS._anims.splice(index, 1);
        }
        this.off();
    },
    /**
     * 需每帧调用
     * @param {Number} dt
    */
    tick:function(dt){
        if(this._isPlay){
            dt *= this.timeScale;

            var time = this._currentTime;
            time += dt;
            if(time > this._endTime){
                this.fire("animationEnd");
                this.iterationCount --;
                if(this.iterationCount <= 0){
                    this.pause();
                }
                else{
                    time = this._startTime + time - this._endTime;
                }
            }

            this.gotoTime(time);
        }
    },
    /**
     * 跳转到第几帧播放
     * @param {Number|String} frameNum 跳转的帧数或名字，默认第0帧
     * @param {Number} iterationCount 播放次数，0为无数次，默认无数次
    */
    play:function(frameNum, iterationCount){
        if(typeof frameNum === "string"){
            var action = this.action[frameNum];
            if(action){
                this._startTime = action.startTime;
                this._endTime = action.endTime;
                this.gotoTime(action.startTime);
            }
            else{
                console.warn("no action:", frameNum);
            }
        }
        else if(typeof frameNum === "number"){
            this._startTime = 0;
            this._endTime = this._totalTime;
            this.gotoFrame(frameNum);
        }

        this._isPlay = true;
        if(iterationCount !== undefined){
            this.iterationCount = iterationCount||Infinity;
        }

    },
    /**
     * 跳转到第几帧停止
     * @param {Number} frameNum 跳转的帧数，默认第0帧
    */
    stop:function(frameNum){
        this.gotoFrame(frameNum);
        this._isPlay = false;
    },
    /**
     * 暂停
    */
    pause:function(){
        this._isPlay = false;
    },
    /**
     * 恢复
    */
    resume:function(){
        this._isPlay = true;
    },
    /**
     * 获取某个元素
     * @param {String} name
     * @return {AnimElem}
    */
    getElem:function(name){
        return this._elemDict[name];
    },
    /**
     * 跳转到某时间
     * @param  {Number} time 时间，单位毫秒，默认第0秒
     */
    gotoTime:function(time){
        time = time||0;
        if(time > this._endTime){
            time = this._endTime;
        }
        this._currentTime = time;

        var elems = this.elems;
        for(var i = this.elems.length - 1;i >= 0;i --){
            elems[i].gotoTime(time);
        }
    },
    /**
     * 跳转到第几帧
     * @param {Number} frameNum 跳转的帧数，默认第0帧
    */
    gotoFrame:function(frameNum){
        frameNum = frameNum||this._startTime;
        this.gotoTime(frameNum * this.frameTime)
    },
    /**
     * 生成动作帧数据
     * @param {Object} actions 动作数据
     */
    _setActions:function(actions){
        if(actions){
            var arr = [];
            var action = {};
            for(var name in actions){
                arr.push({
                    name:name,
                    num:actions[name]
                });
                arr.sort(function(a, b){
                    return a.num - b.num
                });

                for(var i = 0;i < arr.length;i ++){
                    var act = arr[i];
                    var end = arr[i+1]?(arr[i+1].num)*this.frameTime:this._totalTime;
                    action[act.name] = {
                        startTime:act.num * this.frameTime,
                        endTime:end
                    }
                }
            }
            this.action = action;
        }
    }
};

AnimJS._anims = [];

/**
 * @static
 * @param {Object} cfg
 * @param {Object} cfg.data 导出的动画数据
 * @param {String} cfg.image 图片地址
 * @param {Container} cfg.container 动画的容器
 * @param {Number} cfg.iterationCount 循环次数，0为无限循环，默认无限循环
 * @param {Number} cfg.fps 帧频，默认为导出数据中的帧频
 * @param {DisplayBridge} bridge 显示对象桥梁，默认是canvasBridge
 * @return {AnimJS}
*/
AnimJS.build = function(cfg, bridge){
    var that = this;
    var data = cfg.data;
    var image = new Image();
    image.src = cfg.image;
    bridge = bridge||Anim.canvasBridge;
    if(data){
        data = Anim.clone(data);
        var fps = cfg.fps||data.stage.fps;
        var frameTime = 1000/fps;
        var iterationCount = cfg.iterationCount||Infinity;

        var container = bridge.createContainer(data.stage.width, data.stage.height);
        var elems = [];
        var layers = data.layers;
        for(var i = layers.length - 1;i >= 0;i --){
            var elem = that._parseLayer(bridge, container, layers[i], image, data.texture, frameTime);
            if(elem){
                elems.push(elem);
            }
        }

        var anim = new AnimJS({
            frameTime:frameTime,
            width:data.stage.width,
            height:data.stage.height,
            data:data,
            timeScale:1,
            display:container,
            elems:elems,
            iterationCount:iterationCount,
            bridge:bridge,
            container:container
        });

        if(cfg.container){
            bridge.addDisplay(cfg.container, container);
        }
        this._anims.push(anim);
        return anim;
    }
};

/**
 * @param {Number} dt 时间间隔，单位毫秒
*/
AnimJS.tick = function(dt){
    var anims = this._anims;
    for(var i = anims.length - 1;i >= 0;i --){
        anims[i].tick(dt);
    }
};

/**
 * @method _parseLayer
 * @private
 * @static
 * @param {DisplayBridge} bridge
 * @param {Stage} container
 * @param {Object} layer 层数据
 * @param {String} image 图片地址
 * @param {Object} texture 图片数据
 * @param {Number} frameTime 每帧时间，单位毫秒
 * @param {Number} time 播放次数
*/
AnimJS._parseLayer = function(bridge, container, layer, image, texture, frameTime, time){
    var frames = layer.frames;
    if(frames && frames.length){
        var t = 0;
        for(var i = 0, l = frames.length; i < l;i ++){
            frames[i].duration *= frameTime;
            frames[i].startTime = t;
            t += frames[i].duration;
            frames[i].endTime = t;
        }

        var display = bridge.createDisplay(image);
        bridge.addDisplay(container, display);
        var elem = new AnimElem({
            bridge:bridge,
            display:display,
            frames:frames,
            layer:layer,
            frameTime:frameTime,
            name:layer.name,
            texture:texture
        });
        elem.gotoTime(0);
        return elem;
    }
    else{
        console.warn("no frames:" + layer.name);
    }
};

/**
 * 动画结束事件
 * @event AnimJS#animationEnd
 * @type {object}
 */
Anim.AnimJS = AnimJS;

})(window);

(function(window, undefined){

var Anim = window.Anim;

/**
 * @module Anim/domBridge
 * @requires Anim
 * 显示对象桥梁，可以重写适配不同的渲染库
*/

var domBridge = {
    /**
     * 生成容器
     * @param {Number} width
     * @param {Number} height
    */
    createContainer:function(width, height){
        var container = document.createElement("div");
        container.style.width = width + "px";
        container.style.height = height + "px";
        container.style.position = "relative";
        container.style.overflow = "hidden";
        container.className = "flashAnimContainer";

        return container;
    },
    /**
     * 生成显示对象
     * @param {Image} image 图片
    */
    createDisplay:function(image){
        var display = document.createElement("div");
        var style = display.style;
        style.background = "url(" + image.src + ") no-repeat";
        style.position = "absolute";
        style.left = "0px";
        style.top = "0px";
        return display;
    },
    /**
     * 设置显示区域
     * @param {Display} display
     * @param {Array} rect 图片区域, 格式:[x, y, width, height]
     */
    setRect:function(display, rect){
        var style = display.style;
        style.backgroundPosition = "-" + rect[0] + "px -" + rect[1] + "px";
        style.width = rect[2] + "px";
        style.height = rect[3] + "px";
    },
    /**
     * 将显示对象加到容器中
     * @param {Container} container
     * @param {Display} display
     * @param {Number} index 层级，0为最下层，默认加到最上层
    */
    addDisplay:function(container, display, index){
        if(index === undefined){
            index = container.childNodes.length;
        }
        display.style.zIndex = index;
        container.appendChild(display);
    },
    /**
     * 渲染显示对象
     * @param {Display} display
     * @param {Object} data 显示对象属性
     * @param {Number} data.x x位移
     * @param {Number} data.y y位移
     * @param {Number} data.originX 中心点x
     * @param {Number} data.originY 中心点y
     * @param {Number} data.scaleX x方向缩放
     * @param {Number} data.scaleY y方向缩放
     * @param {Number} data.rotation 角度，角度制
     * @param {Number} data.alpha 范围0-100, 0为完全透明, 100完全不透明
    */
    renderDisplay:function(display, data){
        var style = display.style;

        if(data.alpha <= 0){
            style.opacity = 0;
            return;
        }

        var cssVendor = Anim.cssVendor;
        style[cssVendor + "TransformOrigin"] = data.originX + "px " + data.originY + "px";
        style.opacity = data.alpha * 0.01;

        if(Anim.supportTransform3d){
            style[cssVendor + "Transform"] =
            "translate3d(" + (data.x - data.originX) + "px," + (data.y - data.originY) + "px,0px) " +
            "rotate3d(0,0,1," + data.rotation + "deg) " +
            "scale3d(" + data.scaleX + "," + data.scaleY + ",1" + ")";
        }
        else{
            style[cssVendor + "Transform"] =
            "translate(" + (data.x - data.originX) + "px," + (data.y - data.originY) + "px) " +
            "rotate(" + data.rotation + "deg) " +
            "scale(" + data.scaleX + "," + data.scaleY + ")";
        }

    },
    destroy:function(container){
        if(container.parentNode){
            container.parentNode.removeChild(container);
        }
    }
};
Anim.domBridge = domBridge;

})(window);

(function(window, undefined){

var Anim = window.Anim;

/**
 * @module Anim/canvasBridge
 * @requires Anim
 * 显示对象桥梁，可以重写适配不同的渲染库
*/
var DEG2RAD = Math.PI/180;
var canvasBridge = {
    /**
     * 生成容器
     * @param {Number} width
     * @param {Number} height
    */
    createContainer:function(width, height){
        var that = this;
        var container = document.createElement("canvas");
        container.className = "flashAnimContainer";
        container.style.display = "block";

        container.width = width;
        container.height = height;
        var ctx = container.getContext("2d");

        container.views = [];
        container.interval = setInterval(function(){
            ctx.clearRect(0, 0, container.width, container.height);
            that._render(container.views, ctx);
        }, 1000/30);

        return container;
    },
    /**
     * 生成显示对象
     * @param {Image} image 图片
    */
    createDisplay:function(image){
        var display = {
            image:image,
            rect:[0, 0, 0, 0]
        };

        return display;
    },
    /**
     * 设置显示区域
     * @param {Display} display
     * @param {Array} rect 图片区域, 格式:[x, y, width, height]
     */
    setRect:function(display, rect){
        display.rect = rect;
    },
    /**
     * 将显示对象加到容器中
     * @param {Container} container
     * @param {Display} display
     * @param {Number} index 层级，0为最下层，默认加到最上层
    */
    addDisplay:function(container, display, index){
        if(display.getContext){
            container.appendChild(display);
        }
        else{
            if(index === undefined){
                index = container.views.length;
            }
            container.views.splice(index, 0, display);
        }
    },
    /**
     * 渲染显示对象
     * @param {Display} display
     * @param {Object} data 显示对象属性
     * @param {Number} data.x x位移
     * @param {Number} data.y y位移
     * @param {Number} data.originX 中心点x
     * @param {Number} data.originY 中心点y
     * @param {Number} data.scaleX x方向缩放
     * @param {Number} data.scaleY y方向缩放
     * @param {Number} data.rotation 角度，角度制
     * @param {Number} data.alpha 范围0-100, 0为完全透明, 100完全不透明
    */
    renderDisplay:function(display, data){
        display.x = data.x;
        display.y = data.y;
        display.originX = data.originX;
        display.originY = data.originY;
        display.scaleX = data.scaleX;
        display.scaleY = data.scaleY;
        display.rotation = data.rotation;
        display.alpha = data.alpha * 0.01;
    },
    _render:function(views, ctx){
        for(var i = 0, l = views.length;i < l;i ++){
            var view = views[i];
            if(view.alpha <= 0 || !(view.image.complete)){
                continue;
            }

            ctx.globalAlpha = view.alpha;
            ctx.save();
            ctx.translate(view.x, view.y);
            ctx.rotate(view.rotation * DEG2RAD);
            ctx.scale(view.scaleX, view.scaleY);
            ctx.translate(-view.originX, -view.originY);
            var rect = view.rect;
            ctx.drawImage(view.image, rect[0], rect[1], rect[2], rect[3], 0, 0, rect[2], rect[3]);
            ctx.restore();
        }
    },
    destroy:function(container){
        clearInterval(container.interval);
        delete container.views;
        if(container.parentNode){
            container.parentNode.removeChild(container);
        }
    }
};
Anim.canvasBridge = canvasBridge;

})(window);
