<?php
/**
 * Displays a select with categories
 */
GF_Field::add_field( 'select_term',__( 'Select Term', 'gf' ) );
class GF_Field_Select_Term extends GF_Field_Select {
	protected $taxonomy = 'category';

	function set_taxonomy( $taxonomy ) {
		if(!taxonomy_exists($taxonomy)) {
			gf_die( __("<strong>GF_Field_Select_Term</strong>: Invalid taxonomy! Please check if you've spelled the name of the taxonomy correctly and that the taxonomy is already registered!", 'gf') );
		}
		$this->taxonomy = $taxonomy;

		return $this;
	}

	protected function check_options( $default_message ) {
		switch($this->taxonomy) {
			case 'category':
				$filter_name = 'single_cat_title';
				break;
			case 'tag':
				$filter_name = 'single_tag_title';
				break;
			default:
				$filter_name = 'single_term_title';
		}

		$terms = get_terms($this->taxonomy, array( 'hide_empty' => false ));
		foreach($terms as $term) {
			$this->options[$term->term_id] = apply_filters($filter_name, $term->name);
		}

		if(empty($this->options)) {
			if(!$this->no_options_message)
				$this->no_options_message = $default_message;

			echo '<p class="only-child">' . $this->no_options_message . '</p>';
			return false;
		}		

		return true;
	}

	/**
	 * Returns a description for the field, will be used in the settings
	 * 
	 * @return string The description
	 */
	static public function settings_description() {
		return __( 'Displays a drop-down with all available terms from a taxonomy.', 'gf' );
	}

	/**
	 * Adds additional fields to the settings pages
	 * 
	 * @return GF_Field[]
	 */
	static public function additional_settings() {
		$taxonomies = array();
		$taxonomies_raw = get_taxonomies( '', 'objects' ); 
		foreach( $taxonomies_raw as $taxonomy ) {
			$taxonomies[$taxonomy->name] = $taxonomy->labels->name;
		}

		return array(
			GF_Field::factory( 'select', 'taxonomy', __( 'Taxonomy', 'gf' ) )
				->add_options( $taxonomies ),
			GF_Field::factory( 'separator', 'output_data_separator', __( 'Output Settings', 'gf' ) )
				->set_description( __( 'The following settings will affect the output of this field when using the &quot;gf&quot; function or shortcode.', 'gf' ) ),
			GF_Field::factory( 'select', 'output_type', __( 'Output Type', 'gf' ) )
				->add_options( array(
					'link'  => __( 'A link to the chosen term', 'gf' ),
					'url'   => __( 'The URL of the chosen term', 'gf' ),
					'title' => __( 'The name of the chosen term', 'gf' ),
					'id'    => __( 'The ID of the chosen term', 'gf' )
				) )
		);
	}

	/**
	 * Process the value based on the settings in the admin.
	 * 
	 * @param int $value The value of the field
	 * @param mixed $data The settings of the field, added through the Gravity Fields section
	 * 
	 * @return string The content to be shown in the editor.
	 */
	public function process_value( $value, $data ) {
		if( ! isset( $data[ 'output_type' ] ) ) {
			$data[ 'output_type' ] = 'link';
		}

		switch( $data[ 'output_type' ] ) {
			case 'link':
				$title = apply_filters( 'single_cat_title', get_term( intval( $value ), $data[ 'taxonomy' ] )->name );
				$url = get_term_link( intval( $value ), $data[ 'taxonomy' ] );
				return '<a href="' . esc_attr( $url ) . '">' . $title . '</a>';
				break;

			case 'url':
				return get_term_link( intval( $value ), $data[ 'taxonomy' ] );
				break;

			case 'title':
				return apply_filters( 'single_cat_title', get_term( intval( $value ), $data[ 'taxonomy' ] )->name );
				break;
		}

		return $value;	
	}
}