<?php
/**
 * Adds fields to user edit screens.
 * 
 * Provides standard factory-like methods, including add_fields()
 */
class GF_Usermeta extends GF_Container_Base implements GF_Container {
	/** @type string The identifier of the container */
	protected $id;

	/** @type string The title of the container */ 
	protected $title;

	/** @type string The description of the container */
	protected $description;

	/** @type GF_Field[] The fields of the container */
	protected $fields = array();

	/** @type GF_Datastore_Usermeta The default datastore for the container */
	protected $datastore;

	/** @type string Role that this container appears on */
	protected $role;

	/**
	 * Object contructor
	 *
	 * Adds neccessary actions and filters and validates container settings
	 *
	 * @param string $title The title of the panel. If this is variable, use the ID here
	 * @param mixed[] $args Additional arguments that are passed through the constructor
	 */
	public function __construct( $title, $args = null ) {
		# Process title
		$this->set_title( $title );

		# Prepare default datastore
		$this->datastore = new GF_Datastore_Usermeta();

		# Process args
		if( $args ) {
			if( is_array( $args ) ) {
				foreach( $args as $property => $value ) {
					if( method_exists( $this, 'set_' . $property ) ) {
						call_user_func( array( $this, 'set_' . $property ) , $value );
					} else {
						gf_die( '<strong>GF_Usermeta</strong>: ' . $property . ' is not a valid argument!' );
					}
				}
			} else {
				gf_die( '<strong>GF_Usermeta</strong>: Only arrays may be passed as options to the container!' );
			}
		}

		# Attach to the edit screen and the save action
		add_action( 'show_user_profile', array( $this, 'display' ) );
		add_action( 'edit_user_profile', array( $this, 'display' ) );
		add_action( 'personal_options_update', array( $this, 'save' ) );
		add_action( 'edit_user_profile_update', array( $this, 'save' ) );

		# Enqueue scripts
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Enqueues scripts in the admin
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( 'gravity-fields' );
		wp_enqueue_style( 'gravityfields-css' );
	}

	/**
	 * Detaches the panel by removing it's hooks and destroys the object and it's fields
	 */
	public function destroy() {
		# Remove hooks
		remove_action( 'show_user_profile', array( $this, 'display' ) );
		remove_action( 'edit_user_profile', array( $this, 'display' ) );
		remove_action( 'personal_options_update', array( $this, 'save' ) );
		remove_action( 'edit_user_profile_update', array( $this, 'save' ) );

		# Destroy fields
		foreach( $this->fields as $field ) {
			// Call the destroy method of the fields
			$field->destroy();

			unset( $field );
		}

		# Destroy this object
		unset( $this );
	}

	/**
	 * Container factory. Used for method chaining
	 * 
	 * @param string $title The title of the panel, same as __construct()
	 * @param mixed[] $args Same as above
	 * @return GF_Usermeta The new container
	 */
	public static function factory( $title, $args = null ) {
		return new GF_Usermeta( $title, $args );
	}

	/**
	 * Set a title of the container
	 *
	 * Although the title is the first argument for the constructor, it might be multilingual. In such cases, use this method
	 *
	 * @param string $title The title
	 * @return GF_Usermeta The current object
	 */
	public function set_title( $title ){
		$this->title = apply_filters( 'gf_usermeta_container_title', apply_filters( 'gf_container_title', $title, $this ), $this );

		# Set the title as an ID too, if there's no ID set. This happens when the container is added initially
		if( ! $this->id ) {
			$this->set_id( sanitize_title( $this->title ) );
		}

		return $this;
	}

	/**
	 * Get the title of the container
	 *
	 * @return string The title
	 */ 
	public function get_title(){
		return $this->title;
	}

	/**
	 * Set an ID of the container
	 *
	 * Use this method with multilingual contents to make sure that the address of the page wouldn't change
	 *
	 * @param string $id The new ID
	 * @return GF_Usermeta The current instance
	 */
	public function set_id( $id ){
		# Make sure the ID is sanitized and does not contain empty spaces, etc.
		$this->id = sanitize_title( $id );
		$this->id = apply_filters( 'gf_usermeta_container_id', apply_filters( 'gf_container_id', $this->id, $this ), $this );

		return $this;
	}

	/**
	 * Get the ID of the container
	 *
	 * @return string The identifier. Might be autogenerated, be careful
	 */
	public function get_id(){
		return $this->id;
	}

	/**
	 * Set a description of the container. This will appear after the section title
	 *
	 * @param string $description The new description
	 * @return GF_Usermeta The current instance
	 */
	public function set_description($description){
		$this->description = apply_filters( 'gf_usermeta_container_description', apply_filters( 'gf_container_description', $description, $this ), $this );

		return $this;
	}

	/**
	 * Get the description of the container
	 *
	 * @return string The description
	 */
	public function get_description(){
		return $this->description;
	}

	/**
	 * Add a field to the container
	 * 
	 * @param GF_Field $field The field to add
	 * @return GF_Usermeta The current instance
	 */
	public function add_field( GF_Field $field ) {
		$this->fields[] = $field;

		return $this;
	}

	/**
	 * Add multiple fields to the container
	 * 
	 * @param GF_Field[] $fields An array of fields
	 * @return GF_Usermeta The current instance
	 */
	public function add_fields( array $fields ) {
		foreach( $fields as $field ) {
			$this->add_field( $field );
		}

		return $this;
	}

	/**
	 * Display the fields
	 * 
	 * @param WP_User $user - The user whose profile is being edited
	 */
	public function display( WP_User $user ) {
		global $gravityfields;
		
		$this->datastore->set_user( $user );

		foreach( $this->fields as $field ) {
			if( ! $field->get_datastore() && $this->datastore->check_field_id( $field->get_id() ) ) {
				$field->set_datastore( $this->datastore, true );
			}
		}

		include( $gravityfields->themes->path( 'usermeta' ) );

		$this->output_dependencies();

		do_action('gf_after_container');
	}

	/**
	 * Save values
	 * 
	 * @param int $user_id - The ID of the user whose data is being saved
	 */
	public function save( $user_id ) {
		$this->datastore->set_user( get_user_by( 'id', $user_id ) );

		foreach( $this->fields as $field ) {
			if( ! is_a( $field, 'GF_Field' ) ) {
				continue;
			}
			
			if( ! $field->get_datastore() && $this->datastore->check_field_id( $field->get_id() ) ) {
				$field->set_datastore( $this->datastore, true );
			}

			$field->save( $_POST );
		}
	}
}