<?php
/*~ class.smtp.php
.---------------------------------------------------------------------------.
|  Software: PHPMailer - PHP email class                                    |
|   Version: 5.1                                                            |
|   Contact: via sourceforge.net support pages (also www.codeworxtech.com)  |
|      Info: http://phpmailer.sourceforge.net                               |
|   Support: http://sourceforge.net/projects/phpmailer/                     |
| ------------------------------------------------------------------------- |
|     Admin: Andy Prevost (project admininistrator)                         |
|   Authors: Andy Prevost (codeworxtech) codeworxtech@users.sourceforge.net |
|          : Marcus Bointon (coolbru) coolbru@users.sourceforge.net         |
|   Founder: Brent R. Matzelle (original founder)                           |
| Copyright (c) 2004-2009, Andy Prevost. All Rights Reserved.               |
| Copyright (c) 2001-2003, Brent R. Matzelle                                |
| ------------------------------------------------------------------------- |
|   License: Distributed under the Lesser General Public License (LGPL)     |
|            http://www.gnu.org/copyleft/lesser.html                        |
| This program is distributed in the hope that it will be useful - WITHOUT  |
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     |
| FITNESS FOR A PARTICULAR PURPOSE.                                         |
| ------------------------------------------------------------------------- |
| We offer a number of paid services (www.codeworxtech.com):                |
| - Web Hosting on highly optimized fast and secure servers                 |
| - Technology Consulting                                                   |
| - Oursourcing (highly qualified programmers and graphic designers)        |
'---------------------------------------------------------------------------'
*/

/**
 * PHPMailer - PHP SMTP email transport class
 * NOTE: Designed for use with PHP version 5 and up
 * @package PHPMailer
 * @author Andy Prevost
 * @author Marcus Bointon
 * @copyright 2004 - 2008 Andy Prevost
 * @license http://www.gnu.org/copyleft/lesser.html Distributed under the Lesser General Public License (LGPL)
 * @version $Id: class.smtp.php 444 2009-05-05 11:22:26Z coolbru $
 */

/**
 * SMTP is rfc 821 compliant and implements all the rfc 821 SMTP
 * commands except TURN which will always return a not implemented
 * error. SMTP also provides some utility methods for sending mail
 * to an SMTP server.
 * original author: Chris Ryan
 */

class Smtp_class {
	/**
	 * SMTP server port
	 * @var int
	 */
	public $SMTP_PORT = 25;
	
	/**
	 * SMTP reply line ending
	 * @var string
	 */
	public $CRLF = "\r\n";
	
	/**
	 * Sets whether debugging is turned on
	 * @var bool
	 */
	public $do_debug;
	
	/**
	 * Sets VERP use on/off (default is off)
	 * @var bool
	 */
	public $do_verp = false;
	
	/////////////////////////////////////////////////
	// PROPERTIES, PRIVATE AND PROTECTED
	/////////////////////////////////////////////////
	

	private $smtp_conn;
	private $error;
	private $helo_rply;
	
	/**
	 * Initialize the class so that the data is in a known state.
	 * @access public
	 * @return void
	 */
	public function __construct() {
		$this->smtp_conn = 0;
		$this->error = null;
		$this->helo_rply = null;
		
		$this->do_debug = 0;
	}
	
	/////////////////////////////////////////////////
	// CONNECTION FUNCTIONS
	/////////////////////////////////////////////////
	

	/**
	 * Connect to the server specified on the port specified.
	 * If the port is not specified use the default SMTP_PORT.
	 * If tval is specified then a connection will try and be
	 * established with the server for that number of seconds.
	 * If tval is not specified the default is 30 seconds to
	 * try on the connection.
	 *
	 * SMTP CODE SUCCESS: 220
	 * SMTP CODE FAILURE: 421
	 * @access public
	 * @return bool
	 */
	public function Connect($host, $port = 0, $tval = 30) {
		
		$this->error = null;
		
		if ($this->connected ()) {
			
			$this->error = array ("error" => "Already connected to a server" );
			return false;
		}
		
		if (empty ( $port )) {
			$port = $this->SMTP_PORT;
		}
		
		$this->smtp_conn = @fsockopen ( $host, $port, $errno, $errstr, $tval );
		
		if (empty ( $this->smtp_conn )) {
			$this->error = array ("error" => "Failed to connect to server", "errno" => $errno, "errstr" => $errstr );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": $errstr ($errno)" . $this->CRLF . '<br />';
			}
			return false;
		}
		
		if (substr ( PHP_OS, 0, 3 ) != "WIN")
			socket_set_timeout ( $this->smtp_conn, $tval, 0 );
		
		$announce = $this->get_lines ();
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $announce . $this->CRLF . '<br />';
		}
		
		return true;
	}
	
	/**
	 * Initiate a TLS communication with the server.
	 *
	 * SMTP CODE 220 Ready to start TLS
	 * SMTP CODE 501 Syntax error (no parameters allowed)
	 * SMTP CODE 454 TLS not available due to temporary reason
	 * @access public
	 * @return bool success
	 */
	public function StartTLS() {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called StartTLS() without being connected" );
			return false;
		}
		
		fputs ( $this->smtp_conn, "STARTTLS" . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 220) {
			$this->error = array ("error" => "STARTTLS not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		if (! stream_socket_enable_crypto ( $this->smtp_conn, true, STREAM_CRYPTO_METHOD_TLS_CLIENT )) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Performs SMTP authentication.  Must be run after running the
	 * Hello() method.  Returns true if successfully authenticated.
	 * @access public
	 * @return bool
	 */
	public function Authenticate($username, $password) {
		
		fputs ( $this->smtp_conn, "AUTH LOGIN" . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($code != 334) {
			$this->error = array ("error" => "AUTH not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		fputs ( $this->smtp_conn, base64_encode ( $username ) . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($code != 334) {
			$this->error = array ("error" => "Username not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		fputs ( $this->smtp_conn, base64_encode ( $password ) . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($code != 235) {
			$this->error = array ("error" => "Password not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns true if connected to a server otherwise false
	 * @access public
	 * @return bool
	 */
	public function Connected() {
		if (! empty ( $this->smtp_conn )) {
			$sock_status = socket_get_status ( $this->smtp_conn );
			if ($sock_status ["eof"]) {
				
				if ($this->do_debug >= 1) {
					echo "SMTP -> NOTICE:" . $this->CRLF . "EOF caught while checking if connected";
				}
				$this->Close ();
				return false;
			}
			return true;
		}
		return false;
	}
	
	/**
	 * Closes the socket and cleans up the state of the class.
	 * It is not considered good to use this function without
	 * first trying to use QUIT.
	 * @access public
	 * @return void
	 */
	public function Close() {
		$this->error = null;
		$this->helo_rply = null;
		if (! empty ( $this->smtp_conn )) {
			
			fclose ( $this->smtp_conn );
			$this->smtp_conn = 0;
		}
	}
	
	/////////////////////////////////////////////////
	// SMTP COMMANDS
	/////////////////////////////////////////////////
	

	/**
	 * Issues a data command and sends the msg_data to the server
	 * finializing the mail transaction. $msg_data is the message
	 * that is to be send with the headers. Each header needs to be
	 * on a single line followed by a <CRLF> with the message headers
	 * and the message body being seperated by and additional <CRLF>.
	 *
	 * Implements rfc 821: DATA <CRLF>
	 *
	 * SMTP CODE INTERMEDIATE: 354
	 * [data]
	 * <CRLF>.<CRLF>
	 * SMTP CODE SUCCESS: 250
	 * SMTP CODE FAILURE: 552,554,451,452
	 * SMTP CODE FAILURE: 451,554
	 * SMTP CODE ERROR  : 500,501,503,421
	 * @access public
	 * @return bool
	 */
	public function Data($msg_data) {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called Data() without being connected" );
			return false;
		}
		
		fputs ( $this->smtp_conn, "DATA" . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 354) {
			$this->error = array ("error" => "DATA command not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		$msg_data = str_replace ( "\r\n", "\n", $msg_data );
		$msg_data = str_replace ( "\r", "\n", $msg_data );
		$lines = explode ( "\n", $msg_data );
		
		$field = substr ( $lines [0], 0, strpos ( $lines [0], ":" ) );
		$in_headers = false;
		if (! empty ( $field ) && ! strstr ( $field, " " )) {
			$in_headers = true;
		}
		
		$max_line_length = 998;
		
		while ( list ( , $line ) = @each ( $lines ) ) {
			$lines_out = null;
			if ($line == "" && $in_headers) {
				$in_headers = false;
			}
			
			while ( strlen ( $line ) > $max_line_length ) {
				$pos = strrpos ( substr ( $line, 0, $max_line_length ), " " );
				
				if (! $pos) {
					$pos = $max_line_length - 1;
					$lines_out [] = substr ( $line, 0, $pos );
					$line = substr ( $line, $pos );
				} else {
					$lines_out [] = substr ( $line, 0, $pos );
					$line = substr ( $line, $pos + 1 );
				}
				
				if ($in_headers) {
					$line = "\t" . $line;
				}
			}
			$lines_out [] = $line;
			
			while ( list ( , $line_out ) = @each ( $lines_out ) ) {
				if (strlen ( $line_out ) > 0) {
					if (substr ( $line_out, 0, 1 ) == ".") {
						$line_out = "." . $line_out;
					}
				}
				fputs ( $this->smtp_conn, $line_out . $this->CRLF );
			}
		}
		
		fputs ( $this->smtp_conn, $this->CRLF . "." . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 250) {
			$this->error = array ("error" => "DATA not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		return true;
	}
	
	/**
	 * Sends the HELO command to the smtp server.
	 * This makes sure that we and the server are in
	 * the same known state.
	 *
	 * Implements from rfc 821: HELO <SP> <domain> <CRLF>
	 *
	 * SMTP CODE SUCCESS: 250
	 * SMTP CODE ERROR  : 500, 501, 504, 421
	 * @access public
	 * @return bool
	 */
	public function Hello($host = '') {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called Hello() without being connected" );
			return false;
		}
		
		if (empty ( $host )) {
			
			$host = "localhost";
		}
		
		if (! $this->SendHello ( "EHLO", $host )) {
			if (! $this->SendHello ( "HELO", $host )) {
				return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Sends a HELO/EHLO command.
	 * @access private
	 * @return bool
	 */
	private function SendHello($hello, $host) {
		fputs ( $this->smtp_conn, $hello . " " . $host . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER: " . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 250) {
			$this->error = array ("error" => $hello . " not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		$this->helo_rply = $rply;
		
		return true;
	}
	
	/**
	 * Starts a mail transaction from the email address specified in
	 * $from. Returns true if successful or false otherwise. If True
	 * the mail transaction is started and then one or more Recipient
	 * commands may be called followed by a Data command.
	 *
	 * Implements rfc 821: MAIL <SP> FROM:<reverse-path> <CRLF>
	 *
	 * SMTP CODE SUCCESS: 250
	 * SMTP CODE SUCCESS: 552,451,452
	 * SMTP CODE SUCCESS: 500,501,421
	 * @access public
	 * @return bool
	 */
	public function Mail($from) {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called Mail() without being connected" );
			return false;
		}
		
		$useVerp = ($this->do_verp ? "XVERP" : "");
		fputs ( $this->smtp_conn, "MAIL FROM:<" . $from . ">" . $useVerp . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 250) {
			$this->error = array ("error" => "MAIL not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		return true;
	}
	
	/**
	 * Sends the quit command to the server and then closes the socket
	 * if there is no error or the $close_on_error argument is true.
	 *
	 * Implements from rfc 821: QUIT <CRLF>
	 *
	 * SMTP CODE SUCCESS: 221
	 * SMTP CODE ERROR  : 500
	 * @access public
	 * @return bool
	 */
	public function Quit($close_on_error = true) {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called Quit() without being connected" );
			return false;
		}
		
		fputs ( $this->smtp_conn, "quit" . $this->CRLF );
		
		$byemsg = $this->get_lines ();
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $byemsg . $this->CRLF . '<br />';
		}
		
		$rval = true;
		$e = null;
		
		$code = substr ( $byemsg, 0, 3 );
		if ($code != 221) {
			
			$e = array ("error" => "SMTP server rejected quit command", "smtp_code" => $code, "smtp_rply" => substr ( $byemsg, 4 ) );
			$rval = false;
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $e ["error"] . ": " . $byemsg . $this->CRLF . '<br />';
			}
		}
		
		if (empty ( $e ) || $close_on_error) {
			$this->Close ();
		}
		
		return $rval;
	}
	
	/**
	 * Sends the command RCPT to the SMTP server with the TO: argument of $to.
	 * Returns true if the recipient was accepted false if it was rejected.
	 *
	 * Implements from rfc 821: RCPT <SP> TO:<forward-path> <CRLF>
	 *
	 * SMTP CODE SUCCESS: 250,251
	 * SMTP CODE FAILURE: 550,551,552,553,450,451,452
	 * SMTP CODE ERROR  : 500,501,503,421
	 * @access public
	 * @return bool
	 */
	public function Recipient($to) {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called Recipient() without being connected" );
			return false;
		}
		
		fputs ( $this->smtp_conn, "RCPT TO:<" . $to . ">" . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 250 && $code != 251) {
			$this->error = array ("error" => "RCPT not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		return true;
	}
	
	/**
	 * Sends the RSET command to abort and transaction that is
	 * currently in progress. Returns true if successful false
	 * otherwise.
	 *
	 * Implements rfc 821: RSET <CRLF>
	 *
	 * SMTP CODE SUCCESS: 250
	 * SMTP CODE ERROR  : 500,501,504,421
	 * @access public
	 * @return bool
	 */
	public function Reset() {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called Reset() without being connected" );
			return false;
		}
		
		fputs ( $this->smtp_conn, "RSET" . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 250) {
			$this->error = array ("error" => "RSET failed", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		
		return true;
	}
	
	/**
	 * Starts a mail transaction from the email address specified in
	 * $from. Returns true if successful or false otherwise. If True
	 * the mail transaction is started and then one or more Recipient
	 * commands may be called followed by a Data command. This command
	 * will send the message to the users terminal if they are logged
	 * in and send them an email.
	 *
	 * Implements rfc 821: SAML <SP> FROM:<reverse-path> <CRLF>
	 *
	 * SMTP CODE SUCCESS: 250
	 * SMTP CODE SUCCESS: 552,451,452
	 * SMTP CODE SUCCESS: 500,501,502,421
	 * @access public
	 * @return bool
	 */
	public function SendAndMail($from) {
		$this->error = null;
		
		if (! $this->connected ()) {
			$this->error = array ("error" => "Called SendAndMail() without being connected" );
			return false;
		}
		
		fputs ( $this->smtp_conn, "SAML FROM:" . $from . $this->CRLF );
		
		$rply = $this->get_lines ();
		$code = substr ( $rply, 0, 3 );
		
		if ($this->do_debug >= 2) {
			echo "SMTP -> FROM SERVER:" . $rply . $this->CRLF . '<br />';
		}
		
		if ($code != 250) {
			$this->error = array ("error" => "SAML not accepted from server", "smtp_code" => $code, "smtp_msg" => substr ( $rply, 4 ) );
			if ($this->do_debug >= 1) {
				echo "SMTP -> ERROR: " . $this->error ["error"] . ": " . $rply . $this->CRLF . '<br />';
			}
			return false;
		}
		return true;
	}
	
	/**
	 * This is an optional command for SMTP that this class does not
	 * support. This method is here to make the RFC821 Definition
	 * complete for this class and __may__ be implimented in the future
	 *
	 * Implements from rfc 821: TURN <CRLF>
	 *
	 * SMTP CODE SUCCESS: 250
	 * SMTP CODE FAILURE: 502
	 * SMTP CODE ERROR  : 500, 503
	 * @access public
	 * @return bool
	 */
	public function Turn() {
		$this->error = array ("error" => "This method, TURN, of the SMTP " . "is not implemented" );
		if ($this->do_debug >= 1) {
			echo "SMTP -> NOTICE: " . $this->error ["error"] . $this->CRLF . '<br />';
		}
		return false;
	}
	
	/**
	 * Get the current error
	 * @access public
	 * @return array
	 */
	public function getError() {
		return $this->error;
	}
	
	/////////////////////////////////////////////////
	// INTERNAL FUNCTIONS
	/////////////////////////////////////////////////
	

	/**
	 * Read in as many lines as possible
	 * either before eof or socket timeout occurs on the operation.
	 * With SMTP we can tell if we have more lines to read if the
	 * 4th character is '-' symbol. If it is a space then we don't
	 * need to read anything else.
	 * @access private
	 * @return string
	 */
	private function get_lines() {
		$data = "";
		while ( $str = @fgets ( $this->smtp_conn, 515 ) ) {
			if ($this->do_debug >= 4) {
				echo "SMTP -> get_lines(): \$data was \"$data\"" . $this->CRLF . '<br />';
				echo "SMTP -> get_lines(): \$str is \"$str\"" . $this->CRLF . '<br />';
			}
			$data .= $str;
			if ($this->do_debug >= 4) {
				echo "SMTP -> get_lines(): \$data is \"$data\"" . $this->CRLF . '<br />';
			}
			
			if (substr ( $str, 3, 1 ) == " ") {
				break;
			}
		}
		return $data;
	}

}

?>