<?php

/**
 * SOTESHOP/stBasket
 *
 * Ten plik należy do aplikacji stBasket opartej na licencji (Professional License SOTE).
 * Nie zmieniaj tego pliku, jeśli chcesz korzystać z automatycznych aktualizacji oprogramowania.
 * Jeśli chcesz wprowadzać swoje modyfikacje do programu, zapoznaj się z dokumentacją, jak zmieniać
 * oprogramowanie bez zmiany kodu bazowego http://www.sote.pl/modifications
 *
 * @package     stBasket
 * @subpackage  libs
 * @copyright   SOTE (www.sote.pl)
 * @license     http://www.sote.pl/license/sote (Professional License SOTE)
 * @version     $Id: stBasket.class.php 16445 2011-12-12 13:01:54Z marcin $
 */

/**
 * Klasa zarządzająca koszykami w sklepie
 *
 * @author Marcin Butlak <marcin.butlak@sote.pl>
 *
 * @package     stBasket
 * @subpackage  libs
 */
class stBasket
{
   /**
    * Przestrzeń nazw dla sesji
    * @var const
    */
   const SESSION_NAMESPACE = 'soteshop/stBasket';

   const ERR_OUT_OF_STOCK = -1;
   
   const ERR_MIN_QTY = -2;

   protected
   $basket = null,
   $user = null,
   $dispatcher = null,
   $productConfig = null,
   $couponCode = null,
   $basketConfig = null,
   $discountConfig = null;
   protected static $instance = null;

   /**
    * Zwraca instancje obiektu
    *
    * @return     stBasket    object
    */
   public static function getInstance(sfGuardSecurityUser $user)
   {
      if (!isset(self::$instance))
      {
         $class = __CLASS__;
         self::$instance = new $class();
         self::$instance->initialize($user);
      }

      return self::$instance;
   }

   /**
    * Zwraca obiekt do obsługi zdarzeń
    *
    * @return   stEventDispatcher
    */
   public function getDispatcher()
   {
      return $this->dispatcher;
   }

   /**
    * Incjalizuje koszyk
    *
    * @param   sfGuardSecurityUser $user
    */
   public function initialize(sfGuardSecurityUser $user)
   {
      $this->user = $user;

      $this->dispatcher = stEventDispatcher::getInstance();

      if ($this->user->isAnonymous() && !$this->user->hasAttribute('hash_code', self::SESSION_NAMESPACE))
      {
         $this->user->setAttribute('hash_code', md5(uniqid(session_id(), true)), self::SESSION_NAMESPACE);
      }

      if (!$this->user->hasAttribute('product_list', self::SESSION_NAMESPACE))
      {
         $this->user->setAttribute('product_list', array(), self::SESSION_NAMESPACE);
      }

      $this->basketConfig = stConfig::getInstance($user->getContext(), 'stBasket');

      $this->productConfig = stConfig::getInstance($user->getContext(), 'stProduct');

      $this->discountConfig = stConfig::getInstance($user->getContext(), 'stDiscountBackend');
   }

   /**
    * Zapisuje zawartosc koszyka
    */
   public function save()
   {
      if ($this->basket)
      {
         $this->basket->save();
         $this->setSessionItems();
      }
   }

   /**
    * Dodaje produkt do koszyka
    * Jesli dany produkt znajduje sie juz w koszyku zwieksza jego ilosc
    *
    * @param   integer     $id_product         ID produktu
    * @param   integer     $num                ilosc sztuk danego produktu - domyslnie 1
    */
   public function addItem($id, $quantity = 1, &$error = false)
   {
      $this->getDispatcher()->notify(new sfEvent($this, 'stBasket.preAddItem', array('id_product' => $id, 'num' => $quantity)));

      if ($item = $this->getItem($id))
      {
         $item->setQuantity($item->getQuantity() + $quantity);
      }
      else
      {
         if ($product = ProductPeer::retrieveByPK($id))
         {
            $item = $this->populateItem(new BasketProduct, $product);

            $item->setQuantity($quantity);

            $this->getDispatcher()->notify(new sfEvent($this, 'stBasket.modAddItem', array('product' => $product, 'item' => $item)));

            $this->get()->addBasketProduct($item, true);
         }
         else
         {
            return false;
         }
      }

      $this->getDispatcher()->notify(new sfEvent($this, 'stBasket.postAddItem', array('item' => $item)));

      $error = $this->validateQuantity($item);

      return $item;
   }

   public function updateItem($id, $quantity, &$error = false)
   {
      if ($item = $this->getItem($id))
      {
         $item->setQuantity($quantity);
      }
      else
      {
         return false;
      }

      $error = $this->validateQuantity($item);

      return $item;
   }

   /**
    * Usuwa produkt o danym ID z koszyka
    *
    * @param   integer     $id_product         ID produktu
    */
   public function removeItem($id)
   {
      $item = $this->getItem($id);

      if(is_object($item)) $item->delete();
   }

   /**
    * Zwraca wszystkie produkty z koszyka w postaci tablicy modelów BasketProduct
    *
    * @return   array
    */
   public function getItems()
   {
      $c = new Criteria();

      $c->addAscendingOrderByColumn(BasketProductPeer::ID);

      return $this->get()->getBasketProductsJoinProduct($c);
   }

   public function refresh()
   {
      $coupon_code = $this->getCouponCode();

      if ($coupon_code && !$coupon_code->isValid())
      {
         $this->setCouponCode(null);
      }

      foreach ($this->getItems() as $item)
      {
         if (!$item->getProduct())
            continue;
         
         $image = null;

         $this->populateItem($item, $item->getProduct());

         $ids = array();

         foreach ($item->getPriceModifiers() as $pm)
         {
            $ids[$pm['custom']['id']] = $pm['custom']['id'];
         }

         if ($ids)
         {
            $options = ProductOptionsValuePeer::retrieveByPKs($ids);

            if (count($options) != count($ids))
            {
               $item->setProduct(null);

               continue;
            }

            $price_type = ProductOptionsValuePeer::getPriceType($item->getProduct());

            $item->setPriceModifiers(array());

            $stock = $item->getProduct()->getStock();

            foreach ($options as $option)
            {
               stNewProductOptions::addProductPriceModifier($item, $option->getPrice(), $option->getDepth(), $price_type, array('id' => $option->getId(), 'label' => $option->getValue(), 'type' => 'product_options'));

               if ($option->getsfAssetId())
               {
                  $image = $option->getsfAssetId();
               }

               if (null !== $option->getStock() && $stock > $option->getStock())
               {
                  $stock = $option->getStock();
               }
            }

            if ($image && ($sf_asset = sfAssetPeer::retrieveByPK($image)))
            {
               $item->setImage($sf_asset->getRelativePath());
            }

            if (null !== $stock)
            {
               $item->setMaxQuantity($stock);
            }
         }
      }
   }

   /**
    * Zwraca produkt o danych ID
    *
    * @param   integer     $id_product         ID produktu
    * @return   object
    */
   public function getItem($id)
   {
      $id = stBasket::generateItemId($id);

      foreach ($this->getItems() as $item)
      {
         if ($item->getItemId() == $id)
         {
            return $item;
         }
      }

      return null;
   }

   /**
    * Zwraca ID klienta
    *
    * @return   integer
    */
   public function getUserId()
   {
      if ($this->user->isAuthenticated())
      {
         return $this->user->getGuardUser()->getId();
      }

      return false;
   }

   public function getCouponCode()
   {
      if (!$this->discountConfig->get('coupon_code'))
      {
         return null;
      }

      if (null === $this->couponCode)
      {
         $this->couponCode = $this->get()->getDiscountCouponCode();
      }

      return $this->couponCode;
   }

   public function setCouponCode($v)
   {
      $this->get()->setDiscountCouponCode($v);

      $this->couponCode = $v;
   }

   public function getUser()
   {
      return $this->user;
   }

   /**
    * Pobierz laczna sume produktow w koszyku
    *
    * @param   bool        $with_tax           Uwzględnij podatek
    * @param   bool        $with_currency      Uwzględnij walute
    * @return   suma
    */
   public function getTotalAmount($with_tax = false, $with_currency = false)
   {
      $total_amount = 0;

      $basket_products = $this->getItems();

      foreach ($basket_products as $basket_product)
      {
         $total_amount += self::getProductTotalAmount($basket_product, $with_tax, $with_currency);
      }

      return $total_amount;
   }

   /**
    * Pobierz laczna liczbę produktow w koszyku
    *
    * @return   integer
    */
   public function getTotalProductQuantity()
   {
      $items = $this->getItems();

      $quantity = 0;

      foreach ($items as $item)
      {
         $quantity += $item->getQuantity();
      }

      return $quantity;
   }

   /**
    *
    * Pobierz lączną wagę produktów w koszyku
    *
    * @return float
    */
   public function getTotalProductWeight()
   {
      $items = $this->getItems();

      $weight = 0;

      foreach ($items as $item)
      {
         $weight += $item->getWeight() * $item->getQuantity();
      }

      return $weight;
   }

   /**
    * Zmniejsza ilość produktu w koszyku.
    * Jeżeli ilość produktu wynosi jeden, usuwa produkt z koszyka
    *
    * @param   integer     $id_product         ID produktu
    * @param   integer     $num                ilosc odejmowanych sztuk danego produktu - domyslnie 1
    */
   public function decrease($id_product, $num = 1)
   {
      if ($item = $this->getItem($id_product))
      {
         $quantity = $item->getQuantity() - $num;

         $item->setQuantity($quantity);

         return $item->getQuantity();
      }

      return false;
   }

   /**
    * Deletes the current shopping cart
    */
   public function clear()
   {
      $this->deleteBasket($this->get());

      $this->clearSessionItems();
   }

   /**
    * Clears all items in shopping cart
    */
   public function clearItems()
   {
      foreach ($this->getItems() as $item)
      {
         $item->delete();
      }

      $this->clearSessionItems();
   }

   /**
    * Zwraca aktualny koszyk uzytkownika
    *
    * @param   bool        $create_default     Określa czy ma tworzyć domyślny koszyk, jeżeli żaden nie był wcześniej koszykiem domyślnym
    * @return   Basket
    */
   public function get($create_default = true)
   {
      if (null === $this->basket)
      {
         $hash_code = $this->user->getAttribute('hash_code', null, self::SESSION_NAMESPACE);

         $user = $this->getUserId();

         $authenticated = $this->user->isAuthenticated();

         if ($hash_code || $authenticated)
         {
            if ($hash_code)
            {
               $c = new Criteria();

               $c->add(BasketPeer::IS_DEFAULT, true);

               $c->add(BasketPeer::HASH_CODE, $hash_code);

               $this->basket = BasketPeer::doSelectOne($c);
            }

            if (null === $this->basket && $authenticated)
            {
               $c = new Criteria();

               $c->add(BasketPeer::IS_DEFAULT, true);

               $c->add(BasketPeer::SF_GUARD_USER_ID, $user);

               $this->basket = BasketPeer::doSelectOne($c);
            }

            if ($this->basket && $this->basket->getHashCode() !== null && $authenticated)
            {
               if ($this->basket->countBasketProducts() > 0)
               {
                  $c = new Criteria();

                  $c->add(BasketPeer::SF_GUARD_USER_ID, $user);

                  $baskets = BasketPeer::doSelect($c);

                  foreach ($baskets as $basket)
                  {
                     $basket->delete();
                  }
               }

               $this->basket->setSfGuardUserId($user);

               $this->basket->setHashCode(null);

               $this->user->setAttribute('hash_code', null, self::SESSION_NAMESPACE);

               $this->basket->save();
            }
         }

         if (null === $this->basket && $create_default)
         {
            $this->basket = new Basket();

            if ($authenticated)
            {
               $this->basket->setSfGuardUserId($user);
            }
            else
            {
               $this->basket->setHashCode($hash_code);
            }

            $this->basket->setIsDefault(true);
         }
      }

      return $this->basket;
   }

   /**
    * Ustawia domyślny koszyk
    *
    * @param   mixed       $basket             Id koszyka lub instancja modelu Basket
    */
   public function set($basket)
   {
      if (!is_object($basket))
      {
         $basket = BasketPeer::retrieveByPK($basket);
      }

      if ($basket != null)
      {
         if ($prev_basket = $this->get(false))
         {
            $prev_basket->setIsDefault(false);

            $prev_basket->save();
         }
         $basket->setIsDefault(true);

         $basket->setSfGuardUserId($this->getUserId());

         $this->basket = $basket;

         $this->save();
      }
   }

   /**
    * Dodaje nowy koszyk
    *
    * @param   bool        $make_default       Określa czy dodany koszyk ma być ustawiony jako domyślny
    */
   public function addBasket($make_default = false)
   {
      $new_basket = new Basket();
      $new_basket->setSfGuardUserId($this->getUserId());

      if ($make_default)
      {
         $this->set($new_basket);
      }
      else
      {
         $new_basket->save();
      }
   }

   /**
    * Usuwa koszyk
    *
    * @param   mixed       $basket             Id koszyka lub instancja modelu Basket
    */
   public function deleteBasket($basket)
   {
      if (!is_object($basket))
      {
         $basket = BasketPeer::retrieveByPK($basket);
      }

      if ($this->get()->getId() == $basket->getId())
      {
         $this->basket = null;
      }

      $basket->delete();
   }

   /**
    * Koszyk posiada produkty
    *
    * @return   bool
    */
   public function hasItems()
   {
      return (bool) $this->getItems();
   }

   /**
    * Koszyk jest pusty
    *
    * @return   bool
    */
   public function isEmpty()
   {
      return!$this->hasItems();
   }

   /**
    *
    * @param Product $product
    * @return <type>
    */
   protected function populateItem(BasketProduct $item, Product $product)
   {
      $item->setName($product->getName());

      $item->setCode($product->getCode());

      $item->setBasePriceNetto($product->getBasePriceNetto());

      $item->setBasePriceBrutto($product->getBasePriceBrutto());

      $item->setVat($product->getVat());

      $item->setWeight($product->getWeight());

      $coupon_code = $this->getCouponCode();

      if ($coupon_code)
      {
         $item->setDiscount(array('percent' => min($coupon_code->getDiscount(), $product->getMaxDiscount())));
      }
      else
      {
         $item->setDiscount($product->getDiscount());
      }

      $item->setWholesale($product->getWholesale());

      $item->setCurrency(array('price' => $product->getCurrencyPrice(), 'exchange' => $product->getCurrencyExchange(), 'code' => $product->getCurrency()->getShortcut(), 'id' => $product->getCurrencyId()));

      $item->setMaxQuantity($product->getStock());

      $item->setImage($product->getOptImage());

      if ($item->isNew())
      {
         $item->setProductId($product->getId());

         $item->setItemId(stBasket::generateItemId($product->getId()));
      }

      return $item;
   }

   /**
    *
    * @param BasketProduct $item
    * @return <type>
    */
   protected function validateQuantity($item)
   {   
      $max_quantity = $item->getMaxQuantity();

      if ($item->getQuantity() > $max_quantity || $item->getQuantity() == 0)
      {         
         $item->setQuantity($item->getMaxQuantity());

         return self::ERR_OUT_OF_STOCK;
      }
      
      if ($item->getProductMinQty() > $item->getQuantity())
      {
         $item->setQuantity($item->getProductMinQty());
         
         return self::ERR_MIN_QTY;
      }

      return null;
   }

   public static function generateItemId($id)
   {
      if (!is_numeric($id))
      {
         return $id;
      }

      $event = stEventDispatcher::getInstance()->filter(new sfEvent(null, 'stBasket.generateItemId'), $id);

      return $event->getReturnValue();
   }

   /**
    * Zwraca łączną cenę produktu (ilosc x cena)
    *
    * @param   BasketProduct $basket_product   Produkt w koszyku
    * @param   bool        $with_tax           Uwzględnij podatek
    * @param   bool        $with_currency      Uwzględnij walute
    * @return   float
    */
   public static function getProductTotalAmount($basket_product, $with_tax = false, $with_currency = false)
   {
      return $basket_product->getTotalAmount($with_tax, $with_currency);
   }

   /**
    * Weryfikuje czy podany koszyk nalezy do danego uzytkownika
    *
    * @param   mixed       $basket             Id koszyka lub instancja modelu Basket
    * @param   sfGuardSecurityUser $user
    */
   public static function validateBasket($basket, $user)
   {
      if (!is_object($basket))
      {
         $basket = BasketPeer::retrieveByPK($basket);
      }
		
      if ($guard_user = $user->getGuardUser())
      {
         $guard_user_id = $guard_user->getId();
      }
      else
      {
         $guard_user_id = 0;
      }
		
      return isset($basket) && $basket->getUserId() == $guard_user_id;
   }

   public function setSessionItems()
   {
      $sessionItems = array();
      foreach ($this->getItems() as $k => $item)
         if (!$item->isDeleted())
            $sessionItems[] = $item;

      $this->user->setAttribute('product_list', $sessionItems, self::SESSION_NAMESPACE);
   }

   public function clearSessionItems()
   {
      return $this->user->setAttribute('product_list', array(), self::SESSION_NAMESPACE);
   }

   public function getSessionItems()
   {
      $productListSaved = $this->user->getAttribute('product_list_saved', false, self::SESSION_NAMESPACE);

      if ($productListSaved == false && null === $this->basket && $this->user->isAuthenticated())
      {
         $this->setSessionItems();
         $this->user->setAttribute('product_list_saved', true, self::SESSION_NAMESPACE);
      }

      if ($productListSaved == true && null === $this->basket && !$this->user->isAuthenticated())
      {
         $this->clearSessionItems();
         $this->user->setAttribute('product_list_saved', false, self::SESSION_NAMESPACE);
      }
      return $this->user->getAttribute('product_list', array(), self::SESSION_NAMESPACE);
   }

   public function getSessionTotalAmount($with_tax = false, $with_currency = false)
   {
      $total_amount = 0;

      $basket_products = $this->getSessionItems();

      foreach ($basket_products as $basket_product)
      {
         $total_amount += self::getProductTotalAmount($basket_product, $with_tax, $with_currency);
      }

      return $total_amount;
   }

}