<?php

/**
 * SOTESHOP/stBasket
 *
 * Ten plik należy do aplikacji stBasket opartej na licencji (Professional License SOTE).
 * Nie zmieniaj tego pliku, jeśli chcesz korzystać z automatycznych aktualizacji oprogramowania.
 * Jeśli chcesz wprowadzać swoje modyfikacje do programu, zapoznaj się z dokumentacją, jak zmieniać
 * oprogramowanie bez zmiany kodu bazowego http://www.sote.pl/modifications
 *
 * @package     stBasket
 * @subpackage  actions
 * @copyright   SOTE (www.sote.pl)
 * @license     http://www.sote.pl/license/sote (Professional License SOTE)
 * @version     $Id: actions.class.php 16445 2011-12-12 13:01:54Z marcin $
 */

/**
 * Akcje dla koszyka
 *
 * @author Marcin Butlak <marcin.butlak@sote.pl>
 *
 * @package     stBasket
 * @subpackage  actions
 */
class stBasketActions extends stActions
{

   /**
    * Zwraca obiekt obsługi zdarzeń
    *
    * @return   stEventDispatcher
    */
   public function getDispatcher()
   {
      return stEventDispatcher::getInstance();
   }

   /**
    * Zapamietuje poprzednią stronę i przekazuje żądanie do stBasket/index
    */
   public function executeIndexReferer()
   {
      $this->remmemberRerefer();

      return $this->redirect('@stBasket');
   }

   /**
    * Zapamiętuje poprzednią stronę i przekazuje żądanie do stBasket/add
    */
   public function executeAddReferer()
   {
      $this->remmemberRerefer();

      return $this->redirect('@stBasket');
   }

   /**
    * Indeks koszyka
    */
   public function executeIndex()
   {
      $this->getUser()->setAttribute('is_blocked', false, stBasket::SESSION_NAMESPACE);

      $this->smarty = new stSmarty($this->getModuleName());

      $this->basket = stBasket::getInstance($this->getUser());

      $this->delivery = stDeliveryFrontend::getInstance($this->basket);

      $this->basket_config = stConfig::getInstance($this->getContext());

      $this->referer = $this->getReferer();

      $this->checkMinOrderAmount();
   }

   public function executeSetProduct()
   {
      return $this->redirect('@stBasket');
   }

   /**
    * Dodawanie koszyka
    */
   public function executeAddBasket()
   {
      $basket = stBasket::getInstance($this->getUser());

      $basket->addBasket(true);

      return $this->redirect('@stBasket');
   }

   /**
    * Ustawia domyślny koszyk
    */
   public function executeSetBasket()
   {
      $basket = stBasket::getInstance($this->getUser());

      $this->forward404Unless(stBasket::validateBasket($this->getRequestParameter('id'), $this->getUser()));

      $basket->set($this->getRequestParameter('id'));

      return $this->redirect('@stBasket');
   }

   /**
    * Usuwa koszyk
    */
   public function executeDeleteBasket()
   {
      $basket = stBasket::getInstance($this->getUser());

      $this->forward404Unless(stBasket::validateBasket($this->getRequestParameter('id'), $this->getUser()));

      $basket->deleteBasket($this->getRequestParameter('id'));

      return $this->redirect('@stBasket');
   }

   /**
    * Zdejmuje produkt z koszyka
    */
   public function executeRemove()
   {
      $basket = stBasket::getInstance($this->getUser());

      $basket->removeItem($this->getRequestParameter('product_id'));

      $basket->save();

      return $this->redirect('@stBasket');
   }

   /**
    * Weryfikuje poprawność wprowadzanych danych dla akcji SetProduct
    *
    * @return   bool
    */
   public function validateSetProduct()
   {
      $error_exists = false;

      if ($this->hasRequestParameter('product_id'))
      {
         $id = $this->getRequestParameter('product_id');
         
         $basket = stBasket::getInstance($this->getUser());
         
         $item = $basket->getItem($id);

         $quantity = $this->getRequestParameter('quantity', 1);
         
         $product = ProductPeer::retrieveByPK($id);

         if (!$this->validateQuantity($quantity, $error, $item->getProductStockInDecimals()))
         {
            $this->getRequest()->setError('basket{products}{'.$id.'}', $error);

            $this->getRequest()->setParameter('basket[products]['.$id.']', $quantity);

            $error_exists = true;
         }
         else
         {
            $item = $basket->updateItem($id, $quantity, $error);

            if ($item->getQuantity() > 0)
            {
               $basket->save();
            }

            if ($error == stBasket::ERR_OUT_OF_STOCK)
            {
               $this->getRequest()->setParameter('basket[products]['.$item->getItemId().']', $item->getQuantity());

               $this->getRequest()->setError('basket{products}{'.$item->getItemId().'}', $this->getContext()->getI18N()->__('Brak wymaganej ilości towaru w magazynie'));

               $error_exists = true;
            }
            elseif ($error == stBasket::ERR_MIN_QTY)
            {
               sfLoader::loadHelpers(array('Helper', 'stProduct'), 'stProduct');

               $min = $item->getProductMinQty();

               $uom = st_product_uom($item->getProduct());

               $this->getRequest()->setParameter('basket[products]['.$item->getItemId().']', $min);

               $this->getRequest()->setError('basket{products}{'.$item->getItemId().'}', $this->getContext()->getI18N()->__('Minimalna ilość to %quantity% %uom%', array('%quantity%' => $min, '%uom%' => $uom)));

               $error_exists = true;
            }
         }
      }

      $this->getUser()->setAttribute('is_blocked', $error_exists, stBasket::SESSION_NAMESPACE);

      return false == $error_exists;
   }

   /**
    * Weryfikuje poprawność wprowadzanych danych dla akcji addReferer
    *
    * @return   bool
    */
   public function validateAddReferer()
   {
      $error_exists = false;

      if ($this->getRequest()->hasParameter('product_id') && ($this->getRequest()->getMethod() == sfRequest::GET || $this->getRequest()->getMethod() == sfRequest::POST))
      {

         $product_id = $this->getRequestParameter('product_id');

         $quantity = $this->getRequestParameter('quantity', 1);
         
         $product = ProductPeer::retrieveByPK($product_id);

         $basket = stBasket::getInstance($this->getUser());

         if (!$this->validateQuantity($quantity, $error, $product ? $product->getStockInDecimals() : false))
         {
            $item = $basket->addItem($product_id, 0);

            $item->setItemId($product_id);

            $this->getRequest()->setError('basket{products}{'.$product_id.'}', $error);

            $this->getRequest()->setParameter('basket[products]['.$product_id.']', $quantity);

            $error_exists = true;
         }
         else
         {
            $item = $basket->addItem($product_id, $quantity, $error);

            if ($error == stBasket::ERR_OUT_OF_STOCK)
            {
               $this->getRequest()->setParameter('basket[products]['.$item->getItemId().']', $item->getQuantity());

               $this->getRequest()->setError('basket{products}{'.$item->getItemId().'}', $this->getContext()->getI18N()->__('Brak wymaganej ilości towaru w magazynie'));

               $error_exists = true;
            }
            elseif ($error == stBasket::ERR_MIN_QTY)
            {
               sfLoader::loadHelpers(array('Helper', 'stProduct'), 'stProduct');

               $min = $item->getProductMinQty();

               $uom = st_product_uom($item->getProduct());

               $this->getRequest()->setParameter('basket[products]['.$item->getItemId().']', $min);

               $this->getRequest()->setError('basket{products}{'.$item->getItemId().'}', $this->getContext()->getI18N()->__('Minimalna ilość to %quantity% %uom%', array('%quantity%' => $min, '%uom%' => $uom)));

               $error_exists = true;
            }
            else
            {
               $basket->save();
            }
         }
      }

      $this->getUser()->setAttribute('is_blocked', $error_exists, stBasket::SESSION_NAMESPACE);

      return false == $error_exists;
   }

   /**
    * Weryfikuje poprawność wprowadzanych danych dla akcji index
    *
    * @return   bool
    */
   public function validateIndex()
   {
      $error_exists = false;

      $basket = stBasket::getInstance($this->getUser());

      $request = $this->getRequest();

      $i18n = $this->getContext()->getI18N();

      if ($request->getMethod() == sfRequest::POST && !$request->getParameter('submit_save'))
      {
         $products = $this->getRequestParameter('basket[products]', array());
         
         $error_exists = false;

         foreach ($products as $id => $value)
         {
            $item = $basket->getItem($id);

            if ($item && !$item->productValidate())
            {
               $request->setParameter('basket[products]['.$id.']', $value);

               $request->setError('basket{products}{'.$id.'}', $i18n->__('Product wycofany z oferty'));

               $error_exists = true;
            }
            elseif (!$this->validateQuantity($value, $error, $item->getProductStockInDecimals()))
            {
               $error_exists = true;

               $basket->addItem($id, 0);

               $request->setError('basket{products}{'.$id.'}', $error);

               $request->setParameter('basket[products]['.$id.']', $value);
            }
            else
            {
               $item = $basket->updateItem($id, $value, $error);

               if (!$item)
               {
                  $item = $basket->addItem($id, $value, $error);
               }

               if ($error == stBasket::ERR_OUT_OF_STOCK)
               {
                  $request->setParameter('basket[products]['.$id.']', $item->getQuantity());

                  $request->setError('basket{products}{'.$id.'}', $i18n->__('Brak wymaganej ilości towaru w magazynie'));

                  $error_exists = true;
               }
               elseif ($error == stBasket::ERR_MIN_QTY)
               {
                  sfLoader::loadHelpers(array('Helper', 'stProduct'), 'stProduct');

                  $min = $item->getProductMinQty();

                  $uom = st_product_uom($item->getProduct());

                  $this->getRequest()->setParameter('basket[products]['.$item->getItemId().']', $min);

                  $this->getRequest()->setError('basket{products}{'.$item->getItemId().'}', $this->getContext()->getI18N()->__('Minimalna ilość to %quantity% %uom%', array('%quantity%' => $min, '%uom%' => $uom)));

                  $error_exists = true;
               }
               else
               {
                  $basket->save();
               }
            }
         }
      }
      else
      {
         foreach ($basket->getItems() as $item)
         {      
            if (!$item->productValidate())
            {
               $request->setParameter('basket[products]['.$item->getItemId().']', $item->getQuantity());

               $request->setError('basket{products}{'.$item->getItemId().'}', $i18n->__('Produkt wycofany z oferty'));

               $error_exists = true;
            }
            elseif ($item->getQuantity() > $item->getMaxQuantity())
            {
               $request->setParameter('basket[products]['.$item->getItemId().']', $item->getMaxQuantity());

               $request->setError('basket{products}{'.$item->getItemId().'}', $i18n->__('Brak wymaganej ilości towaru w magazynie'));

               $error_exists = true;
            }
         }
      }

      $this->getUser()->setAttribute('is_blocked', $error_exists, stBasket::SESSION_NAMESPACE);

      return false == $error_exists;
   }

   /**
    * Czysci koszyk
    */
   public function executeClear()
   {
      stBasket::getInstance($this->getUser())->clearItems();

      return $this->redirect('@stBasket');
   }

   public function handleErrorSetProduct()
   {
      $this->smarty = new stSmarty($this->getModuleName());

      $this->basket = stBasket::getInstance($this->getUser());

      $this->delivery = stDeliveryFrontend::getInstance($this->basket);

      $this->referer = $this->getReferer();

      $this->basket_config = stConfig::getInstance($this->getContext(), 'stBasket');

      $this->setTemplate('index');

      return sfView::SUCCESS;
   }

   /**
    * Obsługa błędu w akcji index
    *
    * @return   sfView
    */
   public function handleErrorIndex()
   {
      $this->smarty = new stSmarty($this->getModuleName());

      $this->basket = stBasket::getInstance($this->getUser());

      $this->delivery = stDeliveryFrontend::getInstance($this->basket);

      $this->referer = $this->getReferer();

      $this->basket_config = stConfig::getInstance($this->getContext(), 'stBasket');

      $this->checkMinOrderAmount();

      return sfView::SUCCESS;
   }

   /**
    * Obsługa błędu w akcji addReferer
    */
   public function handleErrorAddReferer()
   {
      $this->remmemberRerefer();

      $this->smarty = new stSmarty($this->getModuleName());

      $this->basket = stBasket::getInstance($this->getUser());

      $this->delivery = stDeliveryFrontend::getInstance($this->basket);

      $this->referer = $this->getReferer();

      $this->basket_config = stConfig::getInstance($this->getContext(), 'stBasket');

      $this->setTemplate('index');

      $this->checkMinOrderAmount();

      return sfView::SUCCESS;
   }

   /**
    * Zapamiętuje poprzednia strone
    */
   protected function remmemberRerefer()
   {
      $referer = $this->getRequest()->getReferer();

      // chcemy zapamiętywać wyłącznie strony z produktami lub wywołano akcje 'addReferer'
      if (strpos($referer, '/product/') !== false || $this->getActionName() == 'addReferer' || $referer == '/')
      {
         $this->getUser()->setAttribute('referer', $referer, stBasket::SESSION_NAMESPACE);
      }
   }

   /**
    * Zwraca poprzednia stronevalidateQuantity
    *
    * @return   string
    */
   protected function getReferer()
   {
      $referer = $this->getUser()->getAttribute('referer', '/', stBasket::SESSION_NAMESPACE);

      return $referer;
   }

   protected function validateQuantity($quantity, &$error, $allow_decimals = false)
   {
      $validator = new sfNumberValidator();

      $i18n = $this->getContext()->getI18n();

      $nan_error = $allow_decimals ? $i18n->__('Podana wartość musi być liczbą...') : $i18n->__('Podana wartość musi być liczbą całkowitą...');
      
      $validator->initialize($this->getContext(), array(
          'nan_error' => $nan_error, 
          'type_error' => $nan_error, 
          'min' => $allow_decimals ? 0.01 : 1, 
          'min_error' => $i18n->__('Podana wartość musi byc wieksza od 0'), 
          'type' => $allow_decimals ? 'any' : 'integer'));

      return $validator->execute($quantity, $error);
   }

   protected function checkMinOrderAmount()
   {
      $config = stConfig::getInstance($this->getContext(), 'stOrder');

      $min_amount = $config->get('min_amount', 0);

      if ($min_amount > 0 && !$this->basket->isEmpty() && $min_amount > $this->basket->getTotalAmount(true))
      {

         sfLoader::loadHelpers(array('Helper', 'stCurrency'));

         $this->getUser()->setAttribute('is_blocked', true, stBasket::SESSION_NAMESPACE);

         $this->setFlash('warning', $this->getContext()->getI18N()->__('Minimalna wartość zamówienia wynosi %min_amount%', array('%min_amount%' => st_price($min_amount, true, true)), 'stOrder'));
      }
      elseif (!$this->getRequest()->hasErrors())
      {
         $this->setFlash('warning', null);

         $this->getUser()->setAttribute('is_blocked', false, stBasket::SESSION_NAMESPACE);
      }
   }

}

