<?php
/**
 * SOTESHOP/stUpdate
 *
 * This file is the part of stUpdate application. License: (Open License SOTE) Open License SOTE.
 * Do not modify this file, system will overwrite it during upgrade.
 *
 * @package     stUpdate
 * @subpackage  libs
 * @copyright   SOTE (www.sote.pl)
 * @license     http://www.sote.pl/license/open (Open License SOTE) Open License SOTE
 * @version     $Id: stHtAccess.class.php 16204 2011-11-23 14:13:48Z michal $
 * @author      Marek Jakubowicz <marek.jakubowicz@sote.pl>
 */

/**
 * Htaccess updater.
 * Parse htaccess file. Types: command, define.
 * command - analyze single lines without tags
 * define  - analyze tags
 * Example:
 *    $htaccess1 = new stHtaccess(sfConfig::get('sf_web_dir').DIRECTORY_SEPARATOR.'.htaccess1'); // old version
 *    $htaccess2 = new stHtaccess(sfConfig::get('sf_web_dir').DIRECTORY_SEPARATOR.'.htaccess2'); // new version
 *    $htaccess3 = sfConfig::get('sf_web_dir').DIRECTORY_SEPARATOR.'.htaccess3';                 // merged version
 *    $merge = new stHtaccessMerge($htaccess1,$htaccess2);
 *    $merge->merge();
 *    $merge->save($htaccess3);
 *
 * @package     stUpdate
 * @subpackage  libs
 */
class stHtaccess
{
	var $htaccess_file='';

	public function __construct($file)
	{
		if (file_exists($file))
		{
			$this->htaccess_file=file_get_contents($file);
		}
	}

	public function generate()
	{
		$this->parse();
		return $this->htaccess_array;
	}

	private function parse()
	{
		$result = array();

		$lines = explode("\n", $this->htaccess_file);
		$open = false;

		foreach ($lines as $line) {

			if (preg_match("/^<([a-z]+) ([a-z_0-9\.]+)>/i", $line, $matches)) {
				$open = true;
				$command = $matches[1];
				$module = $matches[2];
				$content = array();
				continue;
			}

			if (!empty($command) && preg_match('/^<\/'.$command.'>/', $line)) {
				$open = false;
				$result[] = array('type'=> 'define' , 'value' => array('command' => $command, 'module' => $module, 'content' => $content));
				continue;
			}

			if ($open == true) {
				$content[] = $line;
			} else {
				$cmd_data = explode(' ', $line, 2);

				if (!empty($cmd_data[1])) $content_cmd = $cmd_data[1];
				else $content_cmd = '';

				if (!empty($cmd_data[0])) $result[] = array('type' => 'command', 'value' => array('command' => $cmd_data[0], 'content' => $content_cmd));
			}
		}

		$this->htaccess_array = $result;
	}

	/**
	 * Get all options 'command' type.
	 *
	 * @return   array
	 */
	public function getAllCommands()
	{
		$result=array();
		foreach ($this->htaccess_array as $id=>$data)
		{

			if ($data['type']=='command')
			{
				$result[]=$data['value']['command'];
			}
		}
		return $result;
	}

	/**
	 * Get all opctions 'define' type.
	 *
	 * @return   array
	 */
	public function getAllDefines()
	{
		$result=array();
		foreach ($this->htaccess_array as $id=>$data)
		{
			if ($data['type']=='define')
			{
				$result[]=$data['value']['command'].' '.$data['value']['module'];
			}
		}
		return $result;
	}

	/**
	 * Get data for provided 'command' type option.
	 *
	 * @param   string      $command            np. Options
	 * @return   array
	 */
	public function getCommand($command)
	{
		foreach ($this->htaccess_array as $id=>$data)
		{

			if ($data['type']=='command')
			{
				if ($data['value']['command']==$command)
				return $data['value']['content'];
			}
		}

		return false;
	}

	/**
	 * Get data for provided 'define' type option.
	 *
	 * @param   string      $command            np. IfModule
	 * @param   string      $module             np. mod_rewrite.c
	 * @return   array
	 */
	public function getDefine($command,$module)
	{
		foreach ($this->htaccess_array as $id=>$data)
		{

			if ($data['type']=='define')
			{
				if (($data['value']['command']==$command) && ($data['value']['module']==$module))
				return $data['value']['content'];
			}
		}

		return false;
	}

	/**
	 * Get information if provided 'command' option exists.
	 *
	 * @param        string      $command
	 * @return   bool
	 */
	public function isCommand($command)
	{
		if (! $this->getCommand($command)) return false;
		return true;
	}

	/**
	 * Get information if provided 'define' option exists.
	 *
	 * @param        string      $command
	 */
	public function isDefine($command,$module)
	{
		if (! $this->getDefine($command,$module)) return false;
		return true;

	}
}

/**
 * Htaccess updater/rules.
 *
 * @package     stUpdate
 * @subpackage  libs
 */
class stHtaccessMerge
{
	/**
	 * @var stHtaccess Object of old htaccess file.
	 */
	var $htaccess1;

	/**
	 * @var stHtaccess Object of new htaccess file.
	 */
	var $htaccess2;

	/**
	 * @var array Data of old htaccess file.
	 *
	 * @see $this->htaccess1
	 */
	var $htaccess1_gen=array();

	/**
	 * @var array Data of new htaccess file.
	 *
	 * @see $this->htaccess2
	 */
	var $htaccess2_gen=array();

	/**
	 * @var string Data of merged htaccess files.
	 */
	var $htaccess_merged='';

	/**
	 * Construct.
	 *
	 * @param    stHtaccess  $htaccess1
	 * @param    stHtaccess  $htaccess2
	 */
	public function __construct(&$htaccess1, &$htaccess2)
	{
		$this->htaccess1=&$htaccess1;
		$this->htaccess2=&$htaccess2;
		$this->htaccess1_gen=$htaccess1->generate();
		$this->htaccess2_gen=$htaccess2->generate();

	}

	/**
	 * Merge htaccess files.
	 */
	public function merge()
	{
		$htaccess_merged=array();

		// Analyze htaccess1 file. If htaccess2 contains elements from htaccess1 which have to be replaced, replace them.
		foreach ($this->htaccess1_gen as $id=>$data)
		{
			switch ($data['type'])
			{
				case 'command':
					$this->command12($data);
					break;

				case 'define':
					$this->define12($data);
					break;
			}
		}

		// If  htaccess2 contains new definitions, add them
		foreach ($this->htaccess2_gen as $id=>$data)
		{
			switch ($data['type'])
			{
				case 'define':
					$this->define21($data);
					break;
			}
		}
	}

	/**
	 * Remember data for type 'command'. Data from htaccess1.
	 *
	 * @param   array       $data               Dane z htaccess1
	 */
	private function command12($data)
	{
		$command=$data['value']['command'];
		$content=$data['value']['content'];
		$this->htaccess_merged.=$command.' '.$content."\n";
	}

	/**
	 * Remember data for type 'define'. Analyze htaccess1 and htaccess2 file.
	 *
	 * @param   array       $data               Dane z htaccess1
	 */
	private function define12($data)
	{
		$command=$data['value']['command'];
		$module=$data['value']['module'];
		if ($this->htaccess2->isDefine($command,$module))
		{
			// kopiuj 'command' z nowego pliku
			$content=$this->htaccess2->getDefine($command,$module);
		} else {
			// kopiuj 'command' z obecnego pliku
			$content=$this->htaccess1->getDefine($command,$module);
		}
		$this->htaccess_merged.='<'.$command.' '.$module.">\n";
		foreach ($content as $c)
		{
			$this->htaccess_merged.=$c."\n";
		}
		$this->htaccess_merged.='</'.$command.">\n\n";
	}

	/**
	 * Remember data for new 'define' type options
	 *
	 * @param   array       $data               Data from htaccess2.
	 */
	public function define21($data)
	{
		$command=$data['value']['command'];
		$module=$data['value']['module'];
		if (! $this->htaccess1->isDefine($command,$module))
		{
			$content=$this->htaccess2->getDefine($command,$module);
			$this->htaccess_merged.='<'.$command.' '.$module.">\n";
			foreach ($content as $c)
			{
				$this->htaccess_merged.=$c."\n";
			}
			$this->htaccess_merged.='</'.$command.">\n\n";
		}
	}

	/**
	 * Get merged htaccess file.
	 *
	 * @return   string
	 */
	public function getHtaccess()
	{
		return $this->htaccess_merged;
	}

	/**
	 * Write merged htaccess file.
	 *
	 * @param   string      $htaccess           Path to merged .htaccess file
	 */
	public function save($htaccess)
	{
		file_put_contents($htaccess,$this->getHtaccess());
	}

}