<?php

/**
 * SOTESHOP/stOrder
 *
 * Ten plik należy do aplikacji stOrder opartej na licencji (Professional License SOTE).
 * Nie zmieniaj tego pliku, jeśli chcesz korzystać z automatycznych aktualizacji oprogramowania.
 * Jeśli chcesz wprowadzać swoje modyfikacje do programu, zapoznaj się z dokumentacją, jak zmieniać
 * oprogramowanie bez zmiany kodu bazowego http://www.sote.pl/modifications
 *
 * @package     stOrder
 * @subpackage  helpers
 * @copyright   SOTE (www.sote.pl)
 * @license     http://www.sote.pl/license/sote (Professional License SOTE)
 * @version     $Id: stOrderHelper.php 13691 2011-06-20 07:00:55Z marcin $
 */
use_helper('Number', 'Tag', 'stCurrency');

function st_order_price_format($amount, $currency)
{
   return $currency->getFrontSymbol().st_format_price($amount).' '.$currency->getBackSymbol();
}

function st_order_product_price(OrderProduct $product, $with_tax = false)
{
   $currency = $product->getGlobalCurrency();

   return st_order_price_format($product->getPrice($with_tax, true), $currency);
}

function st_order_product_total_amount($object, $with_tax = true)
{
   $currency = $object instanceof OrderProduct ? $object->getGlobalCurrency() : $object->getOrderCurrency();

   return st_order_price_format($object->getTotalAmount($with_tax, true), $currency);
}

function st_order_delivery_cost(Order $order, $with_tax = false)
{
   $currency = $order->getOrderCurrency();

   return st_order_price_format($order->getOrderDelivery()->getCost($with_tax, true), $currency);
}

function st_order_display_product_options(OrderProduct $order_product)
{
   echo st_order_render_product_options($order_product);
}

function st_order_render_product_options(OrderProduct $order_product)
{
   if ($order_product->hasOldOptions())
   {
      $options = $order_product->getOptions(true);

      $content = '';

      foreach ($options as $option)
      {
         $content .= content_tag('li', $option);
      }

      return content_tag('ul', $content, array('class' => 'st_order-product-options'));
   }
   else
   {
      $price_modifiers = $order_product->getPriceModifiers();

      $content = '';

      foreach ($price_modifiers as $price_modifier)
      {
         $content .= content_tag('li', $price_modifier['label']);
      }

      return content_tag('ul', $content, array('class' => 'st_order-product-options'));
   }
}

function st_order_status($order_status)
{
   switch ($order_status->getType())
   {
      case 'ST_CANCELED':
         $color = '#b22222';
         break;
      case 'ST_PENDING':
         $color = '#daa520';
         break;
      case 'ST_COMPLETE':
         $color = '#008000';
         break;
   }

   return content_tag('b', $order_status, array('style' => 'white-space: nowrap; color: '.$color));
}

/**
 * Zwraca przekazana cenę w poprawnym formacie i walucie
 *
 * @param                float       $price              Cena
 * @param   OrderCurrency $order_currency   Obiekt waluty
 * @return   string
 */
function st_order_price($price, $order_currency)
{
   return $order_currency->getFrontSymbol().' '.format_currency($order_currency->getPrice($price)).' '.$order_currency->getBackSymbol();
}

/**
 * Zwraca łaczna kwotę zamówienia (uwzglednia format i walutę)
 *
 * @param   Order       $order              Obiekt zamówienia
 * @return   string
 */
function st_order_total_amount($order, $with_tax = true, $with_currency = true)
{
   $order_currency = $order->getOrderCurrency();

   $total_amount = $order->getTotalAmount($with_tax, $with_currency);

   $delivery_cost = $order->getOrderDelivery()->getCost($with_tax, $with_currency);

   if (!$with_currency)
   {
      $order_currency = stCurrency::getInstance(sfContext::getInstance())->getBackendMainCurrency();
   }

   return st_order_price_format($total_amount + $delivery_cost, $order_currency);
}

/**
 * Funkcja pomocnicza obliczajaca łaczna cenę produktu (uwzględnia walutę)
 *
 * @param   OrderProduct $order_product     Obiekt produktu
 * @param   OrderCurrency $order_currency   Obiekt waluty
 * @return   float
 */
function _st_order_product_total_amount($order_product, $order_currency)
{
   return $order_product->getPrice(true, true) * $order_product->getQuantity();
}

/**
 * Funkcja pomocnicza obliczajaca łaczna kwotę zamówienia (uwzglednia walutę)
 *
 * @param   array       $order_products     Lista obiektów OrderProduct
 * @param   OrderCurrency $order_currency   Obiekt waluty
 * @return   float
 */
function _st_order_total_amount($order_products, $order_currency)
{
   $total_amount = 0;

   foreach ($order_products as $order_product)
   {
      $total_amount += _st_order_product_total_amount($order_product, $order_currency);
   }

   return $total_amount;
}