<?php
/**
 * SOTESHOP/stPriceCompare
 *
 * Ten plik należy do aplikacji stPriceCompare opartej na licencji (Professional License SOTE). 
 * Nie zmieniaj tego pliku, jeśli chcesz korzystać z automatycznych aktualizacji oprogramowania. 
 * Jeśli chcesz wprowadzać swoje modyfikacje do programu, zapoznaj się z dokumentacją, jak zmieniać 
 * oprogramowanie bez zmiany kodu bazowego http://www.sote.pl/modifications
 *
 * @package     stPriceCompare
 * @subpackage  libs
 * @copyright   SOTE (www.sote.pl)
 * @license     http://www.sote.pl/license/sote (Professional License SOTE)
 * @version     $Id: stPriceCompareProductParser.class.php 10928 2011-02-09 12:10:58Z michal $
 * @author      Michal Prochowski <michal.prochowski@sote.pl>
 */

/**
 * Klasa stPriceCompareProductParser
 *
 * @package    stPriceCompare
 * @subpackage libs
 */
class stPriceCompareProductParser
{
	/**
	 * Obiekt produktu
	 *
	 * @var Product
	 */
	protected $product = null;

	/**
	 * Obiekt sfContext
	 *
	 * @var sfContext
	 */
	protected $context = null;

	/**
	 * Konstruktor
	 *
	 * @param Product $product obiekt produktu
	 */
	public function __construct($product)
	{
		$this->product = $product;
		$this->context = sfContext::getInstance();
	}

	/**
	 * Sprawdzanie produktu
	 *
	 * @return bool
	 */
	public function checkProduct()
	{
		if (!is_object($this->product)) return false;
		if (!is_object($this->product->getDefaultCategory())) return false;
		return true;
	}

	/**
	 * Metoda __call:
	 *  - zwraca nie zmodyfikowane wartości z produktu
	 *  - dodaje wywyłanie has np. $obj->hasName() 
	 *
	 * @param string $method nazwa wywoływanej metody 
	 * @param array $args argumenty wywoływanej metody
	 * @return bool
	 */
	public function __call($method, $args = array())
	{
		if (!$this->checkProduct()) return false;
		if (method_exists($this->product, $method))
		{
			if (empty($args))
			{
				return $this->product->$method();
			} else {
				return $this->product->$method($args);
			}
		} else {
			if (ereg('has', $method))
			{
				$getMethod = str_replace('has', 'get', $method);

				if (empty($args))
				{
					$value = $this->product->$getMethod();
				} else {
					$value = $this->product->$getMethod($args);
				}

				if (empty($value)) return false;
				return true;
			}
		}
	}

	/**
	 * Pobieranie nazwy
	 *
	 * @return string
	 */
	public function getName()
	{
		$name = $this->product->getName();

		$event = new sfEvent($this, 'stPriceComparePlugin.changeName', array());
		stEventDispatcher::getInstance()->notify($event);
		$changedName = $event->getReturnValue();

		if ($changedName != null) $name = $changedName;
		return $name;
	}

	/**
	 * Pobieranie ceny
	 *
	 * @param int $decimals ilośc liczb po przecinku
	 * @param string $decimalsPoint separator
	 * @param string $thousandsSeparator separator tysiąca
	 * @return string
	 */
	public function getPrice($decimalsPoint = '.', $decimals = 2, $thousandsSeparator = '')
	{
		$price = $this->product->getPriceBrutto();

		$event = new sfEvent($this, 'stPriceComparePlugin.changePrice', array());
		stEventDispatcher::getInstance()->notify($event);
		$changedPrice = $event->getReturnValue();
		if ($changedPrice != null) $price = $changedPrice;
			
		return number_format($price, $decimals, $decimalsPoint, $thousandsSeparator);
	}

	/**
	 * Pobieranie adresu www
	 *
	 * @return string
	 */
	public function getUrl()
	{
		return 'http://'.$this->context->getRequest()->getHost().'/pl/'.$this->product->getFriendlyUrl().'.html';
	}

	/**
	 * Pobieranie ścieżki kategorii
	 *
	 * @param string $separator separator kategorii
	 * @return string
	 */
	public function getCategory($separator = '/', $htmlspecialchars = false)
	{
		$category = '';
		if ($htmlspecialchars == true)
		{
			foreach ($this->product->getDefaultCategory()->getPath() as $categoryPath) $category.= htmlspecialchars($categoryPath->getName()).$separator;
			$category.= htmlspecialchars($this->product->getDefaultCategory()->getName());
		} else {
			foreach ($this->product->getDefaultCategory()->getPath() as $categoryPath) $category.= $categoryPath->getName().$separator;
			$category.= $this->product->getDefaultCategory()->getName();
		}
		return $category;
	}

	/**
	 * Pobieranie nazwy głównej kategorii
	 *
	 * @return string
	 */
	public function getCategoryName()
	{
		return $this->product->getDefaultCategory()->getName();
	}

	/**
	 * Pobieranie id głównej kategorii
	 *
	 * @return string
	 */
	public function getCategoryId()
	{
		return $this->product->getDefaultCategory()->getId();
	}

	/**
	 * Pobieranie adresu zdjęcia 
	 *
	 * @return string
	 */
	public function getPhoto($showBlank = true)
	{
		return htmlspecialchars(st_product_image_path($this->product, 'full', $showBlank, false, true));
	}

	/**
	 * Pobieranie producenta
	 *
	 * @return string
	 */
	public function getProducer($htmlspecialchars = false)
	{
		if (!is_object($this->product->getProducer())) return false;
		if ($htmlspecialchars == true) return htmlspecialchars($this->product->getProducer()->getName());
		return $this->product->getProducer()->getName();
	}

	/**
	 * Pobieranie opisu
	 *
	 * @param bool $stripTags czy ma wyrzucać tagi
	 * @param string $stripTagsArray lista tagów do pozostawienia
	 * @param bool $replaceEntity czy ma zamieniać znaki encji
	 * @return string
	 */
	public function getDescription($stripTags = false, $stripTagsArray = '', $replaceEntity = false)
	{
		$description = $this->product->getDescription();

		if (!empty($description))
		{
			$description = preg_replace('/<script(.*)<\\/script>/sU', '', $description);
				
			if ($stripTags == true) $description = strip_tags($this->product->getDescription(), $stripTagsArray);

			if ($replaceEntity == true)
			{
				$search = array('&nbsp;', '&oacute;');
				$replace = array(' ', 'ó');
				$description = str_replace($search, $replace, $description);
			}
		}

		return $description;
	}

	/**
	 * Pobieranie statusu dostępności
	 *
	 * @param object $priceCompareObject
	 * @param int $default
	 * @return int
	 */
	public function getPriceCompareAvailability($priceCompareObject, $default = 0)
	{
		$availabilityId = $this->getAvailabilityId();
		if (is_null($availabilityId) && !is_null($this->getStock()))
		{
			$c = new Criteria();
			$c->add(AvailabilityPeer::STOCK_FROM, $this->getStock(), Criteria::LESS_EQUAL);
			$c->addDescendingOrderByColumn(AvailabilityPeer::STOCK_FROM);
			$availability = AvailabilityPeer::doSelectOne($c);
			if (is_object($availability)) $availabilityId = $availability->getId();
		}

		if (is_null($availabilityId))
		{
			return $default;
		} else {
			if (is_null($priceCompareObject->getConfig('availability_'.$availabilityId))) return $default;
			return $priceCompareObject->getConfig('availability_'.$availabilityId);
		}
	}
}