<?php

/**
 * SOTESHOP/stDelivery
 *
 * Ten plik należy do aplikacji stDelivery opartej na licencji (Professional License SOTE).
 * Nie zmieniaj tego pliku, jeśli chcesz korzystać z automatycznych aktualizacji oprogramowania.
 * Jeśli chcesz wprowadzać swoje modyfikacje do programu, zapoznaj się z dokumentacją, jak zmieniać
 * oprogramowanie bez zmiany kodu bazowego http://www.sote.pl/modifications
 *
 * @package     stDelivery
 * @subpackage  libs
 * @copyright   SOTE (www.sote.pl)
 * @license     http://www.sote.pl/license/sote (Professional License SOTE)
 * @version     $Id: stDeliveryFrontend.class.php 7534 2010-08-11 13:48:48Z marcin $
 * @author      Marcin Butlak <marcin.butlak@sote.pl>
 */

/**
 * Klasa zarządzająca dostawami w sklepie
 *
 * @author Marcin Butlak <marcin.butlak@sote.pl>
 *
 * @package     stDelivery
 * @subpackage  libs
 */
class stDeliveryFrontend
{
   const SESSION_NAMESPACE = 'soteshop/stDeliveryFrontend';

   protected static $instance = null;

   protected 
      $delivery = null,
      $deliveryCountry = null,
      $deliveries = null,
      $deliveryCountries = null,
      $minOrderAmount = null,
      $minOrderQuantity = null,
      $minOrderWeight = null,
      $maxOrderAmount = null,
      $maxOrderQuantity = null,
      $maxOrderWeight = null,
      $basket = null,
      $dispatcher = null;

   /**
    * Incjalizacja klasy stDelivery
    *
    * @param stBasket $basket
    */
   public function initialize($basket)
   {
      $this->basket = $basket;

      $this->dispatcher = stEventDispatcher::getInstance();
   }

   /**
    * Zwraca instancje obiektu
    *
    * @param stBasket $basket
    * @return   stDeliveryFrontend
    */
   public static function getInstance($basket)
   {
      if (!isset(self::$instance))
      {
         $class = __CLASS__;

         self::$instance = new $class();

         self::$instance->initialize($basket);
      }

      return self::$instance;
   }

   /**
    * Usuwa z sesji domyślna dostawę oraz domyślną płatność
    */
   public function clearSession()
   {
      $this->getUser()->getAttributeHolder()->remove('delivery', self::SESSION_NAMESPACE);

      $this->getUser()->getAttributeHolder()->remove('delivery_payment', self::SESSION_NAMESPACE);
   }

   /**
    *
    * @return stUser
    */
   public function getUser()
   {
      return $this->basket->getUser();
   }

   public function getBasket()
   {
      return $this->basket;
   }

   /**
    * Ustawia domyslna dostawe
    *
    * @param mixed $delivery
    */
   public function setDefaultDelivery($delivery)
   {
      if ($delivery)
      {
         if (!is_object($delivery))
         {
            $this->delivery = $this->getDeliveryById($delivery);
         }
         else
         {
            $this->delivery = $delivery;
         }
      }
   }

   public function setDefaultDeliveryCountry($delivery_country)
   {
      if ($delivery_country)
      {
         if (!is_object($delivery_country))
         {
            $this->deliveryCountry = $this->getCountryById($delivery_country);
         }
         else
         {
            $this->deliveryCountry = $delivery_country;
         }
      }
   }

   /**
    * Pobiera listę aktywnych dostaw
    *
    * @return array of Delivery
    */
   public function getDeliveries()
   {
      if (is_null($this->deliveries))
      {
         $this->deliveries = $this->doSelectDeliveries();
      }

      return $this->deliveries;
   }

   /**
    *
    * Metoda pomocniczna pobierająca listę aktywnych dostaw
    *
    * @param Criteria $c Dodatkowe kryteria filtrujace liste dostaw
    * @return array of Delivery
    */
   public function doSelectDeliveries(Criteria $c = null)
   {
      if (is_null($c))
      {
         $c = new Criteria();
      }
      else
      {
         $c = clone $c;
      }

      $this->setBaseFilterCriteria($c);

      $this->setCountriesFilterCriteria($c);

      $this->setMaxFilterCriteria($c);

      $deliveries = DeliveryPeer::doSelect($c);

      return array_map(array($this, 'deliveryCallback'), $deliveries);
   }

   public function getDeliveryCountries($without_max_restrictions = false)
   {
      if (is_null($this->deliveryCountries))
      {
         $c = new Criteria();

         $c->addJoin(CountriesAreaHasCountriesPeer::COUNTRIES_ID, CountriesPeer::ID);

         $c->addJoin(CountriesAreaHasCountriesPeer::COUNTRIES_AREA_ID, CountriesAreaPeer::ID);

         $c->addJoin(CountriesAreaPeer::ID, DeliveryPeer::COUNTRIES_AREA_ID);

         $c->add(CountriesAreaPeer::IS_ACTIVE, true);

         $this->setBaseFilterCriteria($c);

         if (!$without_max_restrictions)
         {
            $this->setMaxFilterCriteria($c);
         }

         $c->addGroupByColumn(CountriesPeer::ID);

         $this->deliveryCountries = CountriesPeer::doSelectActive($c);

         $this->deliveryCountries = array_map(array($this, 'deliveryCountryCallback'), $this->deliveryCountries);
      }

      return $this->deliveryCountries;
   }

   /**
    *
    * Jest dostawa domyślna?
    *
    * @return bool
    */
   public function hasDefaultDelivery()
   {
      return (bool) $this->getDefaultDelivery();
   }

   /**
    *
    * Są dostawy?
    *
    * @return bool
    */
   public function hasDeliveries()
   {
      $deliveries = $this->getDeliveries();

      return!empty($deliveries);
   }

   /**
    *
    * Zwraca łączny koszt dostawy
    *
    * @return float
    */
   public function getTotalDeliveryCost($with_tax = false, $with_currency = false)
   {
      $total_amount = 0.00;

      $delivery = $this->getDefaultDelivery();

      if ($delivery)
      {
         $total_amount += $delivery->getTotalCost($with_tax, $with_currency);

         $payment = $delivery->getDefaultPayment();

         if ($payment)
         {

            if ($delivery->isFree() && $payment->isFree())
            {
               return 0.00;
            }

            $total_amount += $payment->getCost($with_tax, $with_currency);
         }
      }

      return $total_amount > 0 ? $total_amount : 0.00;
   }

   /**
    * Pobiera domyslna dostawe
    *
    * @return stDeliveryFrontendContainer
    */
   public function getDefaultDelivery()
   {
      if (is_null($this->delivery))
      {
         $deliveries = $this->getDeliveries();

         $default = $this->getUser()->getAttribute('delivery', null, self::SESSION_NAMESPACE);

         foreach ($deliveries as $delivery)
         {
            if ($default)
            {
               if ($default == $delivery->getId())
               {
                  $this->setDefaultDelivery($delivery);

                  break;
               }
            }
            else
            {
               if ($delivery->getDelivery()->getIsDefault())
               {
                  $this->setDefaultDelivery($delivery);

                  break;
               }
            }
         }

         if (is_null($this->delivery) && !empty($deliveries))
         {
            $this->setDefaultDelivery($deliveries[0]);
         }
      }

      return $this->delivery;
   }

   /**
    * Pobiera domyslny kraj dostawy
    *
    * @return stDeliveryCountryFrontendContainer
    */
   public function getDefaultDeliveryCountry()
   {
      if (is_null($this->deliveryCountry))
      {
         $countries = $this->getDeliveryCountries();

         $default = $this->getUser()->getAttribute('delivery_country', null, self::SESSION_NAMESPACE);

         foreach ($countries as $country)
         {
            if ($default)
            {
               if ($default == $country->getId())
               {
                  $this->setDefaultDeliveryCountry($country);

                  break;
               }
            }
            else
            {
               if ($country->getDeliveryCountry()->getIsDefault())
               {
                  $this->setDefaultDeliveryCountry($country);

                  break;
               }
            }
         }

         if (is_null($this->deliveryCountry) && !empty($countries))
         {
            $this->setDefaultDeliveryCountry($countries[0]);
         }
      }

      return $this->deliveryCountry;
   }

   /**
    *
    * Ustawia kryteria filtrowania dla dostaw - podstawowe kryteria filtrowania
    *
    * @param Criteria $c
    */
   protected function setBaseFilterCriteria($c)
   {
      $c->add(DeliveryPeer::ACTIVE, true);

      $this->dispatcher->notify(new sfEvent($this, 'stDeliveryFrontend.postSetBaseFilterCriteria', array('criteria' => $c)));
   }

   /**
    * Ustawia kryteria filtrowania dla dostaw - krytuje filtrowania wedlug domyslnego kraju
    */
   protected function setCountriesFilterCriteria($c)
   {
      $c->addJoin(DeliveryPeer::COUNTRIES_AREA_ID, CountriesAreaHasCountriesPeer::COUNTRIES_AREA_ID);

      $c->addJoin(CountriesAreaHasCountriesPeer::COUNTRIES_ID, CountriesPeer::ID);

      $c->addJoin(CountriesAreaHasCountriesPeer::COUNTRIES_AREA_ID, CountriesAreaPeer::ID);

      $c->addJoin(CountriesAreaPeer::ID, DeliveryPeer::COUNTRIES_AREA_ID);

      $c->add(CountriesAreaPeer::IS_ACTIVE, true);

      if ($this->getDefaultDeliveryCountry())
      {
         $c->add(CountriesAreaHasCountriesPeer::COUNTRIES_ID, $this->getDefaultDeliveryCountry()->getId());
      }
   }

   /**
    *
    * Ustawia kryteria filtrowania dla dostaw - maksymalne wartości zamówienia
    *
    * @param Criteria $c
    */
   public function setMaxFilterCriteria($c)
   {
      $query = '((%1$s = 0 OR %1$s <= %3$s) AND (%2$s = 0 OR %2$s >= %3$s))';

      $c->add(DeliveryPeer::MAX_ORDER_AMOUNT, sprintf($query, DeliveryPeer::MIN_ORDER_AMOUNT, DeliveryPeer::MAX_ORDER_AMOUNT, $this->basket->getTotalAmount(true)), Criteria::CUSTOM);

      $c->add(DeliveryPeer::MAX_ORDER_QUANTITY, sprintf($query, DeliveryPeer::MIN_ORDER_QUANTITY, DeliveryPeer::MAX_ORDER_QUANTITY, $this->basket->getTotalProductQuantity()), Criteria::CUSTOM);

      $c->add(DeliveryPeer::MAX_ORDER_WEIGHT, sprintf($query, DeliveryPeer::MIN_ORDER_WEIGHT, DeliveryPeer::MAX_ORDER_WEIGHT, $this->basket->getTotalProductWeight()), Criteria::CUSTOM);
   }

   /**
    *
    * Pobiera maksymalną dozwoloną kwotę zamówienia
    *
    * @return float
    */
   public function getMaxOrderAmount()
   {
      if (is_null($this->maxOrderAmount))
      {
         $this->maxOrderAmount = $this->getValueByColumn(DeliveryPeer::MAX_ORDER_AMOUNT);
      }

      return $this->maxOrderAmount;
   }

   public function getMinOrderAmount()
   {
      if (is_null($this->minOrderAmount))
      {
         $this->minOrderAmount = $this->getValueByColumn(DeliveryPeer::MIN_ORDER_AMOUNT, false);
      }

      return $this->minOrderAmount;
   }

   /**
    *
    * Pobiera maksymalną dozwoloną wagę zamówienia
    *
    * @return float
    */
   public function getMaxOrderWeight()
   {
      if (is_null($this->maxOrderWeight))
      {
         $this->maxOrderWeight = $this->getValueByColumn(DeliveryPeer::MAX_ORDER_WEIGHT);
      }

      return $this->maxOrderWeight;
   }

   public function getMinOrderWeight()
   {
      if (is_null($this->minOrderWeight))
      {
         $this->minOrderWeight = $this->getValueByColumn(DeliveryPeer::MIN_ORDER_WEIGHT, false);
      }

      return $this->minOrderWeight;
   }

   /**
    *
    * Pobiera maksymlaną dozwoloną ilość sztuk w zamówieniu
    *
    * @return integer
    */
   public function getMaxOrderQuantity()
   {
      if (is_null($this->maxOrderQuantity))
      {
         $this->maxOrderQuantity = $this->getValueByColumn(DeliveryPeer::MAX_ORDER_QUANTITY);
      }

      return $this->maxOrderQuantity;
   }

   public function getMinOrderQuantity()
   {
      if (is_null($this->minOrderQuantity))
      {
         $this->minOrderQuantity = $this->getValueByColumn(DeliveryPeer::MIN_ORDER_QUANTITY, false);
      }

      return $this->minOrderQuantity;
   }

   public static function shutdown()
   {
      if (isset(self::$instance))
      {
         $user = self::$instance->getUser();

         $delivery = self::$instance->getDefaultDelivery();

         if ($delivery)
         {
            $user->setAttribute('delivery', $delivery->getId(), self::SESSION_NAMESPACE);

            $payment = $delivery->getDefaultPayment();

            if ($payment)
            {
               $user->setAttribute('delivery_payment', $payment->getId(), self::SESSION_NAMESPACE);
            }
         }

         $country = self::$instance->getDefaultDeliveryCountry();

         if ($country)
         {
            $user->setAttribute('delivery_country', $country->getId(), self::SESSION_NAMESPACE);
         }
      }
   }

   protected function getValueByColumn($column, $max = true)
   {
      $con = Propel::getConnection();

      $c = new Criteria();

      if ($max)
      {
         $c->addAsColumn('max_value', sprintf('MAX(%s)', $column));
      }
      else
      {
         $c->addAsColumn('max_value', sprintf('MIN(%s)', $column));
      }

      $this->setBaseFilterCriteria($c);

      $this->setCountriesFilterCriteria($c);

      $rs = BasePeer::doSelect($c, $con);

      return $rs->next() ? $rs->get(1) : 0;
   }

   /**
    *
    * Metoda pomocnicza - Zwraca rozszerzony obiekt Delivery
    *
    * @param DeliveryHasPaymentType $delivery_payment
    * @return stDeliveryFrontendContainer Obiekt rozszerzony
    */
   protected function deliveryCallback($delivery)
   {
      $delivery = new stDeliveryFrontendContainer($this, $delivery);

      return $delivery;
   }

   protected function deliveryCountryCallback($delivery_country)
   {
      $delivery_country = new stDeliveryCountryFrontendContainer($this, $delivery_country);

      return $delivery_country;
   }

   /**
    *
    * Pobiera rozszerzoną dostawę na podstawie id
    *
    * @param integer $id Id dostawy
    * @return stDeliveryFrontendContainer Rozszerzona dostawa
    */
   protected function getDeliveryById($id)
   {
      $deliveries = $this->getDeliveries();

      foreach ($deliveries as $delivery)
      {
         if ($delivery->getId() == $id)
         {
            return $delivery;
         }
      }

      return null;
   }

   /**
    *
    * Pobiera kraj dostawy na podstawie id
    *
    * @param integer $id Id dostawy
    * @return stDeliveryCountryFrontendContainer Kraj dostawy
    */
   protected function getCountryById($id)
   {
      $delivery_countries = $this->getDeliveryCountries();

      foreach ($delivery_countries as $delivery_country)
      {
         if ($delivery_country->getId() == $id)
         {
            return $delivery_country;
         }
      }

      return null;
   }

}

/**
 * Kontener rozszerzający funkcjonalność modelu Delivery
 *
 * @see Delivery
 */
class stDeliveryFrontendContainer
{
   protected
      $delivery = null,
      $deliveryPayments = null,
      $deliveryPayment = null,
      $deliveryFrontend = null,
      $additionalCost = null;

   /**
    * Zwraca orginalny obiekt dostawy
    *
    * @return Delivery
    */
   public function getDelivery()
   {
      return $this->delivery;
   }

   public function __construct($delivery_frontend, $delivery)
   {
      $this->delivery = $delivery;

      $this->deliveryFrontend = $delivery_frontend;
   }

   public function getAdditionalCost($with_tax = false, $with_currency = false)
   {
      if ($this->isFree())
      {
         return 0.00;
      }

      if (is_null($this->additionalCost))
      {
         $basket = $this->deliveryFrontend->getBasket();

         $type = $this->delivery->getSectionCostType();

         $c = new Criteria();

         switch ($type)
         {
            case "ST_BY_ORDER_AMOUNT":
               $value = $basket->getTotalAmount(true);
               break;
            case "ST_BY_ORDER_QUANTITY":
               $value = $basket->getTotalProductQuantity();
               break;
            case "ST_BY_ORDER_WEIGHT":
               $value = $basket->getTotalProductWeight();
               break;
            default:
               $value = 0;
               break;
         }

         if ($type)
         {

            $c->add(DeliverySectionsPeer::VALUE_FROM, $value, Criteria::LESS_EQUAL);

            $c->setLimit(1);

            $c->addDescendingOrderByColumn(DeliverySectionsPeer::VALUE_FROM);

            $tmp = $this->getDeliverySectionss($c);

            $this->additionalCost = isset($tmp[0]) ? $tmp[0] : 0.00;
         }
         else
         {
            $this->additionalCost = 0.00;
         }
      }

      if (is_object($this->additionalCost))
      {
         if ($with_tax)
         {
            $cost = $this->additionalCost->getCostBrutto($with_currency);
         }
         else
         {
            $cost = $this->additionalCost->getCostNetto($with_currency);
         }
      }
      else
      {
         $cost = $this->additionalCost;
      }

      return $cost;
   }

   /**
    *
    * Zwraca aktualnego użytkownika
    *
    * @return stUser
    */
   public function getUser()
   {
      return $this->deliveryFrontend->getUser();
   }

   public function getDefaultCost($with_tax = false, $with_currency = false)
   {
      if ($this->isFree())
      {
         return '0.00';
      }

      if ($with_tax)
      {
         $cost = $this->delivery->getCostBrutto($with_currency);
      }
      else
      {
         $cost = $this->delivery->getCostNetto($with_currency);
      }

      return $cost;
   }

   public function getTotalCost($with_tax = false, $with_currency = false)
   {
      return $this->getDefaultCost($with_tax, $with_currency) + $this->getAdditionalCost($with_tax, $with_currency);
   }

   /**
    *
    * Jest darmowa?
    *
    * @return bool
    */
   public function isFree()
   {
      $free_from = $this->delivery->getFreeFrom();

      $total_amount = $this->deliveryFrontend->getBasket()->getTotalAmount(true);

      return $free_from > 0 && $total_amount >= $free_from;
   }

   /**
    *
    * Jest domyślna?
    *
    * @return bool
    */
   public function getIsDefault()
   {
      $delivery = $this->deliveryFrontend->getDefaultDelivery();

      if ($delivery)
      {
         return $delivery->getId() == $this->delivery->getId();
      }

      return $this->delivery->getIsDefault();
   }

   /**
    *
    * Zwraca listę płatności dostawy
    *
    * @return array Of stDeliveryPaymentFrontendContainer
    */
   public function getDeliveryPayments()
   {
      if (is_null($this->deliveryPayments))
      {
         $c = new Criteria();

         $this->setFilterCriteria($c);

         $this->deliveryPayments = DeliveryHasPaymentTypePeer::doSelectJoinPaymentType($c);

         $this->deliveryPayments = array_filter($this->deliveryPayments, array($this, 'paymentFilterCallback'));

         $this->deliveryPayments = array_map(array($this, 'paymentCallback'), $this->deliveryPayments);
      }

      return $this->deliveryPayments;
   }

   /**
    *
    * Ustawia kryteria filtrowania dla płatności
    *
    * @param Criteria $c
    */
   protected function setFilterCriteria($c)
   {
      $c->add(DeliveryHasPaymentTypePeer::DELIVERY_ID, $this->getId());

      $c->add(DeliveryHasPaymentTypePeer::IS_ACTIVE, true);

      $c->add(PaymentTypePeer::ACTIVE, true);

      $c->addAscendingOrderByColumn(DeliveryHasPaymentTypePeer::ID);
   }

   /**
    *
    * Są płatności?
    *
    * @return bool
    */
   public function hasDeliveryPayments()
   {
      $delivery_payments = $this->getDeliveryPayments();

      return!empty($delivery_payments);
   }

   /**
    *
    * @return stDeliveryPaymentFrontendContainer
    */
   public function getDefaultPayment()
   {
      if (is_null($this->deliveryPayment))
      {
         $payments = $this->getDeliveryPayments();

         $default = $this->getUser()->getAttribute('delivery_payment', null, stDeliveryFrontend::SESSION_NAMESPACE);

         foreach ($payments as $payment)
         {
            if ($default)
            {
               if ($default == $payment->getId())
               {
                  $this->setDefaultPayment($payment);

                  break;
               }
            }
            else
            {
               if ($payment->getDeliveryPayment()->getIsDefault())
               {
                  $this->setDefaultPayment($payment);

                  break;
               }
            }
         }

         if (is_null($this->deliveryPayment) && !empty($payments))
         {
            $this->setDefaultPayment(current($payments));
         }
      }

      return $this->deliveryPayment;
   }

   /**
    *
    * Ustawia domyślną płatność
    *
    * @param mixed $delivery_payment Id lub rozszerzony obiekt płatności
    */
   public function setDefaultPayment($delivery_payment)
   {
      if ($delivery_payment)
      {
         if (!is_object($delivery_payment))
         {
            $this->deliveryPayment = $this->getPaymentById($delivery_payment);
         }
         else
         {
            $this->deliveryPayment = $delivery_payment;
         }
      }
   }

   /**
    *
    * Wykonuje metody obiektu Delivery nie występujące w stDeliveryFrontendContainer
    *
    * @param string $name Nazwa metody
    * @param array $arguments Argumenty metody
    * @return mixed
    */
   public function __call($name, $arguments)
   {
      return call_user_func_array(array($this->delivery, $name), $arguments);
   }

   protected function paymentFilterCallback($delivery_payment)
   {
      return $delivery_payment->getPayment() ? $delivery_payment->getPayment()->checkPaymentConfiguration() : false;
   }

   /**
    *
    * Metoda pomocnicza - Zwraca rozszerzony obiekt DeliveryHasPaymentType
    *
    * @param DeliveryHasPaymentType $delivery_payment
    * @return stDeliveryPaymentFrontendContainer Obiekt rozszerzony
    */
   protected function paymentCallback($delivery_payment)
   {
      $delivery_payment = new stDeliveryPaymentFrontendContainer($this->deliveryFrontend, $delivery_payment);

      $delivery_payment->setDelivery($this);

      return $delivery_payment;
   }

   /**
    *
    * Pobiera rozszerzony obiekt stDeliveryPaymentFrontendContainer na podstawie id
    *
    * @param integer $id
    * @return stDeliveryPaymentFrontendContainer
    */
   protected function getPaymentById($id)
   {
      $payments = $this->getDeliveryPayments();

      foreach ($payments as $payment)
      {
         if ($payment->getId() == $id)
         {
            return $payment;
         }
      }

      return null;
   }

}

/**
 * Kontener rozszerzający funkcjonalność modelu DeliveryHasPaymentType
 *
 * @see DeliveryHasPaymentType
 */
class stDeliveryPaymentFrontendContainer
{

   protected 
      $deliveryPayment = null,
      $deliveryFrontend = null;

   public function __construct($delivery_frontend, $delivery_payment)
   {
      $this->deliveryPayment = $delivery_payment;

      $this->deliveryFrontend = $delivery_frontend;
   }

   /**
    * Zwraca orginalny obiekt platnosci dostawt
    *
    * @return DeliveryHasPaymentType
    */
   public function getDeliveryPayment()
   {
      return $this->deliveryPayment;
   }

   public function getId()
   {
      return $this->getPayment() ? $this->getPayment()->getId() : null;
   }

   public function getSocketName()
   {
      return $this->getPayment() ? 'stPayment_show_'.$this->getPayment()->getModuleName().'_info' : null;
   }

   public function getSocketNameExists($type = 'component')
   {
      return (bool) array_key_exists($this->getSocketName(), sfConfig::get('st_socket_'.$type));
   }

   public function getName()
   {
      return $this->getPayment() ? $this->getPayment()->getName() : '';
   }

   public function getDescription()
   {
      return $this->getPayment() ? $this->getPayment()->getDescription() : '';
   }

   public function getCost($with_tax = false, $with_currency = false)
   {
      if ($this->isFree())
      {
         return '0.00';
      }

      if ($with_tax)
      {
         $cost = $this->deliveryPayment->getCostBrutto($with_currency);
      }
      else
      {
         $cost = $this->deliveryPayment->getCostNetto($with_currency);
      }

      return $cost;
   }

   public function isFree()
   {
      $free_from = $this->deliveryPayment->getFreeFrom();

      $total_amount = $this->deliveryFrontend->getBasket()->getTotalAmount(true);

      return $free_from > 0 && $total_amount >= $free_from;
   }

   public function getIsDefault()
   {
      $deliveryPayment = $this->deliveryFrontend->getDefaultDelivery()->getDefaultPayment();

      if ($deliveryPayment)
      {
         return $deliveryPayment->getId() == $this->getId();
      }

      return $this->deliveryPayment->getIsDefault();
   }

   public function __call($name, $arguments)
   {
      return call_user_func_array(array($this->deliveryPayment, $name), $arguments);
   }

}

/**
 * Kontener rozszerzający funkcjonalność modelu Countries
 *
 * @see DeliveryHasPaymentType
 */
class stDeliveryCountryFrontendContainer
{

   protected 
      $deliveryCountry = null,
      $deliveryFrontend = null;

   public function __construct($delivery_frontend, $delivery_country)
   {
      $this->deliveryCountry = $delivery_country;

      $this->deliveryFrontend = $delivery_frontend;
   }

   /**
    * Zwraca orginalny obiekt kraju dostawy
    *
    * @return Countires
    */
   public function getDeliveryCountry()
   {
      return $this->deliveryCountry;
   }

   public function getIsDefault()
   {
      $delivery_country = $this->deliveryFrontend->getDefaultDeliveryCountry();

      if ($delivery_country)
      {
         return $delivery_country->getId() == $this->deliveryCountry->getId();
      }

      return $this->deliveryCountry->getIsDefault();
   }

   public function __call($name, $arguments)
   {
      return call_user_func_array(array($this->deliveryCountry, $name), $arguments);
   }

}