<?php
/**
 * +------------------------------------------------------------------------------
 * LightPHP 轻巧敏捷型PHP框架
 * +------------------------------------------------------------------------------
 * 
 * @package 核心缓存
 * @author webw3cs@gmail.com
+------------------------------------------------------------------------------ 
 */
class Cache
{
    private $tablePre; //表前缀
    private $cacheKey; //缓存标识
    private $cacheTime; //缓存延迟时间
    private $isCache; //是否进行缓存
    /**
     * 构造方法：判断是否使用缓存
     * 
     * @param  $cacheKey 缓存唯一标识
     */
    public function __construct($cacheKey)
    {
        $DBConfig = _config('DB');
        $this -> tablePre = $DBConfig['TablePre']; //数据表前缀
        if ($cacheKey === -1) // 如果cachekey传入的是-1则认为本操作不进行缓存，该功能可实现局部直接操作数据库
        {
            $this -> cacheKey = '';
            $this -> isCache = false;
        } 
        else
        {
            $this -> cacheKey = $cacheKey; //缓存唯一标识
            $this -> isCache = $GLOBALS['C']['IsCache']; //默认操作为使用缓存方法
        } 
    } 

    /**
     * 魔术函数：回调数据查询操作方法：返回数据
     * 
     * @param  $method 数据查询方法，框架DB.class.php中有findAll,findOne,countRecords,用户可自已增加方法
     * @param  $args 查询方法中传入的参数，（数据表|SQL语句|缓存延迟时间）

     * 使用示例：$this->C()->findAll($table,$sql,60);
     */
    public function __call($method, $args)
    {
        $_table = $args[0];
        $sql = $args[1];
        if (is_array($_table))
        {
            foreach($_table as $key => $value) // 如果SQL语句中存在操作多个数据表则将完整的表名循环还原到SQL语句中
            {
                $sql = str_replace($key, $this -> tablePre . $value, $sql); //$this->tablePre.$value为完整的表名
            } 
            $cacheFolder = $_table['TABLE1']; 
            // 将数据表名作为缓存文件名，当操作多个表时取键值为TABLE的表作为缓存文件夹名，所有关于该表的数据缓存将缓存于该文件夹下
        } 
        else
        {
            $sql = str_replace('TABLE', $this -> tablePre . $_table, $sql);
            $cacheFolder = $_table;
        } 

        if ($this -> isCache)
        {
            $cacheDir = _buildDir(CACHE . 'DataCache' . S . $cacheFolder) . S;
            $this -> cacheTime = !isset($args[2])? $GLOBALS['C']['ExpireCTime'] :$args[2] ; 
            // 如果查询时没有局部设置缓存时间则默认使用配置文件中的缓存延迟时间
            $data = $this -> fetchByCache($method, $sql, $cacheDir);
        } 
        else
        {
            $data = $this -> fetchByDB($method, $sql);
        } 

        return $data;
    } 

    /**
     * 数据查询：命名缓存文件与时间文件，若缓存文件存在且没有过期则直接读取，否则生成缓存再读取；返回缓存文件内容
     * 
     * @param  $method 数据查询方法
     * @param  $sql 查询语句
     * @param  $cacheDir 缓存要放置的文件夹
     */
    private function fetchByCache($method, $sql, $cacheDir)
    {
        if (empty($this -> cacheKey)) // 如果缓存标识为空则将 查询方法+SQL作为缓存文件名
        {
            $cacheFile = $cacheDir . md5($sql) . '.php';
            $timeFile = $cacheDir . basename($cacheDir);
        } 
        else
        {
            $cacheFile = $cacheDir . md5($this -> cacheKey) . '.php'; 
            // 传入唯一缓存标识，可实现缓存的异步更新，主要在单条记录的CRUD中使用，将 查询方法+缓存标识作为缓存文件名
            $timeFile = $cacheDir . md5($this -> cacheKey); //时间文件与缓存文件一一相对	
        } 

        if (! file_exists ($cacheFile) || filemtime($cacheFile) + $this -> cacheTime < filemtime($timeFile)) // 判断是否要更新缓存
        {
            $data = $this -> fetchByDB($method, $sql);
            if (empty($data))return $data; //如果为空不生成缓存，防止产生不必要的缓存文件
            if (! file_exists ($timeFile)) $this -> createFile($timeFile);
            $this -> createFile($cacheFile, serialize($data));
            unset($data);
        } 
        if ($readData = unserialize(file_get_contents($cacheFile))) // 读取缓存
        {
            return $readData;
        } 
        else
        {
            _error('readCacheError');
        } 
    } 

    /**
     * 直接从数据库查询数据；返回查询的数据
     * 
     * @param  $method 数据查询方法
     * @param  $sql 查询语句
     */
    private function fetchByDB($method, $sql)
    {
        $database = _class('DB');
        if (method_exists ($database, $method))
        {
            return $database -> $method($sql);
        } 
        else
        {
            _error('methodNotExist', 'Database->' . $method);
        } 
    } 

    /**
     * 添加数据：返回值：新增数据ID
     * 
     * @param  $table 本次操作的数据表
     * @param  $fields 数据插入的字符串
     * @param  $DO 新增记录或替换记录
     */
    public function insert($table, $fields, $DO = 'INSERT')
    {
        $kv = $this -> parseStr($fields, 0); //重组部分SQL语句
        $finalTable = $this -> tablePre . trim($table); //完整的数据表名称
        $sql = str_replace (',)', ' )', " $DO INTO $finalTable ($kv[0]) VALUES ($kv[1])");
        $result = _class('DB') -> update($sql);
        if ($result > 0 && $this -> isCache) // 如果操作成功并且使用缓存时，更新主时间文件
        {
            $cacheDir = _buildDir (CACHE . 'DataCache' . S . $table) . S;
            $this -> createFile($cacheDir . $table);
        } 
        return $result;
    } 

    /**
     * 替换数据
     * 
     * @param  $table 本次操作的数据表
     * @param  $fields 数据插入的字符串
     */
    public function replace($table, $fields)
    {
        $this -> insert($table, $fields, $DO = 'REPLACE');
    } 

    /**
     * 修改数据：返回值：更新的数据记录数
     * 
     * @param  $table 本次操作的数据表
     * @param  $fields 数据更新的字符串
     * @param  $condition 更新条件
     */
    public function update($table, $fields, $condition)
    {
        $setFields = $this -> parseStr($fields, 1);
        $where = 'WHERE ' . $this -> parseStr($condition, 2); //重组部分SQL语句
        $finalTable = $this -> tablePre . trim($table);
        $sql = "UPDATE $finalTable SET $setFields $where";
        $result = _class('DB') -> update($sql);
        if ($result >= 0 && $this -> isCache) // 如果操作成功并且使用缓存时，更新主时间文件，如果缓存标识不为空，更新与缓存文件相对应的时间文件
        {
            $cacheDir = _buildDir (CACHE . 'DataCache' . S . $table) . S;
            $this -> createFile($cacheDir . $table);
            if (!empty($this -> cacheKey)) $this -> createFile($cacheDir . md5($this -> cacheKey));
        } 
        return $result;
    } 

    /**
     * 删除数据：返回值：删除数据的记录数
     * 
     * @param  $table 本次操作的数据表
     * @param  $condition 删除数据的条件
     */
    public function delete($table, $condition)
    {
        $where = 'WHERE ' . $this -> parseStr($condition, 2); //重组部分SQL语句	
        $finalTable = $this -> tablePre . trim($table);
        $sql = "DELETE FROM $finalTable $where";
        $result = _class('DB') -> update($sql);
        if ($result > 0 && $this -> isCache) // 如果操作成功并且使用缓存时，更新主时间文件，如果缓存标识不为空则删除缓存文件与时间文件
        {
            $cacheDir = _buildDir (CACHE . 'DataCache' . S . $table) . S;
            $this -> createFile($cacheDir . $table);
            if (!empty($this -> cacheKey))
            {
                $keyFile = $cacheDir . md5($this -> cacheKey);
                @unlink($keyFile . '.php');
                @unlink($keyFile);
            } 
        } 
        return $result;
    } 

    /**
     * 文件生成：用于缓存文件与时间文件的生成
     * 
     * @param  $file 写入数据的文件
     * @param  $content 写入的内容
     */
    private function createFile($file, $content = 'LIGHTPHP')
    {
        if (! file_put_contents ($file, $content)) _error ('writeCacheError', $file);
    } 

    /**
     * 重组SQL部分语句
     * 
     * @param  $str 要解析的字符串
     * @param  $case 操作类型
     */
    private function parseStr($str, $case)
    {
        $reVal_1 = '';
        $reVal_2 = '';
        $fields = explode (',', $str);
        foreach ($fields as $value)
        {
            $pos = strpos($value, ':');
            $key = substr($value, 0, $pos);
            $val = substr($value, $pos + 1, strlen($value));
            switch ($case)
            {
                case 0: // 增加数据，
                    $reVal_1 .= "$key,";
                    $reVal_2 .= "'$val',";
                    break;
                case 1: // 更新数据
                    $reVal_1 .= "$key='$val',";
                    break;
                case 2: // 数据操作的条件
                    $reVal_1 .= "$key='$val' and ";
                    break;
            } 
        } 
        unset($fields);
        switch ($case)
        {
            case 0:
                return array($reVal_1, $reVal_2);
            case 1:
                return substr($reVal_1, 0, -1);
            case 2:
                return substr($reVal_1, 0, -4);
        } 
    } 

    /**
     * 删除某文件夹下缓存，删除成功返回TRUE
     * 
     * @param  $folder 文件夹名称，以'/'结尾，如'demo/'
     */
    public function clear($folder)
    {
        $path = CACHE . 'DataCache' . S . $folder;
        if (substr(trim($folder), -1) != '/') return false;
        if ($dir = opendir($path))
        {
            while ($file = readdir($dir))
            {
                $isDir = is_dir($file);
                if (!$isDir)
                    @unlink($path . $file);
            } 
            closedir($dir);
        } 

        return true;
    } 

    /**
     * +------------------------------------------------------------------------------
     */
} 

?>
