using System;
using System.Web;
using System.Configuration;
using System.Text.RegularExpressions;

/// <summary>
/// LinkFactory creates absolute links to locations in our site
/// </summary>
public static class LinkFactory
{
  // regular expression that removes characters that aren't a-z, 0-9, dash, underscore or space
  private static Regex purifyUrlRegex = new Regex("[^-a-zA-Z0-9_ ]", RegexOptions.Compiled);

  // regular expression that changes dashes, underscores and spaces to dashes
  private static Regex dashesRegex = new Regex("[-_ ]+", RegexOptions.Compiled);

  // builds a link to a product page
  public static string MakeCategoryProductUrl(
    string categoryName, string categoryId, string productName, string productId)
  {
    // prepare the product name and category name for inclusion in URL
    categoryName = prepareUrlText(categoryName);
    productName = prepareUrlText(productName);

    // read the site domain from configuration file
    string siteDomain = ConfigurationManager.AppSettings["SiteDomain"];

    // build the keyword-rich URL
    string url = String.Format("{0}/Products/{1}-C{2}/{3}-P{4}.html",
      siteDomain, categoryName, categoryId, productName, productId);

    // return the URL
    return HttpUtility.UrlPathEncode(url);
  }

  // builds a link to a media file
  public static string MakeMediaUrl(string name, string extension, string id)
  {
    // prepare the medium name for inclusion in URL
    name = prepareUrlText(name);

    // read the site domain from configuration file
    string siteDomain = ConfigurationManager.AppSettings["SiteDomain"];

    // build the keyword-rich URL 
    string url = String.Format("{0}/{1}-M{2}.{3}", siteDomain, name, id, extension);

    // return the URL
    return HttpUtility.UrlPathEncode(url);
  }

  // prepares a string to be included in an URL
  private static string prepareUrlText(string urlText)
  {
    // remove all characters that aren't a-z, 0-9, dash, underscore or space
    urlText = purifyUrlRegex.Replace(urlText, "");

    // remove all leading and trailing spaces
    urlText = urlText.Trim();

    // change all dashes, underscores and spaces to dashes
    urlText = dashesRegex.Replace(urlText, "-");

    // return the modified string    
    return urlText;
  }
}