using System;
using System.Text.RegularExpressions;
using System.Collections.Generic;

/// <summary>
/// Class that translates custom markup to HTML
/// </summary>
public class CustomMarkup
{
  // store the regular expressions and their string replacements
  private static Dictionary<Regex, string> rs;
    
  // translates the custom markup from the input string to HTML
  public static string Translate(string markupString)
  {
    // apply each regular expression in the collection
    foreach (Regex x in rs.Keys)
    {
      markupString = x.Replace(markupString, rs[x]);
    }

    // return translated string
    return markupString;
  }

  // static constructor creates the collection of Regex objects 
  // and replacement strings used for translation
	static CustomMarkup()
	{
    // create the collection
    rs = new Dictionary<Regex, string>();

    // translate bold tags
    rs.Add(new Regex(@"\{bold}(.*?)\{/bold}", 
      RegexOptions.IgnoreCase), @"<b>$1</b>");

    // translate italic tags
    rs.Add(new Regex(@"\{italic}(.*?)\{/italic}", 
      RegexOptions.IgnoreCase), @"<i>$1</i>");

    // translate underline tags
    rs.Add(new Regex(@"\{underline}(.*?)\{/underline}", 
      RegexOptions.IgnoreCase), @"<u>$1</u>");

    // translate h1 headings
    rs.Add(new Regex(@"\{heading}(.*?)\{/heading}", 
      RegexOptions.IgnoreCase), @"<h1 class=some_class>$1</h1>");

    // translate h2 headings
    rs.Add(new Regex(@"\{subheading}(.*?)\{/subheading}", 
      RegexOptions.IgnoreCase), @"<h2 class=some_other_class>$1</h2>");

    // translate hyperlinks
    rs.Add(new Regex(@"\{link:(.*?)}(.*?)\{/link}", 
      RegexOptions.IgnoreCase), @"<a href=\'$1\'>$2</a>");

    // translate hyperlinks that open in new window
    rs.Add(new Regex(@"\{elink:(.*?)}(.*?)\{/elink}", 
      RegexOptions.IgnoreCase), @"<a href=\'$1\' target=""_blank"">$2</a>");

    // translate unordered list tags
    rs.Add(new Regex(@"\{unordered-list}(.*?)\{/unordered-list}\s*", 
      RegexOptions.IgnoreCase), @"<ul>$1</ul>");

    // translate ordered list tags
    rs.Add(new Regex(@"\{ordered-list}(.*?)\{/ordered-list}\s*", 
      RegexOptions.IgnoreCase), @"<ol>$1</ol>");

    // translate list elements
    rs.Add(new Regex(@"\\s*{list-element}(.*?)\{/list-element}\s*", 
      RegexOptions.IgnoreCase), @"<li>$1</li>");

    // translate picture tags
    rs.Add(new Regex(@"\{picture:(.*?)}", 
      RegexOptions.IgnoreCase), @"<img src=""$1"" border=""0"">");

    // remove tabs
    rs.Add(new Regex(@"\t"), "");

    // remove comments
    rs.Add(new Regex(@"\{comment}(.*?)\{/comment}", 
      RegexOptions.IgnoreCase), "");
  }
}
