using System;
using System.Text.RegularExpressions;

/// <summary>
/// Sanitize contains functionality to remove unaccepted tags or attributes
/// </summary>
public static class Sanitize
{
  // list of accepted/harmeless tags (in lower case)
  private static string[] allowedTags =     
    { "p", "h1", "b", "i", "a", "ul", "li", "pre", "hr", "blockquote", "img" };

  // list of attributes that need to be sanitized
  private static string badAttributes =
    "onerror|onmousemove|onmouseout|onmouseover|" +
     "onkeypress|onkeydown|onkeyup|javascript:";

  // sanitizes the HTML code in $inputHTML
  public static string FixTags(string inputHtml)
  {
    // define the match evaluator
    MatchEvaluator fixThisLink = new MatchEvaluator(Sanitize.fixTag);

    // process each tags in the input string
    string fixedHtml = Regex.Replace(inputHtml,
                                     "(<.*?>)",
                                     fixThisLink,
                                     RegexOptions.IgnoreCase);

    // return the "fixed" input string
    return fixedHtml;
  }

  // remove tag if is not in the list of allowed tags
  private static string fixTag(Match tagMatch)
  {
    string tag = tagMatch.Value;

    // extrag the tag name, such as "a" or "h1"
    Match m = Regex.Match(tag,
                          @"</?(?<tagName>[^\s/]*)[>\s/]",
                          RegexOptions.IgnoreCase);
    string tagName = m.Groups["tagName"].Value.ToLower();

    // if the tag isn't in the list of allowed tags, it should be removed
    if (Array.IndexOf(allowedTags, tagName) < 0)
    {
      return "";
    }

    // remove bad attributes from the tag
    string fixedTag = Regex.Replace(tag,
                        "(" + Sanitize.badAttributes + @")(\s*)(?==)",
                        "SANITIZED", RegexOptions.IgnoreCase);

    // return the altered tag
    return fixedTag;
  }
}
