using System;
using System.Net;
using System.IO;
using System.Text.RegularExpressions;
using System.Web;

/// <summary>
/// Enumeration with the possible levels of spider match confidence
/// </summary>
public enum SpiderMatch
{
  NoMatch, // neither the user agent or IP matches that of a spider
  MatchUA, // the user agent is that of a spider
  MatchIP, // the IP is that of a spider
  MatchUAandIP // both the user agent and the IP are those of a spider
}

/// <summary>
/// Cloak offers functionality for cloaking support
/// </summary>
public static class Cloak
{
  // returns the confidence level of the current visitor being a spider
  // (0 = no match, 2 = user agent matched, 3 = ip matched, 5 = ua + ip)
  public static SpiderMatch IsSpider()
  {
    return IsSpider("");
  }

  public static SpiderMatch IsSpider(string spiderName)
  {
    // default confidence level to 0
    SpiderMatch confidence = 0;

    // visitor's user agent
    string visitorUA = HttpContext.Current.Request["HTTP_USER_AGENT"];
    string visitorIP = HttpContext.Current.Request["REMOTE_ADDR"];

    // verify user agent and IP matches
    bool uaMatches = CloakDb.CheckUA(visitorUA, spiderName);
    bool ipMatches = CloakDb.CheckIP(visitorIP, spiderName);

    // obtain confidence level
    if (uaMatches && ipMatches) confidence = SpiderMatch.MatchUAandIP;
    else if (uaMatches) confidence = SpiderMatch.MatchUA;
    else if (ipMatches) confidence = SpiderMatch.MatchIP;
    else confidence = SpiderMatch.NoMatch;

    // return level of confidence that visitor is a spider
    return confidence;
  }

  // verify if the supplied IP is that of a spider
  public static bool IsSpiderIP(string ip)
  {
    return CloakDb.CheckIP(ip, "");
  }

  // verify if the supplied IP is that of the specified spider
  public static bool IsSpiderIP(string ip, string spiderName)
  {
    return CloakDb.CheckIP(ip, spiderName);
  }

  // verify if the supplied IP is that of a spider
  public static bool IsSpiderUA(string ua)
  {
    return CloakDb.CheckUA(ua, "");
  }

  // verify if the supplied user agent is that of the specified spider
  public static bool IsSpiderUA(string ua, string spiderName)
  {
    return CloakDb.CheckUA(ua, spiderName);
  }

  // updates cloak data; returns true if an update has been performed
  public static bool UpdateAll()
  {
    // read the version and the update date from the database
    string dbVersion = "";
    DateTime dbUpdatedOn = new DateTime();
    CloakDb.GetLastUpdate(ref dbVersion, ref dbUpdatedOn);

    // abort if the database was updated sooner than 7 days ago
    if (DateTime.Now - new TimeSpan(7, 0, 0, 0) < dbUpdatedOn) return false;

    // read the latest cloak version published by iplists
    WebRequest request = HttpWebRequest.Create(
      "http://www.iplists.com/nw/version.php");
    Stream responseStream = request.GetResponse().GetResponseStream();
    StreamReader reader = new StreamReader(responseStream);
    string latestVersion = reader.ReadLine();

    // abort if we alredy have the last version of the database
    if (dbVersion == latestVersion) return false;

    // update the database version and date
    CloakDb.SetLastUpdate(latestVersion, DateTime.Now);

    // update the database
    updateSpiderData("google", "http://www.iplists.com/nw/google.txt");
    updateSpiderData("yahoo", "http://www.iplists.com/nw/inktomi.txt");
    updateSpiderData("msn", "http://www.iplists.com/nw/msn.txt");
    updateSpiderData("ask", "http://www.iplists.com/nw/askjeeves.txt");
    updateSpiderData("altavista", "http://www.iplists.com/nw/altavista.txt");
    updateSpiderData("lycos", "http://www.iplists.com/nw/lycos.txt");
    updateSpiderData("wisenut", "http://www.iplists.com/nw/wisenut.txt");

    // return true to signal that we've updated the data
    return true;
  }

  private static void updateSpiderData(string spiderName, string spiderUrl)
  {
    // create the request object
    WebRequest request = HttpWebRequest.Create(spiderUrl);
    // create a stream to read data from the URL
    Stream responseStream = request.GetResponse().GetResponseStream();

    // load the list of IPs into a string
    StreamReader reader = new StreamReader(responseStream);
    string inputString = reader.ReadToEnd();
    reader.Close();

    // delete the database data we have for the spider
    CloakDb.DeleteSpider(spiderName);

    // use a regular expression to extract the user agents
    MatchCollection matches;
    matches = Regex.Matches(inputString,
      @"^# UA ""(?<url>.*)""$",
      RegexOptions.ExplicitCapture | RegexOptions.Multiline);

    // add each matched user agent to the database
    foreach (Match m in matches)
    {
      string ua = m.Groups["url"].ToString();
      CloakDb.AddSpiderRecord(spiderName, "UA", ua);
    }

    // regular expression that extract the spider IPs
    matches = Regex.Matches(inputString,
      "^(?<ip>[0-9.]+)$", RegexOptions.Multiline);

    // add the matched IPs to the database
    foreach (Match m in matches)
    {
      string ip = m.Groups["ip"].ToString();
      CloakDb.AddSpiderRecord(spiderName, "IP", ip);
    }
  }
}