using System;
using System.Data;
using System.Data.SqlClient;
using System.Configuration;

/// <summary>
/// Class represents the interface to the CloakData database
/// </summary>
public static class CloakDb
{
  // returns the date and the version of the last update
  public static bool GetLastUpdate(ref string version, ref DateTime updatedOn)
  {
    // create a SqlCommand object with a connection to the cloak database
    SqlCommand command = CloakDb.createCommand("CloakDatabase");

    // set the name of the stored procedure to execute
    command.CommandText = "CloakGetLastUpdate";

    // create a new DataTable object
    DataTable results = new DataTable();

    // execute the stored procecure and load the results into the DataTable
    command.Connection.Open();
    try { results.Load(command.ExecuteReader()); }
    finally { command.Connection.Close(); }

    // return false if no update record was found
    if (results.Rows.Count == 0)
    {
      return false;
    }
    else
    {
      version = results.Rows[0]["Version"].ToString();
      updatedOn = (DateTime)results.Rows[0]["UpdatedOn"];
      return true;
    }
  }

  // saves the time and date of the last database update
  public static void SetLastUpdate(string version, DateTime updatedOn)
  {
    // create a SqlCommand object with a connection to the cloak database
    SqlCommand command = CloakDb.createCommand("CloakDatabase");

    // set the name of the stored procedure to execute
    command.CommandText = "CloakSetLastUpdate";

    // add stored procedure parameter
    SqlParameter p = new SqlParameter("@NewVersion", version);
    command.Parameters.Add(p);

    // add stored procedure parameter
    p = new SqlParameter("@UpdatedOn", updatedOn);
    command.Parameters.Add(p);

    // execute the stored procecure and load the results into the DataTable
    command.Connection.Open();
    try { command.ExecuteNonQuery(); }
    finally { command.Connection.Close(); }
  }

  // saves the time and date of the last database update
  public static void AddSpiderRecord(string spiderName, string recordType, string value)
  {
    // create a SqlCommand object with a connection to the cloak database
    SqlCommand command = CloakDb.createCommand("CloakDatabase");

    // set the name of the stored procedure to execute
    command.CommandText = "CloakAddSpiderRecord";

    // @SpiderName
    SqlParameter p = new SqlParameter("@SpiderName", spiderName);
    command.Parameters.Add(p);

    // @RecordType
    p = new SqlParameter("@RecordType", recordType);
    command.Parameters.Add(p);

    // add stored procedure parameter
    p = new SqlParameter("@Value", value);
    command.Parameters.Add(p);

    // execute the stored procecure and load the results into the DataTable
    command.Connection.Open();
    try { command.ExecuteNonQuery(); }
    finally { command.Connection.Close(); }
  }

  // deletes the records of a spider from the database
  public static void DeleteSpider(string spiderName)
  {
    // create a SqlCommand object with a connection to the cloak database
    SqlCommand command = CloakDb.createCommand("CloakDatabase");

    // set the name of the stored procedure to execute
    command.CommandText = "CloakDeleteSpider";

    // @SpiderName
    SqlParameter p = new SqlParameter("@SpiderName", spiderName);
    command.Parameters.Add(p);

    // execute the stored procecure and load the results into the DataTable
    command.Connection.Open();
    try { command.ExecuteNonQuery(); }
    finally { command.Connection.Close(); }
  }

  // verify if the supplied User Agent is that of the specified spider
  // (if spiderName is an empty string, all spiders are checked)
  public static bool CheckUA(string uaValue, string spiderName)
  {
    // create a SqlCommand object with a connection to the cloak database
    SqlCommand command = CloakDb.createCommand("CloakDatabase");

    // set the name of the stored procedure to execute
    command.CommandText = "CloakCheckUA";

    // @UAValue
    SqlParameter p = new SqlParameter("@UAValue", uaValue);
    command.Parameters.Add(p);

    // @SpiderName
    p = new SqlParameter("@SpiderName", spiderName);
    command.Parameters.Add(p);

    // execute the stored procedure
    command.Connection.Open();
    int numberOfMatches;
    try { numberOfMatches = (int)command.ExecuteScalar(); }
    finally { command.Connection.Close(); }

    // return true if there was a match
    return (numberOfMatches > 0);
  }

  // verify if the supplied IP is that of the specified spider
  // (if spiderName is an empty string, all spiders are checked)
  public static bool CheckIP(string ipValue, string spiderName)
  {
    // create a SqlCommand object with a connection to the cloak database
    SqlCommand command = CloakDb.createCommand("CloakDatabase");

    // set the name of the stored procedure to execute
    command.CommandText = "CloakCheckIP";

    // @UAValue
    SqlParameter p = new SqlParameter("@IPValue", ipValue);
    command.Parameters.Add(p);

    // @SpiderName
    p = new SqlParameter("@SpiderName", spiderName);
    command.Parameters.Add(p);

    // execute the stored procedure
    command.Connection.Open();
    int numberOfMatches;
    try { numberOfMatches = (int)command.ExecuteScalar(); }
    finally { command.Connection.Close(); }

    // return true if there was a match
    return (numberOfMatches > 0);
  }

  // creates and prepares a new SqlCommand object on a new connection
  private static SqlCommand createCommand(string connectionName)
  {
    // obtain the database connection string
    string connectionString = ConfigurationManager.ConnectionStrings[connectionName].ConnectionString;

    // obtain a database specific connection object
    SqlConnection conn = new SqlConnection(connectionString);

    // create a database specific command object
    SqlCommand comm = conn.CreateCommand();

    // set the command type to stored procedure
    comm.CommandType = CommandType.StoredProcedure;

    // return the initialized command object
    return comm;
  }
}