using System;
using System.Web;
using System.Data;
using System.Configuration;

/// <summary>
/// Class provides support for URL manipulation and redirection
/// </summary>
public static class UrlTools
{
  // obtain the site domain from the configuration file
  public readonly static string SiteDomain = ConfigurationManager.AppSettings["SiteDomain"];

  /* ensures the current page is being loaded through its standard URL;
   * 301 redirect to the standard URL if it doesn't */
  public static void CheckUrl()
  {
    HttpContext context = HttpContext.Current;
    HttpRequest request = HttpContext.Current.Request;

    // retrieve query string parameters
    string productId = request.QueryString["ProductID"];
    string categoryId = request.QueryString["CategoryID"];
    string page = request.QueryString["Page"];

    // check category-product URLs
    if (categoryId != null && productId != null)
    {
      CheckCategoryProductUrl(categoryId, productId);
    }
    // check category URLs
    else if (categoryId != null && page != null)
    {
      CheckCategoryUrl(categoryId, page);
    }
  }

  // checks a category URL for compliancy
  // 301 redirects to proper URL, or returns 404 if necessary
  public static void CheckCategoryUrl(string categoryId, string page)
  {
    // the current HttpContext
    HttpContext context = HttpContext.Current;

    // the URL requested by the visitor
    string requestedUrl = context.Request.ServerVariables["HTTP_X_REWRITE_URL"];

    // retrieve category data 
    DataRow categoryRow = CatalogDb.GetCategory(categoryId);

    // if the category or the product doesn't exist in the database, return 404
    if (categoryRow == null)
    {
      NotFound();
    }

    // get category name
    string categoryName = categoryRow["Name"].ToString();

    // obtain the standard version of the URL
    string standardUrl = LinkFactory.MakeCategoryUrl(categoryName, categoryId, int.Parse(page));

    // 301 redirect to the proper URL if necessary
    if (SiteDomain + requestedUrl != standardUrl)
    {
      context.Response.Status = "301 Moved Permanently";
      context.Response.AddHeader("Location", standardUrl);
    }
  }

  // checks a category-product URL for compliancy
  // 301 redirects to proper URL, or returns 404 if necessary
  public static void CheckCategoryProductUrl(string categoryId, string productId)
  {
    // the current HttpContext
    HttpContext context = HttpContext.Current;

    // the URL requested by the visitor
    string requestedUrl = context.Request.ServerVariables["HTTP_X_REWRITE_URL"];

    // retrieve product and category data
    DataRow categoryRow = CatalogDb.GetCategory(categoryId);
    DataRow productRow = CatalogDb.GetProduct(productId);

    // if the category or the product doesn't exist in the database, return 404
    if (categoryRow == null || productRow == null)
    {
      NotFound();
    }

    // get product and category names
    string categoryName = categoryRow["Name"].ToString();
    string productName = productRow["Name"].ToString();

    // obtain the standard version of the URL
    string standardUrl = LinkFactory.MakeCategoryProductUrl(categoryName, categoryId, productName, productId);

    // 301 redirect to the proper URL if necessary
    if (SiteDomain + requestedUrl != standardUrl)
    {
      context.Response.Status = "301 Moved Permanently";
      context.Response.AddHeader("Location", standardUrl);
    }
  }

  // Load the 404 page
  public static void NotFound()
  {
    HttpContext.Current.Server.Transfer("~/NotFound.aspx");
  }
}