﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Ink;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using System.Text.RegularExpressions;

namespace GDev.PopupBox
{
    public partial class MessagePage : UserControl, IPopupBox
    {
        #region 属性

        public string Title
        {
            get
            {
                return TitleText.Text;
            }
            set
            {
                TitleText.Text = value;
            }
        }

        public string Message
        {
            get
            {
                return MessageText.Text;
            }
            set
            {
                MessageText.Text = value;
            }
        }

        public MessageBoxIcon Icon
        {
            get;
            set;
        }

        public MessageBoxButtonType ButtonType
        {
            get;
            set;
        }

        public MessageBoxButtonResult Result
        {
            get;
            private set;
        }

        public Border Border
        {
            get
            {
                return LayoutRoot;
            }
        }

        public Panel TitleElement
        {
            get
            {
                return TitlePanel;
            }
        }

        public Panel ContentElement
        {
            get
            {
                return ContentPanel;
            }
        }

        public Panel ButtonElement
        {
            get
            {
                return ButtonPanel;
            }
        }

        #endregion

        #region 事件

        public event EventHandler ButtonClick;

        protected virtual void OnButtonClick(EventArgs e)
        {
            EventHandler handler = ButtonClick;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        public event EventHandler ShowComplete;

        protected virtual void OnShowComplete(EventArgs e)
        {
            EventHandler handler = ShowComplete;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        #endregion

        public MessagePage()
        {
            // 需要初始化变量
            InitializeComponent();
        }

        #region 辅助方法

        private void BuildButton()
        {
            string typeEnum = ButtonType.ToString();
            typeEnum = Regex.Replace(
                typeEnum,
                "[A-Z]",
                match => ":" + match.Value
            );
            string[] types = typeEnum.Split(new char[] { ':' }, StringSplitOptions.RemoveEmptyEntries);
            foreach (string type in types)
            {
                AddButton(type);
            }
        }

        private void AddButton(string text)
        {
            Button button = new Button();
            button.Content = text;
            button.Width = 60;
            button.HorizontalAlignment = HorizontalAlignment.Center;
            button.Margin = new Thickness(5, 3, 5, 3);
            button.Click += new RoutedEventHandler(Button_Click);
            ButtonPanel.Children.Add(button);
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            Button button = sender as Button;
            if (button != null)
            {
                Result = (MessageBoxButtonResult)Enum.Parse(
                    typeof(MessageBoxButtonResult),
                    (string)button.Content,
                    true
                );
                Close();
            }
        }

        private void BuildContent(FrameworkElement icon)
        {
            if (icon != null)
            {
                icon.Width = 64;
                icon.Height = 64;
                icon.HorizontalAlignment = HorizontalAlignment.Left;
                icon.VerticalAlignment = VerticalAlignment.Top;
                icon.Margin = new Thickness(5, 10, 5, 10);
                ContentPanel.Children.Insert(0, icon);
            }
        }

        private void BuildTitle(FrameworkElement icon)
        {
            if (icon != null)
            {
                icon.Width = 18;
                icon.Height = 18;
                icon.HorizontalAlignment = HorizontalAlignment.Left;
                icon.VerticalAlignment = VerticalAlignment.Center;
                icon.Margin = new Thickness(2, 0, 0, 0);
                TitlePanel.Children.Insert(0, icon);
            }
            TitleText.Text = Title;
        }

        #endregion

        #region IPopupBox 成员

        public LayoutMask Mask
        {
            get;
            set;
        }

        public UIElement DragMouseCaptureArea
        {
            get
            {
                return TitlePanel;
            }
        }

        public FrameworkElement Element
        {
            get
            {
                return this;
            }
        }

        public Point LastDragPosition
        {
            get;
            set;
        }

        public bool IsDraggable
        {
            get;
            set;
        }

        public bool IsDragging
        {
            get;
            set;
        }

        public Effect Effect
        {
            get;
            set;
        }

        public bool IsModal
        {
            get;
            private set;
        }

        public void Show()
        {
            //设定ZIndex
            Canvas.SetZIndex(this, Mask.MaxZIndex + 1);

            //先将控件隐藏以显示动画效果
            Visibility = Visibility.Collapsed;

            Mask.AddBox(this);
        }

        public void ShowAsModal()
        {
            IsModal = true;
            Show();
        }

        public void Close()
        {
            //进行退出动画
            Effect.Complete += new EventHandler(Effect_CompleteOnOut);
            Effect.PerformOutEffect();
        }

        #endregion

        #region 事件方法

        private void UserControl_Loaded(object sender, RoutedEventArgs e)
        {
            FrameworkElement titleIcon = null;
            FrameworkElement contentIcon = null;

            if (Icon != null)
            {
                titleIcon = Icon.GetTitleIcon();
                contentIcon = Icon.GetContentIcon();
            }
            BuildTitle(titleIcon);
            BuildContent(contentIcon);
            BuildButton();

            Effect.Complete += new EventHandler(Effect_CompleteOnShow);
            Visibility = Visibility.Visible;
            UpdateLayout();
            //居中
            Mask.CenterToMask(this);
            Effect.PerformInEffect();
        }

        private void UserControl_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            Canvas.SetZIndex(this, Mask.MaxZIndex + 1);
        }

        private void Effect_CompleteOnShow(object sender, EventArgs e)
        {
            Effect.Complete -= Effect_CompleteOnShow;

            OnShowComplete(EventArgs.Empty);
        }

        private void Effect_CompleteOnOut(object sender, EventArgs e)
        {
            //有可能由于多次关闭导致此事件执行多次
            //这种情况下第一次执行是有效的
            //以后的事件会因为Mask.RemoveBox方法的调用使Mask为null
            //因此需要进行判断
            if (Mask != null)
            {
                //从Mask中移除
                Mask.RemoveBox(this);

                Effect.Complete -= Effect_CompleteOnOut;

                OnButtonClick(EventArgs.Empty);
            }
        }

        #endregion
    }
}