﻿using System;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Ink;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using System.Collections;
using System.Collections.Generic;

namespace GDev.PopupBox
{
    public class PopupService
    {
        #region 缓存管理

        private static readonly IDictionary<Panel, PopupService>
            m_Cache = new Dictionary<Panel, PopupService>();

        protected static IDictionary<Panel, PopupService> Cache
        {
            get
            {
                return m_Cache;
            }
        }

        #endregion

        #region 属性

        private readonly Panel m_Owner;

        protected virtual Panel Owner
        {
            get
            {
                return m_Owner;
            }
        }

        private readonly LayoutMask m_Mask;

        protected virtual LayoutMask LayoutMask
        {
            get
            {
                return m_Mask;
            }
        }

        #endregion

        #region 静态实例生产

        private PopupService(Panel owner)
        {
            m_Owner = owner;
            m_Mask = new LayoutMask(owner);
        }

        public static PopupService GetServiceFor(Panel owner)
        {
            if (!Cache.ContainsKey(owner))
            {
                PopupService service = new PopupService(owner);
                Cache[owner] = service;
            }
            return Cache[owner];
        }

        #endregion

        #region 方法

        public BoxPage GetBoxPage(FrameworkElement content, string title, bool draggable, bool showCloseBox, ImageSource closeIcon)
        {
            BoxPage box = new BoxPage()
            {
                ContentElement = content,
                Title = title,
                IsDraggable = draggable,
                ShowCloseBox = showCloseBox,
                CloseIcon = closeIcon,
            };

            RegisterPopupBox(box);

            box.ShowComplete += new EventHandler(Box_ShowComplete);
            return box;
        }

        public BoxPage GetBoxPage(FrameworkElement content, string title, bool draggable, bool showCloseBox)
        {
            return GetBoxPage(content, title, draggable, showCloseBox, null);
        }

        public BoxPage GetBoxPage(FrameworkElement content, string title, bool draggable)
        {
            return GetBoxPage(content, title, draggable, true, null);
        }

        public BoxPage GetBoxPage(FrameworkElement content, string title)
        {
            return GetBoxPage(content, title, true, true, null);
        }

        public BoxPage GetBoxPage(FrameworkElement content)
        {
            return GetBoxPage(content, String.Empty, true, true, null);
        }

        public MessagePage GetMessagePage(string message, string title, bool draggable, MessageBoxButtonType buttonType, MessageBoxIcon icon)
        {
            MessagePage box = new MessagePage()
            {
                Message = message,
                Title = title,
                ButtonType = buttonType,
                Icon = icon,
                IsDraggable = draggable
            };

            RegisterPopupBox(box);

            box.ShowComplete += new EventHandler(Box_ShowComplete);
            return box;
        }

        public MessagePage GetMessagePage(string message, string title, bool draggable, MessageBoxButtonType buttonType)
        {
            return GetMessagePage(message, title, draggable, buttonType, null);
        }

        public MessagePage GetMessagePage(string message, string title, bool draggable)
        {
            return GetMessagePage(message, title, draggable, MessageBoxButtonType.YesNo, null);
        }

        public MessagePage GetMessagePage(string message, string title)
        {
            return GetMessagePage(message, title, true, MessageBoxButtonType.YesNo, null);
        }

        public MessagePage GetMessagePage(string message)
        {
            return GetMessagePage(message, String.Empty, true, MessageBoxButtonType.YesNo, null);
        }

        public void RegisterPopupBox(IPopupBox box)
        {
            box.Mask = LayoutMask;
            if (box.Effect == null)
            {
                box.Effect = Effect.NoEffect(box.Element);
            }
        }

        #endregion

        #region 辅助方法

        private void Box_ShowComplete(object sender, EventArgs e)
        {
            IPopupBox box = sender as IPopupBox;
            if (box != null)
            {
                if (box.IsDraggable)
                {
                    DragService dragService = new DragService(box);
                    dragService.EnableDrag();
                }
            }
        }

        #endregion
    }
}
