﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;

namespace WebMvc.ViewData
{
    public interface IPagedList
    {
        int RowCount
        {
            get;
            set;
        }

        int CurrentPage
        {
            get;
            set;
        }
        /// <summary>
        /// 上一页
        /// </summary>
        int PreviousPage
        {
            get;
            set;
        }
        /// <summary>
        /// 下一页
        /// </summary>
        int NextPage
        {
            get;
            set;
        }

        int PageSize
        {
            get;
            set;
        }
        int TotalPages
        {
            get;
        }
        bool HasPreviousPage
        {
            get;
        }

        bool HasNextPage
        {
            get;
        }
    }

    public class PagedList<T> : List<T>, IPagedList
    {
        public PagedList(IQueryable<T> source, int index, int pageSize)
        {
            this.RowCount = source.Count();
            this.PageSize = pageSize;
            this.CurrentPage = index;
            this.AddRange(source.Skip(index * pageSize).Take(pageSize).ToList());
        }

        public PagedList(List<T> source, int index, int pageSize)
        {
            this.RowCount = source.Count();
            this.PageSize = pageSize;
            this.CurrentPage = index;
            this.AddRange(source.Skip(index * pageSize).Take(pageSize).ToList());
        }

        public int RowCount
        {
            get;
            set;
        }

        public int CurrentPage
        {
            get;
            set;
        }

        public int PreviousPage
        {
            get;
            set;
        }

        public int NextPage
        {
            get;
            set;
        }
        
        public int PageSize
        {
            get;
            set;
        }

        public bool HasPreviousPage
        {
            get
            {
                return (CurrentPage > 0);
            }
        }

        public bool HasNextPage
        {
            get
            {
                return ((CurrentPage + 1) * PageSize) <= RowCount;
                //return (TotalPages - CurrentPage) > 0;
            }
        }

        public int TotalPages
        {
            get
            {
                if (PageSize == 0)
                    throw new ArgumentOutOfRangeException("page");
                int remainder = RowCount % PageSize;
                if (remainder == 0)
                    return RowCount / PageSize;
                else
                    return (RowCount / PageSize) + 1;
            }

        }
                
    }

    public static class Pagination
    {
        public static PagedList<T> ToPagedList<T>(this IQueryable<T> source, int index, int pageSize)
        {
            return new PagedList<T>(source, index, pageSize);
        }

        public static PagedList<T> ToPagedList<T>(this IQueryable<T> source, int index)
        {
            return new PagedList<T>(source, index, 10);
        }

        public static PagedList<T> ToPagedList<T>(this List<T> source, int index, int pageSize)
        {
            return new PagedList<T>(source, index, pageSize);
        }

        public static PagedList<T> ToPagedList<T>(this List<T> source, int index)
        {
            return new PagedList<T>(source, index, 10);
        }
    }

}
