//===--- StringNormalization.swift ------------------------------------------------===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2018 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//
// RUN: %empty-directory(%t)
// RUN: if [ %target-runtime == "objc" ]; \
// RUN: then \
// RUN:   %target-clang -fobjc-arc %S/Inputs/NSSlowString/NSSlowString.m -c -o %t/NSSlowString.o && \
// RUN:   %target-build-swift -I %S/Inputs/NSSlowString/ %t/NSSlowString.o %s -o %t/a.out; \
// RUN: else \
// RUN:   %target-build-swift %s -o %t/a.out; \
// RUN: fi

// RUN: %target-codesign %t/a.out
// RUN: %target-run %t/a.out %S/Inputs/NormalizationTest.txt
// REQUIRES: executable_test
// REQUIRES: objc_interop
// REQUIRES: optimized_stdlib

import Swift
import StdlibUnittest
import StdlibUnicodeUnittest

#if _runtime(_ObjC)
import NSSlowString
#endif

private func expectEqualIterators(
  label: String,
  expected: [UInt8],
  others: [String: [UInt8]],
  _ message: @autoclosure () -> String = "",
  showFrame: Bool = true,
  stackTrace: SourceLocStack = SourceLocStack(),
  file: String = #file,
  line: UInt = #line
) {
  let expectedString = String(decoding: expected, as: UTF8.self)
  let expectedCodeUnits = expectedString._nfcCodeUnits
  for (otherLabel, other) in others {
    let otherString = String(decoding: other, as: UTF8.self)
    expectEqual(
      expectedCodeUnits,
      otherString._nfcCodeUnits,
      "\(label) vs \(otherLabel)",
      stackTrace: stackTrace.pushIf(showFrame, file: file, line: line))
  }
}

var tests = TestSuite("StringNormalization")

tests.test("StringNormalization/ConvertToNFC")
.skip(.custom({
      if #available(macOS 10.14, iOS 12, watchOS 5, tvOS 12, *) { return false }
      return true
    }, reason: "NormalizationTest.txt requires Unicode 11"))
.code {
  for test in normalizationTests {
    expectEqualIterators(
      label: "NFC",
      expected: test.NFC,
      others: [
        "source": test.source,
        "NFC": test.NFC,
        "NFD": test.NFD
      ],
      stackTrace: SourceLocStack(test.loc))
  }
}

tests.test("StringNormalization/ConvertNFK*ToNFKC")
.skip(.custom({
      if #available(macOS 10.14, iOS 12, watchOS 5, tvOS 12, *) { return false }
      return true
    }, reason: "NormalizationTest.txt requires Unicode 11"))
.code {
  for test in normalizationTests {
    expectEqualIterators(
      label: "NFKC",
      expected: test.NFKC,
      others: [
        "NFKC": test.NFKC,
        "NFKD": test.NFKD
      ],
      stackTrace: SourceLocStack(test.loc))
  }
}

let codeUnitNormalizationTestsA: [([UInt8], String)] = [
  (
    [0xef, 0xbd, 0x9a, 0xec, 0xa2, 0x85, 0xe1, 0xbf, 0x90, 0xef, 0xb9, 0xb4, 0xeb, 0x9d, 0x9d, 0xec, 0xa6, 0x81, 0xea, 0xba, 0x91, 0xf0, 0x9d, 0x9b, 0xbb, 0xed, 0x98, 0x9d, 0xe0, 0xb5, 0x8b],
    "ｚ종ῐﹴ띝즁꺑𝛻혝ോ"
  ),
  (
    [0xec, 0xab, 0xa7, 0xc3, 0xa0, 0xd6, 0xae, 0xd8, 0x97, 0xcc, 0x95, 0x62, 0xea, 0xb2, 0xa4, 0xec, 0xa0, 0x91, 0xe8, 0xa3, 0x97, 0xeb, 0xa4, 0xbd, 0xe2, 0x85, 0xaf, 0xed, 0x9b, 0xb7, 0xec, 0xba, 0xa0, 0xeb, 0xb8, 0xa4],
    "쫧à֮ؗ̕b겤접裗뤽Ⅿ훷캠븤"
  ),
  (
    [0x61, 0xcc, 0x96, 0xd6, 0xad, 0xd6, 0x9a, 0xe3, 0x80, 0xae, 0x62, 0xd8, 0xb3, 0xd9, 0x8a, 0xeb, 0xb1, 0x90, 0xeb, 0x91, 0xa8, 0xea, 0xb0, 0xa7, 0xeb, 0x89, 0xb3, 0xd9, 0x86, 0xd9, 0x85, 0xd9, 0x8a, 0xec, 0x80, 0xb7, 0x61, 0xd6, 0xae, 0xdb, 0xa8, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd6, 0xa7, 0xd6, 0x9a, 0x62],
    "a̖֭֚〮bسي뱐둨갧뉳نمي쀷a֮ۨ̀̕ba〪̖֧֚b"
  ),
  (
    [0xed, 0x81, 0x92, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xb3, 0x98, 0xd6, 0x9a, 0x62, 0xc3, 0xa7, 0xe5, 0x8f, 0x83, 0xed, 0x87, 0x81, 0xec, 0x90, 0xa7, 0xec, 0x95, 0xb2, 0xeb, 0xb8, 0x83, 0xec, 0x9d, 0xb4, 0xed, 0x97, 0x8f],
    "큒a〪̖᳘֚bç參퇁쐧앲븃이헏"
  ),
  (
    [0xeb, 0xaa, 0x8c, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xeb, 0x8b, 0xae, 0xeb, 0x8f, 0x88, 0x61, 0xd6, 0xae, 0xf0, 0x96, 0xac, 0xb4, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x90, 0x86, 0xec, 0x84, 0xa8, 0xe2, 0xbd, 0xb1, 0xec, 0x80, 0xaa, 0xef, 0xba, 0x8f, 0xed, 0x83, 0x85],
    "몌̴ᆮ닮돈a֮𖬴̀̕b됆섨⽱쀪ﺏ탅"
  ),
  (
    [0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xb3, 0xcc, 0x95, 0x62, 0xe1, 0xb6, 0xb6, 0xec, 0x93, 0xa1, 0xed, 0x83, 0x97, 0xec, 0xa4, 0x80, 0xec, 0xa0, 0x95, 0xef, 0xbb, 0xaa, 0xec, 0x82, 0x90, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xed, 0x84, 0xa3, 0xed, 0x8b, 0xad, 0xeb, 0x9c, 0xaf, 0xef, 0xb6, 0x8e, 0xe3, 0x8e, 0xa1, 0xef, 0xb6, 0x94, 0xe3, 0x8e, 0x9a, 0xe2, 0x85, 0xb3, 0xef, 0xb0, 0x92, 0xeb, 0x83, 0xa8, 0xed, 0x9d, 0x94, 0xf0, 0x9f, 0x88, 0xb9],
    "à֮𑍳̕bᶶ쓡탗준정ﻪ삐̴ᆵ턣틭뜯ﶎ㎡ﶔ㎚ⅳﰒ냨흔🈹"
  ),
  (
    [0xeb, 0xb3, 0x95, 0xf0, 0xa6, 0xbc, 0xac, 0xef, 0xbc, 0xba, 0xce, 0xa9, 0xec, 0xb2, 0x84, 0xe7, 0xb3, 0x96, 0xed, 0x91, 0xa0, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0xb7, 0xb9, 0xcc, 0x95, 0x62, 0xec, 0xbc, 0x98, 0xe5, 0x8b, 0x9e, 0xec, 0x82, 0x9a, 0xed, 0x88, 0x92, 0xed, 0x87, 0x91, 0xec, 0xaf, 0x99, 0xe6, 0x86, 0xaf, 0xe2, 0xb5, 0xaf, 0xec, 0x9d, 0x8c, 0xe9, 0xb6, 0xb4, 0xe2, 0x93, 0xa2, 0xe2, 0x82, 0x84],
    "볕𦼬ＺΩ첄糖푠à֮ⷹ̕b켘勞삚툒퇑쯙憯ⵯ음鶴ⓢ₄"
  ),
  (
    [0xed, 0x95, 0x8c, 0xec, 0xb7, 0xb6, 0xec, 0xbf, 0x90, 0xed, 0x97, 0xb6, 0x61, 0xcd, 0x9c, 0xcd, 0x9d, 0xe1, 0xb7, 0x8d, 0xcd, 0x85, 0x62, 0xeb, 0xa5, 0x98, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0xa5, 0x89, 0xcc, 0x95, 0x62, 0xec, 0xa4, 0xa9, 0xe5, 0x9c, 0xb0, 0xed, 0x91, 0xb6, 0xeb, 0xa7, 0x94, 0xd9, 0x88, 0xd8, 0xb3, 0xd9, 0x84, 0xd9, 0x85, 0x61, 0xd6, 0xae, 0xf0, 0x9d, 0x86, 0xad, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xad, 0x8d, 0xeb, 0x88, 0x89, 0xed, 0x9b, 0xb2, 0x32, 0x35, 0xed, 0x95, 0xb8, 0xea, 0xbd, 0xa3, 0xec, 0x91, 0x86],
    "핌췶쿐헶a͜͝᷍ͅb류à֮𞥉̕b줩地푶맔وسلمa֮𝆭̀̕b쭍눉훲25핸꽣쑆"
  ),
  (
    [0xec, 0x8c, 0x83, 0xe0, 0xbe, 0xb2, 0xe0, 0xbe, 0x80, 0xeb, 0x9e, 0x8a, 0xeb, 0xad, 0xa4, 0xed, 0x8d, 0x88, 0xed, 0x98, 0x9e, 0xe9, 0xbc, 0xbb, 0xed, 0x91, 0xa6, 0xd8, 0xae, 0xd9, 0x8a, 0xec, 0xb4, 0xa1, 0xeb, 0x89, 0x93, 0xeb, 0x93, 0xbf, 0xec, 0xa1, 0xae, 0xeb, 0xab, 0x95, 0xec, 0xba, 0xa4, 0xed, 0x9c, 0xb4, 0x34, 0x31, 0xeb, 0xb9, 0xab, 0x72, 0x61, 0x64, 0xe2, 0x88, 0x95, 0x73, 0x32, 0xed, 0x81, 0xbf],
    "쌃ྲྀ랊뭤퍈혞鼻푦خي촡뉓듿졮뫕캤휴41빫rad∕s2큿"
  ),
  (
    [0xec, 0x81, 0x8f, 0xe9, 0x9f, 0xb3, 0x4d, 0x57, 0xec, 0x82, 0xa2, 0xec, 0x8d, 0x8d, 0xea, 0xb3, 0x9b, 0xeb, 0xbb, 0x8f, 0xec, 0xa9, 0x84, 0xeb, 0xbb, 0x8d, 0xea, 0xb2, 0x94, 0xe1, 0x84, 0x87, 0xcc, 0xb4, 0xe1, 0x85, 0xaa, 0xed, 0x82, 0xbe, 0x61, 0xe3, 0x80, 0xaa, 0xe1, 0xb3, 0x9f, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xec, 0xa3, 0x86, 0xed, 0x89, 0x86, 0xe1, 0xb9, 0x90, 0xef, 0xb0, 0x86, 0xeb, 0x92, 0xb3, 0xed, 0x82, 0x9f, 0xec, 0xa0, 0x80, 0xcc, 0xb4, 0xe1, 0x86, 0xbc],
    "쁏音MW삢썍곛뻏쩄뻍겔ᄇ̴ᅪ킾a〪᳟̖֚b죆퉆Ṑﰆ뒳킟저̴ᆼ"
  ),
  (
    [0xec, 0x8c, 0x9d, 0xe6, 0x86, 0x8e, 0xec, 0xb0, 0x9a, 0xec, 0xba, 0xb0, 0xe1, 0xb6, 0xb0, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xaa, 0xb4, 0xcc, 0x95, 0x62, 0xec, 0xac, 0x9e, 0xec, 0xa7, 0xb9, 0xea, 0xbe, 0x89, 0xef, 0xad, 0xab, 0xd7, 0x9c, 0xd6, 0xbc, 0xe2, 0x85, 0x85, 0xeb, 0x94, 0x93, 0xe1, 0xbd, 0x92, 0xec, 0xaa, 0xa3, 0xeb, 0xab, 0x85, 0xec, 0xb7, 0x8c, 0xea, 0xb9, 0xa9, 0xec, 0xab, 0x8d, 0xec, 0x8c, 0x84, 0xe5, 0x8f, 0xa5, 0xeb, 0x89, 0x85, 0xeb, 0x8c, 0xa6, 0xed, 0x91, 0xa4, 0xeb, 0x9b, 0x95, 0xe3, 0x89, 0xa5, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x9d, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe4, 0x84, 0xaf, 0xc8, 0x9f, 0xec, 0xba, 0x9b],
    "쌝憎찚캰ᶰà֮᪴̕b쬞짹꾉ﭫלּⅅ딓ὒ쪣뫅췌깩쫍쌄句뉅댦푤뛕㉥a֮𞀝̀̕b䄯ȟ캛"
  ),
  (
    [0xf0, 0x9d, 0x98, 0x92, 0xeb, 0xa7, 0x85, 0xec, 0xaa, 0xaf, 0xe3, 0x85, 0x89, 0xec, 0x90, 0xbc, 0xea, 0xb3, 0xbb, 0x61, 0xcc, 0xb4, 0xe1, 0xb0, 0xb7, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xe6, 0x85, 0x8c, 0xeb, 0xac, 0xac, 0xeb, 0x8a, 0xb8, 0xec, 0x80, 0xbd, 0xec, 0x96, 0xa0, 0xea, 0xb2, 0xa7, 0xf0, 0x9e, 0xb8, 0x9c, 0xeb, 0xb6, 0xb5, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0xb7, 0xb2, 0xcc, 0x95, 0x62, 0xeb, 0xb2, 0x84, 0xec, 0xaa, 0xb3, 0xec, 0x9d, 0x94, 0xe4, 0xaa, 0xb2, 0xf0, 0x9d, 0x86, 0xb9, 0xf0, 0x9d, 0x85, 0xa5, 0xf0, 0x9d, 0x85, 0xaf, 0xed, 0x82, 0xbe, 0xeb, 0x86, 0x94, 0xeb, 0x93, 0xa1, 0xeb, 0xbc, 0x94, 0xeb, 0x9b, 0xbc, 0xec, 0xa8, 0x80, 0xeb, 0x90, 0xbf, 0xef, 0xba, 0xb3, 0x61, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xb4, 0xcc, 0x80, 0xcc, 0x95, 0x62],
    "𝘒맅쪯ㅉ쐼곻a̴᰷़゙b慌묬늸쀽얠겧𞸜붵à֮ⷲ̕b버쪳읔䪲𝆹𝅥𝅯킾놔듡뼔뛼쨀됿ﺳa֮𑍴̀̕b"
  ),
  (
    [0xea, 0xb3, 0xb5, 0xea, 0xbb, 0x86, 0xe1, 0xba, 0x86, 0xeb, 0xab, 0xac, 0xe6, 0xad, 0xb7, 0xec, 0x99, 0x81, 0xed, 0x8d, 0xa8, 0xed, 0x8a, 0xb0, 0xed, 0x8b, 0xa5, 0xed, 0x84, 0xa8, 0xed, 0x89, 0x9e, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x9d, 0x85, 0xbf, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xed, 0x90, 0x8a, 0x20, 0xcc, 0x87, 0xeb, 0xb8, 0x9f, 0xec, 0x92, 0xa7, 0xe1, 0xbb, 0x8c, 0xec, 0x8f, 0xa7, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0xa7, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xd6, 0xae, 0xe1, 0xb3, 0x90, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xbe, 0xac, 0xec, 0xbb, 0x8a, 0xeb, 0x86, 0xb5, 0x61, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0xe3, 0x82, 0x9a, 0xe0, 0xa5, 0x8d, 0x62, 0xec, 0xaf, 0xaf, 0x50, 0x52, 0xeb, 0x82, 0xa9, 0xe1, 0xba, 0xb1, 0xd6, 0xae, 0xcc, 0x95, 0x62, 0xeb, 0xb0, 0xae, 0xe1, 0xbc, 0xb7],
    "공껆Ẇ뫬歷왁퍨튰틥턨퉞a〪𝅿̖֚b퐊 ̇븟쒧Ọ쏧a֮ࣧ̀̕ba֮᳐̀̕b쾬컊놵a़゙゚्b쯯PR납ằ֮̕b밮ἷ"
  ),
  (
    [0xea, 0xb7, 0x9c, 0xed, 0x90, 0xae, 0xe6, 0xba, 0xba, 0x61, 0xd6, 0xae, 0xdb, 0x9b, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xbc, 0x8e, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0x98, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x80, 0xae, 0xed, 0x93, 0x8e, 0xeb, 0xba, 0xab, 0xeb, 0x8a, 0x98, 0xeb, 0xb5, 0x9c, 0xed, 0x81, 0x82, 0xed, 0x94, 0x9e, 0xd9, 0x86, 0xd9, 0x86, 0xeb, 0xac, 0xa0, 0xec, 0x88, 0xa2, 0xeb, 0xb0, 0x99, 0xec, 0xbe, 0x9f, 0xe1, 0xbb, 0xb0, 0xeb, 0x97, 0x88, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x96, 0xac, 0xb0, 0xcc, 0x95, 0x62, 0xed, 0x9d, 0xa4, 0xeb, 0x86, 0x87, 0xec, 0xa6, 0x85, 0xe8, 0x99, 0xa7, 0x61, 0xcc, 0xb4, 0xf0, 0x91, 0x87, 0x8a, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0x28, 0x31, 0x34, 0x29, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x8d, 0xcc, 0x95, 0x62, 0xe9, 0xba, 0x9f, 0xed, 0x99, 0x90],
    "규퐮溺a֮ۛ̀̕b뼎a֮ࣘ̀̕b뀮퓎뺫늘뵜큂픞نن묠숢밙쾟Ự뗈à֮𖬰̕b흤놇즅虧a̴𑇊़゙b(14)à֮̍̕b麟홐"
  ),
  (
    [0xec, 0x95, 0xad, 0xeb, 0xb7, 0x99, 0xeb, 0x94, 0xa1, 0xea, 0xb4, 0x92, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0xa8, 0xcc, 0x95, 0x62, 0xea, 0xb5, 0x92, 0xf0, 0x9d, 0x90, 0x83, 0xeb, 0xab, 0x8e, 0xeb, 0xb9, 0xa5, 0xed, 0x8d, 0xa5, 0xec, 0x8e, 0xb6, 0xc3, 0xa0, 0xd6, 0xae, 0xcd, 0xa6, 0xcc, 0x95, 0x62, 0xeb, 0xb9, 0x87, 0xe2, 0x88, 0xb0, 0xe8, 0xa6, 0x96, 0x61, 0xe3, 0x80, 0xaa, 0xe1, 0xa8, 0x98, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xec, 0xb3, 0x9d, 0xeb, 0x91, 0xa6, 0x62, 0x61, 0x72, 0xec, 0x9c, 0x95, 0xe8, 0x9e, 0x86, 0xeb, 0x8e, 0xbd, 0xf0, 0x9d, 0x9f, 0xa3, 0x6d, 0x32, 0xeb, 0x8c, 0x98, 0xec, 0x86, 0x84, 0xe5, 0x8f, 0xb3, 0xe2, 0xbc, 0x9c, 0xeb, 0x9f, 0xa8, 0xec, 0xb7, 0xa4],
    "앭뷙딡괒à֮𞀨̕b굒𝐃뫎빥퍥쎶à֮ͦ̕b빇∰視a〪ᨘ̖֚b쳝둦bar윕螆뎽𝟣m2댘솄右⼜럨췤"
  ),
  (
    [0xec, 0x89, 0x88, 0xeb, 0x9c, 0xac, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9d, 0x89, 0x83, 0xcc, 0x95, 0x62, 0xec, 0x88, 0x9d, 0xeb, 0x85, 0xbb, 0xe2, 0x85, 0xbb, 0xe9, 0xa3, 0xaf, 0xeb, 0xb7, 0xb2, 0xc8, 0xb3, 0xea, 0xb7, 0xaa, 0xe3, 0x89, 0xa1, 0xea, 0xb3, 0xa8, 0xe3, 0x85, 0x97, 0xed, 0x83, 0x99, 0xeb, 0xaf, 0x97, 0xf0, 0x9d, 0x98, 0x9c, 0xeb, 0x82, 0x80, 0xea, 0xb2, 0xbd, 0xe2, 0x80, 0xb4, 0xea, 0xbd, 0xa1, 0xec, 0xb8, 0x84, 0x61, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0xf0, 0x91, 0x8c, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xec, 0xaf, 0x8b, 0xeb, 0x88, 0xb1, 0xed, 0x83, 0xaf, 0xec, 0x88, 0xbb, 0xc5, 0xb2, 0xc5, 0x90, 0xcb, 0xa4, 0xd7, 0x91, 0xd6, 0xbf, 0xed, 0x9b, 0xb4, 0xe2, 0xbd, 0xa4, 0xec, 0x8c, 0xb4, 0xeb, 0x80, 0xab, 0xec, 0xa3, 0xb4, 0xeb, 0x86, 0x8f, 0xec, 0xb3, 0xba, 0x61, 0xe3, 0x80, 0xaa, 0xdd, 0x82, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe8, 0x98, 0xad, 0xef, 0xaf, 0x94],
    "쉈뜬à֮𝉃̕b숝녻ⅻ飯뷲ȳ귪㉡골ㅗ탙믗𝘜낀경‴꽡츄a̴़𑌼゙b쯋눱탯숻ŲŐˤבֿ훴⽤쌴뀫죴놏쳺a〪݂̖֚b蘭ﯔ"
  ),
  (
    [0xeb, 0xbc, 0xa0, 0xed, 0x8f, 0xa3, 0xf0, 0x91, 0x8d, 0x87, 0xcc, 0xb4, 0xf0, 0x91, 0x8d, 0x97, 0xec, 0x87, 0x82, 0xed, 0x93, 0xaf, 0xf0, 0x9d, 0x94, 0xbb, 0x61, 0xe1, 0xb3, 0xa2, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0xec, 0x8e, 0xbc, 0xef, 0xb1, 0xb2, 0x61, 0xd6, 0xae, 0xe1, 0xb3, 0x91, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x99, 0x9e, 0xf0, 0x9f, 0x88, 0xb3, 0xeb, 0x84, 0x99, 0xeb, 0xb9, 0xa6, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xba, 0xe0, 0xbd, 0xb4, 0x62, 0xe3, 0x8a, 0xb4, 0xeb, 0x97, 0xa9, 0xe3, 0x85, 0x8e, 0xeb, 0x84, 0xaa, 0xeb, 0x82, 0x9a, 0xeb, 0xbb, 0xa8, 0xec, 0x90, 0x9f, 0xec, 0xb8, 0xb4, 0xeb, 0x98, 0x81, 0xeb, 0xa7, 0xae, 0xec, 0xb9, 0x8a, 0xe8, 0x98, 0xbf, 0xec, 0x8d, 0xa5, 0xef, 0xbc, 0xa3, 0xe1, 0xb9, 0x81, 0xe2, 0x84, 0x9c, 0xec, 0x8d, 0xb7, 0xec, 0x97, 0xad, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0x96, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe8, 0x97, 0xba, 0xeb, 0x86, 0xb5, 0xe3, 0x8e, 0xba, 0xeb, 0xba, 0x89, 0xec, 0xa7, 0x97, 0xed, 0x89, 0xb4],
    "뼠폣𑍇̴𑍗쇂퓯𝔻a᳢̴़b쎼ﱲa֮᳑̀̕b홞🈳넙빦aཱིེུb㊴뗩ㅎ넪낚뻨쐟츴똁맮칊蘿썥Ｃṁℜ썷역a֮ᷖ̀̕b藺놵㎺뺉짗퉴"
  ),
]

let codeUnitNormalizationTestsB: [([UInt8], String)] = [
  (
    [0xeb, 0xb5, 0xaf, 0xec, 0x89, 0x90, 0xeb, 0xa2, 0x88, 0x61, 0xcd, 0x9c, 0xcd, 0x9d, 0xcd, 0xa0, 0xcd, 0x85, 0x62, 0xec, 0x9b, 0x98, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa5, 0x91, 0xcc, 0x95, 0x62, 0xeb, 0x88, 0xa4, 0xec, 0xa3, 0xbc, 0xec, 0x9d, 0x98, 0xc5, 0xac, 0xed, 0x80, 0xb0, 0xeb, 0x9f, 0x9c, 0xec, 0xa6, 0x97, 0xeb, 0x99, 0xb3, 0xeb, 0xa4, 0x87, 0xeb, 0xb8, 0x8b, 0xea, 0xb3, 0xa3, 0x35, 0xe6, 0x97, 0xa5, 0xe1, 0x85, 0xb3, 0xec, 0xa4, 0xad, 0xd9, 0x84, 0xd8, 0xa2, 0xec, 0xa7, 0xbf, 0xe7, 0x93, 0xa6, 0xe6, 0xaf, 0x8b, 0xec, 0xbc, 0xb0, 0xc3, 0xa0, 0xd6, 0xae, 0xcd, 0xa7, 0xcc, 0x95, 0x62, 0xea, 0xbf, 0xb7, 0xec, 0x8b, 0x9e, 0xe7, 0xbd, 0x91, 0xc7, 0xa1, 0xe1, 0xbf, 0x99, 0xe0, 0xbe, 0xab, 0xe0, 0xbe, 0xb7, 0xeb, 0x95, 0xb4, 0xeb, 0xa4, 0xb5, 0xed, 0x82, 0xbc, 0xeb, 0x82, 0xb0, 0xec, 0x8a, 0xa7, 0xe5, 0xaf, 0xb3, 0xed, 0x9d, 0xa1, 0xec, 0x9a, 0x80, 0xeb, 0x86, 0x90],
    "뵯쉐뢈a͜͝͠ͅb웘à֮॑̕b눤주의Ŭ퀰럜즗뙳뤇븋곣5日ᅳ줭لآ짿瓦毋켰à֮ͧ̕b꿷싞网ǡῙྫྷ땴뤵킼낰슧寳흡욀놐"
  ),
  (
    [0xec, 0xb4, 0xb8, 0xce, 0xbd, 0xeb, 0xa7, 0x9d, 0x54, 0xeb, 0x96, 0x87, 0xed, 0x88, 0xb8, 0xe5, 0x85, 0xa9, 0xed, 0x82, 0x84, 0xe4, 0xb8, 0xb6, 0xe2, 0x88, 0xa4, 0xe1, 0xb8, 0xa5, 0xed, 0x91, 0xa6, 0xec, 0xa6, 0xbc, 0xec, 0xac, 0xa3, 0xe1, 0x85, 0xa7, 0xd8, 0xab, 0xd9, 0x86, 0xea, 0xbf, 0xbf, 0xec, 0x87, 0xb9, 0xed, 0x8d, 0xa2, 0xec, 0x9f, 0x9e, 0xec, 0x99, 0xb9, 0xca, 0x90, 0xc3, 0xa0, 0xd6, 0xae, 0xd9, 0x9d, 0xcc, 0x95, 0x62, 0xeb, 0xbf, 0xac, 0xea, 0xb4, 0x9c, 0xce, 0xbc, 0x6d, 0xe1, 0xb9, 0xaf, 0x61, 0xd6, 0xae, 0xdc, 0xb0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xef, 0xb8, 0xab, 0xd6, 0x9a, 0x62, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xae, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xad, 0xb3, 0x20, 0xd9, 0x91, 0xd9, 0xb0, 0xeb, 0x89, 0xa3, 0xeb, 0x95, 0x9b, 0xec, 0xb6, 0x85, 0x4c, 0x4a, 0xea, 0xbe, 0x85, 0xd8, 0xa6, 0xd9, 0x86, 0xce, 0xbd, 0xec, 0x81, 0xa2],
    "촸ν망T떇툸兩킄丶∤ḥ푦즼쬣ᅧثن꿿쇹퍢쟞왹ʐà֮ٝ̕b뿬괜μmṯa֮ܰ̀̕ba〪̖︫֚ba֮ⷮ̀̕b쭳 ّٰ뉣땛춅LJ꾅ئنν쁢"
  ),
  (
    [0xeb, 0xa9, 0xb3, 0xea, 0xb4, 0x93, 0xf0, 0x9d, 0x99, 0x91, 0xec, 0x99, 0xac, 0xef, 0xbb, 0xbc, 0xeb, 0x84, 0x91, 0xe8, 0x82, 0x8b, 0xeb, 0xac, 0xa2, 0xf0, 0x9d, 0x91, 0xb2, 0xed, 0x8c, 0x8e, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x84, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x28, 0x6a, 0x29, 0xec, 0x96, 0x9c, 0xc3, 0xa0, 0xd6, 0xae, 0xcd, 0xa5, 0xcc, 0x95, 0x62, 0xf0, 0x9d, 0x9e, 0xb9, 0xea, 0xbd, 0xbc, 0xec, 0x88, 0xa7, 0xec, 0xa8, 0xbf, 0xeb, 0x85, 0xa5, 0xe1, 0xb8, 0x81, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xe1, 0x9f, 0x92, 0xd6, 0xb0, 0x62, 0xea, 0xbb, 0xb3, 0xec, 0x8f, 0xb7, 0xe3, 0x8b, 0xbe, 0xeb, 0xad, 0x9d, 0xec, 0xa0, 0x87, 0xeb, 0xb6, 0x9d, 0xc8, 0x8e, 0xea, 0xb4, 0x8f, 0xec, 0x8a, 0x89, 0xc3, 0x80, 0xef, 0xaf, 0x9c, 0xeb, 0xad, 0xbd, 0xf0, 0xa8, 0xaf, 0xba, 0xef, 0xbb, 0xaf, 0xec, 0xbf, 0xa9, 0xeb, 0x99, 0x87, 0xf0, 0xa6, 0x9e, 0xb5, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x8b, 0xaa, 0xd6, 0xb0, 0x62, 0xeb, 0xae, 0xb9],
    "멳괓𝙑왬ﻼ넑肋묢𝑲팎a֮𞀄̀̕b(j)얜à֮ͥ̕b𝞹꽼숧쨿녥ḁ〪̖֚ba゙्្ְb껳쏷㋾뭝젇붝Ȏ괏슉Àﯜ뭽𨯺ﻯ쿩뙇𦞵a゙्𑋪ְb뮹"
  ),
  (
    [0xef, 0xb1, 0xb3, 0xed, 0x8f, 0x96, 0xeb, 0x94, 0x86, 0xea, 0xb6, 0xa3, 0xeb, 0xbe, 0xa9, 0xed, 0x8d, 0xbb, 0xec, 0x8d, 0xa4, 0xed, 0x82, 0xbe, 0xeb, 0x85, 0xbe, 0xec, 0xa9, 0xbd, 0xf0, 0x9d, 0x94, 0x85, 0xf0, 0x9d, 0x9e, 0x89, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x84, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xbe, 0xab, 0xec, 0x9e, 0x99, 0xec, 0x8d, 0x9a, 0xeb, 0x9e, 0x8e, 0xec, 0xab, 0xa3, 0xef, 0xaf, 0xb9, 0xeb, 0x8e, 0x9d, 0xed, 0x8a, 0xb3, 0xea, 0xbe, 0xba, 0xf0, 0xa6, 0x93, 0x9a, 0xed, 0x8f, 0xa8, 0xed, 0x8b, 0xb4, 0xec, 0xb0, 0x9c, 0xf0, 0x9d, 0x9b, 0x89, 0xec, 0x82, 0xba, 0xec, 0xb0, 0x8d, 0xed, 0x87, 0x86, 0xeb, 0x80, 0xa2, 0x61, 0xd6, 0xae, 0xef, 0xb8, 0xa3, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xb4, 0x9c, 0xec, 0x8d, 0x8d, 0xec, 0x82, 0x80, 0xe8, 0xb1, 0x88, 0xec, 0x8f, 0x9d, 0xed, 0x92, 0xa6, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa7, 0xbe, 0xcc, 0x95, 0x62, 0xef, 0xb7, 0xb8, 0xeb, 0x83, 0xa8, 0xed, 0x85, 0x9d, 0xeb, 0xb5, 0xa4, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xef, 0xb2, 0xb4, 0xe2, 0x91, 0xa2, 0xf0, 0x9d, 0x9c, 0xb2, 0xe5, 0xba, 0xb6, 0xc3, 0xbc, 0xf0, 0x9d, 0x98, 0x96, 0xf0, 0x9d, 0x95, 0xa0],
    "ﱳ폖딆궣뾩퍻썤킾녾쩽𝔅𝞉a֮𞀄̀̕b쾫잙썚랎쫣ﯹ뎝튳꾺𦓚폨틴찜𝛉삺찍퇆뀢a֮︣̀̕bﴜ썍삀豈쏝풦à֮৾̕bﷸ냨텝뵤̴ᆮﲴ③𝜲庶ü𝘖𝕠"
  ),
  (
    [0xe7, 0xa5, 0x9d, 0x61, 0xd6, 0xae, 0xea, 0x99, 0xb4, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xf0, 0x9d, 0x95, 0x94, 0xef, 0xba, 0x8a, 0xed, 0x8c, 0xa4, 0xec, 0xa0, 0xae, 0xe8, 0xb4, 0x9b, 0xed, 0x84, 0xb2, 0xeb, 0x9f, 0xbc, 0xed, 0x81, 0xbd, 0xed, 0x91, 0x9a, 0xf0, 0x9d, 0x9c, 0xbc, 0xeb, 0xbb, 0x89, 0xec, 0x8c, 0xb6, 0xeb, 0x8c, 0xbb, 0xeb, 0x8d, 0x96, 0xec, 0x82, 0xaa, 0xec, 0xab, 0x9d, 0xec, 0xb8, 0xaa, 0x61, 0xcc, 0x96, 0xd6, 0xad, 0xd6, 0x9a, 0xe3, 0x80, 0xae, 0x62, 0xec, 0x84, 0x9c, 0xec, 0xa5, 0xbf, 0xef, 0xb5, 0xa2, 0xe1, 0xbf, 0xa6, 0xed, 0x89, 0xa5, 0xf0, 0x9d, 0x93, 0x90, 0xec, 0x96, 0xa3, 0xec, 0x97, 0x8b, 0xea, 0xb5, 0xa6, 0xe2, 0x91, 0xaf, 0xec, 0xa9, 0xba, 0xea, 0xb0, 0x85, 0xe3, 0x85, 0xa3, 0xe1, 0xbb, 0x94, 0xec, 0x84, 0x8f, 0xeb, 0x9e, 0xae, 0xeb, 0xba, 0xa0, 0xec, 0xbf, 0x86, 0xe7, 0xb9, 0x81, 0xec, 0xa8, 0x8c, 0xef, 0xbd, 0xac, 0xeb, 0xa4, 0x98, 0xe5, 0x83, 0x9a, 0xeb, 0xb7, 0x8a, 0xeb, 0xad, 0x89, 0xec, 0x88, 0xba, 0xec, 0xaa, 0x80, 0xf0, 0x9f, 0x85, 0x85, 0xec, 0x81, 0xaf, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xad, 0x8d, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62],
    "祝a֮ꙴ̀̕b𝕔ﺊ팤젮贛턲럼큽푚𝜼뻉쌶댻덖삪쫝츪a̖֭֚〮b서쥿ﵢῦ퉥𝓐얣엋굦⑯쩺갅ㅣỔ섏랮뺠쿆繁쨌ｬ뤘僚뷊뭉숺쪀🅅쁯a゙୍्ְb"
  ),
  (
    [0xec, 0xb5, 0x95, 0xc5, 0x9f, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0xb7, 0xb0, 0xcc, 0x95, 0x62, 0xed, 0x8c, 0x81, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xb0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x81, 0x93, 0xed, 0x8e, 0xb6, 0xec, 0xa4, 0x8b, 0xd9, 0xbf, 0xeb, 0xac, 0xad, 0xec, 0x8a, 0x8e, 0xe6, 0x9e, 0x97, 0x28, 0xe4, 0xb8, 0x80, 0x29, 0xec, 0xbc, 0xab, 0xeb, 0x80, 0xa5, 0xeb, 0xad, 0x9e, 0xec, 0xba, 0xa6, 0x6b, 0x6c, 0xeb, 0x9b, 0xa4, 0xc2, 0xa6, 0xed, 0x83, 0xbb, 0xed, 0x99, 0xaa, 0xec, 0xb6, 0xb4, 0xed, 0x9e, 0x94, 0xec, 0x92, 0xba, 0xeb, 0xac, 0xae, 0xe3, 0x83, 0xac, 0xea, 0xb1, 0x9c, 0xeb, 0x9a, 0x97, 0x6b, 0x63, 0x61, 0x6c, 0xc3, 0xa0, 0xd6, 0xae, 0xd2, 0x84, 0xcc, 0x95, 0x62, 0xd9, 0x85, 0xd9, 0x8a, 0xec, 0xa1, 0xa4, 0xec, 0xa1, 0xbc, 0xe7, 0xb3, 0x92, 0xd8, 0xaa, 0xd8, 0xac, 0xd9, 0x8a, 0xe1, 0xbf, 0x96, 0xe6, 0xb1, 0xa7, 0x41, 0xeb, 0x83, 0x8e, 0xe0, 0xbd, 0xb1, 0xe0, 0xbe, 0x80, 0xeb, 0xbf, 0x8e, 0xeb, 0x99, 0x81, 0xeb, 0xbf, 0x9e, 0xec, 0xb8, 0x8a, 0x4e, 0xea, 0xba, 0xbe, 0xed, 0x9a, 0x91, 0xe1, 0xbe, 0x8d, 0xd8, 0xaa, 0xd9, 0x85, 0xd8, 0xad],
    "쵕şà֮ⷰ̕b팁a֮ⷰ̀̕b끓펶줋ٿ묭슎林(一)켫뀥뭞캦kl뛤¦탻홪춴힔쒺묮レ걜뚗kcalà֮҄̕bمي졤졼糒تجيῖ汧A냎ཱྀ뿎뙁뿞츊N꺾횑ᾍتمح"
  ),
  (
    [0xec, 0x82, 0xb3, 0xeb, 0x9b, 0xa6, 0xeb, 0xbc, 0x81, 0xea, 0xba, 0xa1, 0xec, 0x91, 0x8f, 0xea, 0xbb, 0xa3, 0xe1, 0xba, 0xa8, 0xe9, 0x84, 0x9b, 0xeb, 0x96, 0xba, 0xec, 0xb6, 0x9a, 0xec, 0xa8, 0xa3, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x96, 0xac, 0xb0, 0xcc, 0x95, 0x62, 0xe6, 0x90, 0xa2, 0xeb, 0xba, 0x8b, 0xeb, 0xb8, 0x87, 0xec, 0x8c, 0x80, 0x69, 0x76, 0xeb, 0x87, 0x9c, 0xe0, 0xa8, 0x97, 0xe0, 0xa8, 0xbc, 0xec, 0x91, 0x8a, 0xeb, 0xa7, 0x8e, 0xec, 0xae, 0xbd, 0xec, 0x83, 0x84, 0xe8, 0xb4, 0x88, 0xec, 0xa8, 0xaa, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9e, 0xa3, 0x95, 0xd6, 0x9a, 0x62, 0xec, 0xbd, 0x9e, 0xe2, 0x86, 0x9a, 0xec, 0xb2, 0x97, 0xec, 0xab, 0xa2, 0xed, 0x86, 0xab, 0xed, 0x9b, 0xa7, 0xeb, 0x98, 0xb5, 0xe4, 0xbb, 0x80, 0xec, 0xa0, 0x80, 0xed, 0x91, 0xbc, 0xec, 0xa1, 0xad, 0xec, 0x86, 0xa1, 0xeb, 0x8c, 0x83, 0xe8, 0xbd, 0xa2, 0xda, 0x98, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xb1, 0x8d, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xec, 0x97, 0xa3, 0x61, 0xd8, 0x99, 0xd8, 0x9a, 0xd9, 0x90, 0xd9, 0x91, 0x62, 0xec, 0x89, 0x81, 0x20, 0xcc, 0x93, 0xcd, 0x82, 0xea, 0xb5, 0xb9, 0xe7, 0x85, 0x85, 0xed, 0x9d, 0xa6, 0xec, 0x94, 0xa1],
    "삳뛦뼁꺡쑏껣Ẩ鄛떺춚쨣à֮𖬰̕b搢뺋븇쌀iv뇜ਗ਼쑊많쮽샄贈쨪a〪̖𞣕֚b콞↚첗쫢톫훧똵什저푼졭송댃轢ژa゙్्ְb엣aؙؚِّb쉁 ̓͂굹煅흦씡"
  ),
  (
    [0xeb, 0xa9, 0xb8, 0xf0, 0x9d, 0x96, 0xac, 0x61, 0xe3, 0x80, 0xaa, 0xe0, 0xa3, 0xb9, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xed, 0x80, 0x8c, 0xeb, 0xb8, 0xb2, 0xeb, 0xac, 0x9a, 0xd7, 0x90, 0xd7, 0x9c, 0xea, 0xb3, 0x90, 0xec, 0xb6, 0x85, 0xe1, 0xbb, 0x8f, 0xeb, 0xb1, 0xae, 0xeb, 0x95, 0xa8, 0xe3, 0xbc, 0x9b, 0xea, 0xb6, 0x83, 0xeb, 0x91, 0x80, 0xeb, 0x8d, 0x87, 0xec, 0xa4, 0xb0, 0xed, 0x8a, 0xb8, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xe3, 0x8c, 0x80, 0xf0, 0x9f, 0x85, 0x8b, 0xec, 0x99, 0xb3, 0xe0, 0xa4, 0xb4, 0xeb, 0xb2, 0x90, 0xeb, 0x9e, 0xb0, 0xeb, 0x87, 0x8b, 0xec, 0x89, 0x90, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0xeb, 0xb3, 0x83, 0xec, 0xb1, 0xb4, 0xf0, 0x9d, 0x9b, 0x9b, 0xe3, 0x85, 0x8f, 0xea, 0xbf, 0x98, 0xef, 0xbe, 0x82, 0x61, 0xe3, 0x82, 0x99, 0xf0, 0x91, 0x91, 0x82, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xe5, 0xb7, 0x9b, 0xeb, 0x83, 0xa4, 0xeb, 0xbb, 0x89, 0xe3, 0x85, 0xa3, 0x61, 0xd6, 0xae, 0xe1, 0xa8, 0x97, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe2, 0x89, 0x87, 0xeb, 0xbe, 0xb9, 0x43, 0x44, 0xc3, 0x8c, 0xef, 0xb5, 0xb2, 0xeb, 0xae, 0xac, 0xef, 0xbd, 0x9e, 0xcb, 0xa0, 0xeb, 0x88, 0xb4, 0xe3, 0x81, 0x92, 0xec, 0xb1, 0x9f, 0xd8, 0xa8, 0xd8, 0xae, 0xd9, 0x8a],
    "멸𝖬a〪ࣹ̖֚b퀌븲묚אל곐춅ỏ뱮땨㼛궃둀덇줰트̴ᆮ㌀🅋왳ऴ벐랰뇋쉐̴ᆼ볃챴𝛛ㅏ꿘ﾂa゙𑑂्ְb巛냤뻉ㅣa֮ᨗ̀̕b≇뾹CDÌﵲ뮬～ˠ눴げ챟بخي"
  ),
  (
    [0xec, 0x9d, 0xa5, 0xec, 0xa2, 0xa0, 0xe1, 0xbf, 0x91, 0xeb, 0x96, 0xa0, 0xc8, 0xa7, 0xeb, 0x82, 0xb3, 0xec, 0x9a, 0x92, 0xed, 0x89, 0xa5, 0xed, 0x81, 0x92, 0xe2, 0x92, 0x88, 0xef, 0xbf, 0xa1, 0xed, 0x99, 0x83, 0xe1, 0xb9, 0xa1, 0xec, 0xba, 0x8a, 0xef, 0xb4, 0xa6, 0xec, 0xaf, 0xa7, 0xec, 0x8d, 0x9c, 0xef, 0xb5, 0xa0, 0xf0, 0x9d, 0x9a, 0xa5, 0xed, 0x95, 0x9d, 0xf0, 0x9d, 0x96, 0x8c, 0xf0, 0x9d, 0x98, 0x92, 0xeb, 0xb6, 0xb1, 0xec, 0xae, 0x93, 0xec, 0x86, 0x9d, 0xf0, 0x9e, 0xb9, 0xb6, 0xe1, 0xbe, 0x92, 0xf0, 0x9d, 0x99, 0x89, 0xec, 0xae, 0x90, 0xea, 0xbe, 0xae, 0xe1, 0x84, 0x80, 0xcc, 0xb4, 0xe1, 0x85, 0xae, 0xeb, 0x89, 0xa3, 0xec, 0xb7, 0xb9, 0xe5, 0x86, 0x8d, 0xec, 0xa8, 0x85, 0xe3, 0x8c, 0xab, 0x61, 0xe0, 0xb8, 0xb8, 0xe0, 0xb9, 0x89, 0xe0, 0xb9, 0x88, 0xe0, 0xba, 0xb8, 0x62, 0xf0, 0x9d, 0x9c, 0x89, 0xec, 0xa6, 0x8c, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9e, 0xa3, 0x90, 0xd6, 0x9a, 0x62, 0xeb, 0x9b, 0xa6, 0xe1, 0xb5, 0xa4, 0xe9, 0xa1, 0x9e, 0xea, 0xb5, 0xb5, 0xeb, 0x80, 0x8d, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0x87, 0xcc, 0x95, 0x62, 0xef, 0xae, 0xa3, 0xe3, 0x8e, 0x91, 0xe3, 0x8b, 0x95, 0xec, 0xa5, 0x87, 0xed, 0x96, 0x8d, 0xeb, 0xa3, 0x83, 0xe1, 0xbd, 0x8a, 0xeb, 0xbb, 0x9f, 0xec, 0x85, 0xa2, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0xb6, 0x97, 0xd6, 0xb0, 0x62, 0xef, 0xbb, 0xb5, 0xe2, 0xbc, 0xbb, 0xe6, 0xb3, 0xa5, 0xeb, 0xb6, 0xa1],
    "읥좠ῑ떠ȧ낳욒퉥큒⒈￡홃ṡ캊ﴦ쯧썜ﵠ𝚥핝𝖌𝘒붱쮓솝𞹶ᾒ𝙉쮐꾮ᄀ̴ᅮ뉣췹再쨅㌫aุ้่ຸb𝜉즌a〪̖𞣐֚b뛦ᵤ類굵뀍à֮᷇̕bﮣ㎑㋕쥇햍룃Ὂ뻟셢a゙्𑶗ְbﻵ⼻泥붡"
  ),
  (
    [0xc7, 0x97, 0xec, 0x88, 0x8c, 0xeb, 0xa4, 0xaa, 0xec, 0x86, 0x8a, 0xec, 0xb3, 0xa0, 0xc8, 0xb2, 0xeb, 0x94, 0x8a, 0xeb, 0xa3, 0x99, 0xe1, 0xb4, 0xb9, 0xe3, 0x8a, 0xb7, 0xf0, 0x9d, 0x97, 0xa4, 0xed, 0x83, 0xa3, 0xeb, 0xba, 0xaf, 0xec, 0xba, 0x84, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0x97, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x85, 0xba, 0x61, 0xcc, 0xb4, 0xf0, 0x96, 0xab, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0xa8, 0xb2, 0xec, 0xae, 0xaa, 0x61, 0xe3, 0x80, 0xaa, 0xe2, 0x83, 0xae, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe2, 0xbd, 0x92, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x96, 0xbf, 0xd6, 0xb0, 0x62, 0xe1, 0xbc, 0x8a, 0xe5, 0xa4, 0x86, 0x61, 0xd6, 0xae, 0xcd, 0x8a, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0x81, 0x9d, 0xeb, 0xba, 0xba, 0xeb, 0xb1, 0x8c, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xea, 0xb6, 0xab, 0x61, 0xcc, 0xb4, 0xf0, 0x91, 0x8c, 0xbb, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xf0, 0x9d, 0x9c, 0x9c, 0xed, 0x92, 0x81, 0xe3, 0x8c, 0x8c, 0xe9, 0x96, 0x8b, 0xeb, 0xad, 0x9c, 0xec, 0xbe, 0x90, 0xec, 0xa9, 0xa0, 0xec, 0xb0, 0x92, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9d, 0x85, 0xbc, 0xd6, 0x9a, 0x62, 0xe1, 0xbb, 0x9c, 0xec, 0xa2, 0xa5, 0xe3, 0x8d, 0xa2, 0xef, 0xb0, 0x96, 0xec, 0xa8, 0xbd, 0xed, 0x84, 0x9a, 0xea, 0xb8, 0x9b, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0xb9, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xce, 0x8c, 0xec, 0x9b, 0x83, 0x61, 0xcc, 0xb4, 0xcc, 0xb7, 0xe0, 0xa4, 0xbc, 0x62, 0xed, 0x8d, 0x9c, 0xc4, 0x86, 0xec, 0xbf, 0xa0, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xea, 0xb9, 0x82, 0xeb, 0xad, 0x92, 0xeb, 0x95, 0x85, 0x61, 0xf0, 0x9d, 0x85, 0xa9, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0xaa, 0xa7, 0xeb, 0x85, 0x8e, 0xeb, 0xbb, 0x8b],
    "Ǘ숌뤪솊쳠Ȳ딊룙ᴹ㊷𝗤탣뺯캄a֮ࣗ̀̕b녺a̴𖫴़b먲쮪a〪⃮̖֚b⽒a゙्𑖿ְbἊ夆a֮͊̀̕b쁝뺺뱌̴ᆵ궫a̴𑌻़゙b𝜜풁㌌開뭜쾐쩠찒a〪̖𝅼֚bỜ좥㍢ﰖ쨽턚긛a〪̹̖֚bΌ웃a̴̷़b퍜Ć쿠̴ᆮ깂뭒땅a𝅩̴़b몧녎뻋"
  ),
  (
    [0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x81, 0x86, 0xd6, 0xb0, 0x62, 0x31, 0x39, 0xed, 0x87, 0x92, 0xeb, 0x95, 0xa0, 0xec, 0xbe, 0x8e, 0xec, 0xad, 0xb7, 0xed, 0x8a, 0x93, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0x9c, 0xd6, 0x9a, 0x62, 0xeb, 0x8e, 0x89, 0x61, 0xcd, 0x9d, 0xcd, 0x85, 0xcd, 0x85, 0x62, 0xec, 0xa0, 0x88, 0xed, 0x85, 0xa3, 0xec, 0xa7, 0xbe, 0x6a, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0xa5, 0xcc, 0x95, 0x62, 0xeb, 0xb9, 0x95, 0xec, 0xa2, 0x90, 0xec, 0x89, 0xa4, 0xec, 0x9a, 0x84, 0xec, 0x8d, 0x87, 0xc4, 0xb7, 0xeb, 0xb9, 0xba, 0xed, 0x88, 0x9b, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb3, 0xb8, 0xcc, 0x95, 0x62, 0x28, 0xea, 0xb0, 0x80, 0x29, 0xed, 0x9c, 0x98, 0xec, 0xb1, 0x91, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0x99, 0xcc, 0x95, 0x62, 0xea, 0xbb, 0xa1, 0xea, 0xba, 0xae, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0x83, 0xb0, 0xcc, 0x95, 0x62, 0xe3, 0x83, 0xb1, 0x61, 0xcc, 0xb4, 0xcc, 0xb7, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0x8f, 0x84, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa3, 0xa0, 0xcc, 0x95, 0x62, 0xe1, 0xbb, 0xb7, 0xea, 0xbb, 0x9f, 0xec, 0xb1, 0x85, 0xed, 0x83, 0x84, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x92, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xd6, 0xae, 0xe0, 0xa0, 0x9f, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x8b, 0x83, 0xec, 0x8d, 0xaa, 0xeb, 0xa1, 0x9b, 0xed, 0x84, 0xa9, 0xea, 0xb9, 0xaf, 0xec, 0xa7, 0xb2, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x84, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xea, 0xb8, 0xa4, 0xeb, 0xbc, 0x9e, 0xeb, 0x8b, 0xa8, 0xec, 0xba, 0x83, 0xed, 0x9d, 0x8a, 0xeb, 0xb0, 0x8d, 0x33, 0x34, 0xec, 0x8b, 0x99, 0xf0, 0xa6, 0x8b, 0x99, 0xed, 0x9a, 0x84, 0xec, 0xaf, 0x99, 0xce, 0xa5],
    "a゙्𑁆ְb19퇒땠쾎쭷튓a〪̖̜֚b뎉a͝ͅͅb절텣짾jà֮ࠥ̕b빕좐쉤욄썇ķ빺툛à֮᳸̕b(가)휘챑à֮࠙̕b껡꺮à֮⃰̕bヱa̴̷़b도à֮࣠̕bỷ껟책탄a֮𞀒̀̕ba֮ࠟ̀̕b닃썪롛턩깯짲a֮𞀄̀̕b긤뼞단캃흊밍34싙𦋙횄쯙Υ"
  ),
  (
    [0xeb, 0x92, 0x9c, 0xe3, 0x80, 0x82, 0xeb, 0xa5, 0x9c, 0xe1, 0xb9, 0xa7, 0xed, 0x98, 0x89, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x90, 0xbd, 0x89, 0xcc, 0x95, 0x62, 0xec, 0x91, 0xbb, 0xed, 0x98, 0x8b, 0xed, 0x9a, 0xa8, 0xec, 0xae, 0x9f, 0xed, 0x94, 0x98, 0xec, 0xa4, 0xb0, 0xeb, 0xab, 0xb5, 0xeb, 0x8c, 0x85, 0xeb, 0xb0, 0xbd, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x90, 0xbd, 0x8d, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x72, 0x61, 0x64, 0xe2, 0x88, 0x95, 0x73, 0xe5, 0x86, 0x95, 0xea, 0xbf, 0xa2, 0xeb, 0x81, 0xb2, 0xec, 0x9a, 0x87, 0x61, 0xcc, 0xb4, 0xe1, 0xb3, 0xa5, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0x93, 0xa5, 0xe1, 0xbb, 0x86, 0xed, 0x87, 0x89, 0xe1, 0xbd, 0x8c, 0xec, 0x9a, 0xb0, 0x20, 0xe3, 0x82, 0x9a, 0x61, 0xe1, 0xb3, 0xa5, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0x9b, 0xb0, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0xec, 0xbc, 0xa4, 0xeb, 0xa5, 0xb1, 0xeb, 0xa2, 0x8e, 0xec, 0xae, 0xbe, 0xe1, 0xbf, 0x96, 0xec, 0xac, 0x95, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe0, 0xa1, 0x9a, 0xd6, 0x9a, 0x62, 0xe7, 0x99, 0xa9, 0xe1, 0xba, 0x95, 0xed, 0x9b, 0xb3, 0xed, 0x9a, 0x86, 0xed, 0x90, 0x99, 0xed, 0x93, 0x90, 0xe0, 0xb3, 0x8b, 0xec, 0xb8, 0x98, 0xec, 0x93, 0xb9, 0xeb, 0x95, 0x99, 0xed, 0x9a, 0xba, 0xec, 0x95, 0x88, 0x61, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0xe1, 0xac, 0xb4, 0xe3, 0x82, 0x99, 0x62, 0xeb, 0x8a, 0x82, 0xeb, 0xb6, 0xa4, 0xec, 0x81, 0x91, 0xec, 0xae, 0xb5, 0xec, 0x91, 0x90, 0xe9, 0xa0, 0x98, 0xe5, 0xbc, 0x84, 0xeb, 0xa2, 0x84, 0xec, 0x88, 0x92, 0xec, 0x8f, 0x8b],
    "뒜。륜ṧ혉à֮𐽉̕b쑻혋효쮟픘줰뫵댅밽a〪𐽍̖֚brad∕s冕꿢끲욇a̴᳥़b듥Ệ퇉Ὄ우 ゚a᳥̴़b뛰̴ᆼ켤륱뢎쮾ῖ쬕a〪̖࡚֚b癩ẕ훳횆퐙퓐ೋ츘쓹땙횺안a̴़᬴゙b늂붤쁑쮵쑐領弄뢄숒쏋"
  ),
  (
    [0xef, 0xbd, 0xa5, 0xeb, 0xaf, 0x9e, 0xea, 0xb3, 0xa9, 0xe1, 0xbb, 0xa2, 0xed, 0x92, 0x83, 0xec, 0xa1, 0x9f, 0xeb, 0x89, 0xaf, 0xec, 0xb1, 0xbd, 0xf0, 0x9d, 0x9c, 0xac, 0x28, 0x59, 0x29, 0xc7, 0x95, 0xf0, 0x9d, 0x9f, 0x88, 0xec, 0x88, 0xbe, 0xed, 0x89, 0x83, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xaa, 0xb3, 0xcc, 0x95, 0x62, 0xed, 0x82, 0x9d, 0xd8, 0xa7, 0xd9, 0xb4, 0xf0, 0x9e, 0xb8, 0x8e, 0x61, 0xd6, 0xae, 0xf0, 0x90, 0x8d, 0xb6, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xa6, 0xa7, 0x32, 0x37, 0xe6, 0x97, 0xa5, 0xe3, 0x83, 0x81, 0xef, 0xba, 0x8f, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0xb4, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x92, 0x92, 0xf0, 0x9d, 0x9e, 0x9d, 0xcf, 0x8c, 0xed, 0x88, 0x8a, 0xeb, 0x97, 0xac, 0xeb, 0xb3, 0x8b, 0xeb, 0x95, 0xbb, 0xec, 0x9b, 0x95, 0xec, 0x85, 0x8a, 0xe6, 0x9f, 0xba, 0xef, 0xb4, 0xb1, 0xec, 0x8b, 0x98, 0xed, 0x9c, 0xa1, 0xec, 0x9d, 0x90, 0xe2, 0xbe, 0x9b, 0xec, 0x88, 0x9b, 0xeb, 0xa1, 0x9c, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0xf0, 0x9d, 0x95, 0xae, 0xe1, 0x84, 0x80, 0xed, 0x91, 0x9a, 0xe3, 0x8b, 0x99, 0xce, 0x89, 0xef, 0xb4, 0xb0, 0x28, 0xe4, 0xba, 0x94, 0x29, 0xef, 0xbd, 0xad, 0xef, 0xb4, 0x9f, 0xec, 0xa8, 0xb9, 0xec, 0x9f, 0xab, 0xeb, 0x8f, 0x98, 0xe4, 0xa9, 0xae, 0xec, 0x80, 0xae, 0xf0, 0x9d, 0x91, 0xa5, 0x61, 0xd6, 0xae, 0xdd, 0x80, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe6, 0xbb, 0x8b, 0xea, 0xb7, 0xa6, 0xf0, 0x9d, 0x93, 0xbe],
    "･믞곩Ợ풃졟뉯챽𝜬(Y)Ǖ𝟈숾퉃à֮᪳̕b킝اٴ𞸎a֮𐍶̀̕b릧27日チﺏa֮ᷴ̀̕b풒𝞝ό툊뗬볋땻웕셊柺ﴱ싘휡읐⾛숛로̴ᆼ𝕮ᄀ푚㋙Ήﴰ(五)ｭﴟ쨹쟫돘䩮쀮𝑥a֮݀̀̕b滋귦𝓾"
  ),
  (
    [0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x90, 0xbd, 0x87, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xeb, 0xbd, 0x97, 0xeb, 0x81, 0xb9, 0xe2, 0x91, 0xb8, 0xd7, 0xb2, 0xd6, 0xb7, 0xea, 0xbb, 0x90, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0xa5, 0x88, 0xcc, 0x95, 0x62, 0xec, 0xa4, 0x8e, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0xa9, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x9d, 0x86, 0x82, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xed, 0x8b, 0xbf, 0xe3, 0x8e, 0x88, 0xec, 0xbe, 0x9a, 0xe7, 0x9d, 0x80, 0xef, 0xb3, 0x98, 0xf0, 0x9d, 0x9b, 0x99, 0xf0, 0x9d, 0x9b, 0x90, 0xec, 0xa8, 0xa7, 0xcc, 0x88, 0xcc, 0x81, 0x61, 0xd6, 0xae, 0xea, 0xa3, 0xb0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x99, 0xba, 0xec, 0xad, 0x99, 0xec, 0x91, 0x99, 0xeb, 0xbd, 0xb4, 0xe9, 0x9b, 0xa3, 0x61, 0xe3, 0x80, 0xaa, 0xdd, 0x88, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe3, 0x89, 0xad, 0xe7, 0xb3, 0x96, 0xeb, 0x8a, 0xa7, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x8f, 0xcc, 0x95, 0x62, 0xf0, 0x9d, 0x9d, 0xa8, 0xe3, 0x8c, 0xa6, 0xe3, 0x8c, 0x84, 0xed, 0x87, 0x8f, 0xeb, 0x85, 0x9a, 0xeb, 0xa2, 0x89, 0xec, 0x82, 0xa8, 0xeb, 0x9c, 0x9a, 0x61, 0xcc, 0xb4, 0xcc, 0xb7, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0xac, 0x83, 0xed, 0x84, 0x84, 0xed, 0x8c, 0x9f, 0xec, 0x81, 0xb7, 0xed, 0x9e, 0x9d, 0x3b, 0xea, 0xb1, 0xb5, 0xeb, 0xb2, 0x90, 0xeb, 0x87, 0xad, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0xac, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xa3, 0x81, 0xf0, 0x9d, 0x90, 0xb8, 0xeb, 0x96, 0xab, 0xec, 0xb0, 0x9b, 0xeb, 0xb3, 0x82, 0xe3, 0x8b, 0xa0, 0xec, 0xa7, 0xbc, 0xeb, 0x94, 0x92, 0xeb, 0xb0, 0x84, 0xec, 0x94, 0xa2, 0xec, 0xaa, 0x8b, 0xec, 0xbc, 0x87, 0xe1, 0x84, 0x86, 0xcc, 0xb4, 0xe1, 0x85, 0xa5, 0xed, 0x91, 0x89, 0xed, 0x99, 0x83, 0xeb, 0x94, 0xb0, 0xe0, 0xa4, 0x95, 0xe0, 0xa4, 0xbc, 0xed, 0x90, 0x86, 0xeb, 0xb5, 0x83, 0xe3, 0x8b, 0xaa, 0xec, 0xb8, 0x91],
    "a〪𐽇̖֚b뽗끹⑸ײַ껐à֮𞥈̕b줎a֮ᷩ̀̕ba〪𝆂̖֚b틿㎈쾚着ﳘ𝛙𝛐쨧̈́a֮꣰̀̕b홺쭙쑙뽴難a〪݈̖֚b㉭糖늧à֮̏̕b𝝨㌦㌄퇏녚뢉삨뜚a̴̷़b묃턄팟쁷힝;걵벐뇭a֮ᷬ̀̕b죁𝐸떫찛볂㋠짼딒밄씢쪋켇ᄆ̴ᅥ푉홃따क़퐆뵃㋪츑"
  ),
  (
    [0xe3, 0xbf, 0xbc, 0xed, 0x8f, 0xbb, 0xc7, 0xbd, 0xed, 0x81, 0xa8, 0xeb, 0x85, 0xa8, 0xec, 0x8c, 0x94, 0xed, 0x86, 0xad, 0xe3, 0x88, 0xa0, 0xeb, 0xa7, 0x84, 0xe1, 0x80, 0xa5, 0xcc, 0xb4, 0xe1, 0x80, 0xae, 0xf0, 0x9f, 0x88, 0x98, 0xec, 0xaa, 0x8e, 0xec, 0xb5, 0x84, 0xed, 0x93, 0x8f, 0xe2, 0x84, 0xb4, 0xec, 0x9b, 0xac, 0xeb, 0xb6, 0x80, 0xf0, 0x9d, 0x9f, 0xa9, 0xe3, 0x8b, 0xa9, 0xeb, 0xa3, 0xa6, 0xec, 0xa4, 0x9d, 0xe6, 0x8e, 0xa0, 0xed, 0x98, 0xb1, 0xeb, 0x99, 0x9e, 0x61, 0xd9, 0x8c, 0xe0, 0xa3, 0xb2, 0xd9, 0x8d, 0xd8, 0x98, 0x62, 0xea, 0xb7, 0xab, 0xeb, 0xb5, 0xb6, 0xeb, 0x8f, 0xb5, 0xed, 0x88, 0xa2, 0xed, 0x9c, 0x90, 0xeb, 0x8b, 0xa7, 0xe2, 0xbf, 0x90, 0xe3, 0x89, 0xb4, 0xeb, 0xac, 0xb8, 0xe1, 0x84, 0x80, 0xea, 0xb0, 0x81, 0xf0, 0x9d, 0x97, 0xac, 0xe3, 0x89, 0x81, 0xe1, 0xba, 0xa8, 0xc4, 0x87, 0xf0, 0x9d, 0x9a, 0x96, 0xec, 0x97, 0xb2, 0xe3, 0x8f, 0x9a, 0xeb, 0xbc, 0xa6, 0xed, 0x8c, 0xb6, 0xeb, 0xb0, 0xae, 0xed, 0x9a, 0xa4, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xe1, 0x9c, 0xb4, 0xd6, 0xb0, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdc, 0xb7, 0xd6, 0x9a, 0x62, 0xed, 0x8a, 0x91, 0xec, 0x8c, 0x97, 0xeb, 0x84, 0x9f, 0x61, 0xe3, 0x80, 0xaa, 0xe0, 0xa1, 0x99, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xbc, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xb4, 0x62, 0xec, 0xb4, 0xa2, 0xed, 0x90, 0x84, 0xf0, 0x9d, 0x9a, 0x9b, 0xec, 0x92, 0xb9, 0xed, 0x8d, 0x8c, 0xeb, 0x90, 0xa6, 0xea, 0xb1, 0x84, 0xed, 0x89, 0x94, 0xec, 0xba, 0xaa, 0xec, 0x89, 0xa8, 0xeb, 0xae, 0x8e, 0xf0, 0xa4, 0xbe, 0xa1, 0xeb, 0x9d, 0x90, 0xeb, 0xb6, 0xbc, 0xed, 0x94, 0x9f, 0xed, 0x8a, 0x80, 0xeb, 0xa0, 0x96],
    "㿼폻ǽ큨녨쌔톭㈠맄ဥ̴ီ🈘쪎쵄퓏ℴ웬부𝟩㋩룦줝掠혱뙞aٌࣲٍؘb귫뵶돵툢휐닧⿐㉴문ᄀ각𝗬㉁Ẩć𝚖엲㏚뼦팶밮횤a゙्᜴ְba〪̖ܷ֚b튑쌗넟a〪࡙̖֚baཱོིུb촢퐄𝚛쒹퍌됦걄퉔캪쉨뮎𤾡띐붼픟튀렖"
  ),
  (
    [0xed, 0x87, 0xa1, 0xe4, 0x8c, 0x81, 0xeb, 0x87, 0xb5, 0xe8, 0xba, 0xab, 0x61, 0xcc, 0x96, 0xd6, 0x9a, 0xe3, 0x80, 0xad, 0xe3, 0x80, 0xae, 0x62, 0xec, 0xbe, 0x86, 0xec, 0xbd, 0xae, 0xeb, 0x8a, 0x82, 0xec, 0xa8, 0x80, 0xeb, 0xbb, 0xb0, 0xe6, 0x98, 0x93, 0xe7, 0x82, 0x99, 0xec, 0xaf, 0x83, 0x61, 0xe3, 0x82, 0x99, 0xf0, 0x90, 0xa8, 0xbf, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xed, 0x90, 0x89, 0xea, 0xb0, 0x9b, 0xeb, 0x8a, 0x92, 0x61, 0xd6, 0xae, 0xe1, 0xb3, 0x91, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x84, 0x8a, 0xed, 0x90, 0x95, 0xec, 0x8c, 0x87, 0xeb, 0x93, 0xbe, 0xec, 0xa8, 0xa7, 0xe1, 0xbc, 0x8a, 0xeb, 0xbe, 0x80, 0xeb, 0x88, 0x8b, 0xd9, 0x80, 0xd9, 0x8f, 0xd9, 0x91, 0xea, 0xb4, 0x82, 0xec, 0xb9, 0xa4, 0xec, 0xaa, 0x82, 0xec, 0x84, 0x86, 0xf0, 0xa6, 0xbe, 0xb1, 0xc8, 0x8e, 0xeb, 0xb8, 0x80, 0xe3, 0x82, 0xad, 0xe3, 0x83, 0xad, 0xe3, 0x83, 0xa1, 0xe3, 0x83, 0xbc, 0xe3, 0x83, 0x88, 0xe3, 0x83, 0xab, 0xe9, 0x86, 0xb4, 0xeb, 0x89, 0xb1, 0xe1, 0xb9, 0x8c, 0xed, 0x9d, 0x8e, 0xed, 0x8c, 0xb8, 0xec, 0x90, 0x8b, 0xec, 0x9e, 0xac, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xc4, 0x8c, 0xec, 0x9c, 0xba, 0xed, 0x99, 0xab, 0xf0, 0xa7, 0xa5, 0xa6, 0xeb, 0xbf, 0xbc, 0xe3, 0xa1, 0xa2, 0xe3, 0x80, 0x91, 0xea, 0xb0, 0xa3, 0xec, 0xb5, 0xa3, 0xeb, 0xaa, 0x83, 0xec, 0xa7, 0x8a, 0xec, 0xb2, 0xbd, 0xea, 0xbd, 0x88, 0xea, 0xb6, 0xaf, 0xeb, 0xac, 0xa7, 0xeb, 0x9c, 0x8f, 0xe5, 0xae, 0x85, 0xe5, 0x85, 0xa7, 0xe1, 0xb8, 0xb7, 0xed, 0x8b, 0xad, 0xec, 0xb5, 0x9f, 0xeb, 0xa8, 0x99, 0xeb, 0x88, 0xa5, 0xeb, 0xb9, 0x97, 0xec, 0x92, 0x9c, 0xeb, 0xac, 0xa2, 0xe6, 0x92, 0x9a, 0xeb, 0xa4, 0xad],
    "퇡䌁뇵身a̖֚〭〮b쾆콮늂쨀뻰易炙쯃a゙𐨿्ְb퐉갛늒a֮᳑̀̕b턊퐕쌇듾쨧Ἂ뾀눋ـُّ괂칤쪂섆𦾱Ȏ븀キロメートル醴뉱Ṍ흎팸쐋재̴ᆵČ윺홫𧥦뿼㡢】갣쵣몃짊첽꽈궯묧뜏宅內ḷ틭쵟먙눥빗쒜묢撚뤭"
  ),
  (
    [0xeb, 0x92, 0x95, 0xc7, 0x8e, 0xd6, 0xae, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe9, 0xbc, 0x8f, 0xe1, 0xbf, 0x86, 0xeb, 0xbb, 0xa9, 0x61, 0xd6, 0xae, 0xe1, 0xa9, 0xbb, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xaa, 0x82, 0xed, 0x92, 0x89, 0xeb, 0xae, 0xa6, 0x61, 0xf0, 0x9b, 0xb2, 0x9e, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0x61, 0xe1, 0xb7, 0x8e, 0xcc, 0x9b, 0xf0, 0x9d, 0x85, 0xb0, 0xe3, 0x80, 0xaa, 0x62, 0xeb, 0x9b, 0xa4, 0xed, 0x81, 0x92, 0xec, 0x8e, 0x85, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9d, 0x86, 0x81, 0xd6, 0x9a, 0x62, 0x28, 0x31, 0x36, 0x29, 0x23, 0xeb, 0xba, 0xa0, 0x32, 0x39, 0xeb, 0xa0, 0xb1, 0xec, 0x9c, 0xb9, 0xe1, 0x86, 0x85, 0xea, 0xb4, 0x9b, 0x61, 0xd6, 0xae, 0xe1, 0xa9, 0xba, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xb9, 0x93, 0x61, 0xd6, 0xb8, 0xd6, 0xb9, 0xd6, 0xb9, 0xd6, 0xbb, 0x62, 0x61, 0xcc, 0xb4, 0xe2, 0x83, 0xab, 0xe0, 0xa4, 0xbc, 0x62, 0xec, 0xb2, 0xa7, 0xe3, 0x81, 0xb3, 0xe1, 0xbb, 0xa3, 0xec, 0xb2, 0xb2, 0xe6, 0x92, 0x9d, 0xea, 0xbe, 0x8b, 0xec, 0xb4, 0x97, 0xe0, 0xa8, 0x97, 0xe0, 0xa8, 0xbc, 0xeb, 0x92, 0x8a, 0xd8, 0xb3, 0xd8, 0xac, 0xeb, 0xbd, 0xbb, 0xe6, 0x83, 0x87, 0xec, 0x92, 0x97, 0xeb, 0x88, 0xa9, 0xeb, 0x9a, 0x8a, 0xed, 0x83, 0xbd, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9e, 0xa3, 0x95, 0xd6, 0x9a, 0x62, 0xe5, 0xa4, 0x86, 0xec, 0xa1, 0xbc, 0xe1, 0xba, 0xae, 0x20, 0xcc, 0xa7, 0xec, 0xab, 0xa4, 0xdb, 0x90, 0xed, 0x96, 0xa1, 0xe7, 0x96, 0x8b, 0xeb, 0x8b, 0x89, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9d, 0x89, 0x83, 0xcc, 0x95, 0x62, 0xc4, 0x83, 0xea, 0xb7, 0x8a, 0xeb, 0xb7, 0x87, 0xeb, 0xa4, 0xb6, 0x61, 0xd6, 0xae, 0xd6, 0x93, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xd8, 0xb5, 0xd9, 0x84, 0xd9, 0x89, 0xed, 0x88, 0xa5, 0xeb, 0x94, 0xbf, 0xeb, 0xab, 0x9e, 0xeb, 0xb5, 0x93, 0x61, 0xd6, 0xae, 0xdb, 0x9b, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd6, 0xa7, 0xd6, 0x9a, 0x62, 0xeb, 0xac, 0x82, 0x61, 0xd6, 0xae, 0xea, 0xa3, 0xac, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe7, 0x9b, 0xa7, 0xeb, 0x93, 0xa5],
    "뒕ǎ֮̀̕b鼏ῆ뻩a֮᩻̀̕b몂풉뮦a𛲞̴़ba᷎̛𝅰〪b뛤큒쎅a〪̖𝆁֚b(16)#뺠29렱윹ᆅ괛a֮᩺̀̕b칓aָֹֹֻba̴⃫़b첧びợ첲撝꾋촗ਗ਼뒊سج뽻惇쒗눩뚊탽a〪̖𞣕֚b夆졼Ắ ̧쫤ې햡疋닉à֮𝉃̕bă귊뷇뤶a֮֓̀̕bصلى툥딿뫞뵓a֮ۛ̀̕ba〪̖֧֚b묂a֮꣬̀̕b盧듥"
  ),
]

let codeUnitNormalizationTestsC: [([UInt8], String)] = [
  (
    [0xc4, 0xbc, 0xec, 0x84, 0xa2, 0xe2, 0xbd, 0x94, 0xc8, 0x86, 0xf0, 0x9d, 0x9d, 0xb1, 0xea, 0xb7, 0x89, 0xec, 0xa0, 0xb8, 0xec, 0xa9, 0x86, 0xd9, 0x81, 0xef, 0xb0, 0xb4, 0xec, 0xb3, 0xb8, 0xec, 0xb5, 0x92, 0xed, 0x8e, 0x81, 0xf0, 0x9f, 0x89, 0x91, 0xea, 0xb4, 0xbe, 0xe3, 0x8e, 0xba, 0xed, 0x90, 0x91, 0xea, 0xbd, 0xbe, 0xed, 0x84, 0xbb, 0xe7, 0xb4, 0x90, 0xeb, 0xb8, 0x8d, 0xf0, 0x9d, 0x94, 0x8f, 0xeb, 0x95, 0x91, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0xaa, 0x99, 0xd6, 0xb0, 0x62, 0xf0, 0x9d, 0x94, 0x96, 0xeb, 0xb4, 0x85, 0xec, 0xb9, 0x99, 0xec, 0xa9, 0x8e, 0xea, 0xb5, 0xa6, 0xef, 0xb3, 0x93, 0xe2, 0x80, 0xb4, 0xe1, 0xb8, 0x92, 0xe2, 0xbd, 0xaf, 0xf0, 0x9d, 0x9b, 0x94, 0xec, 0x9e, 0xb4, 0xe3, 0x8c, 0xb2, 0xe1, 0xba, 0xa0, 0xe1, 0xbb, 0x98, 0xf0, 0x9d, 0x98, 0x84, 0xec, 0x9e, 0xaa, 0xec, 0x95, 0xba, 0xed, 0x9b, 0xad, 0xec, 0x94, 0xb8, 0xed, 0x8f, 0x85, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0xab, 0xcc, 0x95, 0x62, 0xec, 0x9f, 0x9a, 0xec, 0xab, 0x9b, 0xef, 0xb3, 0xb3, 0xe1, 0x84, 0xad, 0xf0, 0x9e, 0xba, 0x95, 0x6d, 0x41, 0xed, 0x87, 0xad, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0xa0, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xef, 0xbd, 0xb5, 0xeb, 0x82, 0x9f, 0xf0, 0xa7, 0x99, 0xa7, 0xef, 0xac, 0x80, 0xeb, 0x85, 0x92, 0xeb, 0xbb, 0xb2, 0xec, 0xab, 0x9e, 0xe4, 0x84, 0xaf, 0xeb, 0xad, 0x92, 0xef, 0xb2, 0xb0, 0xf0, 0x9e, 0xb9, 0xa8, 0xe8, 0xa7, 0x92, 0xeb, 0xb8, 0x98, 0xeb, 0xb0, 0xb1, 0xeb, 0x91, 0x8d, 0xeb, 0x89, 0xb5, 0xec, 0x80, 0x96],
    "ļ섢⽔Ȇ𝝱귉져쩆فﰴ쳸쵒펁🉑괾㎺퐑꽾턻紐븍𝔏땑a゙्𑪙ְb𝔖봅칙쩎굦ﳓ‴Ḓ⽯𝛔잴㌲ẠỘ𝘄잪앺훭씸폅à֮ࠫ̕b쟚쫛ﳳᄭ𞺕mA퇭a〪̠̖֚bｵ낟𧙧ﬀ녒뻲쫞䄯뭒ﲰ𞹨角븘백둍뉵쀖"
  ),
  (
    [0xeb, 0x95, 0xbf, 0xeb, 0x9c, 0x88, 0xf0, 0x9d, 0x99, 0xa7, 0xe3, 0x8f, 0xaa, 0xf0, 0x9d, 0x95, 0x92, 0xeb, 0x85, 0x85, 0xe8, 0xb7, 0xb0, 0xe2, 0x88, 0x8c, 0xed, 0x96, 0xb5, 0xeb, 0x85, 0x99, 0xef, 0xb6, 0xa8, 0xec, 0x96, 0x98, 0xf0, 0x9d, 0x99, 0x8e, 0xed, 0x94, 0xbb, 0xe3, 0x81, 0x94, 0xeb, 0xba, 0x98, 0xec, 0x8c, 0x83, 0xed, 0x99, 0xbc, 0xf0, 0x9d, 0x99, 0x92, 0xea, 0xb8, 0x8a, 0xec, 0xad, 0xbe, 0xeb, 0xa1, 0x8f, 0xef, 0xb6, 0x8c, 0xef, 0xbb, 0xb8, 0xe2, 0xbd, 0xa8, 0xeb, 0xba, 0xa3, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xb3, 0x95, 0xd6, 0x9a, 0x62, 0xf0, 0x9d, 0x86, 0xb9, 0xf0, 0x9d, 0x85, 0xa5, 0xf0, 0x9d, 0x85, 0xae, 0xec, 0xae, 0xa7, 0xef, 0xbd, 0xa7, 0xe3, 0x81, 0xb1, 0xf0, 0x91, 0x92, 0xbe, 0xe1, 0xbe, 0x88, 0xeb, 0xbe, 0xb5, 0xc3, 0xa0, 0xd6, 0xae, 0xef, 0xb8, 0xae, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa7, 0x8d, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xeb, 0x9c, 0xab, 0xef, 0xad, 0xac, 0xec, 0xa2, 0xbc, 0xec, 0x84, 0xa1, 0xec, 0x91, 0xb7, 0xea, 0xb9, 0x90, 0xeb, 0xba, 0x81, 0x61, 0xe3, 0x82, 0x99, 0xea, 0xa7, 0x80, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xed, 0x94, 0xa1, 0xea, 0xba, 0x9d, 0xec, 0xaf, 0x87, 0xeb, 0x81, 0xb6, 0xef, 0xb0, 0xb8, 0xed, 0x96, 0xab, 0xec, 0xb1, 0x8c, 0xf0, 0xa5, 0x98, 0xa6, 0xeb, 0x87, 0xbc, 0xec, 0xb1, 0x87, 0xec, 0x9a, 0x92, 0xf0, 0x9d, 0x93, 0xa0, 0xea, 0xb1, 0x8c, 0xeb, 0x80, 0x8d, 0xe2, 0xbe, 0x9f, 0xe2, 0x93, 0x86, 0xe3, 0xa3, 0x87, 0xef, 0xbd, 0x83, 0xec, 0xaf, 0xa5, 0xed, 0x94, 0xae, 0xed, 0x9e, 0x9f, 0xe2, 0x93, 0xaa, 0xec, 0x9d, 0xaa, 0xec, 0xb0, 0x8d, 0xec, 0xb2, 0x97, 0xeb, 0xae, 0x8e, 0xe6, 0xb6, 0x85, 0xe3, 0x8c, 0xae, 0xed, 0x90, 0x82, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x81, 0xcc, 0x95, 0x62, 0xec, 0x8b, 0xa1, 0xec, 0x91, 0xb4, 0xea, 0xb0, 0x8f, 0xeb, 0xa0, 0xbc, 0xef, 0xb3, 0x9c, 0xeb, 0xbc, 0x80],
    "땿뜈𝙧㏪𝕒녅跰∌햵녙ﶨ얘𝙎픻ご뺘쌃홼𝙒긊쭾롏ﶌﻸ⽨뺣a〪̖᳕֚b𝆹𝅥𝅮쮧ｧぱ𑒾ᾈ뾵à֮︮̕ba゙্्ְb뜫ﭬ좼섡쑷깐뺁a゙꧀्ְb픡꺝쯇끶ﰸ햫챌𥘦뇼챇욒𝓠걌뀍⾟Ⓠ㣇ｃ쯥픮힟⓪읪찍첗뮎涅㌮퐂à֮𞀁̕b싡쑴갏렼ﳜ뼀"
  ),
  (
    [0xeb, 0xad, 0x88, 0xeb, 0x9f, 0xba, 0xe3, 0x82, 0xba, 0xeb, 0x92, 0x81, 0xeb, 0xa0, 0x89, 0xf0, 0x9e, 0xb8, 0x8e, 0xeb, 0xb7, 0xb5, 0x61, 0xcc, 0xb4, 0xe1, 0xb3, 0xa2, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0xa9, 0xa7, 0xec, 0xad, 0x89, 0xec, 0x94, 0x90, 0xf0, 0x9d, 0x9f, 0xb9, 0xec, 0x8a, 0x9a, 0x61, 0xcc, 0xb4, 0xf0, 0x91, 0x9a, 0xb7, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xeb, 0xb6, 0xa3, 0xeb, 0xb7, 0xa1, 0xef, 0xba, 0x97, 0xe9, 0x83, 0x9e, 0xef, 0xbd, 0x99, 0xec, 0x99, 0x8d, 0xec, 0x9f, 0x95, 0xf0, 0x9e, 0xba, 0xae, 0xeb, 0x9f, 0x85, 0xeb, 0x90, 0xa8, 0xea, 0xb9, 0x88, 0xec, 0xad, 0xa4, 0xed, 0x80, 0xb9, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x9c, 0xcc, 0x95, 0x62, 0xeb, 0x83, 0xb2, 0xeb, 0xad, 0x97, 0xeb, 0xb2, 0x8a, 0xe2, 0x82, 0x95, 0xeb, 0x8e, 0x86, 0xeb, 0xa9, 0x93, 0xef, 0xb5, 0x9a, 0xec, 0xac, 0x85, 0xe1, 0xb8, 0x94, 0xe3, 0x85, 0x8d, 0xec, 0xb7, 0xae, 0xf0, 0x9e, 0xba, 0x82, 0xeb, 0xb6, 0x99, 0xe6, 0x99, 0xb4, 0xe5, 0x83, 0xa7, 0xeb, 0xb3, 0xbf, 0xed, 0x8b, 0xa9, 0xe2, 0x82, 0x9a, 0xec, 0x88, 0x9f, 0x61, 0xcd, 0x9c, 0xcd, 0x9e, 0xcd, 0x9d, 0xcd, 0x85, 0x62, 0x61, 0xd6, 0xae, 0xd6, 0x95, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xba, 0xa9, 0xe2, 0xbe, 0xa7, 0xeb, 0xbb, 0x91, 0xec, 0xaf, 0x9a, 0xeb, 0x89, 0x9e, 0xef, 0xbb, 0x82, 0xf0, 0x9d, 0x96, 0xaf, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcd, 0x89, 0xd6, 0x9a, 0x62, 0xeb, 0xa3, 0x87, 0xe1, 0xbb, 0x9d, 0xce, 0x89, 0xed, 0x9b, 0x91, 0xe1, 0xbb, 0xaa, 0xf0, 0x9d, 0x9a, 0xbb, 0xe7, 0x9b, 0x9b, 0xed, 0x85, 0xa1, 0xeb, 0x83, 0x80, 0xec, 0x90, 0x99, 0xec, 0x9c, 0x8b, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0xa2, 0xcc, 0x95, 0x62, 0xeb, 0xa7, 0xb8, 0xec, 0x88, 0xb1, 0xef, 0xb1, 0x87, 0xeb, 0x93, 0x85, 0xcf, 0x8b, 0xec, 0x86, 0x8c, 0xec, 0x83, 0x82, 0xf0, 0x9d, 0x9c, 0xbb, 0xe0, 0xad, 0x8c, 0xec, 0x84, 0xb1, 0xeb, 0x81, 0x85],
    "뭈럺ズ뒁렉𞸎뷵a̴᳢़b멧쭉씐𝟹슚a̴𑚷़゙b붣뷡ﺗ郞ｙ왍쟕𞺮럅됨깈쭤퀹à֮𞀜̕b냲뭗벊ₕ뎆멓ﵚ쬅Ḕㅍ췮𞺂붙晴僧볿틩ₚ숟a͜͞͝ͅba֮֕̀̕bﺩ⾧뻑쯚뉞ﻂ𝖯a〪̖͉֚b룇ờΉ훑Ừ𝚻盛텡냀쐙윋à֮ࠢ̕b맸숱ﱇ듅ϋ소샂𝜻ୌ성끅"
  ),
  (
    [0xeb, 0x94, 0x81, 0xeb, 0x90, 0x95, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0x82, 0x8d, 0xd6, 0x9a, 0x62, 0x28, 0x6c, 0x29, 0xec, 0xba, 0xa3, 0xec, 0x95, 0x83, 0xed, 0x92, 0x8b, 0xec, 0x86, 0x90, 0xec, 0x80, 0xa4, 0xc4, 0x93, 0xec, 0xaf, 0x83, 0xeb, 0xa7, 0xac, 0xec, 0xa2, 0xb0, 0xec, 0x90, 0x83, 0xec, 0xb1, 0x9f, 0xeb, 0xb7, 0xb0, 0xec, 0xa3, 0xb4, 0x37, 0x2c, 0xeb, 0xa6, 0xb1, 0xea, 0xb0, 0xb7, 0x61, 0xd6, 0xae, 0xd8, 0x91, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x80, 0x95, 0xeb, 0x8f, 0x9d, 0xe7, 0xaa, 0xb1, 0xeb, 0xa0, 0x9f, 0xed, 0x89, 0x83, 0xea, 0xb5, 0xba, 0xec, 0xb1, 0x84, 0xec, 0xb9, 0xab, 0xeb, 0xb4, 0x8f, 0x31, 0x31, 0xe6, 0x97, 0xa5, 0xea, 0xb6, 0x9d, 0xeb, 0x9b, 0xb0, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x8f, 0xcc, 0x95, 0x62, 0xec, 0x91, 0x8e, 0xc9, 0xb8, 0xea, 0xb1, 0xb5, 0xec, 0xaf, 0x86, 0xe9, 0xa3, 0xa2, 0xec, 0x8f, 0x83, 0xec, 0x96, 0x88, 0xe8, 0xb5, 0xb7, 0xeb, 0x84, 0x9c, 0xe1, 0xb9, 0xbb, 0xec, 0x89, 0xb1, 0xed, 0x92, 0xb9, 0xea, 0xb3, 0x97, 0x31, 0x36, 0xed, 0x86, 0xbb, 0x61, 0xd6, 0xae, 0xdd, 0x83, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x88, 0x82, 0xed, 0x93, 0xbe, 0xe4, 0xac, 0xb3, 0x32, 0x36, 0xec, 0xbf, 0xb9, 0xed, 0x82, 0x8b, 0xeb, 0xa4, 0xa5, 0xe1, 0xbb, 0xab, 0xeb, 0x94, 0xb4, 0xea, 0xbb, 0x96, 0xe6, 0xae, 0x9f, 0xeb, 0x92, 0xa4, 0x61, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xab, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0x80, 0x97, 0xf0, 0xa1, 0xb7, 0xa4, 0x28, 0xe1, 0x84, 0x86, 0x29, 0xeb, 0x81, 0xae, 0xec, 0x82, 0x84, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x9d, 0x86, 0x81, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xeb, 0x95, 0x90, 0xed, 0x9b, 0xb4, 0xec, 0x9d, 0x80, 0xed, 0x81, 0xb3, 0xe1, 0xb8, 0x9d, 0xeb, 0x99, 0xa0, 0xec, 0x80, 0xba, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xaa, 0xb1, 0xcc, 0x95, 0x62, 0xed, 0x93, 0x9a, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd9, 0x9f, 0xd6, 0x9a, 0x62, 0xec, 0xab, 0x8b],
    "딁됕a〪̖ႍ֚b(l)캣앃풋손쀤ē쯃맬좰쐃챟뷰죴7,릱갷a֮ؑ̀̕b뀕돝窱렟퉃굺채칫봏11日궝뛰̴ᆼà֮𞀏̕b쑎ɸ걵쯆飢쏃얈起넜ṻ쉱풹곗16톻a֮݃̀̕b눂퓾䬳26쿹킋뤥ừ딴껖殟뒤a֮𑍫̀̕b쀗𡷤(ᄆ)끮삄a〪𝆁̖֚b땐훴은큳ḝ뙠쀺à֮᪱̕b퓚a〪̖ٟ֚b쫋"
  ),
  (
    [0x61, 0xe3, 0x80, 0xaa, 0xea, 0xa4, 0xac, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xea, 0xbf, 0xb4, 0xec, 0x9b, 0x92, 0xeb, 0x9a, 0x86, 0xec, 0xa1, 0xa1, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0x98, 0xcc, 0x95, 0x62, 0xed, 0x8d, 0xbb, 0xec, 0xbc, 0xa9, 0xe3, 0x83, 0x8e, 0xeb, 0xb4, 0xa7, 0xe6, 0x89, 0x9d, 0xeb, 0x9c, 0x94, 0xed, 0x9c, 0x85, 0xeb, 0x95, 0x83, 0x61, 0xcd, 0x9c, 0xcd, 0x9d, 0xcd, 0xa0, 0xcd, 0x85, 0x62, 0xe5, 0x90, 0x88, 0xec, 0xb4, 0xad, 0xeb, 0xbe, 0xa9, 0xec, 0x9a, 0x8a, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x89, 0xcc, 0x95, 0x62, 0xeb, 0x9a, 0x91, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xea, 0xa4, 0xab, 0xd6, 0x9a, 0x62, 0xc9, 0xab, 0xeb, 0xb7, 0xaf, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xbc, 0xe0, 0xbd, 0xb4, 0x62, 0x61, 0xd6, 0xae, 0xcc, 0x85, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0x88, 0xb8, 0xed, 0x8e, 0xb8, 0x33, 0x36, 0xec, 0xa2, 0xab, 0xed, 0x9c, 0xac, 0xec, 0x8e, 0x9a, 0xeb, 0xb6, 0xb4, 0xeb, 0x8e, 0x80, 0xed, 0x88, 0xbc, 0xeb, 0xbf, 0x83, 0xed, 0x9c, 0x9f, 0xf0, 0xaa, 0x8a, 0x91, 0xeb, 0xb3, 0xaf, 0x28, 0xe5, 0x8d, 0x81, 0x29, 0xec, 0x8c, 0xb4, 0xec, 0xb6, 0x8d, 0xe3, 0x80, 0x8b, 0xeb, 0x8d, 0x94, 0xec, 0xbf, 0x9e, 0xeb, 0x87, 0xae, 0xed, 0x83, 0xb0, 0xec, 0xa7, 0x86, 0xc3, 0xa0, 0xd6, 0xae, 0xea, 0x99, 0xba, 0xcc, 0x95, 0x62, 0xeb, 0xbd, 0x98, 0xec, 0x86, 0x99, 0xc3, 0x89, 0xd8, 0xb6, 0xec, 0xb1, 0x89, 0xea, 0xb3, 0xab, 0xeb, 0xbc, 0xa2, 0xc3, 0xa0, 0xd6, 0xae, 0xdf, 0xab, 0xcc, 0x95, 0x62, 0xe5, 0x8f, 0xaf, 0xe9, 0xa0, 0xa9, 0xed, 0x8b, 0x87, 0xec, 0xbc, 0xb7, 0xe3, 0x83, 0xa1, 0xe3, 0x82, 0xac, 0xe3, 0x83, 0x88, 0xe3, 0x83, 0xb3, 0xed, 0x91, 0xa7, 0xeb, 0x8a, 0xa8, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xaa, 0xb3, 0xcc, 0x95, 0x62, 0xec, 0x9c, 0xad, 0xec, 0x98, 0xb2, 0x28, 0xed, 0x83, 0x80, 0x29, 0xed, 0x8a, 0xa2, 0xec, 0x99, 0x88, 0xec, 0xb9, 0x88, 0xeb, 0x86, 0xa3, 0xeb, 0x83, 0x8d, 0xec, 0xb6, 0xaf, 0xd8, 0xad, 0xd9, 0x85, 0xd9, 0x89, 0x61, 0xd6, 0xae, 0xe0, 0xa0, 0xa0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xb0, 0x96, 0xe1, 0xbb, 0xaf, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x97, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xec, 0x93, 0x99],
    "a〪꤬̖֚b꿴웒뚆졡à֮࠘̕b퍻켩ノ봧扝뜔휅땃a͜͝͠ͅb合촭뾩욊à֮̉̕b뚑a〪̖꤫֚bɫ뷯aཱིོུba֮̅̀̕b숸편36좫휬쎚붴뎀툼뿃휟𪊑볯(十)쌴춍》더쿞뇮탰짆à֮ꙺ̕b뽘솙Éض챉곫뼢à֮߫̕b可頩틇켷メガトン푧늨à֮᪳̕b윭옲(타)튢왈칈놣냍춯حمىa֮ࠠ̀̕b밖ữa〪̗̖֚b쓙"
  ),
  (
    [0xe1, 0xbd, 0xa0, 0xeb, 0x9e, 0x81, 0xec, 0x85, 0x86, 0xe8, 0xb1, 0x95, 0xed, 0x83, 0xb8, 0xeb, 0xac, 0xab, 0xf0, 0x9d, 0x9f, 0xa5, 0xf0, 0x9d, 0x93, 0x80, 0xe2, 0xbc, 0xa5, 0xed, 0x9b, 0xb8, 0xec, 0xaa, 0xb9, 0xec, 0x91, 0xb3, 0xef, 0xad, 0x8f, 0xec, 0x8a, 0xba, 0xec, 0xa0, 0xa8, 0xec, 0xb3, 0x9b, 0xec, 0x9c, 0xb9, 0xeb, 0x95, 0xa6, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x98, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xb4, 0xb4, 0xef, 0xbd, 0xb7, 0xef, 0xb4, 0xa8, 0x5a, 0xf0, 0x9d, 0x98, 0xa4, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x9c, 0xcc, 0x95, 0x62, 0xd1, 0x93, 0xe8, 0xb2, 0xa9, 0xec, 0x9e, 0x96, 0xeb, 0xa4, 0xa2, 0xeb, 0xb4, 0xa5, 0xea, 0xb4, 0xa8, 0xf0, 0x9d, 0x9a, 0xb1, 0xeb, 0x81, 0xaa, 0xeb, 0xb2, 0x88, 0xec, 0xbd, 0x86, 0xe3, 0x82, 0xad, 0xe3, 0x83, 0xad, 0xe3, 0x83, 0xa1, 0xe3, 0x83, 0xbc, 0xe3, 0x83, 0x88, 0xe3, 0x83, 0xab, 0xed, 0x98, 0x82, 0xf0, 0x9d, 0x9d, 0xa0, 0x61, 0xcc, 0xb4, 0xe1, 0xb3, 0xa5, 0xe0, 0xa4, 0xbc, 0x62, 0xeb, 0xb8, 0x97, 0xec, 0xa5, 0xad, 0xec, 0x95, 0xb6, 0xc3, 0xa0, 0xd6, 0xae, 0xea, 0xaa, 0xb8, 0xcc, 0x95, 0x62, 0xec, 0x9e, 0xbf, 0xf0, 0x9d, 0x96, 0xb1, 0xeb, 0x9a, 0xae, 0xeb, 0xb2, 0xa0, 0xea, 0xb4, 0x92, 0xeb, 0xa3, 0x8c, 0xe1, 0xb8, 0x81, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xad, 0xae, 0xcc, 0x95, 0x62, 0xed, 0x86, 0x92, 0xc8, 0xae, 0xeb, 0xa5, 0xa3, 0xef, 0xb2, 0xb9, 0x61, 0xd6, 0xae, 0xd6, 0xac, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe7, 0x9b, 0xae, 0xeb, 0x99, 0x99, 0xeb, 0x8d, 0xb4, 0xe2, 0xbe, 0xbc, 0xf0, 0xa5, 0x89, 0x89, 0xf0, 0x9d, 0x99, 0xae, 0xe3, 0x8e, 0x9a, 0xe5, 0x95, 0x95, 0xed, 0x8e, 0xb8, 0xeb, 0x97, 0x8c, 0xec, 0xb8, 0x97, 0xec, 0xa9, 0xa0, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9d, 0x86, 0x88, 0xcc, 0x95, 0x62, 0xc3, 0xa0, 0xd6, 0xae, 0xea, 0xa3, 0xac, 0xcc, 0x95, 0x62, 0xec, 0x96, 0xb8, 0xf0, 0x9d, 0x93, 0xa5, 0xea, 0xb1, 0xa3, 0xf0, 0x9d, 0x9d, 0xa6, 0xec, 0xb9, 0xb0, 0xed, 0x9d, 0xba, 0xd8, 0xac, 0xd9, 0x85, 0xe8, 0xab, 0xbe, 0xeb, 0x86, 0x94, 0xe1, 0x85, 0xb0],
    "ὠ랁셆豕탸묫𝟥𝓀⼥훸쪹쑳ﭏ슺젨쳛윹땦a֮𞀘̀̕bﴴｷﴨZ𝘤à֮𞀜̕bѓ販잖뤢봥괨𝚱끪번콆キロメートル혂𝝠a̴᳥़b븗쥭앶à֮ꪸ̕b잿𝖱뚮베괒료ḁ〪̖֚bà֮᭮̕b톒Ȯ륣ﲹa֮֬̀̕b目뙙덴⾼𥉉𝙮㎚啕편뗌츗쩠à֮𝆈̕bà֮꣬̕b언𝓥걣𝝦칰흺جم諾놔ᅰ"
  ),
  (
    [0xec, 0xa9, 0x95, 0xe2, 0xbd, 0x82, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xe1, 0x9c, 0xb4, 0xd6, 0xb0, 0x62, 0x61, 0xd6, 0xae, 0xcd, 0x97, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x85, 0x83, 0xec, 0xaf, 0xb8, 0xe1, 0xbd, 0x9b, 0xe1, 0xbb, 0xad, 0xed, 0x94, 0xa6, 0xec, 0xad, 0x9c, 0xe7, 0x8b, 0xbc, 0x61, 0xd6, 0xae, 0xe1, 0xad, 0xb0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x96, 0x9a, 0xef, 0xb5, 0xb4, 0xeb, 0x82, 0xb4, 0xeb, 0xbb, 0x97, 0xec, 0xa4, 0xa8, 0xe1, 0xbc, 0xbd, 0xeb, 0x8a, 0xa3, 0xef, 0xbb, 0x97, 0xeb, 0xa2, 0x92, 0xec, 0xa8, 0x8a, 0xeb, 0xab, 0x85, 0xeb, 0xb9, 0x99, 0xc3, 0xa0, 0xd6, 0xae, 0xd6, 0xa0, 0xcc, 0x95, 0x62, 0xed, 0x80, 0x8a, 0xc3, 0xa0, 0xd6, 0xae, 0xcd, 0xa3, 0xcc, 0x95, 0x62, 0xec, 0x9a, 0xa5, 0xed, 0x92, 0x88, 0xed, 0x8a, 0x90, 0xe8, 0xa1, 0x8c, 0xeb, 0xa9, 0xa2, 0xeb, 0xa3, 0x89, 0xed, 0x87, 0xa4, 0xeb, 0x98, 0x95, 0xec, 0xb1, 0x85, 0xf0, 0x9f, 0x85, 0x8b, 0xeb, 0xb7, 0x96, 0xec, 0x8b, 0x9f, 0xec, 0x9a, 0x96, 0xe7, 0xbe, 0x85, 0xeb, 0xa4, 0xa1, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xaa, 0xb5, 0xd6, 0x9a, 0x62, 0xeb, 0x81, 0x91, 0xe1, 0xbc, 0x83, 0xf0, 0xa1, 0x9b, 0xaa, 0xeb, 0xa9, 0x8d, 0xed, 0x96, 0x96, 0xed, 0x8b, 0xb3, 0xec, 0x8f, 0x80, 0x61, 0xd6, 0xae, 0xdb, 0xa0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xa0, 0xa7, 0xeb, 0x89, 0x97, 0xe3, 0x86, 0x88, 0xe6, 0x8b, 0x93, 0xec, 0xaf, 0xb6, 0xed, 0x83, 0xbe, 0xea, 0xb5, 0x92, 0xef, 0xbd, 0xaa, 0xe3, 0x85, 0xa8, 0xe1, 0xbb, 0x90, 0xec, 0x98, 0x94, 0xeb, 0xb3, 0xa1, 0x61, 0xcc, 0xb4, 0xf0, 0x9e, 0xa5, 0x8a, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xec, 0x8b, 0xb0, 0xea, 0xb3, 0x9e, 0xea, 0xb6, 0x82, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0xa8, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xd6, 0xae, 0xf0, 0x91, 0x84, 0x80, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xac, 0xa8, 0xec, 0x8f, 0xaf, 0xed, 0x98, 0xa8, 0xeb, 0xba, 0x92, 0xe3, 0x8f, 0xa0, 0xe1, 0xba, 0xa1, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xea, 0xb8, 0x93, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdc, 0xb8, 0xd6, 0x9a, 0x62, 0xed, 0x83, 0xae, 0xec, 0xb4, 0xb1, 0xeb, 0x9e, 0x85, 0xe3, 0xb1, 0x8e, 0xeb, 0x93, 0xb1, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0x99, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9d, 0x85, 0xbf, 0xd6, 0x9a, 0x62, 0xc8, 0xa8, 0xec, 0x96, 0xae, 0xe1, 0xb8, 0xb0, 0xed, 0x90, 0xba, 0xe3, 0x8e, 0x88, 0xed, 0x81, 0xb9],
    "쩕⽂a゙्᜴ְba֮͗̀̕b🅃쯸Ὓử픦쭜狼a֮᭰̀̕b햚ﵴ내뻗줨Ἵ늣ﻗ뢒쨊뫅빙à֮֠̕b퀊à֮ͣ̕b욥품튐行멢룉퇤똕책🅋뷖싟욖羅뤡a〪̖᪵֚b끑ἃ𡛪멍햖틳쏀a֮۠̀̕b젧뉗ㆈ拓쯶탾굒ｪㅨỐ옔볡a̴𞥊़゙b싰곞궂a֮ࣨ̀̕ba֮𑄀̀̕bﬨ쏯혨뺒㏠ạ〪̖֚b긓a〪̖ܸ֚b탮촱랅㱎등à֮ᷙ̕ba〪̖𝅿֚bȨ얮Ḱ퐺㎈큹"
  ),
  (
    [0xeb, 0xac, 0x9a, 0xec, 0xa1, 0xaf, 0xed, 0x98, 0x99, 0xc3, 0xa4, 0xec, 0xba, 0x91, 0xed, 0x81, 0xa0, 0xeb, 0x97, 0xb9, 0xe5, 0x88, 0x83, 0xed, 0x87, 0x82, 0xe1, 0xbf, 0xac, 0xf0, 0x9d, 0x86, 0xb9, 0xf0, 0x9d, 0x85, 0xa5, 0xf0, 0x9d, 0x85, 0xae, 0xed, 0x87, 0xb8, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0x95, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0x96, 0x8f, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xa8, 0x97, 0xcc, 0x95, 0x62, 0xeb, 0x9b, 0x88, 0xed, 0x88, 0x84, 0xec, 0xaa, 0xab, 0xe8, 0xaa, 0xa0, 0xeb, 0xb9, 0x89, 0xeb, 0xa4, 0x8c, 0x61, 0xe1, 0xb7, 0x8e, 0xcc, 0x9b, 0xf0, 0x9d, 0x85, 0xaf, 0xe3, 0x80, 0xaa, 0x62, 0xe2, 0x8a, 0x85, 0xe1, 0xb8, 0xad, 0xec, 0x8f, 0xae, 0xe1, 0xb9, 0x95, 0xec, 0x8e, 0xa4, 0xef, 0xad, 0x9f, 0xeb, 0x8d, 0x96, 0xeb, 0xa6, 0xac, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xb3, 0x9f, 0xd6, 0x9a, 0x62, 0xea, 0xb5, 0x8c, 0xeb, 0x9d, 0x93, 0xef, 0xb9, 0x99, 0xeb, 0xb1, 0x84, 0xec, 0x8b, 0xb0, 0xeb, 0xab, 0x82, 0xf0, 0xa3, 0xa2, 0xa7, 0xec, 0xb1, 0xbb, 0xeb, 0xbe, 0x81, 0xeb, 0xa2, 0x8d, 0xec, 0x92, 0xb0, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0x83, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe3, 0x8d, 0xaf, 0xec, 0x94, 0xab, 0xec, 0xb2, 0x8b, 0xec, 0x9e, 0xb0, 0xed, 0x9e, 0xa3, 0xe2, 0x85, 0xb0, 0xf0, 0xa1, 0xb7, 0xa6, 0xed, 0x92, 0x9b, 0xf0, 0x9d, 0x9b, 0x88, 0xe2, 0x91, 0xaa, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0x86, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0xa8, 0xb4, 0xd6, 0xb0, 0x62, 0xef, 0xac, 0xa1, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x8e, 0xcc, 0x95, 0x62, 0xec, 0xbb, 0x81, 0xe8, 0xa3, 0xba, 0xeb, 0xa0, 0x82, 0xea, 0xbd, 0xbd, 0xec, 0xaa, 0xb2, 0xec, 0x96, 0x8d, 0xed, 0x8c, 0xb9, 0xed, 0x98, 0x8f, 0x61, 0xd6, 0x9a, 0xe3, 0x80, 0xaf, 0xe3, 0x80, 0xae, 0xf0, 0x9d, 0x85, 0xad, 0x62, 0xeb, 0xa5, 0x9b, 0xe1, 0xbd, 0xa6, 0xed, 0x81, 0xba, 0xeb, 0xb6, 0x8d, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0xa4, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xbe, 0x98, 0xec, 0x98, 0xa5, 0xeb, 0xac, 0x9e, 0xec, 0x83, 0xab, 0xec, 0x84, 0x92, 0xed, 0x97, 0xa5, 0xec, 0x9a, 0x93, 0xec, 0x9d, 0xa0, 0xeb, 0x82, 0xbc, 0xea, 0xbd, 0xbe, 0xeb, 0x8a, 0xa9, 0xf0, 0x9d, 0x91, 0x85, 0xec, 0x89, 0x80, 0xec, 0x8b, 0xa0, 0xed, 0x9a, 0x81, 0xec, 0x9e, 0xb0, 0xec, 0xbe, 0xb9, 0xeb, 0x85, 0x8c, 0xeb, 0x89, 0x8e],
    "묚졯혙ä캑큠뗹刃퇂Ῥ𝆹𝅥𝅮퇸a֮ࣕ̀̕b얏à֮ᨗ̕b뛈툄쪫誠빉뤌a᷎̛𝅯〪b⊅ḭ쏮ṕ쎤ﭟ덖리a〪̖᳟֚b굌띓﹙뱄싰뫂𣢧챻뾁뢍쒰a֮᷃̀̕b㍯씫첋잰힣ⅰ𡷦풛𝛈⑪à֮᷆̕ba゙्𑨴ְbﬡà֮𞀎̕b컁裺렂꽽쪲얍팹혏a֚〯〮𝅭b륛ὦ큺붍a֮ࣤ̀̕bﾘ옥묞샫섒헥욓읠낼꽾늩𝑅쉀신횁잰쾹녌뉎"
  ),
]

let codeUnitNormalizationTestsD: [([UInt8], String)] = [
  (
    [0xe3, 0x82, 0xb7, 0xea, 0xb4, 0xa4, 0x28, 0x48, 0x29, 0xe7, 0xbe, 0x95, 0xed, 0x91, 0xae, 0xc4, 0x92, 0xec, 0xb8, 0xba, 0xeb, 0xb3, 0xa8, 0xe3, 0x82, 0xbc, 0xec, 0xae, 0x9d, 0xeb, 0xa8, 0xac, 0xe6, 0xb5, 0xb8, 0xe0, 0xb3, 0x8a, 0xcc, 0xb4, 0xe0, 0xb3, 0x95, 0xce, 0x92, 0x61, 0xd6, 0xae, 0xea, 0x99, 0xb4, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe1, 0xbc, 0xa9, 0xeb, 0xbf, 0x98, 0xec, 0xb0, 0x84, 0xec, 0xab, 0x98, 0xec, 0xba, 0xa3, 0x61, 0xcc, 0xb4, 0xcc, 0xb8, 0xe0, 0xa4, 0xbc, 0x62, 0xed, 0x82, 0xb0, 0xed, 0x8f, 0xaf, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x8e, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xbd, 0xac, 0xeb, 0xa4, 0xa8, 0xc3, 0xa0, 0xd6, 0xae, 0xdc, 0xb6, 0xcc, 0x95, 0x62, 0xec, 0x9a, 0x86, 0xeb, 0x8d, 0xbf, 0xeb, 0xaf, 0xb9, 0xeb, 0xb7, 0xbf, 0xec, 0xb0, 0xae, 0xea, 0xb6, 0x84, 0x2f, 0xed, 0x96, 0xb2, 0xd8, 0xb5, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x82, 0xcc, 0x95, 0x62, 0xe7, 0x8b, 0x80, 0xeb, 0xb1, 0x99, 0xe6, 0x9e, 0x85, 0xec, 0x90, 0x82, 0xec, 0xad, 0x98, 0x61, 0xd6, 0xae, 0xe1, 0xa9, 0xbc, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x99, 0x9d, 0xec, 0xac, 0xa5, 0xe1, 0xbf, 0x83, 0xeb, 0xa5, 0xbe, 0xec, 0xbc, 0x9c, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xb2, 0xeb, 0x90, 0x94, 0xec, 0xa7, 0xb2, 0xc3, 0x8f, 0xeb, 0x98, 0x99, 0xe3, 0x82, 0xa4, 0xe3, 0x83, 0x8b, 0xe3, 0x83, 0xb3, 0xe3, 0x82, 0xb0, 0xe6, 0xac, 0xa1, 0xec, 0xba, 0xbb, 0xec, 0x81, 0xb8, 0xeb, 0xa5, 0x93, 0xeb, 0xa5, 0xae, 0xec, 0x9c, 0x86, 0xeb, 0x93, 0x81, 0xec, 0x86, 0xa7, 0xec, 0xbf, 0x97, 0xec, 0x8c, 0x90, 0xeb, 0xba, 0x98, 0x6d, 0x6d, 0xeb, 0xb0, 0xad, 0xeb, 0x80, 0x97, 0x61, 0xe3, 0x82, 0x99, 0xe1, 0xae, 0xab, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0x31, 0x31, 0xeb, 0x88, 0x82, 0xec, 0xa2, 0x82, 0xed, 0x9a, 0x9a, 0xec, 0xbe, 0x85, 0xeb, 0x86, 0x85, 0xe8, 0x88, 0x84, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd6, 0x96, 0xd6, 0x9a, 0x62, 0xec, 0xab, 0xa6, 0xc3, 0xad, 0xeb, 0xb5, 0x86, 0xeb, 0x86, 0xaa, 0xec, 0xb6, 0x86, 0xeb, 0xb9, 0xb8, 0xed, 0x9b, 0xbc, 0xe1, 0x85, 0xa7, 0xea, 0xb3, 0xbb, 0xe4, 0xbe, 0xbb, 0x31, 0x37, 0xe6, 0x97, 0xa5, 0xe1, 0x85, 0xa5, 0xe1, 0xb8, 0xa4],
    "シ괤(H)羕푮Ē츺볨ゼ쮝먬浸ೊ̴ೕΒa֮ꙴ̀̕bἩ뿘찄쫘캣a̴̸़b킰폯a֮𞀎̀̕b뽬뤨à֮ܶ̕b욆덿믹뷿찮궄/햲صà֮𞀂̕b狀뱙枅쐂쭘a֮᩼̀̕b홝쬥ῃ륾켜̴ᆼཱི됔짲Ï똙イニング次캻쁸륓륮윆듁솧쿗쌐뺘mm밭뀗a゙᮫्ְb11눂좂횚쾅놅舄a〪̖֖֚b쫦í뵆놪춆빸훼ᅧ곻侻17日ᅥḤ"
  ),
  (
    [0xed, 0x84, 0xb7, 0xec, 0x99, 0x94, 0x61, 0xe3, 0x80, 0xaa, 0xe1, 0xb3, 0x9e, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xec, 0x91, 0xb8, 0xec, 0xb9, 0xbe, 0xec, 0xa5, 0xb3, 0x61, 0xe3, 0x82, 0x99, 0xf0, 0x91, 0xb0, 0xbf, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xed, 0x96, 0xbd, 0xed, 0x9a, 0x93, 0xe7, 0x8e, 0x84, 0xea, 0xb8, 0xbc, 0xec, 0x8d, 0x99, 0xe1, 0xb9, 0x83, 0xeb, 0x82, 0x9e, 0xeb, 0xb8, 0x89, 0xeb, 0x96, 0xa5, 0xec, 0x93, 0x82, 0x61, 0xd6, 0xae, 0xea, 0xa3, 0xa1, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xea, 0xb8, 0x81, 0xe6, 0x8d, 0xa8, 0xea, 0xbe, 0x9e, 0xeb, 0x90, 0xa7, 0xe4, 0xb8, 0x83, 0xeb, 0x9e, 0x81, 0xe1, 0xb4, 0x9d, 0xec, 0xab, 0x81, 0xe6, 0x90, 0x9c, 0xe6, 0x9d, 0x9e, 0xec, 0xb4, 0x8f, 0xec, 0xaf, 0x80, 0xec, 0x94, 0x89, 0xed, 0x90, 0xba, 0xec, 0x82, 0x91, 0xed, 0x90, 0xad, 0xec, 0x9b, 0xa6, 0xeb, 0x9f, 0xbf, 0x61, 0xd6, 0xae, 0xdf, 0xae, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xaf, 0x97, 0xec, 0xa1, 0xa3, 0xeb, 0x95, 0xb7, 0xeb, 0x85, 0xad, 0xe9, 0xbe, 0x9f, 0xeb, 0x87, 0xad, 0xeb, 0xa8, 0x94, 0xec, 0x90, 0x85, 0xec, 0x9a, 0xac, 0xec, 0xbb, 0x88, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xc4, 0xa3, 0xec, 0xad, 0x93, 0xeb, 0xa1, 0xa1, 0xea, 0xb4, 0x8c, 0x50, 0x50, 0x56, 0xe9, 0x80, 0xb8, 0xec, 0x9e, 0xbf, 0xed, 0x9c, 0xaf, 0xec, 0xb6, 0x95, 0xe5, 0x9f, 0xb4, 0xeb, 0xa4, 0xad, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x84, 0xcc, 0x95, 0x62, 0xec, 0x8e, 0x91, 0xeb, 0x9e, 0xb7, 0xd9, 0x8a, 0xd8, 0xad, 0xd9, 0x8a, 0xed, 0x8d, 0xbc, 0xed, 0x81, 0xb5, 0xed, 0x80, 0x82, 0xea, 0xb5, 0xb1, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0xbb, 0xd6, 0x9a, 0x62, 0xc8, 0xb7, 0xea, 0xb9, 0xb8, 0x33, 0x33, 0xec, 0xa5, 0xa9, 0xc7, 0xa2, 0xeb, 0xb8, 0xaa, 0xea, 0xbb, 0x9a, 0xeb, 0xa6, 0xb4, 0xe6, 0xb4, 0x9b, 0xeb, 0xbc, 0xbd, 0xeb, 0x98, 0x9c, 0xe4, 0x8c, 0x81, 0xec, 0x83, 0x91, 0xeb, 0xa2, 0xa9, 0xec, 0x9d, 0x94, 0xec, 0x87, 0xb1, 0xeb, 0xb4, 0x9c, 0xea, 0xb4, 0xa3, 0xeb, 0xa7, 0xbb, 0x20, 0xd9, 0x8c, 0xe1, 0xbf, 0x83, 0xec, 0xaf, 0xaf, 0xec, 0x9e, 0x99],
    "턷왔a〪᳞̖֚b쑸칾쥳a゙𑰿्ְb햽횓玄긼썙ṃ낞븉떥쓂a֮꣡̀̕b긁捨꾞됧七랁ᴝ쫁搜杞촏쯀씉퐺삑퐭웦럿a֮߮̀̕b믗졣땷녭龟뇭먔쐅욬컈̴ᆵģ쭓롡괌PPV逸잿휯축埴뤭à֮𞀄̕b쎑랷يحي퍼큵퀂굱a〪̖̻֚bȷ깸33쥩Ǣ븪껚릴洛뼽똜䌁샑뢩읔쇱봜괣맻 ٌῃ쯯잙"
  ),
  (
    [0xef, 0xbe, 0x91, 0xed, 0x89, 0xa0, 0xf0, 0x9d, 0x9f, 0xa9, 0xe2, 0x84, 0x96, 0xeb, 0xb8, 0xa2, 0x44, 0x5a, 0xf0, 0x9d, 0x9b, 0xb2, 0xe3, 0x82, 0xb3, 0xe3, 0x83, 0x88, 0xef, 0xb7, 0xb4, 0xeb, 0xbc, 0x9a, 0xea, 0xbb, 0x93, 0xed, 0x96, 0xa6, 0xf0, 0x9d, 0x96, 0xb3, 0x50, 0x52, 0xd9, 0x87, 0xec, 0xae, 0xaa, 0xed, 0x81, 0x90, 0xec, 0x8c, 0xb2, 0xeb, 0xab, 0x8a, 0xeb, 0x8d, 0xab, 0xe8, 0x82, 0x89, 0xe3, 0x83, 0xb3, 0xec, 0x94, 0x82, 0xec, 0xbd, 0x9f, 0xec, 0x9f, 0xb1, 0xec, 0x8a, 0xb5, 0xea, 0xb9, 0x8e, 0xe3, 0x8a, 0xaf, 0xeb, 0xbf, 0x8f, 0xeb, 0xa6, 0x88, 0xe8, 0x87, 0xa8, 0xeb, 0x9b, 0xac, 0xc7, 0xb9, 0xeb, 0x92, 0x9d, 0xe3, 0x8f, 0x85, 0xec, 0xba, 0xb4, 0x33, 0x36, 0xed, 0x8b, 0x9a, 0xe8, 0xbb, 0x94, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa3, 0xb8, 0xcc, 0x95, 0x62, 0xed, 0x99, 0x8d, 0xd9, 0x89, 0xe1, 0xb6, 0x9e, 0xed, 0x87, 0x93, 0xed, 0x89, 0xbf, 0xe1, 0x87, 0x8c, 0xe0, 0xbe, 0x92, 0xe0, 0xbe, 0xb7, 0xeb, 0xa2, 0x9f, 0x61, 0xe3, 0x82, 0x99, 0xe1, 0x9c, 0x94, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xeb, 0x9d, 0x9d, 0xe1, 0xbf, 0x8a, 0xec, 0xb9, 0xb5, 0xed, 0x89, 0x92, 0xe4, 0xba, 0x8c, 0xe8, 0x98, 0xbf, 0xeb, 0x97, 0xac, 0xeb, 0xb8, 0x87, 0xeb, 0x97, 0xa3, 0xc3, 0xa0, 0xd6, 0xae, 0xdd, 0x83, 0xcc, 0x95, 0x62, 0xec, 0xb3, 0xab, 0xda, 0xb1, 0xeb, 0x9a, 0xa2, 0xec, 0xaa, 0x85, 0xe5, 0x87, 0x9c, 0x49, 0x58, 0xef, 0xbf, 0x9a, 0xec, 0x9f, 0xa8, 0xed, 0x81, 0xa5, 0xf0, 0x9e, 0xb8, 0xb9, 0xf0, 0x9d, 0x91, 0x90, 0xe1, 0xb8, 0x84, 0x43, 0xeb, 0x87, 0x83, 0xf0, 0x9d, 0x9d, 0x8c, 0xec, 0xae, 0xa7, 0xec, 0x87, 0x83, 0xef, 0xb5, 0x98, 0xeb, 0xae, 0x94, 0xeb, 0x84, 0xbd, 0xef, 0xb8, 0xb1, 0xf0, 0x9d, 0x93, 0x81, 0x61, 0xe3, 0x80, 0xaa, 0xd6, 0xaa, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xeb, 0x86, 0x9a, 0xec, 0xb2, 0x87, 0xec, 0x95, 0x8d, 0xc3, 0xa0, 0xd6, 0xae, 0xd6, 0x93, 0xcc, 0x95, 0x62, 0xe3, 0x88, 0xba, 0xec, 0x83, 0xb9, 0xf0, 0x9d, 0x9f, 0xba, 0x61, 0xcc, 0xb4, 0xf0, 0x90, 0xa8, 0xb9, 0xe0, 0xa4, 0xbc, 0x62],
    "ﾑ퉠𝟩№븢DZ𝛲コトﷴ뼚껓햦𝖳PRه쮪큐쌲뫊덫肉ン씂콟쟱습깎㊯뿏릈臨뛬ǹ뒝㏅캴36틚軔à֮ࣸ̕b홍ىᶞ퇓퉿ᇌྒྷ뢟a゙᜔्ְb띝Ὴ칵퉒二蘿뗬븇뗣à֮݃̕b쳫ڱ뚢쪅凜IXￚ쟨큥𞸹𝑐ḄC뇃𝝌쮧쇃ﵘ뮔넽︱𝓁a〪֪̖֚b놚첇앍à֮֓̕b㈺샹𝟺a̴𐨹़b"
  ),
  (
    [0xe1, 0xb6, 0xb3, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0x95, 0x9f, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xf0, 0x9e, 0xb9, 0xaf, 0x6f, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x8c, 0x99, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xc3, 0xa0, 0xd6, 0xae, 0xd8, 0x93, 0xcc, 0x95, 0x62, 0x69, 0xed, 0x87, 0x80, 0x69, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe3, 0x85, 0xba, 0x69, 0xf0, 0x9f, 0xa4, 0x94, 0xe3, 0x8a, 0xaa, 0xc3, 0xa0, 0xd6, 0xae, 0xcd, 0x97, 0xcc, 0x95, 0x62, 0xef, 0xb2, 0xb7, 0x69],
    "ᶳ🤔앟abcdefg🌺🌸𞹯o😂댙🌺🌸à֮ؓ̕bi퇀i🇨🇦ㅺi🤔㊪à֮͗̕bﲷi"
  ),
  (
    [0xec, 0x9d, 0x82, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xac, 0x8f, 0x6f, 0xec, 0xb1, 0x94, 0x75, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x96, 0x9f, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xed, 0x92, 0xb0, 0xea, 0xad, 0x9e, 0xeb, 0x8b, 0xb4, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0xbc, 0x83, 0x75, 0xf0, 0x9f, 0x98, 0x82, 0xf0, 0x9d, 0x95, 0xb8, 0x61, 0xf0, 0x9f, 0x98, 0x82, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x95, 0xcc, 0x95, 0x62, 0x69],
    "읂😂묏o챔u🌺🌸얟🌺🌸풰ꭞ담🇨🇦켃u😂𝕸a😂à֮𞀕̕bi"
  ),
  (
    [0xec, 0x9d, 0x86, 0xec, 0x8d, 0x91, 0x61, 0xeb, 0x98, 0xa1, 0xeb, 0xba, 0xa6, 0x6f, 0xeb, 0x8f, 0xba, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0xa9, 0xbb, 0xed, 0x90, 0xbe, 0x61, 0xe1, 0xb3, 0xa6, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0x65, 0xc3, 0xa0, 0xd6, 0xae, 0xef, 0xb8, 0xa2, 0xcc, 0x95, 0x62, 0xe3, 0x83, 0x9e, 0xe3, 0x83, 0x83, 0xe3, 0x83, 0x8f],
    "읆썑a똡뺦o돺hello쩻퐾a᳦̴़beà֮︢̕bマッハ"
  ),
  (
    [0xeb, 0x95, 0x9d, 0xe2, 0x9a, 0x89, 0xed, 0x8e, 0x9f, 0xf0, 0x9f, 0x98, 0x82, 0xc5, 0x86, 0xec, 0x91, 0xbe, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x81, 0x8d, 0xe2, 0x9a, 0x89, 0xec, 0xa0, 0xa6, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xc3, 0xa0, 0xd6, 0xae, 0xdc, 0xba, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xea, 0xb4, 0xbb, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0xb1, 0xb7, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x83, 0x8a, 0xe2, 0x9a, 0x89],
    "땝⚉펟😂ņ쑾🇨🇦쁍⚉젦👨‍👩‍👧à֮ܺ̕b👨‍👩‍👧괻🇨🇦뱷🌺🌸냊⚉"
  ),
  (
    [0xeb, 0xb2, 0xab, 0xeb, 0xbe, 0xb5, 0xed, 0x80, 0x8e, 0xf0, 0xa8, 0x97, 0xad, 0xed, 0x85, 0x86, 0xe2, 0x9a, 0x89, 0xe3, 0x86, 0x96, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x9e, 0x95, 0xf0, 0x9d, 0x9b, 0x9d, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x88, 0x99, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0xa0, 0x98],
    "벫뾵퀎𨗭텆⚉㆖🇨🇦잕𝛝🇨🇦숙🌺🌸젘"
  ),
  (
    [0xeb, 0x9c, 0x89, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xad, 0xac, 0xd6, 0x9a, 0x62, 0xeb, 0x8b, 0xa6, 0x69, 0xeb, 0xac, 0x9b, 0xe2, 0x9a, 0x89, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xbf, 0xb6, 0x6f, 0xe7, 0xab, 0x8b, 0xed, 0x9d, 0x8a, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x88, 0xa9, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0xa5, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0xb5, 0xcc, 0x95, 0x62, 0x75, 0xed, 0x9a, 0xa6, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe7, 0xab, 0x8b, 0xed, 0x9c, 0xb4, 0x69, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0xac, 0x96, 0x61, 0xeb, 0x8d, 0x8c, 0xeb, 0x80, 0xbf, 0x75, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0xb9, 0x84, 0x61, 0xf0, 0x9d, 0x93, 0xb6, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x88, 0xa9, 0xe3, 0x8a, 0x87, 0xe3, 0x89, 0x9c, 0xe2, 0x9a, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xae, 0xb7, 0x61, 0xf0, 0x9f, 0xa4, 0x94, 0xe1, 0xb8, 0xbd, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0xba, 0xa6, 0xe2, 0xbe, 0xa5, 0x75],
    "뜉🌺🌸a〪̖᭬֚b닦i묛⚉在线服务和个쿶o立흊hello🌺🌸利a֮ᷥ̀̕b在线服务和个à֮᷵̕bu횦🇨🇦立휴i🤔쬖a덌뀿u👨‍👩‍👧칄a𝓶🤔利㊇㉜⚉拥有쮷a🤔ḽabcdefg🌺🌸度⾥u"
  ),
]

let codeUnitNormalizationTestsE: [([UInt8], String)] = [
  (
    [0xf0, 0x9d, 0x93, 0xa2, 0xec, 0xad, 0xb7, 0xea, 0xb7, 0xbc, 0xe2, 0xbf, 0x88, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xef, 0xb2, 0x98, 0xe1, 0xbb, 0xa6, 0xe5, 0x85, 0xac, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xf0, 0x9d, 0x9e, 0xaf, 0xe5, 0x88, 0xa9, 0xf0, 0x9e, 0xba, 0xb7, 0xec, 0xa6, 0xb5, 0xe5, 0x88, 0xa9, 0xec, 0x9a, 0xbf, 0xe2, 0x8a, 0x81, 0xe5, 0xba, 0xa6, 0xea, 0xb4, 0xa8, 0xe5, 0x88, 0xa9, 0xf0, 0x9d, 0x94, 0xa2, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xb2, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe7, 0xab, 0x8b, 0xec, 0xa5, 0x93, 0xe7, 0xab, 0x8b, 0xef, 0xb2, 0x97, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xe3, 0x80, 0xaa, 0xd9, 0x95, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe8, 0xab, 0x8b, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xa6, 0xcc, 0x80, 0xcc, 0x95, 0x62],
    "𝓢쭷근⿈在线服务和个ﲘỦ公a〪̖̖֚b𝞯利𞺷즵利욿⊁度괨利𝔢拥有a֮ⷲ̀̕b立쥓立ﲗ拥有a〪ٕ̖֚b請拥有a֮ⷦ̀̕b"
  ),
  (
    [0xeb, 0x8e, 0xb4, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x9a, 0x96, 0xe8, 0x8c, 0x9d, 0xe5, 0xba, 0xa6, 0xec, 0xb7, 0xa8, 0xc3, 0xa0, 0xd6, 0xae, 0xea, 0x9b, 0xb1, 0xcc, 0x95, 0x62, 0xeb, 0xb2, 0xb1, 0xec, 0xa7, 0xad, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x9d, 0x97, 0xeb, 0xb7, 0xb1, 0xe7, 0xab, 0x8b, 0x53, 0x44, 0x61, 0xe3, 0x80, 0xaa, 0xe2, 0x83, 0xad, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x4c, 0x54, 0x44, 0xc8, 0x83, 0xe5, 0x88, 0xa9, 0xeb, 0x97, 0x9a, 0xe7, 0xab, 0x8b, 0xea, 0x9c, 0xa7, 0xec, 0xbd, 0x87, 0xeb, 0x99, 0xbf, 0xe1, 0xb8, 0x9f, 0xe5, 0x85, 0xac, 0xd9, 0x87, 0xd8, 0xac, 0x7a],
    "뎴拥有횖茝度취à֮꛱̕b벱짭拥有띗뷱立SDa〪⃭̖֚bLTDȃ利뗚立ꜧ콇뙿ḟ公هجz"
  ),
  (
    [0xeb, 0x97, 0xbc, 0xe9, 0x81, 0xbc, 0x65, 0xea, 0xbf, 0xa5, 0xe5, 0xba, 0xa6, 0xec, 0x9d, 0x99, 0x6f, 0xe5, 0x85, 0xac, 0xec, 0xbd, 0x9b, 0xed, 0x92, 0x9c, 0xeb, 0x85, 0x8f, 0xe3, 0x83, 0x83, 0xea, 0xb9, 0x85, 0x4c, 0xc2, 0xb7, 0xeb, 0xae, 0x9a, 0x6f, 0xe5, 0xba, 0xa6, 0x61, 0xe3, 0x82, 0x99, 0xe1, 0x80, 0xba, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x9d, 0x8f, 0xe7, 0xab, 0x8b, 0xeb, 0xb2, 0x94, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x93, 0xb7, 0x61, 0xe0, 0xba, 0xb8, 0xe0, 0xbb, 0x89, 0xe0, 0xbb, 0x88, 0xe0, 0xbd, 0xb1, 0x62, 0xec, 0x95, 0xaf, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x9b, 0x9b, 0x6f, 0xe7, 0xab, 0x8b, 0x61, 0xe3, 0x80, 0xaa, 0xea, 0xaa, 0xb4, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xac, 0x8b, 0x6f, 0xe5, 0xba, 0xa6],
    "뗼遼e꿥度읙o公콛풜녏ッ깅L·뮚o度a゙်्ְb在线服务和个띏立범在线服务和个듷aຸ້່ཱb앯abcdefg훛o立a〪ꪴ̖֚b拥有쬋o度"
  ),
  (
    [0xea, 0xb8, 0x9a, 0xe3, 0x8c, 0x90, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xea, 0xba, 0x98, 0xe1, 0xbb, 0x8d, 0xec, 0x89, 0x9d, 0xed, 0x8e, 0xb5, 0xeb, 0xbc, 0x9e, 0xeb, 0xaf, 0xba, 0x75, 0xe3, 0x8d, 0x81, 0x75, 0xe0, 0xa4, 0xaf, 0xe0, 0xa4, 0xbc, 0xeb, 0x9e, 0x91, 0xed, 0x80, 0xa8, 0x6f, 0xed, 0x8b, 0xa5, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0x9d, 0xba, 0xef, 0xb8, 0xbd, 0xe3, 0x8f, 0xa8, 0x65, 0xf0, 0x9d, 0x99, 0x8a, 0xec, 0x96, 0xa9, 0xec, 0x9d, 0x9a, 0xeb, 0x85, 0xb0],
    "긚㌐abcdefg꺘ọ쉝펵뼞믺u㍁uय़랑퀨o틥hello띺︽㏨e𝙊얩읚녰"
  ),
  (
    [0xeb, 0x82, 0xb9, 0x65, 0xef, 0xb4, 0xb8, 0xec, 0xbb, 0x83, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe0, 0xa8, 0xab, 0xe0, 0xa8, 0xbc, 0x75, 0xeb, 0xb1, 0x9a, 0xed, 0x8f, 0x86, 0x69, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xa9, 0x9e, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xef, 0xb0, 0x95, 0xe5, 0x88, 0xa9, 0xeb, 0x9f, 0xba, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x85, 0xbf, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x85, 0xac, 0xeb, 0xa4, 0x88, 0xeb, 0xa3, 0xa6, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0xb0, 0xab, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x84, 0xbc, 0x75, 0xe1, 0xb8, 0x8a, 0xe5, 0xba, 0xa6, 0xed, 0x96, 0x80, 0x75, 0xe5, 0xba, 0xa6, 0xec, 0x86, 0xab, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x99, 0xbf, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xa1, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe1, 0xb8, 0xa0, 0x65, 0xec, 0xb2, 0xad, 0x65, 0xf0, 0x9d, 0x91, 0xad, 0xec, 0x9c, 0xbc, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0xb3, 0x96, 0xea, 0xb7, 0xaf, 0x75, 0xe5, 0x88, 0xa9, 0xeb, 0x9e, 0x91, 0xe2, 0x85, 0xb9, 0x61, 0xe5, 0x85, 0xac, 0xe2, 0x81, 0xb9, 0x6f, 0xe5, 0x88, 0xa9, 0xf0, 0x9d, 0x9c, 0xbc, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0xba, 0xa6, 0xe1, 0x84, 0x90, 0xcc, 0xb4, 0xe1, 0x85, 0xb0, 0x69, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89],
    "낹eﴸ컃在线服务和个ਫ਼u뱚폆i拥有쩞拥有ﰕ利럺拥有셿hello公뤈룦abcdefg밫o在线服务和个턼uḊ度햀u度솫abcdefg뙿a֮ⷡ̀̕bḠe청e𝑭으hello볖귯u利랑ⅹa公⁹o利𝜼hello度ᄐ̴ᅰi拥有"
  ),
  (
    [0xea, 0xb8, 0xab, 0xef, 0xb9, 0x84, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xef, 0xbe, 0xbd, 0xeb, 0xbd, 0x91, 0xe5, 0xba, 0xa6, 0xec, 0x9f, 0xb5, 0xe5, 0xba, 0xa6, 0xec, 0xb4, 0xbc, 0xf0, 0x9d, 0x90, 0x87, 0xeb, 0x83, 0xab, 0xec, 0x97, 0x86, 0xec, 0xbb, 0x88, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xef, 0xbe, 0xb6, 0xeb, 0x85, 0x8e, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9d, 0xaa, 0xe5, 0xba, 0xa6, 0xeb, 0x9b, 0x9d, 0xe5, 0xba, 0xa6, 0xe5, 0xbb, 0x99, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xef, 0xbe, 0xb3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb0, 0x85, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xae, 0x86, 0xe4, 0xb3, 0x8e, 0xe5, 0x88, 0xa9, 0xeb, 0x91, 0xa5, 0xf0, 0x9f, 0x88, 0x81, 0xe5, 0x88, 0xa9, 0xeb, 0x81, 0x95, 0xec, 0x87, 0x83, 0xe5, 0x85, 0xac, 0xef, 0xb8, 0xb2, 0xec, 0x94, 0x9e, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x88, 0xb7, 0xe5, 0x85, 0xac, 0xef, 0xb4, 0xb5, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xf0, 0x9d, 0x9a, 0x8b, 0xe5, 0x85, 0xac, 0xec, 0x9c, 0x94, 0xe5, 0x85, 0xac, 0xef, 0xb4, 0xb1, 0xe5, 0xba, 0xa6],
    "긫﹄在线服务和个ﾽ뽑度쟵度촼𝐇냫없컈̴ᆵﾶ녎在线服务和个흪度뛝度廙在线服务和个ﾳ拥有밅拥有뮆䳎利둥🈁利끕쇃公︲씞在线服务和个눷公ﴵ拥有𝚋公윔公ﴱ度"
  ),
  (
    [0xed, 0x8e, 0x98, 0xce, 0x9c, 0xed, 0x9b, 0xa0, 0xd9, 0x87, 0xeb, 0x8f, 0x85, 0xeb, 0x80, 0x82, 0xeb, 0x96, 0x89, 0xec, 0xa1, 0x96, 0xe7, 0x80, 0x9e, 0x61, 0xd6, 0xae, 0xe1, 0xaa, 0xb2, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xc3, 0x84, 0xea, 0xb6, 0x84, 0xed, 0x92, 0xbb, 0xeb, 0xb5, 0x91, 0xeb, 0x9a, 0x81, 0xd9, 0x87, 0xd8, 0xac, 0xe8, 0x88, 0x81, 0xeb, 0x95, 0xad, 0xd8, 0xa7, 0xd9, 0x8b, 0xed, 0x9a, 0xb1, 0xec, 0x80, 0x89, 0xea, 0xbe, 0x9e, 0xeb, 0x8a, 0xa2, 0xeb, 0x80, 0xbe, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9d, 0x86, 0x85, 0xcc, 0x95, 0x62, 0x21, 0xec, 0x85, 0xb0, 0xeb, 0xab, 0x96, 0xec, 0xb3, 0xa5, 0xeb, 0x95, 0xa6],
    "페Μ훠ه독뀂떉졖瀞a֮᪲̀̕bÄ궄풻뵑뚁هج舁땭اً횱쀉꾞늢뀾à֮𝆅̕b!셰뫖쳥땦"
  ),
  (
    [0xeb, 0xb1, 0xbc, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe7, 0xab, 0x8b, 0xeb, 0x80, 0xb5, 0xeb, 0xa9, 0xa2, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x85, 0xac, 0xeb, 0xb3, 0x94, 0xe2, 0x9a, 0x89, 0xe5, 0xba, 0xa6, 0xec, 0x8b, 0xa5, 0xf0, 0x9f, 0xa4, 0x94, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x28, 0xe5, 0x91, 0xbc, 0x29, 0xeb, 0xbe, 0xb5, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x9e, 0x87, 0xe5, 0x85, 0xac, 0xe9, 0xbc, 0x8f, 0xe7, 0xab, 0x8b, 0xed, 0x87, 0xb4, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe5, 0xba, 0xa6, 0xec, 0x95, 0x87, 0xf0, 0x9f, 0xa4, 0x94, 0xe7, 0xab, 0x8b, 0xed, 0x89, 0xa8, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xb4, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x9a, 0x8b, 0xec, 0x82, 0x92, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0xa5, 0xcc, 0x95, 0x62, 0xe5, 0x88, 0xa9, 0xea, 0xb0, 0xba, 0xe1, 0xba, 0x88, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x90, 0x91, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xba, 0xa8, 0xe7, 0xab, 0x8b, 0xe7, 0x93, 0xa6, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0xa1, 0xa3, 0xe3, 0x82, 0xa8, 0xe3, 0x83, 0xbc, 0xe3, 0x82, 0xab, 0xe3, 0x83, 0xbc, 0xe5, 0x88, 0xa9, 0xec, 0xb8, 0xbe, 0xe5, 0x88, 0xa9, 0xe5, 0x90, 0x88, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0x8b, 0xb3, 0xec, 0x91, 0xab, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x85, 0xac, 0xed, 0x92, 0xb3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe5, 0x85, 0xb7, 0xeb, 0xa7, 0xb5, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6],
    "뱼👨‍👩‍👧立뀵멢😂公볔⚉度싥🤔拥有(呼)뾵👨‍👩‍👧在线服务和个랇公鼏立퇴̴ᆼ👨‍👩‍👧度앇🤔立퉨à֮𑍴̕b🌺🌸욋삒à֮ࠥ̕b利갺Ẉ拥有퐑🌺🌸拥有꺨立瓦😂졣エーカー利츾利合👨‍👩‍👧닳쑫🇨🇦公풳拥有具맵🇨🇦"
  ),
]

let codeUnitNormalizationTestsF: [([UInt8], String)] = [
  (
    [0xeb, 0xae, 0x8f, 0x61, 0xeb, 0xa8, 0x91, 0xeb, 0xa3, 0xa6, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x82, 0xbf, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x90, 0x96, 0x65, 0xef, 0xb6, 0xba, 0xed, 0x8e, 0xb8, 0x61, 0xeb, 0x94, 0xad, 0xec, 0x84, 0x96, 0xc4, 0x8f, 0xe3, 0x8f, 0x82, 0x61, 0xe3, 0x80, 0xaa, 0xcd, 0x94, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xec, 0x8d, 0xbc, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0x85, 0x99, 0x69, 0xcf, 0x8b, 0xf0, 0x9d, 0x9d, 0x97, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x80, 0x9e, 0xeb, 0xb3, 0xa4, 0x65, 0xed, 0x99, 0xbe, 0x6f, 0xec, 0xaf, 0x92, 0x69, 0xef, 0xb2, 0x83, 0xe5, 0xa2, 0xb3, 0x75, 0xec, 0xaa, 0xbf, 0xe3, 0x88, 0x90, 0xe0, 0xac, 0xa1, 0xe0, 0xac, 0xbc, 0x61, 0xe1, 0xbf, 0xa7, 0xec, 0x98, 0x95, 0x65, 0x61, 0xd6, 0xae, 0xe0, 0xa0, 0x9f, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x31, 0xe6, 0x9c, 0x88, 0xf0, 0x9d, 0x9b, 0xa1, 0x65],
    "뮏a먑룦abcdefg킿abcdefg퐖eﶺ편a딭섖ď㏂a〪͔̖֚b썼hello셙iϋ𝝗abcdefg퀞볤e홾o쯒iﲃ墳u쪿㈐ଡ଼aῧ옕ea֮ࠟ̀̕b1月𝛡e"
  ),
  (
    [0x61, 0xe1, 0xb7, 0x8e, 0xe0, 0xbc, 0xb9, 0xcc, 0x9b, 0xe3, 0x80, 0xaa, 0x62, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xaf, 0x94, 0xed, 0x90, 0xa7, 0x61, 0xe7, 0xab, 0x8b, 0xec, 0x99, 0xa0, 0x6f, 0xe5, 0x85, 0xac, 0xea, 0xbe, 0xb2, 0x6f, 0xec, 0xaf, 0xa3, 0xea, 0xb7, 0x94, 0x69, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x96, 0xa3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x90, 0xae, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0xb2, 0xd6, 0x9a, 0x62, 0x6f, 0xe3, 0xa3, 0xa3, 0x6f, 0xed, 0x9c, 0xad, 0xe7, 0xab, 0x8b, 0xeb, 0x9c, 0xa2, 0x69, 0xe5, 0x85, 0xac, 0xeb, 0xa9, 0x96, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9d, 0x91, 0x87, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xbb, 0xe0, 0xbd, 0xb4, 0x62, 0x75, 0xe5, 0x85, 0xac, 0xef, 0xba, 0xad, 0xeb, 0x86, 0x92, 0xeb, 0xa8, 0x98, 0x69, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xef, 0xb8, 0x94, 0x69, 0xeb, 0xba, 0x91, 0x61, 0xe7, 0xab, 0x8b, 0xeb, 0xaf, 0x86, 0xec, 0x9e, 0xa6, 0x6f, 0xf0, 0x9d, 0x9b, 0x86, 0xe7, 0xab, 0x8b, 0xc3, 0xa0, 0xd6, 0xae, 0xef, 0xb8, 0xaf, 0xcc, 0x95, 0x62, 0x6f, 0xef, 0xbc, 0xb6, 0xeb, 0x9c, 0xaa, 0xe5, 0x85, 0xac, 0xeb, 0x99, 0x91, 0x6f, 0xf0, 0xa5, 0x83, 0xb3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x99, 0xac, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe4, 0x81, 0x86, 0x69, 0xec, 0x9d, 0x82, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x9f, 0xcc, 0x95, 0x62, 0xe1, 0xba, 0x80, 0x75, 0xeb, 0xa3, 0xb8, 0xe3, 0x8a, 0x89, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xbe, 0xa3, 0xe5, 0xba, 0xa6, 0xf0, 0x9d, 0x97, 0x9c, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb6, 0xb3, 0xe5, 0x88, 0xa9, 0xec, 0x8c, 0xa7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67],
    "a᷎༹̛〪bu拥有믔퐧a立왠o公꾲o쯣귔i在线服务和个얣拥有퐮a〪̖̲֚bo㣣o휭立뜢i公멖hello𝑇e在线服务和个aཱིཻུbu公ﺭ높먘i在线服务和个︔i뺑a立믆잦o𝛆立à֮︯̕boＶ뜪公뙑o𥃳拥有뙬abcdefg䁆i읂à֮𞀟̕bẀu룸㊉hello拥有뾣度𝗜u拥有붳利쌧abcdefg"
  ),
  (
    [0xeb, 0xa9, 0x9a, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0x81, 0xcc, 0x95, 0x62, 0xed, 0x8d, 0x86, 0xec, 0xa1, 0x94, 0xe2, 0x91, 0xbc, 0xe5, 0xba, 0xa6, 0xec, 0xa2, 0xa0, 0x61, 0xe1, 0xb7, 0x8e, 0xf0, 0x9d, 0x85, 0xae, 0xcc, 0x9b, 0xe3, 0x80, 0xaa, 0x62, 0xe5, 0xba, 0xa6, 0xe1, 0xb8, 0x80, 0xec, 0x8f, 0x92, 0xe5, 0x88, 0xa9, 0xec, 0xa8, 0xbb, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x8c, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xc5, 0x85, 0xe2, 0xbc, 0x87, 0xf0, 0x9d, 0x9e, 0xa9, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xea, 0xa3, 0x84, 0xd6, 0xb0, 0x62, 0xeb, 0x97, 0xb0, 0xc8, 0x83, 0xe5, 0x88, 0xa9, 0xeb, 0x8f, 0x81, 0xe5, 0x88, 0xa9, 0xec, 0x9a, 0x92, 0xe5, 0x88, 0xa9, 0x61, 0xe3, 0x80, 0xaa, 0xdc, 0xbb, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xbc, 0xb3, 0xe7, 0xbd, 0xb2, 0xe7, 0xab, 0x8b, 0x61, 0xd6, 0xae, 0xcc, 0xbe, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xac, 0x8c, 0xed, 0x81, 0xb6, 0x61, 0xd6, 0xae, 0xea, 0xaa, 0xb8, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe1, 0xb5, 0x9c, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xea, 0xbd, 0xb5, 0xea, 0xb2, 0xb1, 0xe7, 0xab, 0x8b, 0xed, 0x9e, 0x86, 0xeb, 0x9a, 0xba, 0xc3, 0xa0, 0xd6, 0xae, 0xdb, 0xab, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x90, 0xa8, 0xbf, 0xd6, 0xb0, 0x62, 0xe5, 0xba, 0xa6, 0xec, 0x8b, 0xa2, 0xe5, 0x88, 0xa9, 0xec, 0x90, 0xb4, 0xe5, 0xba, 0xa6, 0xeb, 0xbe, 0x91, 0xeb, 0xb1, 0x8e, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x8a, 0xa0, 0xe3, 0x8f, 0x8f, 0xed, 0x9a, 0x9d, 0xe7, 0xab, 0x8b],
    "멚à֮᷁̕b퍆졔⑼度좠a᷎𝅮̛〪b度Ḁ쏒利쨻a֮𞀌̀̕bŅ⼇𝞩在线服务和个a゙्꣄ְb뗰ȃ利돁利욒利a〪ܻ̖֚b拥有뼳署立a֮̾̀̕b묌큶a֮ꪸ̀̕bᵜ在线服务和个꽵겱立힆뚺à֮۫̕ba゙्𐨿ְb度싢利쐴度뾑뱎拥有튠㏏횝立"
  ),
  (
    [0xe6, 0x97, 0xa2, 0xe5, 0xba, 0xa6, 0xea, 0xbb, 0x87, 0xeb, 0xb3, 0xab, 0xd9, 0x81, 0xd8, 0xac, 0xe7, 0xab, 0x8b, 0xeb, 0xb5, 0xb4, 0xe5, 0xba, 0xa6, 0xed, 0x82, 0x8e, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xbd, 0xa7, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x83, 0xcc, 0x95, 0x62, 0xeb, 0xa8, 0x80, 0xe5, 0xba, 0xa6, 0xeb, 0xbc, 0xba, 0xeb, 0x8a, 0xa4, 0xe5, 0x85, 0xac, 0xc9, 0xb2, 0xe5, 0x85, 0xac, 0x20, 0xd9, 0x8f, 0xe5, 0x88, 0xa9, 0xec, 0x97, 0x8e, 0xed, 0x95, 0xa1, 0xd9, 0x84, 0xd8, 0xad, 0xd9, 0x8a, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x87, 0x8a, 0xed, 0x92, 0xac, 0x61, 0xe3, 0x80, 0xaa, 0xd9, 0x96, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xeb, 0x97, 0xa3, 0xe9, 0xb1, 0x97, 0xe5, 0xba, 0xa6, 0xea, 0xb0, 0x9a, 0xe7, 0xab, 0x8b, 0xeb, 0x9f, 0xaa, 0xeb, 0x89, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x99, 0xa5, 0xeb, 0xa2, 0x92, 0xec, 0x9c, 0x84, 0x69, 0x69, 0xea, 0xb5, 0x95, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe1, 0xbc, 0x83, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xa1, 0xb7, 0xe5, 0x85, 0xac, 0xe1, 0xbd, 0x94, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xb8, 0x92, 0xec, 0x8e, 0x87, 0xed, 0x8c, 0x90, 0xe5, 0x85, 0xac, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcd, 0x87, 0xd6, 0x9a, 0x62, 0xec, 0xaf, 0xbc, 0xe5, 0x85, 0xac, 0xe7, 0x80, 0x9b, 0xe7, 0xab, 0x8b, 0xec, 0xbb, 0x96, 0xec, 0x89, 0x8a],
    "既度껇볫فج立뵴度킎拥有꽧à֮𞀃̕b먀度뼺늤公ɲ公 ُ利엎핡لحي在线服务和个쇊풬a〪ٖ̖֚b뗣鱗度갚立럪뉉拥有홥뢒위ii굕在线服务和个ἃ在线服务和个졷公ὔ在线服务和个븒쎇판公a〪̖͇֚b쯼公瀛立컖쉊"
  ),
  (
    [0xed, 0x98, 0x9b, 0xe7, 0xab, 0x8b, 0xd0, 0x99, 0xe7, 0xab, 0x8b, 0xeb, 0xa1, 0x8a, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x28, 0xe7, 0xa5, 0x9d, 0x29, 0xe5, 0x85, 0xac, 0xec, 0x97, 0x9d, 0x6f, 0xed, 0x8d, 0xa5, 0xe7, 0xab, 0x8b, 0xea, 0xb6, 0x91, 0xec, 0x9e, 0xb1, 0xe5, 0x88, 0xa9, 0xea, 0xb1, 0x97, 0xe5, 0xba, 0xa6, 0x35, 0x30, 0xed, 0x83, 0xb8, 0x65, 0xe1, 0xba, 0x89, 0x69, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x88, 0x8b, 0xe5, 0x88, 0xa9, 0xeb, 0x9a, 0x83, 0x69, 0xeb, 0x8d, 0xbe, 0xe5, 0x88, 0xa9, 0xce, 0xb2, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xd8, 0xa6, 0xd8, 0xb2, 0x61, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x99, 0xbd, 0xea, 0xbf, 0xbe, 0xe5, 0x85, 0xac, 0xe0, 0xbe, 0x9c, 0xe0, 0xbe, 0xb7, 0xe5, 0x85, 0xac, 0xeb, 0xa3, 0x85, 0xed, 0x98, 0x92, 0x61, 0xe3, 0x80, 0xaa, 0xe0, 0xa3, 0xad, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x89, 0xa1, 0x65, 0xe5, 0xba, 0xa6, 0xeb, 0xb9, 0xbf, 0x61, 0xe5, 0xba, 0xa6, 0x61, 0xd6, 0xae, 0xe0, 0xa0, 0xa6, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x85, 0xac, 0xe9, 0xa0, 0xbb, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x8b, 0x90, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9b, 0x87, 0xf0, 0xa9, 0xac, 0xb0, 0x6f, 0xe9, 0xbb, 0x91, 0xeb, 0xa1, 0x92, 0xeb, 0x89, 0x94, 0x6f, 0xec, 0xb5, 0x9c, 0x61, 0xe7, 0xab, 0x8b, 0xea, 0xbc, 0xbe, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x9f, 0xb8, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xd9, 0x86, 0xd9, 0x86, 0xec, 0x82, 0x8f, 0xd7, 0x9e, 0xd6, 0xbc, 0x69, 0xe5, 0x85, 0xac, 0xeb, 0xa7, 0xbc],
    "혛立Й立롊abcdefg(祝)公엝o퍥立궑잱利걗度50탸eẉi拥有숋利뚃i덾利βhelloئزa在线服务和个홽꿾公ྜྷ公룅혒a〪࣭̖֚babcdefg쉡e度빿a度a֮ࠦ̀̕bhello公頻在线服务和个싐在线服务和个훇𩬰o黑롒뉔o최a立꼾hello在线服务和个럸e在线服务和个نن삏מּi公맼"
  ),
  (
    [0xf0, 0x9d, 0x93, 0x81, 0xf0, 0x9f, 0xa4, 0x94, 0x28, 0x31, 0x37, 0x29, 0x69, 0xf0, 0x9f, 0xa4, 0x94, 0xe4, 0xb8, 0x81, 0xe2, 0x9a, 0x89, 0xf0, 0x9d, 0x95, 0xaa, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xac, 0x80, 0x61, 0xe6, 0xae, 0xbb, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x72, 0x65, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0xa9, 0xa6, 0xe2, 0x9a, 0x89, 0xd9, 0x8a, 0xd8, 0xb2, 0x75, 0xef, 0xb8, 0xb2, 0xf0, 0x9f, 0xa4, 0x94, 0xf0, 0x9d, 0x91, 0x88, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0xb6, 0xb2, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0xad, 0x95, 0x6f, 0xe2, 0x85, 0x80, 0x75, 0xec, 0xb4, 0xb6, 0x6f, 0xed, 0x89, 0x9f, 0x6f, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0x83, 0xa7, 0xcc, 0x95, 0x62, 0x6f, 0xe2, 0x93, 0xa6, 0x69, 0xe3, 0x82, 0xad, 0xe3, 0x83, 0xad, 0xf0, 0x9f, 0x98, 0x82, 0xcf, 0xb2, 0x69, 0xe0, 0xad, 0x88, 0x65, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb3, 0x91, 0xcc, 0x95, 0x62, 0x61, 0xf0, 0x9f, 0x98, 0x82, 0xcb, 0x9c, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x61, 0xd6, 0xae, 0xdb, 0xab, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0xb8, 0x93, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0xa9, 0x80, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xb8, 0x86, 0x75, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x91, 0xa4, 0xef, 0xbd, 0x9b, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0x9d, 0x9d, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x9c, 0x8a, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xac, 0x8d, 0x65, 0xf0, 0x9f, 0x98, 0x82, 0xef, 0xae, 0xb1, 0xf0, 0xa1, 0x9a, 0xa8, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0xb3, 0xac, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x86, 0x97, 0xeb, 0xac, 0xa4, 0xed, 0x80, 0x9d, 0x69, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x95, 0xa8, 0xeb, 0xa9, 0x8a, 0x69, 0xe2, 0x9a, 0x89],
    "𝓁🤔(17)i🤔丁⚉𝕪👨‍👩‍👧묀a殻🌺🌸re🌺🌸婦⚉يزu︲🤔𝑈abcdefg춲🤔쭕o⅀u촶o퉟oà֮⃧̕boⓦiキロ😂ϲiୈeà֮᳑̕ba😂˜🌺🌸a֮۫̀̕b븓abcdefg🌺🌸멀😂븆u😂둤｛🤔띝abcdefg휊abcdefg😂묍e😂ﮱ𡚨🇨🇦쳬🌺🌸놗묤퀝i🇨🇦땨멊i⚉"
  ),
  (
    [0xeb, 0xbe, 0xa9, 0xeb, 0x81, 0x9a, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x84, 0x80, 0xef, 0xb1, 0xa6, 0xe1, 0xbe, 0x9c, 0xec, 0xae, 0x81, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0xa5, 0xbd, 0xec, 0xa6, 0x84, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x9a, 0xb4, 0xeb, 0xa0, 0x8b, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xf0, 0xa6, 0x9e, 0xb5, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe4, 0xbe, 0xbf, 0xeb, 0x81, 0xa7, 0xed, 0x85, 0xb9, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0xb9, 0xa8, 0xeb, 0x86, 0xaf, 0xed, 0x9a, 0x83, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xed, 0x93, 0x88, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0xa3, 0xb8, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0xa9, 0xb5, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x95, 0xbc, 0xe5, 0x88, 0x87, 0xeb, 0x93, 0x9c, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xe2, 0x9a, 0x89, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0x83, 0x97, 0xcc, 0x95, 0x62, 0xf0, 0x9d, 0x97, 0xb5, 0xf0, 0x9d, 0x94, 0x8a, 0xef, 0xbe, 0xa0, 0xf0, 0x9f, 0xa4, 0x94, 0xe3, 0x8a, 0xae, 0xe0, 0xbe, 0x9c, 0xe0, 0xbe, 0xb7, 0xf0, 0x9f, 0x98, 0x82, 0xe2, 0x85, 0x9a, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdc, 0xbc, 0xd6, 0x9a, 0x62, 0xf0, 0x9e, 0xb9, 0xba, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0x9b, 0xa5, 0xe2, 0x9a, 0x89, 0xe1, 0xb5, 0xa1, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0x9d, 0x97, 0xec, 0x87, 0xa2, 0xf0, 0x9d, 0x9d, 0x83, 0xec, 0x92, 0xaa, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0xa8, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x94, 0x88, 0xef, 0xbc, 0xba, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xf0, 0x9f, 0x84, 0x9f, 0xec, 0x97, 0xa9, 0xe1, 0x84, 0x8e, 0xcc, 0xb4, 0xe1, 0x85, 0xa6, 0xf0, 0x9d, 0x94, 0xb5, 0xf0, 0x9d, 0x95, 0x86, 0xec, 0xb0, 0xb4, 0xe2, 0x9a, 0x89, 0xeb, 0x9a, 0xbb, 0xec, 0x8b, 0xa4, 0xd3, 0x91, 0xf0, 0x9f, 0xa4, 0x94],
    "뾩끚🇨🇦넀ﱦᾜ쮁👨‍👩‍👧쥽즄🌺🌸뚴렋🇨🇦𦞵👨‍👩‍👧便끧텹👨‍👩‍👧침놯횃🇨🇦퓈🇨🇦죸🤔쩵👨‍👩‍👧야切드̴ᆮ⚉à֮⃗̕b𝗵𝔊ﾠ🤔㊮ྜྷ😂⅚👨‍👩‍👧a〪̖ܼ֚b𞹺😂웥⚉ᵡ🤔띗쇢𝝃쒪a֮ࣨ̀̕b딈Ｚ🇨🇦🄟엩ᄎ̴ᅦ𝔵𝕆찴⚉뚻실ӑ🤔"
  ),
  (
    [0xec, 0x89, 0x8f, 0x61, 0xeb, 0xae, 0xa4, 0xec, 0x80, 0x94, 0x75, 0xec, 0xaf, 0x96, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0xae, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xb0, 0x98, 0xea, 0xb3, 0x96, 0x75, 0xf0, 0x9d, 0x91, 0x81, 0xe5, 0xba, 0xb3, 0x69, 0xeb, 0x92, 0xb2, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe3, 0x86, 0x85, 0x6f, 0xed, 0x99, 0x8c, 0x65, 0xe2, 0xbd, 0xbf, 0x61, 0xef, 0xbf, 0xad, 0x65, 0xe3, 0x8b, 0x87, 0x75, 0xec, 0xab, 0xa4, 0xec, 0xa7, 0xbc, 0xe1, 0xbb, 0xb4, 0xe1, 0xba, 0xa8, 0xec, 0x8f, 0xb9, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xed, 0x8d, 0x82, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x8b, 0x9e, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0xa1, 0xa5, 0xec, 0xb2, 0xb3, 0xf0, 0x9d, 0x97, 0xaa, 0x75, 0xeb, 0xa2, 0x81, 0xea, 0xb4, 0xb6, 0xec, 0x95, 0x91, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xea, 0xb6, 0x93, 0xec, 0xa8, 0x84, 0xf0, 0x9d, 0x9e, 0x9e, 0xeb, 0xa4, 0xaf, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe3, 0x8b, 0xb9, 0xec, 0x85, 0xb6, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9d, 0x9b, 0x90, 0xea, 0xb2, 0xa8, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0x65, 0xe1, 0xac, 0xba, 0xcc, 0xb4, 0xe1, 0xac, 0xb5, 0x6f, 0xea, 0xb0, 0x95, 0xec, 0xa4, 0x8d, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x87, 0xcc, 0x95, 0x62, 0xea, 0xb6, 0xb6, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x82, 0x88, 0xec, 0xba, 0xb7, 0xed, 0x90, 0xab, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x97, 0x8c, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xea, 0xab, 0xb6, 0xd6, 0xb0, 0x62, 0x69, 0xed, 0x85, 0xb1, 0x61, 0xf0, 0x9d, 0x99, 0x8a, 0xec, 0x95, 0xb6, 0x75, 0xe1, 0xba, 0xa3, 0x61],
    "쉏a뮤쀔u쯖a֮ᷮ̀̕b찘곖u𝑁庳i뒲helloㆅo홌e⽿a￭e㋇u쫤짼ỴẨ쏹hello퍂abcdefg닞hello롥첳𝗪u뢁괶앑hello궓쨄𝞞뤯hello㋹셶abcdefg𝛐겨̴ᆵeᬺ̴ᬵo강줍à֮̇̕b궶abcdefg삈캷퐫abcdefg엌a゙्꫶ְbi텱a𝙊앶uảa"
  ),
]

let codeUnitNormalizationTestsG: [([UInt8], String)] = [
  (
    [0xec, 0xa5, 0x8e, 0xec, 0x97, 0xba, 0xed, 0x96, 0x8d, 0xec, 0x92, 0x87, 0x75, 0xeb, 0x92, 0xb0, 0x69, 0xeb, 0x83, 0x80, 0xed, 0x8c, 0xa7, 0xec, 0xb7, 0xb8, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x61, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0xea, 0xa6, 0xb3, 0xe3, 0x82, 0x99, 0x62, 0x6f, 0xe5, 0xa3, 0xb0, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0x91, 0xb6, 0xeb, 0xad, 0x8c, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0xb7, 0x91, 0xea, 0xb6, 0xa0, 0x61, 0xec, 0xb5, 0x9b, 0xed, 0x83, 0xb9, 0x69, 0xe1, 0xba, 0xbd, 0x65, 0xd0, 0x80, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xe1, 0xaf, 0xb3, 0xd6, 0xb0, 0x62, 0x61, 0xd6, 0xae, 0xcd, 0xa8, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x61, 0xe2, 0x83, 0xa6, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0xea, 0xb0, 0xad, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x9c, 0xa9, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0x88, 0x83, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa3, 0x99, 0xcc, 0x95, 0x62, 0xe9, 0x99, 0xb5, 0x69, 0xc3, 0xa0, 0xd6, 0xae, 0xcd, 0x97, 0xcc, 0x95, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x80, 0xa0, 0xeb, 0xbd, 0xa4, 0xeb, 0xbf, 0xb0, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x84, 0xb3, 0xd6, 0xb0, 0x62, 0x69, 0xed, 0x94, 0xa0, 0xeb, 0x8e, 0x81, 0x69, 0xeb, 0xbe, 0xa5, 0xeb, 0x87, 0x8c, 0x61, 0xd6, 0xae, 0xdf, 0xac, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0x84, 0x8c, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0x97, 0xac, 0x61, 0xeb, 0x92, 0xa1, 0xeb, 0x82, 0xb8, 0x65, 0x6e, 0x6a, 0xed, 0x98, 0x83, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x88, 0xa9, 0xeb, 0xbd, 0xb3, 0xeb, 0x96, 0x97, 0xeb, 0x8f, 0x83, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe9, 0xa1, 0x9e, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0xb7, 0x98, 0x69, 0xea, 0xb0, 0xbb, 0x6f, 0xeb, 0xae, 0x97, 0x68, 0x65, 0x6c, 0x6c, 0x6f],
    "쥎엺햍쒇u뒰i냀팧췸helloa̴़꦳゙bo声hello둶뭌abcdefg췑궠a쵛탹iẽeЀa゙्᯳ְba֮ͨ̀̕ba⃦̴़b갭abcdefg윩abcdefg刃à֮ࣙ̕b陵ià֮͗̕babcdefg뀠뽤뿰a゙्𑄳ְbi픠뎁i뾥뇌a֮߬̀̕b섌hello여a뒡낸enj혃abcdefg툩뽳떗돃abcdefg類hello췘i갻o뮗hello"
  ),
  (
    [0xd8, 0xb4, 0xd9, 0x85, 0xd8, 0xae, 0xe5, 0xba, 0xa6, 0x28, 0x6a, 0x29, 0xf0, 0x9f, 0xa4, 0x94, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xa3, 0xb3, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xa8, 0x9c, 0xe2, 0x9a, 0x89, 0xe5, 0x85, 0xac, 0xec, 0x99, 0xb4, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x81, 0x81, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0xba, 0xa6, 0xeb, 0x91, 0xa6, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe1, 0xba, 0x8d, 0xe1, 0xb9, 0xa1, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xed, 0x95, 0x81, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0xba, 0xa6, 0xec, 0x9e, 0xb7, 0xe5, 0x85, 0xac, 0xec, 0xb1, 0xb8, 0xeb, 0x82, 0xa2, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe8, 0xb1, 0xb8, 0xe6, 0xad, 0xb7, 0xec, 0xb5, 0xa5, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x88, 0xa9, 0xeb, 0xa3, 0xa7, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0xaa, 0x9a, 0xe2, 0x9a, 0x89, 0xea, 0xbf, 0xa1, 0xe5, 0x88, 0xa9, 0xec, 0x9c, 0xa7, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe7, 0xab, 0x8b, 0xec, 0x80, 0x9b, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0xa0, 0xaa, 0xe2, 0x9a, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xb8, 0xb5, 0xe2, 0x9a, 0x89, 0xeb, 0x8b, 0x94, 0xec, 0x86, 0xa0, 0xe2, 0x9a, 0x89, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe8, 0x8e, 0xad, 0xe7, 0xab, 0x8b, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x8a, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xb3, 0x82, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x88, 0xa9, 0xed, 0x84, 0xab, 0xec, 0x98, 0xb4, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x91, 0x8a, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x85, 0xac, 0x7c, 0xe2, 0x9a, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe1, 0xb9, 0x97, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x88, 0xa9, 0xea, 0xbe, 0xbd, 0xea, 0xbb, 0x92, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xbb, 0xac, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x88, 0x83, 0xe5, 0x90, 0xb9, 0xe7, 0xab, 0x8b, 0x61, 0xe3, 0x82, 0x99, 0xe1, 0xa9, 0xa0, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xeb, 0x91, 0x9a, 0x61, 0xe7, 0xab, 0x8b, 0xec, 0x99, 0xbb, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x94, 0x86, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x8f, 0xb4, 0xe7, 0xab, 0x8b, 0xed, 0x83, 0x91, 0xec, 0xa7, 0x8e, 0xe2, 0x9a, 0x89, 0xe5, 0x85, 0xac, 0xec, 0x84, 0xa4, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9c, 0xa9, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xae, 0x92, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x8a, 0x92, 0xe7, 0xab, 0x8b],
    "شمخ度(j)🤔拥有죳👨‍👩‍👧먜⚉公왴🇨🇦끁🤔度둦🇨🇦ẍṡ🌺🌸핁😂度잷公챸낢🇨🇦豸歷쵥🤔利룧🇨🇦몚⚉꿡利윧🌺🌸立쀛🇨🇦렪⚉拥有층⚉닔솠⚉在线服务和个莭立à֮̊̕b😂볂🇨🇦利턫옴🌺🌸둊😂公|⚉拥有ṗ🇨🇦利꾽껒拥有뻬🌺🌸눃吹立a゙᩠्ְb둚a立왻🇨🇦딆👨‍👩‍👧쏴立탑짎⚉公설🤔在线服务和个휩在线服务和个뮒🌺🌸늒立"
  ),
  (
    [0xeb, 0xb0, 0x95, 0x65, 0xed, 0x8d, 0x89, 0xe7, 0xab, 0x8b, 0xef, 0xb2, 0x94, 0x61, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x80, 0xac, 0x6f, 0x61, 0xe2, 0x83, 0xab, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0x75, 0xe5, 0x88, 0xa9, 0xf0, 0x9f, 0x84, 0x95, 0x6f, 0xd9, 0x86, 0xd9, 0x87, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0xb9, 0xb2, 0x61, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0xbb, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x88, 0x87, 0x65, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0xba, 0xa6, 0xec, 0xaa, 0x8a, 0xeb, 0xa7, 0xad, 0xf0, 0x9f, 0x98, 0x82, 0xd8, 0xa8, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x94, 0x90, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0xba, 0xa6, 0xeb, 0x98, 0x9d, 0xf0, 0x9f, 0xa4, 0x94, 0xe7, 0xab, 0x8b, 0xed, 0x8a, 0xab, 0xeb, 0xbb, 0x87, 0x61, 0x61, 0xcc, 0xb4, 0xe1, 0xb3, 0xa4, 0xe0, 0xa4, 0xbc, 0x62, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe7, 0xab, 0x8b, 0xed, 0x93, 0x9b, 0x69, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x38, 0x2e, 0x75, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x85, 0xac, 0xea, 0xbd, 0xa9, 0x20, 0xcc, 0x93, 0xcc, 0x81, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xef, 0xbc, 0x9d, 0x65, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x93, 0x98, 0x6f, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0xa1, 0x81, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x9d, 0x81, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xe3, 0x80, 0xaa, 0xdc, 0xbc, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x88, 0xa9, 0xeb, 0x92, 0x84, 0xe5, 0xba, 0xa6, 0xe2, 0x85, 0x89, 0x61, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0x96, 0x9c, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0xa3, 0x9f, 0xe5, 0x85, 0xac, 0xeb, 0x92, 0x80, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x8c, 0xa6, 0x6f, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe7, 0xab, 0x8b, 0xef, 0xb5, 0x98, 0x75, 0xe3, 0x85, 0x95, 0x69, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0xb7, 0x81, 0x65, 0xf0, 0x9f, 0xa4, 0x94, 0xe3, 0x8a, 0xb1, 0xec, 0x8f, 0xad, 0x65, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0xa3, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x57, 0x43, 0x61, 0x6d, 0x33, 0x61, 0xeb, 0xa0, 0x9c, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x8c, 0xb1, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x9d, 0x9a, 0x6f, 0xe5, 0x88, 0xa9, 0xeb, 0xa0, 0x88, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xe5, 0x85, 0xac, 0xeb, 0x81, 0x91, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe7, 0xab, 0x8b, 0xef, 0xb5, 0x99, 0xec, 0x9f, 0xbb, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x88, 0xa9, 0xec, 0xb1, 0x9e, 0xf0, 0x9f, 0xa4, 0x94, 0xe7, 0xab, 0x8b, 0xed, 0x93, 0xa1, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa],
    "박e퍉立ﲔa在线服务和个쀬oa⃫̴़bu利🄕oنه🤔干a在线服务和个a〪̻̖֚b🇨🇦숇e🤔度쪊맭😂بo在线服务和个딐😂度똝🤔立튫뻇aa̴᳤़b🌺🌸立퓛i🌺🌸8.u😂公꽩 ̓́🇨🇦＝e👨‍👩‍👧拥有쓘o🤔롁🤔在线服务和个읁在线服务和个a〪ܼ̖֚b🇨🇦利뒄度ⅉa😂얜abcdefg🤔죟公뒀hello拥有팦o🇨🇦立ﵘuㅕi🤔뷁e🤔㊱쏭e拥有a֮𞀣̀̕b拥有WCam3a렜👨‍👩‍👧拥有쌱🌺🌸읚o利레̴ᆵ公끑hello🇨🇦立ﵙ쟻🇨🇦利챞🤔立퓡e在线服务和个"
  ),
  (
    [0xea, 0xb8, 0xa0, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x90, 0xbd, 0x86, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9e, 0xba, 0x88, 0xe3, 0x88, 0xa2, 0x61, 0xec, 0xac, 0x97, 0xeb, 0x85, 0x92, 0xe3, 0x89, 0xb6, 0xeb, 0xb9, 0xb2, 0xed, 0x86, 0xa2, 0x75, 0xe1, 0x84, 0x85, 0xcc, 0xb4, 0xe1, 0x85, 0xad, 0xed, 0x8a, 0x92, 0x6f, 0xec, 0x84, 0x82, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0x86, 0xbe, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0x95, 0x9d, 0xeb, 0x88, 0x82, 0xf0, 0x9d, 0x90, 0x8c, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0x8b, 0xb1, 0xec, 0xb5, 0x95, 0x65, 0xe3, 0x8b, 0x95, 0x75, 0xec, 0xb3, 0x91, 0xeb, 0x81, 0xb8, 0xeb, 0xa3, 0x89, 0x6f, 0xeb, 0xb9, 0xa4, 0x61, 0xed, 0x86, 0xba, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9d, 0x9d, 0x9d, 0x61, 0xef, 0xbe, 0x94, 0x6f, 0xed, 0x90, 0x8c, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0x9c, 0x92, 0xe1, 0xbf, 0xa4, 0x75, 0xeb, 0x96, 0xad, 0xec, 0xad, 0xbd, 0xef, 0xb1, 0x99, 0x6f, 0xeb, 0x92, 0x85, 0x61, 0xeb, 0x9a, 0x8f, 0x65, 0xeb, 0x89, 0xb9, 0x61, 0xe1, 0xb8, 0xb9, 0xeb, 0xb0, 0x85, 0x75, 0x61, 0xd8, 0x98, 0xd8, 0x99, 0xd8, 0x99, 0xd8, 0x9a, 0x62, 0x65, 0xec, 0xbc, 0xbf, 0xec, 0x8e, 0xb6, 0xe9, 0x86, 0xb4, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe0, 0xa4, 0xa2, 0xe0, 0xa4, 0xbc, 0x6f, 0xe3, 0x8f, 0x89, 0xeb, 0x9a, 0xa5, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xef, 0xbf, 0x87, 0xeb, 0xb4, 0xa6, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0xba, 0xad, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x61, 0xcc, 0xb4, 0xe0, 0xac, 0xbc, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0x65, 0xeb, 0xac, 0x84, 0x65, 0xe5, 0x96, 0x87, 0x65, 0xea, 0xb2, 0x8d, 0x69, 0xe3, 0x89, 0xac, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0xb1, 0x8c, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xbc, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xb4, 0x62, 0xe3, 0x8a, 0x8c, 0x75, 0xeb, 0x9d, 0xac, 0x69, 0xef, 0xb6, 0xa5, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcd, 0x96, 0xd6, 0x9a, 0x62, 0x75, 0xed, 0x90, 0xa3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x86, 0xa8],
    "긠a〪𐽆̖֚bhello𞺈㈢a쬗녒㉶빲톢uᄅ̴ᅭ튒o섂hello솾hello땝눂𝐌hello닱쵕e㋕u쳑끸룉o빤a톺abcdefg𝝝aﾔo퐌hello뜒ῤu떭쭽ﱙo뒅a뚏e뉹aḹ밅uaؘؙؙؚbe켿쎶醴abcdefgढ़o㏉뚥helloￇ봦abcdefg뺭abcdefga̴଼़゙be묄e喇e겍i㉬hello챌aཱོིུb㊌u띬iﶥa〪̖͖֚bu퐣abcdefg놨"
  ),
  (
    [0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x8d, 0x8d, 0xd6, 0xb0, 0x62, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe3, 0x8a, 0x94, 0x75, 0xe7, 0xa5, 0x89, 0xeb, 0xb3, 0xb8, 0x75, 0xe5, 0x85, 0xac, 0xeb, 0x81, 0xa0, 0xe5, 0xba, 0xa6, 0xea, 0xbf, 0x92, 0xe5, 0xba, 0xa6, 0xed, 0x9a, 0x9d, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xed, 0x9e, 0x84, 0x75, 0xe3, 0x8c, 0xbe, 0x75, 0xc3, 0xa8, 0x75, 0xe7, 0xab, 0x8b, 0xe6, 0xb5, 0x81, 0x75, 0xea, 0xb2, 0xb4, 0x75, 0xef, 0xad, 0xa7, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb4, 0xb5, 0x61, 0xe5, 0x85, 0xac, 0xed, 0x87, 0x8d, 0x61, 0xe3, 0x80, 0xaa, 0xe0, 0xa3, 0xb9, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x69, 0xec, 0xb5, 0x94, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9d, 0x9a, 0xbd, 0x69, 0xf0, 0x9d, 0x98, 0xba, 0xf0, 0x9f, 0x89, 0x82, 0x75, 0xe1, 0xb5, 0xa8, 0x6f, 0xe7, 0xab, 0x8b, 0xf0, 0x9d, 0x9a, 0xbd, 0xe5, 0x85, 0xac, 0xe1, 0xba, 0xa2, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xb3, 0x98, 0xd6, 0x9a, 0x62, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe5, 0xa8, 0xa7, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x9e, 0xaf, 0x61, 0xe5, 0x88, 0xa9, 0xec, 0xa7, 0x96, 0x75, 0xed, 0x87, 0xa8, 0x6f, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcd, 0x93, 0xd6, 0x9a, 0x62, 0xe5, 0x88, 0xa9, 0xec, 0x8a, 0xb4, 0xec, 0xa1, 0xbc, 0xe5, 0x88, 0xa9, 0xe3, 0x8f, 0x97, 0xe5, 0x85, 0xac, 0xe3, 0x8f, 0x98, 0xe5, 0xba, 0xa6, 0xf0, 0x9e, 0xba, 0x8d, 0x61, 0xe5, 0xba, 0xa6, 0xeb, 0xbc, 0xa1, 0xec, 0xb2, 0xbf, 0xe5, 0xba, 0xa6, 0xc3, 0xa0, 0xe3, 0x80, 0xac, 0xcc, 0x95, 0xcd, 0x9c, 0x62, 0x61, 0xea, 0xbc, 0x90, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xac, 0x91, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xbb, 0x83, 0x65, 0xe5, 0x85, 0xac, 0xe6, 0xb5, 0xaa, 0xe5, 0x85, 0xac, 0xed, 0x87, 0xa5, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x9f, 0x99, 0x65, 0xeb, 0xae, 0x8a, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xef, 0xbd, 0xb3, 0x69, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xbc, 0x93, 0xe6, 0xa5, 0x82, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xa5, 0x88, 0xe7, 0xab, 0x8b, 0xea, 0xbd, 0xb8, 0xe5, 0x85, 0xac, 0xf0, 0x9d, 0x95, 0x93, 0xec, 0xb1, 0x83, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xaa, 0xcc, 0x95, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0x88, 0xa9, 0xec, 0xbe, 0x80, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x91, 0x84, 0x81, 0xcc, 0x95, 0x62, 0xe5, 0xba, 0xa6, 0xed, 0x86, 0xa1, 0xe5, 0xba, 0xa6, 0xec, 0x9f, 0x90, 0xe5, 0x85, 0xac, 0xeb, 0xb8, 0x8e, 0xe5, 0x88, 0xa9, 0xed, 0x8a, 0xbb, 0x61, 0xe5, 0x85, 0xac, 0xed, 0x99, 0x8a, 0x6f, 0xed, 0x85, 0x8d, 0x69],
    "a゙्𑍍ְb拥有㊔u祉본u公끠度꿒度횝hello힄u㌾uèu立流u겴uﭧu拥有봵a公퇍a〪ࣹ̖֚bi쵔abcdefg𝚽i𝘺🉂uᵨo立𝚽公Ảa〪̖᳘֚bu拥有娧在线服务和个잯a利짖u퇨oa〪̖͓֚b利슴졼利㏗公㏘度𞺍a度뼡첿度à〬̕͜ba꼐拥有쬑拥有컃e公浪公퇥abcdefg쟙e뮊helloｳi在线服务和个켓楂u拥有륈立꽸公𝕓챃helloà֮𑍪̕babcdefg利쾀o在线服务和个à֮𑄁̕b度톡度쟐公븎利튻a公홊o텍i"
  ),
  (
    [0xeb, 0xa5, 0xac, 0xe1, 0x84, 0xa7, 0xeb, 0xa2, 0xae, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x90, 0xa8, 0x8f, 0xcc, 0x95, 0x62, 0xec, 0x8c, 0x8b, 0xea, 0xbd, 0x8e, 0xed, 0x88, 0xba, 0xec, 0xbd, 0xaa, 0xec, 0xa9, 0x86, 0xe7, 0x85, 0x85, 0xed, 0x8c, 0xa0, 0xe5, 0xbd, 0x93, 0xec, 0xa7, 0x93, 0xeb, 0x80, 0xbc, 0xe1, 0xac, 0x85, 0xcc, 0xb4, 0xe1, 0xac, 0xb5, 0xea, 0xb2, 0x9d, 0xec, 0x99, 0x94, 0xec, 0xa7, 0xb6, 0xd8, 0xa6, 0xdb, 0x95, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe0, 0xa1, 0x9b, 0xd6, 0x9a, 0x62, 0xed, 0x88, 0x90, 0xeb, 0xbe, 0x89, 0xec, 0xa8, 0xb6, 0xec, 0x8c, 0xa5, 0xed, 0x90, 0x93, 0xeb, 0x8a, 0x8a, 0xec, 0xa1, 0x82, 0xeb, 0x95, 0xb2, 0xeb, 0xbe, 0x81, 0xeb, 0xb4, 0xa5, 0xed, 0x9b, 0x8d, 0xe0, 0xaf, 0x8b, 0xec, 0xbd, 0xaf, 0xe8, 0xae, 0x80, 0xed, 0x88, 0xa0, 0xec, 0x88, 0x8a, 0xec, 0x97, 0x95, 0xec, 0xa0, 0x98, 0xf0, 0xaa, 0x98, 0x80, 0xec, 0x99, 0xa0, 0xec, 0x81, 0xbf, 0xec, 0xac, 0xb9, 0xec, 0xaf, 0x9e, 0xec, 0x97, 0xb7, 0xe5, 0x8c, 0xbb, 0xeb, 0xb4, 0xb3, 0xf0, 0xa9, 0x87, 0x9f, 0xc3, 0xa0, 0xd6, 0xae, 0xdc, 0xb2, 0xcc, 0x95, 0x62, 0xec, 0xb3, 0x97, 0xeb, 0x90, 0xae, 0xeb, 0xbd, 0x8d, 0xed, 0x8d, 0x8f, 0xed, 0x83, 0x93, 0xd8, 0xaa, 0xd8, 0xb2, 0xec, 0xaa, 0x94, 0xeb, 0xb6, 0x8b, 0xe3, 0x83, 0xac, 0xe3, 0x83, 0xa0, 0xce, 0x91, 0xeb, 0x80, 0xaf, 0xec, 0xa3, 0xbf],
    "륬ᄧ뢮à֮𐨏̕b쌋꽎툺콪쩆煅팠当짓뀼ᬅ̴ᬵ겝왔짶ئەa〪̖࡛֚b툐뾉쨶쌥퐓늊졂땲뾁봥훍ோ콯讀툠숊엕젘𪘀왠쁿쬹쯞엷医봳𩇟à֮ܲ̕b쳗됮뽍퍏탓تز쪔붋レムΑ뀯죿"
  ),
  (
    [0xeb, 0x84, 0x83, 0xeb, 0x87, 0xa8, 0xea, 0xb0, 0xbe, 0xe4, 0x83, 0xa3, 0x4b, 0x42, 0xe5, 0x96, 0x87, 0xe1, 0xbe, 0xaa, 0xeb, 0x9b, 0x91, 0xeb, 0xba, 0x95, 0xeb, 0x82, 0xbf, 0xe9, 0x83, 0x8e, 0xe1, 0x86, 0x85, 0xec, 0xbc, 0x8c, 0xeb, 0x8b, 0xa7, 0xec, 0xb6, 0xaf, 0x72, 0x61, 0x64, 0xe2, 0x88, 0x95, 0x73, 0x32, 0xe4, 0xb8, 0x83, 0xed, 0x80, 0xb6, 0xed, 0x9a, 0xaa, 0xeb, 0x93, 0xae, 0xec, 0x9a, 0xa1, 0xed, 0x89, 0x9f, 0xec, 0x9e, 0xbc, 0xeb, 0x86, 0xb5, 0xe3, 0x83, 0x92, 0xd9, 0x85, 0xd8, 0xae, 0xd9, 0x8a, 0xeb, 0x93, 0xa2, 0xeb, 0xb3, 0x9b, 0xed, 0x8c, 0x8d, 0xeb, 0xb8, 0x98, 0xe0, 0xbe, 0xb3, 0xe0, 0xbe, 0x80, 0xe3, 0x83, 0x96, 0xe3, 0x83, 0x83, 0xe3, 0x82, 0xb7, 0xe3, 0x82, 0xa7, 0xe3, 0x83, 0xab, 0xeb, 0x9b, 0x95, 0xea, 0xb2, 0xb7, 0xe8, 0xb3, 0x88, 0xeb, 0xb3, 0xbd, 0xeb, 0xa1, 0xa7, 0xe7, 0xa6, 0x8d, 0xe1, 0xba, 0xbc, 0xeb, 0xb6, 0xa4, 0xec, 0x84, 0x9e, 0xeb, 0x89, 0x94, 0xed, 0x91, 0xb8, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xeb, 0xaf, 0xa1, 0xed, 0x97, 0x89, 0xec, 0xb4, 0xa9, 0xea, 0xbb, 0xb9, 0xec, 0xba, 0xa6, 0xeb, 0xbf, 0xbd, 0xec, 0x8e, 0xbb, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x8d, 0xcc, 0x95, 0x62, 0xeb, 0xae, 0xb9, 0xeb, 0xbd, 0x9b, 0xeb, 0x8e, 0xbb, 0xec, 0xa8, 0xa2, 0xec, 0xaa, 0xa2, 0xeb, 0xa8, 0x87, 0xeb, 0xb4, 0x98, 0x61, 0xd6, 0xae, 0xf0, 0x90, 0x8d, 0xb6, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x97, 0xb7],
    "넃뇨갾䃣KB喇ᾪ뛑뺕낿郎ᆅ켌닧춯rad∕s2七퀶횪듮욡퉟잼놵ヒمخي듢볛팍븘ླྀブッシェル뛕겷賈볽롧禍Ẽ붤섞뉔푸̴ᆵ믡헉촩껹캦뿽쎻à֮̍̕b뮹뽛뎻쨢쪢먇봘a֮𐍶̀̕b헷"
  ),
  (
    [0xf0, 0x9d, 0x93, 0x8f, 0xf0, 0x9d, 0x9c, 0x9e, 0xeb, 0xbc, 0xbb, 0xea, 0xbb, 0xa9, 0xeb, 0xa4, 0x9f, 0x61, 0xec, 0x89, 0x88, 0xc3, 0xa0, 0xd6, 0xae, 0xd2, 0x85, 0xcc, 0x95, 0x62, 0x6f, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xa8, 0x98, 0xd6, 0x9a, 0x62, 0xf0, 0x9d, 0x9a, 0x93, 0xed, 0x8e, 0x89, 0xec, 0x81, 0xb8, 0x6f, 0xef, 0xb1, 0x9a, 0x6f, 0x61, 0xd6, 0xae, 0xef, 0xb8, 0xa5, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xbd, 0xba, 0x65, 0xeb, 0xa0, 0xb1, 0x6f, 0xed, 0x8d, 0xb5, 0xed, 0x8f, 0x88, 0x61, 0xed, 0x97, 0xbd, 0xef, 0xb4, 0xad, 0x75, 0xeb, 0x9a, 0x8b, 0xe3, 0x83, 0x8a, 0xe3, 0x83, 0x8e, 0xc3, 0xa4, 0xd6, 0xae, 0xcc, 0x81, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x69, 0xeb, 0x83, 0xa4, 0x75, 0xed, 0x8b, 0x8d, 0xea, 0xb4, 0x98, 0xf0, 0x9d, 0x99, 0x87, 0x6f, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xea, 0xaa, 0xb4, 0xd6, 0x9a, 0x62, 0x61, 0xec, 0x93, 0x82, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0xbd, 0xa5, 0x61, 0xec, 0xbd, 0x9c, 0xea, 0xb3, 0x9d, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe3, 0x85, 0x8b, 0x65, 0xef, 0xb4, 0xb9, 0x75, 0xec, 0x9f, 0xa3, 0x61, 0xed, 0x9c, 0xa7, 0x75, 0xec, 0xbc, 0x9b, 0xc3, 0x95, 0xec, 0xb7, 0x9c, 0xed, 0x83, 0x85, 0xec, 0xa6, 0xa5, 0xf0, 0x9d, 0x97, 0x8f, 0xec, 0x86, 0x9e, 0xec, 0xa3, 0xa5, 0x6f, 0xeb, 0xa8, 0x93, 0x75, 0xeb, 0xb8, 0xb7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0xb6, 0x98, 0xeb, 0xbc, 0xb3, 0x65, 0xeb, 0xba, 0xac, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x97, 0x90, 0x61, 0xec, 0xa2, 0xa8, 0x75, 0xeb, 0xad, 0x94, 0xe6, 0x83, 0xa1, 0x61, 0xec, 0xb8, 0x8c, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0xb2, 0x88, 0x69, 0xeb, 0x83, 0x80, 0xea, 0xbd, 0x8d, 0xec, 0xa0, 0x9e, 0x65, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x8f, 0xcc, 0x95, 0x62, 0x65, 0xc5, 0x91, 0xef, 0xbc, 0x9e],
    "𝓏𝜞뼻껩뤟a쉈à֮҅̕boa〪̖ᨘ֚b𝚓펉쁸oﱚoa֮︥̀̕bｺe렱o퍵폈a헽ﴭu뚋ナノä֮́̀̕bi냤u틍괘𝙇oa〪̖ꪴ֚ba쓂abcdefg뽥a콜곝abcdefgㅋeﴹu쟣a휧u켛Õ췜탅즥𝗏솞죥o먓u븷abcdefg붘뼳e뺬abcdefg뗐a좨u뭔惡a츌hello첈i냀꽍젞eà֮𞀏̕beő＞"
  ),
  (
    [0xeb, 0xbf, 0x9c, 0xeb, 0xac, 0xae, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xef, 0xb0, 0xa6, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x84, 0x98, 0x69, 0xe2, 0x9a, 0x89, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe1, 0xbc, 0xb3, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xef, 0xae, 0x83, 0x69, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe7, 0xab, 0x8b, 0xec, 0xa0, 0x84, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe2, 0xbf, 0x80, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xd7, 0x91, 0xd6, 0xbc, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe2, 0x9a, 0x89, 0xec, 0xb2, 0xbd, 0x75, 0xe7, 0xab, 0x8b, 0xea, 0xb1, 0xb7, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x91, 0x8d, 0x8b, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9e, 0xa3, 0x92, 0xd6, 0x9a, 0x62, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xea, 0xb1, 0xb1, 0xe2, 0xbe, 0xb6, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xce, 0xac, 0x61, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x91, 0x91, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x88, 0xb7, 0xf0, 0x9f, 0x98, 0x82, 0xe7, 0xab, 0x8b, 0xec, 0x94, 0x9e, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0xba, 0xa6, 0xeb, 0x94, 0x9d, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xf0, 0x9d, 0x9e, 0x8d, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xec, 0xa3, 0x88, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x85, 0xac, 0xef, 0xb3, 0x81, 0x6f, 0xec, 0x86, 0xb0, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0xba, 0xa6, 0xeb, 0x82, 0x84, 0x75, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe7, 0xab, 0x8b, 0xec, 0x9c, 0xb7, 0xeb, 0x9d, 0xa0, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xed, 0x8b, 0xb7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xa3, 0x84, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0x65, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x88, 0xa9, 0xed, 0x8a, 0xbf, 0xea, 0xba, 0xbd, 0xf0, 0x9f, 0x98, 0x82, 0xf0, 0x9d, 0x9c, 0x8b, 0x69, 0xf0, 0x9f, 0xa4, 0x94, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe2, 0xbd, 0xb9, 0x61, 0xe2, 0x9a, 0x89, 0xe7, 0xab, 0x8b, 0xe2, 0xbe, 0xbd, 0xe5, 0x88, 0xa9, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x9c, 0xab, 0xd6, 0xb0, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xed, 0x91, 0x98, 0xec, 0x82, 0x9f, 0x65, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe7, 0xab, 0x8b, 0xeb, 0xac, 0xaa, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xef, 0xb6, 0x87, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x9d, 0x8b, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xb7, 0x90, 0x61, 0xd6, 0xae, 0xea, 0x99, 0xba, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x6f, 0xe7, 0xab, 0x8b, 0xeb, 0xb4, 0xa3, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe7, 0xab, 0x8b, 0xea, 0xbb, 0xa8, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x9c, 0xbd, 0xf0, 0x9f, 0x98, 0x82, 0xd1, 0x97, 0xe2, 0x9a, 0x89, 0xe5, 0x88, 0xa9, 0xeb, 0xa7, 0x85, 0x75, 0xe7, 0xab, 0x8b, 0xf0, 0x9d, 0x99, 0xa8, 0xe5, 0x88, 0xa9, 0xef, 0xb0, 0xb2, 0x75, 0xe9, 0x90, 0x95, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xe3, 0x88, 0xba, 0x61, 0xe5, 0xba, 0xa6, 0xeb, 0x89, 0xab, 0xe2, 0x9a, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x97, 0xb8, 0x65, 0xe7, 0xab, 0x8b, 0xec, 0x98, 0xb9, 0xe2, 0x9a, 0x89, 0xe5, 0xba, 0xa6, 0xec, 0xbb, 0xac, 0x69, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xea, 0xbe, 0xb3, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0xba, 0xa6, 0xeb, 0xa3, 0x9d, 0xeb, 0x8b, 0xbb, 0xec, 0x9d, 0x8a, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xef, 0xbb, 0x89, 0x6f, 0xe5, 0xba, 0xa6, 0xea, 0xbd, 0xa1, 0x61, 0xeb, 0xb8, 0x9c, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe2, 0x9a, 0x89, 0xed, 0x93, 0x8d, 0x65, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0xa5, 0x8e, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x85, 0xac, 0xf0, 0x9d, 0x91, 0xb7, 0x69, 0xec, 0x93, 0x9b, 0x75, 0xeb, 0x9c, 0xbd, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe2, 0x9a, 0x89, 0xea, 0xb3, 0x95, 0xe2, 0x9a, 0x89, 0xe5, 0x88, 0xa9, 0xec, 0xa0, 0xa1, 0xe5, 0x88, 0xa9, 0xf0, 0x9d, 0x95, 0x81, 0xe7, 0xab, 0x8b],
    "뿜묮abcdefgﰦhello😂넘i⚉在线服务和个ἳ😂在线服务和个ﮃi👨‍👩‍👧立전拥有⿀abcdefgבּhello⚉첽u立걷hello𑍋abcdefga〪̖𞣒֚bo在线服务和个걱⾶🇨🇦άa🌺🌸둑🌺🌸눷😂立씞🇨🇦度딝在线服务和个𝞍hello利죈🇨🇦公ﳁo솰hello😂度낄u👨‍👩‍👧立윷띠abcdefg🌺🌸틷拥有죄̴ᆮe😂利튿꺽😂𝜋i🤔拥有⽹a⚉立⾽利a゙्𑜫ְbabcdefg🌺🌸푘삟e👨‍👩‍👧立묪hello👨‍👩‍👧ﶇ🇨🇦띋abcdefg👨‍👩‍👧뷐a֮ꙺ̀̕bo立봣🇨🇦立껨🇨🇦在线服务和个윽😂ї⚉利맅u立𝙨利ﰲu鐕hello利㈺a度뉫⚉拥有헸e立옹⚉度컬i🇨🇦꾳🌺🌸度룝닻읊🌺🌸ﻉo度꽡a븜abcdefg⚉퓍e🇨🇦쥎😂公𝑷i쓛u뜽abcdefg⚉곕⚉利젡利𝕁立"
  ),
  (
    [0xf0, 0x9d, 0x93, 0xa7, 0xf0, 0x9f, 0xa4, 0x94, 0xef, 0xb0, 0xbe, 0xeb, 0x89, 0xab, 0xe7, 0xab, 0x8b, 0xec, 0xb6, 0xbd, 0xe2, 0x9a, 0x89, 0xe5, 0x85, 0xac, 0xe3, 0x86, 0x94, 0xe5, 0x88, 0xa9, 0xed, 0x87, 0xb0, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xa8, 0xb6, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xa2, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xef, 0xb1, 0xa9, 0xe3, 0x8e, 0x91, 0xeb, 0xa0, 0xac, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0xba, 0xa6, 0xed, 0x81, 0x80, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe7, 0xab, 0x8b, 0x61, 0xe0, 0xb8, 0xb8, 0xe0, 0xb9, 0x89, 0xe0, 0xb9, 0x88, 0xe0, 0xba, 0xb8, 0x62, 0xf0, 0x9d, 0x97, 0xb3, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xf0, 0x9d, 0x97, 0xb5, 0xe2, 0x9a, 0x89, 0xec, 0xb0, 0x80, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0xba, 0xa6, 0xea, 0xb0, 0x81, 0xf0, 0x9f, 0x98, 0x82, 0x61, 0xd6, 0xae, 0xcd, 0xa5, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xaa, 0xb4, 0xcc, 0x95, 0x62, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xa5, 0x92, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x8a, 0x86, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe5, 0xba, 0xa6, 0xec, 0x87, 0x87, 0xf0, 0x9f, 0x98, 0x82, 0xe2, 0xbf, 0x8a, 0xe5, 0xba, 0xa6, 0xec, 0xa5, 0xa2, 0xe5, 0x88, 0xa9, 0xed, 0x85, 0xb4, 0xea, 0xb5, 0xb7, 0xe2, 0x9a, 0x89, 0xe7, 0xab, 0x8b, 0xeb, 0x87, 0xa2, 0xe7, 0xab, 0x8b, 0xec, 0x87, 0x8a, 0xed, 0x9b, 0xaa, 0xf0, 0x9f, 0x98, 0x82, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe1, 0xb8, 0x89, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0x97, 0xa8, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x8a, 0x84, 0xe2, 0x9a, 0x89, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xa5, 0x96, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xb4, 0xbc, 0xed, 0x99, 0xa1, 0xe5, 0x88, 0xa9, 0xca, 0xb6, 0xf0, 0x9f, 0xa4, 0x94, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0xa1, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0xa4, 0x94, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x91, 0xa3, 0xe5, 0x88, 0xa9, 0xeb, 0xb9, 0x9b, 0xf0, 0x9f, 0xa4, 0x94, 0xef, 0xb3, 0x8c, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x98, 0xb5, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xc3, 0xa0, 0xd6, 0xae, 0xe2, 0x83, 0x97, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x85, 0xac, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xf0, 0x91, 0x9a, 0xb6, 0xd6, 0xb0, 0x62, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xaa, 0x90, 0xe2, 0x9a, 0x89, 0xe5, 0x88, 0xa9, 0xed, 0x9c, 0x9e, 0xed, 0x87, 0x9e, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xbf, 0xb0, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x97, 0x88, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x88, 0xa9, 0xec, 0xb2, 0xac, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe1, 0xbd, 0x80, 0xeb, 0xbb, 0x9a, 0xe2, 0x9a, 0x89, 0xe5, 0x85, 0xac, 0xec, 0xab, 0xa2, 0xe2, 0x9a, 0x89, 0xe8, 0x88, 0x81, 0xec, 0x9a, 0x85, 0xeb, 0xae, 0x90, 0xe7, 0xab, 0x8b, 0xe3, 0x81, 0x8e, 0xe7, 0xab, 0x8b, 0xf0, 0x9d, 0x98, 0x89, 0xec, 0x97, 0xb1, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x93, 0x80, 0xe5, 0x85, 0xac, 0xeb, 0x92, 0x82, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x96, 0x93, 0xe5, 0x85, 0xac, 0xe1, 0xb9, 0x87, 0xed, 0x96, 0xa9, 0xe3, 0x8f, 0xa9, 0xe7, 0xab, 0x8b, 0x61, 0xe3, 0x80, 0xaa, 0xe1, 0xb7, 0xbd, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xf0, 0x9f, 0xa4, 0x94, 0xed, 0x96, 0xbf, 0xe5, 0xba, 0xa6, 0xf0, 0x9d, 0x95, 0x84, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe2, 0x81, 0xb1, 0xe7, 0xab, 0x8b, 0xf0, 0x9f, 0x84, 0xa3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe5, 0x8a, 0x9b],
    "𝓧🤔ﰾ뉫立춽⚉公㆔利퇰👨‍👩‍👧먶👨‍👩‍👧a֮ⷢ̀̕bﱩ㎑렬🤔度큀🌺🌸立aุ้่ຸb𝗳🇨🇦𝗵⚉찀🌺🌸度각😂a֮ͥ̀̕bà֮᪴̕b在线服务和个륒🌺🌸在线服务和个튆👨‍👩‍👧度쇇😂⿊度쥢利텴굷⚉立뇢立쇊훪😂拥有ḉ🤔뗨🇨🇦슄⚉在线服务和个쥖🇨🇦拥有촼홡利ʶ🤔à֮ᷡ̕b🤔拥有푣利빛🤔ﳌ👨‍👩‍👧옵拥有à֮⃗̕b🌺🌸公a゙्𑚶ְb😂몐⚉利휞퇞拥有꿰😂허🌺🌸利첬🌺🌸ὀ뻚⚉公쫢⚉舁욅뮐立ぎ立𝘉엱😂퓀公뒂😂在线服务和个햓公ṇ햩㏩立a〪᷽̖֚b🤔햿度𝕄🇨🇦在线服务和个ⁱ立🄣拥有力"
  ),
  (
    [0xed, 0x96, 0x99, 0xe5, 0xba, 0xa6, 0xe1, 0xac, 0x8b, 0xcc, 0xb4, 0xe1, 0xac, 0xb5, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x81, 0xaf, 0xe5, 0xba, 0xa6, 0xed, 0x95, 0xa6, 0xe5, 0x88, 0xa9, 0xd9, 0x82, 0xd9, 0x89, 0xc3, 0x87, 0xed, 0x9b, 0xa8, 0xe5, 0xba, 0xa6, 0xea, 0xbb, 0x96, 0xeb, 0x88, 0xb6, 0xec, 0x83, 0xb5, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xb9, 0xb9, 0xec, 0x96, 0xb9, 0xe1, 0xbe, 0xa4, 0xec, 0x81, 0x8d, 0xe5, 0xba, 0xa6, 0xeb, 0x84, 0xa1, 0xe7, 0xab, 0x8b, 0xeb, 0x8d, 0x86, 0xe7, 0xab, 0x8b, 0xf0, 0xa2, 0x86, 0x9f, 0xec, 0xb4, 0x98, 0xe5, 0xba, 0xa6, 0xec, 0x9c, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xb7, 0x84, 0xe5, 0x88, 0xa9, 0xd9, 0x85, 0xd8, 0xae, 0xec, 0x86, 0xa4, 0x28, 0x6f, 0x29, 0xe5, 0x85, 0xac, 0xe1, 0xbd, 0x97, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x95, 0xb6, 0xeb, 0x85, 0x98, 0xeb, 0x99, 0x83, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x93, 0x84, 0xec, 0xb0, 0xa3, 0xe7, 0xab, 0x8b, 0xeb, 0xb8, 0x97, 0xe5, 0x88, 0xa9, 0xe9, 0x89, 0xb6, 0xe5, 0x88, 0xa9, 0xec, 0x94, 0xa7, 0xe5, 0x88, 0xa9, 0xeb, 0x8b, 0x85, 0xea, 0xb6, 0xaa, 0xed, 0x99, 0xb2, 0xe5, 0xba, 0xa6, 0xe8, 0xad, 0x98, 0xec, 0x81, 0x90, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9b, 0xa3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x8b, 0xa7, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0xa6, 0xcc, 0x95, 0x62, 0xe5, 0xba, 0xa6, 0xeb, 0x86, 0x9d, 0xea, 0xb3, 0xac, 0xe6, 0xbc, 0xa2, 0xe5, 0xba, 0xa6, 0xeb, 0xbe, 0xaa, 0xec, 0x8d, 0x96, 0xe5, 0xba, 0xa6, 0x61, 0xd6, 0xae, 0xdb, 0x9c, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb6, 0x98, 0xe7, 0xab, 0x8b, 0xec, 0x96, 0x98, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xea, 0xa7, 0x80, 0xd6, 0xb0, 0x62, 0xe6, 0xb9, 0xae, 0xec, 0x9e, 0xba, 0xeb, 0xbf, 0xa9, 0xe5, 0xba, 0xa6, 0xeb, 0xb8, 0x98, 0xe5, 0x85, 0xac, 0xe1, 0xb8, 0x87, 0xec, 0x87, 0x92, 0xeb, 0xbc, 0xb3, 0xe5, 0x85, 0xac, 0xeb, 0xba, 0x8c, 0xe5, 0xba, 0xa6, 0xe6, 0xb5, 0xaa, 0xeb, 0xb1, 0xa0, 0xe5, 0x85, 0xac, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa5, 0x8d, 0xe0, 0xb4, 0xbc, 0xd6, 0xb0, 0x62, 0xec, 0xaf, 0x86, 0xed, 0x9a, 0xbf, 0xe5, 0xba, 0xa6, 0xea, 0xb8, 0xa4, 0xe7, 0xab, 0x8b, 0xec, 0xa2, 0x96, 0xe5, 0xba, 0xa6, 0xed, 0x9c, 0x85, 0xe7, 0xab, 0x8b, 0xc8, 0xb7, 0xe5, 0xba, 0xa6, 0xec, 0xba, 0x8a, 0xe5, 0x85, 0xac, 0x61, 0xd9, 0xb0, 0xdc, 0x91, 0xdc, 0x91, 0xe0, 0xb1, 0x95, 0x62, 0xeb, 0xa6, 0xb4, 0xe7, 0xab, 0x8b, 0xec, 0xbb, 0x8a, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa],
    "햙度ᬋ̴ᬵ在线服务和个쁯度핦利قىÇ훨度껖눶샵在线服务和个칹얹ᾤ쁍度넡立덆立𢆟촘度윉拥有췄利مخ솤(o)公ὗ在线服务和个땶녘뙃在线服务和个퓄찣立븗利鉶利씧利닅궪홲度識쁐在线服务和个훣拥有닧à֮ᷦ̕b度놝곬漢度뾪썖度a֮ۜ̀̕b拥有붘立얘拥有a゙्꧀ְb湮잺뿩度븘公ḇ쇒뼳公뺌度浪뱠公a゙्഼ְb쯆횿度긤立좖度휅立ȷ度캊公aٰܑܑౕb릴立컊在线服务和个"
  ),
  (
    [0xec, 0xaa, 0xb6, 0xed, 0x99, 0xb8, 0xd9, 0x83, 0xeb, 0xa9, 0xbf, 0xe8, 0x8f, 0x8c, 0xe3, 0x9b, 0xbc, 0xea, 0xb5, 0xab, 0xec, 0x87, 0x93, 0xec, 0x97, 0xb7, 0xec, 0x82, 0xa4, 0xeb, 0x95, 0xb2, 0xec, 0x8e, 0xa3, 0xeb, 0xa7, 0x82, 0xe5, 0x9c, 0x97, 0xec, 0x8f, 0xa2, 0xeb, 0xa4, 0xbb, 0xec, 0x9c, 0xaa, 0xeb, 0xa3, 0x90, 0xed, 0x82, 0x90, 0xed, 0x84, 0xa2, 0xed, 0x82, 0x97, 0xeb, 0xbe, 0x90, 0xec, 0xab, 0xa7, 0xeb, 0xbd, 0x87, 0xeb, 0xa8, 0x95, 0xce, 0xb6, 0xeb, 0x9d, 0xa4, 0xec, 0xaf, 0x8d, 0xec, 0x91, 0x98, 0xea, 0xb1, 0xb7, 0xec, 0x81, 0xb6, 0xeb, 0xbf, 0xa3, 0xe9, 0x8b, 0x98, 0xed, 0x82, 0x86, 0xea, 0xb7, 0xbb, 0xeb, 0xa8, 0xa2, 0xed, 0x8e, 0x93, 0xea, 0xb3, 0x92, 0xec, 0x9f, 0x82, 0xec, 0x9d, 0x98, 0xe1, 0xba, 0xb5, 0xc3, 0xa0, 0xd6, 0xae, 0xdb, 0xa0, 0xcc, 0x95, 0x62, 0xec, 0x96, 0x82, 0xeb, 0xa6, 0x95, 0xec, 0x9d, 0x9d, 0xec, 0xb9, 0xad, 0x61, 0xcc, 0xb4, 0xf0, 0x91, 0x8c, 0xbc, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xeb, 0x82, 0x95, 0xec, 0xac, 0x92, 0xeb, 0xa4, 0xbc, 0xeb, 0x89, 0xb6, 0xeb, 0x82, 0xb7, 0xeb, 0x88, 0x8b, 0xeb, 0xb6, 0x90, 0x61, 0xe0, 0xb9, 0x88, 0xe0, 0xba, 0xb8, 0xe0, 0xba, 0xb8, 0xe0, 0xbb, 0x88, 0x62, 0xec, 0xbc, 0xa0, 0xec, 0xbb, 0xa9, 0xea, 0xbe, 0x87, 0xe1, 0x84, 0xa1, 0xe1, 0xb9, 0xa3, 0xec, 0x92, 0xae, 0xeb, 0xaf, 0xac, 0xed, 0x86, 0x84, 0xed, 0x93, 0xbe, 0xeb, 0xbf, 0xa4, 0xe5, 0x87, 0x9e, 0xed, 0x88, 0xb5, 0xe6, 0x86, 0xb2, 0xeb, 0xa4, 0x87, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xb7, 0xbf, 0xd6, 0x9a, 0x62],
    "쪶홸ك멿菌㛼굫쇓엷삤땲쎣맂圗쏢뤻윪룐킐턢킗뾐쫧뽇먕ζ띤쯍쑘걷쁶뿣鋘킆귻먢펓곒쟂의ẵà֮۠̕b얂릕읝칭a̴𑌼़゙b낕쬒뤼뉶낷눋붐a่ຸຸ່b켠컩꾇ᄡṣ쒮믬톄퓾뿤凞툵憲뤇a〪̖᷿֚b"
  ),
  (
    [0xf0, 0x9d, 0x94, 0x8e, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0x99, 0xd6, 0x9a, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x9a, 0xbd, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe1, 0xbb, 0x85, 0x69, 0xec, 0xa8, 0x8d, 0xeb, 0xa6, 0x9f, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xf0, 0x9d, 0x95, 0x86, 0x75, 0xd9, 0x86, 0xd8, 0xac, 0xd9, 0x8a, 0x61, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0x8f, 0x9d, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe2, 0xbc, 0xa2, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe0, 0xbc, 0x99, 0xd6, 0x9a, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x9a, 0x8f, 0xef, 0xbd, 0x91, 0xe2, 0xbd, 0xa3, 0x69, 0xeb, 0x9a, 0x92, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xed, 0x8d, 0x8c, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9d, 0x97, 0x86, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0x9e, 0x81, 0x75, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0x98, 0x90, 0x6f, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe4, 0x8d, 0x99, 0xeb, 0x9d, 0xa4, 0x65, 0xe9, 0x9a, 0xb9, 0x6f, 0x6a, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xa4, 0x80, 0x6f, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xef, 0xb8, 0x99, 0xe2, 0x9a, 0x89, 0xea, 0xbb, 0x9f, 0xe2, 0x82, 0x96, 0x69, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xf0, 0x9d, 0x9d, 0x99, 0x65, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe1, 0xbc, 0xa7, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xef, 0xb4, 0xba, 0xf0, 0x9e, 0xb8, 0xac, 0x6f, 0xd8, 0xaa, 0xd8, 0xae, 0xeb, 0x8f, 0x89, 0xec, 0xb0, 0x83, 0x69, 0x6b, 0x74, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xa9, 0xa9, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xed, 0x80, 0x85, 0xec, 0x8b, 0x92, 0xe2, 0x9a, 0x89, 0xe5, 0xae, 0x80, 0xed, 0x85, 0x9f, 0xec, 0xa0, 0x94, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x88, 0x83, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xed, 0x91, 0xb1, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0xba, 0xbe, 0xf0, 0x9f, 0xa4, 0x94, 0xef, 0xb1, 0xa4, 0x6f, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0xa5, 0xba, 0x6f, 0xed, 0x92, 0x89, 0x6f, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xed, 0x8a, 0x91, 0x61, 0x62, 0xef, 0xac, 0xa9, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x92, 0x9a, 0xec, 0xb3, 0xa8, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x61, 0xcc, 0xb4, 0xf0, 0x96, 0xab, 0xb0, 0xe0, 0xa4, 0xbc, 0x62, 0xf0, 0x9f, 0x98, 0x82, 0xc4, 0xb0, 0xf0, 0x9f, 0x98, 0x82, 0x28, 0xed, 0x83, 0x80, 0x29, 0x61, 0xe2, 0x9a, 0x89, 0xe0, 0xb3, 0x8a, 0xea, 0xb5, 0xb6, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe7, 0x88, 0xb6, 0xf0, 0x9f, 0xa4, 0x94, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xbb, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xb4, 0x62, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe3, 0x8a, 0xbc, 0x65, 0xe2, 0x9a, 0x89, 0xeb, 0xb2, 0xa5, 0x75, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0x9b, 0xa0, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xea, 0xb8, 0xbc, 0xea, 0xb9, 0xa2, 0x6f, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x84, 0xcc, 0x95, 0x62, 0xeb, 0x96, 0x99, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xc9, 0xb0, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x93, 0xb7, 0xeb, 0x9c, 0xad, 0x75, 0xe2, 0xbd, 0xbf],
    "𝔎a〪̖̙֚babcdefg횽🇨🇦ễi쨍릟👨‍👩‍👧𝕆uنجيa🤔돝👨‍👩‍👧⼢a〪̖༙֚babcdefg👨‍👩‍👧욏ｑ⽣i뚒🇨🇦퍌abcdefg𝗆🤔잁u🤔옐o👨‍👩‍👧䍙띤e隹oj👨‍👩‍👧뤀o🌺🌸︙⚉껟ₖi🇨🇦𝝙e🇨🇦ἧ👨‍👩‍👧ﴺ𞸬oتخ돉찃ikt😂멩👨‍👩‍👧퀅싒⚉宀텟젔abcdefg😂툃👨‍👩‍👧푱🌺🌸캾🤔ﱤo😂쥺o풉o👨‍👩‍👧튑ab﬩hello撚쳨🌺🌸a̴𖫰़b😂İ😂(타)a⚉ೊ굶🇨🇦父🤔aཱཻིུb🇨🇦㊼e⚉벥u👨‍👩‍👧뛠🇨🇦긼깢oà֮𞀄̕b떙helloɰ🇨🇦쓷뜭u⽿"
  ),
  (
    [0xec, 0xb1, 0x87, 0xec, 0x80, 0xb6, 0xe4, 0x8d, 0x99, 0xe1, 0xbd, 0xa0, 0xe3, 0x8b, 0x83, 0xec, 0xa0, 0xac, 0xef, 0xbc, 0xb1, 0xeb, 0xa8, 0x92, 0xf0, 0x9d, 0x92, 0xaa, 0xeb, 0xbb, 0x90, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xe1, 0xbb, 0x9d, 0xeb, 0x9c, 0x83, 0xe1, 0xbd, 0xa4, 0xd3, 0x9a, 0xef, 0xb0, 0xbd, 0xe3, 0xa0, 0xaf, 0xed, 0x9b, 0x85, 0xeb, 0x98, 0xb0, 0xec, 0x98, 0x8f, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdf, 0xbd, 0xd6, 0x9a, 0x62, 0xec, 0x8d, 0x90, 0xeb, 0xb9, 0xa1, 0xeb, 0x9b, 0xa0, 0xed, 0x8a, 0x95, 0xec, 0x98, 0xb1, 0xeb, 0xa1, 0xa1, 0xe3, 0x89, 0xae, 0xeb, 0x92, 0xa0, 0xed, 0x89, 0x80, 0xe4, 0x9a, 0xbe, 0xeb, 0xbd, 0x9a, 0xf0, 0x9d, 0x9d, 0x80, 0xeb, 0xb4, 0xb9, 0xea, 0xb7, 0x8e, 0xeb, 0x83, 0xa8, 0xea, 0xb1, 0x97, 0x61, 0xe3, 0x80, 0xaa, 0xe0, 0xa3, 0xa6, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xef, 0xb9, 0x99, 0xeb, 0x8f, 0xba, 0xeb, 0xb7, 0xa3, 0xea, 0xbf, 0xae, 0xeb, 0x97, 0x81, 0xec, 0x98, 0x81, 0xed, 0x97, 0xa6, 0xed, 0x98, 0x9e, 0xc3, 0xa0, 0xd6, 0xae, 0xef, 0xb8, 0xa4, 0xcc, 0x95, 0x62, 0xe1, 0xb6, 0xa8, 0xf0, 0x9d, 0x9c, 0xb3, 0xec, 0xae, 0xa3, 0xc3, 0xa0, 0xd6, 0xae, 0xd6, 0x9c, 0xcc, 0x95, 0x62, 0xe3, 0x85, 0xb3, 0xeb, 0x9d, 0xb3, 0x61, 0xd8, 0x98, 0xd9, 0x8f, 0xd8, 0x99, 0xd8, 0x9a, 0x62, 0xf0, 0x9e, 0xb8, 0xae, 0xf0, 0x9e, 0xb8, 0xab, 0xe3, 0x8a, 0x8a, 0xec, 0x97, 0x82, 0xef, 0xb0, 0xa0, 0xea, 0xb2, 0xbf, 0xe2, 0xbd, 0xbf, 0xec, 0xa4, 0xa7, 0xec, 0xa9, 0xa8, 0xec, 0xb1, 0x8a, 0xe1, 0xbf, 0xba, 0xe2, 0x86, 0x9a, 0xc3, 0x9a, 0xeb, 0x89, 0x81, 0xec, 0xb7, 0xba, 0x61, 0xe3, 0x80, 0xaa, 0xdb, 0xa3, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xef, 0xbb, 0x80, 0xec, 0xa2, 0xae, 0xed, 0x83, 0xb8, 0xeb, 0x88, 0xb3, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0x9e, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe1, 0xb8, 0x99, 0xeb, 0x9a, 0xaf, 0xc3, 0xa0, 0xd6, 0xae, 0xdc, 0xb0, 0xcc, 0x95, 0x62, 0xea, 0xbf, 0x98, 0xec, 0xb0, 0x86, 0xe8, 0x99, 0x9c],
    "챇쀶䍙ὠ㋃젬Ｑ먒𝒪뻐̴ᆮờ뜃ὤӚﰽ㠯훅똰옏a〪̖߽֚b썐빡뛠튕옱롡㉮뒠퉀䚾뽚𝝀봹귎냨걗a〪ࣦ̖֚b﹙돺뷣꿮뗁영헦혞à֮︤̕bᶨ𝜳쮣à֮֜̕bㅳ띳aؘُؙؚb𞸮𞸫㊊엂ﰠ겿⽿줧쩨챊Ὼ↚Ú뉁췺a〪ۣ̖֚bﻀ좮탸눳a֮ࣞ̀̕bḙ뚯à֮ܰ̕b꿘찆虜"
  ),
  (
    [0xec, 0x87, 0x93, 0xec, 0xa8, 0xac, 0x75, 0xea, 0xbc, 0x87, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x8a, 0x95, 0xe5, 0x88, 0xa9, 0xec, 0xa4, 0xbb, 0xf0, 0x9f, 0x88, 0x9c, 0x69, 0xe5, 0x88, 0xa9, 0xea, 0xbc, 0x81, 0xe5, 0x85, 0xac, 0xf0, 0x9e, 0xba, 0xa8, 0x69, 0xef, 0xb4, 0xb7, 0xe5, 0xba, 0xa6, 0xeb, 0xa7, 0xb6, 0x69, 0xed, 0x86, 0xb5, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xea, 0xbc, 0x81, 0xe5, 0x88, 0xa9, 0xeb, 0x97, 0xa2, 0x75, 0x61, 0xe3, 0x80, 0xaa, 0xe0, 0xa3, 0xaf, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x61, 0xed, 0x9a, 0xad, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x61, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0xe0, 0xaa, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0x61, 0xd6, 0xae, 0xe1, 0xad, 0xab, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe5, 0x88, 0xa9, 0xe2, 0x93, 0x98, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x84, 0x82, 0x65, 0xeb, 0x82, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xb4, 0x84, 0x75, 0xeb, 0xbd, 0xb0, 0x75, 0xd7, 0x91, 0xd6, 0xbc, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x61, 0xe3, 0x80, 0xaa, 0xdf, 0xbd, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xeb, 0x8b, 0x8a, 0x69, 0x61, 0xe0, 0xb9, 0x88, 0xe0, 0xba, 0xb8, 0xe0, 0xba, 0xb8, 0xe0, 0xbb, 0x88, 0x62, 0xe5, 0xba, 0xa6, 0xec, 0x8b, 0xac, 0x65, 0xe5, 0x85, 0xac, 0xec, 0x8d, 0xb7, 0xe5, 0x88, 0xa9, 0xc7, 0x99, 0x69, 0xec, 0x80, 0xbf, 0x61, 0xe5, 0xba, 0xa6, 0xed, 0x87, 0x9c, 0xe5, 0x85, 0xac, 0xe3, 0xa0, 0xaf, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb4, 0x9a, 0x75, 0xe5, 0xba, 0xa6, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xd9, 0x95, 0xd6, 0x9a, 0x62, 0x65, 0xed, 0x81, 0x99, 0xea, 0xb1, 0x8c, 0x69, 0xeb, 0x88, 0x83, 0x69, 0xed, 0x82, 0x88, 0xcc, 0xb4, 0xe1, 0x86, 0xbc, 0x75, 0xeb, 0x81, 0x8d, 0x69, 0xe5, 0x88, 0xa9, 0xf0, 0x9d, 0x93, 0x8e, 0xcf, 0x94, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x8a, 0x8e, 0x61, 0xeb, 0x8a, 0xad, 0x75, 0xe5, 0x85, 0xac, 0xea, 0xb1, 0xae, 0xe5, 0x88, 0xa9, 0xed, 0x87, 0xbb, 0xeb, 0x97, 0xb8, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xec, 0x95, 0x9a, 0x75, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x8e, 0x8a, 0xec, 0x87, 0xa2, 0x65, 0xe5, 0x88, 0xa9, 0xec, 0x9f, 0x86, 0xe5, 0x85, 0xac, 0xeb, 0x90, 0xb2, 0xeb, 0x8b, 0xa3, 0xe5, 0x88, 0xa9, 0xe7, 0x81, 0x8a, 0xec, 0xb0, 0xb4, 0xe5, 0x88, 0xa9, 0xe3, 0x8a, 0x93, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9d, 0x99, 0xa4, 0x61, 0xed, 0x9d, 0x8c, 0x61, 0xed, 0x9c, 0x95, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x8a, 0x91, 0x69, 0xe8, 0x9d, 0xb9, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x87, 0xb3, 0xe7, 0x8e, 0x87, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x89, 0xb7, 0xec, 0x92, 0xb6, 0x6f, 0xe5, 0x85, 0xac, 0xec, 0xbf, 0xa9, 0xe7, 0xab, 0x8b, 0xec, 0xb5, 0x91, 0xe1, 0xba, 0xa9, 0x6f, 0xe5, 0x85, 0xac, 0xed, 0x88, 0xb3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0xba, 0xa6, 0xeb, 0xbc, 0x82, 0x65, 0xeb, 0xbc, 0x9b, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0x86, 0xa5, 0xeb, 0x9a, 0xa4, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xf0, 0x9d, 0x91, 0x90, 0x65, 0xe7, 0xab, 0x8b, 0xec, 0xbc, 0x90, 0xe7, 0xab, 0x8b, 0xed, 0x86, 0x9d, 0xf0, 0x9e, 0xb8, 0xb4, 0xe5, 0x85, 0xac, 0xeb, 0x90, 0xba, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xf0, 0xa5, 0xa5, 0xbc, 0xe5, 0x85, 0xac, 0xef, 0xb5, 0xa2, 0xed, 0x9d, 0xad, 0xe5, 0x85, 0xac],
    "쇓쨬u꼇在线服务和个튕利줻🈜i利꼁公𞺨iﴷ度맶i통abcdefg꼁利뗢ua〪࣯̖֚ba횭abcdefga̴़઼゙ba֮᭫̀̕b利ⓘ拥有넂e낉拥有촄u뽰uבּabcdefga〪߽̖֚b닊ia่ຸຸ່b度심e公썷利Ǚi쀿a度퇜公㠯拥有봚u度a〪̖ٕ֚be큙걌i눃i킈̴ᆼu끍i利𝓎ϔu拥有늎a늭u公걮利퇻뗸hello利앚u在线服务和个쎊쇢e利쟆公됲닣利灊찴利㊓abcdefg𝙤a흌a휕abcdefg늑i蝹拥有쇳率abcdefg쉷쒶o公쿩立쵑ẩo公툳abcdefg度뼂e뼛abcdefg솥뚤拥有𝑐e立켐立톝𞸴公됺在线服务和个𥥼公ﵢ흭公"
  ),
  (
    [0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe1, 0xaa, 0xbd, 0xd6, 0x9a, 0x62, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x87, 0x86, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x82, 0x8c, 0xec, 0x86, 0xbf, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe1, 0xb8, 0x8f, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x80, 0xb1, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x86, 0xbb, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe7, 0x80, 0x9b, 0xec, 0xa5, 0xbd, 0xe2, 0x9a, 0x89, 0xeb, 0xaf, 0xb9, 0xf0, 0x9f, 0xa4, 0x94, 0xed, 0x9c, 0xb7, 0xe2, 0x9a, 0x89, 0x48, 0x67, 0x61, 0xe0, 0xbd, 0xb1, 0xe0, 0xbd, 0xbc, 0xe0, 0xbd, 0xb2, 0xe0, 0xbd, 0xb4, 0x62, 0xe1, 0x84, 0x92, 0xcc, 0xb4, 0xe1, 0x85, 0xad, 0xeb, 0xae, 0x81, 0xf0, 0x9f, 0xa4, 0x94, 0xe1, 0xba, 0xab, 0xec, 0xbf, 0xbe, 0xe2, 0x9a, 0x89, 0x47, 0x48, 0x7a, 0xec, 0x97, 0x83, 0xca, 0x8a, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xa5, 0x86, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0x97, 0xd6, 0x9a, 0x62, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0xa8, 0xaa, 0xd9, 0x86, 0xd9, 0x85, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0xae, 0x9f, 0xed, 0x97, 0x91, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0xb8, 0x9e, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xa6, 0x9b, 0xeb, 0xb7, 0xbc, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x96, 0xa6, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x9a, 0x8b, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0xa8, 0x9f, 0xf0, 0x9f, 0xa4, 0x94, 0xea, 0xb7, 0xae, 0xed, 0x82, 0xb3, 0xf0, 0x9f, 0xa4, 0x94, 0xeb, 0xb5, 0xbf, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0x80, 0x8c, 0xf0, 0x9f, 0x98, 0x82, 0xe3, 0x83, 0x9d, 0xe3, 0x82, 0xa4, 0xe3, 0x83, 0xb3, 0xe3, 0x83, 0x88, 0xe5, 0x80, 0xab, 0xeb, 0x9e, 0xaa, 0xda, 0x80, 0xe1, 0xb9, 0xbd, 0xed, 0x80, 0x89, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xed, 0x81, 0x82, 0xec, 0x87, 0xa9, 0xe1, 0xbe, 0x85, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0x87, 0x9d, 0xec, 0xac, 0xb3, 0xec, 0xbf, 0xa1, 0xe2, 0x9a, 0x89, 0xed, 0x99, 0xb8, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0xa5, 0xa1, 0xeb, 0x80, 0x91, 0xe2, 0x9a, 0x89, 0xeb, 0x96, 0xbd, 0xec, 0xac, 0xad, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb3, 0xb8, 0xcc, 0x95, 0x62, 0xc4, 0x93, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xd8, 0xb4, 0xd8, 0xad, 0xd9, 0x85, 0xf0, 0x9f, 0xa4, 0x94, 0xd8, 0xa8, 0xd8, 0xad, 0xd9, 0x8a, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xed, 0x84, 0x81, 0xeb, 0xa2, 0xbd, 0xe2, 0x9a, 0x89, 0xe1, 0xba, 0x81, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0x95, 0xa0, 0xe2, 0x9a, 0x89, 0xeb, 0xb7, 0xac, 0xea, 0xb1, 0xab, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x8e, 0xb0, 0xec, 0xa3, 0xa0, 0xed, 0x9a, 0xba, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0x49, 0x49, 0x49, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xed, 0x93, 0xbe, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xa6, 0x82, 0xec, 0xb1, 0xa5, 0xe2, 0x9a, 0x89, 0xeb, 0x91, 0xa7, 0xeb, 0x82, 0x98, 0xcc, 0xb4, 0xe1, 0x86, 0xae, 0xec, 0xb9, 0x9f, 0xe2, 0x9a, 0x89, 0xeb, 0xab, 0xaa, 0xf0, 0x9f, 0xa4, 0x94, 0xea, 0xbc, 0xbf, 0xec, 0x92, 0xa0, 0xe5, 0xb7, 0xa2, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0xa7, 0x89, 0xe2, 0x9a, 0x89, 0xe1, 0xbc, 0x82, 0xe2, 0x9a, 0x89, 0xe4, 0xb8, 0xa6, 0xf0, 0x9f, 0xa4, 0x94],
    "a〪̖᪽֚b🇨🇦쇆🌺🌸삌솿👨‍👩‍👧ḏ🇨🇦뀱🌺🌸놻🇨🇦瀛쥽⚉믹🤔휷⚉Hgaཱོིུbᄒ̴ᅭ뮁🤔ẫ쿾⚉GHz엃ʊ😂륆a〪̖̗֚b😂쨪نم🇨🇦쮟헑🌺🌸븞👨‍👩‍👧릛뷼👨‍👩‍👧얦🇨🇦욋🇨🇦먟🤔귮킳🤔뵿👨‍👩‍👧뀌😂ポイント倫랪ڀṽ퀉🇨🇦큂쇩ᾅ🤔쇝쬳쿡⚉홸😂쥡뀑⚉떽쬭à֮᳸̕bē🇨🇦شحم🤔بحي👨‍👩‍👧턁뢽⚉ẁ🇨🇦애⚉뷬걫👨‍👩‍👧쎰죠횺👨‍👩‍👧III🇨🇦퓾👨‍👩‍👧릂챥⚉둧나̴ᆮ칟⚉뫪🤔꼿쒠巢🌺🌸막⚉ἂ⚉並🤔"
  ),
  (
    [0xed, 0x8f, 0xaf, 0xeb, 0xa5, 0x82, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0xa8, 0x86, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0xab, 0xb6, 0xeb, 0xba, 0x8e, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xed, 0x81, 0x88, 0xe3, 0x92, 0x9e, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe4, 0x98, 0xb5, 0xeb, 0x9c, 0xa6, 0xe7, 0xbd, 0xba, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xed, 0x8b, 0xb7, 0xd8, 0xb9, 0xd8, 0xac, 0xd9, 0x85, 0xed, 0x80, 0xaa, 0xeb, 0xae, 0xa0, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x87, 0xaf, 0xeb, 0x94, 0x90, 0xe1, 0xbb, 0xb2, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x9d, 0xbb, 0xe3, 0x81, 0x94, 0xeb, 0x92, 0x85, 0xec, 0xa8, 0x9e, 0xe2, 0x9a, 0x89, 0xf0, 0xa5, 0x98, 0xa6, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x8d, 0x92, 0xf0, 0x9f, 0xa4, 0x94, 0xed, 0x94, 0xb4, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x86, 0x8e, 0xec, 0xb2, 0x9f, 0xec, 0xa2, 0x96, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x9b, 0x86, 0xed, 0x89, 0xa3, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0x31, 0x30, 0xe6, 0x9c, 0x88, 0xeb, 0x9c, 0x97, 0xe8, 0xbc, 0xb8, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x93, 0xb6, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x9d, 0xbe, 0xf0, 0x9f, 0x98, 0x82, 0xc7, 0x95, 0xeb, 0xae, 0x9a, 0xec, 0x84, 0xa8, 0xec, 0xab, 0x99, 0xe2, 0x9a, 0x89, 0xe9, 0xac, 0x92, 0xec, 0x91, 0xac, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x9b, 0xb3, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xed, 0x94, 0xa0, 0xe2, 0x9a, 0x89, 0xeb, 0xa2, 0xba, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xeb, 0x94, 0x97, 0xe4, 0x95, 0xab, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x92, 0xae, 0xeb, 0xb4, 0x9f, 0xf0, 0x9f, 0xa4, 0x94, 0xed, 0x9a, 0x85, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x95, 0xbf, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x95, 0xbf, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xec, 0x98, 0x83, 0xf0, 0x9f, 0xa4, 0x94, 0x4d, 0x44, 0xed, 0x82, 0xae, 0xed, 0x98, 0x84, 0xed, 0x93, 0xac, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0x61, 0xd6, 0xae, 0xdc, 0xb2, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x9c, 0x88, 0xe3, 0x82, 0xaa, 0xe2, 0x9a, 0x89, 0xed, 0x80, 0xa3, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x9d, 0xcc, 0x95, 0x62, 0xe2, 0x9a, 0x89, 0xeb, 0xa8, 0x95, 0xeb, 0x88, 0x9b, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xeb, 0x9b, 0x88, 0xe2, 0x9a, 0x89, 0xeb, 0x9e, 0x95, 0xe2, 0x9a, 0x89, 0xd8, 0xac, 0xd9, 0x8a, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xd8, 0xa2, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0xab, 0x8a, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xea, 0xb2, 0x9f, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xea, 0xbb, 0x9e, 0xe2, 0x9a, 0x89, 0x61, 0xe3, 0x82, 0x99, 0xe1, 0xae, 0xab, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0xe1, 0xb9, 0xb4, 0xd8, 0xb5, 0xd8, 0xb1, 0xea, 0xb0, 0x97, 0xea, 0xbc, 0xa2, 0x31, 0x37, 0xe6, 0x97, 0xa5, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe6, 0xb7, 0xb9, 0xec, 0xa9, 0x9a, 0xec, 0xba, 0x99, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x95, 0x94, 0xeb, 0x9b, 0x9f],
    "폯륂👨‍👩‍👧먆🌺🌸뫶뺎🇨🇦큈㒞🇨🇦䘵뜦罺🌺🌸틷عجم퀪뮠🇨🇦뇯딐Ỳ😂띻ご뒅쨞⚉𥘦👨‍👩‍👧썒🤔픴😂톎첟좖😂뛆퉣👨‍👩‍👧10月뜗輸🤔哶😂흾😂Ǖ뮚섨쫙⚉鬒쑬🌺🌸웳👨‍👩‍👧픠⚉뢺🇨🇦딗䕫😂뒮봟🤔횅👨‍👩‍👧앿👨‍👩‍👧앿👨‍👩‍👧옃🤔MD킮현퓬👨‍👩‍👧a֮ܲ̀̕b😂휈オ⚉퀣👨‍👩‍👧à֮𞀝̕b⚉먕눛👨‍👩‍👧뛈⚉랕⚉جي👨‍👩‍👧آ😂뫊👨‍👩‍👧겟🇨🇦껞⚉a゙᮫्ְbṴصر갗꼢17日🌺🌸淹쩚캙🌺🌸암뛟"
  ),
  (
    [0xeb, 0x80, 0x96, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa3, 0xa1, 0xcc, 0x95, 0x62, 0xea, 0xb7, 0xab, 0xe7, 0xab, 0x8b, 0xec, 0x88, 0xa4, 0x20, 0xcc, 0x94, 0xcc, 0x81, 0xf0, 0x9e, 0xb9, 0xa4, 0xe5, 0x88, 0xa9, 0xef, 0xba, 0x99, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xf0, 0x9d, 0x9e, 0x81, 0xe5, 0x85, 0xac, 0xeb, 0xa0, 0x92, 0xf0, 0x9f, 0x98, 0x82, 0xe7, 0xab, 0x8b, 0xe2, 0x92, 0x92, 0xe5, 0xba, 0xa6, 0xeb, 0x8f, 0xad, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x8a, 0xa9, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe5, 0x88, 0xa9, 0x31, 0x35, 0xf0, 0x9f, 0x98, 0x82, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xaf, 0xb7, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdb, 0xad, 0xd6, 0x9a, 0x62, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xbe, 0xa0, 0xe5, 0x88, 0xa9, 0xec, 0x81, 0xbe, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0x88, 0x98, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x8f, 0x8f, 0xe5, 0xba, 0xa6, 0xec, 0xb0, 0x9b, 0xef, 0xb0, 0xa0, 0xeb, 0xb3, 0xa1, 0xea, 0xb4, 0x96, 0xe5, 0x85, 0xac, 0xec, 0xa2, 0x92, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe2, 0xbc, 0xb1, 0xe3, 0x89, 0x95, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xec, 0x97, 0x87, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x85, 0xac, 0xeb, 0xb8, 0xbc, 0xf0, 0x9d, 0x98, 0x80, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x88, 0xa9, 0xeb, 0xb7, 0x8d, 0xe2, 0x9a, 0x89, 0xec, 0x87, 0xaf, 0xe2, 0x9a, 0x89, 0xe5, 0xba, 0xa6, 0xeb, 0x8c, 0xa1, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xb5, 0x87, 0xf0, 0x9f, 0xa4, 0x94, 0xe9, 0xad, 0xaf, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x61, 0xd6, 0xae, 0xe1, 0x9f, 0x9d, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe5, 0xba, 0xa6, 0xeb, 0xad, 0xab, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0xa5, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x88, 0xa9, 0xeb, 0xb8, 0xa9, 0xe7, 0xab, 0x8b, 0xc3, 0xa0, 0xd6, 0xae, 0xcc, 0x83, 0xcc, 0x95, 0x62, 0xeb, 0x93, 0x9a, 0xf0, 0x9d, 0x97, 0xbf, 0xed, 0x87, 0xbb, 0xeb, 0xa2, 0xb1, 0xe7, 0xab, 0x8b, 0xe3, 0x85, 0xb7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xf0, 0x9f, 0x84, 0xb5, 0xe5, 0x85, 0xac, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0xa5, 0x85, 0xcc, 0x95, 0x62, 0xe2, 0x9a, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe1, 0xb9, 0xaa, 0xe1, 0xb8, 0x92, 0xf0, 0x9f, 0x98, 0x82, 0xeb, 0x82, 0x8a, 0xec, 0x8d, 0x94, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xd8, 0xb9, 0xd8, 0xac, 0xf0, 0x9f, 0x98, 0x82, 0xf0, 0x9d, 0x92, 0xb5, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x85, 0xac, 0xeb, 0xae, 0xaf, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0xb7, 0xba, 0xe5, 0x85, 0xac, 0xe1, 0xb5, 0x96, 0xe5, 0xba, 0xa6, 0x61, 0xf0, 0x96, 0xab, 0xb2, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe6, 0x9c, 0x88, 0xe1, 0xb8, 0x8c, 0xcc, 0x87, 0xeb, 0xb5, 0xb0, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe5, 0x85, 0xac, 0xef, 0xbc, 0xa7, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x99, 0x85, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xa4, 0xb6, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xbb, 0xb2, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x88, 0xa9, 0xeb, 0x80, 0x9e, 0xf0, 0x9f, 0x98, 0x82, 0xec, 0xa9, 0x89, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x97, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe7, 0xab, 0x8b, 0xe7, 0xa6, 0xae, 0xe7, 0xab, 0x8b, 0xe2, 0x93, 0xa3, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe3, 0x8f, 0x80, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x8c, 0xb8, 0xe2, 0x9a, 0x89, 0xe7, 0xab, 0x8b, 0xed, 0x91, 0xab, 0xe5, 0x85, 0xac, 0xec, 0x97, 0x9b, 0xec, 0x8f, 0xa8, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x88, 0xa9, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x8a, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe5, 0x88, 0xa9, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdc, 0xbb, 0xd6, 0x9a, 0x62, 0xec, 0xaa, 0x87, 0xeb, 0xbe, 0xaa, 0xec, 0xab, 0xac, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe7, 0xab, 0x8b, 0xe1, 0xb8, 0x8b, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x85, 0xac, 0xf0, 0x9d, 0x9a, 0x8b, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6],
    "뀖à֮࣡̕b귫立숤 ̔́𞹤利ﺙ🇨🇦在线服务和个𝞁公렒😂立⒒度돭在线服务和个늩👨‍👩‍👧利15😂拥有믷a〪̖ۭ֚b在线服务和个뾠利쁾🤔수🌺🌸在线服务和个쏏度찛ﰠ볡괖公좒👨‍👩‍👧拥有⼱㉕🌺🌸엇🌺🌸公븼𝘀🇨🇦利뷍⚉쇯⚉度댡🌺🌸拥有굇🤔魯🌺🌸a֮៝̀̕b度뭫🤔在线服务和个à֮ᷥ̕b🌺🌸利븩立à֮̃̕b듚𝗿퇻뢱立ㅷ拥有🄵公à֮𞥅̕b⚉拥有ṪḒ😂낊썔拥有عج😂𝒵🇨🇦公뮯🤔췺公ᵖ度a𖫲̴़b🇨🇦月Ḍ̇뵰👨‍👩‍👧公Ｇ🌺🌸拥有왅🇨🇦在线服务和个뤶拥有껲🌺🌸利뀞😂쩉a〪̗̖֚b立禮立ⓣ在线服务和个㏀😂팸⚉立푫公엛쏨🤔利a֮𞀊̀̕b利a〪̖ܻ֚b쪇뾪쫬👨‍👩‍👧立ḋ🇨🇦公𝚋🇨🇦"
  ),
  (
    [0x61, 0xe2, 0x83, 0xaa, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe5, 0x85, 0x80, 0xe7, 0xab, 0x8b, 0xec, 0x99, 0xbd, 0x69, 0xe5, 0xba, 0xa6, 0xec, 0x86, 0xbe, 0xf0, 0x9f, 0x98, 0x82, 0xed, 0x89, 0x81, 0x61, 0xe2, 0x9a, 0x89, 0xe5, 0xba, 0xa6, 0xeb, 0xad, 0x91, 0xe7, 0xab, 0x8b, 0xe0, 0xb3, 0x8b, 0xe5, 0x88, 0xa9, 0xec, 0x84, 0xb6, 0xe1, 0xb8, 0xae, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0xba, 0xa6, 0xec, 0xb5, 0x95, 0xf0, 0x9f, 0xa4, 0x94, 0xec, 0xaf, 0x99, 0x65, 0xf0, 0x9f, 0xa4, 0x94, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xaa, 0xb4, 0xcc, 0x95, 0x62, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xf0, 0x9d, 0x95, 0xa8, 0xeb, 0x8f, 0x9c, 0x75, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xa7, 0xad, 0x65, 0xf0, 0x9d, 0x95, 0x95, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0x88, 0xa9, 0xed, 0x81, 0xbf, 0x6f, 0xf0, 0x9f, 0x98, 0x82, 0xc4, 0x88, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe1, 0xbf, 0xb8, 0x6f, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0xba, 0xa6, 0xe4, 0xbe, 0xbb, 0x61, 0xf0, 0x9f, 0xa4, 0x94, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xf0, 0x9f, 0x84, 0x9c, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe2, 0x9a, 0x89, 0xe1, 0xba, 0xa1, 0x6f, 0xe5, 0xba, 0xa6, 0xec, 0xbc, 0xad, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xb5, 0x89, 0x69, 0xe5, 0x85, 0xac, 0xeb, 0x82, 0xa4, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x90, 0xbd, 0x8f, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x6f, 0xe5, 0x85, 0xac, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xa6, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe2, 0x9a, 0x89, 0xe7, 0xab, 0x8b, 0xeb, 0x9e, 0xb9, 0xf0, 0x9f, 0xa4, 0x94, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9b, 0x81, 0x75, 0xe7, 0xab, 0x8b, 0xeb, 0x9e, 0xaa, 0x61, 0xe2, 0xbf, 0x93, 0x61, 0xec, 0x8f, 0x83, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe2, 0x80, 0x85, 0x75, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe0, 0xa3, 0xb6, 0xd6, 0x9a, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0xa4, 0x94, 0xea, 0xb8, 0xa9, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe7, 0xab, 0x8b, 0x61, 0xcc, 0xb4, 0xe1, 0xb0, 0xb7, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0xe2, 0x9a, 0x89, 0xe5, 0x88, 0xa9, 0xeb, 0x8b, 0xbf, 0x61, 0xe5, 0xba, 0xa6, 0xef, 0xb9, 0x89, 0x6f, 0xc3, 0xaf, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0x8d, 0xaa, 0x75, 0xe5, 0x88, 0xa9, 0xec, 0x86, 0xbf, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xf0, 0x9d, 0x9f, 0x90, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0xba, 0xa6, 0xed, 0x8b, 0x9b, 0x6f, 0xe7, 0xab, 0x8b, 0xec, 0x92, 0xa1, 0x69, 0xec, 0xa2, 0x85, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0x97, 0xb9, 0x65, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xf0, 0x9d, 0x98, 0xa6, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xed, 0x91, 0xab, 0xeb, 0x9d, 0xb3, 0x65, 0xf0, 0x9f, 0x98, 0x82, 0xe7, 0xbe, 0x85, 0xe2, 0x9a, 0x89, 0xe5, 0xba, 0xa6, 0xec, 0xb7, 0xa0, 0xf0, 0x9f, 0x98, 0x82, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe8, 0x8f, 0x8c, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x80, 0x8f, 0x6f, 0xe5, 0x88, 0xa9, 0xeb, 0xbc, 0x96, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9b, 0xa6, 0xec, 0x84, 0x85, 0x61, 0xf0, 0x9f, 0x98, 0x82, 0xd0, 0x81, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe1, 0xbc, 0x89, 0x75, 0xe5, 0xba, 0xa6, 0xed, 0x82, 0x8f, 0x65, 0xe5, 0x85, 0xac, 0xed, 0x96, 0x8a, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0x61, 0xe0, 0xbd, 0xb4, 0xcc, 0xa2, 0xcc, 0xa1, 0xe1, 0xb7, 0x8e, 0x62, 0x61, 0xe5, 0x88, 0xa9, 0xec, 0x9a, 0xa0, 0x65, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xec, 0xb9, 0xae, 0xeb, 0x83, 0xbc, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x99, 0x9f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xb7, 0x88, 0x69, 0xe7, 0xab, 0x8b, 0xec, 0x86, 0x90, 0x75, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xe2, 0x80, 0x84, 0x61, 0xe7, 0xab, 0x8b, 0xeb, 0xb3, 0x9c, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xeb, 0x84, 0x9d, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x8f, 0xa3, 0x6f, 0xf0, 0x9f, 0x91, 0xa8, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa9, 0xe2, 0x80, 0x8d, 0xf0, 0x9f, 0x91, 0xa7, 0xe5, 0xba, 0xa6, 0x61, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0xf0, 0x91, 0x88, 0xb6, 0xe3, 0x82, 0x99, 0x62, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xbe, 0x80, 0x61, 0xe2, 0x84, 0xa1, 0x61, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xed, 0x91, 0xb9, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe7, 0xab, 0x8b, 0xec, 0xbf, 0x83, 0x69, 0xe6, 0x9d, 0x96, 0x6f, 0xec, 0xb4, 0xb2, 0xef, 0xb4, 0xaf, 0xec, 0x8b, 0xac, 0xf0, 0x9f, 0x98, 0x82, 0x61, 0xe3, 0x80, 0xaa, 0xe1, 0x82, 0x8d, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0xe5, 0x85, 0xac, 0xe6, 0xa0, 0x9f, 0x61, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xea, 0xba, 0xbb, 0x6f, 0xf0, 0x9f, 0x98, 0x82, 0xe5, 0x85, 0xac, 0xeb, 0xb1, 0x9d, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0xba, 0xa6, 0xed, 0x95, 0x95, 0xe1, 0xb5, 0x9c, 0x6f, 0xe2, 0x93, 0x92, 0x6f, 0xf0, 0x9f, 0x8c, 0xba, 0xf0, 0x9f, 0x8c, 0xb8, 0xe5, 0x88, 0xa9, 0xe4, 0xbe, 0x8b, 0xed, 0x95, 0x85, 0xf0, 0x9f, 0x87, 0xa8, 0xf0, 0x9f, 0x87, 0xa6, 0xe5, 0xba, 0xa6, 0xeb, 0xa7, 0xba, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe2, 0x9a, 0x89],
    "a⃪̴़bu拥有兀立왽i度솾😂퉁a⚉度뭑立ೋ利섶Ḯabcdefg😂度쵕🤔쯙e🤔à֮᪴̕b🇨🇦在线服务和个𝕨돜u🇨🇦在线服务和个맭e𝕕hello🇨🇦利큿o😂Ĉabcdefg👨‍👩‍👧Ὸo🌺🌸度侻a🤔拥有🄜hello⚉ạo度켭u拥有굉i公낤o在线服务和个a〪𐽏̖֚bo公a֮ⷦ̀̕b⚉立랹🤔在线服务和个훁u立랪a⿓a쏃🇨🇦 ua〪̖ࣶ֚babcdefg🤔긩abcdefg🌺🌸立a̴᰷़゙b⚉利닿a度﹉oïe在线服务和个썪u利솿在线服务和个𝟐abcdefg😂度틛o立쒡i종hello엹e👨‍👩‍👧拥有𝘦🇨🇦푫띳e😂羅⚉度췠😂拥有菌拥有뀏o利뼖😂在线服务和个훦섅a😂Ё在线服务和个Ἁu度킏e公햊🌺🌸aུ̢̡᷎ba利욠e🇨🇦칮냼abcdefg在线服务和个홟拥有귈i立손u在线服务和个 a立볜abcdefg🌺🌸넝拥有폣o👨‍👩‍👧度a̴़𑈶゙b拥有뾀a℡a🌺🌸푹hello立쿃i杖o촲ﴯ심😂a〪ႍ̖֚b公栟a🌺🌸꺻o😂公뱝abcdefg度핕ᵜoⓒo🌺🌸利例핅🇨🇦度맺hello⚉"
  ),
  (
    [0xe1, 0xb8, 0xa0, 0xef, 0xb2, 0x83, 0xea, 0xbe, 0xba, 0xef, 0xbb, 0xae, 0xec, 0x99, 0x8d, 0xeb, 0x90, 0x8e, 0xef, 0xba, 0xa2, 0xeb, 0xb0, 0x86, 0xeb, 0xa2, 0x82, 0xeb, 0x81, 0xbc, 0xe1, 0xba, 0x97, 0xec, 0xbd, 0x9b, 0xef, 0xbc, 0xb0, 0xec, 0xb3, 0x8f, 0xec, 0xa5, 0x99, 0xea, 0xb0, 0xa8, 0xed, 0x89, 0xb1, 0xeb, 0x8a, 0xb3, 0xeb, 0x8b, 0x9d, 0xed, 0x83, 0xae, 0xeb, 0xba, 0x89, 0xeb, 0xaf, 0xb4, 0xcb, 0x99, 0xd7, 0x92, 0xd6, 0xbc, 0xeb, 0xad, 0x8d, 0xec, 0xab, 0x87, 0xeb, 0x8e, 0xad, 0xe3, 0x8b, 0xa1, 0xef, 0xb5, 0xa1, 0xec, 0x9b, 0x93, 0xeb, 0x93, 0xa8, 0xf0, 0x9d, 0x94, 0xa0, 0xec, 0x9b, 0x91, 0xed, 0x96, 0xb6, 0xd3, 0xb9, 0xeb, 0xb6, 0xb8, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0xeb, 0xab, 0xba, 0xec, 0x80, 0xb3, 0xea, 0xb8, 0xb8, 0xf0, 0x9d, 0x9a, 0xa9, 0x61, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xb0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0xa4, 0xcc, 0x95, 0x62, 0xeb, 0xbe, 0xa1, 0xe1, 0xba, 0x90, 0xeb, 0x9f, 0xbb, 0xef, 0xbb, 0x80, 0xed, 0x86, 0xb9, 0xec, 0x9a, 0xa6, 0xeb, 0xbf, 0xa5, 0xed, 0x8e, 0x83, 0xec, 0x86, 0xbc, 0xd3, 0xa2, 0xe1, 0xbd, 0x85, 0xec, 0xbe, 0xbf, 0xeb, 0xbb, 0xa6, 0xec, 0x99, 0xb2, 0xe3, 0x8f, 0x80, 0xec, 0x94, 0x93, 0xea, 0xb6, 0xab, 0xed, 0x86, 0xb6, 0xed, 0x85, 0xa5, 0xeb, 0x95, 0xbe, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcc, 0x9c, 0xd6, 0x9a, 0x62, 0x61, 0xd9, 0x8b, 0xd9, 0x8c, 0xd9, 0x8c, 0xd9, 0x8d, 0x62, 0xed, 0x9c, 0x87, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x91, 0x8d, 0xa7, 0xcc, 0x95, 0x62, 0xe2, 0xbc, 0x83, 0xef, 0xbe, 0xb8, 0xf0, 0x9e, 0xb8, 0x82, 0xeb, 0xb7, 0x94, 0xec, 0xbb, 0xb8, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x9b, 0xcc, 0x95, 0x62, 0xf0, 0x9d, 0x97, 0x81, 0xf0, 0x9e, 0xba, 0x97, 0xec, 0x9c, 0x8a, 0xec, 0x81, 0x9c, 0x61, 0xd6, 0xae, 0xe1, 0xb3, 0x90, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xd7, 0x91, 0xd6, 0xbf, 0xed, 0x8c, 0xb2, 0xe1, 0xb9, 0xbc, 0xd1, 0x90, 0xed, 0x81, 0x8a, 0xed, 0x96, 0xa0, 0x61, 0xe3, 0x80, 0xaa, 0xe1, 0xaa, 0xbd, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xdc, 0xb7, 0xd6, 0x9a, 0x62, 0x61, 0xd6, 0xae, 0xe0, 0xa0, 0x9f, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x9d, 0x8f, 0xed, 0x97, 0xb9, 0xeb, 0x87, 0x9b, 0xef, 0xb5, 0x91],
    "Ḡﲃ꾺ﻮ왍됎ﺢ밆뢂끼ẗ콛Ｐ쳏쥙갨퉱늳닝탮뺉믴˙גּ뭍쫇뎭㋡ﵡ웓듨𝔠웑햶ӹ붸̴ᆵ뫺쀳길𝚩a֮𑍰̀̕bà֮𞀤̕b뾡Ẑ럻ﻀ톹욦뿥펃솼Ӣὅ쾿뻦왲㏀씓궫톶텥땾a〪̖̜֚baًٌٌٍb휇à֮𑍧̕b⼃ﾸ𞸂뷔컸à֮𞀛̕b𝗁𞺗윊쁜a֮᳐̀̕bבֿ팲Ṽѐ큊햠a〪᪽̖֚ba〪̖ܷ֚ba֮ࠟ̀̕b흏헹뇛ﵑ"
  ),
  (
    [0xea, 0xbf, 0x8a, 0xed, 0x96, 0x94, 0x28, 0x6d, 0x29, 0xe7, 0xab, 0x8b, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb7, 0x84, 0xcc, 0x95, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0xba, 0xa6, 0xeb, 0x95, 0x80, 0xeb, 0x83, 0x9f, 0x69, 0xeb, 0x89, 0x9e, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x8e, 0x80, 0xe5, 0xba, 0xa6, 0xec, 0x9e, 0xa4, 0xe5, 0x88, 0xa9, 0xe1, 0xac, 0x89, 0xcc, 0xb4, 0xe1, 0xac, 0xb5, 0xeb, 0x96, 0x91, 0xe5, 0x88, 0xa9, 0xeb, 0xb5, 0xb5, 0x75, 0xed, 0x95, 0xb5, 0xe5, 0xba, 0xa6, 0xed, 0x97, 0x9d, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xea, 0xb7, 0x91, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0x85, 0xac, 0xc4, 0x8d, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0xba, 0xa6, 0xec, 0x84, 0x94, 0xe5, 0xba, 0xa6, 0xeb, 0xbf, 0x96, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe7, 0xab, 0x8b, 0xe7, 0x96, 0x8b, 0xed, 0x89, 0x84, 0x65, 0xed, 0x95, 0xb2, 0x65, 0x4e, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xbb, 0x9b, 0x61, 0xd6, 0xae, 0xea, 0xaa, 0xb2, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x69, 0xec, 0x8a, 0x98, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x93, 0xba, 0x65, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe1, 0x87, 0x8c, 0xe5, 0x85, 0xac, 0xd8, 0xb4, 0xd8, 0xb1, 0xe5, 0x88, 0xa9, 0xeb, 0xb8, 0x90, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xd6, 0xae, 0xd6, 0xa0, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xe5, 0xba, 0xa6, 0xed, 0x9a, 0xab, 0xe5, 0x88, 0xa9, 0xeb, 0x8f, 0xa7, 0xeb, 0x8a, 0xa9, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xaa, 0x85, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x8f, 0x91, 0x61, 0xed, 0x92, 0x9f, 0xe5, 0x85, 0xac, 0xec, 0x90, 0xa2, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x82, 0xb2, 0xe7, 0xab, 0x8b, 0xec, 0x9e, 0x9d, 0x65, 0xe1, 0x84, 0x83, 0x61, 0x61, 0xd6, 0xae, 0xe2, 0x83, 0x91, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x69, 0x6b, 0x50, 0x61, 0x61, 0xeb, 0xac, 0x94, 0x6f, 0xe1, 0xbe, 0xb1, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe7, 0x8b, 0x80, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xed, 0x8c, 0x8c, 0x75, 0xe5, 0xba, 0xa6, 0x61, 0xe3, 0x80, 0xaa, 0xf0, 0x9e, 0xa3, 0x95, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe7, 0xab, 0x8b, 0xec, 0x9a, 0xb9, 0x75, 0xe7, 0xab, 0x8b, 0xec, 0x80, 0x86, 0x69, 0xeb, 0xa0, 0x96, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb7, 0xb2, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0x9f, 0x93, 0x34, 0x30, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xed, 0x8b, 0x8c, 0x61, 0xd6, 0xae, 0xdd, 0x81, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xea, 0xbd, 0x9b, 0x61, 0xe5, 0xba, 0xa6, 0xeb, 0x85, 0x8e, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x20, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0x61, 0xe3, 0x82, 0x99, 0xe0, 0xa7, 0x8d, 0xe0, 0xa5, 0x8d, 0xd6, 0xb0, 0x62, 0x65, 0xe7, 0xab, 0x8b, 0xec, 0x8a, 0x8f, 0x61, 0xe7, 0xab, 0x8b, 0xed, 0x97, 0xaa, 0xed, 0x9b, 0x8d, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0xba, 0xa6, 0xce, 0xbe, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x90, 0xbc, 0x6f, 0xed, 0x9c, 0x9b, 0x6f, 0xe5, 0x88, 0xa9, 0xeb, 0xb6, 0x86, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x91, 0x86, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe7, 0xab, 0x8b, 0xea, 0xba, 0xb4, 0x6f, 0xe5, 0x88, 0xa9, 0xec, 0xb1, 0xb8, 0xed, 0x89, 0xb7, 0x65, 0xec, 0x81, 0xb1, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe5, 0x87, 0x89, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0x92, 0xa9, 0x69, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x28, 0x7a, 0x29, 0xed, 0x88, 0xa0, 0x69, 0xec, 0x93, 0x94, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0x61, 0xe9, 0xa6, 0x96, 0xec, 0xba, 0xa5, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x9d, 0x93, 0xe5, 0x88, 0xa9, 0xe5, 0xa5, 0x91, 0x61, 0xe5, 0x85, 0xac, 0xed, 0x81, 0xb9, 0xe7, 0xab, 0x8b, 0xea, 0xbe, 0xa9, 0x65, 0x61, 0xd6, 0xae, 0xe0, 0xa3, 0xab, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x6f, 0xec, 0x92, 0x9f, 0xe5, 0x85, 0xac, 0xec, 0xa1, 0x9d, 0x69, 0xec, 0xa2, 0xb3, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe1, 0xb9, 0xb5, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x9a, 0xa5, 0x6f, 0xe5, 0xba, 0xa6, 0xeb, 0xa3, 0xb2, 0x69, 0xeb, 0xaf, 0x93, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89],
    "꿊햔(m)立à֮᷄̕bhello度땀냟i뉞abcdefg뎀度잤利ᬉ̴ᬵ떑利뵵u핵度헝e在线服务和个귑abcdefg公čabcdefg度섔度뿖hello立疋퉄e핲eNe在线服务和个컛a֮ꪲ̀̕bi슘拥有듺e拥有ᇌ公شر利븐在线服务和个a֮֠̀̕b度횫利돧늩hello拥有명在线服务和个폑a풟公쐢abcdefg낲立잝eᄃaa֮⃑̀̕bikPaa묔oᾱhello狀abcdefg파u度a〪𞣕̖֚bhello立욹u立쀆i렖o拥有뷲u拥有쟓40hello틌a֮݁̀̕b꽛a度녎hello hello利a゙্्ְbe立슏a立헪훍hello度ξu拥有퐼o휛o利붆拥有푆hello立꺴o利챸퉷e쁱拥有凉hello쒩i拥有(z)툠i쓔̴ᆵa首캥拥有띓利契a公큹立꾩ea֮࣫̀̕bo쒟公졝i좳hello拥有ṵabcdefg뚥o度룲i믓拥有"
  ),
  (
    [0x61, 0xe3, 0x80, 0xaa, 0xe1, 0xb7, 0x8f, 0xcc, 0x96, 0xd6, 0x9a, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x61, 0xd6, 0xae, 0xe2, 0xb7, 0xb4, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xeb, 0x89, 0xba, 0xeb, 0xa1, 0xaf, 0x61, 0xe5, 0x85, 0xac, 0x57, 0x6f, 0xeb, 0x93, 0xb1, 0xe5, 0x88, 0xa9, 0xed, 0x8a, 0xbf, 0xec, 0x8f, 0xb2, 0xe8, 0x8c, 0xb6, 0x65, 0xdb, 0x92, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x87, 0x99, 0x69, 0xe5, 0x85, 0xac, 0xeb, 0x86, 0x92, 0x75, 0xe5, 0x88, 0xa9, 0xed, 0x88, 0x8f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x88, 0x8d, 0x65, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xb1, 0xb8, 0x65, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xa6, 0x95, 0xeb, 0x9a, 0x9c, 0xcc, 0xb4, 0xe1, 0x86, 0xb5, 0x61, 0xeb, 0xa5, 0xa9, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xe2, 0x83, 0x93, 0xcc, 0xb4, 0xe0, 0xa4, 0xbc, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xd6, 0xae, 0xe1, 0xb7, 0x80, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xec, 0xbd, 0x84, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x61, 0xcc, 0xb4, 0xf0, 0x9b, 0xb2, 0x9e, 0xe0, 0xa4, 0xbc, 0x62, 0xe5, 0x89, 0xb7, 0xec, 0xac, 0x95, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x88, 0x91, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa5, 0x91, 0xcc, 0x95, 0x62, 0x75, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa3, 0xbc, 0xcc, 0x95, 0x62, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe0, 0xbc, 0xb7, 0xd6, 0x9a, 0x62, 0xe7, 0xab, 0x8b, 0xec, 0x82, 0xb8, 0x65, 0xec, 0xb8, 0xa2, 0xe5, 0x88, 0xa9, 0xec, 0xa6, 0xa5, 0xe5, 0x85, 0xac, 0xea, 0xbc, 0xa2, 0x75, 0xeb, 0xae, 0xbf, 0xe5, 0x88, 0xa9, 0xed, 0x81, 0x97, 0xf0, 0x9d, 0x86, 0xba, 0xf0, 0x9d, 0x85, 0xa5, 0xf0, 0x9d, 0x85, 0xae, 0x75, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x8e, 0xae, 0x69, 0xe5, 0x88, 0xa9, 0xed, 0x8c, 0x83, 0x65, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x98, 0xcc, 0x95, 0x62, 0x69, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xed, 0x84, 0x89, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0xba, 0xa3, 0xe3, 0x83, 0x9a, 0xe3, 0x83, 0xbc, 0xe3, 0x82, 0xb8, 0x61, 0xeb, 0x92, 0xb4, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x82, 0x96, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0xab, 0x88, 0x6f, 0xe5, 0xba, 0xa6, 0xeb, 0x97, 0x99, 0x75, 0xec, 0x85, 0x89, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x98, 0xbd, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0xba, 0xa6, 0xec, 0xb0, 0xa7, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe7, 0xab, 0x8b, 0xea, 0xb9, 0xa1, 0x75, 0xe7, 0xab, 0x8b, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xad, 0xb3, 0xcc, 0x95, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0xba, 0xa6, 0xc4, 0xbd, 0x65, 0xe5, 0x88, 0xa9, 0xea, 0xbe, 0xb9, 0xed, 0x96, 0x92, 0xe5, 0x85, 0xac, 0xec, 0xb5, 0x96, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x85, 0xac, 0xf0, 0xa3, 0xbb, 0x91, 0x69, 0xe5, 0x85, 0xac, 0xeb, 0x94, 0x9b, 0xeb, 0x89, 0xa7, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0xb1, 0xa6, 0x61, 0xe5, 0x88, 0xa9, 0xea, 0xb0, 0xa3, 0xec, 0x97, 0xac, 0xe5, 0xb6, 0xba, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xeb, 0x98, 0x81, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xeb, 0x92, 0x83, 0x75, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xa0, 0x8d, 0x69, 0xed, 0x8e, 0x80, 0xeb, 0x84, 0xb1, 0x6f, 0x28, 0xe1, 0x84, 0x8e, 0x29, 0xe5, 0x88, 0xa9, 0xed, 0x8d, 0xa9, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x95, 0x80, 0x65, 0xe1, 0xbd, 0x80, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xec, 0xb0, 0xa5, 0x69, 0xeb, 0xb9, 0x91, 0x61, 0xed, 0x91, 0x93, 0x75, 0xec, 0x89, 0xad, 0x69, 0xe8, 0x8a, 0x8b, 0xe5, 0x88, 0xa9, 0xe5, 0x8c, 0x97, 0xe7, 0xab, 0x8b, 0xeb, 0xa7, 0x98, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0x88, 0xa9, 0xe7, 0x8e, 0x87, 0xed, 0x93, 0xa4, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xec, 0x97, 0xad, 0xe5, 0x88, 0xa9, 0x61, 0xcc, 0xb4, 0xf0, 0x91, 0xb5, 0x82, 0xe0, 0xa4, 0xbc, 0xe3, 0x82, 0x99, 0x62, 0x69, 0xed, 0x8e, 0x88, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x94, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x85, 0xac, 0x33, 0x38, 0xe7, 0xab, 0x8b, 0xeb, 0x9d, 0x8c, 0x6f, 0xc3, 0xbc, 0xeb, 0x85, 0x82, 0x61, 0xd6, 0xae, 0xf0, 0x90, 0xb4, 0xa5, 0xcc, 0x80, 0xcc, 0x95, 0x62, 0xed, 0x94, 0x97, 0xec, 0x82, 0xa4, 0x61],
    "a〪᷏̖֚bhelloa֮ⷴ̀̕b뉺롯a公Wo등利튿쏲茶eےabcdefg뇙i公높u利툏拥有눍e拥有걸e拥有즕뚜̴ᆵa륩拥有a⃓̴़bhello拥有a֮᷀̀̕b콄helloa̴𛲞़b剷쬕abcdefg눑在线服务和个à֮॑̕buà֮ࣼ̕ba〪̖༷֚b立삸e츢利즥公꼢u뮿利큗𝆺𝅥𝅮u拥有뎮i利팃eà֮𞀘̕bi拥有턉o在线服务和个뺣ページa뒴在线服务和个낖hello뫈o度뗙u셉拥有똽hello度찧abcdefg立깡u立à֮᭳̕bhello度Ľe利꾹햒公쵖hello公𣻑i公딛뉧拥有뱦a利갣여嶺abcdefg똁hello뒃u在线服务和个젍i펀넱o(ᄎ)利퍩拥有땀eὀhello利찥i빑a푓u쉭i芋利北立맘abcdefg利率퓤hello역利a̴𑵂़゙bi펈拥有a֮𞀔̀̕bhello公38立띌oü녂a֮𐴥̀̕b픗삤a"
  ),
  (
    [0xc2, 0xbe, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xec, 0xb0, 0x8e, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xf0, 0x9d, 0x95, 0x98, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xeb, 0x8e, 0x95, 0xd7, 0x91, 0xd6, 0xbc, 0xeb, 0x83, 0x82, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0xba, 0xa6, 0xed, 0x9a, 0x90, 0xec, 0x90, 0xbd, 0x39, 0xe6, 0x9c, 0x88, 0xe1, 0xbe, 0x9c, 0x61, 0xe5, 0x85, 0xac, 0xeb, 0xbc, 0x83, 0xe1, 0xb9, 0x9c, 0xe5, 0x85, 0xac, 0xef, 0xb2, 0xa7, 0x69, 0x54, 0x4d, 0x61, 0xe5, 0x85, 0xac, 0xec, 0xa5, 0xb4, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x85, 0xac, 0xef, 0xba, 0xae, 0xef, 0xbe, 0xa2, 0x61, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xf0, 0x9d, 0x86, 0x8b, 0xd6, 0x9a, 0x62, 0xe5, 0x85, 0xac, 0xef, 0xb1, 0x88, 0x65, 0xe5, 0x85, 0xac, 0xeb, 0xa2, 0xbf, 0xeb, 0xb0, 0xaa, 0xe5, 0x88, 0xa9, 0xf0, 0x9d, 0x98, 0x83, 0x32, 0x30, 0x2e, 0x6f, 0xe5, 0x85, 0xac, 0xed, 0x85, 0x82, 0xef, 0xb2, 0xa3, 0x6f, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x86, 0xad, 0xe5, 0xba, 0xa6, 0xeb, 0x8b, 0xb4, 0x75, 0xe3, 0x8e, 0xa9, 0xed, 0x91, 0xba, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x85, 0xac, 0xd7, 0x9e, 0xd6, 0xbc, 0xeb, 0x85, 0xa6, 0xe7, 0xab, 0x8b, 0xe5, 0xb1, 0xa5, 0xe2, 0xbd, 0xb4, 0x61, 0xe5, 0x85, 0xac, 0xeb, 0xa5, 0xbf, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xe8, 0x99, 0x9c, 0x6f, 0xed, 0x91, 0x80, 0xed, 0x87, 0xbd, 0x61, 0xe5, 0x88, 0xa9, 0xc3, 0xa0, 0xd6, 0xae, 0xe0, 0xa0, 0x9f, 0xcc, 0x95, 0x62, 0x75, 0xec, 0xa1, 0x98, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xec, 0xba, 0x93, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xe2, 0xbd, 0x8b, 0x65, 0xea, 0xbe, 0xa6, 0x61, 0xe5, 0x88, 0xa9, 0xec, 0xad, 0x84, 0xc3, 0xa7, 0xe5, 0x88, 0xa9, 0xc3, 0xa0, 0xd6, 0xae, 0xf0, 0x9e, 0x80, 0x97, 0xcc, 0x95, 0x62, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xea, 0xb3, 0xae, 0x75, 0xf0, 0x9d, 0x9b, 0x95, 0x65, 0xec, 0x9a, 0xbe, 0xeb, 0xa5, 0xbf, 0x69, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xcd, 0x9a, 0xd6, 0x9a, 0x62, 0xe5, 0x85, 0xac, 0xeb, 0xb1, 0xb2, 0x75, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xec, 0xb6, 0x9c, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe1, 0xb5, 0x8a, 0xe5, 0x85, 0xac, 0xed, 0x8d, 0xb1, 0x69, 0xea, 0xb6, 0xa8, 0xeb, 0xb7, 0x9f, 0xe5, 0x85, 0xac, 0xf0, 0x9d, 0x9f, 0x9d, 0xe5, 0x85, 0xac, 0xeb, 0xb4, 0x9e, 0x69, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x84, 0x93, 0x75, 0xef, 0xbd, 0xaa, 0x61, 0xe5, 0x88, 0xa9, 0x20, 0xcc, 0x93, 0xeb, 0xae, 0xaf, 0xe5, 0x88, 0xa9, 0xe1, 0xb9, 0xb0, 0x75, 0xe5, 0x88, 0xa9, 0xc3, 0xa0, 0xd6, 0xae, 0xe1, 0xb3, 0xb9, 0xcc, 0x95, 0x62, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0xba, 0xa6, 0xeb, 0xae, 0xb2, 0xeb, 0x9a, 0xbd, 0x75, 0xe5, 0x88, 0xa9, 0xec, 0xb9, 0x9f, 0x65, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x8d, 0x82, 0xe0, 0xbe, 0xb2, 0xe0, 0xbd, 0xb1, 0xe0, 0xbe, 0x80, 0x69, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xea, 0xb1, 0xa3, 0x65, 0xe7, 0xab, 0x8b, 0xed, 0x86, 0xa2, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0x61, 0xe0, 0xb8, 0xb8, 0xe0, 0xb9, 0x88, 0xe0, 0xb9, 0x88, 0xe0, 0xba, 0xb8, 0x62, 0xe7, 0xab, 0x8b, 0xec, 0x8c, 0xa8, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe3, 0x82, 0x99, 0x65, 0xe5, 0xba, 0xa6, 0xeb, 0xb6, 0xb3, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0xec, 0xaf, 0xbb, 0x61, 0xe7, 0xab, 0x8b, 0xed, 0x99, 0x98, 0x69, 0xec, 0xbf, 0xac, 0xe7, 0xab, 0x8b, 0xeb, 0xaf, 0xab, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe7, 0xb3, 0xa3, 0x61, 0xec, 0xa7, 0xac, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe6, 0x8b, 0xa5, 0xe6, 0x9c, 0x89, 0x61, 0xe3, 0x80, 0xaa, 0xcc, 0x96, 0xe0, 0xa1, 0x9a, 0xd6, 0x9a, 0x62, 0x65, 0xe5, 0xba, 0xa6, 0xeb, 0xae, 0x90, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xeb, 0x8b, 0xaa, 0xe5, 0x9c, 0xa8, 0xe7, 0xba, 0xbf, 0xe6, 0x9c, 0x8d, 0xe5, 0x8a, 0xa1, 0xe5, 0x92, 0x8c, 0xe4, 0xb8, 0xaa, 0xed, 0x9c, 0xb6, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe5, 0x88, 0xa9, 0xeb, 0x9d, 0xab, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0xe1, 0xb9, 0x9d, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0xe5, 0x88, 0xa9, 0xea, 0xbd, 0xad, 0x61, 0xe3, 0x8e, 0xaf, 0x69, 0xeb, 0xb7, 0x96],
    "¾abcdefg찎hello𝕘拥有뎕בּ냂hello度횐쐽9月ᾜa公뼃Ṝ公ﲧiTMa公쥴hello公ﺮﾢaa〪̖𝆋֚b公ﱈe公뢿밪利𝘃20.o公텂ﲣo在线服务和个놭度담u㎩푺hello公מּ녦立履⽴a公륿hello利虜o푀퇽a利à֮ࠟ̕bu졘hello利캓hello拥有⽋e꾦a利쭄ç利à֮𞀗̕bhello곮u𝛕e욾륿ia〪̖͚֚b公뱲u在线服务和个출helloᵊ公퍱i궨뷟公𝟝公봞i在线服务和个넓uｪa利 ̓뮯利Ṱu利à֮᳹̕babcdefg度뮲뚽u利칟e在线服务和个퍂ྲཱྀi拥有걣e立톢在线服务和个aุ่่ຸb立쌨abcdefg゙e度붳拥有쯻a立환i쿬立믫hello糣a짬abcdefg拥有a〪̖࡚֚be度뮐在线服务和个닪在线服务和个휶hello利띫helloṝabcdefg利꽭a㎯i뷖"
  ),
]

let codeUnitNormalizationTests: [([UInt8], String)] =
    codeUnitNormalizationTestsA
  + codeUnitNormalizationTestsB
  + codeUnitNormalizationTestsC
  + codeUnitNormalizationTestsD
  + codeUnitNormalizationTestsE
  + codeUnitNormalizationTestsF
  + codeUnitNormalizationTestsG


for (i, test) in codeUnitNormalizationTests.enumerated() {
  tests.test("CodeUnitNormalizationTest#\(i)")
.skip(.custom({
      if #available(macOS 10.14, iOS 12, watchOS 5, tvOS 12, *) { return false }
      return true
    }, reason: "NormalizationTest.txt requires Unicode 11"))
.code {
    let codeUnits = test.1._nfcCodeUnits
    expectEqual(test.0, codeUnits)
  }
#if _runtime(_ObjC)
  tests.test("CodeUnitNormalizationTest#\(i)/Opaque")
.skip(.custom({
      if #available(macOS 10.14, iOS 12, watchOS 5, tvOS 12, *) { return false }
      return true
    }, reason: "NormalizationTest.txt requires Unicode 11"))
.code {
    let opaqueString = NSSlowString(string: test.1) as String
    let codeUnits = opaqueString._nfcCodeUnits
    expectEqual(test.0, codeUnits)
  }
#endif
}

runAllTests()
