<?php

namespace App\Modules\PrizePool\Manage;

use App\Helpers\CmsHtmlHelper;
use App\Http\Controllers\Support\BaseController;
use App\Modules\PrizePool\ActivityType;
use App\Modules\PrizePool\CreditType;
use App\Modules\PrizePool\DrawnPhysicalStatus;
use App\Modules\PrizePool\PrizeType;
use App\Modules\PrizePool\RedbagType;
use Edwin404\Base\Support\InputPackage;
use Edwin404\Base\Support\ModelHelper;
use Edwin404\Base\Support\RequestHelper;
use Edwin404\Base\Support\Response;
use Edwin404\Base\Support\TypeHelper;
use Edwin404\Member\Services\MemberReportService;
use Edwin404\Report\ReportService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Input;

class HomeController extends BaseController
{
    public function index(ReportService $reportService)
    {
        $drawLatestReport = $reportService->countDaily(
            'module_prize_pool_drawn',
            ['memberUserId' => $this->memberUserId(),],
            date('Y-m-d', strtotime('-6 day')),
            date('Y-m-d')
        );
        return $this->_view('module.prizePool.index', [
            'drawLatestReport' => $drawLatestReport,
        ]);
    }

    public function prize()
    {
        $prizes = ModelHelper::find('module_prize_pool', ['memberUserId' => $this->memberUserId()], ['id', 'desc']);
        return $this->_view('module.prizePool.prize', [
            'prizes' => $prizes,
        ]);
    }

    public function prizeEdit($id = 0)
    {
        if ($id) {
            $prize = ModelHelper::load('module_prize_pool', ['id' => $id, 'memberUserId' => $this->memberUserId()]);
            if (empty($prize)) {
                return Response::send(-1, '记录不存在');
            }
        } else {
            $prize = null;
        }

        if (RequestHelper::isPost()) {

            $input = InputPackage::buildFromInput();

            $update = [];
            $update['enable'] = $input->getBoolean('enable');
            $update['name'] = $input->getTrimString('name');
            $update['cover'] = $input->getImagePath('cover');
            $update['type'] = $input->getType('type', PrizeType::class);

            if (empty($update['name'])) {
                return Response::send(-1, '奖品名称为空');
            }

            switch ($update['type']) {
                case PrizeType::REDBAG:
                    $update['redbagType'] = $input->getType('redbagType', RedbagType::class);
                    switch ($update['redbagType']) {
                        case RedbagType::FIXED:
                            $update['redbagFixed'] = $input->getDecimal('redbagFixed');
                            if ($update['redbagFixed'] < 0.01 || $update['redbagFixed'] > 200.00) {
                                return Response::send(-1, '红包金额范围0.01-200');
                            }
                            break;
                        case RedbagType::RANDOM:
                            $update['redbagMin'] = $input->getDecimal('redbagMin');
                            $update['redbagMax'] = $input->getDecimal('redbagMax');
                            if ($update['redbagMin'] < 0.01 || $update['redbagMin'] > 200.00) {
                                return Response::send(-1, '红包金额范围0.01-200');
                            }
                            if ($update['redbagMax'] < 0.01 || $update['redbagMax'] > 200.00) {
                                return Response::send(-1, '红包金额范围0.01-200');
                            }
                            if ($update['redbagMin'] > $update['redbagMax']) {
                                return Response::send(-1, '红包范围不正确');
                            }
                            break;
                        default:
                            return Response::send(-1, '红包类型错误');
                    }
                    break;
                case PrizeType::CREDIT:
                    $update['creditType'] = $input->getType('creditType', CreditType::class);
                    switch ($update['creditType']) {
                        case CreditType::FIXED:
                            $update['creditFixed'] = $input->getInteger('creditFixed');
                            if ($update['creditFixed'] < 1) {
                                return Response::send(-1, '积分必须大于0');
                            }
                            break;
                        case CreditType::RANDOM:
                            $update['creditMin'] = $input->getInteger('creditMin');
                            $update['creditMax'] = $input->getInteger('creditMax');
                            if ($update['creditMin'] < 1) {
                                return Response::send(-1, '积分必须大于0');
                            }
                            if ($update['creditMin'] > $update['creditMax']) {
                                return Response::send(-1, '积分范围不正确');
                            }
                            break;
                        default:
                            return Response::send(-1, '积分类型错误');
                    }
                    break;
                case PrizeType::CODE:
                    break;
                case PrizeType::PHYSICAL:
                    break;
                default:
                    return Response::send(-1, '奖品类型错误');
            }

            if (empty($prize)) {
                $update['memberUserId'] = $this->memberUserId();
                $prize = ModelHelper::add('module_prize_pool', $update);
            } else {
                ModelHelper::update('module_prize_pool', ['id' => $prize['id']], $update);
            }

            return Response::send(0, '保存成功', null, '/module/manage/prize_pool_prize');
        }

        return $this->_view('module.prizePool.edit', [
            'prize' => $prize,
        ]);
    }

    public function prizeDelete($id = 0)
    {
        $prize = ModelHelper::load('module_prize_pool', ['id' => $id, 'memberUserId' => $this->memberUserId()]);
        if (empty($prize)) {
            return Response::send(-1, '记录不存在');
        }
        ModelHelper::delete('module_prize_pool', ['id' => $prize['id']]);
        return Response::send(0, '删除成功', null, '[reload]');
    }

    public function prizeView($id = 0)
    {
        $prize = ModelHelper::load('module_prize_pool', ['id' => $id, 'memberUserId' => $this->memberUserId()]);
        if (empty($prize)) {
            return Response::send(-1, '记录不存在');
        }
        return $this->_view('module.prizePool.prizeView', [
            'prize' => $prize,
        ]);
    }

    public function prizeDialog()
    {
        $prizes = ModelHelper::find('module_prize_pool', ['memberUserId' => $this->memberUserId()], ['id', 'desc']);
        return $this->_view('module.prizePool.prizeDialog', [
            'prizes' => $prizes,
        ]);
    }

    public function prizeCodeImport($id = 0)
    {
        $prize = ModelHelper::load('module_prize_pool', ['id' => $id, 'memberUserId' => $this->memberUserId()]);
        if (empty($prize)) {
            return Response::send(-1, '记录不存在');
        }
        if ($prize['type'] != PrizeType::CODE) {
            return Response::send(-1, '奖品类型不正确');
        }

        if (RequestHelper::isPost()) {
            $codesInput = trim(Input::get('codes', ''));
            if (empty($codesInput)) {
                return Response::send(-1, '兑换码不能为空');
            }
            $lines = explode("\n", $codesInput);
            $codes = [];
            foreach ($lines as $lineNo => $line) {
                $line = trim($line);
                if (empty($line)) {
                    continue;
                }
                $line = str_replace('，', ',', $line);
                $pcs = explode(",", $line);
                $item = [];
                switch (count($pcs)) {
                    case 1:
                        $item['no'] = trim($pcs[0]);
                        $item['secret'] = '';
                        $item['expire'] = '';
                        break;
                    case 2:
                        $item['no'] = trim($pcs[0]);
                        $item['secret'] = trim($pcs[1]);
                        $item['expire'] = '';
                        break;
                    case 3:
                        $item['no'] = trim($pcs[0]);
                        $item['secret'] = trim($pcs[1]);
                        $item['expire'] = trim($pcs[2]);
                        break;
                }
                if (empty($item)) {
                    return Response::send(-1, '第' . ($lineNo + 1) . '行数据格式不正确');
                }
                $item['prizeId'] = $prize['id'];
                $codes[] = $item;
            }
            if (count($codes) > 1000) {
                return Response::send(-1, '一次最多导入1000个');
            }
            DB::table('module_prize_pool_code')->insert($codes);
            return Response::send(0, '成功导入' . count($codes) . '个兑换码', null, '[js]parent.api.dialog.dialogCloseAll();');
        }

        return $this->_view('module.prizePool.prizeCodeImport', [
            'prize' => $prize,
        ]);
    }

    public function prizeCode($id = 0)
    {
        $prize = ModelHelper::load('module_prize_pool', ['id' => $id, 'memberUserId' => $this->memberUserId()]);
        if (empty($prize)) {
            return Response::send(-1, '记录不存在');
        }
        if ($prize['type'] != PrizeType::CODE) {
            return Response::send(-1, '奖品类型不正确');
        }

        if (RequestHelper::isPost()) {

            $page = Input::get('page');
            $pageSize = 10;
            $option = [];
            $option['where']['prizeId'] = $prize['id'];
            $option['order'] = ['id', 'desc'];

            $head = [];

            $head[] = ['field' => 'no', 'title' => '卡号',];
            $head[] = ['field' => 'secret', 'title' => '卡密',];
            $head[] = ['field' => 'expire', 'title' => '过期时间',];

            $list = [];
            $paginateData = ModelHelper::modelPaginate('module_prize_pool_code', $page, $pageSize, $option);
            foreach ($paginateData['records'] as &$record) {
                $item = [];
                $item['_id'] = $record['id'];
                $item['no'] = $record['no'];
                $item['secret'] = $record['secret'];
                $item['expire'] = $record['expire'];
                $list[] = $item;
            }

            $data = [];
            $data['head'] = $head;
            $data['list'] = $list;
            $data['total'] = $paginateData['total'];
            $data['pageSize'] = $pageSize;
            $data['page'] = $page;

            return Response::generate(0, null, $data);
        }

        return $this->_view('module.prizePool.prizeCode', [
            'prize' => $prize,
        ]);
    }

    public function drawnListSearchFilter($searchParam = [])
    {
        $search = [];
        if (empty($searchParam) || !is_array($searchParam)) {
            return $search;
        }

        foreach ($searchParam as $searchItem) {
            foreach ($searchItem as $field => $searchInfo) {
                switch ($field) {
                    case 'activityType':
                        if (isset($searchInfo['equal'])) {
                            $search[] = ['activityType' => ['equal' => $searchInfo['equal']]];
                        }
                        break;
                    case 'activityId':
                        if (isset($searchInfo['equal'])) {
                            $search[] = ['activityId' => ['equal' => $searchInfo['equal']]];
                        }
                        break;
                    case 'type':
                        if (isset($searchInfo['equal'])) {
                            $search[] = ['type' => ['equal' => $searchInfo['equal']]];
                        }
                        break;
                    case 'prizeId':
                        if (isset($searchInfo['equal'])) {
                            $search[] = ['prizeId' => ['equal' => $searchInfo['equal']]];
                        }
                        break;
                    case 'checkCode':
                        if (isset($searchInfo['equal'])) {
                            $search[] = ['checkCode' => ['equal' => $searchInfo['equal']]];
                        }
                        break;
                }
            }
        }

        return $search;
    }

    public function drawn()
    {
        if (RequestHelper::isPost()) {

            $page = Input::get('page');
            $pageSize = 10;
            $option = [];
            $option['where']['memberUserId'] = $this->memberUserId();
            $option['order'] = ['id', 'desc'];
            $option['search'] = $this->drawnListSearchFilter(Input::get('search', []));

            $head = [];

            $head[] = ['field' => 'created_at', 'title' => '时间',];
            $head[] = ['field' => 'activityType', 'title' => '活动类型',];
            $head[] = ['field' => 'activityId', 'title' => '活动ID',];
            $head[] = ['field' => 'user', 'title' => '用户',];
            $head[] = ['field' => 'type', 'title' => '类型',];
            $head[] = ['field' => 'name', 'title' => '名称',];
            $head[] = ['field' => 'prize', 'title' => '奖品',];
            $head[] = ['field' => 'physicalStatus', 'title' => '实物状态',];
            $head[] = ['field' => 'checkCode', 'title' => '实物核销码',];
            $head[] = ['field' => 'userName', 'title' => '用户姓名',];
            $head[] = ['field' => 'userPhone', 'title' => '用户手机',];
            $head[] = ['field' => 'userAddress', 'title' => '用户地址',];

            $list = [];
            $paginateData = ModelHelper::modelPaginate('module_prize_pool_drawn', $page, $pageSize, $option);
            ModelHelper::modelJoin($paginateData['records'], 'userId', '_user', 'member_user', 'id');
            foreach ($paginateData['records'] as &$record) {
                $item = [];
                $item['_id'] = $record['id'];
                $item['created_at'] = $record['created_at'];
                $item['activityType'] = TypeHelper::name(ActivityType::class, $record['activityType']);
                $item['activityId'] = $record['activityId'];
                $item['user'] = CmsHtmlHelper::user($record['_user']);
                $item['name'] = htmlspecialchars($record['name']);
                $item['type'] = CmsHtmlHelper::badge(PrizeType::class, $record['type'], [
                    PrizeType::REDBAG => 'danger',
                    PrizeType::CREDIT => 'success',
                    PrizeType::CODE => 'warning',
                    PrizeType::PHYSICAL => 'primary',
                ]);
                switch ($record['type']) {
                    case PrizeType::REDBAG:
                        $item['prize'] = $record['redbagValue'] . ' 元';
                        break;
                    case PrizeType::CREDIT:
                        $item['prize'] = $record['creditValue'] . ' 分';
                        break;
                    case PrizeType::CODE:
                        $values = [];
                        if ($record['codeNo']) {
                            $values[] = '卡号:' . $record['codeNo'];
                        }
                        if ($record['codeSecret']) {
                            $values[] = '卡密:' . $record['codeSecret'];
                        }
                        if ($record['codeExpire']) {
                            $values[] = '过期:' . $record['codeExpire'];
                        }
                        $item['prize'] = join(', ', $values);
                        break;
                    case PrizeType::PHYSICAL:
                        $item['prize'] = $record['name'];
                        break;
                }
                $item['physicalStatus'] = CmsHtmlHelper::status(DrawnPhysicalStatus::class, $record['physicalStatus'], [
                    DrawnPhysicalStatus::DRAWN => 'warning',
                    DrawnPhysicalStatus::WAIT_SHIPPING => 'danger',
                    DrawnPhysicalStatus::SHIPPED => 'success',
                    DrawnPhysicalStatus::CHECKED => 'success',
                ]);
                switch ($record['physicalStatus']) {
                    case DrawnPhysicalStatus::DRAWN:
                        $item['physicalStatus'] .= ' ' . CmsHtmlHelper::pass(
                                action('\App\Modules\PrizePool\Manage\HomeController@drawnCheck', ['id' => $record['id']]),
                                '确认核销'
                            );
                        break;
                    case DrawnPhysicalStatus::WAIT_SHIPPING:
                        $item['physicalStatus'] .= ' ' . CmsHtmlHelper::pass(
                                action('\App\Modules\PrizePool\Manage\HomeController@drawnShipped', ['id' => $record['id']]),
                                '确认已发货'
                            );
                        break;
                }
                $item['checkCode'] = $record['checkCode'] ? $record['checkCode'] : '-';
                $item['userName'] = $record['userName'] ? $record['userName'] : '';
                $item['userPhone'] = $record['userPhone'] ? $record['userPhone'] : '';
                $item['userAddress'] = $record['userAddress'] ? $record['userAddress'] : '';
                $list[] = $item;
            }

            $data = [];
            $data['head'] = $head;
            $data['list'] = $list;
            $data['total'] = $paginateData['total'];
            $data['pageSize'] = $pageSize;
            $data['page'] = $page;

            return Response::generate(0, null, $data);
        }

        return $this->_view('module.prizePool.drawn', []);
    }

    public function drawnCheck($drawnId)
    {
        $drawn = ModelHelper::load('module_prize_pool_drawn', ['id' => $drawnId, 'memberUserId' => $this->memberUserId()]);
        if (empty($drawn)) {
            return Response::send(-1, '记录不存在');
        }
        if ($drawn['physicalStatus'] != DrawnPhysicalStatus::DRAWN) {
            return Response::send(-1, '状态不正确');
        }
        ModelHelper::updateOne('module_prize_pool_drawn', ['id' => $drawnId], [
            'physicalStatus' => DrawnPhysicalStatus::CHECKED,
        ]);
        return Response::send(0, null, null, '[js]lister.load(false)');
    }

    public function drawnShipped($drawnId)
    {
        $drawn = ModelHelper::load('module_prize_pool_drawn', ['id' => $drawnId, 'memberUserId' => $this->memberUserId()]);
        if (empty($drawn)) {
            return Response::send(-1, '记录不存在');
        }
        if ($drawn['physicalStatus'] != DrawnPhysicalStatus::WAIT_SHIPPING) {
            return Response::send(-1, '状态不正确');
        }
        ModelHelper::updateOne('module_prize_pool_drawn', ['id' => $drawnId], [
            'physicalStatus' => DrawnPhysicalStatus::SHIPPED,
        ]);
        return Response::send(0, null, null, '[js]lister.load(false)');
    }

    public function setting()
    {
        $setting = ModelHelper::load('module_prize_pool_setting', ['memberUserId' => $this->memberUserId()]);
        if (empty($setting)) {
            $setting = ModelHelper::add('module_prize_pool_setting', [
                'memberUserId' => $this->memberUserId(),
                'emptyName' => '谢谢参与',
                'emptyCover' => '',
            ]);
        }
        if (RequestHelper::isPost()) {
            $update = [];
            $input = InputPackage::buildFromInput();
            $update['emptyName'] = $input->getTrimString('emptyName');
            $update['emptyCover'] = $input->getImagePath('emptyCover');
            ModelHelper::updateOne('module_prize_pool_setting', ['id' => $setting['id']], $update);
            return Response::send(0, '保存成功', null, '/module/manage/prize_pool');
        }
        return $this->_view('module.prizePool.setting', [
            'setting' => $setting,
        ]);
    }

}