<?php

namespace App\Services;


use Edwin404\Base\Support\ModelHelper;
use Edwin404\Base\Support\Response;
use Edwin404\Config\Facades\ConfigFacade;
use Edwin404\Member\Services\MemberCreditService;
use Edwin404\Member\Services\MemberMoneyService;
use Edwin404\Oauth\Types\OauthType;
use Edwin404\Tecmz\Helpers\ConfigEnvHelper;
use Edwin404\Wechat\Types\WechatAuthType;
use EdwinFound\Utils\RandomUtil;
use Illuminate\Support\Facades\Log;

class PlatformService
{
    private $memberCreditService;

    public function __construct(MemberMoneyService $memberMoneyService)
    {
        $this->memberMoneyService = $memberMoneyService;
    }

    public function getWechatApp($memberUserId)
    {
        return \Edwin404\Wechat\Helpers\WechatHelper::app(0, [
            'enable' => true,
            'appId' => ConfigEnvHelper::get('oauthWechatMobileAppId', ''),
            'appSecret' => ConfigEnvHelper::get('oauthWechatMobileAppSecret', ''),
            'appToken' => '',
            'appEncodingKey' => '',
            'authType' => WechatAuthType::CONFIG,
        ], ['payment' => false,]);
    }

    public function getWechatPaymentApp($memberUserId)
    {
        if (!ConfigFacade::get('payWechatMobileOn', false)) {
            return null;
        }
        return \Edwin404\Wechat\Helpers\WechatHelper::app(0, [
            'enable' => true,
            'appId' => ConfigEnvHelper::get('payWechatMobileAppId', ''),
            'appSecret' => ConfigEnvHelper::get('payWechatMobileAppSecret', ''),
            'appToken' => '',
            'appEncodingKey' => '',
            'authType' => WechatAuthType::CONFIG,
        ], ['payment' => [
            'merchantId' => ConfigFacade::get('payWechatMobileMerchantId'),
            'key' => ConfigFacade::get('payWechatMobileKey'),
            'dataCert' => ConfigFacade::get('payWechatMobileFileCert'),
            'dataKey' => ConfigFacade::get('payWechatMobileFileKey'),
        ]]);
    }

    /**
     * !! 这个方法应该在事务中调用，使用锁进行保护
     * 从MemberUserId划账到UserId
     *
     * @param $memberUserId
     * @param $userId
     * @param $change
     * @param $remark
     * @throws \Exception
     */
    public function remitMoneyToUser($memberUserId, $userId, $change, $remark)
    {
        $this->memberMoneyService->change($userId, $change, $remark);
    }

    public function userPayOpenId($userId)
    {
        $memberOauth = ModelHelper::load('member_oauth', [
            'memberUserId' => $userId,
            'type' => OauthType::WECHAT_MOBILE,
        ]);
        if (empty($memberOauth)) {
            return null;
        }
        return $memberOauth['openId'];
    }


    /**
     * !! 这个方法应该在事务中调用，使用锁进行保护
     *
     * @param $memberUserId
     * @param $userId
     * @param $money
     * @return array
     */
    public function cashMoneyToUser($memberUserId, $userId, $money)
    {

        if ($money < 1.00) {
            return Response::generate(-1, '最小金额为1元');
        }

        $app = $this->getWechatPaymentApp($memberUserId);
        if (empty($app)) {
            return Response::generate(-1, "微信支付未开启");
        }

        $this->memberMoneyService->change($userId, -$money, '用户提现');
        $merchantPay = $app->merchant_pay;

        $user = ModelHelper::load('member_user', ['id' => $userId]);

        $openId = $this->userPayOpenId($userId);
        if (empty($openId)) {
            return Response::generate(-1, "用户OpenId为空");
        }

        $merchantPayData = [
            'partner_trade_no' => 'mzBigWheelT' . time() . '' . RandomUtil::number(4),
            'openid' => $openId,
            'check_name' => 'NO_CHECK',
            're_user_name' => $user['username'],
            'amount' => intval(bcmul($money, 100, 2)),
            'desc' => '用户提现',
            'spbill_create_ip' => '127.0.0.1',
        ];

        try {
            $result = $merchantPay->send($merchantPayData)->toArray();
            if (
                isset($result['return_code']) && $result['return_code'] == 'SUCCESS'
                &&
                isset($result['result_code']) && $result['result_code'] == 'SUCCESS'
            ) {
                return Response::generate(0, null);
            } else {
                // 失败
                return Response::generate(-1, '提现失败:' . json_encode($result, JSON_UNESCAPED_UNICODE));
            }
        } catch (\Exception $e) {
            // 有可能因为网络等错误实际上支付成功了但是没有返回结果
            // 这时候需要假定支付已经发生了但是没有付款成功
            Log::error('支付失败:' . $e->getMessage());
            Log::error('支付失败:' . $e->getTraceAsString());
            return Response::generate(0, null);
        }

    }

}