<?php

/**
 * Yeepay 配置类
 *
 * @author Jeffrey
 * @version $Id: Config.php 32 2010-01-17 11:34:32Z jeffrey $
 */
class Yeepay_Config
{
    /**
     * 商户编号
     *
     * @var string
     */
    private $merchantId;

    /**
     * 商户密钥
     *
     * @var string
     */
    private $merchantKey;

    /**
     * 业务请求地址
     *
     * @var Array<string>
     */
    private $commandUrls = array(
        'Buy' => 'https://www.yeepay.com/app-merchant-proxy/command.action',
        'ChargeCardDirect' => 'https://www.yeepay.com/app-merchant-proxy/command.action', // 充值卡充值
        'WapBank' => 'http://www.yeepay.com/app-merchant-proxy/wap/controller.action', // 手机网银
    );

    /**
     * 从数组内容中创建 Yeepay 配置对象
     *
     * @param Array $config
     * @return Yeepay_Config
     */
    public static function createFromArray(Array $config)
    {
        return new self($config['merchantId'], $config['merchantKey'], $config['commandUrls']);
    }

    /**
     * 从 XML 文件内容创建 Yeepay 配置对象
     *
     * @param string $filename
     * @param string $xpath
     * @return Yeepay_Config
     */
    public static function createFromXml($filename, $xpath = '/')
    {
        $config = simplexml_load_file($filename)->xpath($xpath);
        return self::createFromArray($config);
    }

    /**
     * 从 INI 文件内容创建 Yeepay 配置对象
     *
     * @param string $filename
     * @param boolean $processSections
     * @return Yeepay_Config
     */
    public static function createFromIni($filename, $processSections = false)
    {
        $config = parse_ini_file($filename, $processSections);
        return self::createFromArray($config);
    }

    /**
     * 创建 Yeepay 配置对象
     *
     * @throws Yeepay_Exception
     *
     * @param mixed $merchantId
     * @param mixed $merchantKey
     * @param mixed $commandUrls
     *
     * @return Yeepay_Config
     */
    public function __construct($merchantId, $merchantKey, $commandUrls = array())
    {
        $this->merchantId = $merchantId;
        $this->merchantKey = $merchantKey;

        if (is_array($commandUrls))
        {
            $this->commandUrls = array_merge($this->commandUrls, $commandUrls);
        }

        if (false == $this->validate())
        {
            throw new Yeepay_Exception('Config no valid!');
        }
    }

    /**
     * 获取商户编号
     *
     * @return string
     */
    public function getMerchantId()
    {
        return $this->merchantId;
    }

    /**
     * 获取商户密钥
     *
     * @return string
     */
    public function getMerchantKey()
    {
        return $this->merchantKey;
    }

    /**
     * 获取业务请求地址
     *
     * @param string $command
     */
    public function getCommandUrl($command)
    {
        return $this->commandUrls[$command];
    }

    /**
     * 判断配置是否有效
     *
     * @todo 本方法需要使用更严格和更正规的验证方式
     *
     * @return boolean
     */
    public function validate()
    {
        return false == (empty($this->merchantId) || empty($this->merchantKey) || empty($this->commandUrls));
    }
}