/**
 * @license
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import '../../../test/common-test-setup-karma.js';
import '../gr-diff/gr-diff-line.js';
import './gr-coverage-layer.js';

const basicFixture = fixtureFromElement('gr-coverage-layer');

suite('gr-coverage-layer', () => {
  let element;

  setup(() => {
    const initialCoverageRanges = [
      {
        type: 'COVERED',
        side: 'right',
        code_range: {
          start_line: 1,
          end_line: 2,
        },
      },
      {
        type: 'NOT_COVERED',
        side: 'right',
        code_range: {
          start_line: 3,
          end_line: 4,
        },
      },
      {
        type: 'PARTIALLY_COVERED',
        side: 'right',
        code_range: {
          start_line: 5,
          end_line: 6,
        },
      },
      {
        type: 'NOT_INSTRUMENTED',
        side: 'right',
        code_range: {
          start_line: 8,
          end_line: 9,
        },
      },
    ];

    element = basicFixture.instantiate();
    element.coverageRanges = initialCoverageRanges;
    element.side = 'right';
  });

  suite('annotate', () => {
    function createLine(lineNumber) {
      const lineEl = document.createElement('div');
      lineEl.setAttribute('data-side', 'right');
      lineEl.setAttribute('data-value', lineNumber);
      lineEl.className = 'right';
      return lineEl;
    }

    function checkLine(lineNumber, className, opt_negated) {
      const line = createLine(lineNumber);
      element.annotate(undefined, line, undefined);
      let contains = line.classList.contains(className);
      if (opt_negated) contains = !contains;
      assert.isTrue(contains);
    }

    test('line 1-2 are covered', () => {
      checkLine(1, 'COVERED');
      checkLine(2, 'COVERED');
    });

    test('line 3-4 are not covered', () => {
      checkLine(3, 'NOT_COVERED');
      checkLine(4, 'NOT_COVERED');
    });

    test('line 5-6 are partially covered', () => {
      checkLine(5, 'PARTIALLY_COVERED');
      checkLine(6, 'PARTIALLY_COVERED');
    });

    test('line 7 is implicitly not instrumented', () => {
      checkLine(7, 'COVERED', true);
      checkLine(7, 'NOT_COVERED', true);
      checkLine(7, 'PARTIALLY_COVERED', true);
      checkLine(7, 'NOT_INSTRUMENTED', true);
    });

    test('line 8-9 are not instrumented', () => {
      checkLine(8, 'NOT_INSTRUMENTED');
      checkLine(9, 'NOT_INSTRUMENTED');
    });

    test('coverage correct, if annotate is called out of order', () => {
      checkLine(8, 'NOT_INSTRUMENTED');
      checkLine(1, 'COVERED');
      checkLine(5, 'PARTIALLY_COVERED');
      checkLine(3, 'NOT_COVERED');
      checkLine(6, 'PARTIALLY_COVERED');
      checkLine(4, 'NOT_COVERED');
      checkLine(9, 'NOT_INSTRUMENTED');
      checkLine(2, 'COVERED');
    });
  });
});

