/**
 * @license
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import '../../../test/common-test-setup-karma.js';
import '../../shared/gr-js-api-interface/gr-js-api-interface.js';
import {GrDomHook, GrDomHooksManager} from './gr-dom-hooks.js';
import {_testOnly_initGerritPluginApi} from '../../shared/gr-js-api-interface/gr-gerrit.js';

const pluginApi = _testOnly_initGerritPluginApi();

suite('gr-dom-hooks tests', () => {
  let instance;
  let hook;

  setup(() => {
    let plugin;
    pluginApi.install(p => { plugin = p; }, '0.1',
        'http://test.com/plugins/testplugin/static/test.js');
    instance = new GrDomHooksManager(plugin);
  });

  suite('placeholder', () => {
    setup(()=>{
      sinon.stub(GrDomHook.prototype, '_createPlaceholder');
      hook = instance.getDomHook('foo-bar');
    });

    test('registers placeholder class', () => {
      assert.isTrue(hook._createPlaceholder.calledWithExactly(
          'testplugin-autogenerated-foo-bar'));
    });

    test('getModuleName()', () => {
      const hookName = Object.keys(instance.hooks).pop();
      assert.equal(hookName, 'testplugin-autogenerated-foo-bar');
      assert.equal(hook.getModuleName(), 'testplugin-autogenerated-foo-bar');
    });
  });

  suite('custom element', () => {
    setup(() => {
      hook = instance.getDomHook('foo-bar', 'my-el');
    });

    test('getModuleName()', () => {
      const hookName = Object.keys(instance.hooks).pop();
      assert.equal(hookName, 'foo-bar my-el');
      assert.equal(hook.getModuleName(), 'my-el');
    });

    test('onAttached', () => {
      const onAttachedSpy = sinon.spy();
      hook.onAttached(onAttachedSpy);
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      hook.handleInstanceAttached(el1);
      hook.handleInstanceAttached(el2);
      assert.isTrue(onAttachedSpy.firstCall.calledWithExactly(el1));
      assert.isTrue(onAttachedSpy.secondCall.calledWithExactly(el2));
    });

    test('onDetached', () => {
      const onDetachedSpy = sinon.spy();
      hook.onDetached(onDetachedSpy);
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      hook.handleInstanceDetached(el1);
      assert.isTrue(onDetachedSpy.firstCall.calledWithExactly(el1));
      hook.handleInstanceDetached(el2);
      assert.isTrue(onDetachedSpy.secondCall.calledWithExactly(el2));
    });

    test('getAllAttached', () => {
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      el1.textContent = 'one';
      el2.textContent = 'two';
      hook.handleInstanceAttached(el1);
      hook.handleInstanceAttached(el2);
      assert.deepEqual([el1, el2], hook.getAllAttached());
      hook.handleInstanceDetached(el1);
      assert.deepEqual([el2], hook.getAllAttached());
    });

    test('getLastAttached', () => {
      const beforeAttachedPromise = hook.getLastAttached().then(
          el => assert.strictEqual(el1, el));
      const [el1, el2] = [
        document.createElement(hook.getModuleName()),
        document.createElement(hook.getModuleName()),
      ];
      el1.textContent = 'one';
      el2.textContent = 'two';
      hook.handleInstanceAttached(el1);
      hook.handleInstanceAttached(el2);
      const afterAttachedPromise = hook.getLastAttached().then(
          el => assert.strictEqual(el2, el));
      return Promise.all([
        beforeAttachedPromise,
        afterAttachedPromise,
      ]);
    });
  });
});

