# JavaScript modules vs. NgModules

JavaScript modules and NgModules can help you modularize your code, but they are very different.
Angular apps rely on both kinds of modules.

## JavaScript modules: Files containing code

A [JavaScript module](https://javascript.info/modules "JavaScript.Info - Modules") is an individual file with JavaScript code, usually containing a class or a library of functions for a specific purpose within your app.
JavaScript modules let you spread your work across multiple files.

<div class="alert is-helpful">

To learn more about JavaScript modules, see [ES6 In Depth: Modules](https://hacks.mozilla.org/2015/08/es6-in-depth-modules/).
For the module specification, see the [6th Edition of the ECMAScript standard](https://www.ecma-international.org/ecma-262/6.0/#sec-modules).

</div>

To make the code in a JavaScript module available to other modules, use an `export` statement at the end of the relevant code in the module, such as the following:

```typescript
export class AppComponent { ... }
```

When you need that module’s code in another module, use an `import` statement as follows:

```typescript
import { AppComponent } from './app.component';
```

Each module has its own top-level scope.
In other words, top-level variables and functions in a module are not seen in other scripts or modules.
Each module provides a namespace for identifiers to prevent them from clashing with identifiers in other modules.
With multiple modules, you can prevent accidental global variables by creating a single global namespace and adding sub-modules to it.

The Angular framework itself is loaded as a set of JavaScript modules.

## NgModules: Classes with metadata for compiling

An [NgModule](guide/glossary#ngmodule "Definition of NgModule") is a class marked by the `@NgModule` decorator with a metadata object that describes how that particular part of the app fits together with the other parts.
NgModules are specific to Angular.
While classes with an `@NgModule` decorator are by convention kept in their own files, they differ from JavaScript modules because they include this metadata.

The `@NgModule` metadata plays an important role in guiding the Angular compilation process that converts the app code you write into highly performant JavaScript code.
The metadata describes how to compile a component's template and how to create an [injector](guide/glossary#injector "Definition of injector") at runtime.
It identifies the NgModule's [components](guide/glossary#component "Definition of component"), [directives](guide/glossary#directive "Definition of directive"), and [pipes](guide/glossary#pipe "Definition of pipe)"),
and makes some of them public through the `exports` property so that external components can use them.
You can also use an NgModule to add [providers](guide/glossary#provider "Definition of provider") for [services](guide/glossary#service "Definition of a service"), so that the services are available elsewhere in your app.

Rather than defining all member classes in one giant file as a JavaScript module, declare which components, directives, and pipes belong to the NgModule in the `@NgModule.declarations` list.
These classes are called [declarables](guide/glossary#declarable "Definition of a declarable").
An NgModule can export only the declarable classes it owns or imports from other NgModules.
It doesn't declare or export any other kind of class.
Declarables are the only classes that matter to the Angular compilation process.

For a complete description of the NgModule metadata properties, see [Using the NgModule metadata](guide/ngmodule-api "Using the NgModule metadata").

## An example that uses both

The root NgModule `AppModule` generated by the [Angular CLI](cli) for a new app project demonstrates how you use both kinds of modules:

<code-example path="ngmodules/src/app/app.module.1.ts" header="src/app/app.module.ts (default AppModule)"></code-example>

The root NgModule starts with `import` statements to import JavaScript modules.
It then configures the `@NgModule` with the following arrays:

* `declarations`: The components, directives, and pipes that belong to the NgModule.
  A new app project's root NgModule has only one component, called `AppComponent`.

* `imports`: Other NgModules you are using, so that you can use their declarables.
  The newly generated root NgModule imports [`BrowserModule`](api/platform-browser/BrowserModule "BrowserModule NgModule") in order to use browser-specific services such as [DOM](https://www.w3.org/TR/DOM-Level-2-Core/introduction.html "Definition of Document Object Model") rendering, sanitization, and location.

* `providers`: Providers of services that components in other NgModules can use.
  There are no providers in a newly generated root NgModule.

* `bootstrap`: The [entry component](guide/entry-components "Specifying an entry component") that Angular creates and inserts into the `index.html` host web page, thereby bootstrapping the app.
  This entry component, `AppComponent`, appears in both the `declarations` and the `bootstrap` arrays.

## Next steps

* For more about NgModules, see [Organizing your app with NgModules](guide/ngmodules "Organizing your app with NgModules").
* To learn more about the root NgModule, see [Launching an app with a root NgModule](guide/bootstrapping "Launching an app with a root NgModule").
* To learn about frequently used Angular NgModules and how to import them into your app, see [Frequently-used modules](guide/frequent-ngmodules "Frequently-used modules").
