<?php

/**
 * MyQEE Session 类
 *
 * @author	 jonwang(jonwang@myqee.com)
 * @category   MyQEE
 * @package	System
 * @subpackage Core
 * @copyright  Copyright (c) 2008-2011 myqee.com
 * @license	http://www.myqee.com/license.html
 */
class MyQEE_Session
{

    /**
     * @var Session
     */
    protected static $instance;

    // Protected key names (cannot be set by the user)
    protected static $protect = array('SID' => 1, '_flash_session_' => 1);

    public static $config;

    protected static $flash;

    /**
     * Session驱动
     * @var Session_Driver_Default
     */
    protected $driver;

    /**
     * @var ORM_Admin_Member_Data
     */
    protected static $member;

    /**
     * @return Session
     */
    public static function instance()
    {
        if ( Session::$instance == NULL )
        {
            // Create a new instance
            new Session();
        }
        return Session::$instance;
    }

    /**
     * On first session instance creation, sets up the driver and creates session.
     */
    public function __construct($vars = null)
    {
        // This part only needs to be run once
        if ( Session::$instance === NULL )
        {
            // Load config
            Session::$config = Core::config('session');

            if ( ! isset(Session::$config['name']) || ! preg_match('#^(?=.*[a-z])[a-z0-9_]++$#iD', Session::$config['name']) )
            {
                // Name the session, this will also be the name of the cookie
                Session::$config['name'] = 'PHPSESSINID';
            }

            if ( isset(Session::$config['driver']) && class_exists('Session_Driver_' . Session::$config['driver'], true) )
            {
                $driver_name = 'Session_Driver_' . Session::$config['driver'];
                if ( isset(Session::$config['driver_config']) )
                {
                    $this->driver = new $driver_name(Session::$config['driver_config']);
                }
                else
                {
                    $this->driver = new $driver_name();
                }
            }
            else
            {
                $this->driver = new Session_Driver_Default();
            }

            $this->expire_flash();

            if ( $vars )
            {
                // Set the new data
                $this->set($vars);
            }

            if ( ! isset($_SESSION['_flash_session_']) )
            {
                $_SESSION['_flash_session_'] = array();
            }
            Session::$flash = & $_SESSION['_flash_session_'];

            $_SESSION['SID'] = $this->driver->session_id();

            if ( !isset($_SESSION['_last_actived_time_']) || TIME-600>$_SESSION['_last_actived_time_'] )
            {
                # 更新最后活动时间 10分钟更新一次
                $_SESSION['_last_actived_time_'] = TIME;
            }

            # 确保关闭前执行保存
            Core::register_shutdown_function(array('Session', 'write_close'));

            Session::$instance = $this;

            if ( null===Session::$member && isset($_SESSION['member']) )
            {
                Session::$member = new ORM_Admin_Member_Data($_SESSION['member']);
            }
        }
    }

    /**
     * 开启SESSION
     * @return Session
     */
    public function start()
    {
        return $this;
    }

    /**
     * Get the session id.
     *
     * @return  string
     */
    public function id()
    {
        return $_SESSION['SID'];
    }

    /**
     * 銷毀當前Session
     *
     * @return  void
     */
    public function destroy()
    {
        $this->driver->destroy();
    }

    /**
     * 设置用户
     *
     * @param ORM_Admin_Member_Data $member
     * @return Session
     */
    public function set_member(ORM_Admin_Member_Data $member)
    {
        Session::$member = $member;
        if ( $member->id>0 )
        {
            # 设置用户数据
            $member_data = $member->get_field_data();
            $_SESSION['member'] = $member_data;
        }
        else
        {
            # 游客数据则清空
            unset($_SESSION['member']);
        }
        return $this;
    }

    /**
     * 获取用户对象
     *
     * @return ORM_Admin_Member_Data
     */
    public function member()
    {
        if ( null===Session::$member )
        {
            # 创建一个空的用户对象
            Session::$member = new ORM_Admin_Member_Data();
        }
        return Session::$member;
    }

    public function last_actived_time()
    {
        return $_SESSION['_last_actived_time_'];
    }

    /**
     * 此方法用于保存session数据
     * 只执行一次，系统在关闭前会执行
     *
     * @return  void
     */
    public static function write_close()
    {
        if ( null === Session::$instance )
        {
            return false;
        }
        static $run = null;
        if ( $run === null )
        {
            $run = TRUE;
            Session::$instance->expire_flash();

            if ( ! $_SESSION['_flash_session_'] )
            {
                unset($_SESSION['_flash_session_']);
            }

            if ( Session::$member )
            {
                # 设置用户数据
                $member_data = Session::$member->get_field_data();

                $_SESSION['member'] = $member_data;
            }

            Session::$instance->driver->write_close();
        }
    }

    /**
     * Set a session variable.
     *
     * @param   string|array  key, or array of values
     * @param   mixed		 value (if keys is not an array)
     * @return  void
     */
    public function set($keys, $val = FALSE)
    {
        if ( empty($keys) ) return FALSE;

        if ( ! is_array($keys) )
        {
            $keys = array($keys => $val);
        }

        foreach ( $keys as $key => $val )
        {
            if ( isset(Session::$protect[$key]) ) continue;

            // Set the key
            $_SESSION[$key] = $val;
        }
    }

    /**
     * Set a flash variable.
     *
     * @param   string|array  key, or array of values
     * @param   mixed		 value (if keys is not an array)
     * @return  void
     */
    public function set_flash($keys, $val = FALSE)
    {
        if ( empty($keys) ) return FALSE;

        if ( ! is_array($keys) )
        {
            $keys = array($keys => $val);
        }

        foreach ( $keys as $key => $val )
        {
            if ( $key == FALSE ) continue;

            Session::$flash[$key] = 'new';
            $this->set($key, $val);
        }
    }

    /**
     * Freshen one, multiple or all flash variables.
     *
     * @param   string  variable key(s)
     * @return  void
     */
    public function keep_flash($keys = NULL)
    {
        $keys = ($keys === NULL) ? array_keys(Session::$flash) : func_get_args();

        foreach ( $keys as $key )
        {
            if ( isset(Session::$flash[$key]) )
            {
                Session::$flash[$key] = 'new';
            }
        }
    }

    /**
     * Expires old flash data and removes it from the session.
     *
     * @return  void
     */
    public function expire_flash()
    {
        static $run = null;

        // Method can only be run once
        if ( $run === TRUE ) return;

        if ( ! empty(Session::$flash) )
        {
            foreach ( Session::$flash as $key => $state )
            {
                if ( $state === 'old' )
                {
                    // Flash has expired
                    unset(Session::$flash[$key], $_SESSION[$key]);
                }
                else
                {
                    // Flash will expire
                    Session::$flash[$key] = 'old';
                }
            }
        }

        // Method has been run
        $run = TRUE;
    }

    /**
     * Get a variable. Access to sub-arrays is supported with key.subkey.
     *
     * @param   string  variable key
     * @param   mixed   default value returned if variable does not exist
     * @return  mixed   Variable data if key specified, otherwise array containing all session data.
     */
    public function get($key = FALSE, $default = FALSE)
    {
        if ( empty($key) ) return $_SESSION;

        $result = isset($_SESSION[$key]) ? $_SESSION[$key] : Core::key_string($_SESSION, $key);

        return ($result === NULL) ? $default : $result;
    }

    /**
     * Get a variable, and delete it.
     *
     * @param   string  variable key
     * @param   mixed   default value returned if variable does not exist
     * @return  mixed
     */
    public function get_once($key, $default = FALSE)
    {
        $return = $this->get($key, $default);
        $this->delete($key);

        return $return;
    }

    /**
     * Delete one or more variables.
     *
     * @param   string  variable key(s)
     * @return  void
     */
    public function delete()
    {
        $args = func_get_args();

        foreach ( $args as $key )
        {
            if ( isset(Session::$protect[$key]) ) continue;

            // Unset the key
            unset($_SESSION[$key]);
        }
    }

    public static function session_name()
    {
        return Session::$config['name'];
    }
}