/*!
 * FEA v1.4.0 (http://dev.asthis.net/fea)
 * Copyright 2015-2016 AsThis
 * Licensed under MIT (http://dev.asthis.net/fea/LICENSE)
 */
(function ($, FEA) {
	//'use strict';

	// Class
	var Carousel = function (element, options) {
		var base = this;
		base.init(element, options);
	};
	$.extend(Carousel.prototype, {
		init : function (element, options) {
			var base = this;

			base.$elem = element;
			base.options = $.extend({}, Carousel.defaults, options);

			base.userOptions = options;
			base.loadContent();
		},

		loadContent : function () {
			var base = this, url;

			function getData(data) {
				var i, content = '';
				if(typeof base.options.jsonSuccess === 'function') {
					base.options.jsonSuccess.apply(this, [data]);
				}
				else {
					for (i in data.carousel) {
						if(data.carousel.hasOwnProperty(i)) {
							content += data.carousel[i].item;
						}
					}
					base.$elem.html(content);
				}
				base.logIn();
			}

			if(typeof base.options.beforeInit === 'function') {
				base.options.beforeInit.apply(this, [base.$elem]);
			}

			if(typeof base.options.jsonPath === 'string') {
				url = base.options.jsonPath;
				$.getJSON(url, getData);
			}
			else {
				base.logIn();
			}
		},

		logIn : function () {
			var base = this;

			base.$elem.data('carousel-originalStyles', base.$elem.attr('style'));
			base.$elem.data('carousel-originalClasses', base.$elem.attr('class'));

			base.$elem.css({opacity: 0});
			base.orignalItems = base.options.items;
			base.wrapperWidth = 0;
			base.checkVisible = null;
			base.setVars();
		},

		setVars : function () {
			var base = this;
			if(base.$elem.children().length === 0) {return false; }
			base.baseClass();
			base.eventTypes();
			base.$userItems = base.$elem.children();
			base.itemsAmount = base.$userItems.length;
			base.wrapItems();
			base.$carouselItems = base.$elem.find('.carousel-item');
			base.$carouselWrapper = base.$elem.find('.carousel-wrapper');
			base.playDirection = 'next';
			base.prevItem = 0;
			base.prevArr = [0];
			base.currentItem = 0;
			base.customEvents();
			base.onStartup();
		},

		onStartup : function () {
			var base = this;
			base.updateItems();
			base.calculateAll();
			base.buildControls();
			base.updateControls();
			base.response();
			base.moveEvents();
			base.stopOnHover();
			base.carouselStatus();

			if(base.options.transitionStyle !== false) {
				base.transitionTypes(base.options.transitionStyle);
			}
			if(base.options.autoPlay === true) {
				base.options.autoPlay = 5000;
			}
			base.play();

			base.$elem.find('.carousel-wrapper').css('display', 'block');

			if(!base.$elem.is(':visible')) {
				base.watchVisibility();
			}
			else {
				base.$elem.css('opacity', 1);
			}
			base.onstartup = false;
			base.eachMoveUpdate();
			if(typeof base.options.afterInit === 'function') {
				base.options.afterInit.apply(this, [base.$elem]);
			}
		},

		eachMoveUpdate : function () {
			var base = this;

			if(base.options.lazyLoad === true) {
				base.lazyLoad();
			}
			if(base.options.autoHeight === true) {
				base.autoHeight();
			}
			base.onVisibleItems();

			if(typeof base.options.afterAction === 'function') {
				base.options.afterAction.apply(this, [base.$elem]);
			}
		},

		updateVars : function () {
			var base = this;
			if(typeof base.options.beforeUpdate === 'function') {
				base.options.beforeUpdate.apply(this, [base.$elem]);
			}
			base.watchVisibility();
			base.updateItems();
			base.calculateAll();
			base.updatePosition();
			base.updateControls();
			base.eachMoveUpdate();
			if(typeof base.options.afterUpdate === 'function') {
				base.options.afterUpdate.apply(this, [base.$elem]);
			}
		},

		reload : function () {
			var base = this;
			window.setTimeout(function () {
				base.updateVars();
			}, 0);
		},

		watchVisibility : function () {
			var base = this;

			if(base.$elem.is(':visible') === false) {
				base.$elem.css({opacity: 0});
				window.clearInterval(base.autoPlayInterval);
				window.clearInterval(base.checkVisible);
			}
			else {
				return false;
			}
			base.checkVisible = window.setInterval(function () {
				if(base.$elem.is(':visible')) {
					base.reload();
					base.$elem.animate({opacity: 1}, 200);
					window.clearInterval(base.checkVisible);
				}
			}, 500);
		},

		wrapItems : function () {
			var base = this;
			base.$userItems.wrapAll('<div class="carousel-wrapper">').wrap('<div class="carousel-item"></div>');
			base.$elem.find('.carousel-wrapper').wrap('<div class="carousel-wrapper-outer">');
			base.wrapperOuter = base.$elem.find('.carousel-wrapper-outer');
			base.$elem.css('display', 'block');
		},

		baseClass : function () {
			var base = this,
				hasBaseClass = base.$elem.hasClass(base.options.baseClass),
				hasThemeClass = base.$elem.hasClass(base.options.theme);

			if(!hasBaseClass) {
				base.$elem.addClass(base.options.baseClass);
			}

			if(!hasThemeClass) {
				base.$elem.addClass(base.options.theme);
			}
		},

		updateItems : function () {
			var base = this, width, i;

			if(base.options.responsive === false) {
				return false;
			}
			if(base.options.singleItem === true) {
				base.options.items = base.orignalItems = 1;
				base.options.itemsCustom = false;
				base.options.itemsDesktop = false;
				base.options.itemsDesktopSmall = false;
				base.options.itemsTablet = false;
				base.options.itemsTabletSmall = false;
				base.options.itemsMobile = false;
				return false;
			}

			width = $(base.options.responsiveBaseWidth).width();

			if(width > (base.options.itemsDesktop[0] || base.orignalItems)) {
				base.options.items = base.orignalItems;
			}
			if(base.options.itemsCustom !== false) {
				//Reorder array by screen size
				base.options.itemsCustom.sort(function (a, b) {return a[0] - b[0]; });

				for (i = 0; i < base.options.itemsCustom.length; i += 1) {
					if(base.options.itemsCustom[i][0] <= width) {
						base.options.items = base.options.itemsCustom[i][1];
					}
				}

			}
			else {

				if(width <= base.options.itemsDesktop[0] && base.options.itemsDesktop !== false) {
					base.options.items = base.options.itemsDesktop[1];
				}

				if(width <= base.options.itemsDesktopSmall[0] && base.options.itemsDesktopSmall !== false) {
					base.options.items = base.options.itemsDesktopSmall[1];
				}

				if(width <= base.options.itemsTablet[0] && base.options.itemsTablet !== false) {
					base.options.items = base.options.itemsTablet[1];
				}

				if(width <= base.options.itemsTabletSmall[0] && base.options.itemsTabletSmall !== false) {
					base.options.items = base.options.itemsTabletSmall[1];
				}

				if(width <= base.options.itemsMobile[0] && base.options.itemsMobile !== false) {
					base.options.items = base.options.itemsMobile[1];
				}
			}

			//if number of items is less than declared
			if(base.options.items > base.itemsAmount && base.options.itemsScaleUp === true) {
				base.options.items = base.itemsAmount;
			}
		},

		response : function () {
			var base = this,
				smallDelay,
				lastWindowWidth;

			if(base.options.responsive !== true) {
				return false;
			}
			lastWindowWidth = $(window).width();

			base.resizer = function () {
				if($(window).width() !== lastWindowWidth) {
					if(base.options.autoPlay !== false) {
						window.clearInterval(base.autoPlayInterval);
					}
					window.clearTimeout(smallDelay);
					smallDelay = window.setTimeout(function () {
						lastWindowWidth = $(window).width();
						base.updateVars();
					}, base.options.responsiveRefreshRate);
				}
			};
			$(window).resize(base.resizer);
		},

		updatePosition : function () {
			var base = this;
			base.jumpTo(base.currentItem);
			if(base.options.autoPlay !== false) {
				base.checkAp();
			}
		},

		appendItemsSizes : function () {
			var base = this,
				roundPages = 0,
				lastItem = base.itemsAmount - base.options.items;

			base.$carouselItems.each(function (index) {
				var $this = $(this);
				$this
					.css({'width': base.itemWidth})
					.data('carousel-item', Number(index));

				if(index % base.options.items === 0 || index === lastItem) {
					if(index <= lastItem) {
						roundPages += 1;
					}
				}
				$this.data('carousel-roundPages', roundPages);
			});
		},

		appendWrapperSizes : function () {
			var base = this,
				width = base.$carouselItems.length * base.itemWidth;

			base.$carouselWrapper.css({'width': width * 2}).css($.FEA.langdirection, 0);
			base.appendItemsSizes();
		},

		calculateAll : function () {
			var base = this;
			base.calculateWidth();
			base.appendWrapperSizes();
			base.loops();
			base.max();
		},

		calculateWidth : function () {
			var base = this;
			base.itemWidth = Math.round(base.$elem.width() / base.options.items);
		},

		max : function () {
			var base = this,
				maximum = ((base.itemsAmount * base.itemWidth) - base.options.items * base.itemWidth) * -1;
			if(base.options.items > base.itemsAmount) {
				base.maximumItem = 0;
				maximum = 0;
				base.maximumPixels = 0;
			}
			else {
				base.maximumItem = base.itemsAmount - base.options.items;
				base.maximumPixels = maximum;
			}
			return maximum;
		},

		min : function () {
			return 0;
		},

		loops : function () {
			var base = this,
				prev = 0,
				elWidth = 0,
				i,
				item,
				roundPageNum;

			base.positionsInArray = [0];
			base.pagesInArray = [];

			for (i = 0; i < base.itemsAmount; i += 1) {
				elWidth += base.itemWidth;
				base.positionsInArray.push(-elWidth);

				if(base.options.scrollPerPage === true) {
					item = $(base.$carouselItems[i]);
					roundPageNum = item.data('carousel-roundPages');
					if(roundPageNum !== prev) {
						base.pagesInArray[prev] = base.positionsInArray[i];
						prev = roundPageNum;
					}
				}
			}
		},

		buildControls : function () {
			var base = this;
			if(base.options.navigation === true || base.options.pagination === true) {
				base.carouselControls = $('<div class="carousel-controls"/>').toggleClass('clickable', !FEA.support.touch).appendTo(base.$elem);
			}
			if(base.options.pagination === true) {
				base.buildPagination();
			}
			if(base.options.navigation === true) {
				base.buildButtons();
			}
		},

		buildButtons : function () {
			var base = this,
				buttonsWrapper = $('<div class="carousel-buttons"/>');
			base.carouselControls.append(buttonsWrapper);

			base.buttonPrev = $('<div/>', {
				'class' : 'carousel-prev',
				'html' : base.options.navigationText[0] || ''
			});

			base.buttonNext = $('<div/>', {
				'class' : 'carousel-next',
				'html' : base.options.navigationText[1] || ''
			});

			buttonsWrapper
				.append(base.buttonPrev)
				.append(base.buttonNext);

			buttonsWrapper.on('touchstart.carouselControls mousedown.carouselControls', 'div[class^="carousel"]', function (event) {
				event.preventDefault();
			});

			buttonsWrapper.on('touchend.carouselControls mouseup.carouselControls', 'div[class^="carousel"]', function (event) {
				event.preventDefault();
				if($(this).hasClass('carousel-next')) {
					base.next();
				}
				else {
					base.prev();
				}
			});
		},

		buildPagination : function () {
			var base = this;

			base.paginationWrapper = $('<div class="carousel-pagination"/>');
			base.carouselControls.append(base.paginationWrapper);

			base.paginationWrapper.on('touchend.carouselControls mouseup.carouselControls', '.carousel-page', function (event) {
				event.preventDefault();
				if(Number($(this).data('carousel-page')) !== base.currentItem) {
					base.goTo(Number($(this).data('carousel-page')), true);
				}
			});
		},

		updatePagination : function () {
			var base = this,
				counter,
				lastPage,
				lastItem,
				i,
				paginationButton,
				paginationButtonInner;

			if(base.options.pagination === false) {
				return false;
			}

			base.paginationWrapper.html('');

			counter = 0;
			lastPage = base.itemsAmount - base.itemsAmount % base.options.items;

			for (i = 0; i < base.itemsAmount; i += 1) {
				if(i % base.options.items === 0) {
					counter += 1;
					if(lastPage === i) {
						lastItem = base.itemsAmount - base.options.items;
					}
					paginationButton = $('<div/>', {
						'class' : 'carousel-page'
					});
					paginationButtonInner = $('<span></span>', {
						'text': base.options.paginationNumbers === true ? counter : '',
						'class': base.options.paginationNumbers === true ? 'carousel-numbers' : ''
					});
					paginationButton.append(paginationButtonInner);

					paginationButton.data('carousel-page', lastPage === i ? lastItem : i);
					paginationButton.data('carousel-roundPages', counter);

					base.paginationWrapper.append(paginationButton);
				}
			}
			base.checkPagination();
		},

		checkPagination : function () {
			var base = this;
			if(base.options.pagination === false) {
				return false;
			}
			base.paginationWrapper.find('.carousel-page').each(function () {
				if($(this).data('carousel-roundPages') === $(base.$carouselItems[base.currentItem]).data('carousel-roundPages')) {
					base.paginationWrapper
						.find('.carousel-page')
						.removeClass('active');
					$(this).addClass('active');
				}
			});
		},

		checkNavigation : function () {
			var base = this;

			if(base.options.navigation === false) {
				return false;
			}
			if(base.options.rewindNav === false) {
				if(base.currentItem === 0 && base.maximumItem === 0) {
					base.buttonPrev.addClass('disabled');
					base.buttonNext.addClass('disabled');
				}
				else if(base.currentItem === 0 && base.maximumItem !== 0) {
					base.buttonPrev.addClass('disabled');
					base.buttonNext.removeClass('disabled');
				}
				else if(base.currentItem === base.maximumItem) {
					base.buttonPrev.removeClass('disabled');
					base.buttonNext.addClass('disabled');
				}
				else if(base.currentItem !== 0 && base.currentItem !== base.maximumItem) {
					base.buttonPrev.removeClass('disabled');
					base.buttonNext.removeClass('disabled');
				}
			}
		},

		updateControls : function () {
			var base = this;
			base.updatePagination();
			base.checkNavigation();
			if(base.carouselControls) {
				if(base.options.items >= base.itemsAmount) {
					base.carouselControls.hide();
				}
				else {
					base.carouselControls.show();
				}
			}
		},

		destroyControls : function () {
			var base = this;
			if(base.carouselControls) {
				base.carouselControls.remove();
			}
		},

		next : function (speed) {
			var base = this;

			if(base.isTransition) {
				return false;
			}

			base.currentItem += base.options.scrollPerPage === true ? base.options.items : 1;
			if(base.currentItem > base.maximumItem + (base.options.scrollPerPage === true ? (base.options.items - 1) : 0)) {
				if(base.options.rewindNav === true) {
					base.currentItem = 0;
					speed = 'rewind';
				}
				else {
					base.currentItem = base.maximumItem;
					return false;
				}
			}
			base.goTo(base.currentItem, speed);
		},

		prev : function (speed) {
			var base = this;

			if(base.isTransition) {
				return false;
			}

			if(base.options.scrollPerPage === true && base.currentItem > 0 && base.currentItem < base.options.items) {
				base.currentItem = 0;
			}
			else {
				base.currentItem -= base.options.scrollPerPage === true ? base.options.items : 1;
			}
			if(base.currentItem < 0) {
				if(base.options.rewindNav === true) {
					base.currentItem = base.maximumItem;
					speed = 'rewind';
				}
				else {
					base.currentItem = 0;
					return false;
				}
			}
			base.goTo(base.currentItem, speed);
		},

		goTo : function (position, speed, drag) {
			var base = this,
				goToPixel;

			if(base.isTransition) {
				return false;
			}
			if(typeof base.options.beforeMove === 'function') {
				base.options.beforeMove.apply(this, [base.$elem]);
			}
			if(position >= base.maximumItem) {
				position = base.maximumItem;
			}
			else if(position <= 0) {
				position = 0;
			}

			base.currentItem = base.carousel.currentItem = position;
			if(base.options.transitionStyle !== false && drag !== 'drag' && base.options.items === 1 && FEA.support.translate3d === true) {
				base.swapSpeed(0);
				if(FEA.support.translate3d === true) {
					base.transition3d(base.positionsInArray[position]);
				}
				else {
					base.css2slide(base.positionsInArray[position], 1);
				}
				base.afterGo();
				base.singleItemTransition();
				return false;
			}
			goToPixel = base.positionsInArray[position];

			if(FEA.support.translate3d === true) {
				base.isCss3Finish = false;

				if(speed === true) {
					base.swapSpeed('paginationSpeed');
					window.setTimeout(function () {
						base.isCss3Finish = true;
					}, base.options.paginationSpeed);
				}
				else if(speed === 'rewind') {
					base.swapSpeed(base.options.rewindSpeed);
					window.setTimeout(function () {
						base.isCss3Finish = true;
					}, base.options.rewindSpeed);

				}
				else {
					base.swapSpeed('slideSpeed');
					window.setTimeout(function () {
						base.isCss3Finish = true;
					}, base.options.slideSpeed);
				}
				base.transition3d(goToPixel);
			}
			else {
				if(speed === true) {
					base.css2slide(goToPixel, base.options.paginationSpeed);
				}
				else if(speed === 'rewind') {
					base.css2slide(goToPixel, base.options.rewindSpeed);
				}
				else {
					base.css2slide(goToPixel, base.options.slideSpeed);
				}
			}
			base.afterGo();
		},

		jumpTo : function (position) {
			var base = this;
			if(typeof base.options.beforeMove === 'function') {
				base.options.beforeMove.apply(this, [base.$elem]);
			}
			if(position >= base.maximumItem || position === -1) {
				position = base.maximumItem;
			}
			else if(position <= 0) {
				position = 0;
			}
			base.swapSpeed(0);
			if(FEA.support.translate3d === true) {
				base.transition3d(base.positionsInArray[position]);
			}
			else {
				base.css2slide(base.positionsInArray[position], 1);
			}
			base.currentItem = base.carousel.currentItem = position;
			base.afterGo();
		},

		afterGo : function () {
			var base = this;

			base.prevArr.push(base.currentItem);
			base.prevItem = base.carousel.prevItem = base.prevArr[base.prevArr.length - 2];
			base.prevArr.shift(0);

			if(base.prevItem !== base.currentItem) {
				base.checkPagination();
				base.checkNavigation();
				base.eachMoveUpdate();

				if(base.options.autoPlay !== false) {
					base.checkAp();
				}
			}
			if(typeof base.options.afterMove === 'function' && base.prevItem !== base.currentItem) {
				base.options.afterMove.apply(this, [base.$elem]);
			}
		},

		stop : function () {
			var base = this;
			base.apStatus = 'stop';
			window.clearInterval(base.autoPlayInterval);
		},

		checkAp : function () {
			var base = this;
			if(base.apStatus !== 'stop') {
				base.play();
			}
		},

		play : function () {
			var base = this;
			base.apStatus = 'play';
			if(base.options.autoPlay === false) {
				return false;
			}
			window.clearInterval(base.autoPlayInterval);
			base.autoPlayInterval = window.setInterval(function () {
				base.next(true);
			}, base.options.autoPlay);
		},

		swapSpeed : function (action) {
			var base = this;
			if(action === 'slideSpeed') {
				base.$carouselWrapper.css(base.addCssSpeed(base.options.slideSpeed));
			}
			else if(action === 'paginationSpeed') {
				base.$carouselWrapper.css(base.addCssSpeed(base.options.paginationSpeed));
			}
			else if(typeof action !== 'string') {
				base.$carouselWrapper.css(base.addCssSpeed(action));
			}
		},

		addCssSpeed : function (speed) {
			return {
				'-webkit-transition': 'all ' + speed + 'ms ease',
				'-moz-transition': 'all ' + speed + 'ms ease',
				'-o-transition': 'all ' + speed + 'ms ease',
				'transition': 'all ' + speed + 'ms ease'
			};
		},

		removeTransition : function () {
			return {
				'-webkit-transition': '',
				'-moz-transition': '',
				'-o-transition': '',
				'transition': ''
			};
		},

		doTranslate : function (pixels) {
			return {
				'-webkit-transform': 'translate3d(' + pixels + 'px, 0px, 0px)',
				'-moz-transform': 'translate3d(' + pixels + 'px, 0px, 0px)',
				'-o-transform': 'translate3d(' + pixels + 'px, 0px, 0px)',
				'-ms-transform': 'translate3d(' + pixels + 'px, 0px, 0px)',
				'transform': 'translate3d(' + pixels + 'px, 0px,0px)'
			};
		},

		transition3d : function (value) {
			var base = this;
			value = ('left' === $.FEA.langdirection ? value : value * -1);
			base.$carouselWrapper.css(base.doTranslate(value));
		},

		css2move : function (value) {
			var base = this;
			base.$carouselWrapper.css($.FEA.langdirection, value);
		},

		css2slide : function (value, speed) {
			var base = this;

			base.isCssFinish = false;

			var ani = {};
			ani[$.FEA.langdirection] = value;
			base.$carouselWrapper.stop(true, true).animate(ani, {
				duration : speed || base.options.slideSpeed,
				complete : function () {
					base.isCssFinish = true;
				}
			});
		},

		moveEvents : function () {
			var base = this;
			if(base.options.mouseDrag !== false || base.options.touchDrag !== false) {
				base.gestures();
				base.disabledEvents();
			}
		},

		eventTypes : function () {
			var base = this,
				types = ['s', 'e', 'x'];

			base.evTypes = {};

			if(base.options.mouseDrag === true && base.options.touchDrag === true) {
				types = [
					'touchstart.carousel mousedown.carousel',
					'touchmove.carousel mousemove.carousel',
					'touchend.carousel touchcancel.carousel mouseup.carousel'
				];
			}
			else if(base.options.mouseDrag === false && base.options.touchDrag === true) {
				types = [
					'touchstart.carousel',
					'touchmove.carousel',
					'touchend.carousel touchcancel.carousel'
				];
			}
			else if(base.options.mouseDrag === true && base.options.touchDrag === false) {
				types = [
					'mousedown.carousel',
					'mousemove.carousel',
					'mouseup.carousel'
				];
			}

			base.evTypes.start = types[0];
			base.evTypes.move = types[1];
			base.evTypes.end = types[2];
		},

		disabledEvents :  function () {
			var base = this;
			base.$elem.on('dragstart.carousel', function (event) { event.preventDefault(); });
			base.$elem.on('mousedown.disableTextSelect', function (e) {
				return $(e.target).is('input, textarea, select, option');
			});
		},

		gestures : function () {
			var base = this,
				locals = {
					offsetX : 0,
					offsetY : 0,
					baseElWidth : 0,
					relativePos : 0,
					position: null,
					minSwipe : null,
					maxSwipe: null,
					sliding : null,
					dargging: null,
					targetElement : null
				};

			base.isCssFinish = true;

			function getTouches(event) {
				if(event.touches !== undefined) {
					return {
						x : event.touches[0].pageX,
						y : event.touches[0].pageY
					};
				}

				if(event.touches === undefined) {
					if(event.pageX !== undefined) {
						return {
							x : event.pageX,
							y : event.pageY
						};
					}
					if(event.pageX === undefined) {
						return {
							x : event.clientX,
							y : event.clientY
						};
					}
				}
			}

			function swapEvents(type) {
				if(type === 'on') {
					$(document).on(base.evTypes.move, dragMove);
					$(document).on(base.evTypes.end, dragEnd);
				}
				else if(type === 'off') {
					$(document).off(base.evTypes.move);
					$(document).off(base.evTypes.end);
				}
			}

			function dragStart(event) {
				var ev = event.originalEvent || event || window.event,
					position;

				if(ev.which === 3) {
					return false;
				}
				if(base.itemsAmount <= base.options.items) {
					return;
				}
				if(base.isCssFinish === false && !base.options.dragBeforeAnimFinish) {
					return false;
				}
				if(base.isCss3Finish === false && !base.options.dragBeforeAnimFinish) {
					return false;
				}

				if(base.options.autoPlay !== false) {
					window.clearInterval(base.autoPlayInterval);
				}

				if(FEA.support.touch !== true && !base.$carouselWrapper.hasClass('carousel-grabbing')) {
					base.$carouselWrapper.addClass('carousel-grabbing');
				}

				base.newPosX = 0;
				base.newRelativeX = 0;

				$(this).css(base.removeTransition());

				position = $(this).position();
				locals.relativePos = position.left;

				locals.offsetX = getTouches(ev).x - position.left;
				locals.offsetY = getTouches(ev).y - position.top;

				swapEvents('on');

				locals.sliding = false;
				locals.targetElement = ev.target || ev.srcElement;
			}

			function dragMove(event) {
				var ev = event.originalEvent || event || window.event,
					minSwipe,
					maxSwipe;

				base.newPosX = getTouches(ev).x - locals.offsetX;
				base.newPosY = getTouches(ev).y - locals.offsetY;
				base.newRelativeX = base.newPosX - locals.relativePos;

				if(typeof base.options.startDragging === 'function' && locals.dragging !== true && base.newRelativeX !== 0) {
					locals.dragging = true;
					base.options.startDragging.apply(base, [base.$elem]);
				}

				if((base.newRelativeX > 8 || base.newRelativeX < -8) && (FEA.support.touch === true)) {
					if(ev.preventDefault !== undefined) {
						ev.preventDefault();
					}
					else {
						ev.returnValue = false;
					}
					locals.sliding = true;
				}

				if((base.newPosY > 10 || base.newPosY < -10) && locals.sliding === false) {
					$(document).off('touchmove.carousel');
				}

				minSwipe = function () {
					return base.newRelativeX / 5;
				};

				maxSwipe = function () {
					return base.maximumPixels + base.newRelativeX / 5;
				};

				base.newPosX = ('left' === $.FEA.langdirection ? 
					Math.max(Math.min(base.newPosX, minSwipe()), maxSwipe()) :
					(base.newPosX + (base.itemsAmount + base.options.items) * base.itemWidth) * -1);

				if(FEA.support.translate3d === true) {
					base.transition3d(base.newPosX);
				}
				else {
					base.css2move(base.newPosX);
				}
			}

			function dragEnd(event) {
				var ev = event.originalEvent || event || window.event,
					newPosition,
					handlers,
					carouselStopEvent;

				ev.target = ev.target || ev.srcElement;

				locals.dragging = false;

				if(FEA.support.touch !== true) {
					base.$carouselWrapper.removeClass('carousel-grabbing');
				}

				if(base.newRelativeX < 0) {
					base.dragDirection = base.carousel.dragDirection = 'left';
				}
				else {
					base.dragDirection = base.carousel.dragDirection = 'right';
				}

				if(base.newRelativeX !== 0) {
					newPosition = base.getNewPosition();
					base.goTo(newPosition, false, 'drag');
					if(locals.targetElement === ev.target && FEA.support.touch !== true) {
						$(ev.target).on('click.disable', function (ev) {
							ev.stopImmediatePropagation();
							ev.stopPropagation();
							ev.preventDefault();
							$(ev.target).off('click.disable');
						});
						handlers = $._data(ev.target, 'events').click;
						carouselStopEvent = handlers.pop();
						handlers.splice(0, 0, carouselStopEvent);
					}
				}
				swapEvents('off');
			}
			base.$elem.on(base.evTypes.start, '.carousel-wrapper', dragStart);
		},

		getNewPosition : function () {
			var base = this,
				newPosition = base.closestItem();

			if(newPosition > base.maximumItem) {
				base.currentItem = base.maximumItem;
				newPosition  = base.maximumItem;
			}
			else if(base.newPosX >= 0) {
				newPosition = 0;
				base.currentItem = 0;
			}
			return newPosition;
		},

		closestItem : function () {
			var base = this,
				array = base.options.scrollPerPage === true ? base.pagesInArray : base.positionsInArray,
				goal = base.newPosX,
				closest = null;

			$.each(array, function (i, v) {
				if(goal - (base.itemWidth / 20) > array[i + 1] && goal - (base.itemWidth / 20) < v && $.FEA.langdirection === base.moveDirection()) {
					closest = v;
					if(base.options.scrollPerPage === true) {
						base.currentItem = $.inArray(closest, base.positionsInArray);
					}
					else {
						base.currentItem = i;
					}
				}
				else if(goal + (base.itemWidth / 20) < v && goal + (base.itemWidth / 20) > (array[i + 1] || array[i] - base.itemWidth) && $.FEA.langdirection !== base.moveDirection()) {
					if(base.options.scrollPerPage === true) {
						closest = array[i + 1] || array[array.length - 1];
						base.currentItem = $.inArray(closest, base.positionsInArray);
					}
					else {
						closest = array[i + 1];
						base.currentItem = i + 1;
					}
				}
			});
			return base.currentItem;
		},

		moveDirection : function () {
			var base = this,
				direction;
			if(base.newRelativeX < 0) {
				direction = 'right';
				base.playDirection = ('left' === $.FEA.langdirection ? 'next' : 'prev');
			}
			else {
				direction = 'left';
				base.playDirection = ('left' === $.FEA.langdirection ? 'prev' : 'next');
			}
			return direction;
		},

		customEvents : function () {
			/*jslint unparam: true*/
			var base = this;
			base.$elem.on('carousel.next', function () {
				base.next();
			});
			base.$elem.on('carousel.prev', function () {
				base.prev();
			});
			base.$elem.on('carousel.play', function (event, speed) {
				base.options.autoPlay = speed;
				base.play();
				base.hoverStatus = 'play';
			});
			base.$elem.on('carousel.stop', function () {
				base.stop();
				base.hoverStatus = 'stop';
			});
			base.$elem.on('carousel.goTo', function (event, item) {
				base.goTo(item);
			});
			base.$elem.on('carousel.jumpTo', function (event, item) {
				base.jumpTo(item);
			});
		},

		stopOnHover : function () {
			var base = this;
			if(base.options.stopOnHover === true && FEA.support.touch !== true && base.options.autoPlay !== false) {
				base.$elem.on('mouseover', function () {
					base.stop();
				});
				base.$elem.on('mouseout', function () {
					if(base.hoverStatus !== 'stop') {
						base.play();
					}
				});
			}
		},

		lazyLoad : function () {
			var base = this,
				i,
				$item,
				itemNumber,
				$lazyImg,
				follow;

			if(base.options.lazyLoad === false) {
				return false;
			}
			for (i = 0; i < base.itemsAmount; i += 1) {
				$item = $(base.$carouselItems[i]);

				if($item.data('carousel-loaded') === 'loaded') {
					continue;
				}

				itemNumber = $item.data('carousel-item');
				$lazyImg = $item.find('.lazyCarousel');

				if(typeof $lazyImg.data('src') !== 'string') {
					$item.data('carousel-loaded', 'loaded');
					continue;
				}
				if($item.data('carousel-loaded') === undefined) {
					$lazyImg.hide();
					$item.addClass('carousel-loading').data('carousel-loaded', 'checked');
				}
				if(base.options.lazyFollow === true) {
					follow = itemNumber >= base.currentItem;
				}
				else {
					follow = true;
				}
				if(follow && itemNumber < base.currentItem + base.options.items && $lazyImg.length) {
					base.lazyPreload($item, $lazyImg);
				}
			}
		},

		lazyPreload : function ($item, $lazyImg) {
			var base = this,
				iterations = 0,
				isBackgroundImg;

			if($lazyImg.prop('tagName') === 'DIV') {
				$lazyImg.css('background-image', 'url(' + $lazyImg.data('src') + ')');
				isBackgroundImg = true;
			}
			else {
				$lazyImg[0].src = $lazyImg.data('src');
			}

			function showImage() {
				$item.data('carousel-loaded', 'loaded').removeClass('carousel-loading');
				$lazyImg.removeAttr('data-src');
				if(base.options.lazyEffect === 'fade') {
					$lazyImg.fadeIn(400);
				}
				else {
					$lazyImg.show();
				}
				if(typeof base.options.afterLazyLoad === 'function') {
					base.options.afterLazyLoad.apply(this, [base.$elem]);
				}
			}

			function checkLazyImage() {
				iterations += 1;
				if(base.completeImg($lazyImg.get(0)) || isBackgroundImg === true) {
					showImage();
				}
				else if(iterations <= 100) {//if image loads in less than 10 seconds 
					window.setTimeout(checkLazyImage, 100);
				}
				else {
					showImage();
				}
			}

			checkLazyImage();
		},

		autoHeight : function () {
			var base = this,
				$currentimg = $(base.$carouselItems[base.currentItem]).find('img'),
				iterations;

			function addHeight() {
				var $currentItem = $(base.$carouselItems[base.currentItem]).height();
				base.wrapperOuter.css('height', $currentItem + 'px');
				if(!base.wrapperOuter.hasClass('autoHeight')) {
					window.setTimeout(function () {
						base.wrapperOuter.addClass('autoHeight');
					}, 0);
				}
			}

			function checkImage() {
				iterations += 1;
				if(base.completeImg($currentimg.get(0))) {
					addHeight();
				}
				else if(iterations <= 100) { //if image loads in less than 10 seconds 
					window.setTimeout(checkImage, 100);
				}
				else {
					base.wrapperOuter.css('height', ''); //Else remove height attribute
				}
			}

			if($currentimg.get(0) !== undefined) {
				iterations = 0;
				checkImage();
			}
			else {
				addHeight();
			}
		},

		completeImg : function (img) {
			var naturalWidthType;

			if(!img.complete) {
				return false;
			}
			naturalWidthType = typeof img.naturalWidth;
			if(naturalWidthType !== 'undefined' && img.naturalWidth === 0) {
				return false;
			}
			return true;
		},

		onVisibleItems : function () {
			var base = this,
				i;

			if(base.options.addClassActive === true) {
				base.$carouselItems.removeClass('active');
			}
			base.visibleItems = [];
			for (i = base.currentItem; i < base.currentItem + base.options.items; i += 1) {
				base.visibleItems.push(i);

				if(base.options.addClassActive === true) {
					$(base.$carouselItems[i]).addClass('active');
				}
			}
			base.carousel.visibleItems = base.visibleItems;
		},

		transitionTypes : function (className) {
			var base = this;
			//Currently available: 'fade', 'backSlide', 'goDown', 'fadeUp'
			base.outClass = 'carousel-' + className + '-out';
			base.inClass = 'carousel-' + className + '-in';
		},

		singleItemTransition : function () {
			var base = this,
				outClass = base.outClass,
				inClass = base.inClass,
				$currentItem = base.$carouselItems.eq(base.currentItem),
				$prevItem = base.$carouselItems.eq(base.prevItem),
				prevPos = Math.abs(base.positionsInArray[base.currentItem]) + base.positionsInArray[base.prevItem],
				origin = Math.abs(base.positionsInArray[base.currentItem]) + base.itemWidth / 2,
				animEnd = 'webkitAnimationEnd oAnimationEnd MSAnimationEnd animationend';

			base.isTransition = true;

			base.$carouselWrapper
				.addClass('carousel-origin')
				.css({
					'-webkit-transform-origin' : origin + 'px',
					'-moz-perspective-origin' : origin + 'px',
					'perspective-origin' : origin + 'px'
				});
			function transStyles(prevPos) {
				var picss = {'position' : 'relative'};
				picss[$.FEA.langdirection] = prevPos + 'px';
				return picss;
			}

			$prevItem
				.css(transStyles(prevPos, 10))
				.addClass(outClass)
				.on(animEnd, function () {
					base.endPrev = true;
					$prevItem.off(animEnd);
					base.clearTransStyle($prevItem, outClass);
				});

			$currentItem
				.addClass(inClass)
				.on(animEnd, function () {
					base.endCurrent = true;
					$currentItem.off(animEnd);
					base.clearTransStyle($currentItem, inClass);
				});
		},

		clearTransStyle : function (item, classToRemove) {
			var base = this;
			item.css({'position' : ''}).css($.FEA.langdirection, '').removeClass(classToRemove);

			if(base.endPrev && base.endCurrent) {
				base.$carouselWrapper.removeClass('carousel-origin');
				base.endPrev = false;
				base.endCurrent = false;
				base.isTransition = false;
			}
		},

		carouselStatus : function () {
			var base = this;
			base.carousel = {
				'userOptions' : base.userOptions,
				'baseElement' : base.$elem,
				'userItems' : base.$userItems,
				'carouselItems' : base.$carouselItems,
				'currentItem' : base.currentItem,
				'prevItem' : base.prevItem,
				'visibleItems'  : base.visibleItems,
				'dragDirection' : base.dragDirection
			};
		},

		clearEvents : function () {
			var base = this;
			base.$elem.off('.carousel carousel mousedown.disableTextSelect');
			$(document).off('.carousel carousel');
			$(window).off('resize', base.resizer);
		},

		unWrap : function () {
			var base = this;
			if(base.$elem.children().length !== 0) {
				base.$carouselWrapper.unwrap();
				base.$userItems.unwrap().unwrap();
				if(base.carouselControls) {
					base.carouselControls.remove();
				}
			}
			base.clearEvents();
			base.$elem
				.attr('style', base.$elem.data('carousel-originalStyles') || '')
				.attr('class', base.$elem.data('carousel-originalClasses'));
		},

		destroy : function () {
			var base = this;
			base.stop();
			window.clearInterval(base.checkVisible);
			base.unWrap();
			base.$elem.removeData();
		},

		reinit : function (newOptions) {
			var base = this,
				options = $.extend({}, base.userOptions, newOptions);
			base.unWrap();
			base.init(base.$elem, options);
		},

		addItem : function (htmlString, targetPosition) {
			var base = this,
				position;

			if(!htmlString) {return false; }

			if(base.$elem.children().length === 0) {
				base.$elem.append(htmlString);
				base.setVars();
				return false;
			}
			base.unWrap();
			if(targetPosition === undefined || targetPosition === -1) {
				position = -1;
			}
			else {
				position = targetPosition;
			}
			if(position >= base.$userItems.length || position === -1) {
				base.$userItems.eq(-1).after(htmlString);
			}
			else {
				base.$userItems.eq(position).before(htmlString);
			}

			base.setVars();
		},

		removeItem : function (targetPosition) {
			var base = this,
				position;

			if(base.$elem.children().length === 0) {
				return false;
			}
			if(targetPosition === undefined || targetPosition === -1) {
				position = -1;
			}
			else {
				position = targetPosition;
			}

			base.unWrap();
			base.$userItems.eq(position).remove();
			base.setVars();
		}
	});
	Carousel.defaults = {
		items : 5,
		itemsCustom : false,
		itemsDesktop : [1199, 4],
		itemsDesktopSmall : [979, 3],
		itemsTablet : [768, 2],
		itemsTabletSmall : false,
		itemsMobile : [479, 1],
		singleItem : false,
		itemsScaleUp : false,

		slideSpeed : 200,
		paginationSpeed : 800,
		rewindSpeed : 1000,

		autoPlay : false,
		stopOnHover : false,

		navigation : false,
		navigationText : ['prev', 'next'],
		rewindNav : true,
		scrollPerPage : false,

		pagination : true,
		paginationNumbers : false,

		responsive : true,
		responsiveRefreshRate : 200,
		responsiveBaseWidth : window,

		baseClass : 'carousel',
		theme : 'carousel-theme',

		lazyLoad : false,
		lazyFollow : true,
		lazyEffect : 'fade',

		autoHeight : false,

		jsonPath : false,
		jsonSuccess : false,

		dragBeforeAnimFinish : true,
		mouseDrag : true,
		touchDrag : true,

		addClassActive : false,
		transitionStyle : false,

		beforeUpdate : false,
		afterUpdate : false,
		beforeInit : false,
		afterInit : false,
		beforeMove : false,
		afterMove : false,
		afterAction : false,
		startDragging : false,
		afterLazyLoad: false
	};

	// Plugin
	var old = $.fn.carousel;
	$.fn.carousel = function (option) {
		return this.each(function () {
			var $this = $(this),
				data = $this.data('carousel'),
				options = typeof option === 'object' && option;

			if(!data) {
				$this.data('carousel', (data = new Carousel($this, options)));
			}
		});
	};
	// for common use
	$.fn.carousel.Constructor = Carousel;
	// no conflict
	$.fn.carousel.noConflict = function () {
		$.fn.carousel = old;
		return this;
	};

	// Data API
	$(document).ready(function () {
		$('[data-carousel]').each(function () {
			var element = $(this);
			new Carousel(element, FEA.utils.options(element.data('carousel')));
		});
	});
}(jQuery, jQuery.FEA));

(function ($, FEA) {
	'use strict';

	var uiPrefix = 'fea';

	var $window = $(window);
	var $document = $(document);
	var isTouch = 'createTouch' in document;
	var html = document.documentElement;
	var isIE6 = !('minWidth' in html.style);
	var isLosecapture = !isIE6 && 'onlosecapture' in html;
	var isSetCapture = 'setCapture' in html;
	var getEvent = isTouch ? function (event) {
			if(!event.touches) {
				event = event.originalEvent.touches.item(0);
			}
			return event;
		} : function (event) {
			return event;
		};

	var DragEvent = function () {
		this.start = $.proxy(this.start, this);
		this.over = $.proxy(this.over, this);
		this.end = $.proxy(this.end, this);
		this.onstart = this.onover = this.onend = $.noop;
	};
	DragEvent.types =  {
		start: isTouch ? 'touchstart' : 'mousedown',
		over: isTouch ? 'touchmove' : 'mousemove',
		end: isTouch ? 'touchend' : 'mouseup'
	};
	DragEvent.prototype = {
		start: function (event) {
			event = this.startFix(event);

			$document
			.on(DragEvent.types.over, this.over)
			.on(DragEvent.types.end, this.end);

			this.onstart(event);
			return false;
		},

		over: function (event) {
			event = this.overFix(event);
			this.onover(event);
			return false;
		},

		end: function (event) {
			event = this.endFix(event);

			$document
			.off(DragEvent.types.over, this.over)
			.off(DragEvent.types.end, this.end);

			this.onend(event);
			return false;
		},

		startFix: function (event) {
			event = getEvent(event);

			this.target = $(event.target);
			this.selectstart = function () {
				return false;
			};

			$document
			.on('selectstart', this.selectstart)
			.on('dblclick', this.end);

			if(isLosecapture) {
				this.target.on('losecapture', this.end);
			} else {
				$window.on('blur', this.end);
			}

			if(isSetCapture) {
				this.target[0].setCapture();
			}

			return event;
		},

		overFix: function (event) {
			event = getEvent(event);
			return event;
		},

		endFix: function (event) {
			event = getEvent(event);

			$document
			.off('selectstart', this.selectstart)
			.off('dblclick', this.end);

			if(isLosecapture) {
				this.target.off('losecapture', this.end);
			} else {
				$window.off('blur', this.end);
			}

			if(isSetCapture) {
				this.target[0].releaseCapture();
			}

			return event;
		}
	};
	DragEvent.create = function (elem, event) {
		var $elem = $(elem);
		var dragEvent = new DragEvent();
		var startType = DragEvent.types.start;
		var noop = function () {};
		var className = elem.className.replace(/^\s|\s.*/g, '') + '-drag-start';

		var minX, minY, maxX, maxY;

		var api = {
			onstart: noop,
			onover: noop,
			onend: noop,
			off: function () {
				$elem.off(startType, dragEvent.start);
			}
		};

		dragEvent.onstart = function (event) {
			var isFixed = $elem.css('position') === 'fixed';
			var dl = $document.scrollLeft();
			var dt = $document.scrollTop();
			var w = $elem.width();
			var h = $elem.height();

			minX = 0;
			minY = 0;
			maxX = isFixed ? $window.width() - w + minX : $document.width() - w;
			maxY = isFixed ? $window.height() - h + minY : $document.height() - h;

			var offset = $elem.offset();
			var left = this.startLeft = isFixed ? offset.left - dl : offset.left;
			var top = this.startTop = isFixed ? offset.top - dt  : offset.top;

			this.clientX = event.clientX;
			this.clientY = event.clientY;

			$elem.addClass(className);
			api.onstart.call(elem, event, left, top);
		};

		dragEvent.onover = function (event) {
			var left = event.clientX - this.clientX + this.startLeft;
			var top = event.clientY - this.clientY + this.startTop;
			var style = $elem[0].style;

			left = Math.max(minX, Math.min(maxX, left));
			top = Math.max(minY, Math.min(maxY, top));

			style.left = left + 'px';
			style.top = top + 'px';

			api.onover.call(elem, event, left, top);
		};

		dragEvent.onend = function (event) {
			var position = $elem.position();
			var left = position.left;
			var top = position.top;
			$elem.removeClass(className);
			api.onend.call(elem, event, left, top);
		};


		dragEvent.off = function () {
			$elem.off(startType, dragEvent.start);
		};


		if(event) {
			dragEvent.start(event);
		}
		else {
			$elem.on(startType, dragEvent.start);
		}

		return api;
	};

	var _count = 0;
	var _isIE6 = !('minWidth' in $('html')[0].style);
	var _isFixed = !_isIE6;
	var _expando = new Date() - 0;
	var _isMobile = 'createTouch' in document && !('onmousemove' in document) || /(iPhone|iPad|iPod)/i.test(navigator.userAgent);

	var Popup = function () {
		this.destroyed = false;
		this.__popup = $('<div />')
			.css({
				display: 'none',
				position: 'absolute',
				left: 0,
				top: 0,
				bottom: 'auto',
				right: 'auto',
				margin: 0,
				padding: 0,
				border: '0 none',
				background: 'transparent',
				outline: 0
			}).attr('tabindex', '-1').html(this.innerHTML).appendTo('body');
		this.__backdrop = this.__mask = $('<div />').css({opacity: 0.7, background: '#000'});
		this.node = this.__popup[0];
		this.backdrop = this.__backdrop[0];
		_count++;
	};
	$.extend(Popup.prototype, {
		/* Popup.prototype.onshow */
		/* Popup.prototype.onclose */
		/* Popup.prototype.onbeforeremove */
		/* Popup.prototype.onremove */
		/* Popup.prototype.onreset */
		/* Popup.prototype.onfocus */
		/* Popup.prototype.onblur */
		node: null,
		backdrop: null,
		fixed: false,
		destroyed: true,
		open: false,
		returnValue: '',
		autofocus: true,
		align: 'bottom left',
		innerHTML: '',
		className: uiPrefix + '-popup',
		show: function (anchor) {
			if(this.destroyed) {
				return this;
			}
			var popup = this.__popup;
			var backdrop = this.__backdrop;
			this.__activeElement = this.__getActive();
			this.open = true;
			this.follow = anchor || this.follow;
			if(!this.__ready) {
				popup.addClass(this.className).attr('role', this.modal ? 'alertdialog' : 'dialog').css('position', this.fixed ? 'fixed' : 'absolute');
				if(!_isIE6) {
					$(window).on('resize', $.proxy(this.reset, this));
				}
				if(this.modal) {
					var backdropCss = {
						position: 'fixed',
						left: 0,
						top: 0,
						width: '100%',
						height: '100%',
						overflow: 'hidden',
						userSelect: 'none',
						zIndex: this.zIndex || Popup.zIndex
					};
					popup.addClass(this.className + '-modal');
					if(!_isFixed) {
						$.extend(backdropCss, {
							position: 'absolute',
							width: $(window).width() + 'px',
							height: $(document).height() + 'px'
						});
					}
					backdrop.css(backdropCss).attr({
						tabindex: '0'
					}).on('focus', $.proxy(this.focus, this));
					/* lock tab's focus operation */
					this.__mask = backdrop.clone(true).attr('style', '').insertAfter(popup);
					backdrop.addClass(this.className + '-backdrop').insertBefore(popup);
					this.__ready = true;
				}
				if(!popup.html()) {
					popup.html(this.innerHTML);
				}
			}
			popup.addClass(this.className + '-show').show();
			backdrop.show();
			this.reset().focus();
			this.__dispatchEvent('show');
			return this;
		},
		showModal: function () {
			this.modal = true;
			return this.show.apply(this, arguments);
		},
		close: function (result) {
			if(!this.destroyed && this.open) {
				if(result !== undefined) {
					this.returnValue = result;
				}
				this.__popup.hide().removeClass(this.className + '-show');
				this.__backdrop.hide();
				this.open = false;
				this.blur(); // for keyboard user
				this.__dispatchEvent('close');
			}
			return this;
		},
		remove: function () {
			if(this.destroyed) {
				return this;
			}
			this.__dispatchEvent('beforeremove');
			if(Popup.current === this) {
				Popup.current = null;
			}
			// remove from DOM
			this.__popup.remove();
			this.__backdrop.remove();
			this.__mask.remove();
			if(!_isIE6) {
				$(window).off('resize', this.reset);
			}
			this.__dispatchEvent('remove');
			for(var i in this) {
				delete this[i];
			}
			return this;
		},
		reset: function () {
			var elem = this.follow;
			if(elem) {
				this.__follow(elem);
			} else {
				this.__center();
			}
			this.__dispatchEvent('reset');
			return this;
		},
		focus: function () {
			var node = this.node;
			var popup = this.__popup;
			var current = Popup.current;
			var index = this.zIndex = Popup.zIndex++;
			if(current && current !== this) {
				current.blur(false);
			}
			if(!$.contains(node, this.__getActive())) {
				var autofocus = popup.find('[autofocus]')[0];
				if(!this._autofocus && autofocus) {
					this._autofocus = true;
				} else {
					autofocus = node;
				}
				this.__focus(autofocus);
			}
			popup.css('zIndex', index);
			//this.__backdrop.css('zIndex', index);
			Popup.current = this;
			popup.addClass(this.className + '-focus');
			this.__dispatchEvent('focus');
			return this;
		},
		blur: function () {
			var activeElement = this.__activeElement;
			var isBlur = arguments[0];
			if(isBlur !== false) {
				this.__focus(activeElement);
			}
			this._autofocus = false;
			this.__popup.removeClass(this.className + '-focus');
			this.__dispatchEvent('blur');
			return this;
		},
		addEventListener: function (type, callback) {
			this.__getEventListener(type).push(callback);
			return this;
		},
		removeEventListener: function (type, callback) {
			var listeners = this.__getEventListener(type);
			for(var i = 0; i < listeners.length; i++) {
				if(callback === listeners[i]) {
					listeners.splice(i--, 1);
				}
			}
			return this;
		},
		__getEventListener: function (type) {
			var listener = this.__listener;
			if(!listener) {
				listener = this.__listener = {};
			}
			if(!listener[type]) {
				listener[type] = [];
			}
			return listener[type];
		},
		__dispatchEvent: function (type) {
			var listeners = this.__getEventListener(type);
			if(this['on' + type]) {
				this['on' + type]();
			}
			for(var i = 0; i < listeners.length; i++) {
				listeners[i].call(this);
			}
		},
		__focus: function (elem) {
			// Prevent iframe cross domain permission limit and IE invisible element Error
			try {
				// ie11 bug: iframe page click will jump to top
				if(this.autofocus && !/^iframe$/i.test(elem.nodeName)) {
					elem.focus();
				}
			} catch (e) {}
		},
		__getActive: function () {
			try { // try: ie8~9, iframe #26
				var activeElement = document.activeElement;
				var contentDocument = activeElement.contentDocument;
				var elem = contentDocument && contentDocument.activeElement || activeElement;
				return elem;
			} catch (e) {}
		},
		__center: function () {
			var popup = this.__popup;
			var $window = $(window);
			var $document = $(document);
			var fixed = this.fixed;
			var dl = fixed ? 0 : $document.scrollLeft();
			var dt = fixed ? 0 : $document.scrollTop();
			var ww = $window.width();
			var wh = $window.height();
			var ow = popup.width();
			var oh = popup.height();
			var left = (ww - ow) / 2 + dl;
			var top = (wh - oh) * 382 / 1000 + dt; // Golden ratio
			var style = popup[0].style;
			style.left = Math.max(parseInt(left), dl) + 'px';
			style.top = Math.max(parseInt(top), dt) + 'px';
		},
		__follow: function (anchor) {
			var $elem = anchor.parentNode && $(anchor);
			var popup = this.__popup;
			if(this.__followSkin) {
				popup.removeClass(this.__followSkin);
			}
			/* not available for hidden element */
			if($elem) {
				var o = $elem.offset();
				if(o.left * o.top < 0) {
					return this.__center();
				}
			}
			var that = this;
			var fixed = this.fixed;
			var $window = $(window);
			var $document = $(document);
			var winWidth = $window.width();
			var winHeight = $window.height();
			var docLeft = $document.scrollLeft();
			var docTop = $document.scrollTop();
			var popupWidth = popup.width();
			var popupHeight = popup.height();
			var width = $elem ? $elem.outerWidth() : 0;
			var height = $elem ? $elem.outerHeight() : 0;
			var offset = this.__offset(anchor);
			var x = offset.left;
			var y = offset.top;
			var left = fixed ? x - docLeft : x;
			var top = fixed ? y - docTop : y;
			var minLeft = fixed ? 0 : docLeft;
			var minTop = fixed ? 0 : docTop;
			var maxLeft = minLeft + winWidth - popupWidth;
			var maxTop = minTop + winHeight - popupHeight;
			var css = {};
			var align = this.align.split(' ');
			var className = this.className + '-';
			var reverse = {
				top: 'bottom',
				bottom: 'top',
				left: 'right',
				right: 'left'
			};
			var name = {
				top: 'top',
				bottom: 'top',
				left: 'left',
				right: 'left'
			};
			var temp = [{
				top: top - popupHeight,
				bottom: top + height,
				left: left - popupWidth,
				right: left + width
			}, {
				top: top,
				bottom: top - popupHeight + height,
				left: left,
				right: left - popupWidth + width
			}];
			var center = {
				left: left + width / 2 - popupWidth / 2,
				top: top + height / 2 - popupHeight / 2
			};
			var range = {
				left: [minLeft, maxLeft],
				top: [minTop, maxTop]
			};
			$.each(align, function (i, val) {
				// Beyond the right or bottom edge: use the left or top
				if(temp[i][val] > range[name[val]][1]) {
					val = align[i] = reverse[val];
				}
				// Beyond the left or tp[ edge: use the right or bottom
				if(temp[i][val] < range[name[val]][0]) {
					align[i] = reverse[val];
				}
			});
			if(!align[1]) { // only one param
				name[align[1]] = name[align[0]] === 'left' ? 'top' : 'left';
				temp[1][align[1]] = center[name[align[1]]];
			}
			className += align.join('-') + ' ' + this.className + '-follow';
			that.__followSkin = className;
			if($elem) {
				popup.addClass(className);
			}
			css[name[align[0]]] = parseInt(temp[0][align[0]]);
			css[name[align[1]]] = parseInt(temp[1][align[1]]);
			popup.css(css);
		},
		// Get the position that element relative to the page (including the elements of iframe)
		// does not support more than two iframe
		__offset: function (anchor) {
			var isNode = anchor.parentNode;
			var offset = isNode ? $(anchor).offset() : {
				left: anchor.pageX,
				top: anchor.pageY
			};
			anchor = isNode ? anchor : anchor.target;
			var ownerDocument = anchor.ownerDocument;
			var defaultView = ownerDocument.defaultView || ownerDocument.parentWindow;
			if(defaultView == window) { // jshint ignore:line, for IE <= 8 can only use two equal
				return offset;
			}
			// {Element: Ifarme}
			var frameElement = defaultView.frameElement;
			var $ownerDocument = $(ownerDocument);
			var docLeft = $ownerDocument.scrollLeft();
			var docTop = $ownerDocument.scrollTop();
			var frameOffset = $(frameElement).offset();
			var frameLeft = frameOffset.left;
			var frameTop = frameOffset.top;
			return {
				left: offset.left + frameLeft - docLeft,
				top: offset.top + frameTop - docTop
			};
		}
	});
	/* current top z-index */
	Popup.zIndex = 1024;
	/* instance of the top layer */
	Popup.current = null;

	var Dialog = function (options, ok, cancel) {
		options = options || {};
		if(typeof options === 'string' || options.nodeType === 1) {
			options = {
				content: options,
				fixed: !_isMobile
			};
		}
		options = $.extend(true, {}, Dialog.defaults, options);
		var id = options.id = options.id || _expando + _count;
		var api = Dialog.get(id);
		if(api) {
			return api.focus();
		}
		if(!_isFixed) {
			// disable fixed for mobile device
			options.fixed = false;
		}
		if(options.quickClose) {
			options.modal = true;
			options.backdropOpacity = 0;
		}
		if(!$.isArray(options.button)) {
			options.button = [];
		}
		if(cancel !== undefined) {
			options.cancel = cancel;
		}
		if(options.cancel) {
			options.button.push({
				id: 'cancel',
				value: options.cancelValue,
				callback: options.cancel,
				display: options.cancelDisplay
			});
		}
		if(ok !== undefined) {
			options.ok = ok;
		}
		if(options.ok) {
			options.button.push({
				id: 'ok',
				value: options.okValue,
				callback: options.ok,
				autofocus: true
			});
		}
		return Dialog.list[id] = new Dialog.create(options);
	};
	var prototype = Dialog.prototype = new Popup();
	Dialog.create = function (options) {
		var that = this;
		$.extend(this, new Popup());
		var $popup = $(this.node).html(options.innerHTML);
		var $backdrop = $(this.backdrop);
		this.options = options;
		this._popup = $popup;
		$.each(options, function (name, value) {
			if(typeof that[name] === 'function') {
				that[name](value);
			} else {
				that[name] = value;
			}
		});
		// update zIndex
		if(options.zIndex) {
			Popup.zIndex = options.zIndex;
		}
		// set ARIA info
		$popup.attr({
			'aria-labelledby': this._$('title').attr('id', 'title:' + this.id).attr('id'),
			'aria-describedby': this._$('content').attr('id', 'content:' + this.id).attr('id')
		});
		// close button
		this._$('close').css('display', this.cancel === false ? 'none' : '').attr('title', this.cancelValue).on('click', function (event) {
			that._trigger('cancel');
			event.preventDefault();
		});
		// apply visual parameters
		this._$('dialog').addClass(this.skin);
		this._$('body').css('padding', this.padding);
		// close when click the background
		if(options.quickClose) {
			$backdrop.on('onmousedown' in document ? 'mousedown' : 'click', function () {
				that._trigger('cancel');
				return false;
			});
		}
		// set mask
		this.addEventListener('show', function () {
			$backdrop.css({
				opacity: 0,
				background: options.backdropBackground
			}).animate({
				opacity: options.backdropOpacity
			}, 150);
		});
		// ESC key close dialog
		this._esc = function (event) {
			var target = event.target;
			var nodeName = target.nodeName;
			var rinput = /^input|textarea$/i;
			var isTop = Popup.current === that;
			var keyCode = event.keyCode;
			// Avoid ESC close dialog in the input state
			if(!isTop || rinput.test(nodeName) && target.type !== 'button') {
				return;
			}
			if(keyCode === 27) {
				that._trigger('cancel');
			}
		};
		$(document).on('keydown', this._esc);
		this.addEventListener('remove', function () {
			$(document).off('keydown', this._esc);
			delete Dialog.list[this.id];
		});
		_count++;
		Dialog.oncreate(this);
		return this;
	};
	Dialog.create.prototype = prototype;
	$.extend(prototype, {
		/* Dialog.prototype.show */
		/* Dialog.prototype.showModal */
		/* Dialog.prototype.close */
		/* Dialog.prototype.remove */
		/* Dialog.prototype.reset */
		/* Dialog.prototype.focus */
		/* Dialog.prototype.blur */
		/* Dialog.prototype.addEventListener */
		/* Dialog.prototype.removeEventListener */
		/* Dialog.prototype.onshow */
		/* Dialog.prototype.onclose */
		/* Dialog.prototype.onbeforeremove */
		/* Dialog.prototype.onremove */
		/* Dialog.prototype.onreset */
		/* Dialog.prototype.onfocus */
		/* Dialog.prototype.onblur */

		content: function (html) {
			var $content = this._$('content');
			if(typeof html === 'object') { // HTMLElement
				html = $(html);
				$content.empty('').append(html.show());
				this.addEventListener('remove', function () {
					$('body').append(html.hide());
				});
			}
			else { // String
				$content.html(html);
			}
			return this.reset();
		},
		title: function (text) {
			this._$('title').text(text);
			this._$('header')[text ? 'show' : 'hide']();
			return this;
		},
		width: function (value) {
			this._$('content').css('width', value);
			return this.reset();
		},
		height: function (value) {
			this._$('content').css('height', value);
			return this.reset();
		},
		/* set button. Options: value, callback, autofocus, disabled */
		button: function (args) {
			args = args || [];
			var that = this;
			var html = '';
			var number = 0;
			this.callbacks = {};
			if(typeof args === 'string') {
				html = args;
				number++;
			}
			else {
				$.each(args, function (i, val) {
					var id = val.id = val.id || val.value;
					var style = '';
					that.callbacks[id] = val.callback;
					if(val.display === false) {
						style = ' style="display:none"';
					}
					else {
						number++;
					}

					// val.state: info, success, warning, danger
					var cls = ' class="btn' + 
						(val.state ? ' btn-' + val.state : '') +
						(val.autofocus ? ' btn-info ' + uiPrefix + '-dialog-autofocus' : '') + '"';

					html += '<button' + 
						' type="button"' + 
						' i-id="' + id + '"' + style + (val.disabled ? ' disabled' : '') + 
						cls + (val.autofocus ? ' autofocus' : '') + '>' + 
						val.value + '</button>';
					that._$('button').on('click', '[i-id="' + id + '"]', function (event) {
						var $this = $(this);
						if(!$this.attr('disabled')) { // IE BUG
							that._trigger(id);
						}
						event.preventDefault();
					});
				});
			}
			this._$('button').html(html);
			this._$('footer')[number ? 'show' : 'hide']();
			return this;
		},
		statusbar: function (html) {
			this._$('statusbar').html(html)[html ? 'show' : 'hide']();
			return this;
		},
		_$: function (i) {
			return this._popup.find('[i=' + i + ']');
		},
		/* trigger callback function */
		_trigger: function (id) {
			var fn = this.callbacks[id];
			return typeof fn !== 'function' || fn.call(this) !== false ? this.close().remove() : this;
		}
	});
	Dialog.oncreate = function (api) {
		var options = api.options;
		var originalOptions = options.original;
		var url = options.url;
		var oniframeload = options.oniframeload;
		var $iframe;

		if(url) {
			this.padding = options.padding = 0;
			$iframe = $('<iframe />');
			$iframe.attr({
				src: url,
				name: api.id,
				width: '100%',
				height: '100%',
				allowtransparency: 'yes',
				frameborder: 'no',
				scrolling: 'no'
			})
			.on('load', function () {
				var test;
				try {
					test = $iframe[0].contentWindow.frameElement;
				}
				catch (e) {}
				if(test) {
					if(!options.width) {
						api.width($iframe.contents().width());
					}
					if(!options.height) {
						api.height($iframe.contents().height());
					}
				}

				if(oniframeload) {
					oniframeload.call(api);
				}
			});

			api.addEventListener('beforeremove', function () {
				// important!, for IE6、7. Iframe will stay in memory after it be deleted
				$iframe.attr('src', 'about:blank').remove();
			}, false);

			api.content($iframe[0]);
			api.iframeNode = $iframe[0];
		}

		if(options.time) {
			api._timer = setTimeout(function(){
				api.close().remove();
			}, 1000 * options.time);
		}

		if(!(originalOptions instanceof Object)) {
			var un = function () {
				api.close().remove();
			};

			// find iframe
			for(var i = 0; i < frames.length; i ++) {
				try {
					if(originalOptions instanceof frames[i].Object) {
						/* close dialog when iframe is refresh
						 To prevent the object being forcing recovered, Cause IE error: "can't execute code that has been released Script"*/
						$(frames[i]).one('unload', un);
						break;
					}
				}
				catch (e) {}
			}
		}

		// drag support
		$(api.node).on(DragEvent.types.start, '[i=title]', function (event) {
			if(!api.follow) {
				api.focus();
				DragEvent.create(api.node, event);
			}
		});
	};
	Dialog.getCurrent = function () {
		return Popup.current;
	};
	Dialog.get = function (id) {
		if(id && id.frameElement) {
			var iframe = id.frameElement;
			var list = Dialog.list;
			var api;
			for(var i in list) {
				api = list[i];
				if(api.node.getElementsByTagName('iframe')[0] === iframe) {
					return api;
				}
			}
		}
		else if(id) {
			return Dialog.list[id];
		}
	};
	Dialog.list = {};
	Dialog.defaults = {
		/*align: 'bottom left',
		fixed: false,
		zIndex: 1024, // prevent iframe z-index allways be 1024 */
		backdropBackground: '#000',
		backdropOpacity: 0.7,
		content: '<span class="' + uiPrefix + '-dialog-loading"><i class="icon-spinner icon-spin"></i></span>',
		title: '',
		statusbar: '',
		button: null,
		ok: null,
		cancel: null,
		okValue: 'ok',
		cancelValue: 'cancel',
		cancelDisplay: true,
		width: '',
		height: '',
		padding: '',
		skin: '',
		quickClose: false,
		time: null,
		// use table for the IE7 BUG, use i="***"
		innerHTML: '<div i="dialog" class="' + uiPrefix + '-dialog">' + 
			'<div class="' + uiPrefix + '-dialog-arrow-a"></div>' + 
			'<div class="' + uiPrefix + '-dialog-arrow-b"></div>' + 
			'<table class="' + uiPrefix + '-dialog-grid">' + 
				'<tr>' + 
					'<td i="header" class="' + uiPrefix + '-dialog-header">' + 
						'<span i="close" class="' + uiPrefix + '-dialog-close close"></span>' + 
						'<div i="title" class="' + uiPrefix + '-dialog-title"></div>' + 
					'</td>' + 
				'</tr>' + 
				'<tr>' + 
					'<td i="body" class="' + uiPrefix + '-dialog-body">' + 
						'<div i="content" class="' + uiPrefix + '-dialog-content"></div>' + 
					'</td>' + 
				'</tr>' + 
				'<tr>' + 
					'<td i="footer" class="' + uiPrefix + '-dialog-footer">' + 
						'<div i="statusbar" class="' + uiPrefix + '-dialog-statusbar"></div>' + 
						'<div i="button" class="' + uiPrefix + '-dialog-button"></div>' + 
					'</td>' + 
				'</tr>' + 
			'</table>' + 
		'</div>'
	};

	FEA.dialog = Dialog;

	// Data API
	$(document).ready(function () {
		$('[data-dialog]').each(function () {
			var element = $(this),
				defaults = {
					modal: true,
					title: element.text(),
					target: (element.is('a') ? element.attr('href') : false)
				},
				options = $.extend({}, defaults, FEA.utils.options(element.data('dialog'))),
				id;

			id = ('#' === options.target.substring(0, 1)) ? options.target.substring(1) : options.target;
			options.content = document.getElementById(id);
			$('#' + id).hide();

			element.on('click', function (e) {
				var modal = options.modal,
					d = FEA.dialog(options);

				modal ? d.showModal() : d.show();
				e.preventDefault();
			});
		});
	});

}(jQuery, jQuery.FEA));

(function ($, FEA) {
	'use strict';

	// Class
	var Gotop = function (options) {
		var base = this;

		base.options = $.extend({}, Gotop.defaults, options);

		$('<div id="' + base.options.id + '" data-smooth-scroll="{duration:' + base.options.speed + '}">' + base.options.text + '</div>').appendTo('body');

		$('#' + base.options.id).css({
			'display' : 'none',
			'position' : 'fixed',
			'z-index' : base.options.zindex,
			'cursor' : 'pointer'
		}).addClass(base.options.cls);

		$(window).scroll(function () {
			if (base.options.animation === 'fade') {
				$($(window).scrollTop() > base.options.top ? $('#' + base.options.id).fadeIn(base.options.inSpeed) : $('#' + base.options.id).fadeOut(base.options.outSpeed));
			}
			else if (base.options.animation === 'slide') {
				$($(window).scrollTop() > base.options.top ? $('#' + base.options.id).slideDown(base.options.inSpeed) : $('#' + base.options.id).slideUp(base.options.outSpeed));
			}
			else {
				$($(window).scrollTop() > base.options.top ? $('#' + base.options.id).show(0) : $('#' + base.options.id).hide(0));
			}
		});
	};

	Gotop.defaults = {
		'id' : 'gotop',
		'text' : '<i class="icon-chevron-up"></i>',
		'zindex' : 2147483647,
		'cls' : 'gotop',
		'speed' : 1000,
		'top' : 300,
		'animation' : 'fade',
		'inSpeed' : 200,
		'outSpeed' : 200,
	};

	FEA.gotop = function (option) {
		return new Gotop(option);
	};
	FEA.gotop.Constructor = Gotop;

}(jQuery, jQuery.FEA));

(function ($, FEA) {
	'use strict';

	// Class
	var $html = $('html'),
		$body = $('body'),
		$doc = $(document),
		rtl = ($.FEA.langdirection === 'right'),
		scrollpos,
		Offcanvas = function (element, options) {
			this.element = element;
			this.options = $.extend( { 'target': this.element.is('a') ? this.element.attr('href') : false }, options);

			Offcanvas.show(this.options.target);

			$doc.on('keydown.offcanvas.FEA', function (e) {
				if(e.keyCode === 27) { // ESC
					Offcanvas.hide();
				}
			});
		};

	Offcanvas.show = function (target) {
		var $target = $(target);

		if(!$target.length) {
			return;
		}

		var rtl = ($.FEA.langdirection === 'right'),
			bar = $target.find('.offcanvas-bar:first'),
			dir = (bar.hasClass('offcanvas-bar-flip') ? -1 : 1) * (rtl ? -1 : 1),
			scrollbarwidth =  window.innerWidth - $body.width();

		scrollpos = { x: window.scrollX, y: window.scrollY };
		$target.addClass('active').show();

		$body.css({'width': window.innerWidth - scrollbarwidth, 'height': window.innerHeight}).addClass('offcanvas-page');
		$body.css((rtl ? 'margin-right' : 'margin-left'), (rtl ? -1 : 1) * (bar.outerWidth() * dir)).width(); // .width() - force redraw
		$html.css('margin-top', scrollpos.y * -1);
		bar.addClass('offcanvas-bar-show').width();

		$target.off('.offcanvas').on('click.offcanvas swipeRight.offcanvas swipeLeft.offcanvas', function (e) {
			var target = $(e.target);
			if(!e.type.match(/swipe/)) {
				if(!target.hasClass('offcanvas-close')) {
					if(target.hasClass('offcanvas-bar')) {
						return;
					}
					if(target.parents('.offcanvas-bar:first').length) {
						return;
					}
				}
			}
			e.stopImmediatePropagation();
			Offcanvas.hide();
		});
	};

	Offcanvas.hide = function (force) {
		var panel = $('.offcanvas.active'),
			bar = panel.find('.offcanvas-bar:first');

		if(!panel.length) {
			return;
		}

		if($.FEA.support.transition && !force) {
			$html.one($.FEA.support.transition.end, function () {
				$body.removeClass('offcanvas-page').css({'width': '', 'height': '', 'margin-left': '', 'margin-right': ''});
				panel.removeClass('active').hide();
				$html.css('margin-top', '');
				window.scrollTo(scrollpos.x, scrollpos.y);
			}).css((rtl ? 'margin-right' : 'margin-left'), '');

			setTimeout(function () {
				bar.removeClass('offcanvas-bar-show');
			}, 50);

		}
		else {
			$body.removeClass('offcanvas-page').css({'width': '', 'height': '', 'margin-left': '', 'margin-right': ''});
			panel.removeClass('active').hide();
			bar.removeClass('offcanvas-bar-show');
			window.scrollTo(scrollpos.x, scrollpos.y);
		}
	};

	// Plugin
	var old = $.fn.offcanvas;
	$.fn.offcanvas = function (option) {
		return this.each(function () {
			var $this = $(this),
				data = $this.data('offcanvas'),
				options = typeof option === 'object' && option;

			if(!data) {
				$this.data('offcanvas', (data = new Offcanvas($this, options)));
			}
		});
	};
	// for common use
	$.fn.offcanvas.Constructor = Offcanvas;
	// no conflict
	$.fn.offcanvas.noConflict = function () {
		$.fn.offcanvas = old;
		return this;
	};

	// Data API
	$(document).on('click.offcanvas.FEA', '[data-offcanvas]', function (e) {
		e.preventDefault();
		var element = $(this);
		new Offcanvas(element, FEA.utils.options(element.data('offcanvas')));
	});
}(jQuery, jQuery.FEA));

(function ($, FEA) {
	'use strict';

	// Class
	var Waterfall = function (element, options) {
		var base = this;

		base.element = element;
		base.options = $.extend({}, Waterfall.defaults, options);
		base.items = base.element.find(base.options.item);
		base.columnWidth = (base.element.width() + base.options.margin) / base.options.column;
		base.itemWidth = base.columnWidth - base.options.margin;
		base.pos = [];

		base.init();
		base.addItems(base.items);
		// Calculate the maximum top value assigned to the peripheral div
		base.calcHeight();

		if(base.options.fill){
			base.addFooter(base.element.height());
		}
	};

	Waterfall.prototype.init = function() {
		var base = this, i = 0;

		for(; i < base.options.column; i++) {
			base.pos.push([i * base.columnWidth, 0]);
		}
		base.element.css({'position': 'relative'});
	};
	Waterfall.prototype.addItems = function(items) {
		var base = this;

		items.css({'position': 'absolute', 'margin': 0, 'width': base.itemWidth}).each(function() {
			var that = $(this),
			_temp = 0,
			_height = that.outerHeight() + base.options.margin;

			for(var j = 0; j < base.options.column; j++) {
				if(base.pos[j][1] < base.pos[_temp][1]){
					// index of the minimum top value
					_temp = j;
				}
			}

			that.css({'top': base.pos[_temp][1]})
				.css($.FEA.langdirection, base.pos[_temp][0]);

			// update the top value
			base.pos[_temp][1] += _height;
		});
	};
	Waterfall.prototype.calcHeight = function() {
		var base = this, i = 0, tops = [];
		for(; i < base.options.column; i++) {
			tops.push(base.pos[i][1]);
		}
		tops.sort(function(a, b) {
			return a - b;
		});
		base.element.height(tops[base.options.column - 1]);
	};
	Waterfall.prototype.addFooter = function(max) {
		var base = this,
			footer = $('<div />').css({'position': 'absolute', 'margin': 0, 'padding': 0, 'width': base.itemWidth}).addClass(base.options.footerCls);
		for(var i = 0; i < base.options.column; i++) {
			if(max !== base.pos[i][1]) {
				var _height = max - base.pos[i][1] - base.options.margin,
					clone = footer.clone().attr('waterfall-footer', i).css($.FEA.langdirection, base.pos[i][0]).css({'top': base.pos[i][1], height: _height});
				base.element.append(clone);
			}
		}
	};
	Waterfall.defaults = {
		item: '>*',
		column: 4,
		margin: 16,
		fill: false,
		footerCls: 'placeholder'
	};

	// Plugin
	var old = $.fn.waterfall;
	$.fn.waterfall = function (option) {
		return this.each(function () {
			var $this = $(this),
				data = $this.data('waterfall'),
				options = typeof option === 'object' && option;

			if(!data) {
				$this.data('waterfall', (data = new Waterfall($this, options)));
			}
		});
	};
	// for common use
	$.fn.waterfall.Constructor = Waterfall;
	// no conflict
	$.fn.waterfall.noConflict = function () {
		$.fn.waterfall = old;
		return this;
	};

	// Data API
	$(document).ready(function () {
		$('[data-waterfall]').each(function () {
			var element = $(this);
			new Waterfall(element, FEA.utils.options(element.data('waterfall')));
		});
	});
	$(window).resize(function() {
		$('[waterfall-footer]').remove();
		$('[data-waterfall]').each(function () {
			var element = $(this);
			new Waterfall(element, FEA.utils.options(element.data('waterfall')));
		});
	});
}(jQuery, jQuery.FEA));
