<?php
/*require_once 'Zend/Loader.php';
require_once 'Zend/Uri.php';
require_once 'Zend/Http/Client/Adapter/Interface.php';
require_once 'Zend/Http/Response.php';
require_once 'Zend/Http/Response/Stream.php';*/

class Zend_Http_Client
{
    const GET     = 'GET';
    const POST    = 'POST';
    const PUT     = 'PUT';
    const HEAD    = 'HEAD';
    const DELETE  = 'DELETE';
    const TRACE   = 'TRACE';
    const OPTIONS = 'OPTIONS';
    const CONNECT = 'CONNECT';
    const MERGE   = 'MERGE';
    const AUTH_BASIC = 'basic';
    const HTTP_1 = '1.1';
    const HTTP_0 = '1.0';
    const CONTENT_TYPE   = 'Content-Type';
    const CONTENT_LENGTH = 'Content-Length';
    const ENC_URLENCODED = 'application/x-www-form-urlencoded';
    const ENC_FORMDATA   = 'multipart/form-data';
    const VTYPE_SCALAR  = 'SCALAR';
    const VTYPE_FILE    = 'FILE';
    
    protected $config = array(
        'maxredirects'    => 5,
        'strictredirects' => false,
        'useragent'       => 'Zend_Http_Client',
        'timeout'         => 10,
        'adapter'         => 'Zend_Http_Client_Adapter_Socket',
        'httpversion'     => self::HTTP_1,
        'keepalive'       => false,
        'storeresponse'   => true,
        'strict'          => true,
        'output_stream'   => false,
        'encodecookies'   => true,
        'rfc3986_strict'  => false
    );

    protected $adapter = null;
    protected $uri = null;
    protected $headers = array();
    protected $method = self::GET;
    protected $paramsGet = array();
    protected $paramsPost = array();
    protected $enctype = null;
    protected $raw_post_data = null;
    protected $auth;
    protected $files = array();
    protected $body_field_order = array();
    protected $cookiejar = null;
    protected $last_request = null;
    protected $last_response = null;
    protected $redirectCounter = 0;
    protected $_unmaskStatus = false;
    protected $_queryBracketsEscaped = true;

    static protected $_fileInfoDb = null;

    public function __construct($uri = null, $config = null)
    {
        if ($uri !== null) {
            $this->setUri($uri);
        }
        if ($config !== null) {
            $this->setConfig($config);
        }

        $this->_queryBracketsEscaped = version_compare(phpversion(), '5.1.3', '>=');
    }

    public function setUri($uri)
    {
        if ($uri instanceof Zend_Uri_Http) {
            $uri = clone $uri;
        } elseif (is_string($uri)) {
            $uri = Zend_Uri::factory($uri);
        }

        if (!$uri instanceof Zend_Uri_Http) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Passed parameter is not a valid HTTP URI.');
        }

        if ($uri->getUsername() && $uri->getPassword()) {
            $this->setAuth($uri->getUsername(), $uri->getPassword());
        }

        if (! $uri->getPort()) {
            $uri->setPort(($uri->getScheme() == 'https' ? 443 : 80));
        }

        $this->uri = $uri;

        return $this;
    }

    public function getUri($as_string = false)
    {
        if ($as_string && $this->uri instanceof Zend_Uri_Http) {
            return $this->uri->__toString();
        } else {
            return $this->uri;
        }
    }

    public function setConfig($config = array())
    {
        if ($config instanceof Zend_Config) {
            $config = $config->toArray();

        } elseif (! is_array($config)) {
            //require_once 'Zend/Http/Client/Exception.php';
            throw new Zend_Exception('Array or Zend_Config object expected, got ' . gettype($config));
        }

        foreach ($config as $k => $v) {
            $this->config[strtolower($k)] = $v;
        }

        if ($this->adapter instanceof Zend_Http_Client_Adapter_Interface) {
            $this->adapter->setConfig($config);
        }

        return $this;
    }

    public function setMethod($method = self::GET)
    {
        if (! preg_match('/^[^\x00-\x1f\x7f-\xff\(\)<>@,;:\\\\"\/\[\]\?={}\s]+$/', $method)) {
            //require_once 'Zend/Http/Client/Exception.php';
            throw new Zend_Exception("'{$method}' is not a valid HTTP request method.");
        }

        if (($method == self::POST || $method == self::PUT || $method == self::DELETE) && $this->enctype === null) {
            $this->setEncType(self::ENC_URLENCODED);
        }

        $this->method = $method;

        return $this;
    }

    public function setHeaders($name, $value = null)
    {
        if (is_array($name)) {
            foreach ($name as $k => $v) {
                if (is_string($k)) {
                    $this->setHeaders($k, $v);
                } else {
                    $this->setHeaders($v, null);
                }
            }
        } else {
            if ($value === null && (strpos($name, ':') > 0)) {
                list($name, $value) = explode(':', $name, 2);
            }

            if ($this->config['strict'] && (! preg_match('/^[a-zA-Z0-9-]+$/', $name))) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("{$name} is not a valid HTTP header name");
            }

            $normalized_name = strtolower($name);

            if ($value === null || $value === false) {
                unset($this->headers[$normalized_name]);
            } else {
                if (is_string($value)) {
                    $value = trim($value);
                }
                $this->headers[$normalized_name] = array($name, $value);
            }
        }

        return $this;
    }

    public function getHeader($key)
    {
        $key = strtolower($key);
        if (isset($this->headers[$key])) {
            return $this->headers[$key][1];
        } else {
            return null;
        }
    }

    public function setParameterGet($name, $value = null)
    {
        if (is_array($name)) {
            foreach ($name as $k => $v)
                $this->_setParameter('GET', $k, $v);
        } else {
            $this->_setParameter('GET', $name, $value);
        }

        return $this;
    }

    public function setParameterPost($name, $value = null)
    {
        if (is_array($name)) {
            foreach ($name as $k => $v)
                $this->_setParameter('POST', $k, $v);
        } else {
            $this->_setParameter('POST', $name, $value);
        }

        return $this;
    }

    protected function _setParameter($type, $name, $value)
    {
        $parray = array();
        $type = strtolower($type);
        switch ($type) {
            case 'get':
                $parray = &$this->paramsGet;
                break;
            case 'post':
                $parray = &$this->paramsPost;
                if ( $value === null ) {
                    if (isset($this->body_field_order[$name]))
                        unset($this->body_field_order[$name]);
                } else {
                    $this->body_field_order[$name] = self::VTYPE_SCALAR;
                }
                break;
        }

        if ($value === null) {
            if (isset($parray[$name])) unset($parray[$name]);
        } else {
            $parray[$name] = $value;
        }
    }

    public function getRedirectionsCount()
    {
        return $this->redirectCounter;
    }

    public function setAuth($user, $password = '', $type = self::AUTH_BASIC)
    {
        if ($user === false || $user === null) {
            $this->auth = null;

            if ($this->uri instanceof Zend_Uri_Http) {
                $this->getUri()->setUsername('');
                $this->getUri()->setPassword('');
            }
        } else {
            if (! defined('self::AUTH_' . strtoupper($type))) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("Invalid or not supported authentication type: '$type'");
            }

            $this->auth = array(
                'user' => (string) $user,
                'password' => (string) $password,
                'type' => $type
            );
        }

        return $this;
    }

    public function setCookieJar($cookiejar = true)
    {
        Zend_Loader::loadClass('Zend_Http_CookieJar');

        if ($cookiejar instanceof Zend_Http_CookieJar) {
            $this->cookiejar = $cookiejar;
        } elseif ($cookiejar === true) {
            $this->cookiejar = new Zend_Http_CookieJar();
        } elseif (! $cookiejar) {
            $this->cookiejar = null;
        } else {
            //require_once 'Zend/Http/Client/Exception.php';
            throw new Zend_Exception('Invalid parameter type passed as CookieJar');
        }

        return $this;
    }

    public function getCookieJar()
    {
        return $this->cookiejar;
    }

    public function setCookie($cookie, $value = null)
    {
        Zend_Loader::loadClass('Zend_Http_Cookie');

        if (is_array($cookie)) {
            foreach ($cookie as $c => $v) {
                if (is_string($c)) {
                    $this->setCookie($c, $v);
                } else {
                    $this->setCookie($v);
                }
            }

            return $this;
        }

        if ($value !== null && $this->config['encodecookies']) {
            $value = urlencode($value);
        }

        if (isset($this->cookiejar)) {
            if ($cookie instanceof Zend_Http_Cookie) {
                $this->cookiejar->addCookie($cookie);
            } elseif (is_string($cookie) && $value !== null) {
                $cookie = Zend_Http_Cookie::fromString("{$cookie}={$value}",
                                                       $this->uri,
                                                       $this->config['encodecookies']);
                $this->cookiejar->addCookie($cookie);
            }
        } else {
            if ($cookie instanceof Zend_Http_Cookie) {
                $name = $cookie->getName();
                $value = $cookie->getValue();
                $cookie = $name;
            }

            if (preg_match("/[=,; \t\r\n\013\014]/", $cookie)) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("Cookie name cannot contain these characters: =,; \t\r\n\013\014 ({$cookie})");
            }

            $value = addslashes($value);

            if (! isset($this->headers['cookie'])) {
                $this->headers['cookie'] = array('Cookie', '');
            }
            $this->headers['cookie'][1] .= $cookie . '=' . $value . '; ';
        }

        return $this;
    }

    public function setFileUpload($filename, $formname, $data = null, $ctype = null)
    {
        if ($data === null) {
            if (($data = @file_get_contents($filename)) === false) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("Unable to read file '{$filename}' for upload");
            }

            if (! $ctype) {
                $ctype = $this->_detectFileMimeType($filename);
            }
        }

        $this->setEncType(self::ENC_FORMDATA);

        $this->files[] = array(
            'formname' => $formname,
            'filename' => basename($filename),
            'ctype'    => $ctype,
            'data'     => $data
        );
        
        $this->body_field_order[$formname] = self::VTYPE_FILE;

        return $this;
    }

    public function setEncType($enctype = self::ENC_URLENCODED)
    {
        $this->enctype = $enctype;

        return $this;
    }

    public function setRawData($data, $enctype = null)
    {
        $this->raw_post_data = $data;
        $this->setEncType($enctype);
        if (is_resource($data)) {
            $stat = @fstat($data);
            if($stat) {
                $this->setHeaders(self::CONTENT_LENGTH, $stat['size']);
            }
        }
        return $this;
    }

    public function setUnmaskStatus($status = true)
    {
        $this->_unmaskStatus = (BOOL)$status;
        return $this;
    }

    public function getUnmaskStatus()
    {
        return $this->_unmaskStatus;
    }

    public function resetParameters($clearAll = false)
    {
        $this->paramsGet     = array();
        $this->paramsPost    = array();
        $this->files         = array();
        $this->raw_post_data = null;
        $this->enctype       = null;
        
        if($clearAll) {
            $this->headers = array();
            $this->last_request = null;
            $this->last_response = null;
        } else {
            if (isset($this->headers[strtolower(self::CONTENT_TYPE)])) {
                unset($this->headers[strtolower(self::CONTENT_TYPE)]);
            }
            if (isset($this->headers[strtolower(self::CONTENT_LENGTH)])) {
                unset($this->headers[strtolower(self::CONTENT_LENGTH)]);
            }
        }

        return $this;
    }

    public function getLastRequest()
    {
        return $this->last_request;
    }

    public function getLastResponse()
    {
        return $this->last_response;
    }

    public function setAdapter($adapter)
    {
        if (is_string($adapter)) {
            try {
                Zend_Loader::loadClass($adapter);
            } catch (Zend_Exception $e) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("Unable to load adapter '$adapter': {$e->getMessage()}", 0, $e);
            }

            $adapter = new $adapter;
        }

        if (! $adapter instanceof Zend_Http_Client_Adapter_Interface) {
            //require_once 'Zend/Http/Client/Exception.php';
            throw new Zend_Exception('Passed adapter is not a HTTP connection adapter');
        }

        $this->adapter = $adapter;
        $config = $this->config;
        unset($config['adapter']);
        $this->adapter->setConfig($config);
    }

    public function getAdapter()
    {
         if (null === $this->adapter) {
            $this->setAdapter($this->config['adapter']);
        }

        return $this->adapter;
    }

    public function setStream($streamfile = true)
    {
        $this->setConfig(array("output_stream" => $streamfile));
        return $this;
    }

    public function getStream()
    {
        return $this->config["output_stream"];
    }

    protected function _openTempStream()
    {
        $this->_stream_name = $this->config['output_stream'];
        if(!is_string($this->_stream_name)) {
            $this->_stream_name = tempnam(isset($this->config['stream_tmp_dir'])?$this->config['stream_tmp_dir']:sys_get_temp_dir(),
                 'Zend_Http_Client');
        }

        if (false === ($fp = @fopen($this->_stream_name, "w+b"))) {
                if ($this->adapter instanceof Zend_Http_Client_Adapter_Interface) {
                    $this->adapter->close();
                }
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("Could not open temp file {$this->_stream_name}");
        }

        return $fp;
    }

    public function request($method = null)
    {
        if (! $this->uri instanceof Zend_Uri_Http) {
            //require_once 'Zend/Http/Client/Exception.php';
            throw new Zend_Exception('No valid URI has been passed to the client');
        }

        if ($method) {
            $this->setMethod($method);
        }
        $this->redirectCounter = 0;
        $response = null;

        if ($this->adapter == null) {
            $this->setAdapter($this->config['adapter']);
        }

        do {
            $uri = clone $this->uri;
            if (! empty($this->paramsGet)) {
                $query = $uri->getQuery();
                   if (! empty($query)) {
                       $query .= '&';
                   }
                $query .= http_build_query($this->paramsGet, null, '&');
                if ($this->config['rfc3986_strict']) {
                    $query = str_replace('+', '%20', $query);
                }

                if ($this->getUnmaskStatus()) {
                    if ($this->_queryBracketsEscaped) {
                        $query = preg_replace('/%5B(?:[0-9]|[1-9][0-9]+)%5D=/', '=', $query);
                    } else {
                        $query = preg_replace('/\\[(?:[0-9]|[1-9][0-9]+)\\]=/', '=', $query);
                    }
                }

                $uri->setQuery($query);
            }

            $body = $this->_prepareBody();
            $headers = $this->_prepareHeaders();

            if(is_resource($body) && !($this->adapter instanceof Zend_Http_Client_Adapter_Stream)) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception('Adapter does not support streaming');
            }

            $this->adapter->connect($uri->getHost(), $uri->getPort(),
                ($uri->getScheme() == 'https' ? true : false));

            if($this->config['output_stream']) {
                if($this->adapter instanceof Zend_Http_Client_Adapter_Stream) {
                    $stream = $this->_openTempStream();
                    $this->adapter->setOutputStream($stream);
                } else {
                    //require_once 'Zend/Http/Client/Exception.php';
                    throw new Zend_Exception('Adapter does not support streaming');
                }
            }

            $this->last_request = $this->adapter->write($this->method,
                $uri, $this->config['httpversion'], $headers, $body);

            $response = $this->adapter->read();
            if (! $response) {
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception('Unable to read response, or response is empty');
            }

            if($this->config['output_stream']) {
                $streamMetaData = stream_get_meta_data($stream);
                if ($streamMetaData['seekable']) {
                    rewind($stream);
                }
                $this->adapter->setOutputStream(null);
                $response = Zend_Http_Response_Stream::fromStream($response, $stream);
                $response->setStreamName($this->_stream_name);
                if(!is_string($this->config['output_stream'])) {
                    $response->setCleanup(true);
                }
            } else {
                $response = Zend_Http_Response::fromString($response);
            }

            if ($this->config['storeresponse']) {
                $this->last_response = $response;
            }

            if (isset($this->cookiejar)) {
                $this->cookiejar->addCookiesFromResponse($response, $uri, $this->config['encodecookies']);
            }

            if ($response->isRedirect() && ($location = $response->getHeader('location'))) {
                $location = trim($location);
                if ($response->getStatus() == 303 ||
                   ((! $this->config['strictredirects']) && ($response->getStatus() == 302 ||
                       $response->getStatus() == 301))) {

                    $this->resetParameters();
                    $this->setMethod(self::GET);
                }

                if (($scheme = substr($location, 0, 6)) && ($scheme == 'http:/' || $scheme == 'https:')) {
                    $this->setHeaders('host', null);
                    $this->setUri($location);

                } else {

                    if (strpos($location, '?') !== false) {
                        list($location, $query) = explode('?', $location, 2);
                    } else {
                        $query = '';
                    }
                    $this->uri->setQuery($query);

                    if(strpos($location, '/') === 0) {
                        $this->uri->setPath($location);
                    } else {
                        $path = $this->uri->getPath();
                        $path = rtrim(substr($path, 0, strrpos($path, '/')), "/");
                        $this->uri->setPath($path . '/' . $location);
                    }
                }
                ++$this->redirectCounter;

            } else {
                break;
            }

        } while ($this->redirectCounter < $this->config['maxredirects']);

        return $response;
    }

    protected function _prepareHeaders()
    {
        $headers = array();

        if (! isset($this->headers['host'])) {
            $host = $this->uri->getHost();

            if (! (($this->uri->getScheme() == 'http' && $this->uri->getPort() == 80) ||
                  ($this->uri->getScheme() == 'https' && $this->uri->getPort() == 443))) {
                $host .= ':' . $this->uri->getPort();
            }

            $headers[] = "Host: {$host}";
        }

        if (! isset($this->headers['connection'])) {
            if (! $this->config['keepalive']) {
                $headers[] = "Connection: close";
            }
        }
        
        if (! isset($this->headers['accept-encoding'])) {
            if (function_exists('gzinflate')) {
                $headers[] = 'Accept-encoding: gzip, deflate';
            } else {
                $headers[] = 'Accept-encoding: identity';
            }
        }

        if (($this->method == self::POST || $this->method == self::PUT) &&
           (! isset($this->headers[strtolower(self::CONTENT_TYPE)]) && isset($this->enctype))) {

            $headers[] = self::CONTENT_TYPE . ': ' . $this->enctype;
        }

        if (! isset($this->headers['user-agent']) && isset($this->config['useragent'])) {
            $headers[] = "User-Agent: {$this->config['useragent']}";
        }

        if (is_array($this->auth)) {
            $auth = self::encodeAuthHeader($this->auth['user'], $this->auth['password'], $this->auth['type']);
            $headers[] = "Authorization: {$auth}";
        }

        if (isset($this->cookiejar)) {
            $cookstr = $this->cookiejar->getMatchingCookies($this->uri,
                true, Zend_Http_CookieJar::COOKIE_STRING_CONCAT);

            if ($cookstr) {
                $headers[] = "Cookie: {$cookstr}";
            }
        }

        foreach ($this->headers as $header) {
            list($name, $value) = $header;
            if (is_array($value)) {
                $value = implode(', ', $value);
            }

            $headers[] = "$name: $value";
        }

        return $headers;
    }

    protected function _prepareBody()
    {
        if ($this->method == self::TRACE) {
            return '';
        }

        if (isset($this->raw_post_data) && is_resource($this->raw_post_data)) {
            return $this->raw_post_data;
        }
        if (function_exists('mb_internal_encoding') &&
           ((int) ini_get('mbstring.func_overload')) & 2) {

            $mbIntEnc = mb_internal_encoding();
            mb_internal_encoding('ASCII');
        }
        if (isset($this->raw_post_data)) {
            $this->setHeaders(self::CONTENT_LENGTH, strlen($this->raw_post_data));
            if (isset($mbIntEnc)) {
                mb_internal_encoding($mbIntEnc);
            }

            return $this->raw_post_data;
        }

        $body = '';

        if (count ($this->files) > 0) {
            $this->setEncType(self::ENC_FORMDATA);
        }

        if (count($this->paramsPost) > 0 || count($this->files) > 0) {
            switch($this->enctype) {
                case self::ENC_FORMDATA:
                    $boundary = '---ZENDHTTPCLIENT-' . md5(microtime());
                    $this->setHeaders(self::CONTENT_TYPE, self::ENC_FORMDATA . "; boundary={$boundary}");

                    foreach ($this->body_field_order as $fieldName=>$fieldType) {
                        switch ($fieldType) {
                            case self::VTYPE_FILE:
                                foreach ($this->files as $file) {
                                    if ($file['formname']===$fieldName) {
                                        $fhead = array(self::CONTENT_TYPE => $file['ctype']);
                                        $body .= self::encodeFormData($boundary, $file['formname'], $file['data'], $file['filename'], $fhead);
                                    }
                                }
                                break;
                            case self::VTYPE_SCALAR:
                                if (isset($this->paramsPost[$fieldName])) {
                                    if (is_array($this->paramsPost[$fieldName])) {
                                        $flattened = self::_flattenParametersArray($this->paramsPost[$fieldName], $fieldName);
                                        foreach ($flattened as $pp) {
                                            $body .= self::encodeFormData($boundary, $pp[0], $pp[1]);
                                        }
                                    } else {
                                        $body .= self::encodeFormData($boundary, $fieldName, $this->paramsPost[$fieldName]);
                                    }
                                }
                                break;
                        }
                    }

                    $body .= "--{$boundary}--\r\n";
                    break;

                case self::ENC_URLENCODED:
                    $this->setHeaders(self::CONTENT_TYPE, self::ENC_URLENCODED);
                    $body = http_build_query($this->paramsPost, '', '&');
                    break;

                default:
                    if (isset($mbIntEnc)) {
                        mb_internal_encoding($mbIntEnc);
                    }
                    //require_once 'Zend/Http/Client/Exception.php';
                    throw new Zend_Exception("Cannot handle content type '{$this->enctype}' automatically." .
                        " Please use Zend_Http_Client::setRawData to send this kind of content.");
                    break;
            }
        }
        if ($body || $this->method == self::POST || $this->method == self::PUT) {
            $this->setHeaders(self::CONTENT_LENGTH, strlen($body));
        }

        if (isset($mbIntEnc)) {
            mb_internal_encoding($mbIntEnc);
        }

        return $body;
    }

    protected function _getParametersRecursive($parray, $urlencode = false)
    {
        trigger_error("The " .  __METHOD__ . " method is deprecated and will be dropped in 2.0.",
            E_USER_NOTICE);

        if (! is_array($parray)) {
            return $parray;
        }
        $parameters = array();

        foreach ($parray as $name => $value) {
            if ($urlencode) {
                $name = urlencode($name);
            }

            if (is_array($value)) {
                $name .= ($urlencode ? '%5B%5D' : '[]');
                foreach ($value as $subval) {
                    if ($urlencode) {
                        $subval = urlencode($subval);
                    }
                    $parameters[] = array($name, $subval);
                }
            } else {
                if ($urlencode) {
                    $value = urlencode($value);
                }
                $parameters[] = array($name, $value);
            }
        }

        return $parameters;
    }

    protected function _detectFileMimeType($file)
    {
        $type = null;

        if (function_exists('finfo_open')) {
            if (self::$_fileInfoDb === null) {
                self::$_fileInfoDb = @finfo_open(FILEINFO_MIME);
            }

            if (self::$_fileInfoDb) {
                $type = finfo_file(self::$_fileInfoDb, $file);
            }

        } elseif (function_exists('mime_content_type')) {
            $type = mime_content_type($file);
        }

        if (! $type) {
            $type = 'application/octet-stream';
        }

        return $type;
    }

    public static function encodeFormData($boundary, $name, $value, $filename = null, $headers = array()) {
        $ret = "--{$boundary}\r\n" .
            'Content-Disposition: form-data; name="' . $name .'"';

        if ($filename) {
            $ret .= '; filename="' . $filename . '"';
        }
        $ret .= "\r\n";

        foreach ($headers as $hname => $hvalue) {
            $ret .= "{$hname}: {$hvalue}\r\n";
        }
        $ret .= "\r\n";

        $ret .= "{$value}\r\n";

        return $ret;
    }

    public static function encodeAuthHeader($user, $password, $type = self::AUTH_BASIC)
    {
        $authHeader = null;

        switch ($type) {
            case self::AUTH_BASIC:
                if (strpos($user, ':') !== false) {
                    //require_once 'Zend/Http/Client/Exception.php';
                    throw new Zend_Exception("The user name cannot contain ':' in 'Basic' HTTP authentication");
                }

                $authHeader = 'Basic ' . base64_encode($user . ':' . $password);
                break;
            default:
                //require_once 'Zend/Http/Client/Exception.php';
                throw new Zend_Exception("Not a supported HTTP authentication type: '$type'");
        }

        return $authHeader;
    }

    static protected function _flattenParametersArray($parray, $prefix = null)
    {
        if (! is_array($parray)) {
            return $parray;
        }

        $parameters = array();

        foreach($parray as $name => $value) {
            if ($prefix) {
                if (is_int($name)) {
                    $key = $prefix . '[]';
                } else {
                    $key = $prefix . "[$name]";
                }
            } else {
                $key = $name;
            }

            if (is_array($value)) {
                $parameters = array_merge($parameters, self::_flattenParametersArray($value, $key));

            } else {
                $parameters[] = array($key, $value);
            }
        }

        return $parameters;
    }

}