<?php
//require_once 'Zend/Uri/Http.php';

class Zend_Http_Cookie
{
    protected $name;
    protected $value;
    protected $expires;
    protected $domain;
    protected $path;
    protected $secure;
    protected $encodeValue;

    public function __construct($name, $value, $domain, $expires = null, $path = null, $secure = false)
    {
        if (preg_match("/[=,; \t\r\n\013\014]/", $name)) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception("Cookie name cannot contain these characters: =,; \\t\\r\\n\\013\\014 ({$name})");
        }

        if (! $this->name = (string) $name) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception('Cookies must have a name');
        }

        if (! $this->domain = (string) $domain) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception('Cookies must have a domain');
        }

        $this->value = (string) $value;
        $this->expires = ($expires === null ? null : (int) $expires);
        $this->path = ($path ? $path : '/');
        $this->secure = $secure;
    }

    public function getName()
    {
        return $this->name;
    }

    public function getValue()
    {
        return $this->value;
    }

    public function getDomain()
    {
        return $this->domain;
    }

    public function getPath()
    {
        return $this->path;
    }

    public function getExpiryTime()
    {
        return $this->expires;
    }

    public function isSecure()
    {
        return $this->secure;
    }

    public function isExpired($now = null)
    {
        if ($now === null) $now = time();
        if (is_int($this->expires) && $this->expires < $now) {
            return true;
        } else {
            return false;
        }
    }

    public function isSessionCookie()
    {
        return ($this->expires === null);
    }

    public function match($uri, $matchSessionCookies = true, $now = null)
    {
        if (is_string ($uri)) {
            $uri = Zend_Uri_Http::factory($uri);
        }

        if (! ($uri->valid() && ($uri->getScheme() == 'http' || $uri->getScheme() =='https'))) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception('Passed URI is not a valid HTTP or HTTPS URI');
        }

        if ($this->secure && $uri->getScheme() != 'https') return false;
        if ($this->isExpired($now)) return false;
        if ($this->isSessionCookie() && ! $matchSessionCookies) return false;

        if (! self::matchCookieDomain($this->getDomain(), $uri->getHost())) {
            return false;
        }

        if (! self::matchCookiePath($this->getPath(), $uri->getPath())) {
            return false;
        }

        return true;
    }

    public function __toString()
    {
        if ($this->encodeValue) {
            return $this->name . '=' . urlencode($this->value) . ';';
        }
        return $this->name . '=' . $this->value . ';';
    }

    public static function fromString($cookieStr, $refUri = null, $encodeValue = true)
    {
        if (is_string($refUri)) {
            $refUri = Zend_Uri_Http::factory($refUri);
        }

        $name    = '';
        $value   = '';
        $domain  = '';
        $path    = '';
        $expires = null;
        $secure  = false;
        $parts   = explode(';', $cookieStr);

        if (strpos($parts[0], '=') === false) return false;

        list($name, $value) = explode('=', trim(array_shift($parts)), 2);
        $name  = trim($name);
        if ($encodeValue) {
            $value = urldecode(trim($value));
        }

        if ($refUri instanceof Zend_Uri_Http) {
            $domain = $refUri->getHost();
            $path = $refUri->getPath();
            $path = substr($path, 0, strrpos($path, '/'));
        }

        foreach ($parts as $part) {
            $part = trim($part);
            if (strtolower($part) == 'secure') {
                $secure = true;
                continue;
            }

            $keyValue = explode('=', $part, 2);
            if (count($keyValue) == 2) {
                list($k, $v) = $keyValue;
                switch (strtolower($k))    {
                    case 'expires':
                        if(($expires = strtotime($v)) === false) {
                            //require_once 'Zend/Date.php';

                            $expireDate = new Zend_Date($v);
                            $expires = $expireDate->getTimestamp();
                        }
                        break;

                    case 'path':
                        $path = $v;
                        break;

                    case 'domain':
                        $domain = $v;
                        break;

                    default:
                        break;
                }
            }
        }

        if ($name !== '') {
            $ret = new self($name, $value, $domain, $expires, $path, $secure);
            $ret->encodeValue = ($encodeValue) ? true : false;
            return $ret;
        } else {
            return false;
        }
    }

    public static function matchCookieDomain($cookieDomain, $host)
    {
        if (! $cookieDomain) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception("\$cookieDomain is expected to be a cookie domain");
        }

        if (! $host) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception("\$host is expected to be a host name");
        }

        $cookieDomain = strtolower($cookieDomain);
        $host = strtolower($host);

        if ($cookieDomain[0] == '.') {
            $cookieDomain = substr($cookieDomain, 1);
        }

        return ($cookieDomain == $host ||
                preg_match('/\.' . preg_quote($cookieDomain) . '$/', $host));
    }

    public static function matchCookiePath($cookiePath, $path)
    {
        if (! $cookiePath) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception("\$cookiePath is expected to be a cookie path");
        }

        if (! $path) {
            //require_once 'Zend/Http/Exception.php';
            throw new Zend_Exception("\$path is expected to be a host name");
        }

        return (strpos($path, $cookiePath) === 0);
    }
}