<?php
/*require_once 'Zend/Uri.php';
require_once 'Zend/Validate/Hostname.php';*/

class Zend_Uri_Http extends Zend_Uri
{
    const CHAR_ALNUM    = 'A-Za-z0-9';
    const CHAR_MARK     = '-_.!~*\'()\[\]';
    const CHAR_RESERVED = ';\/?:@&=+$,';
    const CHAR_SEGMENT  = ':@&=+$,;';
    const CHAR_UNWISE   = '{}|\\\\^`';

    protected $_username = '';
    protected $_password = '';
    protected $_host = '';
    protected $_port = '';
    protected $_path = '';
    protected $_query = '';
    protected $_fragment = '';
    protected $_regex = array();

    protected function __construct($scheme, $schemeSpecific = '')
    {
        $this->_scheme = $scheme;
        $this->_regex['escaped']    = '%[[:xdigit:]]{2}';
        $this->_regex['unreserved'] = '[' . self::CHAR_ALNUM . self::CHAR_MARK . ']';
        $this->_regex['segment']    = '(?:' . $this->_regex['escaped'] . '|[' .
            self::CHAR_ALNUM . self::CHAR_MARK . self::CHAR_SEGMENT . '])*';
        
        $this->_regex['path']       = '(?:\/(?:' . $this->_regex['segment'] . ')?)+';
        $this->_regex['uric']       = '(?:' . $this->_regex['escaped'] . '|[' .
            self::CHAR_ALNUM . self::CHAR_MARK . self::CHAR_RESERVED .
            (self::$_config['allow_unwise'] ? self::CHAR_UNWISE : '') . '])';
        
        if (strlen($schemeSpecific) === 0) {
            return;
        }

        $this->_parseUri($schemeSpecific);

        if ($this->valid() === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Invalid URI supplied');
        }
    }

    public static function fromString($uri)
    {
        if (is_string($uri) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('$uri is not a string');
        }

        $uri            = explode(':', $uri, 2);
        $scheme         = strtolower($uri[0]);
        $schemeSpecific = isset($uri[1]) === true ? $uri[1] : '';

        if (in_array($scheme, array('http', 'https')) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Invalid scheme: '$scheme'");
        }

        $schemeHandler = new Zend_Uri_Http($scheme, $schemeSpecific);
        return $schemeHandler;
    }

    protected function _parseUri($schemeSpecific)
    {
        $pattern = '~^((//)([^/?#]*))([^?#]*)(\?([^#]*))?(#(.*))?$~';
        $status  = @preg_match($pattern, $schemeSpecific, $matches);
        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: scheme-specific decomposition failed');
        }

        if ($status === false) {
            return;
        }

        $this->_path     = isset($matches[4]) === true ? $matches[4] : '';
        $this->_query    = isset($matches[6]) === true ? $matches[6] : '';
        $this->_fragment = isset($matches[8]) === true ? $matches[8] : '';

        $combo   = isset($matches[3]) === true ? $matches[3] : '';
        $pattern = '~^(([^:@]*)(:([^@]*))?@)?((?(?=[[])[[][^]]+[]]|[^:]+))(:(.*))?$~';        
        $status  = @preg_match($pattern, $combo, $matches);
        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: authority decomposition failed');
        }

        $this->_username = isset($matches[2]) === true ? $matches[2] : '';
        $this->_password = isset($matches[4]) === true ? $matches[4] : '';
        $this->_host     = isset($matches[5]) === true 
                         ? preg_replace('~^\[([^]]+)\]$~', '\1', $matches[5])  // Strip wrapper [] from IPv6 literal
                         : '';
        $this->_port     = isset($matches[7]) === true ? $matches[7] : '';
    }

    public function getUri()
    {
        if ($this->valid() === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('One or more parts of the URI are invalid');
        }

        $password = strlen($this->_password) > 0 ? ":$this->_password" : '';
        $auth     = strlen($this->_username) > 0 ? "$this->_username$password@" : '';
        $port     = strlen($this->_port) > 0 ? ":$this->_port" : '';
        $query    = strlen($this->_query) > 0 ? "?$this->_query" : '';
        $fragment = strlen($this->_fragment) > 0 ? "#$this->_fragment" : '';

        return $this->_scheme
             . '://'
             . $auth
             . $this->_host
             . $port
             . $this->_path
             . $query
             . $fragment;
    }

    public function valid()
    {
        return $this->validateUsername()
           and $this->validatePassword()
           and $this->validateHost()
           and $this->validatePort()
           and $this->validatePath()
           and $this->validateQuery()
           and $this->validateFragment();
    }

    public function getUsername()
    {
        return strlen($this->_username) > 0 ? $this->_username : false;
    }

    public function validateUsername($username = null)
    {
        if ($username === null) {
            $username = $this->_username;
        }

        if (strlen($username) === 0) {
            return true;
        }

        $status = @preg_match('/^(?:' . $this->_regex['escaped'] . '|[' .
            self::CHAR_ALNUM . self::CHAR_MARK . ';:&=+$,' . '])+$/', $username);

        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: username validation failed');
        }

        return $status === 1;
    }

    public function setUsername($username)
    {
        if ($this->validateUsername($username) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Username \"$username\" is not a valid HTTP username");
        }

        $oldUsername     = $this->_username;
        $this->_username = $username;

        return $oldUsername;
    }

    public function getPassword()
    {
        return strlen($this->_password) > 0 ? $this->_password : false;
    }

    public function validatePassword($password = null)
    {
        if ($password === null) {
            $password = $this->_password;
        }

        if (strlen($password) === 0) {
            return true;
        }

        if (strlen($password) > 0 and strlen($this->_username) === 0) {
            return false;
        }

        $status = @preg_match('/^(?:' . $this->_regex['escaped'] . '|[' .
            self::CHAR_ALNUM . self::CHAR_MARK . ';:&=+$,' . '])+$/', $password);

        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: password validation failed.');
        }

        return $status == 1;
    }

    public function setPassword($password)
    {
        if ($this->validatePassword($password) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Password \"$password\" is not a valid HTTP password.");
        }

        $oldPassword     = $this->_password;
        $this->_password = $password;

        return $oldPassword;
    }

    public function getHost()
    {
        return strlen($this->_host) > 0 ? $this->_host : false;
    }

    public function validateHost($host = null)
    {
        if ($host === null) {
            $host = $this->_host;
        }

        if (strlen($host) === 0) {
            return false;
        }

        $validate = new Zend_Validate_Hostname(Zend_Validate_Hostname::ALLOW_ALL);

        return $validate->isValid($host);
    }

    public function setHost($host)
    {
        if ($this->validateHost($host) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Host \"$host\" is not a valid HTTP host");
        }

        $oldHost     = $this->_host;
        $this->_host = $host;

        return $oldHost;
    }

    public function getPort()
    {
        return strlen($this->_port) > 0 ? $this->_port : false;
    }

    public function validatePort($port = null)
    {
        if ($port === null) {
            $port = $this->_port;
        }

        if (strlen($port) === 0) {
            return true;
        }

        return ctype_digit((string) $port) and 1 <= $port and $port <= 65535;
    }

    public function setPort($port)
    {
        if ($this->validatePort($port) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Port \"$port\" is not a valid HTTP port.");
        }

        $oldPort     = $this->_port;
        $this->_port = $port;

        return $oldPort;
    }

    public function getPath()
    {
        return strlen($this->_path) > 0 ? $this->_path : '/';
    }

    public function validatePath($path = null)
    {
        if ($path === null) {
            $path = $this->_path;
        }

        if (strlen($path) === 0) {
            return true;
        }

        $pattern = '/^' . $this->_regex['path'] . '$/';
        $status  = @preg_match($pattern, $path);
        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: path validation failed');
        }

        return (boolean) $status;
    }

    public function setPath($path)
    {
        if ($this->validatePath($path) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Path \"$path\" is not a valid HTTP path");
        }

        $oldPath     = $this->_path;
        $this->_path = $path;

        return $oldPath;
    }

    public function getQuery()
    {
        return strlen($this->_query) > 0 ? $this->_query : false;
    }

    public function getQueryAsArray()
    {
        $query = $this->getQuery();
        $querryArray = array();
        if ($query !== false) {
            parse_str($query, $querryArray);
        }
        return $querryArray;
    }

    public function validateQuery($query = null)
    {
        if ($query === null) {
            $query = $this->_query;
        }

        if (strlen($query) === 0) {
            return true;
        }

        $pattern = '/^' . $this->_regex['uric'] . '*$/';
        $status  = @preg_match($pattern, $query);
        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: query validation failed');
        }

        return $status == 1;
    }

    public function addReplaceQueryParameters(array $queryParams)
    {
        $queryParams = array_merge($this->getQueryAsArray(), $queryParams);
        return $this->setQuery($queryParams);
    }

    public function removeQueryParameters(array $queryParamKeys)
    {
        $queryParams = array_diff_key($this->getQueryAsArray(), array_fill_keys($queryParamKeys, 0));
        return $this->setQuery($queryParams);
    }

    public function setQuery($query)
    {
        $oldQuery = $this->_query;

        if (empty($query) === true) {
            $this->_query = '';
            return $oldQuery;
        }

        if (is_array($query) === true) {
            $query = http_build_query($query, '', '&');
        } else {
            $query = (string) $query;
            if ($this->validateQuery($query) === false) {
                parse_str($query, $queryArray);
                $query = http_build_query($queryArray, '', '&');
            }
        }

        if ($this->validateQuery($query) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("'$query' is not a valid query string");
        }

        $this->_query = $query;

        return $oldQuery;
    }

    public function getFragment()
    {
        return strlen($this->_fragment) > 0 ? $this->_fragment : false;
    }

    public function validateFragment($fragment = null)
    {
        if ($fragment === null) {
            $fragment = $this->_fragment;
        }

        if (strlen($fragment) === 0) {
            return true;
        }

        $pattern = '/^' . $this->_regex['uric'] . '*$/';
        $status  = @preg_match($pattern, $fragment);
        if ($status === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception('Internal error: fragment validation failed');
        }

        return (boolean) $status;
    }

    public function setFragment($fragment)
    {
        if ($this->validateFragment($fragment) === false) {
            //require_once 'Zend/Exception.php';
            throw new Zend_Exception("Fragment \"$fragment\" is not a valid HTTP fragment");
        }

        $oldFragment     = $this->_fragment;
        $this->_fragment = $fragment;

        return $oldFragment;
    }
}