// Copyright 2020 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// See the License for the specific language governing permissions and
// limitations under the License.

package errno

import "github.com/pingcap/parser/mysql"

// MySQLErrName maps error code to MySQL error messages.
// Note: all ErrMessage to be added should be considered about the log redaction
// by setting the suitable configuration in the second argument of mysql.Message.
// See https://github.com/pingcap/tidb/blob/master/errno/logredaction.md
var MySQLErrName = map[uint16]*mysql.ErrMessage{
	ErrHashchk:                                  mysql.Message("hashchk", nil),
	ErrNisamchk:                                 mysql.Message("isamchk", nil),
	ErrNo:                                       mysql.Message("NO", nil),
	ErrYes:                                      mysql.Message("YES", nil),
	ErrCantCreateFile:                           mysql.Message("Can't create file '%-.200s' (errno: %d - %s)", nil),
	ErrCantCreateTable:                          mysql.Message("Can't create table '%-.200s' (errno: %d)", nil),
	ErrCantCreateDB:                             mysql.Message("Can't create database '%-.192s' (errno: %d)", nil),
	ErrDBCreateExists:                           mysql.Message("Can't create database '%-.192s'; database exists", nil),
	ErrDBDropExists:                             mysql.Message("Can't drop database '%-.192s'; database doesn't exist", nil),
	ErrDBDropDelete:                             mysql.Message("Error dropping database (can't delete '%-.192s', errno: %d)", nil),
	ErrDBDropRmdir:                              mysql.Message("Error dropping database (can't rmdir '%-.192s', errno: %d)", nil),
	ErrCantDeleteFile:                           mysql.Message("Error on delete of '%-.192s' (errno: %d - %s)", nil),
	ErrCantFindSystemRec:                        mysql.Message("Can't read record in system table", nil),
	ErrCantGetStat:                              mysql.Message("Can't get status of '%-.200s' (errno: %d - %s)", nil),
	ErrCantGetWd:                                mysql.Message("Can't get working directory (errno: %d - %s)", nil),
	ErrCantLock:                                 mysql.Message("Can't lock file (errno: %d - %s)", nil),
	ErrCantOpenFile:                             mysql.Message("Can't open file: '%-.200s' (errno: %d - %s)", nil),
	ErrFileNotFound:                             mysql.Message("Can't find file: '%-.200s' (errno: %d - %s)", nil),
	ErrCantReadDir:                              mysql.Message("Can't read dir of '%-.192s' (errno: %d - %s)", nil),
	ErrCantSetWd:                                mysql.Message("Can't change dir to '%-.192s' (errno: %d - %s)", nil),
	ErrCheckread:                                mysql.Message("Record has changed since last read in table '%-.192s'", nil),
	ErrDiskFull:                                 mysql.Message("Disk full (%s); waiting for someone to free some space... (errno: %d - %s)", nil),
	ErrDupKey:                                   mysql.Message("Can't write; duplicate key in table '%-.192s'", nil),
	ErrErrorOnClose:                             mysql.Message("Error on close of '%-.192s' (errno: %d - %s)", nil),
	ErrErrorOnRead:                              mysql.Message("Error reading file '%-.200s' (errno: %d - %s)", nil),
	ErrErrorOnRename:                            mysql.Message("Error on rename of '%-.210s' to '%-.210s' (errno: %d - %s)", nil),
	ErrErrorOnWrite:                             mysql.Message("Error writing file '%-.200s' (errno: %d - %s)", nil),
	ErrFileUsed:                                 mysql.Message("'%-.192s' is locked against change", nil),
	ErrFilsortAbort:                             mysql.Message("Sort aborted", nil),
	ErrFormNotFound:                             mysql.Message("View '%-.192s' doesn't exist for '%-.192s'", nil),
	ErrGetErrno:                                 mysql.Message("Got error %d from storage engine", nil),
	ErrIllegalHa:                                mysql.Message("Table storage engine for '%-.192s' doesn't have this option", nil),
	ErrKeyNotFound:                              mysql.Message("Can't find record in '%-.192s'", nil),
	ErrNotFormFile:                              mysql.Message("Incorrect information in file: '%-.200s'", nil),
	ErrNotKeyFile:                               mysql.Message("Incorrect key file for table '%-.200s'; try to repair it", nil),
	ErrOldKeyFile:                               mysql.Message("Old key file for table '%-.192s'; repair it!", nil),
	ErrOpenAsReadonly:                           mysql.Message("Table '%-.192s' is read only", nil),
	ErrOutofMemory:                              mysql.Message("Out of memory; restart server and try again (needed %d bytes)", nil),
	ErrOutOfSortMemory:                          mysql.Message("Out of sort memory, consider increasing server sort buffer size", nil),
	ErrUnexpectedEOF:                            mysql.Message("Unexpected EOF found when reading file '%-.192s' (errno: %d - %s)", nil),
	ErrConCount:                                 mysql.Message("Too many connections", nil),
	ErrOutOfResources:                           mysql.Message("Out of memory; check if mysqld or some other process uses all available memory; if not, you may have to use 'ulimit' to allow mysqld to use more memory or you can add more swap space", nil),
	ErrBadHost:                                  mysql.Message("Can't get hostname for your address", nil),
	ErrHandshake:                                mysql.Message("Bad handshake", nil),
	ErrDBaccessDenied:                           mysql.Message("Access denied for user '%-.48s'@'%-.64s' to database '%-.192s'", nil),
	ErrAccessDenied:                             mysql.Message("Access denied for user '%-.48s'@'%-.64s' (using password: %s)", nil),
	ErrNoDB:                                     mysql.Message("No database selected", nil),
	ErrUnknownCom:                               mysql.Message("Unknown command", nil),
	ErrBadNull:                                  mysql.Message("Column '%-.192s' cannot be null", nil),
	ErrBadDB:                                    mysql.Message("Unknown database '%-.192s'", nil),
	ErrTableExists:                              mysql.Message("Table '%-.192s' already exists", nil),
	ErrBadTable:                                 mysql.Message("Unknown table '%-.100s'", nil),
	ErrNonUniq:                                  mysql.Message("Column '%-.192s' in %-.192s is ambiguous", nil),
	ErrServerShutdown:                           mysql.Message("Server shutdown in progress", nil),
	ErrBadField:                                 mysql.Message("Unknown column '%-.192s' in '%-.192s'", nil),
	ErrFieldNotInGroupBy:                        mysql.Message("Expression #%d of %s is not in GROUP BY clause and contains nonaggregated column '%s' which is not functionally dependent on columns in GROUP BY clause; this is incompatible with sql_mode=only_full_group_by", nil),
	ErrWrongGroupField:                          mysql.Message("Can't group on '%-.192s'", nil),
	ErrWrongSumSelect:                           mysql.Message("Statement has sum functions and columns in same statement", nil),
	ErrWrongValueCount:                          mysql.Message("Column count doesn't match value count", nil),
	ErrTooLongIdent:                             mysql.Message("Identifier name '%-.100s' is too long", nil),
	ErrDupFieldName:                             mysql.Message("Duplicate column name '%-.192s'", nil),
	ErrDupKeyName:                               mysql.Message("Duplicate key name '%-.192s'", nil),
	ErrDupEntry:                                 mysql.Message("Duplicate entry '%-.64s' for key '%-.192s'", []int{0}),
	ErrWrongFieldSpec:                           mysql.Message("Incorrect column specifier for column '%-.192s'", nil),
	ErrParse:                                    mysql.Message("%s %s", nil),
	ErrEmptyQuery:                               mysql.Message("Query was empty", nil),
	ErrNonuniqTable:                             mysql.Message("Not unique table/alias: '%-.192s'", nil),
	ErrInvalidDefault:                           mysql.Message("Invalid default value for '%-.192s'", nil),
	ErrMultiplePriKey:                           mysql.Message("Multiple primary key defined", nil),
	ErrTooManyKeys:                              mysql.Message("Too many keys specified; max %d keys allowed", nil),
	ErrTooManyKeyParts:                          mysql.Message("Too many key parts specified; max %d parts allowed", nil),
	ErrTooLongKey:                               mysql.Message("Specified key was too long; max key length is %d bytes", nil),
	ErrKeyColumnDoesNotExits:                    mysql.Message("Key column '%-.192s' doesn't exist in table", nil),
	ErrBlobUsedAsKey:                            mysql.Message("BLOB column '%-.192s' can't be used in key specification with the used table type", nil),
	ErrTooBigFieldlength:                        mysql.Message("Column length too big for column '%-.192s' (max = %d); use BLOB or TEXT instead", nil),
	ErrWrongAutoKey:                             mysql.Message("Incorrect table definition; there can be only one auto column and it must be defined as a key", nil),
	ErrReady:                                    mysql.Message("%s: ready for connections.\nVersion: '%s'  socket: '%s'  port: %d", nil),
	ErrNormalShutdown:                           mysql.Message("%s: Normal shutdown\n", nil),
	ErrGotSignal:                                mysql.Message("%s: Got signal %d. Aborting!\n", nil),
	ErrShutdownComplete:                         mysql.Message("%s: Shutdown complete\n", nil),
	ErrForcingClose:                             mysql.Message("%s: Forcing close of thread %d  user: '%-.48s'\n", nil),
	ErrIpsock:                                   mysql.Message("Can't create IP socket", nil),
	ErrNoSuchIndex:                              mysql.Message("Table '%-.192s' has no index like the one used in CREATE INDEX; recreate the table", nil),
	ErrWrongFieldTerminators:                    mysql.Message("Field separator argument is not what is expected; check the manual", nil),
	ErrBlobsAndNoTerminated:                     mysql.Message("You can't use fixed rowlength with BLOBs; please use 'fields terminated by'", nil),
	ErrTextFileNotReadable:                      mysql.Message("The file '%-.128s' must be in the database directory or be readable by all", nil),
	ErrFileExists:                               mysql.Message("File '%-.200s' already exists", nil),
	ErrLoadInfo:                                 mysql.Message("Records: %d  Deleted: %d  Skipped: %d  Warnings: %d", nil),
	ErrAlterInfo:                                mysql.Message("Records: %d  Duplicates: %d", nil),
	ErrWrongSubKey:                              mysql.Message("Incorrect prefix key; the used key part isn't a string, the used length is longer than the key part, or the storage engine doesn't support unique prefix keys", nil),
	ErrCantRemoveAllFields:                      mysql.Message("You can't delete all columns with ALTER TABLE; use DROP TABLE instead", nil),
	ErrCantDropFieldOrKey:                       mysql.Message("Can't DROP '%-.192s'; check that column/key exists", nil),
	ErrInsertInfo:                               mysql.Message("Records: %d  Duplicates: %d  Warnings: %d", nil),
	ErrUpdateTableUsed:                          mysql.Message("You can't specify target table '%-.192s' for update in FROM clause", nil),
	ErrNoSuchThread:                             mysql.Message("Unknown thread id: %d", nil),
	ErrKillDenied:                               mysql.Message("You are not owner of thread %d", nil),
	ErrNoTablesUsed:                             mysql.Message("No tables used", nil),
	ErrTooBigSet:                                mysql.Message("Too many strings for column %-.192s and SET", nil),
	ErrNoUniqueLogFile:                          mysql.Message("Can't generate a unique log-filename %-.200s.(1-999)\n", nil),
	ErrTableNotLockedForWrite:                   mysql.Message("Table '%-.192s' was locked with a READ lock and can't be updated", nil),
	ErrTableNotLocked:                           mysql.Message("Table '%-.192s' was not locked with LOCK TABLES", nil),
	ErrBlobCantHaveDefault:                      mysql.Message("BLOB/TEXT/JSON column '%-.192s' can't have a default value", nil),
	ErrWrongDBName:                              mysql.Message("Incorrect database name '%-.100s'", nil),
	ErrWrongTableName:                           mysql.Message("Incorrect table name '%-.100s'", nil),
	ErrTooBigSelect:                             mysql.Message("The SELECT would examine more than MAXJOINSIZE rows; check your WHERE and use SET SQLBIGSELECTS=1 or SET MAXJOINSIZE=# if the SELECT is okay", nil),
	ErrUnknown:                                  mysql.Message("Unknown error", nil),
	ErrUnknownProcedure:                         mysql.Message("Unknown procedure '%-.192s'", nil),
	ErrWrongParamcountToProcedure:               mysql.Message("Incorrect parameter count to procedure '%-.192s'", nil),
	ErrWrongParametersToProcedure:               mysql.Message("Incorrect parameters to procedure '%-.192s'", nil),
	ErrUnknownTable:                             mysql.Message("Unknown table '%-.192s' in %-.32s", nil),
	ErrFieldSpecifiedTwice:                      mysql.Message("Column '%-.192s' specified twice", nil),
	ErrInvalidGroupFuncUse:                      mysql.Message("Invalid use of group function", nil),
	ErrUnsupportedExtension:                     mysql.Message("Table '%-.192s' uses an extension that doesn't exist in this MySQL version", nil),
	ErrTableMustHaveColumns:                     mysql.Message("A table must have at least 1 column", nil),
	ErrRecordFileFull:                           mysql.Message("The table '%-.192s' is full", nil),
	ErrUnknownCharacterSet:                      mysql.Message("Unknown character set: '%-.64s'", nil),
	ErrTooManyTables:                            mysql.Message("Too many tables; MySQL can only use %d tables in a join", nil),
	ErrTooManyFields:                            mysql.Message("Too many columns", nil),
	ErrTooBigRowsize:                            mysql.Message("Row size too large. The maximum row size for the used table type, not counting BLOBs, is %d. This includes storage overhead, check the manual. You have to change some columns to TEXT or BLOBs", nil),
	ErrStackOverrun:                             mysql.Message("Thread stack overrun:  Used: %d of a %d stack.  Use 'mysqld --threadStack=#' to specify a bigger stack if needed", nil),
	ErrWrongOuterJoin:                           mysql.Message("Cross dependency found in OUTER JOIN; examine your ON conditions", nil),
	ErrNullColumnInIndex:                        mysql.Message("Table handler doesn't support NULL in given index. Please change column '%-.192s' to be NOT NULL or use another handler", nil),
	ErrCantFindUdf:                              mysql.Message("Can't load function '%-.192s'", nil),
	ErrCantInitializeUdf:                        mysql.Message("Can't initialize function '%-.192s'; %-.80s", nil),
	ErrUdfNoPaths:                               mysql.Message("No paths allowed for shared library", nil),
	ErrUdfExists:                                mysql.Message("Function '%-.192s' already exists", nil),
	ErrCantOpenLibrary:                          mysql.Message("Can't open shared library '%-.192s' (errno: %d %-.128s)", nil),
	ErrCantFindDlEntry:                          mysql.Message("Can't find symbol '%-.128s' in library", nil),
	ErrFunctionNotDefined:                       mysql.Message("Function '%-.192s' is not defined", nil),
	ErrHostIsBlocked:                            mysql.Message("Host '%-.64s' is blocked because of many connection errors; unblock with 'mysqladmin flush-hosts'", nil),
	ErrHostNotPrivileged:                        mysql.Message("Host '%-.64s' is not allowed to connect to this MySQL server", nil),
	ErrPasswordAnonymousUser:                    mysql.Message("You are using MySQL as an anonymous user and anonymous users are not allowed to change passwords", nil),
	ErrPasswordNotAllowed:                       mysql.Message("You must have privileges to update tables in the mysql database to be able to change passwords for others", nil),
	ErrPasswordNoMatch:                          mysql.Message("Can't find any matching row in the user table", nil),
	ErrUpdateInfo:                               mysql.Message("Rows matched: %d  Changed: %d  Warnings: %d", nil),
	ErrCantCreateThread:                         mysql.Message("Can't create a new thread (errno %d); if you are not out of available memory, you can consult the manual for a possible OS-dependent bug", nil),
	ErrWrongValueCountOnRow:                     mysql.Message("Column count doesn't match value count at row %d", nil),
	ErrCantReopenTable:                          mysql.Message("Can't reopen table: '%-.192s'", nil),
	ErrInvalidUseOfNull:                         mysql.Message("Invalid use of NULL value", nil),
	ErrRegexp:                                   mysql.Message("Got error '%-.64s' from regexp", nil),
	ErrMixOfGroupFuncAndFields:                  mysql.Message("Mixing of GROUP columns (MIN(),MAX(),COUNT(),...) with no GROUP columns is illegal if there is no GROUP BY clause", nil),
	ErrNonexistingGrant:                         mysql.Message("There is no such grant defined for user '%-.48s' on host '%-.64s'", nil),
	ErrTableaccessDenied:                        mysql.Message("%-.128s command denied to user '%-.48s'@'%-.64s' for table '%-.64s'", nil),
	ErrColumnaccessDenied:                       mysql.Message("%-.16s command denied to user '%-.48s'@'%-.64s' for column '%-.192s' in table '%-.192s'", nil),
	ErrIllegalGrantForTable:                     mysql.Message("Illegal GRANT/REVOKE command; please consult the manual to see which privileges can be used", nil),
	ErrGrantWrongHostOrUser:                     mysql.Message("The host or user argument to GRANT is too long", nil),
	ErrNoSuchTable:                              mysql.Message("Table '%-.192s.%-.192s' doesn't exist", nil),
	ErrNonexistingTableGrant:                    mysql.Message("There is no such grant defined for user '%-.48s' on host '%-.64s' on table '%-.192s'", nil),
	ErrNotAllowedCommand:                        mysql.Message("The used command is not allowed with this MySQL version", nil),
	ErrSyntax:                                   mysql.Message("You have an error in your SQL syntax; check the manual that corresponds to your MySQL server version for the right syntax to use", nil),
	ErrDelayedCantChangeLock:                    mysql.Message("Delayed insert thread couldn't get requested lock for table %-.192s", nil),
	ErrTooManyDelayedThreads:                    mysql.Message("Too many delayed threads in use", nil),
	ErrAbortingConnection:                       mysql.Message("Aborted connection %d to db: '%-.192s' user: '%-.48s' (%-.64s)", nil),
	ErrNetPacketTooLarge:                        mysql.Message("Got a packet bigger than 'maxAllowedPacket' bytes", nil),
	ErrNetReadErrorFromPipe:                     mysql.Message("Got a read error from the connection pipe", nil),
	ErrNetFcntl:                                 mysql.Message("Got an error from fcntl()", nil),
	ErrNetPacketsOutOfOrder:                     mysql.Message("Got packets out of order", nil),
	ErrNetUncompress:                            mysql.Message("Couldn't uncompress communication packet", nil),
	ErrNetRead:                                  mysql.Message("Got an error reading communication packets", nil),
	ErrNetReadInterrupted:                       mysql.Message("Got timeout reading communication packets", nil),
	ErrNetErrorOnWrite:                          mysql.Message("Got an error writing communication packets", nil),
	ErrNetWriteInterrupted:                      mysql.Message("Got timeout writing communication packets", nil),
	ErrTooLongString:                            mysql.Message("Result string is longer than 'maxAllowedPacket' bytes", nil),
	ErrTableCantHandleBlob:                      mysql.Message("The used table type doesn't support BLOB/TEXT columns", nil),
	ErrTableCantHandleAutoIncrement:             mysql.Message("The used table type doesn't support AUTOINCREMENT columns", nil),
	ErrDelayedInsertTableLocked:                 mysql.Message("INSERT DELAYED can't be used with table '%-.192s' because it is locked with LOCK TABLES", nil),
	ErrWrongColumnName:                          mysql.Message("Incorrect column name '%-.100s'", nil),
	ErrWrongKeyColumn:                           mysql.Message("The used storage engine can't index column '%-.192s'", nil),
	ErrWrongMrgTable:                            mysql.Message("Unable to open underlying table which is differently defined or of non-MyISAM type or doesn't exist", nil),
	ErrDupUnique:                                mysql.Message("Can't write, because of unique constraint, to table '%-.192s'", nil),
	ErrBlobKeyWithoutLength:                     mysql.Message("BLOB/TEXT column '%-.192s' used in key specification without a key length", nil),
	ErrPrimaryCantHaveNull:                      mysql.Message("All parts of a PRIMARY KEY must be NOT NULL; if you need NULL in a key, use UNIQUE instead", nil),
	ErrTooManyRows:                              mysql.Message("Result consisted of more than one row", nil),
	ErrRequiresPrimaryKey:                       mysql.Message("This table type requires a primary key", nil),
	ErrNoRaidCompiled:                           mysql.Message("This version of MySQL is not compiled with RAID support", nil),
	ErrUpdateWithoutKeyInSafeMode:               mysql.Message("You are using safe update mode and you tried to update a table without a WHERE that uses a KEY column", nil),
	ErrKeyDoesNotExist:                          mysql.Message("Key '%-.192s' doesn't exist in table '%-.192s'", nil),
	ErrCheckNoSuchTable:                         mysql.Message("Can't open table", nil),
	ErrCheckNotImplemented:                      mysql.Message("The storage engine for the table doesn't support %s", nil),
	ErrCantDoThisDuringAnTransaction:            mysql.Message("You are not allowed to execute this command in a transaction", nil),
	ErrErrorDuringCommit:                        mysql.Message("Got error %d during COMMIT", nil),
	ErrErrorDuringRollback:                      mysql.Message("Got error %d during ROLLBACK", nil),
	ErrErrorDuringFlushLogs:                     mysql.Message("Got error %d during FLUSHLOGS", nil),
	ErrErrorDuringCheckpoint:                    mysql.Message("Got error %d during CHECKPOINT", nil),
	ErrNewAbortingConnection:                    mysql.Message("Aborted connection %d to db: '%-.192s' user: '%-.48s' host: '%-.64s' (%-.64s)", nil),
	ErrDumpNotImplemented:                       mysql.Message("The storage engine for the table does not support binary table dump", nil),
	ErrIndexRebuild:                             mysql.Message("Failed rebuilding the index of  dumped table '%-.192s'", nil),
	ErrFtMatchingKeyNotFound:                    mysql.Message("Can't find FULLTEXT index matching the column list", nil),
	ErrLockOrActiveTransaction:                  mysql.Message("Can't execute the given command because you have active locked tables or an active transaction", nil),
	ErrUnknownSystemVariable:                    mysql.Message("Unknown system variable '%-.64s'", nil),
	ErrCrashedOnUsage:                           mysql.Message("Table '%-.192s' is marked as crashed and should be repaired", nil),
	ErrCrashedOnRepair:                          mysql.Message("Table '%-.192s' is marked as crashed and last (automatic?) repair failed", nil),
	ErrWarningNotCompleteRollback:               mysql.Message("Some non-transactional changed tables couldn't be rolled back", nil),
	ErrTransCacheFull:                           mysql.Message("Multi-statement transaction required more than 'maxBinlogCacheSize' bytes of storage; increase this mysqld variable and try again", nil),
	ErrTooManyUserConnections:                   mysql.Message("User %-.64s already has more than 'maxUserConnections' active connections", nil),
	ErrSetConstantsOnly:                         mysql.Message("You may only use constant expressions with SET", nil),
	ErrLockWaitTimeout:                          mysql.Message("Lock wait timeout exceeded; try restarting transaction", nil),
	ErrLockTableFull:                            mysql.Message("The total number of locks exceeds the lock table size", nil),
	ErrReadOnlyTransaction:                      mysql.Message("Update locks cannot be acquired during a READ UNCOMMITTED transaction", nil),
	ErrDropDBWithReadLock:                       mysql.Message("DROP DATABASE not allowed while thread is holding global read lock", nil),
	ErrCreateDBWithReadLock:                     mysql.Message("CREATE DATABASE not allowed while thread is holding global read lock", nil),
	ErrWrongArguments:                           mysql.Message("Incorrect arguments to %s", nil),
	ErrNoPermissionToCreateUser:                 mysql.Message("'%-.48s'@'%-.64s' is not allowed to create new users", nil),
	ErrUnionTablesInDifferentDir:                mysql.Message("Incorrect table definition; all MERGE tables must be in the same database", nil),
	ErrLockDeadlock:                             mysql.Message("Deadlock found when trying to get lock; try restarting transaction", nil),
	ErrTableCantHandleFt:                        mysql.Message("The used table type doesn't support FULLTEXT indexes", nil),
	ErrCannotAddForeign:                         mysql.Message("Cannot add foreign key constraint", nil),
	ErrNoReferencedRow:                          mysql.Message("Cannot add or update a child row: a foreign key constraint fails", nil),
	ErrRowIsReferenced:                          mysql.Message("Cannot delete or update a parent row: a foreign key constraint fails", nil),
	ErrErrorWhenExecutingCommand:                mysql.Message("Error when executing command %s: %-.128s", nil),
	ErrWrongUsage:                               mysql.Message("Incorrect usage of %s and %s", nil),
	ErrWrongNumberOfColumnsInSelect:             mysql.Message("The used SELECT statements have a different number of columns", nil),
	ErrCantUpdateWithReadlock:                   mysql.Message("Can't execute the query because you have a conflicting read lock", nil),
	ErrMixingNotAllowed:                         mysql.Message("Mixing of transactional and non-transactional tables is disabled", nil),
	ErrDupArgument:                              mysql.Message("Option '%s' used twice in statement", nil),
	ErrUserLimitReached:                         mysql.Message("User '%-.64s' has exceeded the '%s' resource (current value: %d)", nil),
	ErrSpecificAccessDenied:                     mysql.Message("Access denied; you need (at least one of) the %-.128s privilege(s) for this operation", nil),
	ErrLocalVariable:                            mysql.Message("Variable '%-.64s' is a SESSION variable and can't be used with SET GLOBAL", nil),
	ErrGlobalVariable:                           mysql.Message("Variable '%-.64s' is a GLOBAL variable and should be set with SET GLOBAL", nil),
	ErrNoDefault:                                mysql.Message("Variable '%-.64s' doesn't have a default value", nil),
	ErrWrongValueForVar:                         mysql.Message("Variable '%-.64s' can't be set to the value of '%-.200s'", nil),
	ErrWrongTypeForVar:                          mysql.Message("Incorrect argument type to variable '%-.64s'", nil),
	ErrVarCantBeRead:                            mysql.Message("Variable '%-.64s' can only be set, not read", nil),
	ErrCantUseOptionHere:                        mysql.Message("Incorrect usage/placement of '%s'", nil),
	ErrNotSupportedYet:                          mysql.Message("This version of TiDB doesn't yet support '%s'", nil),
	ErrIncorrectGlobalLocalVar:                  mysql.Message("Variable '%-.192s' is a %s variable", nil),
	ErrWrongFkDef:                               mysql.Message("Incorrect foreign key definition for '%-.192s': %s", nil),
	ErrKeyRefDoNotMatchTableRef:                 mysql.Message("Key reference and table reference don't match", nil),
	ErrOperandColumns:                           mysql.Message("Operand should contain %d column(s)", nil),
	ErrSubqueryNo1Row:                           mysql.Message("Subquery returns more than 1 row", nil),
	ErrUnknownStmtHandler:                       mysql.Message("Unknown prepared statement handler (%.*s) given to %s", nil),
	ErrCorruptHelpDB:                            mysql.Message("Help database is corrupt or does not exist", nil),
	ErrCyclicReference:                          mysql.Message("Cyclic reference on subqueries", nil),
	ErrAutoConvert:                              mysql.Message("Converting column '%s' from %s to %s", nil),
	ErrIllegalReference:                         mysql.Message("Reference '%-.64s' not supported (%s)", nil),
	ErrDerivedMustHaveAlias:                     mysql.Message("Every derived table must have its own alias", nil),
	ErrSelectReduced:                            mysql.Message("Select %d was reduced during optimization", nil),
	ErrTablenameNotAllowedHere:                  mysql.Message("Table '%s' from one of the %ss cannot be used in %s", nil),
	ErrNotSupportedAuthMode:                     mysql.Message("Client does not support authentication protocol requested by server; consider upgrading MySQL client", nil),
	ErrSpatialCantHaveNull:                      mysql.Message("All parts of a SPATIAL index must be NOT NULL", nil),
	ErrCollationCharsetMismatch:                 mysql.Message("COLLATION '%s' is not valid for CHARACTER SET '%s'", nil),
	ErrTooBigForUncompress:                      mysql.Message("Uncompressed data size too large; the maximum size is %d (probably, length of uncompressed data was corrupted)", nil),
	ErrZlibZMem:                                 mysql.Message("ZLIB: Not enough memory", nil),
	ErrZlibZBuf:                                 mysql.Message("ZLIB: Not enough room in the output buffer (probably, length of uncompressed data was corrupted)", nil),
	ErrZlibZData:                                mysql.Message("ZLIB: Input data corrupted", nil),
	ErrCutValueGroupConcat:                      mysql.Message("Some rows were cut by GROUPCONCAT(%s)", []int{0}),
	ErrWarnTooFewRecords:                        mysql.Message("Row %d doesn't contain data for all columns", nil),
	ErrWarnTooManyRecords:                       mysql.Message("Row %d was truncated; it contained more data than there were input columns", nil),
	ErrWarnNullToNotnull:                        mysql.Message("Column set to default value; NULL supplied to NOT NULL column '%s' at row %d", nil),
	ErrWarnDataOutOfRange:                       mysql.Message("Out of range value for column '%s' at row %d", nil),
	WarnDataTruncated:                           mysql.Message("Data truncated for column '%s' at row %d", nil),
	ErrWarnUsingOtherHandler:                    mysql.Message("Using storage engine %s for table '%s'", nil),
	ErrCantAggregate2collations:                 mysql.Message("Illegal mix of collations (%s,%s) and (%s,%s) for operation '%s'", nil),
	ErrDropUser:                                 mysql.Message("Cannot drop one or more of the requested users", nil),
	ErrRevokeGrants:                             mysql.Message("Can't revoke all privileges for one or more of the requested users", nil),
	ErrCantAggregate3collations:                 mysql.Message("Illegal mix of collations (%s,%s), (%s,%s), (%s,%s) for operation '%s'", nil),
	ErrCantAggregateNcollations:                 mysql.Message("Illegal mix of collations for operation '%s'", nil),
	ErrVariableIsNotStruct:                      mysql.Message("Variable '%-.64s' is not a variable component (can't be used as XXXX.variableName)", nil),
	ErrUnknownCollation:                         mysql.Message("Unknown collation: '%-.64s'", nil),
	ErrServerIsInSecureAuthMode:                 mysql.Message("Server is running in --secure-auth mode, but '%s'@'%s' has a password in the old format; please change the password to the new format", nil),
	ErrWarnFieldResolved:                        mysql.Message("Field or reference '%-.192s%s%-.192s%s%-.192s' of SELECT #%d was resolved in SELECT #%d", nil),
	ErrUntilCondIgnored:                         mysql.Message("SQL thread is not to be started so UNTIL options are ignored", nil),
	ErrWrongNameForIndex:                        mysql.Message("Incorrect index name '%-.100s'", nil),
	ErrWrongNameForCatalog:                      mysql.Message("Incorrect catalog name '%-.100s'", nil),
	ErrWarnQcResize:                             mysql.Message("Query cache failed to set size %d; new query cache size is %d", nil),
	ErrBadFtColumn:                              mysql.Message("Column '%-.192s' cannot be part of FULLTEXT index", nil),
	ErrUnknownKeyCache:                          mysql.Message("Unknown key cache '%-.100s'", nil),
	ErrWarnHostnameWontWork:                     mysql.Message("MySQL is started in --skip-name-resolve mode; you must restart it without this switch for this grant to work", nil),
	ErrUnknownStorageEngine:                     mysql.Message("Unknown storage engine '%s'", nil),
	ErrWarnDeprecatedSyntax:                     mysql.Message("'%s' is deprecated and will be removed in a future release. Please use %s instead", nil),
	ErrNonUpdatableTable:                        mysql.Message("The target table %-.100s of the %s is not updatable", nil),
	ErrFeatureDisabled:                          mysql.Message("The '%s' feature is disabled; you need MySQL built with '%s' to have it working", nil),
	ErrOptionPreventsStatement:                  mysql.Message("The MySQL server is running with the %s option so it cannot execute this statement", nil),
	ErrDuplicatedValueInType:                    mysql.Message("Column '%-.100s' has duplicated value '%-.64s' in %s", []int{1}),
	ErrTruncatedWrongValue:                      mysql.Message("Truncated incorrect %-.64s value: '%-.128s'", []int{1}),
	ErrTooMuchAutoTimestampCols:                 mysql.Message("Incorrect table definition; there can be only one TIMESTAMP column with CURRENTTIMESTAMP in DEFAULT or ON UPDATE clause", nil),
	ErrInvalidOnUpdate:                          mysql.Message("Invalid ON UPDATE clause for '%-.192s' column", nil),
	ErrUnsupportedPs:                            mysql.Message("This command is not supported in the prepared statement protocol yet", nil),
	ErrGetErrmsg:                                mysql.Message("Got error %d '%-.100s' from %s", nil),
	ErrGetTemporaryErrmsg:                       mysql.Message("Got temporary error %d '%-.100s' from %s", nil),
	ErrUnknownTimeZone:                          mysql.Message("Unknown or incorrect time zone: '%-.64s'", nil),
	ErrWarnInvalidTimestamp:                     mysql.Message("Invalid TIMESTAMP value in column '%s' at row %d", nil),
	ErrInvalidCharacterString:                   mysql.Message("Invalid %s character string: '%.64s'", []int{1}),
	ErrWarnAllowedPacketOverflowed:              mysql.Message("Result of %s() was larger than max_allowed_packet (%d) - truncated", nil),
	ErrConflictingDeclarations:                  mysql.Message("Conflicting declarations: '%s%s' and '%s%s'", nil),
	ErrSpNoRecursiveCreate:                      mysql.Message("Can't create a %s from within another stored routine", nil),
	ErrSpAlreadyExists:                          mysql.Message("%s %s already exists", nil),
	ErrSpDoesNotExist:                           mysql.Message("%s %s does not exist", nil),
	ErrSpDropFailed:                             mysql.Message("Failed to DROP %s %s", nil),
	ErrSpStoreFailed:                            mysql.Message("Failed to CREATE %s %s", nil),
	ErrSpLilabelMismatch:                        mysql.Message("%s with no matching label: %s", nil),
	ErrSpLabelRedefine:                          mysql.Message("Redefining label %s", nil),
	ErrSpLabelMismatch:                          mysql.Message("End-label %s without match", nil),
	ErrSpUninitVar:                              mysql.Message("Referring to uninitialized variable %s", nil),
	ErrSpBadselect:                              mysql.Message("PROCEDURE %s can't return a result set in the given context", nil),
	ErrSpBadreturn:                              mysql.Message("RETURN is only allowed in a FUNCTION", nil),
	ErrSpBadstatement:                           mysql.Message("%s is not allowed in stored procedures", nil),
	ErrUpdateLogDeprecatedIgnored:               mysql.Message("The update log is deprecated and replaced by the binary log; SET SQLLOGUPDATE has been ignored.", nil),
	ErrUpdateLogDeprecatedTranslated:            mysql.Message("The update log is deprecated and replaced by the binary log; SET SQLLOGUPDATE has been translated to SET SQLLOGBIN.", nil),
	ErrQueryInterrupted:                         mysql.Message("Query execution was interrupted", nil),
	ErrSpWrongNoOfArgs:                          mysql.Message("Incorrect number of arguments for %s %s; expected %d, got %d", nil),
	ErrSpCondMismatch:                           mysql.Message("Undefined CONDITION: %s", nil),
	ErrSpNoreturn:                               mysql.Message("No RETURN found in FUNCTION %s", nil),
	ErrSpNoreturnend:                            mysql.Message("FUNCTION %s ended without RETURN", nil),
	ErrSpBadCursorQuery:                         mysql.Message("Cursor statement must be a SELECT", nil),
	ErrSpBadCursorSelect:                        mysql.Message("Cursor SELECT must not have INTO", nil),
	ErrSpCursorMismatch:                         mysql.Message("Undefined CURSOR: %s", nil),
	ErrSpCursorAlreadyOpen:                      mysql.Message("Cursor is already open", nil),
	ErrSpCursorNotOpen:                          mysql.Message("Cursor is not open", nil),
	ErrSpUndeclaredVar:                          mysql.Message("Undeclared variable: %s", nil),
	ErrSpWrongNoOfFetchArgs:                     mysql.Message("Incorrect number of FETCH variables", nil),
	ErrSpFetchNoData:                            mysql.Message("No data - zero rows fetched, selected, or processed", nil),
	ErrSpDupParam:                               mysql.Message("Duplicate parameter: %s", nil),
	ErrSpDupVar:                                 mysql.Message("Duplicate variable: %s", nil),
	ErrSpDupCond:                                mysql.Message("Duplicate condition: %s", nil),
	ErrSpDupCurs:                                mysql.Message("Duplicate cursor: %s", nil),
	ErrSpCantAlter:                              mysql.Message("Failed to ALTER %s %s", nil),
	ErrSpSubselectNyi:                           mysql.Message("Subquery value not supported", nil),
	ErrStmtNotAllowedInSfOrTrg:                  mysql.Message("%s is not allowed in stored function or trigger", nil),
	ErrSpVarcondAfterCurshndlr:                  mysql.Message("Variable or condition declaration after cursor or handler declaration", nil),
	ErrSpCursorAfterHandler:                     mysql.Message("Cursor declaration after handler declaration", nil),
	ErrSpCaseNotFound:                           mysql.Message("Case not found for CASE statement", nil),
	ErrFparserTooBigFile:                        mysql.Message("Configuration file '%-.192s' is too big", nil),
	ErrFparserBadHeader:                         mysql.Message("Malformed file type header in file '%-.192s'", nil),
	ErrFparserEOFInComment:                      mysql.Message("Unexpected end of file while parsing comment '%-.200s'", nil),
	ErrFparserErrorInParameter:                  mysql.Message("Error while parsing parameter '%-.192s' (line: '%-.192s')", nil),
	ErrFparserEOFInUnknownParameter:             mysql.Message("Unexpected end of file while skipping unknown parameter '%-.192s'", nil),
	ErrViewNoExplain:                            mysql.Message("EXPLAIN/SHOW can not be issued; lacking privileges for underlying table", nil),
	ErrFrmUnknownType:                           mysql.Message("File '%-.192s' has unknown type '%-.64s' in its header", nil),
	ErrWrongObject:                              mysql.Message("'%-.192s.%-.192s' is not %s", nil),
	ErrNonupdateableColumn:                      mysql.Message("Column '%-.192s' is not updatable", nil),
	ErrViewSelectDerived:                        mysql.Message("View's SELECT contains a subquery in the FROM clause", nil),
	ErrViewSelectClause:                         mysql.Message("View's SELECT contains a '%s' clause", nil),
	ErrViewSelectVariable:                       mysql.Message("View's SELECT contains a variable or parameter", nil),
	ErrViewSelectTmptable:                       mysql.Message("View's SELECT refers to a temporary table '%-.192s'", nil),
	ErrViewWrongList:                            mysql.Message("View's SELECT and view's field list have different column counts", nil),
	ErrWarnViewMerge:                            mysql.Message("View merge algorithm can't be used here for now (assumed undefined algorithm)", nil),
	ErrWarnViewWithoutKey:                       mysql.Message("View being updated does not have complete key of underlying table in it", nil),
	ErrViewInvalid:                              mysql.Message("View '%-.192s.%-.192s' references invalid table(s) or column(s) or function(s) or definer/invoker of view lack rights to use them", nil),
	ErrSpNoDropSp:                               mysql.Message("Can't drop or alter a %s from within another stored routine", nil),
	ErrSpGotoInHndlr:                            mysql.Message("GOTO is not allowed in a stored procedure handler", nil),
	ErrTrgAlreadyExists:                         mysql.Message("Trigger already exists", nil),
	ErrTrgDoesNotExist:                          mysql.Message("Trigger does not exist", nil),
	ErrTrgOnViewOrTempTable:                     mysql.Message("Trigger's '%-.192s' is view or temporary table", nil),
	ErrTrgCantChangeRow:                         mysql.Message("Updating of %s row is not allowed in %strigger", nil),
	ErrTrgNoSuchRowInTrg:                        mysql.Message("There is no %s row in %s trigger", nil),
	ErrNoDefaultForField:                        mysql.Message("Field '%-.192s' doesn't have a default value", nil),
	ErrDivisionByZero:                           mysql.Message("Division by 0", nil),
	ErrTruncatedWrongValueForField:              mysql.Message("Incorrect %-.32s value: '%-.128s' for column '%.192s' at row %d", []int{0, 1}),
	ErrIllegalValueForType:                      mysql.Message("Illegal %s '%-.192s' value found during parsing", []int{1}),
	ErrViewNonupdCheck:                          mysql.Message("CHECK OPTION on non-updatable view '%-.192s.%-.192s'", nil),
	ErrViewCheckFailed:                          mysql.Message("CHECK OPTION failed '%-.192s.%-.192s'", nil),
	ErrProcaccessDenied:                         mysql.Message("%-.16s command denied to user '%-.48s'@'%-.64s' for routine '%-.192s'", nil),
	ErrRelayLogFail:                             mysql.Message("Failed purging old relay logs: %s", nil),
	ErrPasswdLength:                             mysql.Message("Password hash should be a %d-digit hexadecimal number", nil),
	ErrUnknownTargetBinlog:                      mysql.Message("Target log not found in binlog index", nil),
	ErrIoErrLogIndexRead:                        mysql.Message("I/O error reading log index file", nil),
	ErrBinlogPurgeProhibited:                    mysql.Message("Server configuration does not permit binlog purge", nil),
	ErrFseekFail:                                mysql.Message("Failed on fseek()", nil),
	ErrBinlogPurgeFatalErr:                      mysql.Message("Fatal error during log purge", nil),
	ErrLogInUse:                                 mysql.Message("A purgeable log is in use, will not purge", nil),
	ErrLogPurgeUnknownErr:                       mysql.Message("Unknown error during log purge", nil),
	ErrRelayLogInit:                             mysql.Message("Failed initializing relay log position: %s", nil),
	ErrNoBinaryLogging:                          mysql.Message("You are not using binary logging", nil),
	ErrReservedSyntax:                           mysql.Message("The '%-.64s' syntax is reserved for purposes internal to the MySQL server", nil),
	ErrWsasFailed:                               mysql.Message("WSAStartup Failed", nil),
	ErrDiffGroupsProc:                           mysql.Message("Can't handle procedures with different groups yet", nil),
	ErrNoGroupForProc:                           mysql.Message("Select must have a group with this procedure", nil),
	ErrOrderWithProc:                            mysql.Message("Can't use ORDER clause with this procedure", nil),
	ErrLoggingProhibitChangingOf:                mysql.Message("Binary logging and replication forbid changing the global server %s", nil),
	ErrNoFileMapping:                            mysql.Message("Can't map file: %-.200s, errno: %d", nil),
	ErrWrongMagic:                               mysql.Message("Wrong magic in %-.64s", nil),
	ErrPsManyParam:                              mysql.Message("Prepared statement contains too many placeholders", nil),
	ErrKeyPart0:                                 mysql.Message("Key part '%-.192s' length cannot be 0", nil),
	ErrViewChecksum:                             mysql.Message("View text checksum failed", nil),
	ErrViewMultiupdate:                          mysql.Message("Can not modify more than one base table through a join view '%-.192s.%-.192s'", nil),
	ErrViewNoInsertFieldList:                    mysql.Message("Can not insert into join view '%-.192s.%-.192s' without fields list", nil),
	ErrViewDeleteMergeView:                      mysql.Message("Can not delete from join view '%-.192s.%-.192s'", nil),
	ErrCannotUser:                               mysql.Message("Operation %s failed for %.256s", nil),
	ErrXaerNota:                                 mysql.Message("XAERNOTA: Unknown XID", nil),
	ErrXaerInval:                                mysql.Message("XAERINVAL: Invalid arguments (or unsupported command)", nil),
	ErrXaerRmfail:                               mysql.Message("XAERRMFAIL: The command cannot be executed when global transaction is in the  %.64s state", nil),
	ErrXaerOutside:                              mysql.Message("XAEROUTSIDE: Some work is done outside global transaction", nil),
	ErrXaerRmerr:                                mysql.Message("XAERRMERR: Fatal error occurred in the transaction branch - check your data for consistency", nil),
	ErrXaRbrollback:                             mysql.Message("XARBROLLBACK: Transaction branch was rolled back", nil),
	ErrNonexistingProcGrant:                     mysql.Message("There is no such grant defined for user '%-.48s' on host '%-.64s' on routine '%-.192s'", nil),
	ErrProcAutoGrantFail:                        mysql.Message("Failed to grant EXECUTE and ALTER ROUTINE privileges", nil),
	ErrProcAutoRevokeFail:                       mysql.Message("Failed to revoke all privileges to dropped routine", nil),
	ErrDataTooLong:                              mysql.Message("Data too long for column '%s' at row %d", nil),
	ErrSpBadSQLstate:                            mysql.Message("Bad SQLSTATE: '%s'", nil),
	ErrStartup:                                  mysql.Message("%s: ready for connections.\nVersion: '%s'  socket: '%s'  port: %d  %s", nil),
	ErrLoadFromFixedSizeRowsToVar:               mysql.Message("Can't load value from file with fixed size rows to variable", nil),
	ErrCantCreateUserWithGrant:                  mysql.Message("You are not allowed to create a user with GRANT", nil),
	ErrWrongValueForType:                        mysql.Message("Incorrect %-.32s value: '%-.128s' for function %-.32s", nil),
	ErrTableDefChanged:                          mysql.Message("Table definition has changed, please retry transaction", nil),
	ErrSpDupHandler:                             mysql.Message("Duplicate handler declared in the same block", nil),
	ErrSpNotVarArg:                              mysql.Message("OUT or INOUT argument %d for routine %s is not a variable or NEW pseudo-variable in BEFORE trigger", nil),
	ErrSpNoRetset:                               mysql.Message("Not allowed to return a result set from a %s", nil),
	ErrCantCreateGeometryObject:                 mysql.Message("Cannot get geometry object from data you send to the GEOMETRY field", nil),
	ErrFailedRoutineBreakBinlog:                 mysql.Message("A routine failed and has neither NO SQL nor READS SQL DATA in its declaration and binary logging is enabled; if non-transactional tables were updated, the binary log will miss their changes", nil),
	ErrBinlogUnsafeRoutine:                      mysql.Message("This function has none of DETERMINISTIC, NO SQL, or READS SQL DATA in its declaration and binary logging is enabled (you *might* want to use the less safe logBinTrustFunctionCreators variable)", nil),
	ErrBinlogCreateRoutineNeedSuper:             mysql.Message("You do not have the SUPER privilege and binary logging is enabled (you *might* want to use the less safe logBinTrustFunctionCreators variable)", nil),
	ErrExecStmtWithOpenCursor:                   mysql.Message("You can't execute a prepared statement which has an open cursor associated with it. Reset the statement to re-execute it.", nil),
	ErrStmtHasNoOpenCursor:                      mysql.Message("The statement (%d) has no open cursor.", nil),
	ErrCommitNotAllowedInSfOrTrg:                mysql.Message("Explicit or implicit commit is not allowed in stored function or trigger.", nil),
	ErrNoDefaultForViewField:                    mysql.Message("Field of view '%-.192s.%-.192s' underlying table doesn't have a default value", nil),
	ErrSpNoRecursion:                            mysql.Message("Recursive stored functions and triggers are not allowed.", nil),
	ErrTooBigScale:                              mysql.Message("Too big scale %d specified for column '%-.192s'. Maximum is %d.", nil),
	ErrTooBigPrecision:                          mysql.Message("Too big precision %d specified for column '%-.192s'. Maximum is %d.", nil),
	ErrMBiggerThanD:                             mysql.Message("For float(M,D), double(M,D) or decimal(M,D), M must be >= D (column '%-.192s').", nil),
	ErrWrongLockOfSystemTable:                   mysql.Message("You can't combine write-locking of system tables with other tables or lock types", nil),
	ErrConnectToForeignDataSource:               mysql.Message("Unable to connect to foreign data source: %.64s", nil),
	ErrQueryOnForeignDataSource:                 mysql.Message("There was a problem processing the query on the foreign data source. Data source : %-.64s", nil),
	ErrForeignDataSourceDoesntExist:             mysql.Message("The foreign data source you are trying to reference does not exist. Data source :  %-.64s", nil),
	ErrForeignDataStringInvalidCantCreate:       mysql.Message("Can't create federated table. The data source connection string '%-.64s' is not in the correct format", nil),
	ErrForeignDataStringInvalid:                 mysql.Message("The data source connection string '%-.64s' is not in the correct format", nil),
	ErrCantCreateFederatedTable:                 mysql.Message("Can't create federated table. Foreign data src :  %-.64s", nil),
	ErrTrgInWrongSchema:                         mysql.Message("Trigger in wrong schema", nil),
	ErrStackOverrunNeedMore:                     mysql.Message("Thread stack overrun:  %d bytes used of a %d byte stack, and %d bytes needed.  Use 'mysqld --threadStack=#' to specify a bigger stack.", nil),
	ErrTooLongBody:                              mysql.Message("Routine body for '%-.100s' is too long", nil),
	ErrWarnCantDropDefaultKeycache:              mysql.Message("Cannot drop default keycache", nil),
	ErrTooBigDisplaywidth:                       mysql.Message("Display width out of range for column '%-.192s' (max = %d)", nil),
	ErrXaerDupid:                                mysql.Message("XAERDUPID: The XID already exists", nil),
	ErrDatetimeFunctionOverflow:                 mysql.Message("Datetime function: %-.32s field overflow", nil),
	ErrCantUpdateUsedTableInSfOrTrg:             mysql.Message("Can't update table '%-.192s' in stored function/trigger because it is already used by statement which invoked this stored function/trigger.", nil),
	ErrViewPreventUpdate:                        mysql.Message("The definition of table '%-.192s' prevents operation %.192s on table '%-.192s'.", nil),
	ErrPsNoRecursion:                            mysql.Message("The prepared statement contains a stored routine call that refers to that same statement. It's not allowed to execute a prepared statement in such a recursive manner", nil),
	ErrSpCantSetAutocommit:                      mysql.Message("Not allowed to set autocommit from a stored function or trigger", nil),
	ErrMalformedDefiner:                         mysql.Message("Definer is not fully qualified", nil),
	ErrViewFrmNoUser:                            mysql.Message("View '%-.192s'.'%-.192s' has no definer information (old table format). Current user is used as definer. Please recreate the view!", nil),
	ErrViewOtherUser:                            mysql.Message("You need the SUPER privilege for creation view with '%-.192s'@'%-.192s' definer", nil),
	ErrNoSuchUser:                               mysql.Message("The user specified as a definer ('%-.64s'@'%-.64s') does not exist", nil),
	ErrForbidSchemaChange:                       mysql.Message("Changing schema from '%-.192s' to '%-.192s' is not allowed.", nil),
	ErrRowIsReferenced2:                         mysql.Message("Cannot delete or update a parent row: a foreign key constraint fails (%.192s)", nil),
	ErrNoReferencedRow2:                         mysql.Message("Cannot add or update a child row: a foreign key constraint fails (%.192s)", nil),
	ErrSpBadVarShadow:                           mysql.Message("Variable '%-.64s' must be quoted with `...`, or renamed", nil),
	ErrTrgNoDefiner:                             mysql.Message("No definer attribute for trigger '%-.192s'.'%-.192s'. The trigger will be activated under the authorization of the invoker, which may have insufficient privileges. Please recreate the trigger.", nil),
	ErrOldFileFormat:                            mysql.Message("'%-.192s' has an old format, you should re-create the '%s' object(s)", nil),
	ErrSpRecursionLimit:                         mysql.Message("Recursive limit %d (as set by the maxSpRecursionDepth variable) was exceeded for routine %.192s", nil),
	ErrSpProcTableCorrupt:                       mysql.Message("Failed to load routine %-.192s. The table mysql.proc is missing, corrupt, or contains bad data (internal code %d)", nil),
	ErrSpWrongName:                              mysql.Message("Incorrect routine name '%-.192s'", nil),
	ErrTableNeedsUpgrade:                        mysql.Message("Table upgrade required. Please do \"REPAIR TABLE `%-.32s`\"", nil),
	ErrSpNoAggregate:                            mysql.Message("AGGREGATE is not supported for stored functions", nil),
	ErrMaxPreparedStmtCountReached:              mysql.Message("Can't create more than maxPreparedStmtCount statements (current value: %d)", nil),
	ErrViewRecursive:                            mysql.Message("`%-.192s`.`%-.192s` contains view recursion", nil),
	ErrNonGroupingFieldUsed:                     mysql.Message("Non-grouping field '%-.192s' is used in %-.64s clause", nil),
	ErrTableCantHandleSpkeys:                    mysql.Message("The used table type doesn't support SPATIAL indexes", nil),
	ErrNoTriggersOnSystemSchema:                 mysql.Message("Triggers can not be created on system tables", nil),
	ErrRemovedSpaces:                            mysql.Message("Leading spaces are removed from name '%s'", nil),
	ErrAutoincReadFailed:                        mysql.Message("Failed to read auto-increment value from storage engine", nil),
	ErrUsername:                                 mysql.Message("user name", nil),
	ErrHostname:                                 mysql.Message("host name", nil),
	ErrWrongStringLength:                        mysql.Message("String '%-.70s' is too long for %s (should be no longer than %d)", nil),
	ErrNonInsertableTable:                       mysql.Message("The target table %-.100s of the %s is not insertable-into", nil),
	ErrAdminWrongMrgTable:                       mysql.Message("Table '%-.64s' is differently defined or of non-MyISAM type or doesn't exist", nil),
	ErrTooHighLevelOfNestingForSelect:           mysql.Message("Too high level of nesting for select", nil),
	ErrNameBecomesEmpty:                         mysql.Message("Name '%-.64s' has become ''", nil),
	ErrAmbiguousFieldTerm:                       mysql.Message("First character of the FIELDS TERMINATED string is ambiguous; please use non-optional and non-empty FIELDS ENCLOSED BY", nil),
	ErrForeignServerExists:                      mysql.Message("The foreign server, %s, you are trying to create already exists.", nil),
	ErrForeignServerDoesntExist:                 mysql.Message("The foreign server name you are trying to reference does not exist. Data source :  %-.64s", nil),
	ErrIllegalHaCreateOption:                    mysql.Message("Table storage engine '%-.64s' does not support the create option '%.64s'", nil),
	ErrPartitionRequiresValues:                  mysql.Message("Syntax : %-.64s PARTITIONING requires definition of VALUES %-.64s for each partition", nil),
	ErrPartitionWrongValues:                     mysql.Message("Only %-.64s PARTITIONING can use VALUES %-.64s in partition definition", []int{1}),
	ErrPartitionMaxvalue:                        mysql.Message("MAXVALUE can only be used in last partition definition", nil),
	ErrPartitionSubpartition:                    mysql.Message("Subpartitions can only be hash partitions and by key", nil),
	ErrPartitionSubpartMix:                      mysql.Message("Must define subpartitions on all partitions if on one partition", nil),
	ErrPartitionWrongNoPart:                     mysql.Message("Wrong number of partitions defined, mismatch with previous setting", nil),
	ErrPartitionWrongNoSubpart:                  mysql.Message("Wrong number of subpartitions defined, mismatch with previous setting", nil),
	ErrWrongExprInPartitionFunc:                 mysql.Message("Constant, random or timezone-dependent expressions in (sub)partitioning function are not allowed", nil),
	ErrNoConstExprInRangeOrList:                 mysql.Message("Expression in RANGE/LIST VALUES must be constant", nil),
	ErrFieldNotFoundPart:                        mysql.Message("Field in list of fields for partition function not found in table", nil),
	ErrListOfFieldsOnlyInHash:                   mysql.Message("List of fields is only allowed in KEY partitions", nil),
	ErrInconsistentPartitionInfo:                mysql.Message("The partition info in the frm file is not consistent with what can be written into the frm file", nil),
	ErrPartitionFuncNotAllowed:                  mysql.Message("The %-.192s function returns the wrong type", nil),
	ErrPartitionsMustBeDefined:                  mysql.Message("For %-.64s partitions each partition must be defined", nil),
	ErrRangeNotIncreasing:                       mysql.Message("VALUES LESS THAN value must be strictly increasing for each partition", nil),
	ErrInconsistentTypeOfFunctions:              mysql.Message("VALUES value must be of same type as partition function", nil),
	ErrMultipleDefConstInListPart:               mysql.Message("Multiple definition of same constant in list partitioning", nil),
	ErrPartitionEntry:                           mysql.Message("Partitioning can not be used stand-alone in query", nil),
	ErrMixHandler:                               mysql.Message("The mix of handlers in the partitions is not allowed in this version of MySQL", nil),
	ErrPartitionNotDefined:                      mysql.Message("For the partitioned engine it is necessary to define all %-.64s", nil),
	ErrTooManyPartitions:                        mysql.Message("Too many partitions (including subpartitions) were defined", nil),
	ErrSubpartition:                             mysql.Message("It is only possible to mix RANGE/LIST partitioning with HASH/KEY partitioning for subpartitioning", nil),
	ErrCantCreateHandlerFile:                    mysql.Message("Failed to create specific handler file", nil),
	ErrBlobFieldInPartFunc:                      mysql.Message("A BLOB field is not allowed in partition function", nil),
	ErrUniqueKeyNeedAllFieldsInPf:               mysql.Message("A %-.192s must include all columns in the table's partitioning function", nil),
	ErrNoParts:                                  mysql.Message("Number of %-.64s = 0 is not an allowed value", []int{0}),
	ErrPartitionMgmtOnNonpartitioned:            mysql.Message("Partition management on a not partitioned table is not possible", nil),
	ErrForeignKeyOnPartitioned:                  mysql.Message("Foreign key clause is not yet supported in conjunction with partitioning", nil),
	ErrDropPartitionNonExistent:                 mysql.Message("Error in list of partitions to %-.64s", nil),
	ErrDropLastPartition:                        mysql.Message("Cannot remove all partitions, use DROP TABLE instead", nil),
	ErrCoalesceOnlyOnHashPartition:              mysql.Message("COALESCE PARTITION can only be used on HASH/KEY partitions", nil),
	ErrReorgHashOnlyOnSameNo:                    mysql.Message("REORGANIZE PARTITION can only be used to reorganize partitions not to change their numbers", nil),
	ErrReorgNoParam:                             mysql.Message("REORGANIZE PARTITION without parameters can only be used on auto-partitioned tables using HASH PARTITIONs", nil),
	ErrOnlyOnRangeListPartition:                 mysql.Message("%-.64s PARTITION can only be used on RANGE/LIST partitions", nil),
	ErrAddPartitionSubpart:                      mysql.Message("Trying to Add partition(s) with wrong number of subpartitions", nil),
	ErrAddPartitionNoNewPartition:               mysql.Message("At least one partition must be added", nil),
	ErrCoalescePartitionNoPartition:             mysql.Message("At least one partition must be coalesced", nil),
	ErrReorgPartitionNotExist:                   mysql.Message("More partitions to reorganize than there are partitions", nil),
	ErrSameNamePartition:                        mysql.Message("Duplicate partition name %-.192s", nil),
	ErrNoBinlog:                                 mysql.Message("It is not allowed to shut off binlog on this command", nil),
	ErrConsecutiveReorgPartitions:               mysql.Message("When reorganizing a set of partitions they must be in consecutive order", nil),
	ErrReorgOutsideRange:                        mysql.Message("Reorganize of range partitions cannot change total ranges except for last partition where it can extend the range", nil),
	ErrPartitionFunctionFailure:                 mysql.Message("Partition function not supported in this version for this handler", nil),
	ErrPartState:                                mysql.Message("Partition state cannot be defined from CREATE/ALTER TABLE", nil),
	ErrLimitedPartRange:                         mysql.Message("The %-.64s handler only supports 32 bit integers in VALUES", nil),
	ErrPluginIsNotLoaded:                        mysql.Message("Plugin '%-.192s' is not loaded", nil),
	ErrWrongValue:                               mysql.Message("Incorrect %-.32s value: '%-.128s'", []int{1}),
	ErrNoPartitionForGivenValue:                 mysql.Message("Table has no partition for value %-.64s", []int{0}),
	ErrFilegroupOptionOnlyOnce:                  mysql.Message("It is not allowed to specify %s more than once", nil),
	ErrCreateFilegroupFailed:                    mysql.Message("Failed to create %s", nil),
	ErrDropFilegroupFailed:                      mysql.Message("Failed to drop %s", nil),
	ErrTablespaceAutoExtend:                     mysql.Message("The handler doesn't support autoextend of tablespaces", nil),
	ErrWrongSizeNumber:                          mysql.Message("A size parameter was incorrectly specified, either number or on the form 10M", nil),
	ErrSizeOverflow:                             mysql.Message("The size number was correct but we don't allow the digit part to be more than 2 billion", nil),
	ErrAlterFilegroupFailed:                     mysql.Message("Failed to alter: %s", nil),
	ErrBinlogRowLoggingFailed:                   mysql.Message("Writing one row to the row-based binary log failed", nil),
	ErrEventAlreadyExists:                       mysql.Message("Event '%-.192s' already exists", nil),
	ErrEventStoreFailed:                         mysql.Message("Failed to store event %s. Error code %d from storage engine.", nil),
	ErrEventDoesNotExist:                        mysql.Message("Unknown event '%-.192s'", nil),
	ErrEventCantAlter:                           mysql.Message("Failed to alter event '%-.192s'", nil),
	ErrEventDropFailed:                          mysql.Message("Failed to drop %s", nil),
	ErrEventIntervalNotPositiveOrTooBig:         mysql.Message("INTERVAL is either not positive or too big", nil),
	ErrEventEndsBeforeStarts:                    mysql.Message("ENDS is either invalid or before STARTS", nil),
	ErrEventExecTimeInThePast:                   mysql.Message("Event execution time is in the past. Event has been disabled", nil),
	ErrEventOpenTableFailed:                     mysql.Message("Failed to open mysql.event", nil),
	ErrEventNeitherMExprNorMAt:                  mysql.Message("No datetime expression provided", nil),
	ErrObsoleteColCountDoesntMatchCorrupted:     mysql.Message("Column count of mysql.%s is wrong. Expected %d, found %d. The table is probably corrupted", nil),
	ErrObsoleteCannotLoadFromTable:              mysql.Message("Cannot load from mysql.%s. The table is probably corrupted", nil),
	ErrEventCannotDelete:                        mysql.Message("Failed to delete the event from mysql.event", nil),
	ErrEventCompile:                             mysql.Message("Error during compilation of event's body", nil),
	ErrEventSameName:                            mysql.Message("Same old and new event name", nil),
	ErrEventDataTooLong:                         mysql.Message("Data for column '%s' too long", nil),
	ErrDropIndexFk:                              mysql.Message("Cannot drop index '%-.192s': needed in a foreign key constraint", nil),
	ErrWarnDeprecatedSyntaxWithVer:              mysql.Message("The syntax '%s' is deprecated and will be removed in MySQL %s. Please use %s instead", nil),
	ErrCantWriteLockLogTable:                    mysql.Message("You can't write-lock a log table. Only read access is possible", nil),
	ErrCantLockLogTable:                         mysql.Message("You can't use locks with log tables.", nil),
	ErrForeignDuplicateKeyOldUnused:             mysql.Message("Upholding foreign key constraints for table '%.192s', entry '%-.192s', key %d would lead to a duplicate entry", nil),
	ErrColCountDoesntMatchPleaseUpdate:          mysql.Message("Column count of mysql.%s is wrong. Expected %d, found %d. Created with MySQL %d, now running %d. Please use mysqlUpgrade to fix this error.", nil),
	ErrTempTablePreventsSwitchOutOfRbr:          mysql.Message("Cannot switch out of the row-based binary log format when the session has open temporary tables", nil),
	ErrStoredFunctionPreventsSwitchBinlogFormat: mysql.Message("Cannot change the binary logging format inside a stored function or trigger", nil),
	ErrNdbCantSwitchBinlogFormat:                mysql.Message("The NDB cluster engine does not support changing the binlog format on the fly yet", nil),
	ErrPartitionNoTemporary:                     mysql.Message("Cannot create temporary table with partitions", nil),
	ErrPartitionConstDomain:                     mysql.Message("Partition constant is out of partition function domain", nil),
	ErrPartitionFunctionIsNotAllowed:            mysql.Message("This partition function is not allowed", nil),
	ErrDdlLog:                                   mysql.Message("Error in DDL log", nil),
	ErrNullInValuesLessThan:                     mysql.Message("Not allowed to use NULL value in VALUES LESS THAN", nil),
	ErrWrongPartitionName:                       mysql.Message("Incorrect partition name", nil),
	ErrCantChangeTxCharacteristics:              mysql.Message("Transaction characteristics can't be changed while a transaction is in progress", nil),
	ErrDupEntryAutoincrementCase:                mysql.Message("ALTER TABLE causes autoIncrement resequencing, resulting in duplicate entry '%-.192s' for key '%-.192s'", nil),
	ErrEventModifyQueue:                         mysql.Message("Internal scheduler error %d", nil),
	ErrEventSetVar:                              mysql.Message("Error during starting/stopping of the scheduler. Error code %d", nil),
	ErrPartitionMerge:                           mysql.Message("Engine cannot be used in partitioned tables", nil),
	ErrCantActivateLog:                          mysql.Message("Cannot activate '%-.64s' log", nil),
	ErrRbrNotAvailable:                          mysql.Message("The server was not built with row-based replication", nil),
	ErrBase64Decode:                             mysql.Message("Decoding of base64 string failed", nil),
	ErrEventRecursionForbidden:                  mysql.Message("Recursion of EVENT DDL statements is forbidden when body is present", nil),
	ErrEventsDB:                                 mysql.Message("Cannot proceed because system tables used by Event Scheduler were found damaged at server start", nil),
	ErrOnlyIntegersAllowed:                      mysql.Message("Only integers allowed as number here", nil),
	ErrUnsuportedLogEngine:                      mysql.Message("This storage engine cannot be used for log tables\"", nil),
	ErrBadLogStatement:                          mysql.Message("You cannot '%s' a log table if logging is enabled", nil),
	ErrCantRenameLogTable:                       mysql.Message("Cannot rename '%s'. When logging enabled, rename to/from log table must rename two tables: the log table to an archive table and another table back to '%s'", nil),
	ErrWrongParamcountToNativeFct:               mysql.Message("Incorrect parameter count in the call to native function '%-.192s'", nil),
	ErrWrongParametersToNativeFct:               mysql.Message("Incorrect parameters in the call to native function '%-.192s'", nil),
	ErrWrongParametersToStoredFct:               mysql.Message("Incorrect parameters in the call to stored function '%-.192s'", nil),
	ErrNativeFctNameCollision:                   mysql.Message("This function '%-.192s' has the same name as a native function", nil),
	ErrDupEntryWithKeyName:                      mysql.Message("Duplicate entry '%-.64s' for key '%-.192s'", nil),
	ErrBinlogPurgeEmFile:                        mysql.Message("Too many files opened, please execute the command again", nil),
	ErrEventCannotCreateInThePast:               mysql.Message("Event execution time is in the past and ON COMPLETION NOT PRESERVE is set. The event was dropped immediately after creation.", nil),
	ErrEventCannotAlterInThePast:                mysql.Message("Event execution time is in the past and ON COMPLETION NOT PRESERVE is set. The event was not changed. Specify a time in the future.", nil),
	ErrNoPartitionForGivenValueSilent:           mysql.Message("Table has no partition for some existing values", nil),
	ErrBinlogUnsafeStatement:                    mysql.Message("Unsafe statement written to the binary log using statement format since BINLOGFORMAT = STATEMENT. %s", nil),
	ErrBinlogLoggingImpossible:                  mysql.Message("Binary logging not possible. Message: %s", nil),
	ErrViewNoCreationCtx:                        mysql.Message("View `%-.64s`.`%-.64s` has no creation context", nil),
	ErrViewInvalidCreationCtx:                   mysql.Message("Creation context of view `%-.64s`.`%-.64s' is invalid", nil),
	ErrSrInvalidCreationCtx:                     mysql.Message("Creation context of stored routine `%-.64s`.`%-.64s` is invalid", nil),
	ErrTrgCorruptedFile:                         mysql.Message("Corrupted TRG file for table `%-.64s`.`%-.64s`", nil),
	ErrTrgNoCreationCtx:                         mysql.Message("Triggers for table `%-.64s`.`%-.64s` have no creation context", nil),
	ErrTrgInvalidCreationCtx:                    mysql.Message("Trigger creation context of table `%-.64s`.`%-.64s` is invalid", nil),
	ErrEventInvalidCreationCtx:                  mysql.Message("Creation context of event `%-.64s`.`%-.64s` is invalid", nil),
	ErrTrgCantOpenTable:                         mysql.Message("Cannot open table for trigger `%-.64s`.`%-.64s`", nil),
	ErrCantCreateSroutine:                       mysql.Message("Cannot create stored routine `%-.64s`. Check warnings", nil),
	ErrNoFormatDescriptionEventBeforeBinlogStatement:         mysql.Message("The BINLOG statement of type `%s` was not preceded by a format description BINLOG statement.", nil),
	ErrLoadDataInvalidColumn:                                 mysql.Message("Invalid column reference (%-.64s) in LOAD DATA", nil),
	ErrLogPurgeNoFile:                                        mysql.Message("Being purged log %s was not found", nil),
	ErrXaRbtimeout:                                           mysql.Message("XARBTIMEOUT: Transaction branch was rolled back: took too long", nil),
	ErrXaRbdeadlock:                                          mysql.Message("XARBDEADLOCK: Transaction branch was rolled back: deadlock was detected", nil),
	ErrNeedReprepare:                                         mysql.Message("Prepared statement needs to be re-prepared", nil),
	ErrDelayedNotSupported:                                   mysql.Message("DELAYED option not supported for table '%-.192s'", nil),
	WarnOptionIgnored:                                        mysql.Message("<%-.64s> option ignored", nil),
	WarnPluginDeleteBuiltin:                                  mysql.Message("Built-in plugins cannot be deleted", nil),
	WarnPluginBusy:                                           mysql.Message("Plugin is busy and will be uninstalled on shutdown", nil),
	ErrVariableIsReadonly:                                    mysql.Message("%s variable '%s' is read-only. Use SET %s to assign the value", nil),
	ErrWarnEngineTransactionRollback:                         mysql.Message("Storage engine %s does not support rollback for this statement. Transaction rolled back and must be restarted", nil),
	ErrNdbReplicationSchema:                                  mysql.Message("Bad schema for mysql.ndbReplication table. Message: %-.64s", nil),
	ErrConflictFnParse:                                       mysql.Message("Error in parsing conflict function. Message: %-.64s", nil),
	ErrExceptionsWrite:                                       mysql.Message("Write to exceptions table failed. Message: %-.128s\"", nil),
	ErrTooLongTableComment:                                   mysql.Message("Comment for table '%-.64s' is too long (max = %d)", nil),
	ErrTooLongFieldComment:                                   mysql.Message("Comment for field '%-.64s' is too long (max = %d)", nil),
	ErrFuncInexistentNameCollision:                           mysql.Message("FUNCTION %s does not exist. Check the 'Function Name Parsing and Resolution' section in the Reference Manual", nil),
	ErrDatabaseName:                                          mysql.Message("Database", nil),
	ErrTableName:                                             mysql.Message("Table", nil),
	ErrPartitionName:                                         mysql.Message("Partition", nil),
	ErrSubpartitionName:                                      mysql.Message("Subpartition", nil),
	ErrTemporaryName:                                         mysql.Message("Temporary", nil),
	ErrRenamedName:                                           mysql.Message("Renamed", nil),
	ErrTooManyConcurrentTrxs:                                 mysql.Message("Too many active concurrent transactions", nil),
	WarnNonASCIISeparatorNotImplemented:                      mysql.Message("Non-ASCII separator arguments are not fully supported", nil),
	ErrDebugSyncTimeout:                                      mysql.Message("debug sync point wait timed out", nil),
	ErrDebugSyncHitLimit:                                     mysql.Message("debug sync point hit limit reached", nil),
	ErrDupSignalSet:                                          mysql.Message("Duplicate condition information item '%s'", nil),
	ErrSignalWarn:                                            mysql.Message("Unhandled user-defined warning condition", nil),
	ErrSignalNotFound:                                        mysql.Message("Unhandled user-defined not found condition", nil),
	ErrSignalException:                                       mysql.Message("Unhandled user-defined exception condition", nil),
	ErrResignalWithoutActiveHandler:                          mysql.Message("RESIGNAL when handler not active", nil),
	ErrSignalBadConditionType:                                mysql.Message("SIGNAL/RESIGNAL can only use a CONDITION defined with SQLSTATE", nil),
	WarnCondItemTruncated:                                    mysql.Message("Data truncated for condition item '%s'", nil),
	ErrCondItemTooLong:                                       mysql.Message("Data too long for condition item '%s'", nil),
	ErrUnknownLocale:                                         mysql.Message("Unknown locale: '%-.64s'", nil),
	ErrQueryCacheDisabled:                                    mysql.Message("Query cache is disabled; restart the server with queryCacheType=1 to enable it", nil),
	ErrSameNamePartitionField:                                mysql.Message("Duplicate partition field name '%-.192s'", nil),
	ErrPartitionColumnList:                                   mysql.Message("Inconsistency in usage of column lists for partitioning", nil),
	ErrWrongTypeColumnValue:                                  mysql.Message("Partition column values of incorrect type", nil),
	ErrTooManyPartitionFuncFields:                            mysql.Message("Too many fields in '%-.192s'", nil),
	ErrMaxvalueInValuesIn:                                    mysql.Message("Cannot use MAXVALUE as value in VALUES IN", nil),
	ErrTooManyValues:                                         mysql.Message("Cannot have more than one value for this type of %-.64s partitioning", nil),
	ErrRowSinglePartitionField:                               mysql.Message("Row expressions in VALUES IN only allowed for multi-field column partitioning", nil),
	ErrFieldTypeNotAllowedAsPartitionField:                   mysql.Message("Field '%-.192s' is of a not allowed type for this type of partitioning", nil),
	ErrPartitionFieldsTooLong:                                mysql.Message("The total length of the partitioning fields is too large", nil),
	ErrBinlogRowEngineAndStmtEngine:                          mysql.Message("Cannot execute statement: impossible to write to binary log since both row-incapable engines and statement-incapable engines are involved.", nil),
	ErrBinlogRowModeAndStmtEngine:                            mysql.Message("Cannot execute statement: impossible to write to binary log since BINLOGFORMAT = ROW and at least one table uses a storage engine limited to statement-based logging.", nil),
	ErrBinlogUnsafeAndStmtEngine:                             mysql.Message("Cannot execute statement: impossible to write to binary log since statement is unsafe, storage engine is limited to statement-based logging, and BINLOGFORMAT = MIXED. %s", nil),
	ErrBinlogRowInjectionAndStmtEngine:                       mysql.Message("Cannot execute statement: impossible to write to binary log since statement is in row format and at least one table uses a storage engine limited to statement-based logging.", nil),
	ErrBinlogStmtModeAndRowEngine:                            mysql.Message("Cannot execute statement: impossible to write to binary log since BINLOGFORMAT = STATEMENT and at least one table uses a storage engine limited to row-based logging.%s", nil),
	ErrBinlogRowInjectionAndStmtMode:                         mysql.Message("Cannot execute statement: impossible to write to binary log since statement is in row format and BINLOGFORMAT = STATEMENT.", nil),
	ErrBinlogMultipleEnginesAndSelfLoggingEngine:             mysql.Message("Cannot execute statement: impossible to write to binary log since more than one engine is involved and at least one engine is self-logging.", nil),
	ErrBinlogUnsafeLimit:                                     mysql.Message("The statement is unsafe because it uses a LIMIT clause. This is unsafe because the set of rows included cannot be predicted.", nil),
	ErrBinlogUnsafeInsertDelayed:                             mysql.Message("The statement is unsafe because it uses INSERT DELAYED. This is unsafe because the times when rows are inserted cannot be predicted.", nil),
	ErrBinlogUnsafeAutoincColumns:                            mysql.Message("Statement is unsafe because it invokes a trigger or a stored function that inserts into an AUTOINCREMENT column. Inserted values cannot be logged correctly.", nil),
	ErrBinlogUnsafeNontransAfterTrans:                        mysql.Message("Statement is unsafe because it accesses a non-transactional table after accessing a transactional table within the same transaction.", nil),
	ErrMessageAndStatement:                                   mysql.Message("%s Statement: %s", nil),
	ErrInsideTransactionPreventsSwitchBinlogFormat:           mysql.Message("Cannot modify @@session.binlogFormat inside a transaction", nil),
	ErrPathLength:                                            mysql.Message("The path specified for %.64s is too long.", nil),
	ErrWarnDeprecatedSyntaxNoReplacement:                     mysql.Message("'%s' is deprecated and will be removed in a future release.", nil),
	ErrWrongNativeTableStructure:                             mysql.Message("Native table '%-.64s'.'%-.64s' has the wrong structure", nil),
	ErrWrongPerfSchemaUsage:                                  mysql.Message("Invalid performanceSchema usage.", nil),
	ErrWarnISSkippedTable:                                    mysql.Message("Table '%s'.'%s' was skipped since its definition is being modified by concurrent DDL statement", nil),
	ErrInsideTransactionPreventsSwitchBinlogDirect:           mysql.Message("Cannot modify @@session.binlogDirectNonTransactionalUpdates inside a transaction", nil),
	ErrStoredFunctionPreventsSwitchBinlogDirect:              mysql.Message("Cannot change the binlog direct flag inside a stored function or trigger", nil),
	ErrSpatialMustHaveGeomCol:                                mysql.Message("A SPATIAL index may only contain a geometrical type column", nil),
	ErrTooLongIndexComment:                                   mysql.Message("Comment for index '%-.64s' is too long (max = %d)", nil),
	ErrLockAborted:                                           mysql.Message("Wait on a lock was aborted due to a pending exclusive lock", nil),
	ErrDataOutOfRange:                                        mysql.Message("%s value is out of range in '%s'", []int{1}),
	ErrWrongSpvarTypeInLimit:                                 mysql.Message("A variable of a non-integer based type in LIMIT clause", nil),
	ErrBinlogUnsafeMultipleEnginesAndSelfLoggingEngine:       mysql.Message("Mixing self-logging and non-self-logging engines in a statement is unsafe.", nil),
	ErrBinlogUnsafeMixedStatement:                            mysql.Message("Statement accesses nontransactional table as well as transactional or temporary table, and writes to any of them.", nil),
	ErrInsideTransactionPreventsSwitchSQLLogBin:              mysql.Message("Cannot modify @@session.sqlLogBin inside a transaction", nil),
	ErrStoredFunctionPreventsSwitchSQLLogBin:                 mysql.Message("Cannot change the sqlLogBin inside a stored function or trigger", nil),
	ErrFailedReadFromParFile:                                 mysql.Message("Failed to read from the .par file", nil),
	ErrValuesIsNotIntType:                                    mysql.Message("VALUES value for partition '%-.64s' must have type INT", nil),
	ErrAccessDeniedNoPassword:                                mysql.Message("Access denied for user '%-.48s'@'%-.64s'", nil),
	ErrSetPasswordAuthPlugin:                                 mysql.Message("SET PASSWORD has no significance for users authenticating via plugins", nil),
	ErrGrantPluginUserExists:                                 mysql.Message("GRANT with IDENTIFIED WITH is illegal because the user %-.*s already exists", nil),
	ErrTruncateIllegalFk:                                     mysql.Message("Cannot truncate a table referenced in a foreign key constraint (%.192s)", nil),
	ErrPluginIsPermanent:                                     mysql.Message("Plugin '%s' is forcePlusPermanent and can not be unloaded", nil),
	ErrStmtCacheFull:                                         mysql.Message("Multi-row statements required more than 'maxBinlogStmtCacheSize' bytes of storage; increase this mysqld variable and try again", nil),
	ErrMultiUpdateKeyConflict:                                mysql.Message("Primary key/partition key update is not allowed since the table is updated both as '%-.192s' and '%-.192s'.", nil),
	ErrTableNeedsRebuild:                                     mysql.Message("Table rebuild required. Please do \"ALTER TABLE `%-.32s` FORCE\" or dump/reload to fix it!", nil),
	WarnOptionBelowLimit:                                     mysql.Message("The value of '%s' should be no less than the value of '%s'", nil),
	ErrIndexColumnTooLong:                                    mysql.Message("Index column size too large. The maximum column size is %d bytes.", nil),
	ErrErrorInTriggerBody:                                    mysql.Message("Trigger '%-.64s' has an error in its body: '%-.256s'", nil),
	ErrErrorInUnknownTriggerBody:                             mysql.Message("Unknown trigger has an error in its body: '%-.256s'", nil),
	ErrIndexCorrupt:                                          mysql.Message("Index %s is corrupted", nil),
	ErrUndoRecordTooBig:                                      mysql.Message("Undo log record is too big.", nil),
	ErrPluginNoUninstall:                                     mysql.Message("Plugin '%s' is marked as not dynamically uninstallable. You have to stop the server to uninstall it.", nil),
	ErrPluginNoInstall:                                       mysql.Message("Plugin '%s' is marked as not dynamically installable. You have to stop the server to install it.", nil),
	ErrBinlogUnsafeInsertTwoKeys:                             mysql.Message("INSERT... ON DUPLICATE KEY UPDATE  on a table with more than one UNIQUE KEY is unsafe", nil),
	ErrTableInFkCheck:                                        mysql.Message("Table is being used in foreign key check.", nil),
	ErrUnsupportedEngine:                                     mysql.Message("Storage engine '%s' does not support system tables. [%s.%s]", nil),
	ErrBinlogUnsafeAutoincNotFirst:                           mysql.Message("INSERT into autoincrement field which is not the first part in the composed primary key is unsafe.", nil),
	ErrCannotLoadFromTableV2:                                 mysql.Message("Cannot load from %s.%s. The table is probably corrupted", nil),
	ErrOnlyFdAndRbrEventsAllowedInBinlogStatement:            mysql.Message("Only FormatDescriptionLogEvent and row events are allowed in BINLOG statements (but %s was provided)", nil),
	ErrPartitionExchangeDifferentOption:                      mysql.Message("Non matching attribute '%-.64s' between partition and table", nil),
	ErrPartitionExchangePartTable:                            mysql.Message("Table to exchange with partition is partitioned: '%-.64s'", nil),
	ErrPartitionExchangeTempTable:                            mysql.Message("Table to exchange with partition is temporary: '%-.64s'", nil),
	ErrPartitionInsteadOfSubpartition:                        mysql.Message("Subpartitioned table, use subpartition instead of partition", nil),
	ErrUnknownPartition:                                      mysql.Message("Unknown partition '%-.64s' in table '%-.64s'", nil),
	ErrTablesDifferentMetadata:                               mysql.Message("Tables have different definitions", nil),
	ErrRowDoesNotMatchPartition:                              mysql.Message("Found a row that does not match the partition", nil),
	ErrBinlogCacheSizeGreaterThanMax:                         mysql.Message("Option binlogCacheSize (%d) is greater than maxBinlogCacheSize (%d); setting binlogCacheSize equal to maxBinlogCacheSize.", nil),
	ErrWarnIndexNotApplicable:                                mysql.Message("Cannot use %-.64s access on index '%-.64s' due to type or collation conversion on field '%-.64s'", nil),
	ErrPartitionExchangeForeignKey:                           mysql.Message("Table to exchange with partition has foreign key references: '%-.64s'", nil),
	ErrNoSuchKeyValue:                                        mysql.Message("Key value '%-.192s' was not found in table '%-.192s.%-.192s'", nil),
	ErrRplInfoDataTooLong:                                    mysql.Message("Data for column '%s' too long", nil),
	ErrNetworkReadEventChecksumFailure:                       mysql.Message("Replication event checksum verification failed while reading from network.", nil),
	ErrBinlogReadEventChecksumFailure:                        mysql.Message("Replication event checksum verification failed while reading from a log file.", nil),
	ErrBinlogStmtCacheSizeGreaterThanMax:                     mysql.Message("Option binlogStmtCacheSize (%d) is greater than maxBinlogStmtCacheSize (%d); setting binlogStmtCacheSize equal to maxBinlogStmtCacheSize.", nil),
	ErrCantUpdateTableInCreateTableSelect:                    mysql.Message("Can't update table '%-.192s' while '%-.192s' is being created.", nil),
	ErrPartitionClauseOnNonpartitioned:                       mysql.Message("PARTITION () clause on non partitioned table", nil),
	ErrRowDoesNotMatchGivenPartitionSet:                      mysql.Message("Found a row not matching the given partition set", nil),
	ErrNoSuchPartitionunused:                                 mysql.Message("partition '%-.64s' doesn't exist", nil),
	ErrChangeRplInfoRepositoryFailure:                        mysql.Message("Failure while changing the type of replication repository: %s.", nil),
	ErrWarningNotCompleteRollbackWithCreatedTempTable:        mysql.Message("The creation of some temporary tables could not be rolled back.", nil),
	ErrWarningNotCompleteRollbackWithDroppedTempTable:        mysql.Message("Some temporary tables were dropped, but these operations could not be rolled back.", nil),
	ErrMtsUpdatedDBsGreaterMax:                               mysql.Message("The number of modified databases exceeds the maximum %d; the database names will not be included in the replication event metadata.", nil),
	ErrMtsCantParallel:                                       mysql.Message("Cannot execute the current event group in the parallel mode. Encountered event %s, relay-log name %s, position %s which prevents execution of this event group in parallel mode. Reason: %s.", nil),
	ErrMtsInconsistentData:                                   mysql.Message("%s", nil),
	ErrFulltextNotSupportedWithPartitioning:                  mysql.Message("FULLTEXT index is not supported for partitioned tables.", nil),
	ErrDaInvalidConditionNumber:                              mysql.Message("Invalid condition number", nil),
	ErrInsecurePlainText:                                     mysql.Message("Sending passwords in plain text without SSL/TLS is extremely insecure.", nil),
	ErrForeignDuplicateKeyWithChildInfo:                      mysql.Message("Foreign key constraint for table '%.192s', record '%-.192s' would lead to a duplicate entry in table '%.192s', key '%.192s'", nil),
	ErrForeignDuplicateKeyWithoutChildInfo:                   mysql.Message("Foreign key constraint for table '%.192s', record '%-.192s' would lead to a duplicate entry in a child table", nil),
	ErrTableHasNoFt:                                          mysql.Message("The table does not have FULLTEXT index to support this query", nil),
	ErrVariableNotSettableInSfOrTrigger:                      mysql.Message("The system variable %.200s cannot be set in stored functions or triggers.", nil),
	ErrVariableNotSettableInTransaction:                      mysql.Message("The system variable %.200s cannot be set when there is an ongoing transaction.", nil),
	ErrGtidNextIsNotInGtidNextList:                           mysql.Message("The system variable @@SESSION.GTIDNEXT has the value %.200s, which is not listed in @@SESSION.GTIDNEXTLIST.", nil),
	ErrCantChangeGtidNextInTransactionWhenGtidNextListIsNull: mysql.Message("When @@SESSION.GTIDNEXTLIST == NULL, the system variable @@SESSION.GTIDNEXT cannot change inside a transaction.", nil),
	ErrSetStatementCannotInvokeFunction:                      mysql.Message("The statement 'SET %.200s' cannot invoke a stored function.", nil),
	ErrGtidNextCantBeAutomaticIfGtidNextListIsNonNull:        mysql.Message("The system variable @@SESSION.GTIDNEXT cannot be 'AUTOMATIC' when @@SESSION.GTIDNEXTLIST is non-NULL.", nil),
	ErrSkippingLoggedTransaction:                             mysql.Message("Skipping transaction %.200s because it has already been executed and logged.", nil),
	ErrMalformedGtidSetSpecification:                         mysql.Message("Malformed GTID set specification '%.200s'.", nil),
	ErrMalformedGtidSetEncoding:                              mysql.Message("Malformed GTID set encoding.", nil),
	ErrMalformedGtidSpecification:                            mysql.Message("Malformed GTID specification '%.200s'.", nil),
	ErrGnoExhausted:                                          mysql.Message("Impossible to generate Global Transaction Identifier: the integer component reached the maximal value. Restart the server with a new serverUuid.", nil),
	ErrCantDoImplicitCommitInTrxWhenGtidNextIsSet:            mysql.Message("Cannot execute statements with implicit commit inside a transaction when @@SESSION.GTIDNEXT != AUTOMATIC or @@SESSION.GTIDNEXTLIST != NULL.", nil),
	ErrGtidMode2Or3RequiresEnforceGtidConsistencyOn:          mysql.Message("@@GLOBAL.GTIDMODE = ON or UPGRADESTEP2 requires @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1.", nil),
	ErrCantSetGtidNextToGtidWhenGtidModeIsOff:                mysql.Message("@@SESSION.GTIDNEXT cannot be set to UUID:NUMBER when @@GLOBAL.GTIDMODE = OFF.", nil),
	ErrCantSetGtidNextToAnonymousWhenGtidModeIsOn:            mysql.Message("@@SESSION.GTIDNEXT cannot be set to ANONYMOUS when @@GLOBAL.GTIDMODE = ON.", nil),
	ErrCantSetGtidNextListToNonNullWhenGtidModeIsOff:         mysql.Message("@@SESSION.GTIDNEXTLIST cannot be set to a non-NULL value when @@GLOBAL.GTIDMODE = OFF.", nil),
	ErrFoundGtidEventWhenGtidModeIsOff:                       mysql.Message("Found a GtidLogEvent or PreviousGtidsLogEvent when @@GLOBAL.GTIDMODE = OFF.", nil),
	ErrGtidUnsafeNonTransactionalTable:                       mysql.Message("When @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1, updates to non-transactional tables can only be done in either autocommitted statements or single-statement transactions, and never in the same statement as updates to transactional tables.", nil),
	ErrGtidUnsafeCreateSelect:                                mysql.Message("CREATE TABLE ... SELECT is forbidden when @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1.", nil),
	ErrGtidUnsafeCreateDropTemporaryTableInTransaction:       mysql.Message("When @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1, the statements CREATE TEMPORARY TABLE and DROP TEMPORARY TABLE can be executed in a non-transactional context only, and require that AUTOCOMMIT = 1.", nil),
	ErrGtidModeCanOnlyChangeOneStepAtATime:                   mysql.Message("The value of @@GLOBAL.GTIDMODE can only change one step at a time: OFF <-> UPGRADESTEP1 <-> UPGRADESTEP2 <-> ON. Also note that this value must be stepped up or down simultaneously on all servers; see the Manual for instructions.", nil),
	ErrCantSetGtidNextWhenOwningGtid:                         mysql.Message("@@SESSION.GTIDNEXT cannot be changed by a client that owns a GTID. The client owns %s. Ownership is released on COMMIT or ROLLBACK.", nil),
	ErrUnknownExplainFormat:                                  mysql.Message("Unknown EXPLAIN format name: '%s'", nil),
	ErrCantExecuteInReadOnlyTransaction:                      mysql.Message("Cannot execute statement in a READ ONLY transaction.", nil),
	ErrTooLongTablePartitionComment:                          mysql.Message("Comment for table partition '%-.64s' is too long (max = %d)", nil),
	ErrInnodbFtLimit:                                         mysql.Message("InnoDB presently supports one FULLTEXT index creation at a time", nil),
	ErrInnodbNoFtTempTable:                                   mysql.Message("Cannot create FULLTEXT index on temporary InnoDB table", nil),
	ErrInnodbFtWrongDocidColumn:                              mysql.Message("Column '%-.192s' is of wrong type for an InnoDB FULLTEXT index", nil),
	ErrInnodbFtWrongDocidIndex:                               mysql.Message("Index '%-.192s' is of wrong type for an InnoDB FULLTEXT index", nil),
	ErrInnodbOnlineLogTooBig:                                 mysql.Message("Creating index '%-.192s' required more than 'innodbOnlineAlterLogMaxSize' bytes of modification log. Please try again.", nil),
	ErrUnknownAlterAlgorithm:                                 mysql.Message("Unknown ALGORITHM '%s'", nil),
	ErrUnknownAlterLock:                                      mysql.Message("Unknown LOCK type '%s'", nil),
	ErrMtsResetWorkers:                                       mysql.Message("Cannot clean up worker info tables. Additional error messages can be found in the MySQL error log.", nil),
	ErrColCountDoesntMatchCorruptedV2:                        mysql.Message("Column count of %s.%s is wrong. Expected %d, found %d. The table is probably corrupted", nil),
	ErrDiscardFkChecksRunning:                                mysql.Message("There is a foreign key check running on table '%-.192s'. Cannot discard the table.", nil),
	ErrTableSchemaMismatch:                                   mysql.Message("Schema mismatch (%s)", nil),
	ErrTableInSystemTablespace:                               mysql.Message("Table '%-.192s' in system tablespace", nil),
	ErrIoRead:                                                mysql.Message("IO Read : (%d, %s) %s", nil),
	ErrIoWrite:                                               mysql.Message("IO Write : (%d, %s) %s", nil),
	ErrTablespaceMissing:                                     mysql.Message("Tablespace is missing for table '%-.192s'", nil),
	ErrTablespaceExists:                                      mysql.Message("Tablespace for table '%-.192s' exists. Please DISCARD the tablespace before IMPORT.", nil),
	ErrTablespaceDiscarded:                                   mysql.Message("Tablespace has been discarded for table '%-.192s'", nil),
	ErrInternal:                                              mysql.Message("Internal : %s", nil),
	ErrInnodbImport:                                          mysql.Message("ALTER TABLE '%-.192s' IMPORT TABLESPACE failed with error %d : '%s'", nil),
	ErrInnodbIndexCorrupt:                                    mysql.Message("Index corrupt: %s", nil),
	ErrInvalidYearColumnLength:                               mysql.Message("Supports only YEAR or YEAR(4) column", nil),
	ErrNotValidPassword:                                      mysql.Message("Your password does not satisfy the current policy requirements", nil),
	ErrMustChangePassword:                                    mysql.Message("You must SET PASSWORD before executing this statement", nil),
	ErrFkNoIndexChild:                                        mysql.Message("Failed to add the foreign key constaint. Missing index for constraint '%s' in the foreign table '%s'", nil),
	ErrFkNoIndexParent:                                       mysql.Message("Failed to add the foreign key constaint. Missing index for constraint '%s' in the referenced table '%s'", nil),
	ErrFkFailAddSystem:                                       mysql.Message("Failed to add the foreign key constraint '%s' to system tables", nil),
	ErrFkCannotOpenParent:                                    mysql.Message("Failed to open the referenced table '%s'", nil),
	ErrFkIncorrectOption:                                     mysql.Message("Failed to add the foreign key constraint on table '%s'. Incorrect options in FOREIGN KEY constraint '%s'", nil),
	ErrFkDupName:                                             mysql.Message("Duplicate foreign key constraint name '%s'", nil),
	ErrPasswordFormat:                                        mysql.Message("The password hash doesn't have the expected format. Check if the correct password algorithm is being used with the PASSWORD() function.", nil),
	ErrFkColumnCannotDrop:                                    mysql.Message("Cannot drop column '%-.192s': needed in a foreign key constraint '%-.192s'", nil),
	ErrFkColumnCannotDropChild:                               mysql.Message("Cannot drop column '%-.192s': needed in a foreign key constraint '%-.192s' of table '%-.192s'", nil),
	ErrFkColumnNotNull:                                       mysql.Message("Column '%-.192s' cannot be NOT NULL: needed in a foreign key constraint '%-.192s' SET NULL", nil),
	ErrDupIndex:                                              mysql.Message("Duplicate index '%-.64s' defined on the table '%-.64s.%-.64s'. This is deprecated and will be disallowed in a future release.", nil),
	ErrFkColumnCannotChange:                                  mysql.Message("Cannot change column '%-.192s': used in a foreign key constraint '%-.192s'", nil),
	ErrFkColumnCannotChangeChild:                             mysql.Message("Cannot change column '%-.192s': used in a foreign key constraint '%-.192s' of table '%-.192s'", nil),
	ErrFkCannotDeleteParent:                                  mysql.Message("Cannot delete rows from table which is parent in a foreign key constraint '%-.192s' of table '%-.192s'", nil),
	ErrMalformedPacket:                                       mysql.Message("Malformed communication packet.", nil),
	ErrReadOnlyMode:                                          mysql.Message("Running in read-only mode", nil),
	ErrVariableNotSettableInSp:                               mysql.Message("The system variable %.200s cannot be set in stored procedures.", nil),
	ErrCantSetGtidPurgedWhenGtidModeIsOff:                    mysql.Message("@@GLOBAL.GTIDPURGED can only be set when @@GLOBAL.GTIDMODE = ON.", nil),
	ErrCantSetGtidPurgedWhenGtidExecutedIsNotEmpty:           mysql.Message("@@GLOBAL.GTIDPURGED can only be set when @@GLOBAL.GTIDEXECUTED is empty.", nil),
	ErrCantSetGtidPurgedWhenOwnedGtidsIsNotEmpty:             mysql.Message("@@GLOBAL.GTIDPURGED can only be set when there are no ongoing transactions (not even in other clients).", nil),
	ErrGtidPurgedWasChanged:                                  mysql.Message("@@GLOBAL.GTIDPURGED was changed from '%s' to '%s'.", nil),
	ErrGtidExecutedWasChanged:                                mysql.Message("@@GLOBAL.GTIDEXECUTED was changed from '%s' to '%s'.", nil),
	ErrBinlogStmtModeAndNoReplTables:                         mysql.Message("Cannot execute statement: impossible to write to binary log since BINLOGFORMAT = STATEMENT, and both replicated and non replicated tables are written to.", nil),
	ErrAlterOperationNotSupported:                            mysql.Message("%s is not supported for this operation. Try %s.", nil),
	ErrAlterOperationNotSupportedReason:                      mysql.Message("%s is not supported. Reason: %s. Try %s.", nil),
	ErrAlterOperationNotSupportedReasonCopy:                  mysql.Message("COPY algorithm requires a lock", nil),
	ErrAlterOperationNotSupportedReasonPartition:             mysql.Message("Partition specific operations do not yet support LOCK/ALGORITHM", nil),
	ErrAlterOperationNotSupportedReasonFkRename:              mysql.Message("Columns participating in a foreign key are renamed", nil),
	ErrAlterOperationNotSupportedReasonColumnType:            mysql.Message("Cannot change column type INPLACE", nil),
	ErrAlterOperationNotSupportedReasonFkCheck:               mysql.Message("Adding foreign keys needs foreignKeyChecks=OFF", nil),
	ErrAlterOperationNotSupportedReasonIgnore:                mysql.Message("Creating unique indexes with IGNORE requires COPY algorithm to remove duplicate rows", nil),
	ErrAlterOperationNotSupportedReasonNopk:                  mysql.Message("Dropping a primary key is not allowed without also adding a new primary key", nil),
	ErrAlterOperationNotSupportedReasonAutoinc:               mysql.Message("Adding an auto-increment column requires a lock", nil),
	ErrAlterOperationNotSupportedReasonHiddenFts:             mysql.Message("Cannot replace hidden FTSDOCID with a user-visible one", nil),
	ErrAlterOperationNotSupportedReasonChangeFts:             mysql.Message("Cannot drop or rename FTSDOCID", nil),
	ErrAlterOperationNotSupportedReasonFts:                   mysql.Message("Fulltext index creation requires a lock", nil),
	ErrDupUnknownInIndex:                                     mysql.Message("Duplicate entry for key '%-.192s'", nil),
	ErrIdentCausesTooLongPath:                                mysql.Message("Long database name and identifier for object resulted in path length exceeding %d characters. Path: '%s'.", nil),
	ErrAlterOperationNotSupportedReasonNotNull:               mysql.Message("cannot silently convert NULL values, as required in this SQLMODE", nil),
	ErrMustChangePasswordLogin:                               mysql.Message("Your password has expired. To log in you must change it using a client that supports expired passwords.", nil),
	ErrRowInWrongPartition:                                   mysql.Message("Found a row in wrong partition %s", []int{0}),
	ErrGeneratedColumnFunctionIsNotAllowed:                   mysql.Message("Expression of generated column '%s' contains a disallowed function.", nil),
	ErrGeneratedColumnRowValueIsNotAllowed:                   mysql.Message("Expression of generated column '%s' cannot refer to a row value", nil),
	ErrUnsupportedAlterInplaceOnVirtualColumn:                mysql.Message("INPLACE ADD or DROP of virtual columns cannot be combined with other ALTER TABLE actions.", nil),
	ErrWrongFKOptionForGeneratedColumn:                       mysql.Message("Cannot define foreign key with %s clause on a generated column.", nil),
	ErrBadGeneratedColumn:                                    mysql.Message("The value specified for generated column '%s' in table '%s' is not allowed.", nil),
	ErrUnsupportedOnGeneratedColumn:                          mysql.Message("'%s' is not supported for generated columns.", nil),
	ErrGeneratedColumnNonPrior:                               mysql.Message("Generated column can refer only to generated columns defined prior to it.", nil),
	ErrDependentByGeneratedColumn:                            mysql.Message("Column '%s' has a generated column dependency.", nil),
	ErrGeneratedColumnRefAutoInc:                             mysql.Message("Generated column '%s' cannot refer to auto-increment column.", nil),
	ErrWarnConflictingHint:                                   mysql.Message("Hint %s is ignored as conflicting/duplicated.", nil),
	ErrUnresolvedHintName:                                    mysql.Message("Unresolved name '%s' for %s hint", nil),
	ErrInvalidFieldSize:                                      mysql.Message("Invalid size for column '%s'.", nil),
	ErrInvalidArgumentForLogarithm:                           mysql.Message("Invalid argument for logarithm", nil),
	ErrAggregateOrderNonAggQuery:                             mysql.Message("Expression #%d of ORDER BY contains aggregate function and applies to the result of a non-aggregated query", nil),
	ErrIncorrectType:                                         mysql.Message("Incorrect type for argument %s in function %s.", nil),
	ErrFieldInOrderNotSelect:                                 mysql.Message("Expression #%d of ORDER BY clause is not in SELECT list, references column '%s' which is not in SELECT list; this is incompatible with %s", nil),
	ErrAggregateInOrderNotSelect:                             mysql.Message("Expression #%d of ORDER BY clause is not in SELECT list, contains aggregate function; this is incompatible with %s", nil),
	ErrInvalidJSONData:                                       mysql.Message("Invalid JSON data provided to function %s: %s", nil),
	ErrInvalidJSONText:                                       mysql.Message("Invalid JSON text: %-.192s", []int{0}),
	ErrInvalidJSONPath:                                       mysql.Message("Invalid JSON path expression %s.", nil),
	ErrInvalidTypeForJSON:                                    mysql.Message("Invalid data type for JSON data in argument %d to function %s; a JSON string or JSON type is required.", nil),
	ErrInvalidJSONPathWildcard:                               mysql.Message("In this situation, path expressions may not contain the * and ** tokens.", nil),
	ErrInvalidJSONContainsPathType:                           mysql.Message("The second argument can only be either 'one' or 'all'.", nil),
	ErrJSONUsedAsKey:                                         mysql.Message("JSON column '%-.192s' cannot be used in key specification.", nil),
	ErrJSONDocumentNULLKey:                                   mysql.Message("JSON documents may not contain NULL member names.", nil),
	ErrSecureTransportRequired:                               mysql.Message("Connections using insecure transport are prohibited while --require_secure_transport=ON.", nil),
	ErrBadUser:                                               mysql.Message("User %s does not exist.", nil),
	ErrUserAlreadyExists:                                     mysql.Message("User %s already exists.", nil),
	ErrInvalidJSONPathArrayCell:                              mysql.Message("A path expression is not a path to a cell in an array.", nil),
	ErrInvalidEncryptionOption:                               mysql.Message("Invalid encryption option.", nil),
	ErrTooLongValueForType:                                   mysql.Message("Too long enumeration/set value for column %s.", nil),
	ErrPKIndexCantBeInvisible:                                mysql.Message("A primary key index cannot be invisible", nil),
	ErrWindowNoSuchWindow:                                    mysql.Message("Window name '%s' is not defined.", nil),
	ErrWindowCircularityInWindowGraph:                        mysql.Message("There is a circularity in the window dependency graph.", nil),
	ErrWindowNoChildPartitioning:                             mysql.Message("A window which depends on another cannot define partitioning.", nil),
	ErrWindowNoInherentFrame:                                 mysql.Message("Window '%s' has a frame definition, so cannot be referenced by another window.", nil),
	ErrWindowNoRedefineOrderBy:                               mysql.Message("Window '%s' cannot inherit '%s' since both contain an ORDER BY clause.", nil),
	ErrWindowFrameStartIllegal:                               mysql.Message("Window '%s': frame start cannot be UNBOUNDED FOLLOWING.", nil),
	ErrWindowFrameEndIllegal:                                 mysql.Message("Window '%s': frame end cannot be UNBOUNDED PRECEDING.", nil),
	ErrWindowFrameIllegal:                                    mysql.Message("Window '%s': frame start or end is negative, NULL or of non-integral type", nil),
	ErrWindowRangeFrameOrderType:                             mysql.Message("Window '%s' with RANGE N PRECEDING/FOLLOWING frame requires exactly one ORDER BY expression, of numeric or temporal type", nil),
	ErrWindowRangeFrameTemporalType:                          mysql.Message("Window '%s' with RANGE frame has ORDER BY expression of datetime type. Only INTERVAL bound value allowed.", nil),
	ErrWindowRangeFrameNumericType:                           mysql.Message("Window '%s' with RANGE frame has ORDER BY expression of numeric type, INTERVAL bound value not allowed.", nil),
	ErrWindowRangeBoundNotConstant:                           mysql.Message("Window '%s' has a non-constant frame bound.", nil),
	ErrWindowDuplicateName:                                   mysql.Message("Window '%s' is defined twice.", nil),
	ErrWindowIllegalOrderBy:                                  mysql.Message("Window '%s': ORDER BY or PARTITION BY uses legacy position indication which is not supported, use expression.", nil),
	ErrWindowInvalidWindowFuncUse:                            mysql.Message("You cannot use the window function '%s' in this context.'", nil),
	ErrWindowInvalidWindowFuncAliasUse:                       mysql.Message("You cannot use the alias '%s' of an expression containing a window function in this context.'", nil),
	ErrWindowNestedWindowFuncUseInWindowSpec:                 mysql.Message("You cannot nest a window function in the specification of window '%s'.", nil),
	ErrWindowRowsIntervalUse:                                 mysql.Message("Window '%s': INTERVAL can only be used with RANGE frames.", nil),
	ErrWindowNoGroupOrderUnused:                              mysql.Message("ASC or DESC with GROUP BY isn't allowed with window functions; put ASC or DESC in ORDER BY", nil),
	ErrWindowExplainJSON:                                     mysql.Message("To get information about window functions use EXPLAIN FORMAT=JSON", nil),
	ErrWindowFunctionIgnoresFrame:                            mysql.Message("Window function '%s' ignores the frame clause of window '%s' and aggregates over the whole partition", nil),
	ErrRoleNotGranted:                                        mysql.Message("%s is is not granted to %s", nil),
	ErrMaxExecTimeExceeded:                                   mysql.Message("Query execution was interrupted, max_execution_time exceeded.", nil),
	ErrLockAcquireFailAndNoWaitSet:                           mysql.Message("Statement aborted because lock(s) could not be acquired immediately and NOWAIT is set.", nil),
	ErrNotHintUpdatable:                                      mysql.Message("Variable '%s' cannot be set using SET_VAR hint.", nil),
	ErrDataTruncatedFunctionalIndex:                          mysql.Message("Data truncated for expression index '%s' at row %d", nil),
	ErrDataOutOfRangeFunctionalIndex:                         mysql.Message("Value is out of range for expression index '%s' at row %d", nil),
	ErrFunctionalIndexOnJSONOrGeometryFunction:               mysql.Message("Cannot create an expression index on a function that returns a JSON or GEOMETRY value", nil),
	ErrFunctionalIndexRefAutoIncrement:                       mysql.Message("Expression index '%s' cannot refer to an auto-increment column", nil),
	ErrCannotDropColumnFunctionalIndex:                       mysql.Message("Cannot drop column '%s' because it is used by an expression index. In order to drop the column, you must remove the expression index", nil),
	ErrFunctionalIndexPrimaryKey:                             mysql.Message("The primary key cannot be an expression index", nil),
	ErrFunctionalIndexOnLob:                                  mysql.Message("Cannot create an expression index on an expression that returns a BLOB or TEXT. Please consider using CAST", nil),
	ErrFunctionalIndexFunctionIsNotAllowed:                   mysql.Message("Expression of expression index '%s' contains a disallowed function", nil),
	ErrFulltextFunctionalIndex:                               mysql.Message("Fulltext expression index is not supported", nil),
	ErrSpatialFunctionalIndex:                                mysql.Message("Spatial expression index is not supported", nil),
	ErrWrongKeyColumnFunctionalIndex:                         mysql.Message("The used storage engine cannot index the expression '%s'", nil),
	ErrFunctionalIndexOnField:                                mysql.Message("Expression index on a column is not supported. Consider using a regular index instead", nil),
	ErrFKIncompatibleColumns:                                 mysql.Message("Referencing column '%s' in foreign key constraint '%s' are incompatible", nil),
	ErrFunctionalIndexRowValueIsNotAllowed:                   mysql.Message("Expression of expression index '%s' cannot refer to a row value", nil),
	ErrDependentByFunctionalIndex:                            mysql.Message("Column '%s' has an expression index dependency and cannot be dropped or renamed", nil),
	ErrInvalidJSONValueForFuncIndex:                          mysql.Message("Invalid JSON value for CAST for expression index '%s'", nil),
	ErrJSONValueOutOfRangeForFuncIndex:                       mysql.Message("Out of range JSON value for CAST for expression index '%s'", nil),
	ErrFunctionalIndexDataIsTooLong:                          mysql.Message("Data too long for expression index '%s'", nil),
	ErrFunctionalIndexNotApplicable:                          mysql.Message("Cannot use expression index '%s' due to type or collation conversion", nil),
	ErrUnsupportedConstraintCheck:                            mysql.Message("%s is not supported", nil),
	// MariaDB errors.
	ErrOnlyOneDefaultPartionAllowed:         mysql.Message("Only one DEFAULT partition allowed", nil),
	ErrWrongPartitionTypeExpectedSystemTime: mysql.Message("Wrong partitioning type, expected type: `SYSTEM_TIME`", nil),
	ErrSystemVersioningWrongPartitions:      mysql.Message("Wrong Partitions: must have at least one HISTORY and exactly one last CURRENT", nil),
	ErrSequenceRunOut:                       mysql.Message("Sequence '%-.64s.%-.64s' has run out", nil),
	ErrSequenceInvalidData:                  mysql.Message("Sequence '%-.64s.%-.64s' values are conflicting", nil),
	ErrSequenceAccessFail:                   mysql.Message("Sequence '%-.64s.%-.64s' access error", nil),
	ErrNotSequence:                          mysql.Message("'%-.64s.%-.64s' is not a SEQUENCE", nil),
	ErrUnknownSequence:                      mysql.Message("Unknown SEQUENCE: '%-.300s'", nil),
	ErrWrongInsertIntoSequence:              mysql.Message("Wrong INSERT into a SEQUENCE. One can only do single table INSERT into a sequence object (like with mysqldump).  If you want to change the SEQUENCE, use ALTER SEQUENCE instead.", nil),
	ErrSequenceInvalidTableStructure:        mysql.Message("Sequence '%-.64s.%-.64s' table structure is invalid (%s)", nil),

	// TiDB errors.
	ErrMemExceedThreshold:         mysql.Message("%s holds %dB memory, exceeds threshold %dB.%s", nil),
	ErrForUpdateCantRetry:         mysql.Message("[%d] can not retry select for update statement", nil),
	ErrAdminCheckTable:            mysql.Message("TiDB admin check table failed.", nil),
	ErrTxnTooLarge:                mysql.Message("Transaction is too large, size: %d", nil),
	ErrWriteConflictInTiDB:        mysql.Message("Write conflict, txnStartTS %d is stale", nil),
	ErrInvalidPluginID:            mysql.Message("Wrong plugin id: %s, valid plugin id is [name]-[version], both name and version should not contain '-'", nil),
	ErrInvalidPluginManifest:      mysql.Message("Cannot read plugin %s's manifest", nil),
	ErrInvalidPluginName:          mysql.Message("Plugin load with %s but got wrong name %s", nil),
	ErrInvalidPluginVersion:       mysql.Message("Plugin load with %s but got %s", nil),
	ErrDuplicatePlugin:            mysql.Message("Plugin [%s] is redeclared", nil),
	ErrInvalidPluginSysVarName:    mysql.Message("Plugin %s's sysVar %s must start with its plugin name %s", nil),
	ErrRequireVersionCheckFail:    mysql.Message("Plugin %s require %s be %v but got %v", nil),
	ErrUnsupportedReloadPlugin:    mysql.Message("Plugin %s isn't loaded so cannot be reloaded", nil),
	ErrUnsupportedReloadPluginVar: mysql.Message("Reload plugin with different sysVar is unsupported %v", nil),
	ErrTableLocked:                mysql.Message("Table '%s' was locked in %s by %v", nil),
	ErrNotExist:                   mysql.Message("Error: key not exist", nil),
	ErrTxnRetryable:               mysql.Message("Error: KV error safe to retry %s ", []int{0}),
	ErrCannotSetNilValue:          mysql.Message("can not set nil value", nil),
	ErrInvalidTxn:                 mysql.Message("invalid transaction", nil),
	ErrEntryTooLarge:              mysql.Message("entry too large, the max entry size is %d, the size of data is %d", nil),
	ErrNotImplemented:             mysql.Message("not implemented", nil),
	ErrInfoSchemaExpired:          mysql.Message("Information schema is out of date: schema failed to update in 1 lease, please make sure TiDB can connect to TiKV", nil),
	ErrInfoSchemaChanged:          mysql.Message("Information schema is changed during the execution of the statement(for example, table definition may be updated by other DDL ran in parallel). If you see this error often, try increasing `tidb_max_delta_schema_count`", nil),
	ErrBadNumber:                  mysql.Message("Bad Number", nil),
	ErrCastAsSignedOverflow:       mysql.Message("Cast to signed converted positive out-of-range integer to it's negative complement", nil),
	ErrCastNegIntAsUnsigned:       mysql.Message("Cast to unsigned converted negative integer to it's positive complement", nil),
	ErrInvalidYearFormat:          mysql.Message("invalid year format", nil),
	ErrInvalidYear:                mysql.Message("invalid year", nil),
	ErrIncorrectDatetimeValue:     mysql.Message("Incorrect datetime value: '%s'", []int{0}),
	ErrInvalidTimeFormat:          mysql.Message("invalid time format: '%v'", []int{0}),
	ErrInvalidWeekModeFormat:      mysql.Message("invalid week mode format: '%v'", nil),
	ErrFieldGetDefaultFailed:      mysql.Message("Field '%s' get default value fail", nil),
	ErrIndexOutBound:              mysql.Message("Index column %s offset out of bound, offset: %d, row: %v", []int{2}),
	ErrUnsupportedOp:              mysql.Message("operation not supported", nil),
	ErrRowNotFound:                mysql.Message("can not find the row: %s", []int{0}),
	ErrTableStateCantNone:         mysql.Message("table %s can't be in none state", nil),
	ErrColumnStateCantNone:        mysql.Message("column %s can't be in none state", nil),
	ErrColumnStateNonPublic:       mysql.Message("can not use non-public column", nil),
	ErrIndexStateCantNone:         mysql.Message("index %s can't be in none state", nil),
	ErrInvalidRecordKey:           mysql.Message("invalid record key", nil),
	ErrUnsupportedValueForVar:     mysql.Message("variable '%s' does not yet support value: %s", nil),
	ErrUnsupportedIsolationLevel:  mysql.Message("The isolation level '%s' is not supported. Set tidb_skip_isolation_level_check=1 to skip this error", nil),
	ErrInvalidDDLWorker:           mysql.Message("Invalid DDL worker", nil),
	ErrUnsupportedDDLOperation:    mysql.Message("Unsupported %s", nil),
	ErrNotOwner:                   mysql.Message("TiDB server is not a DDL owner", nil),
	ErrCantDecodeRecord:           mysql.Message("Cannot decode %s value, because %v", nil),
	ErrInvalidDDLJob:              mysql.Message("Invalid DDL job", nil),
	ErrInvalidDDLJobFlag:          mysql.Message("Invalid DDL job flag", nil),
	ErrWaitReorgTimeout:           mysql.Message("Timeout waiting for data reorganization", nil),
	ErrInvalidStoreVersion:        mysql.Message("Invalid storage current version: %d", nil),
	ErrUnknownTypeLength:          mysql.Message("Unknown length for type %d", nil),
	ErrUnknownFractionLength:      mysql.Message("Unknown length for type %d and fraction %d", nil),
	ErrInvalidDDLJobVersion:       mysql.Message("Version %d of DDL job is greater than current one: %d", nil),
	ErrInvalidSplitRegionRanges:   mysql.Message("Failed to split region ranges", nil),
	ErrReorgPanic:                 mysql.Message("Reorg worker panic", nil),
	ErrInvalidDDLState:            mysql.Message("Invalid %s state: %v", nil),
	ErrCancelledDDLJob:            mysql.Message("Cancelled DDL job", nil),
	ErrRepairTable:                mysql.Message("Failed to repair table: %s", nil),
	ErrLoadPrivilege:              mysql.Message("Load privilege table fail: %s", nil),
	ErrInvalidPrivilegeType:       mysql.Message("unknown privilege type %s", nil),
	ErrUnknownFieldType:           mysql.Message("unknown field type", nil),
	ErrInvalidSequence:            mysql.Message("invalid sequence", nil),
	ErrInvalidType:                mysql.Message("invalid type", nil),
	ErrCantGetValidID:             mysql.Message("Cannot get a valid auto-ID when retrying the statement", nil),
	ErrCantSetToNull:              mysql.Message("cannot set variable to null", nil),
	ErrSnapshotTooOld:             mysql.Message("snapshot is older than GC safe point %s", nil),
	ErrInvalidTableID:             mysql.Message("invalid TableID", nil),
	ErrInvalidAutoRandom:          mysql.Message("Invalid auto random: %s", nil),
	ErrInvalidHashKeyFlag:         mysql.Message("invalid encoded hash key flag", nil),
	ErrInvalidListIndex:           mysql.Message("invalid list index", nil),
	ErrInvalidListMetaData:        mysql.Message("invalid list meta data", nil),
	ErrWriteOnSnapshot:            mysql.Message("write on snapshot", nil),
	ErrInvalidKey:                 mysql.Message("invalid key", nil),
	ErrInvalidIndexKey:            mysql.Message("invalid index key", nil),
	ErrDataInConsistent:           mysql.Message("data isn't equal", nil),
	ErrDDLReorgElementNotExist:    mysql.Message("DDL reorg element does not exist", nil),
	ErrDDLJobNotFound:             mysql.Message("DDL Job:%v not found", nil),
	ErrCancelFinishedDDLJob:       mysql.Message("This job:%v is finished, so can't be cancelled", nil),
	ErrCannotCancelDDLJob:         mysql.Message("This job:%v is almost finished, can't be cancelled now", nil),
	ErrUnknownAllocatorType:       mysql.Message("Invalid allocator type", nil),
	ErrAutoRandReadFailed:         mysql.Message("Failed to read auto-random value from storage engine", nil),
	ErrInvalidIncrementAndOffset:  mysql.Message("Invalid auto_increment settings: auto_increment_increment: %d, auto_increment_offset: %d, both of them must be in range [1..65535]", nil),

	ErrWarnOptimizerHintInvalidInteger:  mysql.Message("integer value is out of range in '%s'", nil),
	ErrWarnOptimizerHintUnsupportedHint: mysql.Message("Optimizer hint %s is not supported by TiDB and is ignored", nil),
	ErrWarnOptimizerHintInvalidToken:    mysql.Message("Cannot use %s '%s' (tok = %d) in an optimizer hint", nil),
	ErrWarnMemoryQuotaOverflow:          mysql.Message("Max value of MEMORY_QUOTA is %d bytes, ignore this invalid limit", nil),
	ErrWarnOptimizerHintParseError:      mysql.Message("Optimizer hint syntax error at %v", nil),

	ErrSequenceUnsupportedTableOption:      mysql.Message("Unsupported sequence table-option %s", nil),
	ErrColumnTypeUnsupportedNextValue:      mysql.Message("Unsupported sequence default value for column type '%s'", nil),
	ErrAddColumnWithSequenceAsDefault:      mysql.Message("Unsupported using sequence as default value in add column '%s'", nil),
	ErrUnsupportedType:                     mysql.Message("Unsupported type %T", nil),
	ErrAnalyzeMissIndex:                    mysql.Message("Index '%s' in field list does not exist in table '%s'", nil),
	ErrCartesianProductUnsupported:         mysql.Message("Cartesian product is unsupported", nil),
	ErrPreparedStmtNotFound:                mysql.Message("Prepared statement not found", nil),
	ErrWrongParamCount:                     mysql.Message("Wrong parameter count", nil),
	ErrSchemaChanged:                       mysql.Message("Schema has changed", nil),
	ErrUnknownPlan:                         mysql.Message("Unknown plan", nil),
	ErrPrepareMulti:                        mysql.Message("Can not prepare multiple statements", nil),
	ErrPrepareDDL:                          mysql.Message("Can not prepare DDL statements with parameters", nil),
	ErrResultIsEmpty:                       mysql.Message("Result is empty", nil),
	ErrBuildExecutor:                       mysql.Message("Failed to build executor", nil),
	ErrBatchInsertFail:                     mysql.Message("Batch insert failed, please clean the table and try again.", nil),
	ErrGetStartTS:                          mysql.Message("Can not get start ts", nil),
	ErrPrivilegeCheckFail:                  mysql.Message("privilege check fail", nil), // this error message should begin lowercased to be compatible with the test
	ErrInvalidWildCard:                     mysql.Message("Wildcard fields without any table name appears in wrong place", nil),
	ErrMixOfGroupFuncAndFieldsIncompatible: mysql.Message("In aggregated query without GROUP BY, expression #%d of SELECT list contains nonaggregated column '%s'; this is incompatible with sql_mode=only_full_group_by", nil),
	ErrUnsupportedSecondArgumentType:       mysql.Message("JSON_OBJECTAGG: unsupported second argument type %v", nil),
	ErrLockExpire:                          mysql.Message("TTL manager has timed out, pessimistic locks may expire, please commit or rollback this transaction", nil),
	ErrTableOptionUnionUnsupported:         mysql.Message("CREATE/ALTER table with union option is not supported", nil),
	ErrTableOptionInsertMethodUnsupported:  mysql.Message("CREATE/ALTER table with insert method option is not supported", nil),

	ErrBRIEBackupFailed:  mysql.Message("Backup failed: %s", nil),
	ErrBRIERestoreFailed: mysql.Message("Restore failed: %s", nil),
	ErrBRIEImportFailed:  mysql.Message("Import failed: %s", nil),
	ErrBRIEExportFailed:  mysql.Message("Export failed: %s", nil),

	ErrInvalidTableSample: mysql.Message("Invalid TABLESAMPLE: %s", nil),

	ErrJSONObjectKeyTooLong:  mysql.Message("TiDB does not yet support JSON objects with the key length >= 65536", nil),
	ErrPartitionStatsMissing: mysql.Message("Build table: %s global-level stats failed due to missing partition-level stats", nil),

	ErrInvalidPlacementSpec:   mysql.Message("Invalid placement policy '%s': %s", nil),
	ErrPlacementPolicyCheck:   mysql.Message("Placement policy didn't meet the constraint, reason: %s", nil),
	ErrMultiStatementDisabled: mysql.Message("client has multi-statement capability disabled. Run SET GLOBAL tidb_multi_statement_mode='ON' after you understand the security risk", nil),

	// TiKV/PD errors.
	ErrPDServerTimeout:           mysql.Message("PD server timeout", nil),
	ErrTiKVServerTimeout:         mysql.Message("TiKV server timeout", nil),
	ErrTiKVServerBusy:            mysql.Message("TiKV server is busy", nil),
	ErrTiFlashServerTimeout:      mysql.Message("TiFlash server timeout", nil),
	ErrTiFlashServerBusy:         mysql.Message("TiFlash server is busy", nil),
	ErrResolveLockTimeout:        mysql.Message("Resolve lock timeout", nil),
	ErrRegionUnavailable:         mysql.Message("Region is unavailable", nil),
	ErrGCTooEarly:                mysql.Message("GC life time is shorter than transaction duration, transaction starts at %v, GC safe point is %v", nil),
	ErrWriteConflict:             mysql.Message("Write conflict, txnStartTS=%d, conflictStartTS=%d, conflictCommitTS=%d, key=%s", []int{3}),
	ErrTiKVStoreLimit:            mysql.Message("Store token is up to the limit, store id = %d", nil),
	ErrPrometheusAddrIsNotSet:    mysql.Message("Prometheus address is not set in PD and etcd", nil),
	ErrTiKVStaleCommand:          mysql.Message("TiKV server reports stale command", nil),
	ErrTiKVMaxTimestampNotSynced: mysql.Message("TiKV max timestamp is not synced", nil),
}
