/*
 *  Copyright (c) 2020 NetEase Inc.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/*
 * Project: curve
 * File Created: Monday, 10th December 2018 3:22:12 pm
 * Author: tongguangxun
 */

#include <fcntl.h>
#include <glog/logging.h>
#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include <json/json.h>

#include <climits>
#include <memory>
#include <thread>

#include "src/chunkserver/datastore/file_pool.h"
#include "src/common/crc32.h"
#include "src/common/curve_define.h"
#include "src/fs/local_filesystem.h"
#include "test/fs/mock_local_filesystem.h"

using ::testing::_;
using ::testing::DoAll;
using ::testing::ElementsAre;
using ::testing::Ge;
using ::testing::Gt;
using ::testing::Mock;
using ::testing::NotNull;
using ::testing::Return;
using ::testing::ReturnArg;
using ::testing::ReturnPointee;
using ::testing::SetArgPointee;
using ::testing::StrEq;

using curve::chunkserver::FilePool;
using curve::chunkserver::FilePoolHelper;
using curve::chunkserver::FilePoolOptions;
using curve::chunkserver::FilePoolState_t;
using curve::common::kFilePoolMaigic;
using curve::fs::FileSystemType;
using curve::fs::LocalFileSystem;
using curve::fs::LocalFsFactory;

#define TOTAL_FILE_NUM 10000
#define THREAD_NUM 1000
#define FILE_NUM_PER_THEAD 10

const char POOL1_DIR[] = "./cspooltest/pool1/";
const char POOL2_DIR[] = "./cspooltest/pool2/";
const char FILEPOOL_DIR[] = "./cspooltest/filePool/";

class CSFilePool_test : public testing::Test {
 public:
    void SetUp() {
        fsptr = LocalFsFactory::CreateFs(FileSystemType::EXT4, "");

        chunkFilePoolPtr_ = std::make_shared<FilePool>(fsptr);
        if (chunkFilePoolPtr_ == nullptr) {
            LOG(FATAL) << "allocate chunkfile pool failed!";
        }
        /* remove all pool dir */
        if (fsptr->DirExists(FILEPOOL_DIR)) {
            ASSERT_EQ(0, fsptr->Delete(FILEPOOL_DIR));
        }
        if (fsptr->DirExists(POOL1_DIR)) {
            ASSERT_EQ(0, fsptr->Delete(POOL1_DIR));
        }
        if (fsptr->DirExists(POOL2_DIR)) {
            ASSERT_EQ(0, fsptr->Delete(POOL2_DIR));
        }

        int count = 1;
        ASSERT_EQ(0, fsptr->Mkdir("./cspooltest/"));
        ASSERT_EQ(0, fsptr->Mkdir(FILEPOOL_DIR));
        while (count < 51) {
            std::string filename = FILEPOOL_DIR + std::to_string(count);
            int fd = fsptr->Open(filename.c_str(), O_RDWR | O_CREAT);
            ASSERT_GT(fd, 0);
            char data[8192];
            memset(data, 'a', 8192);
            fsptr->Write(fd, data, 0, 8192);
            fsptr->Close(fd);
            count++;
        }

        uint32_t chunksize = 4096;
        uint32_t metapagesize = 4096;

        int ret = FilePoolHelper::PersistEnCodeMetaInfo(
            fsptr, chunksize, metapagesize, FILEPOOL_DIR,
            "./cspooltest/filePool.meta");

        if (ret == -1) {
            LOG(ERROR) << "persist chunkfile pool meta info failed!";
            return;
        }

        int fd = fsptr->Open("./cspooltest/filePool.meta2", O_RDWR | O_CREAT);
        if (fd < 0) {
            return;
        }

        char* buffer = new char[2048];
        memset(buffer, 1, 2048);
        ret = fsptr->Write(fd, buffer, 0, 2048);
        if (ret != 2048) {
            delete[] buffer;
            return;
        }
        delete[] buffer;
        fsptr->Close(fd);
    }

    void TearDown() {
        fsptr->Delete("./cspooltest");
        chunkFilePoolPtr_->UnInitialize();
    }

    std::shared_ptr<FilePool> chunkFilePoolPtr_;
    std::shared_ptr<LocalFileSystem> fsptr;
};

bool CheckFileOpenOrNot(const std::string& filename) {
    std::string syscmd;
    syscmd.append("lsof ").append(filename);
    FILE* fp;
    char buffer[4096];
    memset(buffer, 0, 4096);
    fp = popen(syscmd.c_str(), "r");
    fgets(buffer, sizeof(buffer), fp);
    pclose(fp);

    std::string out(buffer, 4096);
    return out.find("No such file or directory") != out.npos;
}

TEST_F(CSFilePool_test, InitializeTest) {
    std::string filePool = "./cspooltest/filePool.meta";
    const std::string filePoolPath = FILEPOOL_DIR;

    FilePoolOptions cfop;
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());

    // initialize
    ASSERT_TRUE(chunkFilePoolPtr_->Initialize(cfop));
    ASSERT_EQ(50, chunkFilePoolPtr_->Size());
    // 初始化阶段会扫描FilePool内的所有文件，在扫描结束之后需要关闭这些文件
    // 防止过多的文件描述符被占用
    ASSERT_FALSE(CheckFileOpenOrNot(filePoolPath + "1"));
    ASSERT_FALSE(CheckFileOpenOrNot(filePoolPath + "2"));
    cfop.fileSize = 8192;
    cfop.metaPageSize = 4096;
    // test meta content wrong
    ASSERT_FALSE(chunkFilePoolPtr_->Initialize(cfop));
    cfop.fileSize = 8192;
    cfop.metaPageSize = 4096;
    ASSERT_FALSE(chunkFilePoolPtr_->Initialize(cfop));
    // invalid file name
    std::string filename = filePoolPath + "a";
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    int fd = fsptr->Open(filename.c_str(), O_RDWR | O_CREAT);
    char data[8192];
    memset(data, 'a', 8192);
    fsptr->Write(fd, data, 0, 8192);
    fsptr->Close(fd);
    ASSERT_FALSE(chunkFilePoolPtr_->Initialize(cfop));
    // test meta file wrong
    filePool = "./cspooltest/filePool.meta2";
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());
    ASSERT_FALSE(chunkFilePoolPtr_->Initialize(cfop));
    // test meta file not exist
    filePool = "./cspooltest/FilePool.meta3";
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());
    ASSERT_FALSE(chunkFilePoolPtr_->Initialize(cfop));

    fsptr->Delete(filePoolPath + "a");
    fsptr->Delete("./cspooltest/filePool.meta3");
}

TEST_F(CSFilePool_test, GetFileTest) {
    std::string filePool = "./cspooltest/filePool.meta";
    FilePoolOptions cfop;
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());
    // test get chunk success
    char metapage[4096];
    memset(metapage, '1', 4096);
    ASSERT_EQ(-1, chunkFilePoolPtr_->GetFile("./new_exit", metapage));
    ASSERT_EQ(-2, fsptr->Delete("./new_exit"));
    chunkFilePoolPtr_->Initialize(cfop);
    ASSERT_EQ(50, chunkFilePoolPtr_->Size());
    ASSERT_EQ(0, chunkFilePoolPtr_->GetFile("./new1", metapage));
    ASSERT_EQ(49, chunkFilePoolPtr_->Size());
    ASSERT_TRUE(fsptr->FileExists("./new1"));
    int fd = fsptr->Open("./new1", O_RDWR);
    char data[4096];
    ASSERT_GE(fd, 0);
    int len = fsptr->Read(fd, data, 0, 4096);
    ASSERT_EQ(4096, len);
    for (int i = 0; i < 4096; i++) {
        ASSERT_EQ(data[i], '1');
    }
    ASSERT_EQ(0, fsptr->Close(fd));
    ASSERT_EQ(0, fsptr->Delete("./new1"));

    // test get chunk success
    ASSERT_EQ(0, chunkFilePoolPtr_->GetFile("./new2", metapage));
    ASSERT_TRUE(fsptr->FileExists("./new2"));
    ASSERT_NE(49, chunkFilePoolPtr_->Size());
    ASSERT_EQ(0, fsptr->Delete("./new2"));
}

TEST_F(CSFilePool_test, RecycleFileTest) {
    std::string filePool = "./cspooltest/filePool.meta";
    const std::string filePoolPath = FILEPOOL_DIR;
    FilePoolOptions cfop;
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());

    chunkFilePoolPtr_->Initialize(cfop);
    FilePoolState_t currentStat = chunkFilePoolPtr_->GetState();
    ASSERT_EQ(50, currentStat.preallocatedChunksLeft);
    ASSERT_EQ(50, chunkFilePoolPtr_->Size());
    char metapage[4096];
    memset(metapage, '1', 4096);
    ASSERT_EQ(0, chunkFilePoolPtr_->GetFile("./new1", metapage));
    ASSERT_TRUE(fsptr->FileExists("./new1"));
    ASSERT_EQ(49, chunkFilePoolPtr_->Size());

    currentStat = chunkFilePoolPtr_->GetState();
    ASSERT_EQ(49, currentStat.preallocatedChunksLeft);

    chunkFilePoolPtr_->RecycleFile("./new1");
    ASSERT_EQ(50, chunkFilePoolPtr_->Size());

    currentStat = chunkFilePoolPtr_->GetState();
    ASSERT_EQ(50, currentStat.preallocatedChunksLeft);

    ASSERT_FALSE(fsptr->FileExists("./new1"));
    ASSERT_TRUE(fsptr->FileExists(filePoolPath + "4"));
    ASSERT_EQ(0, fsptr->Delete(filePoolPath + "4"));
}

TEST_F(CSFilePool_test, UsePoolConcurrentGetAndRecycle) {
    std::string filePool = "./cspooltest/filePool.meta";
    const std::string filePoolPath = FILEPOOL_DIR;
    FilePoolOptions cfop;
    memcpy(cfop.filePoolDir, filePoolPath.c_str(), filePoolPath.size());
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    cfop.getFileFromPool = true;
    cfop.retryTimes = 1;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());

    /* step 1. prepare file for filePool and pool2 */
    int count = 1;
    while (count <= TOTAL_FILE_NUM) {
        std::string filename = filePoolPath + std::to_string(count);
        int fd = fsptr->Open(filename.c_str(), O_RDWR | O_CREAT);
        char data[8192];
        memset(data, 'a', 8192);
        fsptr->Write(fd, data, 0, 8192);
        fsptr->Close(fd);
        count++;
    }

    fsptr->Mkdir(POOL1_DIR);
    fsptr->Mkdir(POOL2_DIR);
    count = 1;
    while (count <= TOTAL_FILE_NUM) {
        std::string filename = POOL2_DIR + std::to_string(count);
        int fd = fsptr->Open(filename.c_str(), O_RDWR | O_CREAT);
        char data[8192];
        memset(data, 'b', 8192);
        fsptr->Write(fd, data, 0, 8192);
        fsptr->Close(fd);
        count++;
    }

    /* step 2. init filepool */
    chunkFilePoolPtr_->Initialize(cfop);

    /* step 3. start multiple threads, get files from filePool to pool1  */
    std::vector<std::thread> threads;
    for (int i = 0; i < THREAD_NUM; i++) {
        int id = i;
        auto task = [this, id]() {
            char metapage[4096];
            memset(metapage, '1', 4096);
            for (int filenum = id * FILE_NUM_PER_THEAD + 1;
                 filenum <= (id * FILE_NUM_PER_THEAD + FILE_NUM_PER_THEAD);
                 filenum++) {
                ASSERT_EQ(0,
                          chunkFilePoolPtr_->GetFile(
                              POOL1_DIR + std::to_string(filenum), metapage));
            }
        };
        std::thread currThread(task);
        threads.push_back(std::move(currThread));
    }

    /* step 4. start multiple thread, recycle files from pool2 to filePool */
    for (int i = 0; i < THREAD_NUM; i++) {
        int id = i;
        auto task = [this, id]() {
            for (int filenum = id * FILE_NUM_PER_THEAD + 1;
                 filenum <= (id * FILE_NUM_PER_THEAD + FILE_NUM_PER_THEAD);
                 filenum++) {
                ASSERT_EQ(0, chunkFilePoolPtr_->RecycleFile(
                                 POOL2_DIR + std::to_string(filenum)));
            }
        };
        std::thread currThread(task);
        threads.push_back(std::move(currThread));
    }

    for (auto iter = threads.begin(); iter != threads.end(); iter++) {
        iter->join();
    }

    /* step 5. verify file numbers in filePool, pool1 and poo2 */
    {
        std::vector<std::string> filename;
        fsptr->List(filePoolPath, &filename);
        LOG(INFO) << "file Pool size=" << filename.size();
        ASSERT_EQ(filename.size(), TOTAL_FILE_NUM);
    }
    {
        std::vector<std::string> filename;
        fsptr->List(POOL1_DIR, &filename);
        LOG(INFO) << "pool1 size=" << filename.size();
        ASSERT_EQ(filename.size(), TOTAL_FILE_NUM);
    }
    {
        std::vector<std::string> filename;
        fsptr->List(POOL2_DIR, &filename);
        LOG(INFO) << "pool2 size=" << filename.size();
        ASSERT_EQ(filename.size(), 0);
    }
}

TEST_F(CSFilePool_test, WithoutPoolConcurrentGetAndRecycle) {
    std::string filePool = "./cspooltest/filePool.meta";
    const std::string filePoolPath = FILEPOOL_DIR;
    FilePoolOptions cfop;
    memcpy(cfop.filePoolDir, filePoolPath.c_str(), filePoolPath.size());
    cfop.fileSize = 4096;
    cfop.metaPageSize = 4096;
    cfop.getFileFromPool = false;
    cfop.retryTimes = 1;
    memcpy(cfop.metaPath, filePool.c_str(), filePool.size());

    /* step 1. prepare file for filePool and pool2 */
    ASSERT_EQ(0, fsptr->Delete(filePoolPath));
    fsptr->Mkdir(filePoolPath);
    fsptr->Mkdir(POOL1_DIR);
    fsptr->Mkdir(POOL2_DIR);
    int count = 1;
    while (count <= TOTAL_FILE_NUM) {
        std::string filename = POOL2_DIR + std::to_string(count);
        int fd = fsptr->Open(filename.c_str(), O_RDWR | O_CREAT);
        char data[8192];
        memset(data, 'b', 8192);
        fsptr->Write(fd, data, 0, 8192);
        fsptr->Close(fd);
        count++;
    }

    /* step 2. init filepool */
    chunkFilePoolPtr_->Initialize(cfop);

    /* step 3. start multiple threads, get files from filePool to pool1  */
    std::vector<std::thread> threads;
    for (int i = 0; i < THREAD_NUM; i++) {
        int id = i;
        auto task = [this, id]() {
            char metapage[4096];
            memset(metapage, '1', 4096);
            for (int filenum = id * FILE_NUM_PER_THEAD + 1;
                 filenum <= (id * FILE_NUM_PER_THEAD + FILE_NUM_PER_THEAD);
                 filenum++) {
                ASSERT_EQ(0,
                          chunkFilePoolPtr_->GetFile(
                              POOL1_DIR + std::to_string(filenum), metapage));
            }
        };
        std::thread currThread(task);
        threads.push_back(std::move(currThread));
    }

    /* step 4. start multiple thread, recycle files from pool2 to filePool */
    for (int i = 0; i < THREAD_NUM; i++) {
        int id = i;
        auto task = [this, id]() {
            for (int filenum = id * FILE_NUM_PER_THEAD + 1;
                 filenum <= (id * FILE_NUM_PER_THEAD + FILE_NUM_PER_THEAD);
                 filenum++) {
                ASSERT_EQ(0, chunkFilePoolPtr_->RecycleFile(
                                 POOL2_DIR + std::to_string(filenum)));
            }
        };
        std::thread currThread(task);
        threads.push_back(std::move(currThread));
    }

    for (auto iter = threads.begin(); iter != threads.end(); iter++) {
        iter->join();
    }

    /* step 5. verify file numbers in filePool, pool1 and poo2 */
    {
        std::vector<std::string> filename;
        fsptr->List(filePoolPath, &filename);
        LOG(INFO) << "file Pool size=" << filename.size();
        ASSERT_EQ(filename.size(), 0);
    }
    {
        std::vector<std::string> filename;
        fsptr->List(POOL1_DIR, &filename);
        LOG(INFO) << "pool1 size=" << filename.size();
        ASSERT_EQ(filename.size(), TOTAL_FILE_NUM);
    }
    {
        std::vector<std::string> filename;
        fsptr->List(POOL2_DIR, &filename);
        LOG(INFO) << "pool2 size=" << filename.size();
        ASSERT_EQ(filename.size(), 0);
    }
}

TEST(CSFilePool, GetFileDirectlyTest) {
    std::shared_ptr<FilePool> chunkFilePoolPtr_;
    std::shared_ptr<LocalFileSystem> fsptr;
    fsptr = LocalFsFactory::CreateFs(FileSystemType::EXT4, "");
    const std::string filePoolPath = FILEPOOL_DIR;
    // create chunkfile in chunkfile pool dir
    // if chunkfile pool 的getFileFromPool开关关掉了，那么
    // FilePool的size是一直为0，不会从pool目录中找
    std::string filename = filePoolPath + "1000";
    fsptr->Mkdir(filePoolPath);
    int fd = fsptr->Open(filename.c_str(), O_RDWR | O_CREAT);

    char data[8192];
    memset(data, 'a', 8192);
    ASSERT_EQ(8192, fsptr->Write(fd, data, 0, 8192));
    fsptr->Close(fd);
    ASSERT_TRUE(fsptr->FileExists(filePoolPath + "1000"));

    FilePoolOptions cspopt;
    cspopt.getFileFromPool = false;
    cspopt.fileSize = 16 * 1024;
    cspopt.metaPageSize = 4 * 1024;
    cspopt.metaFileSize = 4 * 1024;
    cspopt.retryTimes = 5;
    strcpy(cspopt.filePoolDir, filePoolPath.c_str());  // NOLINT

    chunkFilePoolPtr_ = std::make_shared<FilePool>(fsptr);
    if (chunkFilePoolPtr_ == nullptr) {
        LOG(FATAL) << "allocate chunkfile pool failed!";
    }
    ASSERT_TRUE(chunkFilePoolPtr_->Initialize(cspopt));
    ASSERT_EQ(0, chunkFilePoolPtr_->Size());

    // 测试获取chunk，chunkfile pool size不变一直为0
    char metapage[4096];
    memset(metapage, '1', 4096);

    ASSERT_EQ(0, chunkFilePoolPtr_->GetFile("./new1", metapage));
    ASSERT_EQ(0, chunkFilePoolPtr_->Size());

    ASSERT_TRUE(fsptr->FileExists("./new1"));
    fd = fsptr->Open("./new1", O_RDWR);
    ASSERT_GE(fd, 0);

    char buf[4096];
    ASSERT_EQ(4096, fsptr->Read(fd, buf, 0, 4096));
    for (int i = 0; i < 4096; i++) {
        ASSERT_EQ(buf[i], '1');
    }

    // 测试回收chunk,文件被删除，FilePool Size不受影响
    chunkFilePoolPtr_->RecycleFile("./new1");
    ASSERT_EQ(0, chunkFilePoolPtr_->Size());
    ASSERT_FALSE(fsptr->FileExists("./new1"));

    // 删除测试文件及目录
    ASSERT_EQ(0, fsptr->Close(fd));
    ASSERT_EQ(0, fsptr->Delete(filePoolPath + "1000"));
    ASSERT_EQ(0, fsptr->Delete(filePoolPath));
    chunkFilePoolPtr_->UnInitialize();
}
