<?php

namespace App\Repos;

use App\Library\Paginator\Adapter\QueryBuilder as PagerQueryBuilder;
use App\Models\Category as CategoryModel;
use App\Models\Chapter as ChapterModel;
use App\Models\ChapterUser as ChapterUserModel;
use App\Models\Consult as ConsultModel;
use App\Models\Course as CourseModel;
use App\Models\CourseCategory as CourseCategoryModel;
use App\Models\CourseFavorite as CourseFavoriteModel;
use App\Models\CoursePackage as CoursePackageModel;
use App\Models\CourseRating as CourseRatingModel;
use App\Models\CourseRelated as CourseRelatedModel;
use App\Models\CourseUser as CourseUserModel;
use App\Models\ImGroup as ImGroupModel;
use App\Models\Package as PackageModel;
use App\Models\Review as ReviewModel;
use App\Models\User as UserModel;
use Phalcon\Mvc\Model;
use Phalcon\Mvc\Model\Resultset;
use Phalcon\Mvc\Model\ResultsetInterface;

class Course extends Repository
{

    public function paginate($where = [], $sort = 'latest', $page = 1, $limit = 15)
    {
        $builder = $this->modelsManager->createBuilder();

        $builder->from(CourseModel::class);

        $builder->where('1 = 1');

        if (!empty($where['category_id'])) {
            $where['id'] = $this->getCategoryCourseIds($where['category_id']);
        } elseif (!empty($where['teacher_id'])) {
            $where['id'] = $this->getTeacherCourseIds($where['teacher_id']);
        }

        if (!empty($where['id'])) {
            if (is_array($where['id'])) {
                $builder->inWhere('id', $where['id']);
            } else {
                $builder->andWhere('id = :id:', ['id' => $where['id']]);
            }
        }

        if (!empty($where['title'])) {
            $builder->andWhere('title LIKE :title:', ['title' => "%{$where['title']}%"]);
        }

        if (!empty($where['model'])) {
            if (is_array($where['model'])) {
                $builder->inWhere('model', $where['model']);
            } else {
                $builder->andWhere('model = :model:', ['model' => $where['model']]);
            }
        }

        if (!empty($where['level'])) {
            if (is_array($where['level'])) {
                $builder->inWhere('level', $where['model']);
            } else {
                $builder->andWhere('level = :level:', ['level' => $where['level']]);
            }
        }

        if (isset($where['free'])) {
            if ($where['free'] == 1) {
                $builder->andWhere('market_price = 0');
            } else {
                $builder->andWhere('market_price > 0');
            }
        }

        if (isset($where['featured'])) {
            $builder->andWhere('featured = :featured:', ['featured' => $where['featured']]);
        }

        if (isset($where['published'])) {
            $builder->andWhere('published = :published:', ['published' => $where['published']]);
        }

        if (isset($where['deleted'])) {
            $builder->andWhere('deleted = :deleted:', ['deleted' => $where['deleted']]);
        }

        if ($sort == 'free') {
            $builder->andWhere('market_price = 0');
        } elseif ($sort == 'featured') {
            $builder->andWhere('featured = 1');
        } elseif ($sort == 'vip_discount') {
            $builder->andWhere('vip_price < market_price');
            $builder->andWhere('vip_price > 0');
        } elseif ($sort == 'vip_free') {
            $builder->andWhere('market_price > 0');
            $builder->andWhere('vip_price = 0');
        }

        switch ($sort) {
            case 'score':
                $orderBy = 'score DESC';
                break;
            case 'rating':
                $orderBy = 'rating DESC';
                break;
            case 'popular':
                $orderBy = 'user_count DESC';
                break;
            default:
                $orderBy = 'id DESC';
                break;
        }

        $builder->orderBy($orderBy);

        $pager = new PagerQueryBuilder([
            'builder' => $builder,
            'page' => $page,
            'limit' => $limit,
        ]);

        return $pager->paginate();
    }

    /**
     * @param array $where
     * @param string $sort
     * @return ResultsetInterface|Resultset|CourseModel[]
     */
    public function findAll($where = [], $sort = 'latest')
    {
        /**
         * 一个偷懒的实现，适用于中小体量数据
         */
        $paginate = $this->paginate($where, $sort, 1, 10000);

        return $paginate->items;
    }

    /**
     * @param int $id
     * @return CourseModel|Model|bool
     */
    public function findById($id)
    {
        return CourseModel::findFirst([
            'conditions' => 'id = :id:',
            'bind' => ['id' => $id],
        ]);
    }

    /**
     * @param array $ids
     * @param array|string $columns
     * @return ResultsetInterface|Resultset|CourseModel[]
     */
    public function findByIds($ids, $columns = '*')
    {
        return CourseModel::query()
            ->columns($columns)
            ->inWhere('id', $ids)
            ->execute();
    }

    /**
     * @param int $courseId
     * @return CourseRatingModel|Model|bool
     */
    public function findCourseRating($courseId)
    {
        return CourseRatingModel::findFirst([
            'conditions' => 'course_id = :course_id:',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    /**
     * @param int $courseId
     * @return ImGroupModel|Model|bool
     */
    public function findImGroup($courseId)
    {
        return ImGroupModel::findFirst([
            'conditions' => 'course_id = :course_id:',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    /**
     * @param int $courseId
     * @return ResultsetInterface|Resultset|UserModel[]
     */
    public function findTeachers($courseId)
    {
        $roleType = CourseUserModel::ROLE_TEACHER;

        return $this->modelsManager->createBuilder()
            ->columns('u.*')
            ->addFrom(UserModel::class, 'u')
            ->join(CourseUserModel::class, 'u.id = cu.user_id', 'cu')
            ->where('cu.course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('cu.role_type = :role_type:', ['role_type' => $roleType])
            ->andWhere('cu.deleted = :deleted:', ['deleted' => 0])
            ->getQuery()->execute();
    }

    /**
     * @param int $courseId
     * @return ResultsetInterface|Resultset|CategoryModel[]
     */
    public function findCategories($courseId)
    {
        return $this->modelsManager->createBuilder()
            ->columns('c.*')
            ->addFrom(CategoryModel::class, 'c')
            ->join(CourseCategoryModel::class, 'c.id = cc.category_id', 'cc')
            ->where('cc.course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('c.published = 1')
            ->getQuery()->execute();
    }

    /**
     * @param int $courseId
     * @return ResultsetInterface|Resultset|PackageModel[]
     */
    public function findPackages($courseId)
    {
        return $this->modelsManager->createBuilder()
            ->columns('p.*')
            ->addFrom(PackageModel::class, 'p')
            ->join(CoursePackageModel::class, 'p.id = cp.package_id', 'cp')
            ->where('cp.course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('p.published = 1')
            ->getQuery()->execute();
    }

    /**
     * @param int $courseId
     * @return ResultsetInterface|Resultset|CourseModel[]
     */
    public function findRelatedCourses($courseId)
    {
        return $this->modelsManager->createBuilder()
            ->columns('c.*')
            ->addFrom(CourseModel::class, 'c')
            ->join(CourseRelatedModel::class, 'c.id = cr.related_id', 'cr')
            ->where('cr.course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('c.published = 1')
            ->getQuery()->execute();
    }

    /**
     * @param int $courseId
     * @return ResultsetInterface|Resultset|ChapterModel[]
     */
    public function findChapters($courseId)
    {
        return ChapterModel::query()
            ->where('course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('deleted = 0')
            ->execute();
    }

    /**
     * @param int $courseId
     * @return ResultsetInterface|Resultset|ChapterModel[]
     */
    public function findLessons($courseId)
    {
        return ChapterModel::query()
            ->where('course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('parent_id > 0')
            ->andWhere('deleted = 0')
            ->execute();
    }

    /**
     * @param int $courseId
     * @param int $userId
     * @param int $planId
     * @return ResultsetInterface|Resultset|ChapterUserModel[]
     */
    public function findUserLearnings($courseId, $userId, $planId)
    {
        return ChapterUserModel::query()
            ->where('course_id = :course_id:', ['course_id' => $courseId])
            ->andWhere('user_id = :user_id:', ['user_id' => $userId])
            ->andWhere('plan_id = :plan_id:', ['plan_id' => $planId])
            ->execute();
    }

    public function countCourses()
    {
        return (int)CourseModel::count(['conditions' => 'deleted = 0']);
    }

    public function countLessons($courseId)
    {
        return (int)ChapterModel::count([
            'conditions' => 'course_id = :course_id: AND parent_id > 0 AND deleted = 0',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    public function countPackages($courseId)
    {
        return (int)CoursePackageModel::count([
            'conditions' => 'course_id = :course_id:',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    public function countUsers($courseId)
    {
        return (int)CourseUserModel::count([
            'conditions' => 'course_id = :course_id: AND deleted = 0',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    public function countConsults($courseId)
    {
        return (int)ConsultModel::count([
            'conditions' => 'course_id = :course_id: AND published = 1',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    public function countReviews($courseId)
    {
        return (int)ReviewModel::count([
            'conditions' => 'course_id = :course_id: AND published = 1',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    public function countFavorites($courseId)
    {
        return (int)CourseFavoriteModel::count([
            'conditions' => 'course_id = :course_id:',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    public function averageRating($courseId)
    {
        return (int)ReviewModel::average([
            'column' => 'rating',
            'conditions' => 'course_id = :course_id: AND published = 1',
            'bind' => ['course_id' => $courseId],
        ]);
    }

    protected function getCategoryCourseIds($categoryId)
    {
        $categoryIds = is_array($categoryId) ? $categoryId : [$categoryId];

        $repo = new CourseCategory();

        $rows = $repo->findByCategoryIds($categoryIds);

        $result = [];

        if ($rows->count() > 0) {
            $result = kg_array_column($rows->toArray(), 'course_id');
        }

        return $result;
    }

    protected function getTeacherCourseIds($teacherId)
    {
        $teacherIds = is_array($teacherId) ? $teacherId : [$teacherId];

        $repo = new CourseUser();

        $rows = $repo->findByTeacherIds($teacherIds);

        $result = [];

        if ($rows->count() > 0) {
            $result = kg_array_column($rows->toArray(), 'course_id');
        }

        return $result;
    }

}
