﻿using RFrameGenerate.Common;
using RFrameGenerate.Model;
using RFramework.SQLQuery.SQLExpands;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text;

namespace RFrameGenerate.BLL.Member
{
    public class MemberLevelManage
    {
        public static List<T_BAS_MemberLevel> GetPageList(GridModel gridModel)
        {
            using (var query = Factory.CreateT_BAS_MemberLevel())
            {
                return query.GetPageList(gridModel);
            }
        }

        public static List<T_BAS_MemberLevel> GetList()
        {
            using (var query = Factory.CreateT_BAS_MemberLevel())
            {
                return query.GetList(null, false, m => m.LevelExp, DAL.Enums.OrderMethods.ASC);
            }
        }

        public static List<R> GetList<R>(Guid couponGuid, Expression<Func<T_BAS_MemberLevel, T_VIP_VipCoupon, R>> selectField)
        {
            using (var sql = new SqlModel<T_BAS_MemberLevel, T_VIP_VipCoupon>())
            {
                return sql.SqlSelect(selectField)
                          .SqlJionInner((a, b) => b).SqlOn((a, b) => a.MemberLevelGuid == b.MemberLevelGuid)
                          .SqlWhere((a, b) => b.CouponGuid == couponGuid && a.MerchantGuid == CurrentManage.MerchantGuid)
                          .ToList();
            }
        }

        public static Result Delete(Guid? memberLevelGuid)
        {
            var level = GetMemberLevel(memberLevelGuid);

            if (level == null) return new Result(false, "等级为空，不能删除！");

            if (Member.VipCouponManage.IsUsingMemberLevel(level.MemberLevelGuid)) return new Result(false, "等级已经使用，不能删除！");

            if (Member.ProductDiscountManage.IsUsingMemberLevel(level.MemberLevelGuid)) return new Result(false, "等级已经使用，不能删除！");

            using (var query = Factory.CreateT_BAS_MemberLevel())
            {
                query.Delete(level.MemberLevelGuid);
            }
            return new Result(true);
        }

        public static Result Save(Guid? memberLevelGuid, string levelName, string levelImg, int exp)
        {
            if (string.IsNullOrEmpty(levelName)) return new Result(false, "等级名称不能为空！");

            var level = GetMemberLevel(memberLevelGuid);

            if (level == null)
            {
                level = new T_BAS_MemberLevel()
                {
                    MemberLevelGuid = Guid.NewGuid(),
                    LevelName = levelName,
                    LevelImg = levelImg,
                    LevelExp = exp,
                    MerchantGuid = RFrameGenerate.Common.CurrentManage.MerchantGuid,
                };
                using (var query = Factory.CreateT_BAS_MemberLevel())
                {
                    query.Add(level);
                }
                return new Result(true);
            }
            else
            {
                using (var query = Factory.CreateT_BAS_MemberLevel())
                {
                    query.Update(m => new object[] {
                            m.LevelName == levelName,
                            m.LevelImg == levelImg,
                            m.LevelExp == exp
                        }, v => v.MemberLevelGuid == memberLevelGuid.Value && v.MerchantGuid == RFrameGenerate.Common.CurrentManage.MerchantGuid);
                }
                return new Result(true);
            }
        }

        public static T_BAS_MemberLevel GetMemberLevel(Guid? memberLevelGuid)
        {
            if (memberLevelGuid == null || memberLevelGuid.Value == Guid.Empty) return null;

            using (var query = Factory.CreateT_BAS_MemberLevel())
            {
                return query.GetFirstData(m => m.MemberLevelGuid == memberLevelGuid.Value && m.MerchantGuid == RFrameGenerate.Common.CurrentManage.MerchantGuid);
            }
        }

        public static Result SaveMemberLevelState(bool state)
        {
            return ParameterManage.UpdateParam(ParameterName.MarketingName.MemberLevel, state.ToString().ToLower());
        }

        /// <summary>
        /// 当前用户等级
        /// </summary>        
        public static T_BAS_MemberLevel CurrentMemberLevel()
        {
            if (!UsingMemberLevel()) return null;

            var member = MemberManage.GetMemberByGuid(CurrentUser.GetCurrUserGuid());

            if (member == null) return null;

            if (member.GrowthValue == null) return null;

            using (var query = Factory.CreateT_BAS_MemberLevel())
            {
                return query.GetFirstData(m => m.LevelExp <= member.GrowthValue, false, v => v.LevelExp, DAL.Enums.OrderMethods.DESC);
            }
        }

        /// <summary>
        /// 是否使用用户等级
        /// </summary> 
        public static bool UsingMemberLevel()
        {
            var value = ParameterManage.GetParamValue(ParameterName.MarketingName.MemberLevel);

            if (string.IsNullOrEmpty(value)) return false;

            return value == "true";
        }

        /// <summary>
        /// 是否使用用户等级
        /// </summary> 
        public static bool UsingMemberLevel(Guid merchantGuid)
        {
            using (var query = Factory.CreateT_SYS_Parameter())
            {
                var model = query.GetFirstData(a => a.MerchantGuid == merchantGuid && a.ParameterName == ParameterName.MarketingName.MemberLevel, isFilterMerchant: false);

                if (model == null) return false;

                if (string.IsNullOrEmpty(model.ParameterValue)) return false;

                return model.ParameterValue == "true";
            }
        }


    }
}
