﻿using RFrameGenerate.Common;
using RFrameGenerate.Model;
using RFramework.SQLQuery.SQLExpands;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text;

namespace RFrameGenerate.BLL.Member
{
    public class ProductDiscountManage
    {
        public static PageList<R> GetPageList<R>(GridModel gridModel, Expression<Func<T_Vip_ProductDiscount, T_BAS_Products, T_BAS_MemberLevel, R>> expression)
        {
            using (var sql = new SqlModel<T_Vip_ProductDiscount, T_BAS_Products, T_BAS_MemberLevel>())
            {
                return sql.SqlPage(expression, gridModel.page, gridModel.rows)
                          .SqlJionLeft((a, b, c) => b).SqlOn((a, b, c) => a.ProductGuid == b.ProductGuid)
                          .SqlJionInner((a, b, c) => c).SqlOn((a, b, c) => a.MemberLevelGuid == c.MemberLevelGuid)
                          .SqlWhere((a, b, c) => a.MerchantGuid == Common.CurrentManage.MerchantGuid)
                          .ToPageList();
            }
        }

        public static T_Vip_ProductDiscount GetDiscount(Guid? discountGuid)
        {
            if (discountGuid == null || discountGuid == Guid.Empty) return null;

            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                return query.GetFirstData(m => m.MerchantGuid == Common.CurrentManage.MerchantGuid && m.ProductDiscountGuid == discountGuid);
            }
        }

        public static Result SaveProductDiscount(Model.T_Vip_ProductDiscount discount)
        {
            var chResult = CheckError(discount);

            if (!chResult.State) return new Result(false, chResult.Msg);

            if (discount.ProductDiscountGuid == Guid.Empty || !IsExists(discount.ProductDiscountGuid))
                return Save(discount);
            else
                return Update(discount);
        }

        public static Result DeleteProductDiscount(Guid discountGuid)
        {
            if (!IsExists(discountGuid)) return new Result(false, "不存在折扣!");

            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                query.Delete(m => m.ProductDiscountGuid == discountGuid);
            }
            return new Result(true);
        }

        public static bool IsUsingMemberLevel(Guid memberLevel)
        {
            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                return query.IsExist(m => m.MerchantGuid == Common.CurrentManage.MerchantGuid && m.MemberLevelGuid == memberLevel);
            }
        }

        /// <summary>
        /// 当前商品折扣
        /// </summary>        
        public static decimal? CurrentDiscount(T_BAS_MemberLevel level, Guid? productGuid)
        {
            if (level == null) return null;

            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                var discount = query.GetFirstData(m => m.MerchantGuid == CurrentManage.MerchantGuid && m.MemberLevelGuid == level.MemberLevelGuid && m.ProductGuid == productGuid);

                if (discount != null) return discount.DiscountProportion.Value / 100;

                discount = query.GetFirstData(m => m.MerchantGuid == CurrentManage.MerchantGuid && m.MemberLevelGuid == level.MemberLevelGuid && m.DiscountType == 1);

                if (discount != null) return discount.DiscountProportion.Value / 100;

                return null;
            }
        }

        private static Result Save(Model.T_Vip_ProductDiscount discount)
        {
            discount.MerchantGuid = Common.CurrentManage.MerchantGuid;
            discount.ProductDiscountGuid = Guid.NewGuid();

            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                query.Add(discount);
            }
            return new Result(true);
        }

        private static Result Update(Model.T_Vip_ProductDiscount discount)
        {
            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                query.Update(m => new object[] 
                {
                    m.MemberLevelGuid == discount.MemberLevelGuid,
                    m.ProductGuid == discount.ProductGuid,
                    m.DiscountProportion == discount.DiscountProportion,
                    m.DiscountType == discount.DiscountType
                }
                , v => v.ProductDiscountGuid == discount.ProductDiscountGuid);
            }
            return new Result(true);
        }

        private static bool IsExists(Guid discountGuid)
        {
            if (discountGuid == Guid.Empty) return false;

            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                return query.IsExist(m => m.MerchantGuid == Common.CurrentManage.MerchantGuid && m.ProductDiscountGuid == discountGuid);
            }
        }

        private static Result CheckError(Model.T_Vip_ProductDiscount discount)
        {
            if (discount == null) return new Result(false, "保存对象为空！");

            if (discount.DiscountProportion == null || discount.DiscountProportion <= 0.0m) return new Result(false, "折扣比例不能为空或小于等于0");

            if (discount.DiscountType == 2 && (discount.ProductGuid == null || discount.ProductGuid == Guid.Empty))
                return new Result(false, "指定商品不能为空！");

            using (var query = Factory.CreateT_Vip_ProductDiscount())
            {
                if (discount.DiscountType == 1 && query.IsExist(m => m.MemberLevelGuid == discount.MemberLevelGuid && m.MerchantGuid == Common.CurrentManage.MerchantGuid && m.DiscountType == 1 && m.ProductDiscountGuid != discount.ProductDiscountGuid))
                    return new Result(false, "已存在全店铺折扣！");

                if (query.IsExist(m => m.MemberLevelGuid == discount.MemberLevelGuid && m.MerchantGuid == Common.CurrentManage.MerchantGuid && m.ProductGuid == discount.ProductGuid && m.ProductDiscountGuid != discount.ProductDiscountGuid))
                    return new Result(false, "已存在商品折扣！");
            }

            return new Result(true);
        }
    }
}
