﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using System.Data.SqlClient;
using System.Data;
using System.Transactions;
using RFrameGenerate.Model; 
using RFrameGenerate.DAL.Enums;
using RFrameGenerate.DAL.Interface;
using RFrameGenerate.Common;
using RFramework.SQLQuery;
using RFramework.SQLServer;
using RFramework.SQLServer.Enums;

namespace RFrameGenerate.DAL.SQLServer
{
    internal class CoursesDAL : IQuery<Courses>
    {
        /// <summary>
        /// 事务处理
        /// </summary>
        public SqlRTransaction sqlRTransaction { get; set; }
        
        #region IsExist
        
        /// <summary>
        /// 判断记录是否存在，根据记录的GUID值
        /// </summary>
        /// <param name="dataGuid">待判断的记录的GUID值</param>
        /// <returns>返回判断结果，存在为True，不存在为False</returns>
        public bool IsExist(Guid dataGuid)
        {
            return IsExist("CoursesGuid", dataGuid);
        } 
        
		/// <summary>
        /// 判断记录是否存在，根据字段名称和字段值
        /// </summary>
        /// <param name="field">字段名称</param>
        /// <param name="value">字段值</param>
        /// <returns>返回判断结果，存在为True，不存在为False</returns>
        public bool IsExist<T>(string field, T value)
        {
            SqlParameter[] parameters = { 
                                            new SqlParameter("@iCount", SqlDbType.Int),
                                            new SqlParameter("@strField", SqlDbType.NVarChar),
                                            new SqlParameter("@strValue", SqlDbType.NVarChar)
                                        };
            parameters[0].Direction = ParameterDirection.Output;
            parameters[1].Value = field;
            parameters[2].Value = value.ToString();
            try
            {
                SqlHelper.ExecuteNonQuery("PCourses_IsExist", CommandType.StoredProcedure, parameters, sqlRTransaction);
                return parameters[0].Value.ToString() != "0" ? true : false;
            }
            catch (Exception ex)
            {
                Dispose();
                throw new Exception(string.Format("IsExist()参数:strField={0},strValue={1}获取数据出错,异常:{2}", field, value, ex.Message), ex);
            }
        }
         /// <summary>
        /// 判断是否存在
        /// </summary>
        /// <param name="conditionExpression">条件</param>
        /// <returns></returns>
        public bool IsExist(Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false)
        {
            QueryTranslator where = new QueryTranslator();
            string conditionString = where.Translate(conditionExpression);
            if (isFilterNull == true)
                conditionString = conditionString.SqlFilterNull();
            return IsExist(conditionString);
        }
        /// <summary>
        /// 判断是否存在
        /// </summary>
        /// <param name="conditionString">条件</param>
        /// <returns></returns>
        public bool IsExist(string conditionString)
        {
            SqlParameter[] parameters = { 
                                            new SqlParameter("@iCount", SqlDbType.Int),
                                            new SqlParameter("@strCondition", SqlDbType.NVarChar)
                                        };
            parameters[0].Direction = ParameterDirection.Output;
            parameters[1].Value = conditionString;
            try
            {
                SqlHelper.ExecuteNonQuery("PCourses_IsExist", CommandType.StoredProcedure, parameters, sqlRTransaction);
                return parameters[0].Value.ToString() != "0" ? true : false;
            }
            catch (Exception ex)
            {
                Dispose();
                throw new Exception(string.Format("IsExist()参数:strCondition={0}获取数据出错,异常:{1}", conditionString, ex.Message), ex);
            }
        }   
        #endregion
		
        #region GetDataByGuid
         
        /// <summary>
        /// 获取一条记录，根据记录的GUID值
        /// </summary>
        /// <param name="dataGuid">待获取的记录的GUID值</param>
        /// <returns>返回指定记录，记录不存在则返回Null</returns>
        public Courses GetDataByGuid(Guid dataGuid)
        {
            SqlParameter[] parameters = {new SqlParameter("@CoursesGuid", SqlDbType.UniqueIdentifier)};
            parameters[0].Value = dataGuid;
            
            SqlDataReader dataReader = SqlHelper.ExecuteReader("PCourses_GetDataByGuid", CommandType.StoredProcedure, parameters, sqlRTransaction);

            Courses data = null;
            try
            {
                if (dataReader.Read())
                    data = LoadModel(dataReader);
            }
            catch (Exception ex)
            {
                dataReader.Close();
                Dispose();
                throw new Exception(string.Format("GetDataByGuid()参数:dataGuid={0}获取数据出错,异常:{1}", dataGuid, ex.Message), ex);
            }        
                
            dataReader.Close();
             
            return data;
        }
        #endregion
        
        #region Add
        /// <summary>
        /// 批量更新记录集，根据dataList中各个记录的状态
        /// 记录的状态为ToBeAdd的，新增这条记录到数据库
        /// 记录的状态为ToBeDelete的，删除这条记录到数据库
        /// 记录的状态为ToBeUpdate的，更新这条记录到数据库
        /// 记录的状态为Original的，不对这条记录进行任何操作
        /// </summary>
        /// <param name="dataList">待更新的记录集</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>        
        public void AddList(List<Courses> dataList)
        {
            bool istran = false;
            if (sqlRTransaction.IsTransaction == false)
            {
                istran = true;
                sqlRTransaction.BeginTransaction();
            }

            DataTable dataListAdd = SqlHelper.GetTables("Courses",sqlRTransaction);

            
            foreach (var data in dataList)
            {
                DataRow dr = dataListAdd.NewRow();
                if(data.CoursesGuid != null)
    			    dr["CoursesGuid"] = data.CoursesGuid;
                else
                    dr["CoursesGuid"] = DBNull.Value;
                    
                if(data.Subject != null)
    			    dr["Subject"] = data.Subject;
                else
                    dr["Subject"] = DBNull.Value;
                    
                if(data.Grade != null)
    			    dr["Grade"] = data.Grade;
                else
                    dr["Grade"] = DBNull.Value;
                    
                dr["SCreateDate"] = DateTime.Now;
                dr["SCreateUserName"] = CurrentManage.UserName;
                dataListAdd.Rows.Add(dr);
            }
            
            #region 更新到数据库
            if (dataListAdd.Rows.Count>0)
            { 
            //添加
            SqlHelper.BatchAddData(new DataTable[] { dataListAdd }, sqlRTransaction);
            }
            #endregion

            if (istran == true)
                sqlRTransaction.Commit();

            dataListAdd.Clear();
        }
        /// <summary>
        /// 新增一条记录
        /// </summary>
        /// <param name="data">待新增的记录</param>
        /// <returns>返回新增结果，成功为True，失败为False</returns>
        public void Add(Courses data)
        {
            SqlParameter[] parameters = {
                                            new SqlParameter("@SCreateDate", SqlDbType.DateTime),
                                            new SqlParameter("@SCreateUserName", SqlDbType.NVarChar),
											new SqlParameter("@CoursesGuid", SqlDbType.UniqueIdentifier),
											new SqlParameter("@Subject", SqlDbType.NVarChar),
											new SqlParameter("@Grade", SqlDbType.Decimal),
										};
            
            parameters[0].Direction = ParameterDirection.Output;
            parameters[1].Value = CurrentManage.UserName;
            parameters[2].Value = data.CoursesGuid;
            parameters[3].Value = data.Subject;
            parameters[4].Value = data.Grade;
			try
            {
                int rowsAffected = SqlHelper.ExecuteNonQuery("PCourses_Add", CommandType.StoredProcedure, parameters, sqlRTransaction);
                bool result = rowsAffected != -1 ? true : false;
                if(result == false)
                    throw new Exception("新增数据失败");
            }
            catch(Exception ex)
            {
                Dispose();
                throw new Exception(string.Format("Add()参数:data={0}添加数据出错,异常:{1}", data.ToString(), ex.Message), ex);
            }
        }
        #endregion
		
        #region GetFirstData
        
		/// <summary>
        /// 获取第一条记录，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="orderString">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的记录集</returns>
        public Courses GetFirstData(Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<Courses, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default,LockType flag=LockType.Default)
        {
            QueryTranslator where = new QueryTranslator();
            string conditionString = where.Translate(conditionExpression);
            string orderString = string.Empty;
            if (orderExpression != null)
                orderString = where.Translate(orderExpression);
            if (isFilterNull == true)
                conditionString = conditionString.SqlFilterNull();
            
            return GetFirstData(conditionString, orderString, orderMethod, flag);        
        }

        /// <summary>
        /// 获取第一条记录，根据查询条件及排序字段、排序方法
        /// </summary>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <param name="orderString">排序字段字符串，SQL语句中的OrderBy表达式，字段间用“,”号隔开</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的第一条记录，记录不存在则返回Null</returns>
        public Courses GetFirstData(string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default,LockType flag=LockType.Default)
        { 
            if (conditionString == null)
                conditionString = string.Empty;
            if (orderString == null)
                orderString = string.Empty;
            SqlParameter[] parameters = {
								            new SqlParameter("@strCondition", SqlDbType.NVarChar), 
								            new SqlParameter("@strOrder", SqlDbType.NVarChar),
                                            new SqlParameter("@strLockType", SqlDbType.NVarChar)
                                        };
            parameters[0].Value = conditionString.Trim();
            parameters[1].Value = orderString.Trim();
            parameters[2].Value = flag.ToSqlLock();
            if (orderString.Trim() != string.Empty)
                parameters[1].Value += orderMethod == OrderMethods.DESC ? " DESC" : " ASC";
                
            SqlDataReader dataReader = SqlHelper.ExecuteReader("PCourses_GetFirstData", CommandType.StoredProcedure, parameters, sqlRTransaction);

            Courses data = null;
            try
            {
                if (dataReader.Read())
                    data = LoadModel(dataReader);
            }
            catch (Exception ex)
            {
                dataReader.Close();
                Dispose();
                throw new Exception(string.Format("GetFirstData()参数:conditionString={0}获取数据出错,异常:{1}", conditionString, ex.Message), ex);
            }         
            dataReader.Close();
            
            return data;
        }
        #endregion

        #region Update
        /// 更新一条记录
        /// </summary>
        /// <param name="valueExpression">待更新的记录</param>
        /// <param name="conditionExpression">要更新的条件</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>        
        public void Update(Expression<Func<Courses, object[]>> valueExpression, Expression<Func<Courses, bool>> conditionExpression)
        {            
            QueryTranslator query = new QueryTranslator();
            string value = query.Translate(valueExpression).Replace(")(", ",").Substring(1);
            value = value.Substring(0, value.Length - 1);
            string condition = query.Translate(conditionExpression);
            if (condition == string.Empty)
                throw new Exception("删除条件不能为null或空");


            SqlParameter[] parameters = {
                                            new SqlParameter("@value", SqlDbType.NVarChar),
                                            new SqlParameter("@strCondition", SqlDbType.NVarChar),
										};

            parameters[0].Value = value;
            parameters[1].Value = condition;
            try
            {
                int rowsAffected = SqlHelper.ExecuteNonQuery("PCourses_SingleUpdate", CommandType.StoredProcedure, parameters, sqlRTransaction);
                bool result = rowsAffected != -1 ? true : false;

                if (result == false)
                    throw new Exception("更新数据失败");
            }
            catch (Exception ex)
            {
                Dispose();
                throw new Exception(string.Format("Update()参数:conditionString={0}更新数据失败,异常:{1}", condition, ex.Message), ex);
            }
        }
        /// <summary>
        /// 更新一条记录
        /// </summary>
        /// <param name="data">待更新的记录</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>
        public void Update(Courses data)
        {
            SqlParameter[] parameters = {
                                            new SqlParameter("@SUpdateDate", SqlDbType.DateTime),
                                            new SqlParameter("@SUpdateUserName", SqlDbType.NVarChar),
											new SqlParameter("@CoursesGuid", SqlDbType.UniqueIdentifier),
											new SqlParameter("@Subject", SqlDbType.NVarChar),
											new SqlParameter("@Grade", SqlDbType.Decimal),
										};										
			
            parameters[0].Direction = ParameterDirection.Output;
            parameters[1].Value = CurrentManage.UserName;
			parameters[2].Value = data.CoursesGuid;
			parameters[3].Value = data.Subject;
			parameters[4].Value = data.Grade;
			try
            {
                int rowsAffected = SqlHelper.ExecuteNonQuery("PCourses_Update", CommandType.StoredProcedure, parameters, sqlRTransaction);
                bool result = rowsAffected != -1 ? true : false;
                
                if (result == false)
                    throw new Exception("更新数据失败");
            }
            catch (Exception ex)
            {
                Dispose();
                throw new Exception(string.Format("Update()参数:conditionString={0}更新数据失败,异常:{1}", data.ToString(), ex.Message), ex);
            }
        }
        
        /// <summary>
        /// 批量更新记录集，根据dataList中各个记录的状态
        /// 记录的状态为ToBeAdd的，新增这条记录到数据库
        /// 记录的状态为ToBeDelete的，删除这条记录到数据库
        /// 记录的状态为ToBeUpdate的，更新这条记录到数据库
        /// 记录的状态为Original的，不对这条记录进行任何操作
        /// </summary>
        /// <param name="dataList">待更新的记录集</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>        
        public void UpdateList(List<Courses> dataList)
        {
            bool istran = false;
            if (sqlRTransaction.IsTransaction == false)
            {
                istran = true;
                sqlRTransaction.BeginTransaction();
            }

            DataTable dataListAdd = SqlHelper.GetTables("Courses",sqlRTransaction);

            List<Courses> dataListUpdate = new List<Courses>();
            List<string> dataListDelete = new List<string>();            
            StringBuilder sbDelete = new StringBuilder();
            sbDelete.Append(" CoursesGuid in (");
            int deleteCount = 0;
            string strSql = string.Empty;
            foreach (var data in dataList)
            {
                switch (data.EntityState)
                {
                    case EntityStates.ToBeAdd:
                        DataRow dr = dataListAdd.NewRow();
            			dr["CoursesGuid"] = data.CoursesGuid;
            			dr["Subject"] = data.Subject;
            			dr["Grade"] = data.Grade;
                        dr["SCreateDate"] = DateTime.Now;
                        dr["SCreateUserName"] = CurrentManage.UserName;
                        dataListAdd.Rows.Add(dr);
                        break;
                    case EntityStates.ToBeDelete:
                        sbDelete.AppendFormat(" '{0}',", data.CoursesGuid);
                        deleteCount++;
                        if (deleteCount == 20)
                        {
                            sbDelete.Append(")");
                            deleteCount = 0;
                            strSql = sbDelete.ToString();
                            dataListDelete.Add(strSql.Replace(",)",") "));
                            sbDelete.Clear();
                            sbDelete.Append(" CoursesGuid in (");
                        }
                        break;
                    case EntityStates.ToBeUpdated:
                        dataListUpdate.Add(data);
                        break;
                }
            }
            if (sbDelete.Length>0 && sbDelete.ToString() != " CoursesGuid in (")
            {
                sbDelete.Append(")");
                strSql = sbDelete.ToString();
                dataListDelete.Add(strSql.Replace(",)",") "));
                sbDelete.Clear();
            }

            #region 更新到数据库
            if (dataListAdd.Rows.Count>0)
            { 
            //添加
            SqlHelper.BatchAddData(new DataTable[] { dataListAdd }, sqlRTransaction);
            }
            //修改
            foreach(var v in dataListUpdate)
            {
                Update(v);
            }
            //删除
            foreach(var v in dataListDelete)
            {
                Delete(v);
            }
            #endregion

            if (istran == true)
                sqlRTransaction.Commit();

            dataListAdd.Clear();
            dataListUpdate.Clear();
            dataListDelete.Clear();
            dataList.ForEach(a => a.EntityState = EntityStates.Original);
        }

        #endregion
         
        #region Delete
        /// <summary>
        /// 删除一条记录，根据记录的GUID值
        /// </summary>
        /// <param name="dataGuid">待删除的记录的GUID值</param>
        /// <returns>返回删除结果，成功为True，失败为False</returns>
        public void Delete(Guid dataGuid)
        { 
			Delete(string.Format("CoursesGuid = '{0}'",dataGuid));
        }
                
         /// <summary>
        /// 根据条件删除记录
        /// </summary>
        /// <param name="dataGuid">删除条件</param>
        /// <returns>返回删除结果，成功为True，失败为False</returns>
        public void Delete(Expression<Func<Courses, bool>> conditionExpression)
        { 
            QueryTranslator where = new QueryTranslator();
            string conditionString = where.Translate(conditionExpression);
            Delete(conditionString);
        }
        
        /// <summary>
        /// 批量删除数据
        /// 记录的状态为ToBeAdd的，新增这条记录到数据库
        /// 记录的状态为ToBeDelete的，删除这条记录到数据库
        /// 记录的状态为ToBeUpdate的，更新这条记录到数据库
        /// 记录的状态为Original的，不对这条记录进行任何操作
        /// </summary>
        /// <param name="dataList">待更新的记录集</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>        
        public void DeleteList(List<Courses> dataList)
        {
            bool istran = false;
            if (sqlRTransaction.IsTransaction == false)
            {
                istran = true;
                sqlRTransaction.BeginTransaction();
            }
            List<string> dataListDelete = new List<string>();            
            StringBuilder sbDelete = new StringBuilder();
            sbDelete.Append(" CoursesGuid in (");
            int deleteCount = 0;
            string strSql = string.Empty;
            foreach (var data in dataList)
            {              
                sbDelete.AppendFormat(" '{0}',", data.CoursesGuid);
                deleteCount++;
                if (deleteCount == 20)
                {
                    sbDelete.Append(")");
                    deleteCount = 0;
                    strSql = sbDelete.ToString();
                    dataListDelete.Add(strSql.Replace(",)",") "));
                    sbDelete.Clear();
                    sbDelete.Append(" CoursesGuid in (");
                }
            }
            if (sbDelete.Length>0 && sbDelete.ToString() != " CoursesGuid in (")
            {
                sbDelete.Append(")");
                strSql = sbDelete.ToString();
                dataListDelete.Add(strSql.Replace(",)",") "));
                sbDelete.Clear();
            }

            #region 更新到数据库           
            //删除
            foreach(var v in dataListDelete)
            {
                Delete(v);
            }
            #endregion

            if (istran == true)
                sqlRTransaction.Commit();

            dataListDelete.Clear();
        }
        
         /// <summary>
        /// 根据条件删除记录
        /// </summary>
        /// <param name="dataGuid">删除条件</param>
        /// <returns>返回删除结果，成功为True，失败为False</returns>
        public void Delete(string conditionString = "")
        {
            if (conditionString == null || conditionString ==String.Empty)
                throw new Exception("删除条件不能为null或空");
            SqlParameter[] parameters =
            {
                new SqlParameter("@strCondition", SqlDbType.NVarChar)
            };
            parameters[0].Value = conditionString; 
            try
            {
                int rowsAffected = SqlHelper.ExecuteNonQuery("PCourses_Delete", CommandType.StoredProcedure, parameters, sqlRTransaction);
                
                if(rowsAffected == -1)
                    throw new Exception("删除数据失败");
            }
            catch (Exception ex)
            {
                Dispose();
                throw new Exception(string.Format("Delete()参数:conditionString={0}删除数据失败,异常:{1}", conditionString, ex.Message), ex);
            }
        }
        
        #endregion
        
        #region GetList
         
		/// <summary>
        /// 获取记录集，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="orderString">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的记录集</returns>
        public List<Courses> GetList(Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<Courses, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default,LockType flag=LockType.Default)
        {
            QueryTranslator where = new QueryTranslator();
            string conditionString = where.Translate(conditionExpression);
            string orderString = string.Empty;
            if (orderExpression != null)
                orderString = where.Translate(orderExpression);
            if (isFilterNull == true)
                conditionString = conditionString.SqlFilterNull();
            
            return GetList(conditionString, orderString, orderMethod, flag);
         }
        
        /// <summary>
        /// 获取记录集，根据查询条件及排序字段、排序方法
        /// </summary>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <param name="orderString">排序字段字符串，SQL语句中的OrderBy表达式，字段间用“,”号隔开</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的记录集</returns>
        public List<Courses> GetList(string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default,LockType flag=LockType.Default)
        { 
            if (conditionString == null)
                conditionString = string.Empty;
            if (orderString == null)
                orderString = string.Empty;
			SqlParameter[] parameters = {
											new SqlParameter("@strCondition", SqlDbType.NVarChar), 
											new SqlParameter("@strOrder", SqlDbType.NVarChar),
                                            new SqlParameter("@strLockType", SqlDbType.NVarChar)
										};
            parameters[0].Value = conditionString.Trim();
            parameters[1].Value = orderString.Trim();
            parameters[2].Value = flag.ToSqlLock();
			if (orderString.Trim() != string.Empty)
				parameters[1].Value += orderMethod == OrderMethods.DESC ? " DESC" : " ASC";
			
            List<Courses> dataList = new List<Courses>();
			           
            SqlDataReader dataReader = SqlHelper.ExecuteReader("PCourses_GetList",CommandType.StoredProcedure, parameters, sqlRTransaction);

			try
            {
                while (dataReader.Read())
                {
                    dataList.Add(LoadModel(dataReader));
                }
            }
            catch (Exception ex)
            {
                dataReader.Close();
                Dispose();
                throw new Exception(string.Format("GetList获取数据出错,conditionString={0},异常:{1}", conditionString, ex.Message), ex);
            }
            dataReader.Close();
    						
            return dataList;             
        }	
        
        #endregion
        
        #region GetCount
		
		/// <summary>
        /// 获取记录数，根据查询条件Lambda表达式
        /// </summary>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <returns>返回满足条件的记录集</returns>
        public int GetCount(Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false)
        {
            QueryTranslator where = new QueryTranslator();
            string conditionString = where.Translate(conditionExpression);
            if (isFilterNull == true)
                conditionString = conditionString.SqlFilterNull();
            return GetCount(conditionString);
        }

        /// <summary>
        /// 获取记录数，根据查询条件
        /// </summary>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <returns>返回满足条件的记录数</returns>
        public int GetCount(string conditionString = "")
        { 
            if (conditionString == null)
                conditionString = string.Empty; 
            SqlParameter[] parameters = {
											new SqlParameter("@strCondition", SqlDbType.NVarChar),
                                            new SqlParameter("@iRecordCount", SqlDbType.Int),
										};
            parameters[0].Value = conditionString.Trim();
            parameters[1].Direction = ParameterDirection.Output;
            int recordCount;
            SqlDataReader dataReader =  SqlHelper.ExecuteReader("PCourses_GetCount", CommandType.StoredProcedure, parameters, sqlRTransaction);
            try
            {
                recordCount = (int)parameters[1].Value;
            }
            catch (Exception ex)
            {
                dataReader.Close();
                Dispose();
                throw new Exception(string.Format("GetCount获取总记录出错,conditionString={0},异常:{1}", conditionString, ex.Message), ex);
            }
            dataReader.Close();
            
            return recordCount;
        } 
        
        #endregion
        
        #region GetPageList
		
		/// <summary>
        /// 获取记录集，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="page">页号</param>
        /// <param name="pageSize">每页的记录数</param>
        /// <param name="pageCount">返回总页数</param>
        /// <param name="recordCount">返回总记录数</param>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="orderString">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的记录集</returns>
        public List<Courses> GetPageList(int page, int pageSize, out int pageCount, out int recordCount, Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<Courses, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default,LockType flag=LockType.Default)
        {
            QueryTranslator where = new QueryTranslator();
            string conditionString = where.Translate(conditionExpression);
            string orderString = string.Empty;
            if (orderExpression != null)
                orderString = where.Translate(orderExpression);
            if (isFilterNull == true)
                conditionString = conditionString.SqlFilterNull();
           
            return GetPageList(page, pageSize,out pageCount, out recordCount, conditionString, orderString, orderMethod, flag);
        }
        /// <summary>
        /// 获取记录集，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="gridModel">查询条件</param>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="orderString">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的记录集</returns>
        public List<Courses> GetPageList(GridModel gridModel, Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<Courses, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default)
        {
            return GetPageList(gridModel.page, gridModel.rows, out gridModel.pageCount, out gridModel.recordCount, conditionExpression,isFilterNull, orderExpression, orderMethod, flag);
        }
        /// <summary>
        /// 获取记录集，根据查询条件及排序字段、排序方法，分页
        /// </summary>      
        /// <param name="page">页号</param>
        /// <param name="pageSize">每页的记录数</param>
        /// <param name="pageCount">返回总页数</param>
        /// <param name="recordCount">返回总记录数</param>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <param name="orderString">排序字段字符串，SQL语句中的OrderBy表达式，字段间用“,”号隔开</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <returns>返回满足条件的记录集，分页</returns>
        public List<Courses> GetPageList(int page, int pageSize, out int pageCount, out int recordCount, string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default,LockType flag=LockType.Default)
        { 
            if (conditionString == null)
                conditionString = string.Empty;
            if (orderString == null)
                orderString = string.Empty;
            List<Courses> dataList = new List<Courses>();
            SqlParameter[] parameters = {
											new SqlParameter("@iPage", SqlDbType.Int),
											new SqlParameter("@iPageSize", SqlDbType.Int),
											new SqlParameter("@iPageCount", SqlDbType.Int),
											new SqlParameter("@iRecordCount", SqlDbType.Int),
											new SqlParameter("@strCondition", SqlDbType.NVarChar),
										    new SqlParameter("@strOrder", SqlDbType.NVarChar),
                                            new SqlParameter("@strLockType", SqlDbType.NVarChar)
                                        };
            parameters[0].Value = page;
            parameters[1].Value = pageSize;
            parameters[2].Direction = ParameterDirection.Output;
            parameters[3].Direction = ParameterDirection.Output;
            parameters[4].Value = conditionString.Trim();
            parameters[5].Value = orderString.Trim();
            parameters[6].Value = flag.ToSqlLock();
			if (orderString.Trim() != string.Empty)
				parameters[5].Value += orderMethod == OrderMethods.DESC ? " DESC" : " ASC";			
			
			SqlDataReader dataReader = SqlHelper.ExecuteReader("PCourses_GetPageList",CommandType.StoredProcedure, parameters,sqlRTransaction);
            try
            {
                while (dataReader.Read())
                {
                    dataList.Add(LoadModel(dataReader));
                }
			}
            catch (Exception ex)
            {
                dataReader.Close();
                Dispose();
                throw new Exception(string.Format("GetPageList获取数据记录出错,conditionString={0},异常:{1}", conditionString, ex.Message), ex);
            }
            dataReader.Close();
			
            pageCount = (int)parameters[2].Value;
            recordCount = (int)parameters[3].Value;
			
            return dataList;
        }
        
        #endregion
   
        /// <summary>
        /// 获取记录实体，从 DataReader 的当前记录装载
        /// </summary>
        /// <param name="dataReader">DataReader 对象</param>
        /// <returns>返回 DataReader 里下一条记录，记录不存在则返回Null</returns>
        private Courses LoadModel(SqlDataReader dataReader)
        {
            Courses data = new Courses();
			if (dataReader["CoursesGuid"] != DBNull.Value)
				data.CoursesGuid = (Guid)dataReader["CoursesGuid"];
			if (dataReader["Subject"] != DBNull.Value)
				data.Subject = (string)dataReader["Subject"];
			if (dataReader["Grade"] != DBNull.Value)
				data.Grade = (decimal)dataReader["Grade"];
            data.EntityState = EntityStates.Original; 
            return data;
        }
        
          #region IDisposable 成员
        /// <summary>
        /// 回收
        /// </summary>
        public void Dispose()
        {
            sqlRTransaction.Dispose();
        }

        #endregion



        public List<Courses> GetTopData(string top, Expression<Func<Courses, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<Courses, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default)
        {
            throw new NotImplementedException();
        }

        public List<Courses> GetTopData(string top, string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default)
        {
            throw new NotImplementedException();
        }
    }
}

