﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Linq.Expressions;
using System.Data.SqlClient;
using RFramework.SQLServer;
using RFramework.SQLServer.Enums;
using RFrameGenerate.DAL.Enums;
using RFrameGenerate.Model;

namespace YDT_2.DAL.Interface
{
    /// <summary>
    /// 数据表访问接口
    /// </summary>
    /// <typeparam name="T">记录类型</typeparam>
    public interface IQuery2000<T> : IDisposable
    {
        SqlRTransaction sqlRTransaction { get; set; }

        #region IsExist*
        /// <summary>
        /// 判断记录是否存在，根据字段名称和字段值
        /// </summary>
        /// <param name="conditionString">条件</param>
        /// <returns>返回判断结果，存在为True，不存在为False</returns>
        bool IsExist(Expression<Func<T, bool>> conditionExpression, bool isFilterNull = false);
        /// <summary>
        /// 判断记录是否存在，根据字段名称和字段值
        /// </summary>
        /// <param name="conditionString">条件</param>
        /// <returns>返回判断结果，存在为True，不存在为False</returns>
        bool IsExist(string conditionString);
        #endregion
        
        #region Add
        /// <summary>
        /// 新增一条记录
        /// </summary>
        /// <param name="data">待新增的记录</param>
        /// <returns>返回新增结果，成功为True，失败为False</returns>
        void Add(T data);
        /// <summary>
        /// 批量更新记录集，根据dataList中各个记录的状态
        /// </summary>
        /// <param name="dataList">待更新的记录集</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>        
        void AddList(List<T> dataList);
        #endregion

        #region GetFirstData
        /// <summary>
        /// 获取第一条记录，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="isFilterNull">查询条件是否过滤</param>
        /// <param name="orderExpression">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录集</returns>
        T GetFirstData(Expression<Func<T, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<T, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);

        /// <summary>
        /// 获取第一条记录，根据查询条件及排序字段、排序方法
        /// </summary>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <param name="orderString">排序字段字符串，SQL语句中的OrderBy表达式，字段间用“,”号隔开</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的第一条记录，记录不存在则返回Null</returns>
        T GetFirstData(string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);
        #endregion

        #region Update*
        /// <summary>
        /// 更新一条记录
        /// </summary>
        /// <param name="valueExpression">待更新的记录</param>
        /// <param name="conditionExpression">更新的条件</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>
        void Update(Expression<Func<T, object[]>> valueExpression, Expression<Func<T, bool>> conditionExpression);
        /// <summary>
        /// 更新一条记录
        /// </summary>
        /// <param name="data">待更新的记录</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>
        void Update(T data, Expression<Func<T, bool>> conditionExpression);
        /// <summary>
        /// 更新一条记录
        /// </summary>
        /// <param name="data">待更新的记录</param>
        /// <returns>返回更新结果，成功为True，失败为False</returns>
        void Update(T data, string conditionString);
        /// <summary>
        /// 批量更新
        /// </summary>
        /// <param name="list"></param>
        void UpdateList(List<T> list);
        #endregion

        #region Delete*

        /// <summary>
        /// 根据条件删除记录
        /// </summary>
        /// <param name="dataGuid">删除条件</param>
        /// <returns>返回删除结果，成功为True，失败为False</returns>
        void Delete(Expression<Func<T, bool>> conditionExpression);
        /// <summary>
        /// 根据条件删除记录
        /// </summary>
        /// <param name="dataGuid">删除条件</param>
        /// <returns>返回删除结果，成功为True，失败为False</returns>
        void Delete(string conditionString = "");

        #endregion

        #region GetList
        /// <summary>
        /// 获取记录集，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="isFilterNull">是否过滤条件中的空字段</param>
        /// <param name="orderExpression">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录集</returns>
        List<T> GetList(Expression<Func<T, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<T, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);

        /// <summary>
        /// 获取记录集，根据查询条件及排序字段、排序方法
        /// </summary>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <param name="orderString">排序字段字符串，SQL语句中的OrderBy表达式，字段间用“,”号隔开</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录集</returns>
        List<T> GetList(string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);
        #endregion

        #region GetCount
        /// <summary>
        /// 获取记录数，根据查询条件Lambda表达式
        /// </summary>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录数</returns>
        int GetCount(Expression<Func<T, bool>> conditionExpression, bool isFilterNull = false);

        /// <summary>
        /// 获取记录数，根据查询条件
        /// </summary>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <returns>返回满足条件的记录数</returns>
        int GetCount(string conditionString = "");
        #endregion

        #region GetPageList
        /// <summary>
        /// 获取记录集，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="page">页号</param>
        /// <param name="pageSize">每页的记录数</param>
        /// <param name="pageCount">返回总页数</param>
        /// <param name="recordCount">返回总记录数</param>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="isFilterNull">查询条件是否过滤</param>
        /// <param name="orderString">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录集</returns>
        List<T> GetPageList(int page, int pageSize, out int pageCount, out int recordCount, Expression<Func<T, bool>> conditionExpression, bool isFilterNull = false, Expression<Func<T, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);
        /// <summary>
        /// 获取记录集，根据查询条件Lambda表达式及排序字段Lambda表达式、排序方法
        /// </summary>
        /// <param name="gridModel">分页条件</param>
        /// <param name="conditionExpression">查询条件Lambda表达式</param>
        /// <param name="isFilterNull">查询条件是否过滤</param>
        /// <param name="orderString">排序字段Lambda表达式，字段间用“+”号连接</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录集</returns>
        List<T> GetPageList(GridModel gridModel, Expression<Func<T, bool>> conditionExpression = null, bool isFilterNull = false, Expression<Func<T, object>> orderExpression = null, OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);

        /// <summary>
        /// 获取记录集，根据查询条件及排序字段、排序方式，分页
        /// </summary>      
        /// <param name="page">页号</param>
        /// <param name="pageSize">每页的记录数</param>
        /// <param name="pageCount">返回总页数</param>
        /// <param name="recordCount">返回总记录数</param>
        /// <param name="conditionString">查询条件字符串，SQL语句中的WHERE表达式</param>
        /// <param name="orderString">排序字段字符串，SQL语句中的OrderBy表达式，字段间用“,”号隔开</param>
        /// <param name="orderMethod">排序方法，升序或降序排列</param>
        /// <param name="flag">锁类型</param>
        /// <returns>返回满足条件的记录集，分页</returns>
        List<T> GetPageList(int page, int pageSize, out int pageCount, out int recordCount, string conditionString = "", string orderString = "", OrderMethods orderMethod = OrderMethods.Default, LockType flag = LockType.Default);
        #endregion
    }
}
